/*
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied.  See the License for the
 * specific language governing permissions and limitations
 * under the License.
 */
package org.apache.spark.sql.sedona_sql.expressions

import org.apache.sedona.common.{Functions, FunctionsGeoTools}
import org.apache.sedona.common.sphere.{Haversine, Spheroid}
import org.apache.spark.sql.catalyst.InternalRow
import org.apache.spark.sql.catalyst.expressions.codegen.CodegenFallback
import org.apache.spark.sql.catalyst.expressions.{Expression, Generator}
import org.apache.spark.sql.catalyst.util.ArrayData
import org.apache.spark.sql.sedona_sql.UDT.GeometryUDT
import org.apache.spark.sql.sedona_sql.expressions.implicits._
import org.apache.spark.sql.types._
import org.locationtech.jts.algorithm.MinimumBoundingCircle
import org.locationtech.jts.geom._
import org.apache.spark.sql.sedona_sql.expressions.InferrableFunctionConverter._

/**
  * Return the distance between two geometries.
  *
  * @param inputExpressions This function takes two geometries and calculates the distance between two objects.
  */
case class ST_Distance(inputExpressions: Seq[Expression])
  extends InferredExpression(Functions.distance _) {

  protected def withNewChildrenInternal(newChildren: IndexedSeq[Expression]) = {
    copy(inputExpressions = newChildren)
  }
}


case class ST_YMax(inputExpressions: Seq[Expression])
  extends InferredExpression(Functions.yMax _) {

  protected def withNewChildrenInternal(newChildren: IndexedSeq[Expression]) = {
    copy(inputExpressions = newChildren)
  }
}

case class ST_YMin(inputExpressions: Seq[Expression])
  extends InferredExpression(Functions.yMin _) {

  protected def withNewChildrenInternal(newChildren: IndexedSeq[Expression]) = {
    copy(inputExpressions = newChildren)
  }
}

/**
  * Return the Z maxima of the geometry.
  *
  * @param inputExpressions This function takes a geometry and returns the maximum of all Z-coordinate values.
  */
case class ST_ZMax(inputExpressions: Seq[Expression])
  extends InferredExpression(Functions.zMax _) {

  protected def withNewChildrenInternal(newChildren: IndexedSeq[Expression]) = {
    copy(inputExpressions = newChildren)
  }
}

/**
  * Return the Z minima of the geometry.
  *
  * @param inputExpressions This function takes a geometry and returns the minimum of all Z-coordinate values.
  */
case class ST_ZMin(inputExpressions: Seq[Expression])
  extends InferredExpression(Functions.zMin _) {

  protected def withNewChildrenInternal(newChildren: IndexedSeq[Expression]) = {
    copy(inputExpressions = newChildren)
  }
}

case class ST_3DDistance(inputExpressions: Seq[Expression])
  extends InferredExpression(Functions.distance3d _) {

  protected def withNewChildrenInternal(newChildren: IndexedSeq[Expression]) = {
    copy(inputExpressions = newChildren)
  }
}

/**
  * Return the concave hull of a Geometry.
  *
  * @param inputExpressions
  */
case class ST_ConcaveHull(inputExpressions: Seq[Expression])
  extends InferredExpression(Functions.concaveHull _) {

  protected def withNewChildrenInternal(newChildren: IndexedSeq[Expression]): Expression = {
    copy(inputExpressions = newChildren)
  }
}

/**
  * Return the convex hull of a Geometry.
  *
  * @param inputExpressions
  */
case class ST_ConvexHull(inputExpressions: Seq[Expression])
  extends InferredExpression(Functions.convexHull _) {

  protected def withNewChildrenInternal(newChildren: IndexedSeq[Expression]) = {
    copy(inputExpressions = newChildren)
  }
}

case class ST_CrossesDateLine(inputExpressions: Seq[Expression])
  extends InferredExpression(Functions.crossesDateLine _) {

  protected def withNewChildrenInternal(newChildren: IndexedSeq[Expression]) = {
    copy(inputExpressions = newChildren)
  }
}

/**
  * Return the number of Points in geometry.
  *
  * @param inputExpressions
  */
case class ST_NPoints(inputExpressions: Seq[Expression])
  extends InferredExpression(Functions.nPoints _) {

  protected def withNewChildrenInternal(newChildren: IndexedSeq[Expression]) = {
    copy(inputExpressions = newChildren)
  }
}

/**
  * Return the number of Dimensions in geometry.
  *
  * @param inputExpressions
  */
case class ST_NDims(inputExpressions: Seq[Expression])
  extends InferredExpression(Functions.nDims _) {

  protected def withNewChildrenInternal(newChildren: IndexedSeq[Expression]) = {
    copy(inputExpressions = newChildren)
  }
}

/**
  * Returns a geometry/geography that represents all points whose distance from this Geometry/geography is less than or equal to distance.
  *
  * @param inputExpressions
  */
case class ST_Buffer(inputExpressions: Seq[Expression])
  extends InferredExpression(inferrableFunction2(Functions.buffer), inferrableFunction3(Functions.buffer), inferrableFunction4(Functions.buffer)) {

  protected def withNewChildrenInternal(newChildren: IndexedSeq[Expression]) = {
    copy(inputExpressions = newChildren)
  }
}

case class ST_BestSRID(inputExpressions: Seq[Expression])
  extends InferredExpression(Functions.bestSRID _) {

  protected def withNewChildrenInternal(newChildren: IndexedSeq[Expression]) = {
    copy(inputExpressions = newChildren)
  }
}

case class ST_ShiftLongitude(inputExpressions: Seq[Expression])
  extends InferredExpression(Functions.shiftLongitude _) {

  protected def withNewChildrenInternal(newChildren: IndexedSeq[Expression]) = {
    copy(inputExpressions = newChildren)
  }
}

/**
  * Return the bounding rectangle for a Geometry
  *
  * @param inputExpressions
  */
case class ST_Envelope(inputExpressions: Seq[Expression])
  extends InferredExpression(Functions.envelope _) {

  protected def withNewChildrenInternal(newChildren: IndexedSeq[Expression]) = {
    copy(inputExpressions = newChildren)
  }
}

/**
  * Return the length measurement of a Geometry
  *
  * @param inputExpressions
  */
case class ST_Length(inputExpressions: Seq[Expression])
  extends InferredExpression(Functions.length _) {

  protected def withNewChildrenInternal(newChildren: IndexedSeq[Expression]) = {
    copy(inputExpressions = newChildren)
  }
}

/**
  * Return the area measurement of a Geometry.
  *
  * @param inputExpressions
  */
case class ST_Area(inputExpressions: Seq[Expression])
  extends InferredExpression(Functions.area _) {

  protected def withNewChildrenInternal(newChildren: IndexedSeq[Expression]) = {
    copy(inputExpressions = newChildren)
  }
}

/**
  * Return mathematical centroid of a geometry.
  *
  * @param inputExpressions
  */
case class ST_Centroid(inputExpressions: Seq[Expression])
  extends InferredExpression(Functions.getCentroid _) {

  protected def withNewChildrenInternal(newChildren: IndexedSeq[Expression]) = {
    copy(inputExpressions = newChildren)
  }
}

/**
  * Given a geometry, sourceEPSGcode, and targetEPSGcode, convert the geometry's Spatial Reference System / Coordinate Reference System.
  *
  * @param inputExpressions
  */
case class ST_Transform(inputExpressions: Seq[Expression])
  extends InferredExpression(inferrableFunction4(FunctionsGeoTools.transform), inferrableFunction3(FunctionsGeoTools.transform),
                              inferrableFunction2(FunctionsGeoTools.transform)) {

  protected def withNewChildrenInternal(newChildren: IndexedSeq[Expression]) = {
    copy(inputExpressions = newChildren)
  }
}

/**
  * Return the intersection shape of two geometries. The return type is a geometry
  *
  * @param inputExpressions
  */
case class ST_Intersection(inputExpressions: Seq[Expression])
  extends InferredExpression(Functions.intersection _) {

  protected def withNewChildrenInternal(newChildren: IndexedSeq[Expression]) = {
    copy(inputExpressions = newChildren)
  }
}

/**
  * Given an invalid geometry, create a valid representation of the geometry.
  * See: http://lin-ear-th-inking.blogspot.com/2021/05/fixing-invalid-geometry-with-jts.html
  *
  * @param inputExpressions
  */
case class ST_MakeValid(inputExpressions: Seq[Expression])
  extends InferredExpression(Functions.makeValid _) {

  protected def withNewChildrenInternal(newChildren: IndexedSeq[Expression]) = {
    copy(inputExpressions = newChildren)
  }
}

/**
  * Test if Geometry is valid.
  *
  * @param inputExpressions
  */
case class ST_IsValid(inputExpressions: Seq[Expression])
  extends InferredExpression(inferrableFunction2(Functions.isValid), inferrableFunction1(Functions.isValid)) {

  protected def withNewChildrenInternal(newChildren: IndexedSeq[Expression]) = {
    copy(inputExpressions = newChildren)
  }
}

/**
  * Test if Geometry is simple.
  *
  * @param inputExpressions
  */
case class ST_IsSimple(inputExpressions: Seq[Expression])
  extends InferredExpression(Functions.isSimple _) {

  protected def withNewChildrenInternal(newChildren: IndexedSeq[Expression]) = {
    copy(inputExpressions = newChildren)
  }
}

/**
  * Simplifies a geometry and ensures that the result is a valid geometry having the same dimension and number of components as the input,
  * and with the components having the same topological relationship.
  * The simplification uses a maximum-distance difference algorithm similar to the Douglas-Peucker algorithm.
  *
  * @param inputExpressions first arg is geometry
  *                         second arg is distance tolerance for the simplification(all vertices in the simplified geometry will be within this distance of the original geometry)
  */
case class ST_SimplifyPreserveTopology(inputExpressions: Seq[Expression])
  extends InferredExpression(Functions.simplifyPreserveTopology _) {

  protected def withNewChildrenInternal(newChildren: IndexedSeq[Expression]) = {
    copy(inputExpressions = newChildren)
  }
}

/**
  * Reduce the precision of the given geometry to the given number of decimal places
  *
  * @param inputExpressions The first arg is a geom and the second arg is an integer scale, specifying the number of decimal places of the new coordinate. The last decimal place will
  *                         be rounded to the nearest number.
  */
case class ST_ReducePrecision(inputExpressions: Seq[Expression])
  extends InferredExpression(Functions.reducePrecision _) {

  protected def withNewChildrenInternal(newChildren: IndexedSeq[Expression]) = {
    copy(inputExpressions = newChildren)
  }
}

case class ST_AsText(inputExpressions: Seq[Expression])
  extends InferredExpression(Functions.asWKT _) {

  protected def withNewChildrenInternal(newChildren: IndexedSeq[Expression]) = {
    copy(inputExpressions = newChildren)
  }
}

case class ST_AsGeoJSON(inputExpressions: Seq[Expression])
  extends InferredExpression(Functions.asGeoJson _) {

  protected def withNewChildrenInternal(newChildren: IndexedSeq[Expression]) = {
    copy(inputExpressions = newChildren)
  }
}

case class ST_AsBinary(inputExpressions: Seq[Expression])
  extends InferredExpression(Functions.asWKB _) {

  protected def withNewChildrenInternal(newChildren: IndexedSeq[Expression]) = {
    copy(inputExpressions = newChildren)
  }
}

case class ST_AsEWKB(inputExpressions: Seq[Expression])
  extends InferredExpression(Functions.asEWKB _) {

  protected def withNewChildrenInternal(newChildren: IndexedSeq[Expression]) = {
    copy(inputExpressions = newChildren)
  }
}

case class ST_SRID(inputExpressions: Seq[Expression])
  extends InferredExpression(Functions.getSRID _) {

  protected def withNewChildrenInternal(newChildren: IndexedSeq[Expression]) = {
    copy(inputExpressions = newChildren)
  }
}

case class ST_SetSRID(inputExpressions: Seq[Expression])
  extends InferredExpression(Functions.setSRID _) {

  protected def withNewChildrenInternal(newChildren: IndexedSeq[Expression]) = {
    copy(inputExpressions = newChildren)
  }
}

case class ST_GeometryType(inputExpressions: Seq[Expression])
  extends InferredExpression(Functions.geometryType _) {

  protected def withNewChildrenInternal(newChildren: IndexedSeq[Expression]) = {
    copy(inputExpressions = newChildren)
  }
}

/**
  * Returns a LineString formed by sewing together the constituent line work of a MULTILINESTRING.
  * Only works for MultiLineString. Using other geometry will return GEOMETRYCOLLECTION EMPTY
  * If the MultiLineString is can't be merged, the original multilinestring is returned
  *
  * @param inputExpressions Geometry
  */
case class ST_LineMerge(inputExpressions: Seq[Expression])
  extends InferredExpression(Functions.lineMerge _) {

  protected def withNewChildrenInternal(newChildren: IndexedSeq[Expression]) = {
    copy(inputExpressions = newChildren)
  }
}

case class ST_Azimuth(inputExpressions: Seq[Expression])
  extends InferredExpression(Functions.azimuth _) {

  protected def withNewChildrenInternal(newChildren: IndexedSeq[Expression]) = {
    copy(inputExpressions = newChildren)
  }
}

case class ST_X(inputExpressions: Seq[Expression])
  extends InferredExpression(Functions.x _) {

  protected def withNewChildrenInternal(newChildren: IndexedSeq[Expression]) = {
    copy(inputExpressions = newChildren)
  }
}


case class ST_Y(inputExpressions: Seq[Expression])
  extends InferredExpression(Functions.y _) {

  protected def withNewChildrenInternal(newChildren: IndexedSeq[Expression]) = {
    copy(inputExpressions = newChildren)
  }
}

case class ST_Z(inputExpressions: Seq[Expression])
  extends InferredExpression(Functions.z _) {

  protected def withNewChildrenInternal(newChildren: IndexedSeq[Expression]) = {
    copy(inputExpressions = newChildren)
  }
}

case class ST_StartPoint(inputExpressions: Seq[Expression])
  extends InferredExpression(Functions.startPoint _) {

  protected def withNewChildrenInternal(newChildren: IndexedSeq[Expression]) = {
    copy(inputExpressions = newChildren)
  }
}

case class ST_Snap(inputExpressions: Seq[Expression])
  extends InferredExpression(Functions.snap _) {

  protected def withNewChildrenInternal(newChildren: IndexedSeq[Expression]) = {
    copy(inputExpressions = newChildren)
  }
}

case class ST_Boundary(inputExpressions: Seq[Expression])
  extends InferredExpression(Functions.boundary _) {

  protected def withNewChildrenInternal(newChildren: IndexedSeq[Expression]) = {
    copy(inputExpressions = newChildren)
  }
}


case class ST_MinimumBoundingRadius(inputExpressions: Seq[Expression])
  extends Expression with FoldableExpression with CodegenFallback {

  override def nullable: Boolean = true

  private val geometryFactory = new GeometryFactory()

  override def eval(input: InternalRow): Any = {
    val expr = inputExpressions(0)
    val geometry = expr match {
      case s: SerdeAware => s.evalWithoutSerialization(input)
      case _ => expr.toGeometry(input)
    }

    geometry match {
      case geometry: Geometry => getMinimumBoundingRadius(geometry)
      case _ => null
    }
  }

  private def getMinimumBoundingRadius(geom: Geometry): InternalRow = {
    val minimumBoundingCircle = new MinimumBoundingCircle(geom)
    val centerPoint = geometryFactory.createPoint(minimumBoundingCircle.getCentre)
    InternalRow(centerPoint.toGenericArrayData, minimumBoundingCircle.getRadius)
  }

  override def dataType: DataType = DataTypes.createStructType(
    Array(
      DataTypes.createStructField("center", GeometryUDT, false),
      DataTypes.createStructField("radius", DataTypes.DoubleType, false)
    )
  )

  override def children: Seq[Expression] = inputExpressions

  protected def withNewChildrenInternal(newChildren: IndexedSeq[Expression]) = {
    copy(inputExpressions = newChildren)
  }
}


case class ST_MinimumBoundingCircle(inputExpressions: Seq[Expression])
  extends InferredExpression(Functions.minimumBoundingCircle _) {

  protected def withNewChildrenInternal(newChildren: IndexedSeq[Expression]) = {
    copy(inputExpressions = newChildren)
  }
}


/**
  * Return a linestring being a substring of the input one starting and ending at the given fractions of total 2d length.
  * Second and third arguments are Double values between 0 and 1. This only works with LINESTRINGs.
  *
  * @param inputExpressions
  */
case class ST_LineSubstring(inputExpressions: Seq[Expression])
  extends InferredExpression(Functions.lineSubString _) {

  protected def withNewChildrenInternal(newChildren: IndexedSeq[Expression]) = {
    copy(inputExpressions = newChildren)
  }
}

/**
  * Returns a point interpolated along a line. First argument must be a LINESTRING.
  * Second argument is a Double between 0 and 1 representing fraction of
  * total linestring length the point has to be located.
  *
  * @param inputExpressions
  */
case class ST_LineInterpolatePoint(inputExpressions: Seq[Expression])
  extends InferredExpression(Functions.lineInterpolatePoint _) {

  protected def withNewChildrenInternal(newChildren: IndexedSeq[Expression]) = {
    copy(inputExpressions = newChildren)
  }
}

/**
 * Returns a float between 0 and 1 representing the location of the closest point on a LineString to the given Point,
 * as a fraction of 2d line length.
 *
 * @param inputExpressions
 */
case class ST_LineLocatePoint(inputExpressions: Seq[Expression])
  extends InferredExpression(Functions.lineLocatePoint _) {

  protected def withNewChildrenInternal(newChildren: IndexedSeq[Expression]) = {
    copy(inputExpressions = newChildren)
  }
}

case class ST_EndPoint(inputExpressions: Seq[Expression])
  extends InferredExpression(Functions.endPoint _) {

  protected def withNewChildrenInternal(newChildren: IndexedSeq[Expression]) = {
    copy(inputExpressions = newChildren)
  }
}

case class ST_ExteriorRing(inputExpressions: Seq[Expression])
  extends InferredExpression(Functions.exteriorRing _) {

  protected def withNewChildrenInternal(newChildren: IndexedSeq[Expression]) = {
    copy(inputExpressions = newChildren)
  }
}


case class ST_GeometryN(inputExpressions: Seq[Expression])
  extends InferredExpression(Functions.geometryN _) {

  protected def withNewChildrenInternal(newChildren: IndexedSeq[Expression]) = {
    copy(inputExpressions = newChildren)
  }
}

case class ST_InteriorRingN(inputExpressions: Seq[Expression])
  extends InferredExpression(Functions.interiorRingN _) {

  protected def withNewChildrenInternal(newChildren: IndexedSeq[Expression]) = {
    copy(inputExpressions = newChildren)
  }
}

case class ST_Dump(inputExpressions: Seq[Expression])
  extends InferredExpression(Functions.dump _) {

  protected def withNewChildrenInternal(newChildren: IndexedSeq[Expression]) = {
    copy(inputExpressions = newChildren)
  }
}

case class ST_DumpPoints(inputExpressions: Seq[Expression])
  extends InferredExpression(Functions.dumpPoints _) {

  protected def withNewChildrenInternal(newChildren: IndexedSeq[Expression]) = {
    copy(inputExpressions = newChildren)
  }
}


case class ST_IsClosed(inputExpressions: Seq[Expression])
  extends InferredExpression(Functions.isClosed _) {

  protected def withNewChildrenInternal(newChildren: IndexedSeq[Expression]) = {
    copy(inputExpressions = newChildren)
  }
}

case class ST_NumInteriorRings(inputExpressions: Seq[Expression])
  extends InferredExpression(Functions.numInteriorRings _) {

  protected def withNewChildrenInternal(newChildren: IndexedSeq[Expression]) = {
    copy(inputExpressions = newChildren)
  }
}

case class ST_AddPoint(inputExpressions: Seq[Expression])
  extends InferredExpression(inferrableFunction3(Functions.addPoint)) {

  protected def withNewChildrenInternal(newChildren: IndexedSeq[Expression]) = {
    copy(inputExpressions = newChildren)
  }
}

case class ST_RemovePoint(inputExpressions: Seq[Expression])
  extends InferredExpression(inferrableFunction2(Functions.removePoint)) {

  protected def withNewChildrenInternal(newChildren: IndexedSeq[Expression]) = {
    copy(inputExpressions = newChildren)
  }
}

case class ST_SetPoint(inputExpressions: Seq[Expression])
  extends InferredExpression(Functions.setPoint _) {

  protected def withNewChildrenInternal(newChildren: IndexedSeq[Expression]) = {
    copy(inputExpressions = newChildren)
  }
}

case class ST_ClosestPoint(inputExpressions: Seq[Expression])
  extends InferredExpression(Functions.closestPoint _) {

  protected def withNewChildrenInternal(newChildren: IndexedSeq[Expression]) = {
    copy(inputExpressions = newChildren)
  }
}

case class ST_IsPolygonCW(inputExpressions: Seq[Expression])
  extends InferredExpression(Functions.isPolygonCW _) {
  protected def withNewChildrenInternal(newChildren: IndexedSeq[Expression]) = {
    copy(inputExpressions = newChildren)
  }
}

case class ST_IsRing(inputExpressions: Seq[Expression])
  extends InferredExpression(ST_IsRing.isRing _) {

  protected def withNewChildrenInternal(newChildren: IndexedSeq[Expression]) = {
    copy(inputExpressions = newChildren)
  }
}

object ST_IsRing {
  def isRing(geom: Geometry): Option[Boolean] = {
    geom match {
      case _: LineString => Some(Functions.isRing(geom))
      case _ => None
    }
  }
}

/**
  * Returns the number of Geometries. If geometry is a GEOMETRYCOLLECTION (or MULTI*) return the number of geometries,
  * for single geometries will return 1
  *
  * This method implements the SQL/MM specification. SQL-MM 3: 9.1.4
  *
  * @param inputExpressions Geometry
  */
case class ST_NumGeometries(inputExpressions: Seq[Expression])
  extends InferredExpression(Functions.numGeometries _) {

  protected def withNewChildrenInternal(newChildren: IndexedSeq[Expression]) = {
    copy(inputExpressions = newChildren)
  }
}

/**
  * Returns a version of the given geometry with X and Y axis flipped.
  *
  * @param inputExpressions Geometry
  */
case class ST_FlipCoordinates(inputExpressions: Seq[Expression])
  extends InferredExpression(Functions.flipCoordinates _) {

  protected def withNewChildrenInternal(newChildren: IndexedSeq[Expression]) = {
    copy(inputExpressions = newChildren)
  }
}

case class ST_SubDivide(inputExpressions: Seq[Expression])
  extends InferredExpression(Functions.subDivide _) {

  protected def withNewChildrenInternal(newChildren: IndexedSeq[Expression]) = {
    copy(inputExpressions = newChildren)
  }
}

case class ST_SubDivideExplode(children: Seq[Expression])
  extends Generator with CodegenFallback {
  children.validateLength(2)

  override def eval(input: InternalRow): TraversableOnce[InternalRow] = {
    val geometryRaw = children.head
    val maxVerticesRaw = children(1)
    geometryRaw.toGeometry(input) match {
      case geom: Geometry => ArrayData.toArrayData(
        Functions.subDivide(geom, maxVerticesRaw.toInt(input)).map(_.toGenericArrayData)
      )
        Functions.subDivide(geom, maxVerticesRaw.toInt(input)).map(_.toGenericArrayData).map(InternalRow(_))
      case _ => new Array[InternalRow](0)
    }
  }
  override def elementSchema: StructType = {
    new StructType()
      .add("geom", GeometryUDT, true)
  }

  protected def withNewChildrenInternal(newChildren: IndexedSeq[Expression]) = {
    copy(children = newChildren)
  }
}

case class ST_MakeLine(inputExpressions: Seq[Expression])
  extends InferredExpression(inferrableFunction2(Functions.makeLine), inferrableFunction1(Functions.makeLine)) {

  protected def withNewChildrenInternal(newChildren: IndexedSeq[Expression]) = {
    copy(inputExpressions = newChildren)
  }
}

case class ST_Polygon(inputExpressions: Seq[Expression])
  extends InferredExpression(Functions.makepolygonWithSRID _) {

  protected def withNewChildrenInternal(newChildren: IndexedSeq[Expression]) = {
    copy(inputExpressions = newChildren)
  }
}

case class ST_Polygonize(inputExpressions: Seq[Expression])
  extends InferredExpression(Functions.polygonize _) {

  protected def withNewChildrenInternal(newChildren: IndexedSeq[Expression]) = {
    copy(inputExpressions = newChildren)
  }
}

case class ST_MakePolygon(inputExpressions: Seq[Expression])
  extends InferredExpression(InferrableFunction.allowRightNull(Functions.makePolygon)) {

  protected def withNewChildrenInternal(newChildren: IndexedSeq[Expression]) = {
    copy(inputExpressions = newChildren)
  }
}

case class ST_GeoHash(inputExpressions: Seq[Expression])
  extends InferredExpression(Functions.geohash _) {

  protected def withNewChildrenInternal(newChildren: IndexedSeq[Expression]) = {
    copy(inputExpressions = newChildren)
  }
}

/**
  * Return the difference between geometry A and B
  *
  * @param inputExpressions
  */
case class ST_Difference(inputExpressions: Seq[Expression])
  extends InferredExpression(Functions.difference _) {

  protected def withNewChildrenInternal(newChildren: IndexedSeq[Expression]) = {
    copy(inputExpressions = newChildren)
  }
}

/**
  * Return the symmetrical difference between geometry A and B
  *
  * @param inputExpressions
  */
case class ST_SymDifference(inputExpressions: Seq[Expression])
  extends InferredExpression(Functions.symDifference _) {

  protected def withNewChildrenInternal(newChildren: IndexedSeq[Expression]) = {
    copy(inputExpressions = newChildren)
  }
}

/**
  * Return the union of geometry A and B
  *
  * @param inputExpressions
  */
case class ST_Union(inputExpressions: Seq[Expression])
  extends InferredExpression(inferrableFunction2(Functions.union), inferrableFunction1(Functions.union)) {

  protected def withNewChildrenInternal(newChildren: IndexedSeq[Expression]) = {
    copy(inputExpressions = newChildren)
  }
}

case class ST_Multi(inputExpressions: Seq[Expression])
  extends InferredExpression(Functions.createMultiGeometryFromOneElement _) {

  protected def withNewChildrenInternal(newChildren: IndexedSeq[Expression]) = {
    copy(inputExpressions = newChildren)
  }
}

/**
  * Returns a POINT guaranteed to lie on the surface.
  *
  * @param inputExpressions Geometry
  */
case class ST_PointOnSurface(inputExpressions: Seq[Expression])
  extends InferredExpression(Functions.pointOnSurface _) {

  protected def withNewChildrenInternal(newChildren: IndexedSeq[Expression]) = {
    copy(inputExpressions = newChildren)
  }
}

/**
  * Returns the geometry with vertex order reversed
  *
  * @param inputExpressions
  */
case class ST_Reverse(inputExpressions: Seq[Expression])
  extends InferredExpression(Functions.reverse _) {

  protected def withNewChildrenInternal(newChildren: IndexedSeq[Expression]) = {
    copy(inputExpressions = newChildren)
  }
}

/**
  * Returns the nth point in the geometry, provided it is a linestring
  *
  * @param inputExpressions sequence of 2 input arguments, a geometry and a value 'n'
  */
case class ST_PointN(inputExpressions: Seq[Expression])
  extends InferredExpression(Functions.pointN _) {

  protected def withNewChildrenInternal(newChildren: IndexedSeq[Expression]) = {
    copy(inputExpressions = newChildren)
  }
}

/*
* Forces the geometries into a "2-dimensional mode" so that all output representations will only have the X and Y coordinates.
*
* @param inputExpressions
*/
case class ST_Force_2D(inputExpressions: Seq[Expression])
  extends InferredExpression(Functions.force2D _) {

  protected def withNewChildrenInternal(newChildren: IndexedSeq[Expression]) = {
    copy(inputExpressions = newChildren)
  }
}

/**
  * Returns the geometry in EWKT format
  *
  * @param inputExpressions
  */
case class ST_AsEWKT(inputExpressions: Seq[Expression])
  extends InferredExpression(Functions.asEWKT _) {

  protected def withNewChildrenInternal(newChildren: IndexedSeq[Expression]) = {
    copy(inputExpressions = newChildren)
  }
}

case class ST_AsGML(inputExpressions: Seq[Expression])
  extends InferredExpression(Functions.asGML _) {

  protected def withNewChildrenInternal(newChildren: IndexedSeq[Expression]) = {
    copy(inputExpressions = newChildren)
  }
}

case class ST_AsKML(inputExpressions: Seq[Expression])
  extends InferredExpression(Functions.asKML _) {

  protected def withNewChildrenInternal(newChildren: IndexedSeq[Expression]) = {
    copy(inputExpressions = newChildren)
  }
}

/**
  * Test if Geometry is empty geometry.
  *
  * @param inputExpressions
  */
case class ST_IsEmpty(inputExpressions: Seq[Expression])
  extends InferredExpression(Functions.isEmpty _) {

  protected def withNewChildrenInternal(newChildren: IndexedSeq[Expression]) = {
    copy(inputExpressions = newChildren)
  }
}

/**
  * Test if returning Max X coordinate value.
  *
  * @param inputExpressions
  */
case class ST_XMax(inputExpressions: Seq[Expression])
  extends InferredExpression(Functions.xMax _) {

  protected def withNewChildrenInternal(newChildren: IndexedSeq[Expression]) = {
    copy(inputExpressions = newChildren)
  }
}

/**
  * Test if returning Min X coordinate value.
  *
  * @param inputExpressions
  */
case class ST_XMin(inputExpressions: Seq[Expression])
  extends InferredExpression(Functions.xMin _) {

  protected def withNewChildrenInternal(newChildren: IndexedSeq[Expression]) = {
    copy(inputExpressions = newChildren)
  }
}


/**
  * Returns the areal geometry formed by the constituent linework of the input geometry assuming all inner geometries represent holes
  *
  * @param inputExpressions
  */
case class ST_BuildArea(inputExpressions: Seq[Expression])
  extends InferredExpression(Functions.buildArea _) {

  protected def withNewChildrenInternal(newChildren: IndexedSeq[Expression]): Expression = {
    copy(inputExpressions = newChildren)
  }
}

/**
  * Returns the input geometry in its normalized form.
  *
  * @param inputExpressions
  */
case class ST_Normalize(inputExpressions: Seq[Expression])
  extends InferredExpression(Functions.normalize _) {

  protected def withNewChildrenInternal(newChildren: IndexedSeq[Expression]): Expression = {
    copy(inputExpressions = newChildren)
  }
}

/**
  * Returns the LineString geometry given a MultiPoint geometry
  *
  * @param inputExpressions
  */
case class ST_LineFromMultiPoint(inputExpressions: Seq[Expression])
  extends InferredExpression(Functions.lineFromMultiPoint _) {

  protected def withNewChildrenInternal(newChildren: IndexedSeq[Expression]) = {
    copy(inputExpressions = newChildren)
  }
}

/**
  * Returns a multi-geometry that is the result of splitting the input geometry by the blade geometry
  *
  * @param inputExpressions
  */
case class ST_Split(inputExpressions: Seq[Expression])
  extends InferredExpression(Functions.split _) {

  protected def withNewChildrenInternal(newChildren: IndexedSeq[Expression]) = {
    copy(inputExpressions = newChildren)
  }
}

case class ST_S2CellIDs(inputExpressions: Seq[Expression])
  extends InferredExpression(Functions.s2CellIDs _) {

  protected def withNewChildrenInternal(newChildren: IndexedSeq[Expression]) = {
    copy(inputExpressions = newChildren)
  }
}

case class ST_S2ToGeom(inputExpressions: Seq[Expression])
  extends InferredExpression(Functions.s2ToGeom _) {
  protected def withNewChildrenInternal(newChildren: IndexedSeq[Expression]) = {
    copy(inputExpressions = newChildren)
  }
}

case class ST_H3CellIDs(inputExpressions: Seq[Expression])
  extends InferredExpression(Functions.h3CellIDs _) {

  protected def withNewChildrenInternal(newChildren: IndexedSeq[Expression]) = {
    copy(inputExpressions = newChildren)
  }
}

case class ST_H3CellDistance(inputExpressions: Seq[Expression])
  extends InferredExpression(Functions.h3CellDistance _) {

  protected def withNewChildrenInternal(newChildren: IndexedSeq[Expression]) = {
    copy(inputExpressions = newChildren)
  }
}

case class ST_H3KRing(inputExpressions: Seq[Expression])
  extends InferredExpression(Functions.h3KRing _) {

  protected def withNewChildrenInternal(newChildren: IndexedSeq[Expression]) = {
    copy(inputExpressions = newChildren)
  }
}

case class ST_H3ToGeom(inputExpressions: Seq[Expression])
  extends InferredExpression(Functions.h3ToGeom _) with FoldableExpression {

  protected def withNewChildrenInternal(newChildren: IndexedSeq[Expression]) = {
    copy(inputExpressions = newChildren)
  }
}

case class ST_CollectionExtract(inputExpressions: Seq[Expression])
  extends InferredExpression(InferrableFunction.allowRightNull(Functions.collectionExtract)) {

  protected def withNewChildrenInternal(newChildren: IndexedSeq[Expression]) = {
    copy(inputExpressions = newChildren)
  }
}


/**
  * Returns a POINT Computes the approximate geometric median of a MultiPoint geometry using the Weiszfeld algorithm.
  * The geometric median provides a centrality measure that is less sensitive to outlier points than the centroid.
  *
  * @param inputExpressions Geometry
  */
case class ST_GeometricMedian(inputExpressions: Seq[Expression])
  extends InferredExpression(inferrableFunction4(Functions.geometricMedian)) with FoldableExpression {

  protected def withNewChildrenInternal(newChildren: IndexedSeq[Expression]) = {
    copy(inputExpressions = newChildren)
  }
}

case class ST_DistanceSphere(inputExpressions: Seq[Expression])
  extends InferredExpression(inferrableFunction2(Haversine.distance), inferrableFunction3(Haversine.distance)) {

  protected def withNewChildrenInternal(newChildren: IndexedSeq[Expression]) = {
    copy(inputExpressions = newChildren)
  }
}

case class ST_DistanceSpheroid(inputExpressions: Seq[Expression])
  extends InferredExpression(Spheroid.distance _) {

  protected def withNewChildrenInternal(newChildren: IndexedSeq[Expression]) = {
    copy(inputExpressions = newChildren)
  }
}

case class ST_AreaSpheroid(inputExpressions: Seq[Expression])
  extends InferredExpression(Spheroid.area _) {

  protected def withNewChildrenInternal(newChildren: IndexedSeq[Expression]) = {
    copy(inputExpressions = newChildren)
  }
}

case class ST_LengthSpheroid(inputExpressions: Seq[Expression])
  extends InferredExpression(Spheroid.length _) {

  protected def withNewChildrenInternal(newChildren: IndexedSeq[Expression]) = {
    copy(inputExpressions = newChildren)
  }
}

case class ST_NumPoints(inputExpressions: Seq[Expression])
  extends InferredExpression(Functions.numPoints _) {
  protected def withNewChildrenInternal(newChildren: IndexedSeq[Expression]) = {
    copy(inputExpressions = newChildren)
  }
}

case class ST_Force3D(inputExpressions: Seq[Expression])
  extends InferredExpression(inferrableFunction2(Functions.force3D)) {

  protected def withNewChildrenInternal(newChildren: IndexedSeq[Expression]) = {
    copy(inputExpressions = newChildren)
  }
}

case class ST_ForcePolygonCW(inputExpressions: Seq[Expression])
  extends InferredExpression(Functions.forcePolygonCW _) {
  protected def withNewChildrenInternal(newChildren: IndexedSeq[Expression]) = {
    copy(inputExpressions = newChildren)
  }
}

case class ST_NRings(inputExpressions: Seq[Expression])
  extends InferredExpression(Functions.nRings _) {
  protected def withNewChildrenInternal(newChildren: IndexedSeq[Expression]) = {
    copy(inputExpressions = newChildren)
  }
}

case class ST_IsPolygonCCW(inputExpressions: Seq[Expression]) extends InferredExpression(Functions.isPolygonCCW _) {
  protected def withNewChildrenInternal(newChildren: IndexedSeq[Expression]) = {
    copy(inputExpressions = newChildren)
  }
}

case class ST_ForcePolygonCCW(inputExpressions: Seq[Expression])
  extends InferredExpression(Functions.forcePolygonCCW _) {
  protected def withNewChildrenInternal(newChildren: IndexedSeq[Expression]) = {
    copy(inputExpressions = newChildren)
  }
}

case class ST_Translate(inputExpressions: Seq[Expression])
  extends InferredExpression(inferrableFunction4(Functions.translate)) with FoldableExpression {
  protected def withNewChildrenInternal(newChildren: IndexedSeq[Expression]) = {
    copy(inputExpressions = newChildren)
  }
}

case class ST_VoronoiPolygons(inputExpressions: Seq[Expression])
  extends InferredExpression(nullTolerantInferrableFunction3(FunctionsGeoTools.voronoiPolygons)) with FoldableExpression {
  protected def withNewChildrenInternal(newChildren: IndexedSeq[Expression]) = {
    copy(inputExpressions = newChildren)
  }
}

case class ST_FrechetDistance(inputExpressions: Seq[Expression])
  extends InferredExpression(Functions.frechetDistance _) with FoldableExpression {
  protected def withNewChildrenInternal(newChildren: IndexedSeq[Expression]) = {
    copy(inputExpressions = newChildren)
  }
}

case class ST_Affine(inputExpressions: Seq[Expression])
  extends InferredExpression(inferrableFunction13(Functions.affine), inferrableFunction7(Functions.affine)) {
  protected def withNewChildrenInternal(newChildren: IndexedSeq[Expression]) = {
    copy(inputExpressions = newChildren)
  }
}

case class ST_Dimension(inputExpressions: Seq[Expression])
  extends InferredExpression(Functions.dimension _) with FoldableExpression {
  protected def withNewChildrenInternal(newChildren: IndexedSeq[Expression]) = {
    copy(inputExpressions = newChildren)
  }
}

case class ST_BoundingDiagonal(inputExpressions: Seq[Expression])
  extends InferredExpression(Functions.boundingDiagonal _) {
  protected def withNewChildrenInternal(newChildren: IndexedSeq[Expression]) = {
    copy(inputExpressions = newChildren)
  }
}

case class ST_HausdorffDistance(inputExpressions: Seq[Expression])
  extends InferredExpression(inferrableFunction3(Functions.hausdorffDistance), inferrableFunction2(Functions.hausdorffDistance)) {
  protected def withNewChildrenInternal(newChildren: IndexedSeq[Expression]) = {
    copy(inputExpressions = newChildren)
  }
}

case class ST_Angle(inputExpressions: Seq[Expression])
  extends InferredExpression(inferrableFunction4(Functions.angle _), inferrableFunction3(Functions.angle _), inferrableFunction2(Functions.angle _)) with FoldableExpression {
  protected def withNewChildrenInternal(newChildren: IndexedSeq[Expression]) = {
    copy(inputExpressions = newChildren)
  }
}

case class GeometryType(inputExpressions: Seq[Expression])
  extends InferredExpression(Functions.geometryTypeWithMeasured _) with FoldableExpression {
  protected def withNewChildrenInternal(newChildren: IndexedSeq[Expression]) = {
    copy(inputExpressions = newChildren)
  }
}

case class ST_Degrees(inputExpressions: Seq[Expression])
  extends InferredExpression(Functions.degrees _) with FoldableExpression {
  protected def withNewChildrenInternal(newChildren: IndexedSeq[Expression]) = {
    copy(inputExpressions = newChildren)
  }
}

/**
  * Return the number of ddimensions in geometry.
  *
  * @param inputExpressions
  * */
case class ST_CoordDim(inputExpressions: Seq[Expression])
  extends InferredExpression(Functions.nDims _) {
  protected def withNewChildrenInternal(newChildren: IndexedSeq[Expression]): Expression = {
    copy(inputExpressions = newChildren)
  }
}

/**
  * Returns True if geometry is a collection of geometries
  *
  * @param inputExpressions
  */
case class ST_IsCollection(inputExpressions: Seq[Expression])
  extends InferredExpression(Functions.isCollection _) {
  protected def withNewChildrenInternal(newChildren: IndexedSeq[Expression]): Expression = {
    copy(inputExpressions = newChildren)
  }
}

/**
 * Returns a text description of the validity of the geometry considering the specified flags.
 * If flag not specified, it defaults to OGC SFS validity semantics.
 *
 * @param geom  The geometry to validate.
 * @param flag The validation flags.
 * @return A string describing the validity of the geometry.
 */
case class ST_IsValidReason(inputExpressions: Seq[Expression])
  extends InferredExpression(inferrableFunction2(Functions.isValidReason), inferrableFunction1(Functions.isValidReason)) {

  protected def withNewChildrenInternal(newChildren: IndexedSeq[Expression]) = copy(inputExpressions = newChildren)
}
