# Licensed to the Apache Software Foundation (ASF) under one
# or more contributor license agreements.  See the NOTICE file
# distributed with this work for additional information
# regarding copyright ownership.  The ASF licenses this file
# to you under the Apache License, Version 2.0 (the
# "License"); you may not use this file except in compliance
# with the License.  You may obtain a copy of the License at
#
#   http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing,
# software distributed under the License is distributed on an
# "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
# KIND, either express or implied.  See the License for the
# specific language governing permissions and limitations
# under the License.

"""Schedule for injective operators"""

import tvm

import numpy as np


def schedule_injective(outs):
    """Schedule for injective op.

    Parameters
    ----------
    outs: Array of Tensor
        The computation graph description of injective in the format
        of an array of tensors.

    Returns
    -------
    sch: Schedule
        The computation schedule for the op.
    """
    outs = [outs] if isinstance(outs, tvm.te.tensor.Tensor) else outs
    s = tvm.te.create_schedule([x.op for x in outs])
    tvm.te.schedule.AutoInlineInjective(s)
    # Fuse axes and vectorize inner elements
    for x in outs:
        fused = s[x].fuse(*x.op.axis)
        _, inner = s[x].split(fused, factor=128 // np.dtype(x.dtype).itemsize)
        s[x].vectorize(inner)
    return s


def schedule_softmax(outs):
    return schedule_injective(outs)


def schedule_elemwise(outs):
    return schedule_injective(outs)


def schedule_broadcast(outs):
    return schedule_injective(outs)
