# $Id: ResConf.pm,v 1.38 2002/02/07 09:34:30 m-kasahr Exp $
#
# Copyright (c) 2001 Japan Network Information Center.  All rights reserved.
#  
# By using this file, you agree to the terms and conditions set forth bellow.
# 
#                      LICENSE TERMS AND CONDITIONS 
# 
# The following License Terms and Conditions apply, unless a different
# license is obtained from Japan Network Information Center ("JPNIC"),
# a Japanese association, Fuundo Bldg., 1-2 Kanda Ogawamachi, Chiyoda-ku,
# Tokyo, Japan.
# 
# 1. Use, Modification and Redistribution (including distribution of any
#    modified or derived work) in source and/or binary forms is permitted
#    under this License Terms and Conditions.
# 
# 2. Redistribution of source code must retain the copyright notices as they
#    appear in each source code file, this License Terms and Conditions.
# 
# 3. Redistribution in binary form must reproduce the Copyright Notice,
#    this License Terms and Conditions, in the documentation and/or other
#    materials provided with the distribution.  For the purposes of binary
#    distribution the "Copyright Notice" refers to the following language:
#    "Copyright (c) Japan Network Information Center.  All rights reserved."
# 
# 4. Neither the name of JPNIC may be used to endorse or promote products
#    derived from this Software without specific prior written approval of
#    JPNIC.
# 
# 5. Disclaimer/Limitation of Liability: THIS SOFTWARE IS PROVIDED BY JPNIC
#    "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
#    LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A
#    PARTICULAR PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL JPNIC BE LIABLE
#    FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
#    CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
#    SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR
#    BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
#    WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR
#    OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF
#    ADVISED OF THE POSSIBILITY OF SUCH DAMAGES.
# 
# 6. Indemnification by Licensee
#    Any person or entities using and/or redistributing this Software under
#    this License Terms and Conditions shall defend indemnify and hold
#    harmless JPNIC from and against any and all judgements damages,
#    expenses, settlement liabilities, cost and other liabilities of any
#    kind as a result of use and redistribution of this Software or any
#    claim, suite, action, litigation or proceeding by any third party
#    arising out of or relates to this License Terms and Conditions.
# 
# 7. Governing Law, Jurisdiction and Venue
#    This License Terms and Conditions shall be governed by and and
#    construed in accordance with the law of Japan. Any person or entities
#    using and/or redistributing this Software under this License Terms and
#    Conditions hereby agrees and consent to the personal and exclusive
#    jurisdiction and venue of Tokyo District Court of Japan.
#
package MDN::ResConf;

use strict;
use Carp;
use vars qw($VERSION @ISA @EXPORT @EXPORT_OK $AUTOLOAD);

require Exporter;
require DynaLoader;

@ISA = qw(Exporter DynaLoader);
# Items to export into callers namespace by default. Note: do not export
# names by default without a very good reason. Use EXPORT_OK instead.
# Do not simply export all your public functions/methods/constants.
@EXPORT = qw();
$VERSION = '2.3';

bootstrap MDN::ResConf $VERSION;

# Preloaded methods go here.

# Autoload methods go after =cut, and are processed by the autosplit program.

sub local_to_utf8 {
    my ($self, $from) = @_;
    return $self->nameconv('l', $from);
}

sub utf8_to_local {
    my ($self, $from) = @_;
    return $self->nameconv('L', $from);
}

sub delimiter_map {
    my ($self, $from) = @_;
    return $self->nameconv('d', $from);
}

sub local_map {
    my ($self, $from) = @_;
    return $self->nameconv('M', $from);
}

sub map {
    my ($self, $from) = @_;
    return $self->nameconv('m', $from);
}

sub normalize {
    my ($self, $from) = @_;
    return $self->nameconv('n', $from);
}

sub prohibit_check {
    my ($self, $from) = @_;
    return $self->nameconv('p', $from);
}

sub nameprep {
    my ($self, $from) = @_;
    return $self->nameconv('N', $from);
}

sub nameprep_check {
    my ($self, $from) = @_;
    return $self->nameconv('!N', $from);
}

sub unassigned_check {
    my ($self, $from) = @_;
    return $self->nameconv('u', $from);
}

sub utf8_to_idn {
    my ($self, $from) = @_;
    return $self->nameconv('I', $from);
}

sub idn_to_utf8 {
    my ($self, $from) = @_;
    return $self->nameconv('i', $from);
}

sub local_to_idn {
    my ($self, $from) = @_;
    return $self->nameconv('ldMNI', $from);
}

sub idn_to_local {
    my ($self, $from) = @_;
    return $self->nameconv('i!NL', $from);
}

1;
__END__

=head1 NAME

MDN::ResConf - Perl interface to domain name conversion methods of libmdn

=head1 SYNOPSIS

  use MDN::ResConf;
  $res = MDN::ResConf->new();
  $res->load_file('/usr/local/etc/mdn.conf');

  $idn_domain_name = $res->local_to_idn($local_domain_name);
  $local_domain_name = $res->idn_to_local($idn_domain_name);

=head1 DESCRIPTION

C<MDN::ResConf> provides a Perl object interface to domain name
conversion methods of libmdn, the multilingual domain name library
included in the mDNkit.

This module provides low-level conversion interface to the library.

=head1 CONSTRUCTOR

=over 4

=item new

Creates an object.
It returns C<undef> if no enough memory is remained.

=back

=head1 METHODS

=over 4

=item load_file([$filename])

Loads a configuration file.
If C<$filename> is omitted, the default file
(e.g. F</usr/local/etc/mdn.conf>) is loaded.

=item set_local_encoding([$encoding [, @options]])

Sets local encoding.
If no encoding name is specified, the library determines it using locale
information.

The library currently understands two options,
C<'delayedopen'> and C<'rtcheck'>.
C<'delayedopen'> delays opening the internal I<iconv> converter until
the conversion is actually performed.
C<'rtcheck'> enables round trip check, which is to ensure that the
conversion is correctly performed.

The method returns 1 upon success, C<undef> otherwise.

=item set_idn_encoding($encoding [, @options])

Sets IDN encoding name (e.g. C<'AMC-ACE-Z'>).
The meaning of C<@options> is the same as that of C<set_local_encoding>.
This method corresponds with C<idn-encoding> entry in libmdn
configuration file.

It returns 1 upon success, C<undef> otherwise.

=item set_nameprep_version($version)

Sets NAMEPREP version (e.g. C<'nameprep-05'>).
This method corresponds with C<nameprep> entry in libmdn
configuration file.

It returns 1 upon success, C<undef> otherwise.

=item add_delimiter_map(@ucs)

Recoginzes specified UCS code points (e.g. C<0x3002>) as domain name
delimiter.
This method corresponds with C<delimiter-map> entry in libmdn
configuration file.

It returns 1 upon success, C<undef> otherwise.

=item add_local_map($tld, @names)

Add elements in C<@names> (e.g. C<'filemap:/some/where'>) to
a list of local mapping schemes for the top level domain C<$tld>
(e.g. C<'jp'>).
This method corresponds with C<local-map> entry in libmdn
configuration file.

It returns 1 upon success, C<undef> otherwise.

=item add_nameprep_map(@names)

Add elements in C<@names> (e.g. C<'nameprep-05'>) to a list of
NAMEPREP mapping schemes.
This method corresponds with C<nameprep-map> entry in libmdn
configuration file.

It returns 1 upon success, C<undef> otherwise.

=item add_nameprep_normalize(@names)

Add elements in C<@names> (e.g. C<'unicode-form-kc'>) to a list of
NAMEPREP normalization schemes.
This method corresponds with C<nameprep-normalize> entry in libmdn
configuration file.

It returns 1 upon success, C<undef> otherwise.

=item add_nameprep_prohibit(@names)

Add elements in C<@names> (e.g. C<'fileset:/some/where'>) to a list
of NAMEPREP prohibited-character check schemes.
This method corresponds with C<nameprep-prohibit> entry in libmdn
configuration file.

It returns 1 upon success, C<undef> otherwise.

=item add_nameprep_unassigned(@names)

Add elements in C<@names> (e.g. C<'nameprep-05'>) to a list of NAMEPREP
unassigned-codepoint check schemes.
This method corresponds with C<nameprep-unassigned> entry in libmdn
configuration file.

It returns 1 upon success, C<undef> otherwise.

=item unset_local_encoding()

Unsets local encoding name, set by C<set_local_encoding> method.

=item unset_idn_encoding()

Unsets IDN encoding name, set by C<set_idn_encoding> method
or C<idn-encoding> entry in the loaded configuration file.

=item unset_nameprep_version()

Unsets NAMEPREP version, set by C<set_nameprep_version> method
or C<nameprep> entry in the loaded configuration file.

This methods unsets all of the NAMEPREP mapping, normalization,
prohibited-character check and unassigned-codepoint check schemes,
registered in the object.  Therefore,

    $res->unset_nameprep_version();

is equivalent to the following codes:

    $res->unset_nameprep_map();
    $res->unset_nameprep_normalize();
    $res->unset_nameprep_prohibit();
    $res->unset_nameprep_unassigned();

=item unset_delimiter_map()

Unset delimiters, registered by C<add_delimiter_map> method
or C<delimiter-map> entries in the loaded configuration file.

=item unset_local_map()

Unsets local mapping schemes, registered by C<add_local_map> method
or C<local-map> entries in the loaded configuration file.

=item unset_nameprep_map()

Unsets NAMEPREP mapping schemes, registered by C<add_nameprep_map> method
or C<nameprep-map> entries in the loaded configuration file.

=item unset_nameprep_normalize()

Unsets NAMEPREP normalization schemes,
registered by C<add_nameprep_normalize> method
or C<nameprep-normalize> entries in the loaded configuration file.

=item unset_nameprep_prohibit()

Unsets NAMEPREP prohibited-codepoint check schemes,
registered by C<add_nameprep_prohibit> method
or C<nameprep-prohibit> entries in the loaded configuration file.

=item unset_nameprep_unassigned()

Unsets NAMEPREP unassigned-codepoint check schemes,
set by C<add_nameprep_unassigned> method
or C<nameprep-unassigned> entries in the loaded configuration file.

=item nameconv($actions, $from)

Converts a domain name C<$from> according with C<$actions>, and returns
the result.
Similer to C<pack> and C<unpack>, C<$actions> is a sequence of characters
listed below:

    l       convert the local encoding to UTF-8.
    L       convert the UTF-8 to the local encoding.
    d       covnert local delimiters to periods ('.').
    M       perfrom local mapping.
    m       perform the NAMEPREP mapping.
    n       perform NAMEPREP normalization.
    p       check whether the string contains NAMEPREP prohibited
            character.
    N       perform NAMEPREP (equivalent to 'mnp').
    u       check whether the string contains NAMEPREP unassigned
            codepoint.
    I       convert the UTF-8 string to ACE.
    i       convert the ACE string to UTF-8.
    !m      inspect if NAMEPREP mapping has been performed to the
            string.  If hasn't, convert the string to ACE.
    !n      inspect if NAMEPREP normalizaion has been performed
            to the string.  If hasn't, convert the string to ACE.
    !p      search the string for NAMEPREP prohibited character.
            If found, convert the string to ACE.
    !N      inspect if NAMEPREP has been performed to the string.
            If hasn't, convert the string to ACE. (equivalent to
            `!m!n!p').
    !u      search the string for NAMEPREP unassigned codepoint.
            If found, convert the string to ACE.

The characters in C<$actions> are applied left to right.

=item enable($on_off)

Enable or disable mulitilingual domain name support provided by
libmdn.
If C<$on_off> is zero, the support is disabled, otherwise enabled.
If this method has not been called, the C<MDN_DISABLE> environment
variabe is used to determine whether mulitilingual domain name
support is enabled.
The support is disabled if that variable is defined, 

In disabled condition, conversion methods (e.g. C<nameconv()>)
simply return a copy of input string as the result of conversion.

=item local_to_utf8($from)

Is equivalent to C<nameconv('l', $from)>.

=item utf8_to_local($from)

Is equivalent to C<nameconv('L', $from)>.

=item delimiter_map($from)

Is equivalent to C<nameconv('d', $from)>.

=item local_map($from)

Is equivalent to C<nameconv('M', $from)>.

=item map($from)

Is equivalent to C<nameconv('m', $from)>.

=item normalize($from)

Is equivalent to C<nameconv('n', $from)>.

=item prohibit_check($from)

Is equivalent to C<nameconv('p', $from)>.

=item nameprep($from)

Is equivalent to C<nameconv('N', $from)>.

=item nameprep_check($from)

Is equivalent to C<nameconv('!N', $from)>.

=item unassigned_check($from)

Is equivalent to C<nameconv('u', $from)>.

=item utf8_to_idn($from)

Is equivalent to C<nameconv('I', $from)>.

=item idn_to_utf8($from)

Is equivalent to C<nameconv('i', $from)>.

=item local_to_idn($from)

Is equivalent to C<nameconv('ldMNI', $from)>.

=item idn_to_local($from)

Is equivalent to C<nameconv('i!NL', $from)>.

=back

=head1 CLASS METHOD

=over 4

=item lasterror()

Returns the error message string corresponding to the last error occurred
in this module.

=back

=head1 ISSUE OF HANDLING UNICODE CHARACTERS

See the ``ISSUE OF HANDLING UNICODE CHARACTERS'' section in
L<MDN::UTF8>.

=head1 SEE ALSO

L<mdn.conf(5)>, L<MDN::API>, L<MDN::UTF8>

MDN library specification

=cut
