/*
 * Copyright (c) 2002-2005 Sendmail, Inc. and its suppliers.
 *	All rights reserved.
 *
 * By using this file, you agree to the terms and conditions set
 * forth in the LICENSE file which can be found at the top level of
 * the sendmail distribution.
 */

#include "sm/generic.h"
SM_RCSID("@(#)$Id: smar.c,v 1.20 2005/10/21 22:13:28 ca Exp $")

#include "sm/common.h"
#include "sm/error.h"
#include "sm/assert.h"
#include "sm/heap.h"
#include "sm/io.h"
#include "sm/sysexits.h"
#include "smar.h"
#include "log.h"

/*
**  SMAR_LOOP -- SMAR loop
**
**	Parameters:
**		smar_ctx -- SMAR context
**
**	Returns:
**		usual sm_error code
*/

static sm_ret_T
smar_loop(smar_ctx_P smar_ctx)
{
	SM_IS_SMAR_CTX(smar_ctx);
	SM_REQUIRE(smar_ctx->smar_ev_ctx != NULL);

	if (smar_ctx->smar_status != SMAR_ST_START)
		return sm_error_perm(SM_EM_AR, SM_E_STARTFAIL);
	smar_ctx->smar_status = SMAR_ST_OK;
	sm_log_write(smar_ctx->smar_lctx,
		AR_LCAT_INIT, AR_LMOD_CONFIG, SM_LOG_INFO, 10,
		"sev=INFO, func=smar_loop, status=ready_to_serve");

	/* wait for events, schedule tasks */
	return evthr_loop(smar_ctx->smar_ev_ctx);
}

/*
**  MAIN -- SMAR (SendMail Address Resolver)
**
**	Parameters:
**		argc -- number of arguments
**		argv -- vector of arguments
**
**	Returns:
**		exit code
*/

int
main(int argc, char *argv[])
{
	sm_ret_T ret;
	smar_ctx_T smar_ctx;
	char *prg;

	prg = argv[0];
	if (getuid() == 0 || geteuid() == 0)
	{
		sm_io_fprintf(smioerr, SM_DONTRUNASROOT "\n", prg);
		exit(EX_USAGE);
	}

#if SM_HEAP_CHECK
	SmHeapCheck = 1;
#endif

	/* basic initialization */
	ret = smar_init0(&smar_ctx);
	if (sm_is_err(ret))
		goto error;

	/* read config */
	ret = smar_rdcf(&smar_ctx, argc, argv);
	if (sm_is_err(ret))
		goto error;

	/* initialization */
	ret = smar_init1(&smar_ctx);
	if (sm_is_err(ret))
		goto error;

	SMAR_LEV_DPRINTF(4, (SMAR_DEBFP, "smar: uid=%ld, gid=%ld, euid=%ld, egid=%ld\n",
		(long) getuid(), (long) getgid(), (long) geteuid(), (long) getegid()));

	/* startup ar */
	ret = smar_start(&smar_ctx);
	if (sm_is_err(ret))
		goto error;

#if SM_HEAP_CHECK
	if (HEAP_CHECK)
		sm_heap_report(smioerr, 3);
#endif

	/* run main loop */
	ret = smar_loop(&smar_ctx);
	if (sm_is_err(ret))
		goto error;

	ret = smar_stop(&smar_ctx);
#if SM_HEAP_CHECK
	if (HEAP_CHECK)
		sm_heap_report(smioerr, 3);
#endif
	return 0;

  error:
	/* ignore shutdown errors... */
	(void) smar_stop(&smar_ctx);

	/* select an appropriate error here... */
	return sm_error_value(ret);
}
