/* 
   Unix SMB/Netbios implementation.
   Version 0.1
   WINS server routines and daemon - version 3
   Copyright (C) Andrew Tridgell 1994-1996 Luke Leighton 1996
   
   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.
   
   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.
   
   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
   
   Module name: winsserv.c

   Revision History:

   14 jan 96: lkcl@pires.co.uk
   added multiple workgroup domain master support

   04 jul 96: lkcl@pires.co.uk
   module nameserv contains name server management functions
*/

#include "includes.h"

extern int ClientNMB[];

extern int DEBUGLEVEL;

extern struct name_record *namelist;
extern time_t names_last_modified;

extern struct in_addr ipgrp;

/* a name query challenge structure */
struct nmb_query_confirm
{
	struct in_addr ip;
};

static void response_query_confirm(time_t timestamp, struct packet_struct *p,
                                   struct response_record *n);

/*******************************************************************
  queries names occasionally. an over-cautious, non-trusting WINS server!

  this function has been added because nmbd could be restarted. it
  is generally a good idea to check all the names that have been
  reloaded from file.

  XXXX which names to poll and which not can be refined at a later date.
  ******************************************************************/
void query_refresh_names(time_t time_now)
{
    struct name_record *n;

    static time_t lasttime = 0;

    int count = 0;
    int name_refresh_time = NAME_POLL_REFRESH_TIME;
    int max_count = name_refresh_time / NAME_POLL_INTERVAL;
    if (max_count > 10) max_count = 10;

    name_refresh_time = NAME_POLL_INTERVAL * max_count;

    if (!lp_wins_polling()) return; /* polling of registered names allowed */

    if (!lasttime) lasttime = time_now;

    if (lasttime + NAME_POLL_INTERVAL > time_now) return;

    lasttime = time_now;

    for (n = namelist; n; n = n->next)
    {
		int i;

        for (i = 0; i < n->num_ips; i++)
        {
            /* only do registered names but not group names (255.255.255.255) */

            if (n->ip_flgs[i].source != REGISTER &&
                n->ip_flgs[i].source != SELF) continue;

            if (ip_equal(n->ip_flgs[i].ip, ipgrp)) continue;

            if (n->ip_flgs[i].refresh_time &&
			    n->ip_flgs[i].refresh_time < time_now)
            {
              struct nmb_query_confirm *nmb_data;

              nmb_data = (struct nmb_query_confirm*)(malloc(sizeof(*nmb_data)));

              if (nmb_data)
              {
                int fd = ClientNMB[iface_idx(n->ip_flgs[i].ip)];

                DEBUG(3,("Polling name %s\n", namestr(&n->name)));

                putip(&nmb_data->ip, &n->ip_flgs[i].ip);
                netbios_name_query(time_now, fd, response_query_confirm,
                      (void*)nmb_data, &n->name, 
                      False,False,n->ip_flgs[i].ip);
                count++;
              }
              /* this name will be checked on again, if it's not removed */
              n->ip_flgs[i].refresh_time += name_refresh_time;
            }

            if (count >= max_count)
            {
                /* don't do too many of these at once, but do enough to
                   cover everyone in the list */
                return;
            }
        }
    }
}


/****************************************************************************
  response from a name query to sync browse lists or to update our netbios
  entry. states of type NAME_QUERY_CONFIRM 
  ****************************************************************************/
static void response_query_confirm(time_t timestamp, struct packet_struct *p,
                                   struct response_record *n)
{
	struct nmb_packet *nmb = p ? &p->packet.nmb : NULL;
	struct nmb_name *ans_name = (nmb&&nmb->answers)?&nmb->answers->rr_name:NULL;
	char *rdata = (nmb && nmb->answers) ? nmb->answers->rdata : NULL;

	int rcode = nmb ? nmb->header.rcode : 0;

	struct nmb_ip found;
	struct nmb_ip *data = NULL;
	
	struct nmb_query_confirm *nbd = (struct nmb_query_confirm *)n->nmb_data;

	found.source = REGISTER;
	found.death_time = GET_TTL(0);

	if (rcode == 0 && rdata)
	{
		/* copy the netbios flags and the ip address out of the reply data */
		found.nb_flags = rdata[0];
		putip((char*)&found.ip,&rdata[2]);
		
		data = &found;
	}

    DEBUG(4, ("Name query at %s ip %s - ",
          namestr(&n->name), inet_ntoa(n->send_ip)));

    if (!nbd) return;

    if (ans_name && !name_equal(&n->name, ans_name))
    {
        /* someone gave us the wrong name as a reply. oops. */
        DEBUG(4,("unexpected name received: %s\n", namestr(ans_name)));
        return;
    }

    if (data)
    {
        if (!ip_equal(nbd->ip, data->ip))
        {
            /* someone gave us the wrong ip as a reply. oops. */
            DEBUG(4,("expected ip: %s\n", inet_ntoa(nbd->ip)));
            DEBUG(4,("unexpected ip: %s\n", inet_ntoa(data->ip)));
            return;
        }

        DEBUG(4, (" OK: %s\n", inet_ntoa(data->ip)));

        /* update our netbios name list */
        add_netbios_entry(timestamp, &namelist,&names_last_modified,
                                ans_name,
                                data,True,False);
    }
    else
    {
        DEBUG(4, (" NEGATIVE RESPONSE!\n"));

        if (n->num_msgs == 0)
        {
            /* oops. name query had no response. */

            /* remove the name that had been registered with us,
               and we're now getting no response when challenging.
               see rfc1001.txt 15.5.2
             */

            remove_netbios_name(timestamp, &namelist,
			                   &names_last_modified,
			                   &n->name, nbd->ip, True);
        }
    }
}


/*******************************************************************
  expires or refreshes old names in the namelist
  ******************************************************************/
void check_expire_names(time_t t)
{
    struct name_record *n;
    struct name_record *next;

    /* expire old names */
	for (n = namelist; n; n = next)
	{
		int i;
		next = n->next;
						  
		for (i = 0; n && i < n->num_ips; i++)
		{
			if (n->ip_flgs[i].source != EXPIRED &&
                n->ip_flgs[i].death_time &&
                n->ip_flgs[i].death_time < t)
			{
				n = remove_name(t, &namelist, &names_last_modified,
				                n, n->ip_flgs[i].ip, True);
				i--; /* just removed one: do same index on next iteration */
			}
		}
	}
}

