/* server.c -- pam_smb IPC server code 
   
   This code is (c) Dave Airlie 1998 
   and is released under the GNU GPL
   as provided with this software
   */

#include <stdio.h>
#include <syslog.h>
#include <stdlib.h>
#include <signal.h>
#include <errno.h>

#include "config.h"
#include "pam_smb_queue.h"
#include "cache.h"

#include <fcntl.h>
#include <time.h>
#include <sys/time.h>

#ifdef USE_DB2
#include <db_185.h>
#else
#include <db.h>
#endif

void sighup_handler(int signum);
void sigusr2_handler(int signum);
void sigterm_handler(int signum);

int mid;
int debug=1;

DB *cache_db;
int cur_cache_size=0;

domain_list smb_domlist;

int main(void)
{
  int key;
  struct sigaction sighup_act, sigusr2_act, sigterm_act, sigalrm_act;
  struct itimerval cachetimer;
  int fileDesc=0;

  switch(fork())
    {
    case 0:
      break;
    case -1:
      printf("error in fork\n");
      exit(-1);
    default:
      exit(0);
    }

  close(0);
  close(1);
  close(2);
  setsid();
  
  switch(fork())
    {
    case 0:
      break;
    case -1:
      perror("fork()");
      exit(-1);
    default:
      exit(0);
    }

  chdir("/");
  umask(0);

  fileDesc = open("/dev/null", O_RDWR);
  (void) dup(fileDesc);
  (void) dup(fileDesc);

  openlog("pamsmbd", LOG_PID, LOG_AUTHPRIV);

  cachetimer.it_interval.tv_sec=60;
  cachetimer.it_interval.tv_usec=0;
  cachetimer.it_value.tv_sec=60;
  cachetimer.it_value.tv_usec=0;
  
  setitimer(ITIMER_REAL, &cachetimer, NULL);

  /* set up signal handlers */
  /* SIGHUP re-read configuration file */
  /* SIGUSR2 dumps cache to /tmp */ 
  /* SIGTERM close msgqueue and ends program */

  sigalrm_act.sa_handler=sigalrm_handler;
  sigalrm_act.sa_flags=0;
  sigaction(SIGALRM, &sigalrm_act, NULL);
  sighup_act.sa_handler=sighup_handler;
  sighup_act.sa_flags=0;
  sigaction(SIGHUP, &sighup_act, NULL);
  sigusr2_act.sa_handler=sigusr2_handler;
  sigusr2_act.sa_flags=0;
  sigaction(SIGUSR2, &sigusr2_act, NULL);
  sigterm_act.sa_handler=sigterm_handler;
  sigterm_act.sa_flags=0;
  sigaction(SIGTERM, &sigterm_act, NULL);
  sigaction(SIGINT, &sigterm_act, NULL);
  sigaction(SIGQUIT, &sigterm_act, NULL);

  /* Read configuration */
  smb_readpamconf(&smb_domlist);

  serv_initcachedb();

  /* Setup the SYSV IPC msg queue */
  mid=setup_queue();

  /* Start reading the queue */
  do_queue(mid);

  closelog();
}

/****************************************************************************
 Set up the SYSV message queue
****************************************************************************/
int setup_queue(void)
{
  int mid;
  int key=PAMSMBKEY;
  
  /* if queue exists remove it and re-create it */
  if((mid=msgget(key, 0)) != -1)
    {
      msgctl(mid, IPC_RMID, NULL);
    }
#ifdef PAM_SMB_ROOT_ONLY  
  mid=msgget(key, IPC_CREAT | 0600);
#else
  mid=msgget(key, IPC_CREAT | 0666);
#endif
  if (mid==-1)
    {
      perror("msgget: ");
      exit(-1);
    }

  return mid;
}

/****************************************************************************
 Process a message in the SYSV message queue and return a reply
****************************************************************************/
/* function to get queue entries and validate them */  
int do_queue(int mid)
{
  int n;
  int ok=0;
  MESSAGE msg;
  int user_num;

  int cache_status;
  int valid=0;
  int position;
  
  usercache_data internal_entry, *newentry=&internal_entry;

  while(1)
    {
      ok=0;
      do {
	n=msgrcv(mid, (struct msgbuf *)&msg, sizeof(msg), SERVER, 0);
	if (n==-1)
	  {
	    if (errno!=EINTR)
	      {		
		perror("Server: msgrcv");
		msgctl(mid ,IPC_RMID, NULL);
		exit(-1);
	      }
	  }
	else
		ok=1;
      } while (ok==0);

      memset(newentry,0, sizeof(usercache_data));

      strncpy(newentry->nt_user, msg.username, 21);
      strncpy(newentry->ux_user, msg.username, 21);
      strncpy(newentry->password, msg.password, 39);
   
      pam_map_user(newentry);

      /* Check cache for user code */
      /* Cache will *always* be checked on UNIX username */

      user_num = cur_cache_size;
      position=0;
      ok=0;
    

      cache_status=check_entry_in_cache(newentry);

      if (cache_status!=CACHE_OK)
	{
	  /* if nothing in the map for primary, backup and domain
	     user defaults from conf file */

	  if ((strlen(newentry->pdc_nbname)==0) || (strlen(newentry->nt_domain)==0))
	    {
	      strcpy(newentry->pdc_nbname, smb_domlist.controllers[0].servers[0].sername);
	      strcpy(newentry->bdc_nbname, smb_domlist.controllers[0].servers[1].sername);
	      strcpy(newentry->nt_domain, smb_domlist.controllers[0].domain);
	    }
	  if (debug) syslog(LOG_AUTHPRIV | LOG_DEBUG, "Reauthenticating user %s %s %s %s %s\n", newentry->ux_user, newentry->nt_user, newentry->nt_domain, newentry->pdc_nbname, newentry->bdc_nbname);
	  /* this is the actual validation routine call here <------------*/

	  valid=Valid_MapUser(newentry);

	  if (valid==0)
	    {
	      add_entry_to_cache(newentry);
	    }
	}
      else
	valid=0;
      
      msg.return_code = valid;

      msg.msg_to=msg.msg_fm;
      msg.msg_fm=SERVER;

      strncpy(msg.username, newentry->ux_user, 16);
      memset(msg.password, 0, 16);
      memset(msg.domain, 0, 16);
      n=msgsnd(mid, (struct msgbuf *)&msg, sizeof(msg), 0);
      if (n==-1) {
	perror("server : msgsnd");
	exit(-1);
      }
    }
}

/****************************************************************************
 Handler for SIGHUP signal, re-reads config file 
****************************************************************************/
void sighup_handler(int signum)
{
  smb_readpamconf(&smb_domlist);
  syslog(LOG_AUTHPRIV | LOG_NOTICE, "Got SIGHUP : reloading conf file\n");
  return;
}

/****************************************************************************
 Handler for SIGUSR2 signal, dumps num users in cache
****************************************************************************/
void sigusr2_handler(int signum)
{
  
  syslog(LOG_AUTHPRIV | LOG_NOTICE, "Got SIGUSR2: number of users in cache is %d",cur_cache_size);
  
  return;
}


/****************************************************************************
 Handler for SIGTERM signal, cleans up before exit
****************************************************************************/
void sigterm_handler(int signum)
{
  /* kill the IPC msg queue */
  int n;
  
  n=msgctl(mid ,IPC_RMID, NULL);
  if (n!=0)
    {
      syslog(LOG_AUTHPRIV | LOG_ERR, "Error killing msg queue");
      closelog();
      exit(-1);
    }
  (cache_db->close)(cache_db);
  closelog();
  exit(0);

}



