/* server.c -- pam_smb IPC server code 
   
   Copyright (c) Dave Airlie 2000
   airlied@samba.org

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

*/

#include <stdio.h>
#include <syslog.h>
#include <stdlib.h>
#include <signal.h>
#include <errno.h>
#include <unistd.h>
#include <string.h>
#include "config.h"
#include "constants.h"
#include "pam_smb_queue.h"
#include "cache.h"

#ifdef CRYPT
#include <crypt.h>
#else
#include "md5driver.h"
#endif

#include "tdb.h"

#include <fcntl.h>
#include <time.h>
#include <sys/time.h>

#include "pamsmbd.h"

void sighup_handler(int signum);
void sigusr2_handler(int signum);
void sigterm_handler(int signum);

char map_file[255];

int mid;
int debug=1;  /* default value...should be set in pam.conf */

int cur_cache_size=0;

domain_list smb_domlist;
char *defaultdomain;
int use_map=0;

int main(int argc, char **argv)
{
  int res;
  struct sigaction sighup_act, sigusr2_act, sigterm_act, sigusr1_act;
  struct itimerval;
  
   if (argc>1)
	   	{
				fprintf(stderr,"pam_smb v2.0.0-rc6, usage: pamsmbd");
				fprintf(stderr,"\nContact: airlied@samba.org, pam_smb@csn.ul.ie\n");
				exit(-1);
		}

  openlog("pamsmbd", LOG_PID, LOG_AUTHPRIV);

  /* set up signal handlers */
  /* SIGHUP re-read configuration file */
  /* SIGUSR2 dumps cache to /tmp */ 
  /* SIGTERM close msgqueue and ends program */

  sigusr1_act.sa_handler=sigusr1_handler;
  sigusr1_act.sa_flags=0;
  sigaction(SIGUSR1, &sigusr1_act, NULL);
  sighup_act.sa_handler=sighup_handler;
  sighup_act.sa_flags=0;
  sigaction(SIGHUP, &sighup_act, NULL);
  sigusr2_act.sa_handler=sigusr2_handler;
  sigusr2_act.sa_flags=0;
  sigaction(SIGUSR2, &sigusr2_act, NULL);
  sigterm_act.sa_handler=sigterm_handler;
  sigterm_act.sa_flags=0;
  sigaction(SIGTERM, &sigterm_act, NULL);
  sigaction(SIGINT, &sigterm_act, NULL);
  sigaction(SIGQUIT, &sigterm_act, NULL);

  switch(fork())
	 {
	case 0:
		break;
	case -1:
		printf("error in fork\n");
		exit(-1);
	default:
		exit(0);
	}

  setsid();

  /* cachetimer is now controled via pam.conf
  cachetimer.it_interval.tv_sec=CACHE_CHECK;
  cachetimer.it_interval.tv_usec=0;
  cachetimer.it_value.tv_sec=CACHE_CHECK;
  cachetimer.it_value.tv_usec=0;
  
  setitimer(ITIMER_REAL, &cachetimer, NULL); 
  */
  
  safestrcpy(map_file, DEFAULT_MAP_FILE, 255);

  /* Read configuration */
  smb_readpamconf(&smb_domlist);

  defaultdomain=smb_domlist.controllers[0].domain;

  map_user_initdb();
  res=map_user_read_map_file(map_file);
  if (res==0)
    {
      if (debug)
	syslog(LOG_AUTHPRIV | LOG_DEBUG, "Using map file %s\n", map_file);
      use_map=1;
    }
  serv_initcachedb();

  /* Setup the SYSV IPC msg queue */
  mid=setup_queue();

  /* Start reading the queue */
  do_queue(mid);

  closelog();
  return 0;
}

/*
 * get entry in domain list and fill out servers 
 */
int get_domain_servers(usercache_data *entry)
{
  int x;
  
  for (x=0; x<smb_domlist.numdomains; x++)
    {
      if(strcasecmp(entry->nt_domain, smb_domlist.controllers[x].domain)==0)
	break;
    }

  if (x==smb_domlist.numdomains)
    return -1;
  
  if (smb_domlist.controllers[x].numservers==0)
    return -1;
  if (smb_domlist.controllers[x].numservers>=1)
  {
    safestrcpy(entry->pdc_nbname, smb_domlist.controllers[x].servers[0].sername, MAX_SERV_LEN);
  }
  if (smb_domlist.controllers[x].numservers>=2)
  {
    safestrcpy(entry->bdc_nbname, smb_domlist.controllers[x].servers[1].sername, MAX_SERV_LEN);
  }

  return 0;
}

/*
 * setup_queue()
 *
 * Set up the SYSV message queue 
 */
int setup_queue(void)
{
  int mid;
  int key=PAMSMBKEY;
  
  /* if queue exists remove it and re-create it */
  if((mid=msgget(key, 0)) != -1)
    {
      msgctl(mid, IPC_RMID, NULL);
    }
#ifdef PAM_SMB_ROOT_ONLY  
  mid=msgget(key, IPC_CREAT | 0600);
#else
  mid=msgget(key, IPC_CREAT | 0666);
#endif
  if (mid==-1)
    {
      perror("msgget: ");
      exit(-1);
    }

  return mid;
}

/*
 * do_queue()
 *
 * Process a message in the SYSV message queue and return a reply
 * function to get queue entries and validate them 
 *
 */
int do_queue(int mid)
{
  int n;
  int ok=0, res;
  MESSAGE msg;
  int user_num;

  int cache_status;
  int valid=0;
  int position;
  
  usercache_data internal_entry, *newentry=&internal_entry;

  while(1)
    {
      ok=0;
      do {
	n=msgrcv(mid, (struct msgbuf *)&msg, sizeof(msg), SERVER_ID, 0);
	if (n==-1)
	  {
	    if (errno!=EINTR)
	      {		
		perror("Server: msgrcv");
		msgctl(mid, IPC_RMID, NULL);
		exit(-1);
	      }
	  }
	else
	  ok=1;
      } while (ok==0);

      debug=msg.debug;
      
      /* this resets newentry...all char * in new entry are 1+MAX...
       * therefore, this memset should gaurantee that all strings
       * will be null terminated before hand (not the best but it works)*/
      memset(newentry,0, sizeof(usercache_data));

      strncpy(newentry->nt_user, msg.username, MAX_NT_LEN);
      strncpy(newentry->ux_user, msg.username, MAX_UU_LEN);
      strncpy(newentry->password, msg.password, MAX_PASS_LEN);

      /* copy in the domain from the remote end or default if none passed */
      if (msg.domain[0] != '\0')
	strncpy(newentry->nt_domain, msg.domain, MAX_NTDOM_LEN);
      else
	strncpy(newentry->nt_domain, defaultdomain, MAX_NTDOM_LEN);

      if (use_map)
	map_user(newentry);

      /* Need to dig entry for domain out of conf file list */
      res=get_domain_servers(newentry);

      /* this next "if" is needed for allowing a non-usermapped user
       * to authenticate against *any* of the trusted domains on a
       * PDC / BDC.  Requires userid accross trusted domains to be
       * unique.  This is very common in Large Corporations */
      if (!strncmp(newentry->nt_domain, TRUSTEDDOMAINS ,MAX_NTDOM_LEN))
	memset(newentry->nt_domain, 0, MAX_NTDOM_LEN);

      /* Check cache for user code */
      /* Cache will *always* be checked on UNIX username */
      user_num = cur_cache_size;
      position=0;
      ok=0;
    
      cache_status=check_entry_in_cache(newentry,msg.cachetime,msg.failcachetime);
      if (cache_status!=CACHE_OK)
	{
	  
	  if (debug) 
	    syslog(LOG_AUTHPRIV | LOG_DEBUG, 
		   "server: remote auth user unix:%s nt:%s NTDOM:%s PDC:%s BDC:%s\n", 
		   newentry->ux_user, newentry->nt_user, newentry->nt_domain, 
		   newentry->pdc_nbname, newentry->bdc_nbname);

	  /* this is the actual validation routine call here <------------*/
	  if (res==0)
	    valid=Valid_MapUser(newentry);
	  else 
	    {
	      syslog(LOG_AUTHPRIV | LOG_ERR, "no or incorrect domain record for domain %s in pam_smb configuration", newentry->nt_domain);
	      valid=1;
	    }

          /* See if servers are down and allow failover cache if so */
          if (valid != 0 && valid != NTV_LOGON_ERROR && cache_status == CACHE_OK_FAILOVER)
		valid=0;
	  else if (valid==0)
	    {
#ifdef CRYPT
			strncpy(newentry->password,
				crypt((const char*)newentry->password,(const char*)salt()),13);
			memset(newentry->password + 13, 0, MAX_PASS_LEN - 13);
#else	
			memcpy((void *) newentry->password,
				(void *) md5string((char *)newentry->password), MD5_LENGTH);
			memset(newentry->password + MD5_LENGTH, 0, MAX_PASS_LEN - MD5_LENGTH);
#endif
			if (msg.cachetime != SMB_NO_CACHE || msg.failcachetime != SMB_NO_CACHE)
			  add_entry_to_cache(newentry);
	    }
	}
      else
	valid=0;
      
      msg.return_code = valid;

      msg.msg_to=msg.msg_fm;
      msg.msg_fm=SERVER_ID;

      safestrcpy(msg.username, newentry->ux_user, MAX_UU_LEN);
      memset(msg.password, 0, MAX_PASS_LEN);
      memset(msg.domain, 0, MAX_NTDOM_LEN);
      n=msgsnd(mid, (struct msgbuf *)&msg, sizeof(msg), 0);
      if (n==-1) {
	perror("server : msgsnd");
	exit(-1);
      }
    }
}

/*
 * sighup_handler()
 *
 * Handler for SIGHUP signal, re-reads config file 
 *
 */

void sighup_handler(int signum)
{
  smb_readpamconf(&smb_domlist);
  syslog(LOG_AUTHPRIV | LOG_NOTICE, "Got SIGHUP : reloading conf file\n");
  if(use_map){   
    map_user_closedb();
    if (debug)
      syslog(LOG_AUTHPRIV | LOG_DEBUG, "Closed Map file");
    map_user_initdb(); 
    map_user_read_map_file(map_file);
    if(debug) 
      syslog(LOG_AUTHPRIV | LOG_DEBUG, "Re-opened Map file"); 
  }   
  return;
}

/*
 * sigusr2_handler()
 *
 * Handler for SIGUSR2 signal, dumps num users in cache
 *
 */
void sigusr2_handler(int signum)
{
  
  syslog(LOG_AUTHPRIV | LOG_NOTICE, "Got SIGUSR2: number of users in cache is %d",cur_cache_size);
  
  return;
}


/*
 * sigterm_handler() 
 *
 * Handler for SIGTERM signal, cleans up before exit
 *
 */
void sigterm_handler(int signum)
{
  /* kill the IPC msg queue */
  int n;
  
  n=msgctl(mid ,IPC_RMID, NULL);
  if (n!=0)
    {
      syslog(LOG_AUTHPRIV | LOG_ERR, "Error killing msg queue");
      closelog();
      exit(-1);
    }
  serv_closecachedb();

  if(use_map){   
    map_user_closedb();
    if (debug)
      syslog(LOG_AUTHPRIV | LOG_DEBUG, "Closed Map file");
  }   
    
  closelog();
  exit(0);

}



