/*
 * SOURCE:  main.c
 * PROJECT: EasyTeX
 *
 * PURPOSE: main routine, menu definitions and main user shell
 *
 * HISTORY: 08/15/1991 - major rewrite
 *          10/16/1991 - fixed bug in searchstring()
 *          10/21/1991 - version 1.2
 *                       customize menu
 *          11/16/1991 - memory management improved
 *          11/28/1991 - visible marking added
 *          12/01/1991 - help database
 *          12/05/1991 - options added
 *
 *                   ...
 *
 *          03/11/1992 - version 1.8
 *                       lots of bugs fixed
 *          03/29/1992 - version 1.9
 *                       file requester excluded
 *                       released to public
 *          04/02/1992 - version 2.0
 *                       advanced help database
 *          04/14/1992 - version 2.1
 *                       file compresion for help database
 *                       bug to do with 'critical_error' fixed
 *          04/27/1992 - bug in _msgbox fixed
 *                       (NULLpointer assignment made Delete() hang up)
 *          05/05/1992 - version 2.2
 *                       MARKEND added to textgadget
 *                       bug in Create() fixed
 *                       ('pop ds' came too late, 'errnum' not set properly)
 *          05/13/1992 - version 2.3
 *                       Video-Mode settings rearranged (mainly option /F)
 *          06/13/1992 - version 2.4
 *                       several bugs fixed
 *                       overlay handling
 *          06/25/1992 - bug in ReadInput fixed
 *                       overlay handling removed (conflict with critical_error)
 *          10/15/1992 - version 2.5
 *                       TeX command line added
 *          11/15/1992 - version 3.0
 *                       new: file loader written in assembler.
 *                       Shared memory region
 *          08/21/1993 - version 3.1
 *                       bugs fixed
 *          10/07/1993 - Hey, this is my birthday, but I had a nasty examen today,
 *                       so I had no time the last several weeks, and now I'm
 *                       relaxing a bit. There's not much to do any more, just
 *                       some more options (/a, /i), file arguments allowed...
 *          03/17/1994 - /q removed.
 *                       postscript driver added.
 *
 *
 *
 *** IMPORTANT NOTE: *******************************************************
 **                                                                       **
 **     The LaTeX user shell 'EasyTeX' is shareware. You might give       **
 **  EasyTeX to your friends and everybody you know as long as all        **
 **  executable files remain unchanged.                                   **
 **     You might modify and use any of my functions for your purpose     **
 **  as long as they contain the copyright remark and your executable     **
 **  program contains a way to display credits and those credits contain  **
 **  the copyright remark.                                                **
 **     If you have any suggestions for further releases, please          **
 **  send the source code to me, I'll try to organize upgrades.           **
 **     'EasyTeX' has been developed using MICROSOFT(R) C 6.0, all        **
 **  source code sent in should be compatible.                            **
 **                                                                       **
 ** 'EasyTeX' has been written by:                                        **
 **                                                                       **
 **     Michael Schollmeyer                                               **
 **     Institut fr Thermodynamik                                        **
 **       der Luft- und Raumfahrt                                         **
 **     Pfaffenwaldring 31                                                **
 **                                                                       **
 **     D-7000 Stuttgart 80                                               **
 **                                                                       **
 **     email: sm@itlrsun1.luftfahrt.uni-stuttgart.de (INTERNET)          **
 **                                                                       **
 ***************************************************************************
 *
 * Files you need to build EasyTeX:
 *
 *      makefile   - tells nmake how to do things
 *
 * to build et.ovl:
 *
 *      setting.h  - user defined switches and settings
 *      macros.h   - some macro definitions
 *      main.c     - user shell
 *      main.h     - include file for main.c
 *      applic.c   - application and option routines
 *      applic.h   - include file for applic.c
 *      memory.c   - memory management routines
 *      memory.h   - include file for memory.c
 *      edit.c     - main editing routines
 *      edit.h     - include file for edit.c
 *      profile.c  - stuff to do with ini file handling
 *      profile.h  - include file for profile.c
 *      intui.c    - intuition shell, windows, gadgets, menus...
 *      intui.h    - include file for intui.c
 *      help.c     - help engine
 *      help.h     - include file for help.c
 *      video.c    - routines for direct video access
 *      video.h    - include file for video.c
 *      bios.c     - low level bios routines
 *      bios.h     - include file for bios.c
 *      dos.c      - low level dos routines
 *      dos.h      - include file for dos.c
 *      error.c    - error messages
 *      error.h    - include file for error.c
 *      filereq.c  - file requester
 *      filereq.h  - include file for filereq.c
 *      compress.c - file compression
 *      compress.h - include file for compress.h
 *
 * to build et.com:
 *
 *     version.inc - version and release string made by 'version.exe'
 *     et.asm      - source code for overlay
 *
 * to build rdel.exe:
 *
 *     rdel.c      - main module
 *     rdel.h      - include file for rdel.c
 *
 * to build help database:
 *
 *     makehelp.c  - main module
 *     compress.c  - file compression
 *     compress.h  - include file for compress.h
 *     help.txt    - help source file
 *
 * to build version.exe:
 *
 *     version.c   - source code
 *
 * additional files:
 *
 *     et.ini      - initialization file
 *     readme.txt  - information for users
 *
 *******************************************************************
 *
 * (c)M.Schollmeyer
 */
#include "main.h"

                                /* function prototypes */
void main( int, char ** );
static void setfile( void );
static BOOL addfile( char * );

struct IntuitionBase *IntuitionBase;
struct SharedMemory *SharedMemory;

char *TmpPath, *IniName, *MyPath;
BOOL ini_loaded = FALSE;
BOOL periodic_backup = FALSE;
BOOL ignore_palette = FALSE;
int old_breakflag;

extern struct Editor ed;       // 'ed' is defined in edit.c
extern struct MemoryHeader mh; // 'mh' is defined in memory.h

/*******************************************************************
 *
 * Menu and MenuItem Definitions
 *
 * All Menu and MenuItem structures are nodes of a linked list,
 * there is one list for the Menus and one list for all MenuItems.
 * Since C is not able to process forward references in static
 * variables (that's what the structures are), I enumerate all
 * structures in reverse order (from the bottom to the top) of the
 * menu window.
 *
 *******************************************************************/

static struct MenuItem i_mouse = {
    NULL, "&Mouse Functions...",
    "Display Help on Mouse Functions", HLP_MOUSE
};
static struct MenuItem i_keyass = {
    &i_mouse, "&Key Assignments...",
    "Display Help on Key Assignments", HLP_KEYS
};
static struct MenuItem i_currtopic = {
    &i_keyass, "&Current Topic...",
    "Display Help on current Dialogbox", -1, 0x3b00
};
static struct MenuItem i_help = {
    &i_currtopic, "&Last Topic...",
    "Display last help window", -1, 0x5e00
};
static struct MenuItem i_basic = {
    &i_help, "&Contents...",
    "Display table of contents for Help", -1
};

static struct Menu m_help = {
    NULL, &i_basic, "&Help"
};

static struct MenuItem i_miscv = {
    NULL, "&View Output",
    "View the DOS Screen", -1
};
static struct MenuItem i_miscsec = {
    &i_miscv, NULL, NULL, -1, 0, ITEM_SECTION
};
static struct MenuItem i_mem = {
    &i_miscsec, "&Memory...",
    "Show memory statistics", HLP_MEMORY
};
static struct MenuItem i_info = {
    &i_mem, "&About...",
    "Display credit information", HLP_ABOUT
};

static struct Menu m_misc = {
    &m_help, &i_info, "&Miscellaneous"
};


static struct MenuItem io_custom = {
    NULL, "&Customize Menu...",
    "Edit the Utilities menu", HLP_CUSTOMMENU
};
static struct MenuItem io_sec = {
    &io_custom, NULL, NULL, -1, 0, ITEM_SECTION
};
static struct MenuItem io_pages = {
    &io_sec, "Pa&ges...",
    "Select Pages for Drivers", HLP_PAGES
};
static struct MenuItem io_driver = {
    &io_pages, "&Driver...",
    "Set options for Drivers", HLP_DRIVER
};
static struct MenuItem io_pscript = {
    &io_driver, "P&ostscript...",
    "Set options for Postscript", HLP_PRINTCMD
};
static struct MenuItem io_print = {
    &io_pscript, "&Print...",
    "Set options for Print", HLP_PRINTCMD
};
static struct MenuItem io_prev = {
    &io_print, "Pre&View...",
    "Set options for PreView", HLP_PREVIEWCMD
};
static struct MenuItem io_tex = {
    &io_prev, "&TeX...",
    "Set options for TeX", HLP_LATEXCMD
};

static struct Menu m_opts = {
    &m_misc, &io_tex, "&Options"
};

static struct MenuItem i_settings = {
    NULL, "&Editor Settings...",
    "Configure the Editor", HLP_SETTINGS
};
static struct MenuItem i_pack = {
    &i_settings, "P&ack",
    "Pack the Edit Buffer", HLP_PACK, 0x1910
};
static struct MenuItem i_edsec4 = {
    &i_pack, NULL, NULL, -1, 0, ITEM_SECTION
};
static struct MenuItem i_redo = {
    &i_edsec4, "Red&o",
    "Restore the previously undone line", HLP_UNDO, 0x180f
};
static struct MenuItem i_undo = {
    &i_redo, "&Undo",
    "Restore the previously edited line", HLP_UNDO, 0x0e7f
};
static struct MenuItem i_edsec3 = {
    &i_undo, NULL, NULL, -1, 0, ITEM_SECTION
};
static struct MenuItem i_setma = {
    &i_edsec3, "Set Mar&k",
    "Set the Marker to the current Cursor position", -1, 0x1e00
};
static struct MenuItem i_goma = {
    &i_setma, "Goto &Mark",
    "Set the Cursor to the Marker position", -1, 0x1312
};
static struct MenuItem i_goln = {
    &i_goma, "&Goto Line...",
    "Set the Cursor to a specified line number", HLP_GOTO, 0x2207
};
static struct MenuItem i_edsec2 = {
    &i_goln, NULL, NULL, -1, 0, ITEM_SECTION
};
static struct MenuItem i_edpa = {
    &i_edsec2, "&Paste",
    "Insert the Clipboard at the Cursor", HLP_CLIPBOARD, 0x5200
};
static struct MenuItem i_edcp = {
    &i_edpa, "Cop&y",
    "Copy the marked region to the Clipboard", HLP_CLIPBOARD, 0x4e2b
};
static struct MenuItem i_cut = {
    &i_edcp, "Cu&t",
    "Copy the marked region to the Clipboard and delete in Buffer", HLP_CLIPBOARD, 0x4a2d
};
static struct MenuItem i_edsec1 = {
    &i_cut, NULL, NULL, -1, 0, ITEM_SECTION
};
static struct MenuItem i_cont = {
    &i_edsec1, "&Continue",
    "Repeat the last command of 'Search' and 'Query s&r'", HLP_SEARCH, 0x1000
};
static struct MenuItem i_eds_r = {
    &i_cont, "Query s&&r...",
    "Search a string and replace by another", HLP_SEARCH, 0x1011
};
static struct MenuItem i_edsrc = {
    &i_eds_r, "&Search...",
    "Search a string", HLP_SEARCH, 0x1f13
};
static struct Menu m_edit = {
    &m_opts, &i_edsrc, "&Edit"
};


static struct MenuItem i_utilsec2 = {
    NULL, NULL, NULL, -1, 0, ITEM_SECTION
};
static struct MenuItem i_vlog = {
    &i_utilsec2, "View &Error Log...",
    "View more information on the displayed TeX-error", -1, 0x7100
};
static struct MenuItem i_verr = {
    &i_vlog, "V&iew Errors",
    "View the next TeX-error", -1, 0x7000
};
static struct MenuItem i_utilsec = {
    &i_verr, NULL, NULL, -1, 0, ITEM_SECTION
};
static struct MenuItem i_pscript = {
    &i_utilsec, "P&ostscript...",
    "Convert the current file to Postscript", HLP_PRINT, 0x6f00
};
static struct MenuItem i_print = {
    &i_pscript, "&Print...",
    "Print the current file", HLP_PRINT, 0x6e00
};
static struct MenuItem i_preview = {
    &i_print, "Pre&View",
    "View layout of the current file", -1, 0x6d00
};
static struct MenuItem i_tex = {
    &i_preview, "&TeX",
    "Run TeX over the current file", -1, 0x6c00
};

static struct Menu m_util = {
    &m_edit, &i_tex, "&Utilities"
};

static struct MenuItem i_exit = {
    NULL, "E&xit",
    "Exit EasyTeX", -1, 0x6b00
};
static struct MenuItem i_savex = {
    &i_exit, "Save & Exi&t",
    "Save the current file and exit EasyTeX", -1, 0x5700
};
static struct MenuItem i_dsks2 = {
    &i_savex, NULL, NULL, -1, 0, ITEM_SECTION
};
static struct MenuItem i_doscom = {
    &i_dsks2, "DOS &Command...",
    "Execute a DOS command", HLP_DOSCMD
};
static struct MenuItem i_dos = {
    &i_doscom, "&DOS Shell",
    "Run a DOS Shell from EasyTeX", -1
};
static struct MenuItem i_dsks = {
    &i_dos, NULL, NULL, -1, 0, ITEM_SECTION
};
static struct MenuItem i_merge = {
    &i_dsks, "&Merge...",
    "Insert a file at the current Cursor position", HLP_MERGE
};
static struct MenuItem i_next = {
    &i_merge, "&Next",
    "Load the next file of the file list", HLP_FILESYSTEM, 0x6900
};
static struct MenuItem i_prev = {
    &i_next, "&Previous",
    "Load the previous file of the file list", HLP_FILESYSTEM, 0x5f00
};
static struct MenuItem i_last = {
    &i_prev, "&Last",
    "Load the last previously edited file", HLP_FILESYSTEM, 0x3c00
};
static struct MenuItem i_save = {
    &i_last, "&Save...",
    "Save the current file", HLP_SAVEFILE, 0x5500
};
static struct MenuItem i_visit = {
    &i_save, "&Visit...",
    "Load a file", HLP_VISIT, 0x6a00
};
static struct MenuItem i_add = {
    &i_visit, "&Add...",
    "Load a file and add it to the file list", HLP_ADDFILE, 0x5600
};
static struct MenuItem i_load = {
    &i_add, "S&et...",
    "Load a file of the file list", HLP_SETFILE, 0x3d00
};

static struct Menu m_file = {
    &m_util, &i_load, "&File"
};

BOOL setvmode( int vm ) {

    BOOL ret;
    int i;
    char *cp;

    if( !(VideoBase = InitVideo(vm) ) )
        return FALSE;

    if( VideoBase->ScreenWidth != 80 )
        return FALSE;

    if( ignore_palette == FALSE ) {
        if( cp = GetProfileData( PR_PALETTE, NULL ) )
            VideoBase->Palette = cp;
    }
    /* Take default palette from colors.c if there is no palette */
    if( ignore_palette || VideoBase->Palette == NULL )
        getdefpalette();

    IntuitionBase->InputEvent->mousestatus = ResetMouse();

    ToggleBlink( FALSE );

    ed.EditHeight = VideoBase->ScreenHeight-2;

    if( VideoBase->ScreenHeight != 25 ) {
        SetPointerPage( 0 );
        SetActualPage( 0 );
        SetVisualPage( 0 );

        i_mem.Next = NULL;
    } else {
        SetPointerPage( WORKPAGE );
        SetActualPage( WORKPAGE );
        VideoSetPage( WORKPAGE );
        SetVisualPage( WORKPAGE );

        i_mem.Next = &i_miscsec;
    }

    e_redraw();
    HideMouse();
    e_drawmr();
    ShowMouse();
    e_status( 1 );

    return TRUE;
}


/*
 *    Name: main
 *  Return: void
 * Purpose:     This is the main routine, called by c startup code. Does
 *          some initialization tasks first. I will place that in it's own
 *          routine if it gets more sophisticated.
 *              Then processes the main message loop.
 *          If this loop is exited, main() does some cleanup tasks and
 *          exits.
 *
 * (c)M.Schollmeyer
 */

void main( int argc, char **argv )
{
    int ret = 1;

    static void load_arg_files(void);
    static void remove_obsolete_entries();
    if( maininit( argv, argc ) ) {
        e_loadcboard();            /* load the clipboard       */
        load_arg_files();
        e_setfile(ed.CurrFile); /* try to load current file */
        e_refresh();

        remove_obsolete_entries();
        /* MAIN MESSAGE LOOP */
        while( EventHandler( ReadInput() ) );
        ret = 0;
    }
    die( ret );
}


static void remove_obsolete_entries() {

    /* This is the place to remove obsolete profile entries
       of previous EasyTeX Releases. */
    ClearProfileData( PR_ENVIRONMENT );

}

/* Loop once per entry in the file arg list, and add the files to the file
   list. If all files of the file list has been added, it loads either the
   last file added to the file list or the current file. */
static void load_arg_files() {

    char *cp;
    int filenum;
    char buffer[_MAX_PATH];

    while (cp=GetFromFileArgsList()) {
        strcpy(buffer, cp);     /* This is necessary because IsFileInFileList
                                   will expand the contents of cp. */
        free(cp);
        /* Add file only if it is not already in file list. */
        filenum = IsFileInFileList(buffer, NULL, TRUE);
        if (filenum) {
            /* File is in file list, make it the current one. */
            ed.CurrFile = filenum-1;
        } else {
            /* File is not in file list, add it if there is a free slot. */
            for( filenum = 0; filenum < MAXFILES; ++filenum ) {
                /* Look if there is an empty slot */
                if( GetProfileString( PR_FILENAME( filenum ), NULL ) == NULL ) {
                    PutProfileString( PR_FILENAME( filenum ), buffer );
                    ed.CurrFile = filenum;
                    ed.MarkX = ed.MarkY = 0;
                    e_goto( 0, 0, 0, 0 );
                    m_savevirtual();
                    e_status( 1 );
                    e_refresh();
                    break;
                }
            }
        }
    }
}


void ShowMenu( void ) {

    DrawMenuTitle( &m_file, MENUWIDTH, 0L );
    e_drawflags();
    _printtime( 1 );
}

void HideMenu( void ) {

    DrawMenuTitle( &m_file, MENUWIDTH, MENU_DISABLE );
}

/*
 *    Name: EventHandler
 *  Return: FALSE if user aborted program, TRUE if not
 * Purpose: This routine handles all user input.
 *
 *
 *
 * (c)M.Schollmeyer
 */
int EventHandler( struct InputEvent *ie ) {

    unsigned long qual;
    int ret = 0, i;

    static int topics[] = { HLP_DEFAULT, -1, -1, HLP_KEYS, HLP_MOUSE };
    static unsigned long clock = 0L;

    static void gotoline( void );
    static void visitfile( void );

    static void mergefile( void );

    static void dos_shell( void );
    static void view_output( void );
    static void settings( void );

    static void checkmenu( struct Menu *, struct MenuItem *, unsigned long );

    if( periodic_backup ) {
        qual = GetClock();
        if( qual > clock ) {
            m_save( SAVE_NORMAL );
            clock = GetClock() + GetProfileLong( PR_PERISAV );
        }
    }

    qual = CheckMenu( &m_file, ie, checkmenu );
    ClearTopicStack();
    if( qual != MENUNULL ) {
        switch( MENUNUM( qual ) ) {
            case 1: switch( ITEMNUM( qual ) ) {
                case  1: HideMenu();
                         setfile();
                         break;
                case  2: HideMenu();
                         addfile(NULL);
                         break;
                case  3: HideMenu();
                         visitfile();
                         break;
                case  4: HideMenu();
                         m_save( SAVE_QUERY );
                         e_savekoords();
                         break;
                case  5: HideMenu();
                         e_last();
                         e_refresh();
                         break;
                case  6: e_savekoords();
                         e_prev();
                         e_refresh();
                         break;
                case  7: e_savekoords();
                         e_next();
                         e_refresh();
                         break;
                case  8: HideMenu();
                         mergefile();
                         break;
                case 10: dos_shell();
                         break;
                case 11: HideMenu();
                         dos_command();
                         break;
                case 13: if( m_save( SAVE_ALWAYS ) ) {
                            e_savekoords();
                            return FALSE;
                         }
                case 14: if( ed.Flags & (MODIFIED|LINECHANGED) ) {
                            qual = DoMessageBox( DMB_YES|DMB_NO, HLP_MODIFIED,
                                  lpszWarning,
                  "Buffer has been modified!\nDo you really want to exit?" );
                            if( qual != DMB_YES )
                                    break;
                         }
                         return FALSE;
                }
                break;
            case 2: switch( ITEMNUM( qual ) ) {
                case 1: exe_tex();      /* tex */
                        break;
                case 2: exe_preview();  /* preview */
                        break;
                case 3: HideMenu();
                        exe_print();      /* print */
                        break;
                case 4: HideMenu();
                        exe_pscript();
                        break;
                case 6: viewerrors( VIEW_NEXT );
                        break;
                case 7: HideMenu();
                        viewerrors( VIEW_TEXT );
                         break;
                default:exe_custom( (int)ITEMNUM(qual) - 9 );
                        break;
                }
                break;
            case 3: switch( ITEMNUM( qual ) ) {
                case 1: searchstring( 0 );
                        break;
                case 2: searchstring( QUERYSEARCH );
                        break;
                case 3: searchstring( CONTSEARCH );
                        break;
                case 5: m_beginundogroup();
                        e_cut( 0, 1 );
                        m_endundogroup();
                        e_status( 0 );
                        e_refresh();
                        break;
                case 6: e_copy( 0, 1 );
                        e_refresh();
                        break;
                case 7: if( ie->shift & IES_SHIFT || !ACCELERATOR(qual) ) {
                            m_beginundogroup();
                            e_paste( 0, 1 );
                            m_endundogroup();
                            e_refresh();
                        } else
                            e_insmode( 0, 0 );
                        break;
                case 9: gotoline();
                        break;
                case 10: e_gomark( 0, 1 );
                         e_status(0);
                         e_refresh();
                         break;
                case 11: e_setmark( 0, 1 );
                         e_status(0);
                         e_refresh();
                         break;
                case 13: m_undo();
                         e_redraw();
                         e_drawmr();
                         e_status( 0 );
                         break;
                case 14: m_redo();
                         e_redraw();
                         e_drawmr();
                         e_status( 0 );
                         break;
                case 16: HideMenu();
                         m_pack( 0, 0 );
                         break;
                case 17: HideMenu();
                         settings();
                         break;
                }
                break;
            case 4: HideMenu();
                    switch( ITEMNUM( qual ) ) {
                case 1: opt_tex();      /* tex */
                        break;
                case 2: opt_preview();  /* preview */
                        break;
                case 3: opt_print();
                        break;
                case 4: opt_pscript();
                        break;
                case 5: opt_driver();      /* driver */
                        break;
                case 6: opt_pages();
                        break;
                case 8: customizemenu();
                        break;
                }
                break;
            case 5: HideMenu();
                    switch( ITEMNUM( qual ) ) {

                case 1: about();
                        break;
                case 2: mem();
                        break;
                case 4: view_output();
                        break;
                }
                break;
            case 6: switch ( ITEMNUM( qual ) ) {
                case 2: lasthelp();
                        break;
                default:help( topics[ITEMNUM( qual )-1] );
                        break;
                }
                break;
        }
        DrawMenuTitle( &m_file, MENUWIDTH, 0L );

    } else if( !edit( ie ) ) {
        if( ie->buttons & DOUBLE_CLICK( MBUTTON_RIGHT ) ) {
            HideMenu();
            setfile();
            DrawMenuTitle( &m_file, MENUWIDTH, 0L );
        }
    }
    return TRUE;
}

static void checkmenu( struct Menu *m, struct MenuItem *mi, unsigned long flags ) {

    struct Clipboard cboard;
    int i, j;

    if( flags & MH_CHECKITEM ) {
        ClearTopicStack();
        if( mi ) {
            PushTopic( mi->HlpIndex );
            if( mi->Message )
                m_putmsg( MSG_STNDRD, mi->Message );
        }
    } else if( flags & MH_OPENMENU ) {
        if( m == &m_edit ) {
            e_getclipboardstruct( &cboard );
            if( cboard.flags & CLIP_EMPTY )
                SETBIT( i_edpa.Flags, ITEM_DISABLE );
            else
                CLEARBIT( i_edpa.Flags, ITEM_DISABLE );
            if( m_isrestoreable() )
                CLEARBIT( i_undo.Flags, ITEM_DISABLE );
            else
                SETBIT( i_undo.Flags, ITEM_DISABLE );
            if( m_isredoable() )
                CLEARBIT( i_redo.Flags, ITEM_DISABLE );
            else
                SETBIT( i_redo.Flags, ITEM_DISABLE );

        } else if( m == &m_file ) {
            for( i = j = 0; i < 2 && j < MAXFILES; ++j )
                if( GetProfileString( PR_FILENAME(j), NULL ) ) ++i;
            if( i == 2 ) {
                CLEARBIT( i_next.Flags, ITEM_DISABLE );
                CLEARBIT( i_prev.Flags, ITEM_DISABLE );
                CLEARBIT( i_last.Flags, ITEM_DISABLE );

            } else {
                SETBIT( i_next.Flags, ITEM_DISABLE );
                SETBIT( i_prev.Flags, ITEM_DISABLE );
                SETBIT( i_last.Flags, ITEM_DISABLE );
            }
        } else if( m == &m_util ) {
            if( ed.Flags & PSEUDOFILE ) {
                SETBIT( i_tex.Flags, ITEM_DISABLE );
                SETBIT( i_preview.Flags, ITEM_DISABLE );
                SETBIT( i_verr.Flags, ITEM_DISABLE );
                SETBIT( i_vlog.Flags, ITEM_DISABLE );
            } else {
                CLEARBIT( i_tex.Flags, ITEM_DISABLE );
                CLEARBIT( i_preview.Flags, ITEM_DISABLE );
                CLEARBIT( i_verr.Flags, ITEM_DISABLE );
                CLEARBIT( i_vlog.Flags, ITEM_DISABLE );
            }
        }
    } else if( flags & MH_DESTROYMENU ) {
        e_status( 1 );
    }
}


/*
 *    Name: gotoline
 *  Return: void
 * Purpose: user interface for e_goto()
 *
 *
 *
 * (c)M.Schollmeyer
 */
#define WIN_WIDTH 50
#define WIN_HEIGHT 9
static void gotoline( void ) {

    static struct Window win = {
        -1, -1, WIN_WIDTH, WIN_HEIGHT, 0, 0, "Goto Line", 0L
    };

    static struct Gadget g_help = {
        NULL, NULL, WIN_WIDTH-10, WIN_HEIGHT-4,
        8, 3, BOOLGADGET | ENDGADGET | DEFHELPGADGET,
    };
    static struct Gadget g_cancel = {
        &g_help, NULL, WIN_WIDTH-22, WIN_HEIGHT-4,
        10, 3, BOOLGADGET | ENDGADGET | DEFCANCELGADGET,
    };

    static struct Gadget g_ok = {
        &g_cancel, NULL, 2, WIN_HEIGHT-4,
        6, 3, BOOLGADGET | ENDGADGET | DEFOKGADGET,
    };

    static struct IntuiText it_linegad = {
        NULL, NULL, "&Goto line:", 0, -1
    };
    static struct StringInfo si_linegad = {
        NULL, 49, 0L
    };
    static struct Gadget g_line = {
        &g_ok, &it_linegad, 2, 3, 46, 1,
        TEXTGADGET | LONGINT | BORDERLESS | ACTIVATED,
        (APTR)&si_linegad,
    };

    struct Gadget *g;
    long line;

    si_linegad.LongInt = ed.CursorY;

    OpenWindow( &win, HLP_GOTO );
    do {
        g = OpenGadget( &g_line, win.Left, win.Top, 0L );
        if( g == &g_help ) help( -1 );
    } while( g == &g_help );
    CloseWindow( &win );

    if( g == &g_ok || g == &g_line )
    {
        line = si_linegad.LongInt;
        if( line < 0L ) return;

        e_goto( ed.WindowX, ed.WindowY, ed.CursorX, (int)line );
        e_refresh();
    }
}
#undef WIN_WIDTH
#undef WIN_HEIGHT


/*
 *    Name: addfile
 *  Return: TRUE on success
 * Purpose: requests for a file name (using DOS_Request()) and adds
 *          it to the current file list. Then tries to load that file.
 *
 *
 * (c)M.Schollmeyer
 */
static BOOL addfile(char *name) {

    char buffer[_MAX_PATH];
    unsigned int num;

    for( num = 0; num < MAXFILES; ++num ) {
        /* Look if there is an empty slot */
        if( GetProfileString( PR_FILENAME( num ), NULL ) == NULL ) {
            if( m_save( SAVE_NORMAL ) ) {
                e_savekoords();
                if (!name) {
                    DOS_Request( "Add File", buffer, "", HLP_ADDFILE );
                    name = buffer;
                }

                if( name[0] ) {
                    if( e_load( name ) ) {
                        PutProfileString( PR_FILENAME( num ), name );
                        ed.LastFile = ed.CurrFile;
                        ed.CurrFile = num;
                        ed.MarkX = ed.MarkY = 0;
                        e_goto( 0, 0, 0, 0 );
                        m_savevirtual();
                        e_status( 1 );
                        e_refresh();
                        return FALSE;
                    } else {
                        e_refresh();
                        return FALSE;
                    }

                    e_refresh();
                } else
                    return FALSE;
            } else
                return FALSE;

            return TRUE;
        }
    }

    DoErrorBox( HLP_NOSLOTS,
                "There are no more free slots.\n"
                "Choose 'Disk/Close' to close one or\n"
                "more files and try again...\n" );

    return FALSE;
}


/*
 *    Name: visitfile
 *  Return: void
 * Purpose: same as addfile() except does not add the file to the
 *          file list.
 *
 *
 *
 * (c)M.Schollmeyer
 */
static void visitfile( void ) {

    char buffer[127];

    if( ! m_save( SAVE_NORMAL ) )
        return;

    e_savekoords();
    DOS_Request( "Visit File", buffer, "", HLP_VISIT );
    if( buffer[0] )
    {
        if( e_load( buffer ) ) {
            e_goto( 0, 0, 0, 0 );
            e_status( 1 );
            ed.Flags |= PSEUDOFILE;
            e_drawflags();
        }
        e_refresh();
    }
}


/*
 *    Name: mergefile
 *  Return: void
 * Purpose: same as addfile() except does not add the file to the
 *          file list.
 *
 *
 *
 * (c)M.Schollmeyer
 */
static void mergefile( void ) {

    char buffer[127];

    DOS_Request( "Merge File", buffer, "", HLP_MERGE );
    if( buffer[0] )
    {
        if( e_merge( buffer ) ) {
            e_status( 1 );
            e_drawflags();
        }
        e_refresh();
    }
}


/*
 *    Name: dos_shell
 *  Return: void
 * Purpose: launches a dos shell.
 *
 *
 *
 * (c)M.Schollmeyer
 */
static void dos_shell( void )
{
    char buffer[BUFSIZE];

    if (!smakec( buffer, "", NULL)) return;
    LaunchOverlay( "", LO_NOECHO );
}

/*
 *    Name: view_output
 *  Return: void
 * Purpose: shows the dos output page, waits for user input
 *          and returns to EasyTeX.
 *
 *
 * (c)M.Schollmeyer
 */
static void view_output( void ) {

    if( VideoBase->ScreenHeight != 25 )
        return; // should never happen
    else {
        HideMouse();
        SetVisualPage( DOSPAGE );

        IntuitionBase->InputEvent->key = 0;
        IntuitionBase->InputEvent->buttons = 0;

        ReadInput();

        SetVisualPage( WORKPAGE );
        ShowMouse();
    }

}

/*
 *    Name: settings
 *  Return: void
 * Purpose: dialogbox for editor switches and values.
 *
 *
 *
 * (c)M.Schollmeyer
 */
static void settings( void )
{
    char           buffer[BUFSIZE],
                  *newini = NULL,
                  *cp;
    int            i, ret;

    ret = settingdlg();

    /* change the ini-file ? */
    if( ret == 1 ) {
        /* first, save file */
        if( m_save( SAVE_NORMAL ) ) {
            e_savekoords();
            /* now, request for new ini-filename */
            DOS_Request( "Change Ini-File", buffer, IniName, HLP_CHANGEINI );
            if( buffer[0] == '\0' )
                goto done;
            /* request successful, proceed... */
            newini = HeapStrDup( buffer );
            if( newini ) {
                DeleteTmps();
                if( put_init() ) {
                    /* delete file list */
                    for( i = 0; i < MAXFILES; ++i )
                        ClearProfileData( PR_FILENAME(i) );

                    /* delete custom menus */
                    for( i = 0; i < PR_MAXCUSTOM; ++i ) {
                        ClearProfileData( PR_CUSTOMCMD(i) );
                        ClearProfileData( PR_CUSTOMOPT(i) );
                        ClearProfileData( PR_CUSTOMNAME(i) );
                    }

                    free( IniName );
                    IniName = newini;
                    for( cp = newini + strlen( newini ); cp >= newini; --cp ) {
                        if( *cp == '\\' || *cp == ':' ) {
                            ++cp;
                            break;
                        }
                    }

                    get_init( TRUE );
                    if( ignore_palette == FALSE ) {
                        if( cp = GetProfileData( PR_PALETTE, NULL ) ) {
                            for( i = 0; i < CR_MAXVAL; ++i, ++cp )
                                VideoBase->Palette[i] = *cp;
                        }
                    }
                    updatemenu();
                    e_setfile(ed.CurrFile);
                }
            }
        }
    }

    /* anything could have changed */
    HideMouse();
    e_drawmr();
    e_redraw();
    e_status(1);
    DrawMenuTitle( &m_file, MENUWIDTH, 0L );
    _printtime(1);
    ShowMouse();

done:
    e_drawflags();

    return;
}


/*
 *    Name: MakeShortName
 *  Return: void
 * Purpose: makes a copy of <src> to <dst>, if <src> is longer than <len>,
 *          MakeShortName copies only the tail of <src> and the first three
 *          chars.
 *
 *
 * (c)M.Schollmeyer
 */
void MakeShortName( char *dst, char *src, int len ) {

    if( strlen( src ) <= len ) {
        strcpy( dst, src );
        return;
    }

    // copy firt three chars
    *dst++ = *src++;
    *dst++ = *src++;
    *dst++ = *src++;
    // make dots
    *dst++ = '.';
    *dst++ = '.';
    *dst++ = '.';
    // copy tail
    src += strlen( src ) - len + 6;
    strcpy( dst, src );
}


unsigned int _printtime( int verb ) {

    static unsigned char secs = (unsigned char)101;
    char buffer[10];
    unsigned long memfree;

    struct RealTimeClock rtc;

    if( GetRTC( &rtc ) ) {
        if( secs != rtc.Seconds || verb ) {
            if (ed.Flags & DISPLAY_MEM)
                sprintf( buffer, "%8lu", MemFree()+
                    (unsigned long)((unsigned char _huge *)mh.desc + 4 -
                                        (4 * mh.numlines) - mh.endlines ) );


            else
                sprintf( buffer, "%02d:%02d.%02d",
                             rtc.Hours, rtc.Minutes, rtc.Seconds );

            VideoStr( CR_FLAGDISABLE, buffer, SCREEN_WIDTH-8, 0 );
            secs = rtc.Seconds;
        }
    }

}




/********
 *
 *      The following code are general file routines
 */

/*
 *    Name: _saverequest
 *  Return: TRUE if file should be saved,
 *          FALSE if file sould not be saved,
 *          CANCEL if actiob should be aborted.
 *
 * Purpose: This function is called from e_save() when file
 *          AUTOSAVE flag is not set.
 *
 *
 *
 * (c)M.Schollmeyer
 */

BOOL _saverequest( char *name ) {

    unsigned int ret = DoMessageBox( DMB_YES|DMB_CANCEL|DMB_NO,
                                     HLP_SAVEREQ, "Message",
                                     "'%s'\nBuffer has been modified\n"
                                     "Save?", name );

    if( ret == DMB_YES )
        return TRUE;
    if( ret == DMB_NO )
        return FALSE;
    if( ret == DMB_CANCEL )
        return CANCEL;
}

BOOL _createrequest( char *name ) {


    unsigned int ret = DoMessageBox( DMB_YES|DMB_CANCEL, HLP_CREATEREQ,
                                     "Message",
                                     "'%s'\nFile does not exist\nCreate?",
                                     name );

    if( ret == DMB_YES )
        return TRUE;
    if( ret == DMB_CANCEL )
        return CANCEL;
}


/*
 *    Name: setfile
 *  Return: void
 * Purpose: displays the file list and shows info for files.
 *
 *
 *
 * (c)M.Schollmeyer
 */
static int pathcmp( char *, char * );

#define WIN_HEIGHT 20
#define WIN_WIDTH  48

// #pragma optimize( "elg", off )
static void setfile( void ) {

    static void _moveentry( int, int );

    static struct Window win = {
        -1, -1, WIN_WIDTH, WIN_HEIGHT, 0, 0,
        "Set File", 0L
    };

    static struct Gadget g_help = {
        NULL, NULL, WIN_WIDTH-10, WIN_HEIGHT-4,
        8, 3, BOOLGADGET | ENDGADGET | DEFHELPGADGET,
    };
    static struct Gadget g_cancel = {
        &g_help, NULL, WIN_WIDTH-22, WIN_HEIGHT-4,
        10, 3, BOOLGADGET | ENDGADGET | DEFCANCELGADGET,
    };

    static struct IntuiText it_close = {
        NULL, NULL, "&Close", 1, 0
    };
    static struct Gadget g_close = {
        &g_cancel, &it_close, 12, WIN_HEIGHT-4,
        9, 3, BOOLGADGET | ENDGADGET,
    };

    static struct Gadget g_ok = {
        &g_close, NULL, 2, WIN_HEIGHT-4,
        6, 3, BOOLGADGET | ENDGADGET | DEFOKGADGET,
    };

    static struct IntuiText it_move = {
        NULL, NULL, "&Move", 1, 0
    };
    static struct Gadget g_move = {
        &g_ok, &it_move, 23, WIN_HEIGHT-8,
        8, 3, BOOLGADGET | ENDGADGET,
    };

    static struct IntuiText it_info = {
        NULL, NULL, "&Info...", 1, 0
    };
    static struct Gadget g_info = {
        &g_move, &it_info, 11, WIN_HEIGHT-8,
        11, 3, BOOLGADGET | ENDGADGET,
    };

    static struct IntuiText it_sort = {
        NULL, NULL, "&Sort", 1, 0
    };
    static struct Gadget g_sort = {
        &g_info, &it_sort, 2, WIN_HEIGHT-8,
        8, 3, BOOLGADGET | ENDGADGET,
    };

    struct IntuiText *ita[MAXFILES+1];
    struct IntuiText it[MAXFILES];
    static struct Request req = { NULL, NULL, MULTISELECT };
    static struct Gadget gad_req = {
        &g_sort, NULL, 1, 2, 46, 10, REQGADGET | ACTIVATED,
        (APTR)&req, 0L
    };

    int i, j, k, move_reg = -1;
    char *fname1, *fname2;
    BOOL dogetfile = FALSE;

    struct Gadget *g;

    req.Items = ita;

    SETBIT( req.Flags, RESETREQUEST );

    OpenWindow( &win, HLP_SETFILE );
    req.CurrIt = ed.CurrFile;

    while( 1 ) {
        for( i = 0, j = 0; i < MAXFILES; ++i )
        {
            fname1 = GetProfileString( PR_FILENAME(i), NULL);
            ita[i] = &it[i];
            it[i].Flags = 0;
            it[i].Next = NULL;
            it[i].TextBorder = NULL;
            if( ! (it[i].Text = AllocHeap( 44 ) ) ) {
                for( j = 0; j < i; ++j )
                    free( it[i].Text );
                goto done;
            }
            it[i].Left = 0;
            it[i].Top = 0;
            if( fname1 ) {
                MakeShortName( it[i].Text, fname1, 41 );
                j = i + 1;
            } else
                it[i].Text[0] = '\0';
        }

        ita[j] = NULL;
        if( j < 2 ) {
            SETBIT( g_move.Flags, DISABLED );
            SETBIT( g_sort.Flags, DISABLED );
        } else {
            CLEARBIT( g_move.Flags, DISABLED );
            CLEARBIT( g_sort.Flags, DISABLED );
        }

        if( j ) {
            CLEARBIT( g_close.Flags, DISABLED );
            CLEARBIT( g_info.Flags, DISABLED );
        } else {
            SETBIT( g_close.Flags, DISABLED );
            SETBIT( g_info.Flags, DISABLED );
        }

        do {
            g = OpenGadget( &gad_req, win.Left, win.Top, 0L );
            if( g == &g_help ) help( -1 );
        } while( g == &g_help );

        for( i = 0; i < MAXFILES; ++i )
            free( it[i].Text );

        i = req.CurrIt;

        if( g == &g_sort ) {
            for( j = 0; j < MAXFILES-1; ++j ) {
                fname1 = GetProfileString( PR_FILENAME(j  ), NULL );
                fname2 = GetProfileString( PR_FILENAME(j+1), NULL );
                if( pathcmp( fname1, fname2 ) > 0 ) {
                    /* swap entries */

                    if( j == ed.CurrFile )
                        ++ed.CurrFile;
                    else if( j+1 == ed.CurrFile )
                        --ed.CurrFile;

                    if( j == ed.LastFile )
                        ++ed.LastFile;
                    else if( j+1 == ed.LastFile )
                        --ed.LastFile;

                    SwapProfileData( PR_FILENAME(j), PR_FILENAME(j+1) );
                    SwapProfileData( PR_VIRTFILE(j), PR_VIRTFILE(j+1) );
                    SwapProfileData( PR_FILEKOORDS(j), PR_FILEKOORDS(j+1) );

                    j = -1;
                }
            }
            req.CurrIt = ed.CurrFile;
        } else if( g == &g_close ) {
            SetMSGadget( &req );
            SETBIT( req.Flags, RESETREQUEST );

            /* search for selected files */
            for( j = 0; j < MAXFILES; ++j ) {
                if( ita[j] == NULL ) break;
                if( ita[j]->Flags & IT_SELECTED ) {
                    if( j != ed.CurrFile || ed.Flags & PSEUDOFILE ) {
                        /* This one can be simply closed */
                        e_close( j );
                        req.CurrIt = ed.CurrFile;
                    } else if( e_close( j ) ) {
                        /* This one is (was) the current file */
                        dogetfile = TRUE;
                    }
                }
            }
            if( dogetfile ) {
                e_setfile(ed.LastFile);
                req.CurrIt = ed.CurrFile;
            }
        } else if( g == &g_info )
            disp_info( i );
        else if( g == &g_move ) {
            SETBIT( gad_req.Flags, SINGLESELECT );
            move_reg = req.CurrIt;
        } else if( g == &gad_req ) {
            if( move_reg == -1 )
                break;
            else {
                _moveentry( move_reg, req.CurrIt );
                DrawGadget( &gad_req, win.Left, win.Top );
                while( GetInput()->buttons & MBUTTON_LEFT );
            }
            CLEARBIT( gad_req.Flags, SINGLESELECT );
            move_reg = -1;
        } else if( g == NULL || g == &g_cancel || g == &g_ok )
            break;
        else
            move_reg = -1;

    }

    CloseWindow( &win );

    if( g == &g_ok || g == &gad_req ) {
        if( i == -1 || !GetProfileString( PR_FILENAME(i), NULL ) )
            return;
        if( i == ed.CurrFile && !(ed.Flags & PSEUDOFILE) ) {
            if( ! ( ed.Flags & (MODIFIED|LINECHANGED) ) )
                goto done;
            if( DoMessageBox( DMB_YES|DMB_CANCEL, HLP_SETFILE2,
                lpszWarning, "Refresh buffer?\nChanges will be lost!" )
                == DMB_YES ) {
                if( !e_setfile( i ) ) {
                    ClearProfileData( PR_FILENAME(i));
                    e_setfile(ed.LastFile);
                }
            }
            goto done;
        }
        if( m_save( SAVE_NORMAL ) ) {
            e_savekoords();
            if( !e_setfile( i ) )
                ClearProfileData( PR_FILENAME(i) );
        }
    }
done:
    e_refresh();
}
#undef WIN_WIDTH
#undef WIN_HEIGHT
// #pragma optimize( "", on )

static void _moveentry( int from, int to ) {

    int i, direction;

    if( from < to )
        direction = 1;
    else if( from > to )
        direction = -1;
    else
        return;

    for( i = from; i != to; i += direction ) {
        if( i == ed.CurrFile )
            ++ed.CurrFile;
        else if( i+1 == ed.CurrFile )
            --ed.CurrFile;

        if( i == ed.LastFile )
            ++ed.LastFile;
        else if( i+1 == ed.LastFile )
            --ed.LastFile;

        SwapProfileData( PR_FILENAME(i), PR_FILENAME(i+direction) );
        SwapProfileData( PR_VIRTFILE(i), PR_VIRTFILE(i+direction) );
        SwapProfileData( PR_FILEKOORDS(i), PR_FILEKOORDS(i+direction) );
    }

}


/*
 *    Name: pathcmp
 *  Return: a value expressing the relation between <path1> and <path2>
 * Purpose: This function compares two paths and returns a value
 *          expressing the relation of the two strings. Comparable
 *          to the ansi strcmp() function pathcmp() compares the
 *          strings byte for byte.
 *          In difference to the strcmp() function, pathcmp() compares
 *          the paths under notice of path levels.
 *
 * (c)M.Schollmeyer
 */

static int pathcmp( path1, path2 )
char *path1, *path2;
{
    BOOL islevel[2];

    /* check if paths set */
    if( path2 == NULL )
        return -1;

    if( path1 == NULL )
        return 1;

    /* scan strings for differences */
    while( ( tolower(*path1) == tolower(*path2) ) && *path1 ) {
        ++path2;
        ++path1;
    }

    /* end of string reached, strings are equal */
    if( *path1 == '\0' || *path2 == '\0' )
        return 0;

    /* since <path1> and <path2> now point to the first difference,
       we have 4 cases:
       - no more further levels: compare next char.
       - both have further levels: compare next char.
       - path1 or path2 has further levels: return relation.
    */

    islevel[0] = strchr( path1, '\\' ) ? TRUE : FALSE;
    islevel[1] = strchr( path2, '\\' ) ? TRUE : FALSE;

    if( !( islevel[1] ^ islevel[0] ) )
        return (int)(tolower(*path1) - tolower(*path2));
    if( islevel[1] )
        return -1;
    return 1;
}


/*
 *    Name: LaunchOverlay
 *  Return: void
 * Purpose: tries to execute the command <cmd> in an overlay.
 *
 *
 *
 * (c)M.Schollmeyer
 */
void LaunchOverlay( char *cmd, int flags ) {

    int bflag;
    int ret;
    char *command;
    char *defcommand = "\\command.com";
    char *cmdopt = "/c";

    command = getenv( "COMSPEC" );
    if( command == NULL )
        command = defcommand;

    if( e_savecboard() ) {

        if( m_save( SAVE_NORMAL ) ) {
            e_savekoords();
            put_init();
            HideMouse();

            if( VideoBase->ScreenHeight != 25 ) {
                HideMouse();
                ClearScreen();
                ShowMouse();
                SetCursorPos( 0, 0 );
            } else
                SetVisualPage( DOSPAGE );

            SetPointerPage( DOSPAGE );
            UnSetHandler();
            bflag = BreakFlag( old_breakflag );

            /* These are the options for the next call to et.ovl. */
            sprintf( SharedMemory->OvlArgs, "%s /i%s",
                                             MAGICOVERLAY, IniName );

            if( !(flags & LO_NOECHO) )
                printf( "\t%s\n", cmd );

            if( flags & LO_WAIT )
                SharedMemory->WaitFlag = 1;
            else
                SharedMemory->WaitFlag = 0;

            /* Freeing the edit buffer is a good idea at this time.
               This makes room for the execlp function. We exit at this
               time in any case. */
            FreeMem( FP_SEG(mh.base) );

            SharedMemory->ExitCode = 1; /* Call me again. */

            if( *cmd )
                ret = execlp( command, command, cmdopt, cmd, NULL );
            else {
                puts( "\nType 'exit' to return to EasyTeX" );
                ret = execlp( command, command, NULL );
            }

            /* Now something went wrong with exec */
            BreakFlag( bflag );
            ShowMouse();
            SetHandler( critical_error );

            if( VideoBase->ScreenHeight != 25 ) {
                ShowMenu();
                e_status( 1 );
            } else {
                SetVisualPage( WORKPAGE );
                SetPointerPage( WORKPAGE );
            }

            if( ret != -1 ) {
                DoErrorBox( HLP_LAUNCH1,
"Something's wrong, I tried to execute\n\
'%s %s %s'\nBut that didn't work, sorry...", command, cmdopt, cmd );
            } else {
                DoErrorBox( HLP_LAUNCH2,
"Error executing overlay:\n'%s %s %s'\nError: %s",
                            command, cmdopt, cmd,
                            sys_errlist[errno] );
            }
            SharedMemory->WaitFlag = 0;
            ExitProc(0);
        }
    }
}


/*
 *    Name: searchstring
 *  Return: void
 * Purpose: user interface to e_search() and e_replace().
 *
 *
 *
 * (c)M.Schollmeyer
 */
#define WIN_WIDTH 50
#define WIN_HEIGHT 15
#pragma optimize( "elg", off )
void searchstring( int flag )
{
    static char *cap_src = "Search String";
    static char *cap_s_r = "Query Search & Replace";

    static struct Window win = {
        -1, -1, WIN_WIDTH, WIN_HEIGHT, 0, 0,
        NULL, 0L
    };

    static struct Gadget g_help = {
        NULL, NULL, WIN_WIDTH-10, WIN_HEIGHT-4,
        8, 3, BOOLGADGET | ENDGADGET | DEFHELPGADGET,
    };
    static struct Gadget g_cancel = {
        &g_help, NULL, WIN_WIDTH-22, WIN_HEIGHT-4,
        10, 3, BOOLGADGET | ENDGADGET | DEFCANCELGADGET,
    };
    static struct IntuiText it_minus = {
        NULL, NULL, "&Up", 2, 0
    };
    static struct Gadget g_minus = {
        &g_cancel, &it_minus, 11, WIN_HEIGHT-4,
        8, 3, BOOLGADGET | TOGGLESELECT | ENDGADGET | NODESELECT,
        NULL, 1L
    };

    static struct IntuiText it_plus = {
        NULL, NULL, "&Down", 1, 0
    };
    static struct Gadget g_plus = {
        &g_minus, &it_plus, 2, WIN_HEIGHT-4,
        8, 3, BOOLGADGET | TOGGLESELECT | ENDGADGET | NODESELECT,
        NULL, 1L
    };

    static struct IntuiText it_files = {
        NULL, NULL, "&Search Across Files", 4, 0
    };
    static struct Gadget g_files = {
        &g_plus, &it_files, 2, 8,
        18, 1, CHECKGADGET|TOGGLESELECT,
    };

    static struct IntuiText it_case = {
        NULL, NULL, "&Case Sensitive", 4, 0
    };
    static struct Gadget g_case = {
        &g_files, &it_case, 2, 7,
        18, 1, CHECKGADGET|TOGGLESELECT,
    };

    static struct IntuiText it_rep = {
        NULL, NULL, "&Replace by:", 0, -1
    };
    static struct StringInfo si_rep = {
        NULL, EDIT_WIDTH-1, 0L
    };
    static struct Gadget g_rep = {
        &g_case, &it_rep, 2, 5, 46, 1, TEXTGADGET | BORDERLESS | MARKEND,
        (APTR)&si_rep,
    };

    static struct IntuiText it_src = {
        NULL, NULL, "&Enter Search String:", 0, -1
    };
    static struct StringInfo si_src = {
        NULL, EDIT_WIDTH-1, 0L
    };
    static struct Gadget g_src = {
        NULL, &it_src, 2, 3, 46, 1, TEXTGADGET | BORDERLESS | MARKEND | ACTIVATED,
        (APTR)&si_src,
    };

    static struct IntuiText rit_request = {
        NULL, NULL, "Do you want to replace this occurrence?", 0, 2 };

    static struct Window rwin = {
        -1, 0,
        58, 8, 0, 0, "Query Search & Replace", 0L
    };

    static struct Gadget rg_help = {
        NULL, NULL, 48, 4, 8, 3, BOOLGADGET | ENDGADGET | DEFHELPGADGET,
    };
    static struct Gadget rg_cancel = {
        &rg_help, NULL, 36, 4, 10, 3, BOOLGADGET | ENDGADGET | DEFCANCELGADGET,
    };

    static struct IntuiText rit_all = {
        NULL, NULL, "Replace &All", 1, 0
    };
    static struct Gadget rg_all = {
        &rg_cancel, &rit_all, 19, 4, 15, 3, BOOLGADGET | ENDGADGET,
    };

    static struct IntuiText rit_no = {
        NULL, NULL, "&No", 1, 0
    };
    static struct Gadget rg_no = {
        &rg_all, &rit_no, 11, 4, 6, 3, BOOLGADGET | ENDGADGET,
    };

    static struct IntuiText rit_yes = {
        NULL, NULL, "&Yes", 1, 0
    };
    static struct Gadget rg_yes = {
        &rg_no, &rit_yes, 2, 4, 7, 3,
        BOOLGADGET | ENDGADGET | ACTIVATED
    };

    unsigned char replaceall = 0;
    struct Gadget *g;

    static struct MarkRegion mr = { 0,0,0,0,"",MR_INITVAL};
    static int currfile=-1;

    int numreps = 0, ret;
    BOOL filechanged = FALSE;

    si_src.Buffer = si_rep.Buffer = NULL;

    if( !(si_src.Buffer = AllocHeap( EDIT_WIDTH )))
        goto done;
    if( !(si_rep.Buffer = AllocHeap( EDIT_WIDTH )))
        goto done;

    strncpy( si_src.Buffer, GetProfileString( PR_SEARCHSTRING, &profnull ),
        EDIT_WIDTH );

    strncpy( si_rep.Buffer, GetProfileString( PR_REPLACESTRING, &profnull ),
        EDIT_WIDTH );

    CLEARBIT( ed.SearchFlags, SEARCH_REGULAR );

    if( ed.SearchFlags == 0 )
        SETBIT(ed.SearchFlags,SEARCH_PLUS);

    /* we need this only for the first call of searchstring or if someone
       changes ed.Searchflags...
     */
    CLEARBIT(g_plus.Flags,SELECTED);
    CLEARBIT(g_minus.Flags,SELECTED);

    if( ed.SearchFlags & SEARCH_PLUS )
        SETBIT(g_plus.Flags,SELECTED);
    else
        SETBIT(g_minus.Flags,SELECTED);

    if( ed.SearchFlags & SEARCH_CASE )
        SETBIT(g_case.Flags,SELECTED);
    else
        CLEARBIT(g_case.Flags,SELECTED);

    if( ed.SearchFlags & SEARCH_FILES )
        SETBIT(g_files.Flags,SELECTED);
    else
        CLEARBIT(g_files.Flags,SELECTED);

    if( (flag != CONTSEARCH) || (mr.flags & MR_INITVAL) ) {
        currfile=-1;
        mr = *(m_getmr());
    }

    if( flag == CONTSEARCH ) {
        if( !(mr.flags & MR_MULTILINE) ) {
            mr.UpperX = mr.UpperY = 0;
            mr.LowerY = mh.numlines;
            mr.LowerX = EDIT_WIDTH;
        }
    } else {
        if( flag == QUERYSEARCH ) {
            SETBIT( ed.SearchFlags, SEARCH_QUERY );
            win.Title = cap_s_r;
            g_src.Next = &g_rep;
        } else {
            CLEARBIT( ed.SearchFlags, SEARCH_QUERY );
            win.Title = cap_src;
            g_src.Next = &g_case;
        }

        if( mr.flags & MR_MARKED && !(mr.flags & MR_MULTILINE) )
            strncpy( si_src.Buffer, mr.buffer, EDIT_WIDTH );

        CLEARBIT( ed.Flags, MARKED );

        OpenWindow( &win, HLP_SEARCH );

        do {
            g = OpenGadget( &g_src, win.Left, win.Top, 0L );
            if( g == &g_help ) help( -1 );
        } while( g == &g_help );

        CloseWindow( &win );

        if( g == &g_cancel || g == NULL ) {
            goto done;
        } else {
            /* save search strings */
            PutProfileString( PR_SEARCHSTRING, si_src.Buffer );
            PutProfileString( PR_REPLACESTRING, si_rep.Buffer );
        }

        if( flag == QUERYSEARCH && si_rep.Buffer[0] == '\0' ) {
            if( DoMessageBox( DMB_OK|DMB_CANCEL, HLP_CONFEMPTY, "Warning",
                    "Confirm empty replacement string" ) == DMB_CANCEL )
                goto done;
        }

        if( g == &g_plus ) {
            SETBIT(ed.SearchFlags, SEARCH_PLUS);
            CLEARBIT(ed.SearchFlags, SEARCH_MINUS);
        } else if( g == &g_minus ) {
            SETBIT(ed.SearchFlags, SEARCH_MINUS);
            CLEARBIT(ed.SearchFlags, SEARCH_PLUS);
        }

        if( g_case.Flags & SELECTED )
            SETBIT(ed.SearchFlags,SEARCH_CASE);
        else
            CLEARBIT(ed.SearchFlags,SEARCH_CASE);

        if( g_files.Flags & SELECTED )
            SETBIT(ed.SearchFlags,SEARCH_FILES);
        else
            CLEARBIT(ed.SearchFlags,SEARCH_FILES);

        if( (mr.flags & MR_MARKED) && (mr.flags & MR_MULTILINE) ) {
            if( !m_putcurrentline() )
                goto done;
            if( ed.SearchFlags & SEARCH_PLUS ) {
                ed.CursorX = mr.UpperX;
                ed.CursorY = mr.UpperY;
                ed.MarkX = mr.LowerX;
                ed.MarkY = mr.LowerY;
            } else {
                ed.CursorX = mr.LowerX;
                ed.CursorY = mr.LowerY;
                ed.MarkX = mr.UpperX;
                ed.MarkY = mr.UpperY;
            }
            m_getcurrentline();
        } else {
            /* no region marked, search from top of buffer to end */
            mr.UpperX = mr.UpperY = 0;
            mr.LowerY = mh.numlines;
            mr.LowerX = EDIT_WIDTH;
        }
    }
again:
    while( e_search( si_src.Buffer, ed.SearchFlags, &mr ) ) {

        if( ed.SearchFlags & SEARCH_QUERY ) {
            if( replaceall == FALSE ) {
                /* request replacement */
                if( ed.CursorY - ed.WindowY < ed.EditHeight/2 )
                    rwin.NewTop = EDITW_TOP+ed.EditHeight/2+1;
                else
                    rwin.NewTop = EDITW_TOP+1;

                OpenWindow( &rwin, HLP_QUERY );
                CenterText( &rit_request, rwin.Width );
                PrintIText( &rit_request, rwin.Left, rwin.Top );
                do {
                    g = OpenGadget( &rg_yes, rwin.Left, rwin.Top, 0L );
                    if( g == &rg_help ) help( -1 );
                } while( g == &rg_help );
                CloseWindow( &rwin );

                if( g == &rg_no ) {
                    if( ed.SearchFlags & SEARCH_PLUS )
                        ed.CursorX += strlen( si_src.Buffer );
                    continue;
                } else if( g == &rg_all ) {
                    --verbose;
                    replaceall = TRUE;
                } else if( g == &rg_cancel || g == NULL ) {
                    goto cancelrep;
                }
            }
            if( !e_replace( strlen( si_src.Buffer ), si_rep.Buffer,
                ed.SearchFlags ) )
                break;
            ++numreps;

        } else {
            goto done;
        }
    }

    /* check for 'search across files' */
    if( (ed.SearchFlags & SEARCH_FILES) &&
      ( (mr.flags & (MR_MULTILINE|MR_MARKED)) != (MR_MULTILINE|MR_MARKED) ) ) {
        /* first we have to check for currfile to avoid dead end cycles */
        if( currfile == -1 )
            currfile = ed.CurrFile;
        else if( currfile == ed.CurrFile ) {
            ret = DoMessageBox( DMB_YES|DMB_NO, HLP_SRCFILES, "Message",
                   "No more files to scan\nStart again ?" );
            if( (ret == DMB_NO) || (ret == DMB_CANCEL) ) {
                if( filechanged ) {
                    if( ed.SearchFlags & SEARCH_PLUS )
                        e_goto(0,0,0,mh.numlines);
                    else
                        e_goto(0,0,0,0);
                }
                currfile = -1;
                e_refresh();
                goto done;
            }
        }
        if( ed.SearchFlags & SEARCH_PLUS ) {
            /* load next file */
            if( e_next() )
                e_goto(0,0,0,0);
        } else {
            /* load prev file */
            if( e_prev() ) {
                /* This seems to be strange, since the cursor is outside
                   the window, but this will be adjusted by search */
                e_goto(0,0,EDIT_WIDTH-1,mh.numlines);
                ed.WindowX = 0;
            }
        }
        /* we have to reinitialize mr since it has been changed
           by e_next() or e_prev() */
        mr.UpperX = mr.UpperY = 0;
        mr.LowerY = mh.numlines;
        mr.LowerX = EDIT_WIDTH;
        filechanged = TRUE;
        goto again;
    }

    if( replaceall ) {
        ++verbose;
cancelrep:
        e_refresh();
        SetCursorPos( ed.CursorX - ed.WindowX , ed.CursorY - ed.WindowY + EDITW_TOP );
        m_putmsg( MSG_STNDRD, "%d occurrences replaced", numreps );
        goto done;
    }

    SETBIT( mr.flags, MR_INITVAL );
    m_putmsg( MSG_ERROR, "search string not found" );
done:
    if( si_src.Buffer ) free( si_src.Buffer );
    if( si_rep.Buffer ) free( si_rep.Buffer );
}
#undef WIN_WIDTH
#undef WIN_HEIGHT
#pragma optimize( "", on )


/*
 *    Name: CreateTmp
 *  Return: handle if successful, -1 if not
 * Purpose: this routine opens a temporary file named 'ETxxxxxx.TMP' on
 *          the directory specified by the environment variable 'TMP'.
 *          CreateTmp() uses isvirtualfile() (wich is defined below) to
 *          check if the temporary file is not in use by any other routine.
 *          This is neccessary to prevent different routines from accessing
 *          their temporary files.
 *          e.g. you have 2 slots filled with 2 files called HOBO.TEX for
 *          slot 1 and MARY.TXT for slot 2 and their corresponding virtual
 *          file ET000000.TMP for slot 1 and ET000001.TMP for slot 2.
 *          Assume you then launch a dos shell and delete both temporary
 *          files. After returning to EasyTeX, the current file (assume
 *          in slot 2) is trying to recover (using e_reload(). This fails,
 *          because the temporary file does not exist. The file is reloaded
 *          using e_load() wich saves the virtual file using CreateTmp(),
 *          wich will open a file named ET000000.TMP (wich does not exist)
 *          for use. When you then use e_prev() to get the previous file,
 *          e_reload() assumes that its virtual file resides in ET000000.TMP
 *          again and tries to reload it. Actually MARY.TXT is loaded from
 *          that file wich is not correct, so we have to keep track of all
 *          temporary files.
 *
 *
 * (c)M.Schollmeyer
 */
int CreateTmp( char *buf, unsigned int attr ) {

    int fptr, i;
    unsigned long l = 0x0428a0L;

    static BOOL isvirtualfile( char * );

    do {
        sprintf( buf, "%set%06x.tmp", TmpPath, l );
        fptr = Open( buf, ACCESS_READ );
        if( fptr == -1 && !isvirtualfile( buf ) ) {
            fptr = Create( buf, attr );
            return fptr;
        }
        Close( fptr );
        ++l;
    } while( l < 0xffffff );

    return -1;
}



static BOOL isvirtualfile( char *name ) {

    int i;

    for( i = 0; virtfiles[i] != 0 ; ++i ) {
        if( stricmp( name,
            GetProfileString( virtfiles[i], &profnull ) ) == 0 )
            return TRUE;
    }

    return FALSE;
}


/*
 *    Name: DeleteTmps
 *  Return: void
 * Purpose: delete all temporary files.
 *
 *
 *
 * (c)M.Schollmeyer
 */
void DeleteTmps( void ) {

    int i;
    char *cp;

    for( i = 0; virtfiles[i] != 0 ; ++i ) {
        cp = GetProfileString( virtfiles[i], NULL );
        if( cp ) {
            Delete( cp );
            ClearProfileData( virtfiles[i] );
        }
    }
}

void DeleteTeXTmps( void ) {

    char *cp;
    char ovlname[BUFSIZE];

    cp = GetProfileString( PR_INIPATH, NULL );
    if( cp ) {
        SetCurrDir( cp );
    }

    cp = GetProfileString( PR_TMPTEXFILES, NULL );
    if( cp ) {
        strcpy( ovlname, MyPath );
        strcat( ovlname, "rdel.exe" );
        execlp( ovlname, ovlname, cp, NULL );
    }
}


void updatemenu( void ) {

    static struct MenuItem custom[PR_MAXCUSTOM];

    struct MenuItem *mi;
    char *cp;
    int i, key;

    i_vlog.Next = NULL;

    for( i = 0, mi = &i_utilsec2, key = 0x7800;
         i < PR_MAXCUSTOM ; mi = mi->Next, ++i ) {
        cp = GetProfileString( PR_CUSTOMNAME(i), NULL );
        if( cp == NULL )
            break;
        else
            i_vlog.Next = &i_utilsec2;

        mi->Next = &custom[i];
        custom[i].ItemName = cp;

        if( GetProfileLong( PR_CUSTOMFLAGS(i) ) & CM_SECTION ) {
            custom[i].Flags = ITEM_SECTION;
            continue;
        }

        if( key == 0x8100 )
            key = 0;

        custom[i].ShortKey = key;

        if( key )
            key += 0x100;

        custom[i].Flags = 0L;
        custom[i].Message = GetProfileString( PR_CUSTOMOPT(i), &profnull );
    }

    mi->Next = NULL;
}

/*
 *    Name: critical_error
 *  Return: void (Well, actually the user response in ax)
 * Purpose: This is the cricical error handler.
 *          Remember, don't optimize _interrupt-routines, since global
 *          optimization will remove the _ax = ... at the end of this
 *          routine 'cause it's quiet useless for C, but the return value
 *          of this routine.
 *
 * (c)M.Schollmeyer
 */
#pragma optimize( "elg", off )
void _interrupt _far critical_error( unsigned _es, unsigned _ds,
                                     unsigned _di, unsigned _si,
                                     unsigned _bp, unsigned _sp,
                                     unsigned _bx, unsigned _dx,
                                     unsigned _cx, unsigned _ax,
                                     unsigned _ip, unsigned _cs,
                                     unsigned flags )

{
    int ret;
    unsigned int ( *helpfunc )();

    /* first, disable online help */
    helpfunc = SetHook( NULL, HOOK_HELP );

    IntPreamble();
    _enable();

    errnum = _di+0x13;

    ret = PutError( errnum, NULL );

    if( ret == DMB_CANCEL || ret == DMB_OK )
        _ax = 0;
    else
        _ax = 1;

    IntPostamble();

    /* restore online help */
    SetHook( helpfunc, HOOK_HELP );
}
#pragma optimize( "", on )

/* converts dta date stamp to ascii format */
void dta_datetoa( unsigned long stamp, char _far *cp ) {

    unsigned int i;

    i = DOS_DAYS(stamp);
    *cp++ = _i2asc( i / 10 );
    *cp++ = _i2asc( i % 10 );
    *cp++ = '.';
    i = DOS_MONTHS(stamp);
    *cp++ = _i2asc( i / 10 );
    *cp++ = _i2asc( i % 10 );
    *cp++ = '.';
    i = DOS_YEARS(stamp);
    *cp++ = _i2asc( i / 1000 );
    i = i % 1000;
    *cp++ = _i2asc( i /  100 );
    i = i % 100;
    *cp++ = _i2asc( i /   10 );
    *cp++ = _i2asc( i %   10 );
    *cp++ = ' ';
    i = DOS_HOURS(stamp);
    *cp++ = _i2asc( i / 10 );
    *cp++ = _i2asc( i % 10 );
    *cp++ = ':';
    i = DOS_MINUTES(stamp);
    *cp++ = _i2asc( i / 10 );
    *cp++ = _i2asc( i % 10 );

    *cp = '\0';
}

int packstr( char *str ) {

    char *cp;

    for( cp = str; *cp; ++cp ) {
        if( isspace( (int)*cp ) )
            continue;
        if( cp == str )
            return 0;
        while( *cp )
            *str++ = *cp++;
        return 1;
    }
    *str = '\0';
    return -1;
}

/* end of file main.c */
