/* maketex.c: this program emulates the MakeTeX* shell scripts from web2c.

Copyright (C) 1997 Fabrice POPINEAU.

This library is free software; you can redistribute it and/or
modify it under the terms of the GNU Library General Public
License as published by the Free Software Foundation; either
version 2 of the License, or (at your option) any later version.

This library is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
Library General Public License for more details.

You should have received a copy of the GNU Library General Public
License along with this library; if not, write to the Free Software
Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.  */

/* Maximum number of programs taken into account. */
#define MAX_PROGS 10
/* Maximum number of lines per usage message. */
#define MAX_LINES 12

#define SEPARATORS " \t\n\""

#define mvfile(a,b) CopyFile(a, b, FALSE)

#include <direct.h>
#include <process.h>

#include <kpathsea/config.h>
#include <kpathsea/c-ctype.h>
#include <kpathsea/c-fopen.h>
#include <kpathsea/c-stat.h>
#include <kpathsea/expand.h>
#include <kpathsea/getopt.h>
#include <kpathsea/line.h>
#include <kpathsea/pathsearch.h>
#include <kpathsea/c-memstr.h>
#include <kpathsea/c-pathch.h>
#include <kpathsea/c-pathmx.h>
#include <kpathsea/c-vararg.h>
#include <kpathsea/proginit.h>
#include <kpathsea/tex-file.h>
#include <kpathsea/tex-glyph.h>
#include <kpathsea/variable.h>

/*
#if defined(__STDC__) || defined(WIN32)
#define NeedVarargsPrototypes   1
#include <stdarg.h>
#else
#define NeedVarargsPrototypes   0
#include <varargs.h>
#endif
*/

#if 0
#define fopen(f, m) (fprintf(stderr, "opening %s\n", f), fopen(f, m))
#define fclose(f) (fprintf(stderr, "closing %x\n", f), fclose(f))
#endif

/* We are keeping trace of the environment (ie: cwd, file
  redirections) with the help of these ops and structures. There is a
  global stack inidcating wich actions have been taken. Popping the
  stack enables us to undo these actions at any moment (eg: aborting
  in the middle of an operation, restoring the cwd).  
  */
typedef enum { CHDIR = 1, REDIRECT } op_env;
typedef struct mod_env {
  op_env op;
  union {
  string path;
  int oldfd[3];
  } data;
} mod_env;

/* The global stack. */
mod_env stack_env[256];
int index_env = 0;

extern DllImport char* kpathsea_version_string;

/* Variables for MakeTeX* scripts */
static int program_number = -1;
static string progname;
static string texmf;
static string ls_R_magic = "% ls-R -- maintained by MakeTeXls-R; do not change this line.\n";
static char tfmname[PATH_MAX];
static char pkname[PATH_MAX];
static char pkdestdir[PATH_MAX];
static char tfmdestdir[PATH_MAX];
char tmpdir[PATH_MAX]; /* Working temporary directory */
string output = "astdout";
FILE *fout;
static boolean downcase_names;

/* Variables for MakeTeX.site */
static string mode;
static int dpi;
static string bdpi;
static string ps_to_pk;
static string dcfontdir;
static string ecfontdir;
static string fcfontdir;
static string tcfontdir;
static string tsfontdir;
static string sauterfontdir;
static string mt_features;
boolean mt_dosnames = FALSE;
boolean mt_nomode = FALSE;
boolean mt_strip = FALSE;
boolean mt_varfonts = FALSE;

static struct alist {
  string key;
  string *var;
  string val;
} vardef[] = {
  {"MODE", &mode, "ljfour"},
  {"BDPI", &bdpi, "600"},
  {"ps_to_pk", &ps_to_pk, "gsftopk"},
  {"dcfontdir", &dcfontdir, "$TEXMF/fonts/source/jknappen/dc"},
  {"ecfontdir", &ecfontdir, "$TEXMF/fonts/source/jknappen/ec"},
  {"fcfontdir", &ecfontdir, "$TEXMF/fonts/source/jknappen/fc"},
  {"tcfontdir", &tcfontdir, "$TEXMF/fonts/source/jknappen/tc"},
  {"tsfontdir", &tsfontdir, "$TEXMF/fonts/source/jknappen/ts"},
  {"sauterfontdir", &sauterfontdir, "$TEXMF/fonts/source/public/sauter"},
  {"MT_FEATURES", &mt_features, ""},
  {NULL, NULL, NULL}
};

/* Variables for MakeTeXnames.cnf */
static string mt_dir_perms;
static string mt_pkname;
static string mt_namepart;
static string mt_destroot;

/* Type of function to execute */
typedef void (*execfn)(string, struct stat*);
static struct stat stat_buf;


/* Test whether getopt found an option ``A''.
   Assumes the option index is in the variable `option_index', and the
   option table in a variable `long_options'.  */
#define ARGUMENT_IS(a) STREQ (long_options[option_index].name, a)

static struct option long_options [] = {
  { "debug",               1, 0, 0},
  { "help",                0, 0, 0},
  { "version",             0, 0, 0},
  {0, 0, 0, 0}
};

/*
  This array describes all the programs that are taken into account.
  Each program is caracterized by :
  name, args required, args max, usage_msg
  */

void maketexpk(int, char**);
void maketextfm(int, char**);
void maketextex(int, char**);
void maketexmf(int, char**);
void maketexlsr(int, char**);
void maketexupdate(int, char**);
void maketexmkdir(int, char**);
void maketexrmdir(int, char**);
void maketexnames(int, char**);

void rec_rmdir(string);
void do_rmdir(string);
void do_makedir(string);
void delmulslash(string );

/* Pointer on functions that take argc and argv[] */
typedef void (*fp)(int, string*);

/* Description of on functionality : long name, short name, min args,
   max args and pointer on function to execute. */
static struct program_description {
  string name;
  string altname;
  int arg_min;
  int arg_max;
  fp prog;
} makedesc [] = {
  {"MakeTeXPK",     "makepk",    4, 6, maketexpk },
  {"MakeTeXTFM",    "maketfm",   2, 2, maketextfm },
  {"MakeTeXTeX",    "maketex",   4, 6, maketextex },
  {"MakeTeXMF",     "makemf",    2, 2, maketexmf },
  {"MakeTeXls-R",   "makelsr",   1, 2, maketexlsr },
  {"MakeTeXupdate", "makeupd",   3, 3, maketexupdate },
  {"MakeTeXmkdir",  "makemd",    2, 255, maketexmkdir },
  {"MakeTeXrmdir",  "makerd",    2, 255, maketexrmdir },
  {"MakeTeXnames",  "makename",  2, 4, maketexnames },
  {0, 0, 0}
};

/*
  First  argument takes progname;
  Other arguments take no arguments.
  */
static string usage_msg[][MAX_LINES] = {
  /* MakeTeXPK */
  {"Usage: %s NAME DPI BDPI MAG [MODE [DESTDIR]].\n",
   "Try to create a PK file for NAME at resolution DPI,\n",
   "with an assumed device base resolution of BDPI,\n",
   "and a Metafont `mag' of MAG.\n",
   "Use MODE for the Metafont mode if supplied, unless MODE is `default', in\n",
   "which case guess as usual. (This is so you can specify DESTDIR without MODE.)\n",
   "Use DESTDIR for the root of where to install into, either the absolute\n",
   "directory name to use (if it starts with a /) or relative to the default\n",
   "DESTDIR (if not).\n",
   0
  },
  /* MakeTeXTFM */
  {
    "Usage : %s FONT.\n",
    "Makes a TFM file for FONT, if possible.\n",
    0
  },
  /* MakeTeXTeX */
  {
    0
  },
  /* MakeTeXMF */
  {
    "Usage : %s FONT.\n",
    "Makes the Metafont source file for FONT, if possible.\n",
    "For example, 'dcr12' or 'cmr11'.\n",
    0
  },
  /* MakeTeXls-R */
  {
    "Usage : %s [DIR].\n",
    "Rebuild the ls-R file completely for the default or specified texmf DIR.\n",
    0
  },
  /* MakeTeXupdate */
  {
    "Usage : %s DIR FILE.\n",
    "Update the ls-R file with an entry for FILE in DIR.\n",
    0
  },
  /* MakeTeXmkdir */
  {
    "Usage : %s DIRS ...\n",
    "Create each DIR, including any missing leading directories.\n",
    0
  },
  /* MakeTeXrmdir */
  {
    "Usage : %s DIRS ...\n",
    "Recursively remove each DIR.\n",
    0
  },
  /* MakeTeXnames */
  {
    "Usage : %s NAME [DPI MODE] [DESTDIR].\n",
    "Output the PK and TFM names for a font NAME.\n",
    0
  }
};

/* extern void mt_exit(int); */
extern BOOL HandlerRoutine(DWORD);

/* Reading MakeTeX.site */
void read_mtsite()
{
  FILE *f;
  boolean found;
  string line;
  int len;
  struct alist *p;
  string mtsite = kpse_find_file("MakeTeX.site", kpse_cnf_format, true);
  string t;

  /* First assigning default values to all variables. */
  for(p = vardef; p->key != NULL; p++) {
    /* testing if there is an env var */
    line = concat("$", p->key);
    *(p->var) = kpse_var_expand(line);
    if (*(p->var) == NULL || **(p->var) == '\0')
      *(p->var) = p->val;
    free(line);
  }

  /* and next overriding those variables in MakeTeX.site */
  if ((f = fopen(mtsite, "r")) == NULL) 
    return;
  while ((line  = read_line(f)) != NULL) {
    if (*line == '#' || *line == '%' || isspace(*line)
	|| *line == '\0' || *line == '\n')
      continue;
    
    found = FALSE;
    for(p = vardef; !found && p->key != NULL; p++) {
      len = strlen(p->key);
      if (strncmp(p->key, line, len) == 0 && line[len] == '=') {
	*(p->var) = xstrdup(line+len+1);
	found = TRUE;
      }
    }
    if (!found) 
      fprintf(stderr, "MakeTeX.site: unrecognized keyword in '%s'.\n",
	      line);
    free(line);
  }
  fclose(f);
  /* Processing of mt_features (roughly equivalent to MakeTeXnames.cnf) */
  /* We just set booleans for future use */
  t = strtok(mt_features, SEPARATORS);
  while(t != NULL && *t != '\0') {
    fprintf(stderr, "MT_FEATURES : %s\n", t);
    if (strcmp(t, "dosnames") == 0)
      mt_dosnames = TRUE;
    else if (strcmp(t, "nomode") == 0)
      mt_nomode = TRUE;
    else if (strcmp(t, "strip") == 0)
      mt_strip = TRUE;
    else if (strcmp(t, "varfonts") == 0) {
      mt_varfonts = TRUE;
    }
    t = strtok(NULL, SEPARATORS);
  }
  free(mtsite);
}

void output_and_cleanup(int code)
{
  FILE *f;
  string output_name;
  string line;

  output_name = concat3(tmpdir, DIR_SEP_STRING, output);

  /* output result if any */
  if (code == 0 && (f = fopen(output_name, "r")) != NULL) {
    while((line = read_line(f)) != NULL) {
      fputs(line, stdout);
      free (line);
    }
    fclose(f);
  }

  free(output_name);

  rec_rmdir(tmpdir);
}

void usage()
{
  int i;
  fprintf(stderr, "%s version of kpathsea %s\n", progname, kpathsea_version_string);
  fprintf(stderr,usage_msg[program_number][0], progname );
  fputs("\n", stderr);
  for(i = 1; usage_msg[program_number][i]; ++i)
    fputs(usage_msg[program_number][i], stderr);
}

/* ensure that the path is written with the same DIR_SEP */
void slashify(string path)
{
  while(*path) {
    if (IS_DIR_SEP(*path))
      *path = DIR_SEP;
    path++;
  }
}


int main(int argc, char* argv[])
{
  int i,j;
  int g; /* getopt return code */
  int option_index;
  struct program_description * program;
  string tempenv;
  FILE *fnul;
  int newfd[3];

  if (!progname)
    progname = argv[0];
  kpse_set_progname (progname);

  /* NULL for no fallback font. */
  kpse_init_prog (uppercasify (progname), dpi, mode, NULL);

  for(i = 0; makedesc[i].name; ++i)
    if (!strcasecmp(program_invocation_short_name, makedesc[i].name)
        || !strcasecmp(program_invocation_short_name, makedesc[i].altname)) {
      program_number = i;
      progname = makedesc[i].name;
      program = makedesc+i;
      break;
    }

  if (!makedesc[i].name) {
    fprintf(stderr, "This program was incorrectly copied to the name %s\n", 
	    argv[0]);
    exit(1);
  }

  for(;;) {
    g = getopt_long_only (argc, argv, "", long_options, &option_index);

    if (g == EOF)
      break;

    if (g == '?')
      exit (1);  /* Unknown option.  */

    /* assert (g == 0); /* We have no short option names.  */
    
    if (ARGUMENT_IS ("debug")) {
      kpathsea_debug |= atoi (optarg);
    }
    else if (ARGUMENT_IS ("help")) {
      usage();
      exit(0);
    }
    else if (ARGUMENT_IS ("version")) {
      fprintf(stderr, "%s of %s.\n", progname, kpathsea_version_string);
      exit(0);
    }

  }
#if 0
  fprintf(stderr, "Before Arg count = %d\n", argc);
  for (i = 0; i < argc; i++)
    fprintf(stderr, "argv[%d] = %s\n", i, argv[i]);
#endif
  /* shifting options from argv[] list */
  for (i = j = 1; i < argc; i++) {
    if (*(argv[i]) != '-') {
      argv[j] = argv[i];
      j++;
    }
  }
  argc = j;
#if 0
  fprintf(stderr, "After Arg count = %d\n", argc);
  for (i = 0; i < argc; i++)
    fprintf(stderr, "argv[%d] = %s\n", i, argv[i]);
#endif
  if ((argc < program->arg_min)
      || (argc  > program->arg_max)) {
    fprintf(stderr, 
	    "Bad number of arguments. Use %s --help if you need it.\n", 
	    progname);
    exit(1);
  }

  /* texmf may contain multiple pathes ! */
  texmf = kpse_path_expand("$TEXMF");

  tempenv = getenv("TEMP");
  sprintf(tmpdir, "%s/mtXXXXXX", (tempenv ? tempenv : "/tmp"));
  mktemp(tmpdir);
  do_makedir(tmpdir);

  pushd(tmpdir);
  
  if ((fout = fopen(output, "w")) == NULL) {
    perror(output);
    mt_exit(1);
  }
  
  xputenv("KPSE_DOT", tmpdir);
  
  SetConsoleCtrlHandler((PHANDLER_ROUTINE)HandlerRoutine, TRUE);
  
  read_mtsite();

  fnul = fopen("nul", "r");	/* fopen /dev/null */
  
  newfd[0] = fileno(fnul);
  newfd[1] = 2;
  newfd[2] = 2;
  
  push_fd(newfd);
  program->prog(argc, argv);
  
  fclose(fnul);
  fclose(fout);

  mt_exit(0);

}

/* Like the test function from Unix */
boolean test_file(char c, string path)
{
  boolean file_exists = (stat(path, &stat_buf) != -1);
  if (!file_exists) return FALSE;
  switch (c) {
  case 'd':
    return ((stat_buf.st_mode & S_IFMT) == S_IFDIR);
  case 'e':
    return TRUE;
  case 'f':
  case 'r':
    return ((stat_buf.st_mode & S_IFMT) == S_IFREG);
  case 's':
    return (stat_buf.st_size > 0);
  }
}

/*
  Recursive walk through the directory tree. Depth-first order. 
*/
void recurse_dir(string path, execfn before, execfn after)
{
    /* In depth traversal of the subdir tree */
#if defined(_WIN32)
  WIN32_FIND_DATA find_file_data;
  HANDLE hnd;
  int index;
#else
    DIR *dp;
    struct dirent *ep;
#endif
    struct stat stat_buf;	/* We have to have one local because
				   of after */

    int path_len = strlen(path);

    /* current node */
    if (stat(path, &stat_buf))
	perror(path);

    /* execute before for the current node */
    if (before)
      (*before)(path, &stat_buf);

    /* if it is a directory, recurse through all sons */
    if ((stat_buf.st_mode & S_IFMT) == S_IFDIR) {
#if defined(_WIN32)
      index = strlen(path);
      strcat(path, "/*");
#if 0
      fprintf(stderr, "Opening hnd on %s\n", path);
#endif
      hnd = FindFirstFile(path, &find_file_data);
      while (hnd != INVALID_HANDLE_VALUE && 
	     FindNextFile(hnd, &find_file_data) != FALSE) { 
	if(!strcmp(find_file_data.cFileName, ".")
	   || !strcmp(find_file_data.cFileName, "..")) 
	  continue;
	path[index+1] = '\0';
	strcat(path, find_file_data.cFileName);
	recurse_dir(path, before, after);
      }
      path[index] = '\0';
#if 0
      fprintf(stderr, "Closing hnd on %s\n", path);
#endif
      FindClose(hnd);
#else
	if (dp = opendir(path)) {
	    while (ep = readdir(dp))
		if (strcmp(ep->d_name, ".") &&
		    strcmp(ep->d_name, "..")) {
		    path[path_len] = '/';
		    strcpy(path+path_len+1, ep->d_name);
		    recurse_dir(path, opt);
		}
	    closedir(dp);
	}
	else
	    perror(NULL);
#endif
    }
    /* execute after for the current node */
    if (after)
      (*after)(path, &stat_buf);
}

/* remove something which already exists */
void remove_path(string path, struct stat* st)
{
#if 0
  fprintf(stderr, "Removing %s.\n", path);
#endif
  if ((st->st_mode & S_IFMT) == S_IFDIR) {
    if (rmdir(path) == -1)
      perror(path);
  }
  else {
  if (unlink(path) == -1)
    perror(path);
  }
}

/* Creates path, and all intermediate directories if necessary. */

void do_makedir(string path)
{
  string p = path;

  /* We can't authrize multiple slashes here */
  delmulslash(path);

  if (stat(path, &stat_buf) == 0 &&
      (stat_buf.st_mode & S_IFMT) == S_IFDIR)
    return;			/* The path already exists */

  /* Finding the [drive letter][:] */
  if (IS_DEVICE_SEP(*(p+1)))
    p +=2;

  if (IS_DIR_SEP(*p))
    p++;

  for ( ; *p ; p++) {
    if (IS_DIR_SEP(*p)) {
      *p = '\0';
      if ((_mkdir(path) == -1)
	  && (errno != EEXIST)) { /* We do it for each path elt, even if it
				     already exists */
	perror(path);
	fprintf(stderr, "%s: MakeTeXmkdir %s failed.\n", progname, path);
	mt_exit(1);
      }
      *p = DIR_SEP;
    }
  }
  /* and for the whole path */
  if (_mkdir(path) == -1) {
    perror(path);
    fprintf(stderr, "%s: MakeTeXmkdir %s failed.\n", progname, path);
    mt_exit(1);
  }

}

/* Output the string in lower case or without modification according
   to downcase_names. */
void output_name(string name, FILE *f)
{
  string p;
  if (downcase_names) {
    for(p = name; *p; p++)
      *p = tolower(*p);
  }
  fputs(name, f);
}

/* Updates ls-R db for directory pointed to by path. */
void print_path(string path, struct stat* st)
{
  WIN32_FIND_DATA ffd;
  HANDLE hnd;
  int index;
  if ((st->st_mode & S_IFMT) == S_IFDIR) {
    output_name(path, stdout);
    putchar(':'); putchar('\n');
    index = strlen(path);
    strcat(path, "/*");
    hnd = FindFirstFile(path, &ffd);
    while (FindNextFile(hnd, &ffd)) { 
      if (*ffd.cFileName == '.'
	  && ((ffd.cFileName[1] == '\0')
	      || (ffd.cFileName[1] == '.' && ffd.cFileName[2] == '\0')))
	continue;
      output_name(ffd.cFileName, stdout);
      putchar('\n');/* We do want a \n */
    }
    path[index] = '\0';
    FindClose(hnd);
  }
}

/* return true if the fs on which path lies is case sensitive.
   Insufficient. Some people may have copied the complete tree in
   upper case rfom a case insensitive fs.
   Second test : look at the texmf name. If GetShortName !=
   GetLongName it is ok. 
*/
boolean is_casesensitive(string path)
{
  char name[PATH_MAX];
  char volname[PATH_MAX];
  DWORD sernum;
  DWORD maxcomplen;
  DWORD fsflags;
  char fsname[PATH_MAX];
  
  /* How should the db be output ? Preserve case or lower case ? 
   First, determine the root volume : if path has the drive, ok. If
   not, get current drive. */
  
  if (IS_DEVICE_SEP(path[1])) {	/* the shortest path is "." and has 2 chars */
    name[0] =  path[0];
  }
  else
    name[0] = _getdrive() + 'A' - 1;

  strcpy(name+1, ":\\");

  if (GetVolumeInformation(name, volname, 256, &sernum, &maxcomplen,
			   &fsflags, fsname, 256) == FALSE) {
    fprintf(stderr, "Error getting volume info on %s.\n", name);
    mt_exit(1);
  }
  return (boolean)((fsflags & FS_CASE_IS_PRESERVED) != 0);

}

void do_maketexlsr(string path)
{
  static char name[PATH_MAX];

  downcase_names = ! is_casesensitive(path);

  strcpy(name,path);
  recurse_dir(name, print_path, NULL);
}

void maketexlsr(int argc, char **argv)
{
  /* We have to take multiple root trees into account. */

  string lsrname, lsrdir, elt;
  char tmpname[32];
  FILE* lsrfile;
  int old, newfd[3];

  /* If only no argument, we must use the default texmf tree.*/
  if (argc == 1)
    lsrdir = xstrdup(texmf);
  else {
    lsrdir = xstrdup(argv[1]);
    if (!test_file('d', lsrdir)) {
      fprintf(stderr, "%s : %s is not a directory !\n", progname, lsrdir);
      mt_exit(1);
    }
  }
  for (elt = kpse_path_element(lsrdir); elt;
       elt = kpse_path_element(NULL)) {
    lsrname = concat(elt, "/ls-R");
    strcpy(tmpname, "lsXXXXXX");	/* temporary ls-R file */
    if (mktemp(tmpname) == NULL) {
      perror("lsXXXXXX");
      mt_exit(1);
    }
    if ((lsrfile = fopen(tmpname, "w")) == NULL) {
      perror(lsrname);
      mt_exit(1);
    }
  
    newfd[0] = 0;
    newfd[1] = fileno(lsrfile);
    newfd[2] = 2;
    push_fd(newfd);
    
    puts(ls_R_magic);
    putchar('\n');
    
    pushd(elt);
    do_maketexlsr(".");
    
    popd();			/* back to tmpdir */
    /* Restore original stdout */
    pop_fd();

    fflush(stdout);
    fclose(lsrfile);
    
    if (mvfile(tmpname, lsrname) == FALSE) {
      fprintf(stderr, "Error : can't copy %s on %s.\n", tmpname, lsrname);
      mt_exit(1);
    }
    free(lsrname);
  }
  free(lsrdir);
}

string findmap(string name)
{
  FILE *f;
  string line;
  string token;
  string filename;
  string namepart = "";
  string source = "";
  string typeface = "";

  filename = kpse_find_file("special.map", kpse_fontmap_format, true);
  f = filename ? fopen(filename, FOPEN_R_MODE) : NULL;;
  if (f == NULL) oops("Cannot find file special.map.");

  /* FIXME : there is a seg fault if the font is not found here */
  /* Is it ok ? 03/02/97 ;.. */
  while((line = read_line(f)) != NULL) {
    token = strtok(line, SEPARATORS);
    if (!token || !*token)
      continue;
#if 0
    fprintf(stderr, "%s %s %d %c\n", token, name, strlen(token), name[strlen(name)-1]);
#endif
    if (!strcmp(token, name)
	|| (!strncmp(token, name, strlen(token))
	    && isdigit(name[strlen(name)-1])
	    && !isdigit(token[strlen(token)-1]))) {
      source = strtok(NULL, SEPARATORS);
      typeface = strtok(NULL, SEPARATORS);
      namepart = concat3(source, DIR_SEP_STRING, typeface);
      break;
    }
    free(line);
  }
  if (!namepart || !*namepart) {
    free(filename);
    filename = kpse_find_file("supplier.map", kpse_fontmap_format, true);
    f = filename ? freopen(filename, FOPEN_R_MODE, f) : NULL;
    if (f == NULL) oops("Cannot find file supplier.map.");

    while((line = read_line(f)) != NULL) {
      token = strtok(line, SEPARATORS);
      if ((strlen(token) == 1) && (*token == *name)) {
	source = xstrdup(strtok(NULL, SEPARATORS));
	break;
      }
      free(line);
    }
    free(filename);
    if (source) {
      filename = kpse_find_file("typeface.map", kpse_fontmap_format, true);
      f = filename ? freopen(filename, FOPEN_R_MODE, f) : NULL;
      if (f == NULL) oops("Cannot find file typeface.map.");
      while((line = read_line(f)) != NULL) {
	token = strtok(line, SEPARATORS);
	if ((strlen(token) == 2)
	    && (name[1] == token[0]) && (name[2] == token[1])) {
	  typeface = xstrdup(strtok(NULL, SEPARATORS));
	  break;
	}
	free(line);
      }
      if (typeface) {
	namepart = concat3(source, DIR_SEP_STRING, typeface);
	free(source);
	free(typeface);
      }
    }
  }
  free(filename);
  fclose(f);
  return namepart;
}

void delmulslash(string path)
{
  string p, q;
#if 0
  fprintf(stderr, "Delmulslash before : %s\n", path);
#endif
  /* Delete any multiple slashes */
  for (p = q = path; *p != '\0'; p++, q++) {
    if (*p == '/') {
      *q = *p;
      q++;
      while (*++p == '/');
    }
    *q = *p;
  }
  *q = '\0';
#if 0
  fprintf(stderr, "Delmulslash after : %s\n", path);
#endif
}  

/* Test if two pathes are equal. Up to the shortest one. */
boolean path_included(const_string p1, const_string p2)
{
  boolean ret = true;
#if 0
  fprintf(stderr, "Path_included found that %s included in %s is", p1, p2);
#endif
  for( ; *p1 && *p2 && ret; p1++, p2++)
    ret = ret && (IS_DIR_SEP(*p1) ? IS_DIR_SEP(*p2) :
		  toupper(*p1) == toupper(*p2));
#if 0
  fprintf(stderr, " %s\n", ret ? "true" : "false");
#endif
  return ret;
}

/* return the path element in path which can match dir else NULL */
string path_in_pathes(const_string dir, const_string path)
{
  string ret = NULL;
  string elt;

  for (elt = kpse_path_element (path); 
       elt && !ret;
       elt = kpse_path_element (NULL)) {
    /* Not sufficient to have both equal, there should be a DIR_SEP
       on the next char of dir */
    if (path_included(elt, dir) &&
	IS_DIR_SEP(*(dir+strlen(elt))))
      ret = xstrdup(elt);
  }
#if 0
  fprintf(stderr, "Path_in_Pathes found that %s is in %s\n",
	  dir, ret);
#endif
  return ret;
  
}

/* Look if .tfm file exists, else if .mf file exists. Will have
   to enhance this for MakeTeXMF. */
string try_to_find_root(const_string name)
{
  string tfm_name = concat(name, ".tfm");
  string mf_name = concat(name, ".mf");
  
  string location;
  string elt, p;
  string ret = NULL;
#if 0
  fprintf(stderr, "Try to find root : %s\ntfm = %s, mf = %s\n", name, tfm_name, mf_name);
  fprintf(stderr, "%s : %s\n", 
	  tfm_name, kpse_find_file(tfm_name, kpse_tfm_format, true));
  fprintf(stderr, "%s : %s\n", 
	  mf_name, kpse_find_file(mf_name, kpse_mf_format, true));
#endif
  location = kpse_find_file(tfm_name, kpse_tfm_format, true);
  if (location == NULL)
    location = kpse_find_file(mf_name, kpse_mf_format, true);
#if 0
  fprintf(stderr, "Location : %s\n", location);
#endif
  free (tfm_name);
  free (mf_name);
  
  if (location && *location) {
    /* Looking for which path_element in texmf is matched by location. */
    ret = path_in_pathes(location, texmf);
    free(location);
  }

  /* We have a texmf tree in which to put something.
     Is it possible ? At first sight, yes if it is not on
     a CD-ROM ... in which case we return $TEXMFMAIN */
  if (NAME_BEGINS_WITH_DEVICE(ret)
      && IS_DIR_SEP(ret[2])) {
    char drv[4];
    int type;
    strncpy(drv, ret, 3);
    drv[3] = '\0';
    if ((type = GetDriveType(drv)) == DRIVE_CDROM
	/*	|| type == 0 The drive type cannot be determined ? */
	|| type == 1) {
      free(ret);
      ret == NULL;
    }
    fprintf(stderr, "Drive type : %d\n", type);
  }

  if (!ret) {
    ret = kpse_var_expand("$TEXMFMAIN");
    if (!ret)
      oops("%s:  No $TEXMFMAIN; set the environment variable or in texmf.cnf.\n", progname);
  }

#if 0
  fprintf(stderr, "%s\n", ret);
#endif
  return ret;
}
       

void do_maketexnames(string name, string dest, int dpi, string mode)
{
  string mt_destroot, mt_namepart, mt_mode;
  string mt_tfmpart = "tfm";
  string mt_pkpart = "pk";
  char sdpi[24];

  /* we do not need to determine fontname, just need to call
     kpse_find_file with special.map ... */
  /*  if ((fontname == NULL) || (*fontname == '\0'))
    fontname = concat(texmf, "/fontname"); */

  /* find namepart in *.map */
  if ((mt_namepart = findmap(name)) == NULL)
    mt_namepart = "tmp";
  
  if (mt_varfonts) {
    mt_destroot = kpse_var_expand("$VARTEXFONTS");
    mt_namepart = "";
    if (!mt_destroot || !*mt_destroot) {
      fprintf(stderr, "%s: You asked for varfonts in MT_FEATURES, but VARTEXFONTS\n", progname);
      fprintf(stderr, "%s: is not set as an environment variable or in texmf.cnf.\n", progname);
    }
  } else
    mt_destroot = try_to_find_root(name);

  if (mt_nomode)
    mt_mode = "";
  else
    mt_mode = mode;

  if (mt_strip)
    mt_namepart = "";
    
  *pkdestdir = '\0';
  if (dest)
    if (kpse_absolute_p(dest, false))
      strcpy(pkdestdir,dest);
    else
      mt_namepart = dest;

  if (!*pkdestdir)
    sprintf(pkdestdir, "%s/fonts/%s/%s/%s", mt_destroot, mt_pkpart, mt_mode, mt_namepart);
  sprintf(tfmdestdir, "%s/fonts/%s/%s", mt_destroot, mt_tfmpart, mt_namepart);

  if (mt_dosnames) {
    strcat(pkdestdir, "/dpi");
    itoa(dpi, sdpi, 10);
    strcat(pkdestdir, sdpi);
    sprintf(pkname, "%s/%s.pk", pkdestdir, name);
  }
  else
    sprintf(pkname, "%s/%s.%dpk", pkdestdir, name, dpi);

  /*  tfmname = tfmdestdir/name.tfm */
  sprintf(tfmname, "%s/%s.tfm\n", tfmdestdir, name);
  
  delmulslash(pkdestdir );
  delmulslash(tfmdestdir);
  delmulslash(pkname);
  delmulslash(tfmname);
}

void maketexnames(int argc, char **argv)
{
  string name = argv[1];
  string dest = NULL;

  argc--; /* We are eliminating argv[0] */
  if (argc <= 2) 
    dpi = atoi(bdpi);
  if (argc == 2)
    dest = argv[2];
  if (argc >= 3) {
    dpi = atoi(argv[2]);
    mode = argv[3];
  }
  if (argc == 4)
    dest = argv[4];

  do_maketexnames(name, dest, dpi, mode);

  fputs(pkname, fout); fputc(' ', fout);
  fputs(tfmname, fout); fputc(' ', fout);
  fputc('\n', fout);
}

void maketexmkdir(int argc, char **argv)
{
  int i;
  string path;
  for(i = 1; i < argc; i++) {
    /* We have to concat the original wd if argv[i] is not absolute */
    if (kpse_absolute_p(argv[i], false))
      do_makedir(argv[i]);
    else {
      /* original path is in stack_env[0] */
      path = concat3(stack_env[0].data.path, DIR_SEP_STRING, argv[i]);
      do_makedir(path);
      free(path);
    }
  }
      
}

void do_rmdir(string path)
{
  static char name[PATH_MAX];
  strcpy(name,path);

  if (test_file('d', path)) {
    recurse_dir(name, NULL, remove_path);
#if 0
    if (rmdir(path) == -1) perror(path);
#endif
  }
  else
    DeleteFile(path);
}

/* Recursively destructs files & directories from path. */
void rec_rmdir(string path)
{
  string p;
#if 0
  fprintf(stderr, "Path : %s, l = %d, last : %c\n", path, strlen(path), path[strlen(path)-1]);
#endif
  for(p=path+strlen(path)-1; IS_DIR_SEP(*p) && (p > path) ; p--) {
    *p = '\0';
  }
  do_rmdir(path);
}

void maketexrmdir(int argc, char **argv)
{
  int i;
  string path;
  for(i = 1; i < argc; i++) {
      do_rmdir(argv[i]);
  }
      
}

void do_maketexupdate(string destdir, char *name)
{
  string path, db_file, localtexmf;
  char buf[256];
  FILE *f;

  slashify(destdir);
  if (!test_file('d', destdir)) {
    fprintf(stderr, "%s: %s is not a directory.\n", progname, destdir);
    mt_exit(1);
  }
  path = concat3(destdir, DIR_SEP_STRING, name);
  if (!test_file('f', path)) {
    fprintf(stderr, "%s: %s is not a file.\n", progname, name);
    mt_exit(1);
  }
  free(path);
  /* We should look for each path_elt of texmf ! */
#if 0
  fprintf(stderr, "first : %s\n", path_in_pathes(destdir, texmf));
  fprintf(stderr, "second : %s\n", path_in_pathes(destdir, kpse_var_expand("$VARTEXFONTS")));
#endif

  if ((localtexmf = kpse_var_expand("$TEXMFLS_R")) == NULL
      && (localtexmf = path_in_pathes(destdir, texmf)) == NULL
      && (localtexmf = 
	  path_in_pathes(destdir, 
			 kpse_var_expand("$VARTEXFONTS"))) == NULL) {
    fprintf(stderr, 
	    "%s: MakeTeXupdate found that %s isn't a directory in the %s tree.\n", 
	    progname, destdir, texmf);
    mt_exit(1);
  }
  
  db_file = concat(localtexmf, "/ls-R");
  if (!test_file('f', db_file)) {
    char *arg[2];
    arg[0] = "MakeTeXls-R";
    arg[1] = localtexmf;
    maketexlsr(2, arg );
    if (!test_file('f', db_file)) {
      fprintf(stderr, "%s: %s does not exist.\n", progname, db_file);
      mt_exit(1);
    }
  }
  if ((f = fopen(db_file, "r+")) == NULL) {
    fprintf(stderr, "%s: %s unwritable.\n", progname, db_file);
    mt_exit(1);
  }
  if (!fgets(buf, 256, f) || strncmp(buf, ls_R_magic, strlen(ls_R_magic))) {
    fprintf(stderr, "%s: %s lacks magic string\n%s.\n", progname, db_file,
	    ls_R_magic);
    mt_exit(1);
  }

  destdir = destdir + strlen(localtexmf) - 1;
  *destdir = '.';
  fseek(f, 0L, SEEK_END);
  /* does the fs preserve case ? */
  if (!is_casesensitive(localtexmf)) {
    string p;
    for(p = destdir; *p; p++)
      *p = tolower(*p);
    for(p = name; *p; p++)
      *p = tolower(*p);
  }
  fprintf(f, "%s:\n%s\n", destdir, name); 
  /*    fputs(destdir, f); fputs(":\n", f);
	fputs(name, f); fputc('\n', f); */
  fclose(f);
}

void maketexupdate(int argc, char **argv)
{
  do_maketexupdate(argv[1], argv[2]);
}

void maketexmf(int argc, char **argv)
{

}

void maketextex(int argc, char **argv)
{

}

void maketextfm(int argc, char **argv)
{

}

int get_mode_from_mf(string mode)
{
  int std_in[2], std_out[2];
  int new_fd[3], oldin, oldout, pid, status, result = 0;
  char buf[1024], *p;
  FILE *mf_in, *mf_out;

  if (pipe(std_in, 0, _O_TEXT) != 0 
      || pipe(std_out, 0, _O_TEXT) != 0) {
    perror("pipe");
    mt_exit(1);
  }
  /*
  flushall();
  
  oldin = dup(0);
  oldout = dup(1);
  
  if (dup2(std_in[0], 0) != 0)
    perror("dup2 std_in");
  if (dup2(std_out[1], 1) != 0)
    perror("dup2 std_out");
  flushall();
  */
  new_fd[0] = std_in[0];
  new_fd[1] = std_out[1];
  new_fd[2] = 2;
  push_fd(new_fd);

  pid = spawnlp(_P_NOWAIT, "mf", "mf", NULL);
  Sleep(50); /* Don't know why, but emacs is doing it too ... */

  /*
  dup2(oldin, 0);
  dup2(oldout, 1);
  */
  pop_fd();
  /* close(10); */
  /* close(12); */
  /*
  close(std_in[0]);
  close(std_out[1]);
  */

  if (pid == -1) {
    perror("Spawning mf.exe");
    close(std_in[1]); close(std_out[0]);
    mt_exit(1);
  }

  mf_in = fdopen(std_in[1], "w");
  mf_out = fdopen(std_out[0], "r");

  sprintf(buf, "\\mode:=%s; mode_setup; message\"BDPI= \"&decimal round pixels_per_inch;end.\n", mode);
  fprintf(mf_in, buf);
  flushall();
  for(; result==0 && !feof(mf_out) ;) {
    memset(buf, '\0', 1024);
    fgets(buf, 1024, mf_out);
    for(p = buf; *p; p++)
      if (strncmp(p, "DPI= ", 5) ==0) {
	result=atoi(p+5);
	break;
      }
  }

  if (_cwait(&status, pid, 0) == -1) {
    perror("Waiting for mf.");
    mt_exit(1);
  }
  fclose(mf_in);
  fclose(mf_out);
  return result;
}

void maketexpk(int argc, char **argv)
{
  string name = argv[1];
  string dpi = argv[2];
  string bdpi = argv[3];
  string mag = argv[4];
  string dest = NULL;
  char line[256], cmd[256];
  string pkbasename, psfontmapname;
  char gfname[PATH_MAX];
  char gfpkname[PATH_MAX];
  char pktempname[PATH_MAX];
  FILE *config_file;
  string index;
  int len = strlen(name);
  int mf_bdpi, ibdpi;
  boolean res;
  int retcode;

  if (argc >= 6)
    mode = argv[5];
  if (argc == 7)
    dest = argv[6];

  /* look for name in psfont.map */
  config_file = kpse_open_file("psfonts.map", kpse_dvips_config_format);
  if (config_file == NULL) oops("Cannot find file psfonts.map.");
  /* looking for pattern "^r$name(0|8r)?([ \t]|$)" */
  res = false;
  while (fgets(line,255,config_file) != NULL) {
    index = line;
    if (*line == 'r' && !strcmp(line+1, name))
      index += len+1;
    else if (!strcmp(line, name))
      index += len;
    else
      continue;
    if (*index == '0')
      index++;
    else if (*index == '8' && *(index+1) == 'r')
      index += 2;
    else continue;
    if (*index == '\0' || isspace(*index)) {
      res = true;
      break;
    }
  }
  fclose(config_file);
  if (res == true) {
    strcpy(mode, "modeless");
    sprintf(cmd, "%s %s %s", ps_to_pk, name, dpi);
  }
  else if (strlen(mode) > 0) {
    if ((mf_bdpi = get_mode_from_mf(mode)) != atoi(bdpi)) {
      fprintf(stderr, "%s: Mismatched mode %s and resolution %s; ignoring mode.\n", progname, mode, bdpi);
      strcpy(mode, "");
    }
    if (strlen(mode) == 0  || strcmp(mode, "default") == 0) {
      ibdpi = atoi(bdpi);
      if (ibdpi == 300)
	mode = "cx";
      else if (ibdpi == 600)
	mode = "ljfour";
      else {
	fprintf(stderr, "%s: Can't guess mode for %s dpi devices.\n",
		progname, bdpi);
	fprintf(stderr, "%s: Use a config file, or update me.\n", progname);
	mt_exit(1);
      }
    }
    sprintf(cmd, "mf \\mode:=%s; mag:=%s; scrollmode; input %s",
	    mode, mag, name);
  }
    
  /* Put files into proper place */
  do_maketexnames(name, dest, atoi(dpi), mode);
  do_makedir(pkdestdir);

  pkbasename = basename(pkname);

  sprintf(gfname, "%s.%sgf", name, dpi);
  sprintf(gfpkname, "%s.%spk", name, dpi);

  if (test_file('e', pkname)) {
    fprintf(stderr, "%s: %s already exists.\n", progname, pkname);
    fputs(pkname, fout); fputc('\n', fout);
    do_maketexupdate(pkdestdir, pkbasename);
    return;
  }

  /* Now run metafont or gsftopk */
  retcode = system(cmd);
  if (retcode == -1)
    perror(mode);
  if (retcode != 0) {
    fprintf(stderr, "%s: '%s' failed.\n", progname, cmd);
    mt_exit(1);
  }
  
  if (test_file('e', gfname)) {
    sprintf(cmd, "gftopk ./%s %s", gfname, pkbasename);
    retcode = system(cmd);
    if (retcode == -1)
      perror("gsftopk");
    if (retcode != 0) {
      fprintf(stderr, "%s: gsftopk failed to make %s.\n", progname, pkbasename);
      mt_exit(1);
    }
  }
  if (!test_file('e', pkbasename) && test_file('e', gfpkname))
    if (mvfile(gfpkname, pkbasename) == FALSE) {
      fprintf(stderr, "%s: can't move %s to %s.\n", gfpkname, pkbasename);
      mt_exit(1);
    }
  if (!test_file('s', pkbasename)) {
    fprintf(stderr, "%s: '%s' failed to make %s.\n", progname, cmd,
	    pkbasename);
    mt_exit(1);
  }

  sprintf(pktempname, "%s/pktmp.XXX", pkdestdir);
  mktemp(pktempname);
  if (mvfile(pkbasename, pktempname) == FALSE) {
    fprintf(stderr, "%s: can't move %s to %s.\n", progname, pkbasename, 
	    pktempname);
    mt_exit(1);
  }
  if (!test_file('e', pkname))
    if (mvfile(pktempname, pkname) == FALSE 
	|| DeleteFile(pktempname) == FALSE ) {
      fprintf(stderr, "%s: can't move %s to %s.\n", progname, pktempname, 
	      pkname);
    mt_exit(1);
    }
  
  do_maketexupdate(pkdestdir, pkbasename);

  /* Maybe we should have been waiting before restoring stdin & stdout*/
  fputs(pkname, fout); fputc('\n', fout);
}

/* This one should help to clean up a font tree in a TDS way. It
  should also regenerate any pk font older than the same tfm font. 
  */
void maketextdsify(int arc, char **argv)
{

}
