package Mojolicious::Plugin::CoverDb;
use Mojo::Base 'Mojolicious::Plugin', -signatures;
use Mojo::Collection 'c';

our $VERSION = '0.05';

sub register ($self, $app, $conf = {}) {
    if ($app->mode eq 'development') {
        my $route = $conf->{route} // 'coverdb';
        $route =~ s#/$##;

        my $dir = $conf->{dir} // 'cover_db';
        $dir =~ s#/$##;

        my $r = $app->routes;
           $r = $r->under($route);

        push @{$app->renderer->classes}, __PACKAGE__;

        $r->get('/' => sub ($c) {
            if (-e "$dir/coverage.html") {
                return $c->reply->file(Mojo::File->new($dir, 'coverage.html')->to_abs);
            }
            my $files = c(glob("$dir/*.html"))->map(sub { $_ =~ s#$dir/##; return $_; });
            $c->stash('files' => $files);
            $c->stash('dir'   => $dir);
            $c->stash('route' => $route);
            $c->render(template => 'coverdb_index', layout => undef);
        });
        $r->get('/#file' => sub ($c) {
            my $file = Mojo::File->new($dir, $c->param('file'));

            return $c->reply->not_found unless -e $file;

            $c->reply->file($file->to_abs);
        })->name('coverdb_file');
    }
}

1;

=encoding utf8

=head1 NAME

Mojolicious::Plugin::CoverDb - Mojolicious Plugin

=head1 SYNOPSIS

  # Mojolicious
  $self->plugin('CoverDb');
  $self->plugin('CoverDb' => { dir => 'cover_db', route => 'coverdb' });

  # Mojolicious::Lite
  plugin 'CoverDb';
  plugin 'CoverDb' => { dir => 'cover_db', route => 'coverdb' };

=head1 DESCRIPTION

L<Mojolicious::Plugin::CoverDb> is a L<Mojolicious> plugin to conveniently expose a C<cover_db> directory
generated by L<Devel::Cover>. You automatically get C<coverage.html> when going to the exposed route (if
the file exists, otherwise, you’get a list of the files).

As L<Devel::Cover> is a development tool, the directory is only exposed if C<MOJO_MODE> is C<development>.

=head1 OPTIONS

L<Mojolicious::Plugin::StaticCache> supports the following options.

=head2 dir

The directory to expose. It may be an absolute path or a path relative to your mojolicious app’s directory.

Default is C<cover_db>.

=head2 route

The route to the exposed directory.

Default is C<coverdb>.

=head1 METHODS

L<Mojolicious::Plugin::CoverDb> inherits all methods from
L<Mojolicious::Plugin> and implements the following new ones.

=head2 register

  $plugin->register(Mojolicious->new);

Register plugin in L<Mojolicious> application.

=head1 BUGS and SUPPORT

The latest source code can be browsed and fetched at:

  https://framagit.org/fiat-tux/mojolicious/mojolicious-plugin-coverdb
  git clone https://framagit.org/fiat-tux/mojolicious/mojolicious-plugin-coverdb.git

Bugs and feature requests will be tracked at:

  https://framagit.org/fiat-tux/mojolicious/mojolicious-plugin-coverdb/issues

=head1 AUTHOR

  Luc DIDRY
  CPAN ID: LDIDRY
  ldidry@cpan.org
  https://fiat-tux.fr/

=head1 COPYRIGHT

This program is free software; you can redistribute
it and/or modify it under the same terms as Perl itself.

The full text of the license can be found in the
LICENSE file included with this module.

=head1 SEE ALSO

L<Mojolicious>, L<Mojolicious::Guides>, L<https://mojolicious.org>.

=cut

__DATA__

@@ coverdb_index.html.ep
<!DOCTYPE html>
<html>
    <head>
        <title>Availables files in <%= $dir %></title>
    </head>
    <body>
        % if ($files->size) {
            <h1>Availables files in <code><%= $dir %></code></h1>
            <ul>
                % $files->each(sub { my ($e, $num) = @_;
                    <li>
                        %= link_to $e => 'coverdb_file' => { file => $e }
                    </li>
                % });
            </ul>
        % } else {
            <p>Sorry, no file found in <%= $dir %>.</p>
        % }
    </body>
</html>
