use strict;
use warnings;

use Test2::V0;
use Test::Lib;
use Time::HiRes qw/ usleep time /;

use MIDI::RtMidi::FFI::Device;
use MIDI::RtMidi::FFI::TestUtils;

plan skip_all => "Sanity check failed" unless sanity_check;
plan skip_all => 'Cannot open virtual ports on this platform' if no_virtual;

my ( $in, $out ) = ( RtMidiIn->new, RtMidiOut->new );
connect_devices( $in, $out );

$out->set_14bit_mode( 'await' ); # same as 'midi'
my $tests = [
    {
        out => [ [ 0x01, 0x06, 0x1337 ] ],
        in  => [ [ 0x01, 0x06, 0x1337 >> 7 ], [ 0x01, 0x06 | 0x20, 0x1337 & 0x7F ] ],
    },
    {   # Can send our own fine adjust to channel > 31
        out => [ [ 0x01, 0x06 | 0x20, 0x39 ] ],
        in  => [ [ 0x01, 0x06 | 0x20, 0x39 ] ],
    },
    {   # No new MSB
        out => [ [ 0x01, 0x06, 0x1333 ] ],
        in  => [ [ 0x01, 0x06 | 0x20, 0x1333 & 0x7F ] ],
    },
    {
        out => [ [ 0x00, 0x00, 0x0000 ] ],
        in  => [ [ 0x00, 0x00, 0x00 ], [ 0x00, 0x20, 0x00 ] ],
    },
    {
        out => [ [ 0x0F, 0x1F, 0x3FFF ] ],
        in  => [ [ 0x0F, 0x1F, 0x3FFF >> 7 ], [ 0x0F, 0x1F | 0x20, 0x3FFF & 0x7F ] ],
    },
    {   # 7-bit CCs
        out => [ [ 0x01, 0x48, 0x7F ], [ 0x03, 0x6F, 0x00 ], [ 0x0F, 0x5A, 0x5A ] ],
        in  => [ [ 0x01, 0x48, 0x7F ], [ 0x03, 0x6F, 0x00 ], [ 0x0F, 0x5A, 0x5A ] ],
    },
];
test_cc( $in, $out, $tests );

$out->set_14bit_mode( 'pair' );
$tests = [
    {
        out => [ [ 0x01, 0x06, 0x1337 ] ],
        in  => [ [ 0x01, 0x06, 0x1337 >> 7 ], [ 0x01, 0x06 | 0x20, 0x1337 & 0x7F ] ],
    },
    {   # Can send our own fine adjust to channel > 31
        out => [ [ 0x01, 0x06 | 0x20, 0x39 ] ],
        in  => [ [ 0x01, 0x06 | 0x20, 0x39 ] ],
    },
    {   # No MSB change, but we still get complete pair
        out => [ [ 0x01, 0x06, 0x1333 ] ],
        in  => [ [ 0x01, 0x06, 0x1333 >> 7 ],  [ 0x01, 0x06 | 0x20, 0x1333 & 0x7F ] ],
    },
    {
        out => [ [ 0x00, 0x00, 0x0000 ] ],
        in  => [ [ 0x00, 0x00, 0x00 ], [ 0x00, 0x20, 0x00 ] ],
    },
    {
        out => [ [ 0x0F, 0x1F, 0x3FFF ] ],
        in  => [ [ 0x0F, 0x1F, 0x3FFF >> 7 ], [ 0x0F, 0x1F | 0x20, 0x3FFF & 0x7F ] ],
    },
    {   # 7-bit CCs
        out => [ [ 0x01, 0x48, 0x7F ], [ 0x03, 0x6F, 0x00 ], [ 0x0F, 0x5A, 0x5A ] ],
        in  => [ [ 0x01, 0x48, 0x7F ], [ 0x03, 0x6F, 0x00 ], [ 0x0F, 0x5A, 0x5A ] ],
    },
];
test_cc( $in, $out, $tests );

$out->set_14bit_mode( 'backwards' );
$tests = [
    {
        out => [ [ 0x01, 0x06, 0x1337 ] ],
        in  => [ [ 0x01, 0x06 | 0x20, 0x1337 & 0x7F ], [ 0x01, 0x06, 0x1337 >> 7 ] ],
    },
    {   # Can send our own fine adjust to channel > 31
        out => [ [ 0x01, 0x06 | 0x20, 0x39 ] ],
        in  => [ [ 0x01, 0x06 | 0x20, 0x39 ] ],
    },
    {   # No MSB change, but we still get complete pair
        out => [ [ 0x01, 0x06, 0x1333 ] ],
        in  => [ [ 0x01, 0x06 | 0x20, 0x1333 & 0x7F ], [ 0x01, 0x06, 0x1333 >> 7 ] ],
    },
    {
        out => [ [ 0x00, 0x00, 0x0000 ] ],
        in  => [ [ 0x00, 0x20, 0x00 ], [ 0x00, 0x00, 0x00 ] ],
    },
    {
        out => [ [ 0x0F, 0x1F, 0x3FFF ] ],
        in  => [ [ 0x0F, 0x1F | 0x20, 0x3FFF & 0x7F ], [ 0x0F, 0x1F, 0x3FFF >> 7 ] ],
    },
    {   # 7-bit CCs
        out => [ [ 0x01, 0x48, 0x7F ], [ 0x03, 0x6F, 0x00 ], [ 0x0F, 0x5A, 0x5A ] ],
        in  => [ [ 0x01, 0x48, 0x7F ], [ 0x03, 0x6F, 0x00 ], [ 0x0F, 0x5A, 0x5A ] ],
    },
];
test_cc( $in, $out, $tests );

$out->set_14bit_mode( 'backwait' );
$tests = [
    {
        out => [ [ 0x01, 0x06, 0x1337 ] ],
        in  => [ [ 0x01, 0x06 | 0x20, 0x1337 & 0x7F ], [ 0x01, 0x06, 0x1337 >> 7 ] ],
    },
    {   # Can send our own fine adjust to channel > 31
        out => [ [ 0x01, 0x06 | 0x20, 0x39 ] ],
        in  => [ [ 0x01, 0x06 | 0x20, 0x39 ] ],
    },
    {   # No MSB change, get LSB
        out => [ [ 0x01, 0x06, 0x1333 ] ],
        in  => [ [ 0x01, 0x06 | 0x20, 0x1333 & 0x7F ] ],
    },
    {   # MSB Change, get complete pair
        out => [ [ 0x00, 0x00, 0x0000 ] ],
        in  => [ [ 0x00, 0x20, 0x00 ], [ 0x00, 0x00, 0x00 ] ],
    },
    {
        out => [ [ 0x0F, 0x1F, 0x3FFF ] ],
        in  => [ [ 0x0F, 0x1F | 0x20, 0x3FFF & 0x7F ], [ 0x0F, 0x1F, 0x3FFF >> 7 ] ],
    },
    {   # 7-bit CCs
        out => [ [ 0x01, 0x48, 0x7F ], [ 0x03, 0x6F, 0x00 ], [ 0x0F, 0x5A, 0x5A ] ],
        in  => [ [ 0x01, 0x48, 0x7F ], [ 0x03, 0x6F, 0x00 ], [ 0x0F, 0x5A, 0x5A ] ],
    },
];
test_cc( $in, $out, $tests );

# MSB/LSB pair, MSB on high controller
$out->set_14bit_mode( 'doubleback' );
$tests = [
    {
        out => [ [ 0x01, 0x06, 0x1337 ] ],
        in  => [ [ 0x01, 0x06 | 0x20, 0x1337 >> 7 ], [ 0x01, 0x06, 0x1337 & 0x7F ] ],
    },
    {   # No MSB change, but we still get complete pair
        out => [ [ 0x01, 0x06, 0x1333 ] ],
        in  => [ [ 0x01, 0x06 | 0x20, 0x1333 >> 7 ], [ 0x01, 0x06, 0x1333 & 0x7F ] ],
    },
    {
        out => [ [ 0x00, 0x00, 0x0000 ] ],
        in  => [ [ 0x00, 0x20, 0x00 ], [ 0x00, 0x00, 0x00 ] ],
    },
    {
        out => [ [ 0x0F, 0x1F, 0x3FFF ] ],
        in  => [ [ 0x0F, 0x1F | 0x20, 0x3FFF >> 7 ], [ 0x0F, 0x1F, 0x3FFF & 0x7F ] ],
    },
    {   # 7-bit CCs
        out => [ [ 0x01, 0x48, 0x7F ], [ 0x03, 0x6F, 0x00 ], [ 0x0F, 0x5A, 0x5A ] ],
        in  => [ [ 0x01, 0x48, 0x7F ], [ 0x03, 0x6F, 0x00 ], [ 0x0F, 0x5A, 0x5A ] ],
    },
];
test_cc( $in, $out, $tests );

# LSB/MSB pair, MSB on high controller
$out->set_14bit_mode( 'bassack' );
$tests = [
    {
        out => [ [ 0x01, 0x06, 0x1337 ] ],
        in  => [ [ 0x01, 0x06, 0x1337 & 0x7F ], [ 0x01, 0x06 | 0x20, 0x1337 >> 7 ] ],
    },
    {   # No MSB change, but we still get complete pair
        out => [ [ 0x01, 0x06, 0x1333 ] ],
        in  => [ [ 0x01, 0x06, 0x1333 & 0x7F ], [ 0x01, 0x06| 0x20, 0x1333 >> 7 ] ],
    },
    {
        out => [ [ 0x00, 0x00, 0x0000 ] ],
        in  => [ [ 0x00, 0x00, 0x00 ], [ 0x00, 0x20, 0x00 ] ],
    },
    {
        out => [ [ 0x0F, 0x1F, 0x3FFF ] ],
        in  => [ [ 0x0F, 0x1F, 0x3FFF & 0x7F ], [ 0x0F, 0x1F | 0x20, 0x3FFF >> 7 ] ],
    },
    {   # 7-bit CCs
        out => [ [ 0x01, 0x48, 0x7F ], [ 0x03, 0x6F, 0x00 ], [ 0x0F, 0x5A, 0x5A ] ],
        in  => [ [ 0x01, 0x48, 0x7F ], [ 0x03, 0x6F, 0x00 ], [ 0x0F, 0x5A, 0x5A ] ],
    },
];
test_cc( $in, $out, $tests );

# MIDI 1.0 compatible callback
sub callback {
    my ( $device, $channel, $controller, $value ) = @_;
    my $msb = $value >> 7 & 0x7F;
    my $lsb = $value & 0x7F;
    my $last_msb = $device->get_last( control_change => $channel, $controller );
    if ( !defined $last_msb || $last_msb->{ val } != $msb ) {
        $device->send_message_encoded_cb( control_change => $channel, $controller, $msb )
    }
    $device->send_message_encoded_cb( control_change => $channel, $controller | 0x20, $lsb );
}
$out->set_14bit_mode( \&callback );
is ref $out->get_14bit_mode, 'CODE', "14 bit mode is a callback";
$tests = [
    {
        out => [ [ 0x01, 0x06, 0x1337 ] ],
        in  => [ [ 0x01, 0x06, 0x1337 >> 7 ], [ 0x01, 0x06 | 0x20, 0x1337 & 0x7F ] ],
    },
    {   # Can send our own fine adjust to channel > 31
        out => [ [ 0x01, 0x06 | 0x20, 0x39 ] ],
        in  => [ [ 0x01, 0x06 | 0x20, 0x39 ] ],
    },
    {   # No new MSB
        out => [ [ 0x01, 0x06, 0x1333 ] ],
        in  => [ [ 0x01, 0x06 | 0x20, 0x1333 & 0x7F ] ],
    },
    {
        out => [ [ 0x00, 0x00, 0x0000 ] ],
        in  => [ [ 0x00, 0x00, 0x00 ], [ 0x00, 0x20, 0x00 ] ],
    },
    {
        out => [ [ 0x0F, 0x1F, 0x3FFF ] ],
        in  => [ [ 0x0F, 0x1F, 0x3FFF >> 7 ], [ 0x0F, 0x1F | 0x20, 0x3FFF & 0x7F ] ],
    },
    {   # 7-bit CCs
        out => [ [ 0x01, 0x48, 0x7F ], [ 0x03, 0x6F, 0x00 ], [ 0x0F, 0x5A, 0x5A ] ],
        in  => [ [ 0x01, 0x48, 0x7F ], [ 0x03, 0x6F, 0x00 ], [ 0x0F, 0x5A, 0x5A ] ],
    },
];
test_cc( $in, $out, $tests );

$out->set_14bit_mode( 'My son is also named bort' );
my $e = dies { $out->cc( 0x01, 0x01, 0x1337 ) };
like $e, qr/Unknown 14 bit midi mode: My son is also named bort/;

# Fix this - SIGSEGV if don't explicitly tear down instances
undef $in; undef $out;

done_testing;
