package Color::Library::Dictionary::NBS_ISCC::S;

use strict;
use warnings;

use base qw/Color::Library::Dictionary/;

__PACKAGE__->_register_dictionary;

package Color::Library::Dictionary::NBS_ISCC::S;

=pod

=head1 NAME

Color::Library::Dictionary::NBS_ISCC::S - (NBS/ISCC S) Postage-Stamp Color Names

=head1 DESCRIPTION

    Kenneth L. Kelly and Deanne B. Judd.
    "Color: Universal Language and Dictionary of Names",
    National Bureau of Standards,
    Spec. Publ. 440, Dec. 1976, 189 pages.


    Color Name Dictionary: (S) Postage-Stamp Color Names

    Postage-Stamp Color Names,
    William H. Beck 

This dictionary is a hybrid of combinatorial and idiosyncratic styles. 178 names map to 905 colors, using 223 of 267 centroids. Despite covering 84% of the centroids, averaging over 5 colors per name makes this dictionary uselessly unspecific.

L<http://swissnet.ai.mit.edu/~jaffer/Color/Dictionaries#S>

=head1 COLORS

	apple green        applegreen        #8db600

	apple green        applegreen        #7e9f2e

	apple green        applegreen        #44944a

	bistre             bistre            #a67b5b

	bistre             bistre            #8e8279

	bistre             bistre            #e3a857

	bistre             bistre            #be8a3d

	bistre             bistre            #996515

	bistre             bistre            #c19a6b

	bistre             bistre            #826644

	bistre             bistre            #ae9b82

	bistre             bistre            #7e6d5a

	bistre             bistre            #af8d13

	bistre             bistre            #c9ae5d

	bistre             bistre            #ab9144

	bistre             bistre            #f3e5ab

	bistre             bistre            #c2b280

	bistre             bistre            #a18f60

	bistre             bistre            #bfb8a5

	bistre             bistre            #967117

	bistre             bistre            #6c541e

	bistre             bistre            #b9b459

	bistre             bistre            #8c8767

	bistre             bistre            #8a8776

	bistre brown       bistrebrown       #d9a6a9

	bistre brown       bistrebrown       #a67b5b

	bistre brown       bistrebrown       #8e8279

	bistre brown       bistrebrown       #c19a6b

	bistre brown       bistrebrown       #826644

	bistre brown       bistrebrown       #967117

	bistre brown       bistrebrown       #867e36

	black              black             #57554c

	black              black             #b9b8b5

	black              black             #848482

	black              black             #555555

	black              black             #222222

	black brown        blackbrown        #635147

	black brown        blackbrown        #3e322c

	black brown        blackbrown        #8e8279

	black brown        blackbrown        #5b504f

	black brown        blackbrown        #5b5842

	black brown        blackbrown        #57554c

	black violet       blackviolet       #50404d

	black violet       blackviolet       #291e29

	black violet       blackviolet       #673147

	black violet       blackviolet       #38152c

	blue               blue              #66ada4

	blue               blue              #317873

	blue               blue              #239eba

	blue               blue              #007791

	blue               blue              #2e8495

	blue               blue              #9cd1dc

	blue               blue              #66aabc

	blue               blue              #367588

	blue               blue              #004958

	blue               blue              #4997d0

	blue               blue              #0067a5

	blue               blue              #00416a

	blue               blue              #70a3cc

	blue               blue              #436b95

	blue               blue              #00304e

	blue               blue              #bcd4e6

	blue               blue              #536878

	blue black         blueblack         #81878b

	blue black         blueblack         #51585e

	blue gray          bluegray          #b2beb5

	blue gray          bluegray          #81878b

	blue green         bluegreen         #3eb489

	blue green         bluegreen         #007959

	blue green         bluegreen         #8ed1b2

	blue green         bluegreen         #6aab8e

	blue green         bluegreen         #3b7861

	blue green         bluegreen         #008882

	blue green         bluegreen         #00a693

	blue green         bluegreen         #007a74

	blue green         bluegreen         #66ada4

	blue green         bluegreen         #317873

	blue green         bluegreen         #367588

	blue violet        blueviolet        #8791bf

	brick red          brickred          #905d5d

	brick red          brickred          #674c47

	bright blue        brightblue        #007a74

	bright blue        brightblue        #007791

	bright blue        brightblue        #0067a5

	bright blue        brightblue        #436b95

	bright blue green  brightbluegreen   #008856

	bright blue green  brightbluegreen   #3eb489

	bright blue green  brightbluegreen   #007959

	bright green       brightgreen       #27a64c

	bright green       brightgreen       #008856

	bright green       brightgreen       #007959

	bright green       brightgreen       #00543d

	bright red violet  brightredviolet   #9a4eae

	bright red violet  brightredviolet   #875692

	bright red violet  brightredviolet   #870074

	bright red violet  brightredviolet   #9e4f88

	bright rose        brightrose        #de6fa1

	bright rose        brightrose        #ce4676

	bright rose        brightrose        #915f6d

	bright rose violet brightroseviolet  #de6fa1

	bright ultramarine brightultramarine #00a1c2

	bright ultramarine brightultramarine #0067a5

	bright ultramarine brightultramarine #30267a

	bright ultramarine brightultramarine #545aa7

	bright violet      brightviolet      #9065ca

	bright violet      brightviolet      #604e81

	bright violet      brightviolet      #9a4eae

	bright violet      brightviolet      #875692

	bright violet      brightviolet      #b695c0

	bright violet      brightviolet      #86608e

	bright yellow      brightyellow      #ffc14f

	bright yellow      brightyellow      #eaa221

	bright yellow      brightyellow      #fbc97f

	bright yellow      brightyellow      #e3a857

	bright yellow      brightyellow      #fada5e

	bright yellow      brightyellow      #d4af37

	bright yellow      brightyellow      #f8de7e

	bright yellow      brightyellow      #c9ae5d

	brown              brown             #c1b6b3

	brown              brown             #905d5d

	brown              brown             #5c504f

	brown              brown             #cb6d51

	brown              brown             #a87c6d

	brown              brown             #79443b

	brown              brown             #977f73

	brown              brown             #674c47

	brown              brown             #ae6938

	brown              brown             #80461b

	brown              brown             #a67b5b

	brown              brown             #6f4e37

	brown              brown             #958070

	brown              brown             #635147

	brown              brown             #8e8279

	brown              brown             #5b504f

	brown              brown             #ae9b82

	brown              brown             #7e6d5a

	brown bistre       brownbistre       #c19a6b

	brown bistre       brownbistre       #ae9b82

	brown bistre       brownbistre       #c9ae5d

	brown bistre       brownbistre       #967117

	brown-black        brownblack        #5b504f

	brown carmine      browncarmine      #bc3f4a

	brown carmine      browncarmine      #5c0923

	brown carmine      browncarmine      #722f37

	brown lake         brownlake         #ab4e52

	brown lake         brownlake         #905d5d

	brown olive        brownolive        #996515

	brown olive        brownolive        #967117

	brown orange       brownorange       #d9a6a9

	brown orange       brownorange       #cb6d51

	brown orange       brownorange       #b4745e

	brown orange       brownorange       #ed872d

	brown orange       brownorange       #be6516

	brown orange       brownorange       #d99058

	brown orange       brownorange       #ae6938

	brown orange       brownorange       #a67b5b

	brown orange       brownorange       #eaa221

	brown orange       brownorange       #e3a857

	brown orange       brownorange       #c19a6b

	brown orange       brownorange       #c9ae5d

	brown purple       brownpurple       #796878

	brown red          brownred          #be0032

	brown red          brownred          #ad8884

	brown red          brownred          #905d5d

	brown red          brownred          #674c47

	brown violet       brownviolet       #673147

	brown violet       brownviolet       #915f6d

	buff               buff              #f4c2c2

	buff               buff              #d9a6a9

	buff               buff              #ecd5c5

	buff               buff              #c2ac99

	buff               buff              #fab57f

	buff               buff              #d99058

	buff               buff              #e3a857

	buff               buff              #fad6a5

	buff               buff              #c19a6b

	buff               buff              #ae9b82

	buff               buff              #f3e5ab

	buff               buff              #f0ead6

	buff               buff              #bfb8a5

	burnt orange       burntorange       #d9603b

	burnt orange       burntorange       #be6516

	burnt orange       burntorange       #d99058

	burnt orange       burntorange       #ae6938

	carmine            carmine           #ffb5ba

	carmine            carmine           #ea9399

	carmine            carmine           #e4717a

	carmine            carmine           #c08081

	carmine            carmine           #be0032

	carmine            carmine           #bc3f4a

	carmine            carmine           #841b2d

	carmine            carmine           #ab4e52

	carmine            carmine           #905d5d

	carmine            carmine           #ce4676

	carmine            carmine           #b3446c

	carmine            carmine           #78184a

	carmine            carmine           #a8516e

	carmine lake       carminelake       #be0032

	carmine lake       carminelake       #bc3f4a

	carmine lake       carminelake       #ab4e52

	carmine lake       carminelake       #b3446c

	carmine lake       carminelake       #a8516e

	carmine rose       carminerose       #ffb5ba

	carmine rose       carminerose       #ea9399

	carmine rose       carminerose       #e4717a

	carmine rose       carminerose       #be0032

	carmine rose       carminerose       #bc3f4a

	carmine rose       carminerose       #ab4e52

	carmine rose       carminerose       #905d5d

	carmine rose       carminerose       #e66761

	carmine rose       carminerose       #de6fa1

	carmine rose       carminerose       #b3446c

	carmine rose       carminerose       #a8516e

	cerise             cerise            #ce4676

	cerise             cerise            #78184a

	chalky blue        chalkyblue        #91a3b0

	chestnut brown     chestnutbrown     #9e4732

	chocolate          chocolate         #905d5d

	chocolate          chocolate         #8f817f

	chocolate          chocolate         #5c504f

	chocolate          chocolate         #56070c

	chocolate          chocolate         #674c47

	claret             claret            #c17e91

	claret             claret            #915f6d

	claret brown       claretbrown       #841b2d

	claret brown       claretbrown       #722f37

	claret brown       claretbrown       #796878

	crimson            crimson           #be0032

	crimson            crimson           #bc3f4a

	dark blue          darkblue          #2e8495

	dark blue          darkblue          #367588

	dark blue          darkblue          #004958

	dark blue          darkblue          #0067a5

	dark blue          darkblue          #00416a

	dark blue          darkblue          #436b95

	dark blue          darkblue          #00304e

	dark blue          darkblue          #536878

	dark blue          darkblue          #36454f

	dark blue          darkblue          #202830

	dark blue          darkblue          #272458

	dark blue          darkblue          #8791bf

	dark blue green    darkbluegreen     #007959

	dark blue green    darkbluegreen     #1b4d3e

	dark blue green    darkbluegreen     #1c352d

	dark blue green    darkbluegreen     #3a4b47

	dark blue green    darkbluegreen     #1a2421

	dark blue green    darkbluegreen     #007a74

	dark blue green    darkbluegreen     #66ada4

	dark blue green    darkbluegreen     #317873

	dark blue green    darkbluegreen     #367588

	dark brown         darkbrown         #543d3f

	dark brown         darkbrown         #8f817f

	dark brown         darkbrown         #5c504f

	dark brown         darkbrown         #79443b

	dark brown         darkbrown         #977f73

	dark brown         darkbrown         #6f4e37

	dark brown         darkbrown         #635147

	dark brown         darkbrown         #8e8279

	dark brown         darkbrown         #5b504f

	dark brown         darkbrown         #7e6d5a

	dark brown         darkbrown         #bfb8a5

	dark carmine       darkcarmine       #be0032

	dark carmine       darkcarmine       #ab4e52

	dark gray          darkgray          #51585e

	dark green         darkgreen         #00622d

	dark green         darkgreen         #355e3b

	dark green         darkgreen         #007959

	dark green         darkgreen         #00543d

	dark green         darkgreen         #3b7861

	dark green         darkgreen         #1b4d3e

	dark green         darkgreen         #5e716a

	dark green         darkgreen         #3a4b47

	dark green         darkgreen         #7d8984

	dark green         darkgreen         #317873

	dark ochre         darkochre         #967117

	dark olive         darkolive         #5b5842

	dark olive         darkolive         #57554c

	dark olive         darkolive         #515744

	dark olive green   darkolivegreen    #4a5d23

	dark red brown     darkredbrown      #674c47

	dark rose          darkrose          #ab4e52

	dark slate green   darkslategreen    #1b4d3e

	dark slate green   darkslategreen    #1c352d

	dark slate green   darkslategreen    #3a4b47

	dark slate green   darkslategreen    #1a2421

	dark violet        darkviolet        #563c5c

	dark violet        darkviolet        #aa98a9

	dark violet        darkviolet        #796878

	dark violet        darkviolet        #8b8589

	dark violet        darkviolet        #915f6d

	dark violet brown  darkvioletbrown   #50404d

	dark violet brown  darkvioletbrown   #291e29

	dark violet brown  darkvioletbrown   #673147

	dark violet brown  darkvioletbrown   #38152c

	deep bistre        deepbistre        #967117

	deep blue          deepblue          #2e8495

	deep blue          deepblue          #367588

	deep blue          deepblue          #00a1c2

	deep blue          deepblue          #0067a5

	deep blue          deepblue          #00416a

	deep blue          deepblue          #70a3cc

	deep blue          deepblue          #436b95

	deep blue          deepblue          #536878

	deep brown         deepbrown         #79443b

	deep brown         deepbrown         #80461b

	deep brown         deepbrown         #6f4e37

	deep claret        deepclaret        #722f37

	deep green         deepgreen         #7e9f2e

	deep green         deepgreen         #8a9a5b

	deep green         deepgreen         #27a64c

	deep green         deepgreen         #83d37d

	deep green         deepgreen         #44944a

	deep green         deepgreen         #00622d

	deep green         deepgreen         #003118

	deep green         deepgreen         #93c592

	deep green         deepgreen         #007959

	deep green         deepgreen         #00543d

	deep green         deepgreen         #3b7861

	deep green         deepgreen         #7d8984

	deep green         deepgreen         #007a74

	deep green         deepgreen         #66ada4

	deep green         deepgreen         #317873

	deep green         deepgreen         #004b49

	deep lake          deeplake          #673147

	deep magenta       deepmagenta       #ce4676

	deep olive green   deepolivegreen    #4a5d23

	deep orange        deeporange        #f88379

	deep orange        deeporange        #d99058

	deep plum          deepplum          #a8516e

	deep ultramarine   deepultramarine   #00a1c2

	deep ultramarine   deepultramarine   #436b95

	deep ultramarine   deepultramarine   #30267a

	deep violet        deepviolet        #7e73b8

	deep violet        deepviolet        #875692

	deep violet        deepviolet        #86608e

	deep violet        deepviolet        #702963

	dull blue          dullblue          #0067a5

	dull blue          dullblue          #00416a

	dull blue          dullblue          #70a3cc

	dull blue          dullblue          #436b95

	dull blue          dullblue          #536878

	dull blue          dullblue          #51585e

	dull blue          dullblue          #4c516d

	dull buff          dullbuff          #ecd5c5

	dull buff          dullbuff          #c7ada3

	dull buff          dullbuff          #fbc97f

	dull buff          dullbuff          #e3a857

	dull buff          dullbuff          #fad6a5

	dull green         dullgreen         #515744

	dull green         dullgreen         #93c592

	dull green         dullgreen         #355e3b

	dull green         dullgreen         #6aab8e

	dull green         dullgreen         #3b7861

	dull green         dullgreen         #3a4b47

	dull green         dullgreen         #66ada4

	dull lilac         dulllilac         #604e81

	dull lilac         dulllilac         #aa98a9

	dull lilac         dulllilac         #915f6d

	dull orange        dullorange        #c1b6b3

	dull red           dullred           #d9603b

	dull rose          dullrose          #ea9399

	dull rose          dullrose          #ad8884

	dull rose          dullrose          #d597ae

	dull vermilion     dullvermilion     #cb6d51

	dull violet        dullviolet        #796878

	dull violet        dullviolet        #836479

	dull violet        dullviolet        #673147

	dull violet        dullviolet        #af868e

	dull violet        dullviolet        #915f6d

	dull yellow        dullyellow        #fbc97f

	dull yellow        dullyellow        #e3a857

	emerald            emerald           #27a64c

	emerald            emerald           #83d37d

	emerald            emerald           #44944a

	emerald            emerald           #00622d

	emerald            emerald           #008856

	emerald            emerald           #3eb489

	emerald            emerald           #6aab8e

	fawn               fawn              #d9a6a9

	fawn               fawn              #79443b

	gray               gray              #bfb8a5

	gray               gray              #8a8776

	gray               gray              #8da399

	gray               gray              #b2beb5

	gray               gray              #7d8984

	gray               gray              #4e5755

	gray               gray              #e9e9ed

	gray               gray              #51585e

	gray               gray              #bfb9bd

	gray               gray              #b9b8b5

	gray               gray              #848482

	gray bistre        graybistre        #c19a6b

	gray bistre        graybistre        #ae9b82

	gray bistre        graybistre        #967117

	gray black         grayblack         #8e8279

	gray black         grayblack         #8a8776

	gray black         grayblack         #57554c

	gray black         grayblack         #7d8984

	gray black         grayblack         #848482

	gray blue          grayblue          #436b95

	gray blue          grayblue          #91a3b0

	gray blue          grayblue          #b4bcc0

	gray blue          grayblue          #8c92ac

	gray blue          grayblue          #9690ab

	gray blue          grayblue          #bfb9bd

	gray brown         graybrown         #977f73

	gray brown         graybrown         #8e8279

	gray brown         graybrown         #967117

	gray brown         graybrown         #8a8776

	gray green         graygreen         #8a8776

	gray green         graygreen         #8a9a5b

	gray green         graygreen         #8f9779

	gray green         graygreen         #679267

	gray green         graygreen         #6aab8e

	gray green         graygreen         #3b7861

	gray green         graygreen         #66ada4

	gray lilac         graylilac         #8f817f

	gray lilac         graylilac         #8e8279

	gray lilac         graylilac         #d6cadd

	gray lilac         graylilac         #bfb9bd

	gray lilac         graylilac         #efbbcc

	gray violet        grayviolet        #8b8589

	green              green             #7e9f2e

	green              green             #8a9a5b

	green              green             #27a64c

	green              green             #83d37d

	green              green             #44944a

	green              green             #00622d

	green              green             #93c592

	green              green             #679267

	green              green             #008856

	green              green             #3eb489

	green              green             #007959

	green              green             #00543d

	green              green             #8ed1b2

	green              green             #6aab8e

	green              green             #3b7861

	green              green             #c7e6d7

	green              green             #8da399

	green              green             #5e716a

	green              green             #b2beb5

	green              green             #00a693

	green              green             #66ada4

	green              green             #317873

	greenish black     greenishblack     #8da399

	greenish blue      greenishblue      #317873

	greenish blue      greenishblue      #007791

	greenish blue      greenishblue      #367588

	henna brown        hennabrown        #841b2d

	indigo             indigo            #00416a

	indigo             indigo            #4c516d

	lake               lake              #ab4e52

	lake               lake              #722f37

	lake               lake              #905d5d

	lake               lake              #b4745e

	lake               lake              #a8516e

	lemon              lemon             #f3c300

	lemon              lemon             #fada5e

	lemon              lemon             #d4af37

	lemon              lemon             #af8d13

	lemon              lemon             #dcd300

	lemon yellow       lemonyellow       #af8d13

	lemon yellow       lemonyellow       #9b9400

	light blue         lightblue         #96ded1

	light blue         lightblue         #66ada4

	light blue         lightblue         #317873

	light blue         lightblue         #239eba

	light blue         lightblue         #66aabc

	light blue         lightblue         #367588

	light blue         lightblue         #0067a5

	light blue         lightblue         #70a3cc

	light blue         lightblue         #bcd4e6

	light brown        lightbrown        #c7ada3

	light green        lightgreen        #44944a

	light green        lightgreen        #93c592

	light green        lightgreen        #3eb489

	light olive green  lightolivegreen   #867e36

	light olive green  lightolivegreen   #7e9f2e

	light olive green  lightolivegreen   #c9dc89

	light red brown    lightredbrown     #a87c6d

	lilac              lilac             #c1b6b3

	lilac              lilac             #dcd0ff

	lilac              lilac             #8c82b6

	lilac              lilac             #d5badb

	lilac              lilac             #d6cadd

	lilac              lilac             #aa98a9

	lilac              lilac             #796878

	lilac              lilac             #e8e3e5

	lilac              lilac             #bfb9bd

	lilac              lilac             #8b8589

	lilac              lilac             #915c83

	lilac              lilac             #aa8a9e

	lilac              lilac             #e8ccd7

	lilac              lilac             #c3a6b1

	lilac              lilac             #af868e

	lilac              lilac             #915f6d

	lilac brown        lilacbrown        #c1b6b3

	lilac brown        lilacbrown        #8f817f

	lilac brown        lilacbrown        #8e8279

	lilac gray         lilacgray         #bfb9bd

	lilac rose         lilacrose         #e4717a

	lilac rose         lilacrose         #ad8884

	lilac rose         lilacrose         #9e4f88

	lilac rose         lilacrose         #de6fa1

	lilac rose         lilacrose         #efbbcc

	lilac rose         lilacrose         #a8516e

	magenta            magenta           #9065ca

	magenta            magenta           #875692

	magenta            magenta           #9e4f88

	magenta            magenta           #5d3954

	magenta            magenta           #b3446c

	magenta            magenta           #a8516e

	magenta            magenta           #673147

	magenta            magenta           #915f6d

	maroon             maroon            #722f37

	maroon             maroon            #673147

	mauve              mauve             #915c83

	milky blue         milkyblue         #66aabc

	myrtle green       myrtlegreen       #317873

	myrtle green       myrtlegreen       #004b49

	ochre              ochre             #be6516

	ochre              ochre             #d99058

	ochre              ochre             #a67b5b

	ochre              ochre             #ffc14f

	ochre              ochre             #e3a857

	ochre              ochre             #fad6a5

	ochre              ochre             #996515

	ochre              ochre             #fada5e

	ochre              ochre             #d4af37

	ochre              ochre             #f8de7e

	ochre              ochre             #c9ae5d

	olive              olive             #6c541e

	olive              olive             #5b5842

	olive              olive             #8f9779

	olive bistre       olivebistre       #c98500

	olive bistre       olivebistre       #be8a3d

	olive bistre       olivebistre       #996515

	olive bistre       olivebistre       #826644

	olive bistre       olivebistre       #af8d13

	olive bistre       olivebistre       #c9ae5d

	olive bistre       olivebistre       #ab9144

	olive bistre       olivebistre       #f3e5ab

	olive bistre       olivebistre       #c2b280

	olive bistre       olivebistre       #a18f60

	olive bistre       olivebistre       #bfb8a5

	olive bistre       olivebistre       #967117

	olive bistre       olivebistre       #867e36

	olive bistre       olivebistre       #8a8776

	olive bistre       olivebistre       #8f9779

	olive bistre       olivebistre       #b9b8b5

	olive black        oliveblack        #5b5842

	olive black        oliveblack        #57554c

	olive brown        olivebrown        #6f4e37

	olive brown        olivebrown        #8e8279

	olive brown        olivebrown        #4b3621

	olive brown        olivebrown        #7e6d5a

	olive brown        olivebrown        #483c32

	olive brown        olivebrown        #6c541e

	olive brown        olivebrown        #5b5842

	olive brown        olivebrown        #57554c

	olive gray         olivegray         #665d1e

	olive gray         olivegray         #8a8776

	olive gray         olivegray         #57554c

	olive gray         olivegray         #8f9779

	olive gray         olivegray         #7d8984

	olive gray         olivegray         #4e5755

	olive green        olivegreen        #ab9144

	olive green        olivegreen        #f3e5ab

	olive green        olivegreen        #c2b280

	olive green        olivegreen        #a18f60

	olive green        olivegreen        #bfb8a5

	olive green        olivegreen        #9b9400

	olive green        olivegreen        #867e36

	olive green        olivegreen        #665d1e

	olive green        olivegreen        #8c8767

	olive green        olivegreen        #5b5842

	olive green        olivegreen        #8a8776

	olive green        olivegreen        #57554c

	olive green        olivegreen        #7e9f2e

	olive green        olivegreen        #8a9a5b

	olive green        olivegreen        #8f9779

	olive green        olivegreen        #404f00

	olive green        olivegreen        #4a5d23

	olive green        olivegreen        #515744

	olive green        olivegreen        #5e716a

	olive green        olivegreen        #3a4b47

	olive green        olivegreen        #7d8984

	olive green        olivegreen        #4e5755

	olive yellow       oliveyellow       #ecd5c5

	olive yellow       oliveyellow       #c2ac99

	olive yellow       oliveyellow       #fad6a5

	olive yellow       oliveyellow       #f8de7e

	olive yellow       oliveyellow       #c9ae5d

	olive yellow       oliveyellow       #f3e5ab

	olive yellow       oliveyellow       #bfb8a5

	orange             orange            #ffb7a5

	orange             orange            #f88379

	orange             orange            #f4c2c2

	orange             orange            #d9a6a9

	orange             orange            #e25822

	orange             orange            #d9603b

	orange             orange            #cb6d51

	orange             orange            #f38400

	orange             orange            #fd943f

	orange             orange            #ed872d

	orange             orange            #be6516

	orange             orange            #fab57f

	orange             orange            #d99058

	orange             orange            #f6a600

	orange             orange            #ffc14f

	orange             orange            #eaa221

	orange             orange            #c98500

	orange             orange            #fbc97f

	orange             orange            #e3a857

	orange             orange            #be8a3d

	orange             orange            #fad6a5

	orange             orange            #c19a6b

	orange             orange            #fada5e

	orange             orange            #d4af37

	orange             orange            #f8de7e

	orange             orange            #c9ae5d

	orange             orange            #f3e5ab

	orange brown       orangebrown       #722f37

	orange brown       orangebrown       #cb6d51

	orange brown       orangebrown       #b4745e

	orange brown       orangebrown       #882d17

	orange brown       orangebrown       #be6516

	orange brown       orangebrown       #d99058

	orange brown       orangebrown       #ae6938

	orange brown       orangebrown       #80461b

	orange brown       orangebrown       #a67b5b

	orange brown       orangebrown       #6f4e37

	orange brown       orangebrown       #958070

	orange brown       orangebrown       #ae9b82

	orange buff        orangebuff        #d9a6a9

	orange red         orangered         #c08081

	orange red         orangered         #d9603b

	orange red         orangered         #cb6d51

	orange vermilion   orangevermilion   #cb6d51

	orange vermilion   orangevermilion   #be6516

	orange yellow      orangeyellow      #f38400

	orange yellow      orangeyellow      #fd943f

	orange yellow      orangeyellow      #ed872d

	orange yellow      orangeyellow      #fab57f

	orange yellow      orangeyellow      #ffc14f

	orange yellow      orangeyellow      #eaa221

	orange yellow      orangeyellow      #c98500

	orange yellow      orangeyellow      #fbc97f

	orange yellow      orangeyellow      #e3a857

	pale blue          paleblue          #536878

	pale blue          paleblue          #bfb9bd

	pale brown         palebrown         #d99058

	pale brown         palebrown         #a67b5b

	pale gray green    palegraygreen     #dadfb7

	pale green         palegreen         #c9dc89

	pale green         palegreen         #8a9a5b

	pale green         palegreen         #8ed1b2

	pale lilac         palelilac         #e8ccd7

	pale olive green   paleolivegreen    #8a9a5b

	pale yellow-green  paleyellowgreen   #bdda57

	pale yellow-green  paleyellowgreen   #c9dc89

	pale yellow-green  paleyellowgreen   #83d37d

	peacock blue       peacockblue       #007791

	pink               pink              #ea9399

	pink               pink              #e4717a

	pink               pink              #f88379

	pink               pink              #ffc8d6

	pink               pink              #e68fac

	pink               pink              #de6fa1

	pink               pink              #a8516e

	plum               plum              #870074

	plum               plum              #702963

	plum               plum              #78184a

	prussian blue      prussianblue      #2e8495

	prussian blue      prussianblue      #004958

	prussian green     prussiangreen     #007a74

	prussian green     prussiangreen     #00443f

	prussian green     prussiangreen     #317873

	prussian green     prussiangreen     #2e8495

	purple             purple            #272458

	purple             purple            #4e5180

	purple             purple            #604e97

	purple             purple            #604e81

	purple             purple            #554c69

	purple             purple            #9a4eae

	purple             purple            #875692

	purple             purple            #602f6b

	purple             purple            #86608e

	purple             purple            #915c83

	purple             purple            #5d3954

	purple             purple            #836479

	purple brown       purplebrown       #796878

	purplish bistre    purplishbistre    #c2ac99

	red                red               #e4717a

	red                red               #dea5a4

	red                red               #c4aead

	red                red               #bc3f4a

	red                red               #905d5d

	red                red               #f88379

	red                red               #e66761

	red                red               #f4c2c2

	red                red               #d9a6a9

	red                red               #d9603b

	red                red               #aa381e

	red                red               #cb6d51

	red brown          redbrown          #c4aead

	red brown          redbrown          #722f37

	red brown          redbrown          #ad8884

	red brown          redbrown          #905d5d

	red brown          redbrown          #8f817f

	red brown          redbrown          #ecd5c5

	red brown          redbrown          #c7ada3

	red brown          redbrown          #cb6d51

	red brown          redbrown          #9e4732

	red brown          redbrown          #a87c6d

	red brown          redbrown          #79443b

	red brown          redbrown          #be6516

	red brown          redbrown          #d99058

	red brown          redbrown          #ae6938

	red brown          redbrown          #a67b5b

	red brown          redbrown          #6f4e37

	red brown          redbrown          #996515

	red brown          redbrown          #826644

	red brown          redbrown          #6c541e

	reddish brown      reddishbrown      #674c47

	red lilac          redlilac          #c4aead

	red lilac          redlilac          #a8516e

	red lilac          redlilac          #915f6d

	red orange         redorange         #be0032

	red orange         redorange         #ffb7a5

	red orange         redorange         #e25822

	red orange         redorange         #d9603b

	red orange         redorange         #ed872d

	red purple         redpurple         #702963

	red purple         redpurple         #673147

	red violet         redviolet         #9a4eae

	red violet         redviolet         #875692

	red violet         redviolet         #602f6b

	red violet         redviolet         #b695c0

	red violet         redviolet         #86608e

	red violet         redviolet         #9e4f88

	red violet         redviolet         #702963

	red violet         redviolet         #b784a7

	red violet         redviolet         #915c83

	red violet         redviolet         #efbbcc

	red violet         redviolet         #d597ae

	red violet         redviolet         #c3a6b1

	red violet         redviolet         #915f6d

	rose               rose              #ffb5ba

	rose               rose              #ea9399

	rose               rose              #e4717a

	rose               rose              #f9ccca

	rose               rose              #dea5a4

	rose               rose              #c08081

	rose               rose              #bc3f4a

	rose               rose              #841b2d

	rose               rose              #ab4e52

	rose               rose              #905d5d

	rose               rose              #ffb7a5

	rose               rose              #f88379

	rose               rose              #e66761

	rose               rose              #f4c2c2

	rose               rose              #d9a6a9

	rose               rose              #c48379

	rose               rose              #ecd5c5

	rose               rose              #c7ada3

	rose               rose              #cb6d51

	rose               rose              #aa98a9

	rose               rose              #e68fac

	rose               rose              #de6fa1

	rose               rose              #efbbcc

	rose               rose              #d597ae

	rose               rose              #c17e91

	rose               rose              #ce4676

	rose               rose              #b3446c

	rose               rose              #a8516e

	rose               rose              #915f6d

	rose brown         rosebrown         #be0032

	rose brown         rosebrown         #841b2d

	rose brown         rosebrown         #5c0923

	rose brown         rosebrown         #78184a

	rose brown         rosebrown         #54133b

	rose carmine       rosecarmine       #e4717a

	rose carmine       rosecarmine       #ce4676

	rose carmine       rosecarmine       #b3446c

	rose carmine       rosecarmine       #a8516e

	rose lake          roselake          #dea5a4

	rose lake          roselake          #673147

	rose lake          roselake          #915f6d

	rose lilac         roselilac         #9e4f88

	rose lilac         roselilac         #b784a7

	rose lilac         roselilac         #d597ae

	rose lilac         roselilac         #c17e91

	rose lilac         roselilac         #a8516e

	rose lilac         roselilac         #915f6d

	rose pink          rosepink          #ce4676

	rose red           rosered           #e4717a

	rose red           rosered           #ab4e52

	rose red           rosered           #722f37

	rose violet        roseviolet        #9e4f88

	rose violet        roseviolet        #702963

	rose violet        roseviolet        #915c83

	rose violet        roseviolet        #78184a

	rose violet        roseviolet        #a8516e

	royal blue         royalblue         #00416a

	salmon             salmon            #f88379

	salmon             salmon            #f4c2c2

	salmon             salmon            #d9a6a9

	salmon rose        salmonrose        #ea9399

	salmon rose        salmonrose        #f9ccca

	salmon rose        salmonrose        #f88379

	salmon rose        salmonrose        #f4c2c2

	salmon rose        salmonrose        #d9a6a9

	scarlet            scarlet           #e4717a

	scarlet            scarlet           #be0032

	scarlet            scarlet           #bc3f4a

	scarlet            scarlet           #ab4e52

	scarlet            scarlet           #e25822

	sepia              sepia             #8e8279

	slate              slate             #8da399

	slate              slate             #3a4b47

	slate              slate             #7d8984

	slate              slate             #4e5755

	slate              slate             #36454f

	slate              slate             #202830

	slate              slate             #81878b

	slate              slate             #51585e

	slate              slate             #252440

	slate              slate             #4c516d

	slate              slate             #555555

	slate blue         slateblue         #317873

	slate blue         slateblue         #536878

	slate green        slategreen        #8c8767

	slate green        slategreen        #8a8776

	slate green        slategreen        #404f00

	slate green        slategreen        #4a5d23

	slate green        slategreen        #3b7861

	slate green        slategreen        #3a4b47

	slate green        slategreen        #7d8984

	straw              straw             #c9ae5d

	turquoise blue     turquoiseblue     #66aabc

	turquoise green    turquoisegreen    #008882

	turquoise green    turquoisegreen    #00a693

	ultramarine        ultramarine       #00a1c2

	ultramarine        ultramarine       #4997d0

	ultramarine        ultramarine       #0067a5

	ultramarine        ultramarine       #00416a

	ultramarine        ultramarine       #70a3cc

	ultramarine        ultramarine       #436b95

	ultramarine        ultramarine       #91a3b0

	ultramarine        ultramarine       #536878

	ultramarine        ultramarine       #30267a

	ultramarine        ultramarine       #6c79b8

	ultramarine        ultramarine       #545aa7

	ultramarine        ultramarine       #b3bce2

	ultramarine        ultramarine       #8791bf

	ultramarine        ultramarine       #4e5180

	ultramarine        ultramarine       #c0c8e1

	ultramarine        ultramarine       #8c92ac

	ultramarine        ultramarine       #4c516d

	ultramarine        ultramarine       #8c82b6

	umber brown        umberbrown        #635147

	vermilion          vermilion         #c08081

	vermilion          vermilion         #bc3f4a

	vermilion          vermilion         #ab4e52

	vermilion          vermilion         #ad8884

	vermilion          vermilion         #905d5d

	vermilion          vermilion         #f88379

	vermilion          vermilion         #e66761

	vermilion          vermilion         #c48379

	vermilion          vermilion         #d9603b

	vermilion          vermilion         #aa381e

	vermilion          vermilion         #cb6d51

	vermilion          vermilion         #9e4732

	violet             violet            #722f37

	violet             violet            #4e5180

	violet             violet            #7e73b8

	violet             violet            #8c82b6

	violet             violet            #604e81

	violet             violet            #9690ab

	violet             violet            #d399e6

	violet             violet            #875692

	violet             violet            #b695c0

	violet             violet            #86608e

	violet             violet            #d6cadd

	violet             violet            #aa98a9

	violet             violet            #796878

	violet             violet            #50404d

	violet             violet            #702963

	violet             violet            #b784a7

	violet             violet            #aa8a9e

	violet             violet            #836479

	violet             violet            #d597ae

	violet             violet            #e8ccd7

	violet             violet            #c3a6b1

	violet             violet            #af868e

	violet black       violetblack       #543d3f

	violet black       violetblack       #5d555b

	violet blue        violetblue        #545aa7

	violet brown       violetbrown       #905d5d

	violet brown       violetbrown       #796878

	violet brown       violetbrown       #673147

	yellow             yellow            #f4c2c2

	yellow             yellow            #d9a6a9

	yellow             yellow            #f38400

	yellow             yellow            #fd943f

	yellow             yellow            #ed872d

	yellow             yellow            #fab57f

	yellow             yellow            #d99058

	yellow             yellow            #a67b5b

	yellow             yellow            #f6a600

	yellow             yellow            #ffc14f

	yellow             yellow            #eaa221

	yellow             yellow            #c98500

	yellow             yellow            #fbc97f

	yellow             yellow            #e3a857

	yellow             yellow            #be8a3d

	yellow             yellow            #fad6a5

	yellow             yellow            #c19a6b

	yellow             yellow            #f3c300

	yellow             yellow            #fada5e

	yellow             yellow            #d4af37

	yellow             yellow            #af8d13

	yellow             yellow            #f8de7e

	yellow             yellow            #c9ae5d

	yellow             yellow            #ab9144

	yellow             yellow            #f3e5ab

	yellow             yellow            #c2b280

	yellow             yellow            #f0ead6

	yellow brown       yellowbrown       #c2ac99

	yellow brown       yellowbrown       #9e4732

	yellow brown       yellowbrown       #a87c6d

	yellow brown       yellowbrown       #977f73

	yellow brown       yellowbrown       #d99058

	yellow brown       yellowbrown       #ae6938

	yellow brown       yellowbrown       #80461b

	yellow brown       yellowbrown       #a67b5b

	yellow brown       yellowbrown       #be8a3d

	yellow brown       yellowbrown       #996515

	yellow brown       yellowbrown       #867e36

	yellow buff        yellowbuff        #d9a6a9

	yellow buff        yellowbuff        #d99058

	yellow buff        yellowbuff        #e3a857

	yellow buff        yellowbuff        #fad6a5

	yellow buff        yellowbuff        #c19a6b

	yellow buff        yellowbuff        #f3e5ab

	yellow buff        yellowbuff        #c2b280

	yellow green       yellowgreen       #7e9f2e

	yellow green       yellowgreen       #8a9a5b

	yellow green       yellowgreen       #27a64c

	yellow green       yellowgreen       #83d37d

	yellow green       yellowgreen       #44944a

	yellow green       yellowgreen       #00622d

	yellow green       yellowgreen       #b6e5af

	yellow green       yellowgreen       #93c592

	yellow green       yellowgreen       #679267

	yellow green       yellowgreen       #355e3b

	yellow green       yellowgreen       #008856

	yellow green       yellowgreen       #3eb489

	yellow green       yellowgreen       #007959

	yellow green       yellowgreen       #8ed1b2

	yellow green       yellowgreen       #6aab8e

	yellow green       yellowgreen       #3b7861

	yellow orange      yelloworange      #ed872d

	yellow orange      yelloworange      #d99058


=cut

sub _load_color_list() {
    return [
['nbs-iscc-s:applegreen.115','applegreen','apple green',[141,182,0],'8db600',9287168],
['nbs-iscc-s:applegreen.117','applegreen','apple green',[126,159,46],'7e9f2e',8298286],
['nbs-iscc-s:applegreen.131','applegreen','apple green',[68,148,74],'44944a',4494410],
['nbs-iscc-s:bistre.57','bistre','bistre',[166,123,91],'a67b5b',10910555],
['nbs-iscc-s:bistre.63','bistre','bistre',[142,130,121],'8e8279',9339513],
['nbs-iscc-s:bistre.71','bistre','bistre',[227,168,87],'e3a857',14919767],
['nbs-iscc-s:bistre.72','bistre','bistre',[190,138,61],'be8a3d',12487229],
['nbs-iscc-s:bistre.74','bistre','bistre',[153,101,21],'996515',10052885],
['nbs-iscc-s:bistre.76','bistre','bistre',[193,154,107],'c19a6b',12687979],
['nbs-iscc-s:bistre.77','bistre','bistre',[130,102,68],'826644',8545860],
['nbs-iscc-s:bistre.79','bistre','bistre',[174,155,130],'ae9b82',11443074],
['nbs-iscc-s:bistre.80','bistre','bistre',[126,109,90],'7e6d5a',8285530],
['nbs-iscc-s:bistre.85','bistre','bistre',[175,141,19],'af8d13',11504915],
['nbs-iscc-s:bistre.87','bistre','bistre',[201,174,93],'c9ae5d',13217373],
['nbs-iscc-s:bistre.88','bistre','bistre',[171,145,68],'ab9144',11243844],
['nbs-iscc-s:bistre.89','bistre','bistre',[243,229,171],'f3e5ab',15984043],
['nbs-iscc-s:bistre.90','bistre','bistre',[194,178,128],'c2b280',12759680],
['nbs-iscc-s:bistre.91','bistre','bistre',[161,143,96],'a18f60',10588000],
['nbs-iscc-s:bistre.93','bistre','bistre',[191,184,165],'bfb8a5',12564645],
['nbs-iscc-s:bistre.94','bistre','bistre',[150,113,23],'967117',9859351],
['nbs-iscc-s:bistre.95','bistre','bistre',[108,84,30],'6c541e',7099422],
['nbs-iscc-s:bistre.102','bistre','bistre',[185,180,89],'b9b459',12170329],
['nbs-iscc-s:bistre.109','bistre','bistre',[140,135,103],'8c8767',9209703],
['nbs-iscc-s:bistre.112','bistre','bistre',[138,135,118],'8a8776',9078646],
['nbs-iscc-s:bistrebrown.29','bistrebrown','bistre brown',[217,166,169],'d9a6a9',14263977],
['nbs-iscc-s:bistrebrown.57','bistrebrown','bistre brown',[166,123,91],'a67b5b',10910555],
['nbs-iscc-s:bistrebrown.63','bistrebrown','bistre brown',[142,130,121],'8e8279',9339513],
['nbs-iscc-s:bistrebrown.76','bistrebrown','bistre brown',[193,154,107],'c19a6b',12687979],
['nbs-iscc-s:bistrebrown.77','bistrebrown','bistre brown',[130,102,68],'826644',8545860],
['nbs-iscc-s:bistrebrown.94','bistrebrown','bistre brown',[150,113,23],'967117',9859351],
['nbs-iscc-s:bistrebrown.106','bistrebrown','bistre brown',[134,126,54],'867e36',8814134],
['nbs-iscc-s:black.113','black','black',[87,85,76],'57554c',5723468],
['nbs-iscc-s:black.264','black','black',[185,184,181],'b9b8b5',12171445],
['nbs-iscc-s:black.265','black','black',[132,132,130],'848482',8684674],
['nbs-iscc-s:black.266','black','black',[85,85,85],'555555',5592405],
['nbs-iscc-s:black.267','black','black',[34,34,34],'222222',2236962],
['nbs-iscc-s:blackbrown.61','blackbrown','black brown',[99,81,71],'635147',6508871],
['nbs-iscc-s:blackbrown.62','blackbrown','black brown',[62,50,44],'3e322c',4076076],
['nbs-iscc-s:blackbrown.63','blackbrown','black brown',[142,130,121],'8e8279',9339513],
['nbs-iscc-s:blackbrown.64','blackbrown','black brown',[91,80,79],'5b504f',5984335],
['nbs-iscc-s:blackbrown.110','blackbrown','black brown',[91,88,66],'5b5842',5986370],
['nbs-iscc-s:blackbrown.113','blackbrown','black brown',[87,85,76],'57554c',5723468],
['nbs-iscc-s:blackviolet.229','blackviolet','black violet',[80,64,77],'50404d',5259341],
['nbs-iscc-s:blackviolet.230','blackviolet','black violet',[41,30,41],'291e29',2694697],
['nbs-iscc-s:blackviolet.259','blackviolet','black violet',[103,49,71],'673147',6762823],
['nbs-iscc-s:blackviolet.260','blackviolet','black violet',[56,21,44],'38152c',3675436],
['nbs-iscc-s:blue.163','blue','blue',[102,173,164],'66ada4',6729124],
['nbs-iscc-s:blue.164','blue','blue',[49,120,115],'317873',3242099],
['nbs-iscc-s:blue.168','blue','blue',[35,158,186],'239eba',2334394],
['nbs-iscc-s:blue.169','blue','blue',[0,119,145],'007791',30609],
['nbs-iscc-s:blue.170','blue','blue',[46,132,149],'2e8495',3048597],
['nbs-iscc-s:blue.171','blue','blue',[156,209,220],'9cd1dc',10277340],
['nbs-iscc-s:blue.172','blue','blue',[102,170,188],'66aabc',6728380],
['nbs-iscc-s:blue.173','blue','blue',[54,117,136],'367588',3569032],
['nbs-iscc-s:blue.174','blue','blue',[0,73,88],'004958',18776],
['nbs-iscc-s:blue.177','blue','blue',[73,151,208],'4997d0',4822992],
['nbs-iscc-s:blue.178','blue','blue',[0,103,165],'0067a5',26533],
['nbs-iscc-s:blue.179','blue','blue',[0,65,106],'00416a',16746],
['nbs-iscc-s:blue.181','blue','blue',[112,163,204],'70a3cc',7381964],
['nbs-iscc-s:blue.182','blue','blue',[67,107,149],'436b95',4418453],
['nbs-iscc-s:blue.183','blue','blue',[0,48,78],'00304e',12366],
['nbs-iscc-s:blue.184','blue','blue',[188,212,230],'bcd4e6',12375270],
['nbs-iscc-s:blue.186','blue','blue',[83,104,120],'536878',5466232],
['nbs-iscc-s:blueblack.191','blueblack','blue black',[129,135,139],'81878b',8488843],
['nbs-iscc-s:blueblack.192','blueblack','blue black',[81,88,94],'51585e',5331038],
['nbs-iscc-s:bluegray.154','bluegray','blue gray',[178,190,181],'b2beb5',11714229],
['nbs-iscc-s:bluegray.191','bluegray','blue gray',[129,135,139],'81878b',8488843],
['nbs-iscc-s:bluegreen.140','bluegreen','blue green',[62,180,137],'3eb489',4109449],
['nbs-iscc-s:bluegreen.141','bluegreen','blue green',[0,121,89],'007959',31065],
['nbs-iscc-s:bluegreen.143','bluegreen','blue green',[142,209,178],'8ed1b2',9359794],
['nbs-iscc-s:bluegreen.144','bluegreen','blue green',[106,171,142],'6aab8e',6990734],
['nbs-iscc-s:bluegreen.145','bluegreen','blue green',[59,120,97],'3b7861',3897441],
['nbs-iscc-s:bluegreen.158','bluegreen','blue green',[0,136,130],'008882',34946],
['nbs-iscc-s:bluegreen.159','bluegreen','blue green',[0,166,147],'00a693',42643],
['nbs-iscc-s:bluegreen.160','bluegreen','blue green',[0,122,116],'007a74',31348],
['nbs-iscc-s:bluegreen.163','bluegreen','blue green',[102,173,164],'66ada4',6729124],
['nbs-iscc-s:bluegreen.164','bluegreen','blue green',[49,120,115],'317873',3242099],
['nbs-iscc-s:bluegreen.173','bluegreen','blue green',[54,117,136],'367588',3569032],
['nbs-iscc-s:blueviolet.199','blueviolet','blue violet',[135,145,191],'8791bf',8884671],
['nbs-iscc-s:brickred.19','brickred','brick red',[144,93,93],'905d5d',9461085],
['nbs-iscc-s:brickred.46','brickred','brick red',[103,76,71],'674c47',6769735],
['nbs-iscc-s:brightblue.160','brightblue','bright blue',[0,122,116],'007a74',31348],
['nbs-iscc-s:brightblue.169','brightblue','bright blue',[0,119,145],'007791',30609],
['nbs-iscc-s:brightblue.178','brightblue','bright blue',[0,103,165],'0067a5',26533],
['nbs-iscc-s:brightblue.182','brightblue','bright blue',[67,107,149],'436b95',4418453],
['nbs-iscc-s:brightbluegreen.139','brightbluegreen','bright blue green',[0,136,86],'008856',34902],
['nbs-iscc-s:brightbluegreen.140','brightbluegreen','bright blue green',[62,180,137],'3eb489',4109449],
['nbs-iscc-s:brightbluegreen.141','brightbluegreen','bright blue green',[0,121,89],'007959',31065],
['nbs-iscc-s:brightgreen.129','brightgreen','bright green',[39,166,76],'27a64c',2598476],
['nbs-iscc-s:brightgreen.139','brightgreen','bright green',[0,136,86],'008856',34902],
['nbs-iscc-s:brightgreen.141','brightgreen','bright green',[0,121,89],'007959',31065],
['nbs-iscc-s:brightgreen.142','brightgreen','bright green',[0,84,61],'00543d',21565],
['nbs-iscc-s:brightredviolet.216','brightredviolet','bright red violet',[154,78,174],'9a4eae',10112686],
['nbs-iscc-s:brightredviolet.218','brightredviolet','bright red violet',[135,86,146],'875692',8869522],
['nbs-iscc-s:brightredviolet.236','brightredviolet','bright red violet',[135,0,116],'870074',8847476],
['nbs-iscc-s:brightredviolet.237','brightredviolet','bright red violet',[158,79,136],'9e4f88',10375048],
['nbs-iscc-s:brightrose.248','brightrose','bright rose',[222,111,161],'de6fa1',14577569],
['nbs-iscc-s:brightrose.254','brightrose','bright rose',[206,70,118],'ce4676',13518454],
['nbs-iscc-s:brightrose.262','brightrose','bright rose',[145,95,109],'915f6d',9527149],
['nbs-iscc-s:brightroseviolet.248','brightroseviolet','bright rose violet',[222,111,161],'de6fa1',14577569],
['nbs-iscc-s:brightultramarine.176','brightultramarine','bright ultramarine',[0,161,194],'00a1c2',41410],
['nbs-iscc-s:brightultramarine.178','brightultramarine','bright ultramarine',[0,103,165],'0067a5',26533],
['nbs-iscc-s:brightultramarine.194','brightultramarine','bright ultramarine',[48,38,122],'30267a',3155578],
['nbs-iscc-s:brightultramarine.196','brightultramarine','bright ultramarine',[84,90,167],'545aa7',5528231],
['nbs-iscc-s:brightviolet.205','brightviolet','bright violet',[144,101,202],'9065ca',9463242],
['nbs-iscc-s:brightviolet.211','brightviolet','bright violet',[96,78,129],'604e81',6311553],
['nbs-iscc-s:brightviolet.216','brightviolet','bright violet',[154,78,174],'9a4eae',10112686],
['nbs-iscc-s:brightviolet.218','brightviolet','bright violet',[135,86,146],'875692',8869522],
['nbs-iscc-s:brightviolet.222','brightviolet','bright violet',[182,149,192],'b695c0',11965888],
['nbs-iscc-s:brightviolet.223','brightviolet','bright violet',[134,96,142],'86608e',8806542],
['nbs-iscc-s:brightyellow.67','brightyellow','bright yellow',[255,193,79],'ffc14f',16761167],
['nbs-iscc-s:brightyellow.68','brightyellow','bright yellow',[234,162,33],'eaa221',15376929],
['nbs-iscc-s:brightyellow.70','brightyellow','bright yellow',[251,201,127],'fbc97f',16501119],
['nbs-iscc-s:brightyellow.71','brightyellow','bright yellow',[227,168,87],'e3a857',14919767],
['nbs-iscc-s:brightyellow.83','brightyellow','bright yellow',[250,218,94],'fada5e',16439902],
['nbs-iscc-s:brightyellow.84','brightyellow','bright yellow',[212,175,55],'d4af37',13938487],
['nbs-iscc-s:brightyellow.86','brightyellow','bright yellow',[248,222,126],'f8de7e',16309886],
['nbs-iscc-s:brightyellow.87','brightyellow','bright yellow',[201,174,93],'c9ae5d',13217373],
['nbs-iscc-s:brown.10','brown','brown',[193,182,179],'c1b6b3',12695219],
['nbs-iscc-s:brown.19','brown','brown',[144,93,93],'905d5d',9461085],
['nbs-iscc-s:brown.23','brown','brown',[92,80,79],'5c504f',6049871],
['nbs-iscc-s:brown.37','brown','brown',[203,109,81],'cb6d51',13331793],
['nbs-iscc-s:brown.42','brown','brown',[168,124,109],'a87c6d',11041901],
['nbs-iscc-s:brown.43','brown','brown',[121,68,59],'79443b',7947323],
['nbs-iscc-s:brown.45','brown','brown',[151,127,115],'977f73',9928563],
['nbs-iscc-s:brown.46','brown','brown',[103,76,71],'674c47',6769735],
['nbs-iscc-s:brown.54','brown','brown',[174,105,56],'ae6938',11430200],
['nbs-iscc-s:brown.55','brown','brown',[128,70,27],'80461b',8406555],
['nbs-iscc-s:brown.57','brown','brown',[166,123,91],'a67b5b',10910555],
['nbs-iscc-s:brown.58','brown','brown',[111,78,55],'6f4e37',7294519],
['nbs-iscc-s:brown.60','brown','brown',[149,128,112],'958070',9797744],
['nbs-iscc-s:brown.61','brown','brown',[99,81,71],'635147',6508871],
['nbs-iscc-s:brown.63','brown','brown',[142,130,121],'8e8279',9339513],
['nbs-iscc-s:brown.64','brown','brown',[91,80,79],'5b504f',5984335],
['nbs-iscc-s:brown.79','brown','brown',[174,155,130],'ae9b82',11443074],
['nbs-iscc-s:brown.80','brown','brown',[126,109,90],'7e6d5a',8285530],
['nbs-iscc-s:brownbistre.76','brownbistre','brown bistre',[193,154,107],'c19a6b',12687979],
['nbs-iscc-s:brownbistre.79','brownbistre','brown bistre',[174,155,130],'ae9b82',11443074],
['nbs-iscc-s:brownbistre.87','brownbistre','brown bistre',[201,174,93],'c9ae5d',13217373],
['nbs-iscc-s:brownbistre.94','brownbistre','brown bistre',[150,113,23],'967117',9859351],
['nbs-iscc-s:brownblack.64','brownblack','brown-black',[91,80,79],'5b504f',5984335],
['nbs-iscc-s:browncarmine.12','browncarmine','brown carmine',[188,63,74],'bc3f4a',12336970],
['nbs-iscc-s:browncarmine.14','browncarmine','brown carmine',[92,9,35],'5c0923',6031651],
['nbs-iscc-s:browncarmine.16','browncarmine','brown carmine',[114,47,55],'722f37',7483191],
['nbs-iscc-s:brownlake.15','brownlake','brown lake',[171,78,82],'ab4e52',11226706],
['nbs-iscc-s:brownlake.19','brownlake','brown lake',[144,93,93],'905d5d',9461085],
['nbs-iscc-s:brownolive.74','brownolive','brown olive',[153,101,21],'996515',10052885],
['nbs-iscc-s:brownolive.94','brownolive','brown olive',[150,113,23],'967117',9859351],
['nbs-iscc-s:brownorange.29','brownorange','brown orange',[217,166,169],'d9a6a9',14263977],
['nbs-iscc-s:brownorange.37','brownorange','brown orange',[203,109,81],'cb6d51',13331793],
['nbs-iscc-s:brownorange.39','brownorange','brown orange',[180,116,94],'b4745e',11826270],
['nbs-iscc-s:brownorange.50','brownorange','brown orange',[237,135,45],'ed872d',15566637],
['nbs-iscc-s:brownorange.51','brownorange','brown orange',[190,101,22],'be6516',12477718],
['nbs-iscc-s:brownorange.53','brownorange','brown orange',[217,144,88],'d99058',14258264],
['nbs-iscc-s:brownorange.54','brownorange','brown orange',[174,105,56],'ae6938',11430200],
['nbs-iscc-s:brownorange.57','brownorange','brown orange',[166,123,91],'a67b5b',10910555],
['nbs-iscc-s:brownorange.68','brownorange','brown orange',[234,162,33],'eaa221',15376929],
['nbs-iscc-s:brownorange.71','brownorange','brown orange',[227,168,87],'e3a857',14919767],
['nbs-iscc-s:brownorange.76','brownorange','brown orange',[193,154,107],'c19a6b',12687979],
['nbs-iscc-s:brownorange.87','brownorange','brown orange',[201,174,93],'c9ae5d',13217373],
['nbs-iscc-s:brownpurple.228','brownpurple','brown purple',[121,104,120],'796878',7956600],
['nbs-iscc-s:brownred.11','brownred','brown red',[190,0,50],'be0032',12451890],
['nbs-iscc-s:brownred.18','brownred','brown red',[173,136,132],'ad8884',11372676],
['nbs-iscc-s:brownred.19','brownred','brown red',[144,93,93],'905d5d',9461085],
['nbs-iscc-s:brownred.46','brownred','brown red',[103,76,71],'674c47',6769735],
['nbs-iscc-s:brownviolet.259','brownviolet','brown violet',[103,49,71],'673147',6762823],
['nbs-iscc-s:brownviolet.262','brownviolet','brown violet',[145,95,109],'915f6d',9527149],
['nbs-iscc-s:buff.28','buff','buff',[244,194,194],'f4c2c2',16040642],
['nbs-iscc-s:buff.29','buff','buff',[217,166,169],'d9a6a9',14263977],
['nbs-iscc-s:buff.31','buff','buff',[236,213,197],'ecd5c5',15521221],
['nbs-iscc-s:buff.33','buff','buff',[194,172,153],'c2ac99',12758169],
['nbs-iscc-s:buff.52','buff','buff',[250,181,127],'fab57f',16430463],
['nbs-iscc-s:buff.53','buff','buff',[217,144,88],'d99058',14258264],
['nbs-iscc-s:buff.71','buff','buff',[227,168,87],'e3a857',14919767],
['nbs-iscc-s:buff.73','buff','buff',[250,214,165],'fad6a5',16438949],
['nbs-iscc-s:buff.76','buff','buff',[193,154,107],'c19a6b',12687979],
['nbs-iscc-s:buff.79','buff','buff',[174,155,130],'ae9b82',11443074],
['nbs-iscc-s:buff.89','buff','buff',[243,229,171],'f3e5ab',15984043],
['nbs-iscc-s:buff.92','buff','buff',[240,234,214],'f0ead6',15788758],
['nbs-iscc-s:buff.93','buff','buff',[191,184,165],'bfb8a5',12564645],
['nbs-iscc-s:burntorange.35','burntorange','burnt orange',[217,96,59],'d9603b',14245947],
['nbs-iscc-s:burntorange.51','burntorange','burnt orange',[190,101,22],'be6516',12477718],
['nbs-iscc-s:burntorange.53','burntorange','burnt orange',[217,144,88],'d99058',14258264],
['nbs-iscc-s:burntorange.54','burntorange','burnt orange',[174,105,56],'ae6938',11430200],
['nbs-iscc-s:carmine.1','carmine','carmine',[255,181,186],'ffb5ba',16758202],
['nbs-iscc-s:carmine.2','carmine','carmine',[234,147,153],'ea9399',15373209],
['nbs-iscc-s:carmine.3','carmine','carmine',[228,113,122],'e4717a',14971258],
['nbs-iscc-s:carmine.6','carmine','carmine',[192,128,129],'c08081',12615809],
['nbs-iscc-s:carmine.11','carmine','carmine',[190,0,50],'be0032',12451890],
['nbs-iscc-s:carmine.12','carmine','carmine',[188,63,74],'bc3f4a',12336970],
['nbs-iscc-s:carmine.13','carmine','carmine',[132,27,45],'841b2d',8657709],
['nbs-iscc-s:carmine.15','carmine','carmine',[171,78,82],'ab4e52',11226706],
['nbs-iscc-s:carmine.19','carmine','carmine',[144,93,93],'905d5d',9461085],
['nbs-iscc-s:carmine.254','carmine','carmine',[206,70,118],'ce4676',13518454],
['nbs-iscc-s:carmine.255','carmine','carmine',[179,68,108],'b3446c',11748460],
['nbs-iscc-s:carmine.256','carmine','carmine',[120,24,74],'78184a',7870538],
['nbs-iscc-s:carmine.258','carmine','carmine',[168,81,110],'a8516e',11030894],
['nbs-iscc-s:carminelake.11','carminelake','carmine lake',[190,0,50],'be0032',12451890],
['nbs-iscc-s:carminelake.12','carminelake','carmine lake',[188,63,74],'bc3f4a',12336970],
['nbs-iscc-s:carminelake.15','carminelake','carmine lake',[171,78,82],'ab4e52',11226706],
['nbs-iscc-s:carminelake.255','carminelake','carmine lake',[179,68,108],'b3446c',11748460],
['nbs-iscc-s:carminelake.258','carminelake','carmine lake',[168,81,110],'a8516e',11030894],
['nbs-iscc-s:carminerose.1','carminerose','carmine rose',[255,181,186],'ffb5ba',16758202],
['nbs-iscc-s:carminerose.2','carminerose','carmine rose',[234,147,153],'ea9399',15373209],
['nbs-iscc-s:carminerose.3','carminerose','carmine rose',[228,113,122],'e4717a',14971258],
['nbs-iscc-s:carminerose.11','carminerose','carmine rose',[190,0,50],'be0032',12451890],
['nbs-iscc-s:carminerose.12','carminerose','carmine rose',[188,63,74],'bc3f4a',12336970],
['nbs-iscc-s:carminerose.15','carminerose','carmine rose',[171,78,82],'ab4e52',11226706],
['nbs-iscc-s:carminerose.19','carminerose','carmine rose',[144,93,93],'905d5d',9461085],
['nbs-iscc-s:carminerose.27','carminerose','carmine rose',[230,103,97],'e66761',15099745],
['nbs-iscc-s:carminerose.248','carminerose','carmine rose',[222,111,161],'de6fa1',14577569],
['nbs-iscc-s:carminerose.255','carminerose','carmine rose',[179,68,108],'b3446c',11748460],
['nbs-iscc-s:carminerose.258','carminerose','carmine rose',[168,81,110],'a8516e',11030894],
['nbs-iscc-s:cerise.254','cerise','cerise',[206,70,118],'ce4676',13518454],
['nbs-iscc-s:cerise.256','cerise','cerise',[120,24,74],'78184a',7870538],
['nbs-iscc-s:chalkyblue.185','chalkyblue','chalky blue',[145,163,176],'91a3b0',9544624],
['nbs-iscc-s:chestnutbrown.38','chestnutbrown','chestnut brown',[158,71,50],'9e4732',10372914],
['nbs-iscc-s:chocolate.19','chocolate','chocolate',[144,93,93],'905d5d',9461085],
['nbs-iscc-s:chocolate.22','chocolate','chocolate',[143,129,127],'8f817f',9404799],
['nbs-iscc-s:chocolate.23','chocolate','chocolate',[92,80,79],'5c504f',6049871],
['nbs-iscc-s:chocolate.41','chocolate','chocolate',[86,7,12],'56070c',5637900],
['nbs-iscc-s:chocolate.46','chocolate','chocolate',[103,76,71],'674c47',6769735],
['nbs-iscc-s:claret.251','claret','claret',[193,126,145],'c17e91',12680849],
['nbs-iscc-s:claret.262','claret','claret',[145,95,109],'915f6d',9527149],
['nbs-iscc-s:claretbrown.13','claretbrown','claret brown',[132,27,45],'841b2d',8657709],
['nbs-iscc-s:claretbrown.16','claretbrown','claret brown',[114,47,55],'722f37',7483191],
['nbs-iscc-s:claretbrown.228','claretbrown','claret brown',[121,104,120],'796878',7956600],
['nbs-iscc-s:crimson.11','crimson','crimson',[190,0,50],'be0032',12451890],
['nbs-iscc-s:crimson.12','crimson','crimson',[188,63,74],'bc3f4a',12336970],
['nbs-iscc-s:darkblue.170','darkblue','dark blue',[46,132,149],'2e8495',3048597],
['nbs-iscc-s:darkblue.173','darkblue','dark blue',[54,117,136],'367588',3569032],
['nbs-iscc-s:darkblue.174','darkblue','dark blue',[0,73,88],'004958',18776],
['nbs-iscc-s:darkblue.178','darkblue','dark blue',[0,103,165],'0067a5',26533],
['nbs-iscc-s:darkblue.179','darkblue','dark blue',[0,65,106],'00416a',16746],
['nbs-iscc-s:darkblue.182','darkblue','dark blue',[67,107,149],'436b95',4418453],
['nbs-iscc-s:darkblue.183','darkblue','dark blue',[0,48,78],'00304e',12366],
['nbs-iscc-s:darkblue.186','darkblue','dark blue',[83,104,120],'536878',5466232],
['nbs-iscc-s:darkblue.187','darkblue','dark blue',[54,69,79],'36454f',3556687],
['nbs-iscc-s:darkblue.188','darkblue','dark blue',[32,40,48],'202830',2107440],
['nbs-iscc-s:darkblue.197','darkblue','dark blue',[39,36,88],'272458',2565208],
['nbs-iscc-s:darkblue.199','darkblue','dark blue',[135,145,191],'8791bf',8884671],
['nbs-iscc-s:darkbluegreen.141','darkbluegreen','dark blue green',[0,121,89],'007959',31065],
['nbs-iscc-s:darkbluegreen.146','darkbluegreen','dark blue green',[27,77,62],'1b4d3e',1789246],
['nbs-iscc-s:darkbluegreen.147','darkbluegreen','dark blue green',[28,53,45],'1c352d',1848621],
['nbs-iscc-s:darkbluegreen.151','darkbluegreen','dark blue green',[58,75,71],'3a4b47',3820359],
['nbs-iscc-s:darkbluegreen.152','darkbluegreen','dark blue green',[26,36,33],'1a2421',1713185],
['nbs-iscc-s:darkbluegreen.160','darkbluegreen','dark blue green',[0,122,116],'007a74',31348],
['nbs-iscc-s:darkbluegreen.163','darkbluegreen','dark blue green',[102,173,164],'66ada4',6729124],
['nbs-iscc-s:darkbluegreen.164','darkbluegreen','dark blue green',[49,120,115],'317873',3242099],
['nbs-iscc-s:darkbluegreen.173','darkbluegreen','dark blue green',[54,117,136],'367588',3569032],
['nbs-iscc-s:darkbrown.20','darkbrown','dark brown',[84,61,63],'543d3f',5520703],
['nbs-iscc-s:darkbrown.22','darkbrown','dark brown',[143,129,127],'8f817f',9404799],
['nbs-iscc-s:darkbrown.23','darkbrown','dark brown',[92,80,79],'5c504f',6049871],
['nbs-iscc-s:darkbrown.43','darkbrown','dark brown',[121,68,59],'79443b',7947323],
['nbs-iscc-s:darkbrown.45','darkbrown','dark brown',[151,127,115],'977f73',9928563],
['nbs-iscc-s:darkbrown.58','darkbrown','dark brown',[111,78,55],'6f4e37',7294519],
['nbs-iscc-s:darkbrown.61','darkbrown','dark brown',[99,81,71],'635147',6508871],
['nbs-iscc-s:darkbrown.63','darkbrown','dark brown',[142,130,121],'8e8279',9339513],
['nbs-iscc-s:darkbrown.64','darkbrown','dark brown',[91,80,79],'5b504f',5984335],
['nbs-iscc-s:darkbrown.80','darkbrown','dark brown',[126,109,90],'7e6d5a',8285530],
['nbs-iscc-s:darkbrown.93','darkbrown','dark brown',[191,184,165],'bfb8a5',12564645],
['nbs-iscc-s:darkcarmine.11','darkcarmine','dark carmine',[190,0,50],'be0032',12451890],
['nbs-iscc-s:darkcarmine.15','darkcarmine','dark carmine',[171,78,82],'ab4e52',11226706],
['nbs-iscc-s:darkgray.192','darkgray','dark gray',[81,88,94],'51585e',5331038],
['nbs-iscc-s:darkgreen.132','darkgreen','dark green',[0,98,45],'00622d',25133],
['nbs-iscc-s:darkgreen.137','darkgreen','dark green',[53,94,59],'355e3b',3497531],
['nbs-iscc-s:darkgreen.141','darkgreen','dark green',[0,121,89],'007959',31065],
['nbs-iscc-s:darkgreen.142','darkgreen','dark green',[0,84,61],'00543d',21565],
['nbs-iscc-s:darkgreen.145','darkgreen','dark green',[59,120,97],'3b7861',3897441],
['nbs-iscc-s:darkgreen.146','darkgreen','dark green',[27,77,62],'1b4d3e',1789246],
['nbs-iscc-s:darkgreen.150','darkgreen','dark green',[94,113,106],'5e716a',6189418],
['nbs-iscc-s:darkgreen.151','darkgreen','dark green',[58,75,71],'3a4b47',3820359],
['nbs-iscc-s:darkgreen.155','darkgreen','dark green',[125,137,132],'7d8984',8227204],
['nbs-iscc-s:darkgreen.164','darkgreen','dark green',[49,120,115],'317873',3242099],
['nbs-iscc-s:darkochre.94','darkochre','dark ochre',[150,113,23],'967117',9859351],
['nbs-iscc-s:darkolive.110','darkolive','dark olive',[91,88,66],'5b5842',5986370],
['nbs-iscc-s:darkolive.113','darkolive','dark olive',[87,85,76],'57554c',5723468],
['nbs-iscc-s:darkolive.127','darkolive','dark olive',[81,87,68],'515744',5330756],
['nbs-iscc-s:darkolivegreen.125','darkolivegreen','dark olive green',[74,93,35],'4a5d23',4873507],
['nbs-iscc-s:darkredbrown.46','darkredbrown','dark red brown',[103,76,71],'674c47',6769735],
['nbs-iscc-s:darkrose.15','darkrose','dark rose',[171,78,82],'ab4e52',11226706],
['nbs-iscc-s:darkslategreen.146','darkslategreen','dark slate green',[27,77,62],'1b4d3e',1789246],
['nbs-iscc-s:darkslategreen.147','darkslategreen','dark slate green',[28,53,45],'1c352d',1848621],
['nbs-iscc-s:darkslategreen.151','darkslategreen','dark slate green',[58,75,71],'3a4b47',3820359],
['nbs-iscc-s:darkslategreen.152','darkslategreen','dark slate green',[26,36,33],'1a2421',1713185],
['nbs-iscc-s:darkviolet.224','darkviolet','dark violet',[86,60,92],'563c5c',5651548],
['nbs-iscc-s:darkviolet.227','darkviolet','dark violet',[170,152,169],'aa98a9',11180201],
['nbs-iscc-s:darkviolet.228','darkviolet','dark violet',[121,104,120],'796878',7956600],
['nbs-iscc-s:darkviolet.233','darkviolet','dark violet',[139,133,137],'8b8589',9143689],
['nbs-iscc-s:darkviolet.262','darkviolet','dark violet',[145,95,109],'915f6d',9527149],
['nbs-iscc-s:darkvioletbrown.229','darkvioletbrown','dark violet brown',[80,64,77],'50404d',5259341],
['nbs-iscc-s:darkvioletbrown.230','darkvioletbrown','dark violet brown',[41,30,41],'291e29',2694697],
['nbs-iscc-s:darkvioletbrown.259','darkvioletbrown','dark violet brown',[103,49,71],'673147',6762823],
['nbs-iscc-s:darkvioletbrown.260','darkvioletbrown','dark violet brown',[56,21,44],'38152c',3675436],
['nbs-iscc-s:deepbistre.94','deepbistre','deep bistre',[150,113,23],'967117',9859351],
['nbs-iscc-s:deepblue.170','deepblue','deep blue',[46,132,149],'2e8495',3048597],
['nbs-iscc-s:deepblue.173','deepblue','deep blue',[54,117,136],'367588',3569032],
['nbs-iscc-s:deepblue.176','deepblue','deep blue',[0,161,194],'00a1c2',41410],
['nbs-iscc-s:deepblue.178','deepblue','deep blue',[0,103,165],'0067a5',26533],
['nbs-iscc-s:deepblue.179','deepblue','deep blue',[0,65,106],'00416a',16746],
['nbs-iscc-s:deepblue.181','deepblue','deep blue',[112,163,204],'70a3cc',7381964],
['nbs-iscc-s:deepblue.182','deepblue','deep blue',[67,107,149],'436b95',4418453],
['nbs-iscc-s:deepblue.186','deepblue','deep blue',[83,104,120],'536878',5466232],
['nbs-iscc-s:deepbrown.43','deepbrown','deep brown',[121,68,59],'79443b',7947323],
['nbs-iscc-s:deepbrown.55','deepbrown','deep brown',[128,70,27],'80461b',8406555],
['nbs-iscc-s:deepbrown.58','deepbrown','deep brown',[111,78,55],'6f4e37',7294519],
['nbs-iscc-s:deepclaret.16','deepclaret','deep claret',[114,47,55],'722f37',7483191],
['nbs-iscc-s:deepgreen.117','deepgreen','deep green',[126,159,46],'7e9f2e',8298286],
['nbs-iscc-s:deepgreen.120','deepgreen','deep green',[138,154,91],'8a9a5b',9083483],
['nbs-iscc-s:deepgreen.129','deepgreen','deep green',[39,166,76],'27a64c',2598476],
['nbs-iscc-s:deepgreen.130','deepgreen','deep green',[131,211,125],'83d37d',8639357],
['nbs-iscc-s:deepgreen.131','deepgreen','deep green',[68,148,74],'44944a',4494410],
['nbs-iscc-s:deepgreen.132','deepgreen','deep green',[0,98,45],'00622d',25133],
['nbs-iscc-s:deepgreen.133','deepgreen','deep green',[0,49,24],'003118',12568],
['nbs-iscc-s:deepgreen.135','deepgreen','deep green',[147,197,146],'93c592',9684370],
['nbs-iscc-s:deepgreen.141','deepgreen','deep green',[0,121,89],'007959',31065],
['nbs-iscc-s:deepgreen.142','deepgreen','deep green',[0,84,61],'00543d',21565],
['nbs-iscc-s:deepgreen.145','deepgreen','deep green',[59,120,97],'3b7861',3897441],
['nbs-iscc-s:deepgreen.155','deepgreen','deep green',[125,137,132],'7d8984',8227204],
['nbs-iscc-s:deepgreen.160','deepgreen','deep green',[0,122,116],'007a74',31348],
['nbs-iscc-s:deepgreen.163','deepgreen','deep green',[102,173,164],'66ada4',6729124],
['nbs-iscc-s:deepgreen.164','deepgreen','deep green',[49,120,115],'317873',3242099],
['nbs-iscc-s:deepgreen.165','deepgreen','deep green',[0,75,73],'004b49',19273],
['nbs-iscc-s:deeplake.259','deeplake','deep lake',[103,49,71],'673147',6762823],
['nbs-iscc-s:deepmagenta.254','deepmagenta','deep magenta',[206,70,118],'ce4676',13518454],
['nbs-iscc-s:deepolivegreen.125','deepolivegreen','deep olive green',[74,93,35],'4a5d23',4873507],
['nbs-iscc-s:deeporange.26','deeporange','deep orange',[248,131,121],'f88379',16286585],
['nbs-iscc-s:deeporange.53','deeporange','deep orange',[217,144,88],'d99058',14258264],
['nbs-iscc-s:deepplum.258','deepplum','deep plum',[168,81,110],'a8516e',11030894],
['nbs-iscc-s:deepultramarine.176','deepultramarine','deep ultramarine',[0,161,194],'00a1c2',41410],
['nbs-iscc-s:deepultramarine.182','deepultramarine','deep ultramarine',[67,107,149],'436b95',4418453],
['nbs-iscc-s:deepultramarine.194','deepultramarine','deep ultramarine',[48,38,122],'30267a',3155578],
['nbs-iscc-s:deepviolet.206','deepviolet','deep violet',[126,115,184],'7e73b8',8287160],
['nbs-iscc-s:deepviolet.218','deepviolet','deep violet',[135,86,146],'875692',8869522],
['nbs-iscc-s:deepviolet.223','deepviolet','deep violet',[134,96,142],'86608e',8806542],
['nbs-iscc-s:deepviolet.238','deepviolet','deep violet',[112,41,99],'702963',7350627],
['nbs-iscc-s:dullblue.178','dullblue','dull blue',[0,103,165],'0067a5',26533],
['nbs-iscc-s:dullblue.179','dullblue','dull blue',[0,65,106],'00416a',16746],
['nbs-iscc-s:dullblue.181','dullblue','dull blue',[112,163,204],'70a3cc',7381964],
['nbs-iscc-s:dullblue.182','dullblue','dull blue',[67,107,149],'436b95',4418453],
['nbs-iscc-s:dullblue.186','dullblue','dull blue',[83,104,120],'536878',5466232],
['nbs-iscc-s:dullblue.192','dullblue','dull blue',[81,88,94],'51585e',5331038],
['nbs-iscc-s:dullblue.204','dullblue','dull blue',[76,81,109],'4c516d',5001581],
['nbs-iscc-s:dullbuff.31','dullbuff','dull buff',[236,213,197],'ecd5c5',15521221],
['nbs-iscc-s:dullbuff.32','dullbuff','dull buff',[199,173,163],'c7ada3',13086115],
['nbs-iscc-s:dullbuff.70','dullbuff','dull buff',[251,201,127],'fbc97f',16501119],
['nbs-iscc-s:dullbuff.71','dullbuff','dull buff',[227,168,87],'e3a857',14919767],
['nbs-iscc-s:dullbuff.73','dullbuff','dull buff',[250,214,165],'fad6a5',16438949],
['nbs-iscc-s:dullgreen.127','dullgreen','dull green',[81,87,68],'515744',5330756],
['nbs-iscc-s:dullgreen.135','dullgreen','dull green',[147,197,146],'93c592',9684370],
['nbs-iscc-s:dullgreen.137','dullgreen','dull green',[53,94,59],'355e3b',3497531],
['nbs-iscc-s:dullgreen.144','dullgreen','dull green',[106,171,142],'6aab8e',6990734],
['nbs-iscc-s:dullgreen.145','dullgreen','dull green',[59,120,97],'3b7861',3897441],
['nbs-iscc-s:dullgreen.151','dullgreen','dull green',[58,75,71],'3a4b47',3820359],
['nbs-iscc-s:dullgreen.163','dullgreen','dull green',[102,173,164],'66ada4',6729124],
['nbs-iscc-s:dulllilac.211','dulllilac','dull lilac',[96,78,129],'604e81',6311553],
['nbs-iscc-s:dulllilac.227','dulllilac','dull lilac',[170,152,169],'aa98a9',11180201],
['nbs-iscc-s:dulllilac.262','dulllilac','dull lilac',[145,95,109],'915f6d',9527149],
['nbs-iscc-s:dullorange.10','dullorange','dull orange',[193,182,179],'c1b6b3',12695219],
['nbs-iscc-s:dullred.35','dullred','dull red',[217,96,59],'d9603b',14245947],
['nbs-iscc-s:dullrose.2','dullrose','dull rose',[234,147,153],'ea9399',15373209],
['nbs-iscc-s:dullrose.18','dullrose','dull rose',[173,136,132],'ad8884',11372676],
['nbs-iscc-s:dullrose.250','dullrose','dull rose',[213,151,174],'d597ae',13997998],
['nbs-iscc-s:dullvermilion.37','dullvermilion','dull vermilion',[203,109,81],'cb6d51',13331793],
['nbs-iscc-s:dullviolet.228','dullviolet','dull violet',[121,104,120],'796878',7956600],
['nbs-iscc-s:dullviolet.245','dullviolet','dull violet',[131,100,121],'836479',8610937],
['nbs-iscc-s:dullviolet.259','dullviolet','dull violet',[103,49,71],'673147',6762823],
['nbs-iscc-s:dullviolet.261','dullviolet','dull violet',[175,134,142],'af868e',11503246],
['nbs-iscc-s:dullviolet.262','dullviolet','dull violet',[145,95,109],'915f6d',9527149],
['nbs-iscc-s:dullyellow.70','dullyellow','dull yellow',[251,201,127],'fbc97f',16501119],
['nbs-iscc-s:dullyellow.71','dullyellow','dull yellow',[227,168,87],'e3a857',14919767],
['nbs-iscc-s:emerald.129','emerald','emerald',[39,166,76],'27a64c',2598476],
['nbs-iscc-s:emerald.130','emerald','emerald',[131,211,125],'83d37d',8639357],
['nbs-iscc-s:emerald.131','emerald','emerald',[68,148,74],'44944a',4494410],
['nbs-iscc-s:emerald.132','emerald','emerald',[0,98,45],'00622d',25133],
['nbs-iscc-s:emerald.139','emerald','emerald',[0,136,86],'008856',34902],
['nbs-iscc-s:emerald.140','emerald','emerald',[62,180,137],'3eb489',4109449],
['nbs-iscc-s:emerald.144','emerald','emerald',[106,171,142],'6aab8e',6990734],
['nbs-iscc-s:fawn.29','fawn','fawn',[217,166,169],'d9a6a9',14263977],
['nbs-iscc-s:fawn.43','fawn','fawn',[121,68,59],'79443b',7947323],
['nbs-iscc-s:gray.93','gray','gray',[191,184,165],'bfb8a5',12564645],
['nbs-iscc-s:gray.112','gray','gray',[138,135,118],'8a8776',9078646],
['nbs-iscc-s:gray.149','gray','gray',[141,163,153],'8da399',9282457],
['nbs-iscc-s:gray.154','gray','gray',[178,190,181],'b2beb5',11714229],
['nbs-iscc-s:gray.155','gray','gray',[125,137,132],'7d8984',8227204],
['nbs-iscc-s:gray.156','gray','gray',[78,87,85],'4e5755',5134165],
['nbs-iscc-s:gray.189','gray','gray',[233,233,237],'e9e9ed',15329773],
['nbs-iscc-s:gray.192','gray','gray',[81,88,94],'51585e',5331038],
['nbs-iscc-s:gray.232','gray','gray',[191,185,189],'bfb9bd',12564925],
['nbs-iscc-s:gray.264','gray','gray',[185,184,181],'b9b8b5',12171445],
['nbs-iscc-s:gray.265','gray','gray',[132,132,130],'848482',8684674],
['nbs-iscc-s:graybistre.76','graybistre','gray bistre',[193,154,107],'c19a6b',12687979],
['nbs-iscc-s:graybistre.79','graybistre','gray bistre',[174,155,130],'ae9b82',11443074],
['nbs-iscc-s:graybistre.94','graybistre','gray bistre',[150,113,23],'967117',9859351],
['nbs-iscc-s:grayblack.63','grayblack','gray black',[142,130,121],'8e8279',9339513],
['nbs-iscc-s:grayblack.112','grayblack','gray black',[138,135,118],'8a8776',9078646],
['nbs-iscc-s:grayblack.113','grayblack','gray black',[87,85,76],'57554c',5723468],
['nbs-iscc-s:grayblack.155','grayblack','gray black',[125,137,132],'7d8984',8227204],
['nbs-iscc-s:grayblack.265','grayblack','gray black',[132,132,130],'848482',8684674],
['nbs-iscc-s:grayblue.182','grayblue','gray blue',[67,107,149],'436b95',4418453],
['nbs-iscc-s:grayblue.185','grayblue','gray blue',[145,163,176],'91a3b0',9544624],
['nbs-iscc-s:grayblue.190','grayblue','gray blue',[180,188,192],'b4bcc0',11844800],
['nbs-iscc-s:grayblue.203','grayblue','gray blue',[140,146,172],'8c92ac',9212588],
['nbs-iscc-s:grayblue.214','grayblue','gray blue',[150,144,171],'9690ab',9867435],
['nbs-iscc-s:grayblue.232','grayblue','gray blue',[191,185,189],'bfb9bd',12564925],
['nbs-iscc-s:graybrown.45','graybrown','gray brown',[151,127,115],'977f73',9928563],
['nbs-iscc-s:graybrown.63','graybrown','gray brown',[142,130,121],'8e8279',9339513],
['nbs-iscc-s:graybrown.94','graybrown','gray brown',[150,113,23],'967117',9859351],
['nbs-iscc-s:graybrown.112','graybrown','gray brown',[138,135,118],'8a8776',9078646],
['nbs-iscc-s:graygreen.112','graygreen','gray green',[138,135,118],'8a8776',9078646],
['nbs-iscc-s:graygreen.120','graygreen','gray green',[138,154,91],'8a9a5b',9083483],
['nbs-iscc-s:graygreen.122','graygreen','gray green',[143,151,121],'8f9779',9410425],
['nbs-iscc-s:graygreen.136','graygreen','gray green',[103,146,103],'679267',6787687],
['nbs-iscc-s:graygreen.144','graygreen','gray green',[106,171,142],'6aab8e',6990734],
['nbs-iscc-s:graygreen.145','graygreen','gray green',[59,120,97],'3b7861',3897441],
['nbs-iscc-s:graygreen.163','graygreen','gray green',[102,173,164],'66ada4',6729124],
['nbs-iscc-s:graylilac.22','graylilac','gray lilac',[143,129,127],'8f817f',9404799],
['nbs-iscc-s:graylilac.63','graylilac','gray lilac',[142,130,121],'8e8279',9339513],
['nbs-iscc-s:graylilac.226','graylilac','gray lilac',[214,202,221],'d6cadd',14076637],
['nbs-iscc-s:graylilac.232','graylilac','gray lilac',[191,185,189],'bfb9bd',12564925],
['nbs-iscc-s:graylilac.249','graylilac','gray lilac',[239,187,204],'efbbcc',15711180],
['nbs-iscc-s:grayviolet.233','grayviolet','gray violet',[139,133,137],'8b8589',9143689],
['nbs-iscc-s:green.117','green','green',[126,159,46],'7e9f2e',8298286],
['nbs-iscc-s:green.120','green','green',[138,154,91],'8a9a5b',9083483],
['nbs-iscc-s:green.129','green','green',[39,166,76],'27a64c',2598476],
['nbs-iscc-s:green.130','green','green',[131,211,125],'83d37d',8639357],
['nbs-iscc-s:green.131','green','green',[68,148,74],'44944a',4494410],
['nbs-iscc-s:green.132','green','green',[0,98,45],'00622d',25133],
['nbs-iscc-s:green.135','green','green',[147,197,146],'93c592',9684370],
['nbs-iscc-s:green.136','green','green',[103,146,103],'679267',6787687],
['nbs-iscc-s:green.139','green','green',[0,136,86],'008856',34902],
['nbs-iscc-s:green.140','green','green',[62,180,137],'3eb489',4109449],
['nbs-iscc-s:green.141','green','green',[0,121,89],'007959',31065],
['nbs-iscc-s:green.142','green','green',[0,84,61],'00543d',21565],
['nbs-iscc-s:green.143','green','green',[142,209,178],'8ed1b2',9359794],
['nbs-iscc-s:green.144','green','green',[106,171,142],'6aab8e',6990734],
['nbs-iscc-s:green.145','green','green',[59,120,97],'3b7861',3897441],
['nbs-iscc-s:green.148','green','green',[199,230,215],'c7e6d7',13100759],
['nbs-iscc-s:green.149','green','green',[141,163,153],'8da399',9282457],
['nbs-iscc-s:green.150','green','green',[94,113,106],'5e716a',6189418],
['nbs-iscc-s:green.154','green','green',[178,190,181],'b2beb5',11714229],
['nbs-iscc-s:green.159','green','green',[0,166,147],'00a693',42643],
['nbs-iscc-s:green.163','green','green',[102,173,164],'66ada4',6729124],
['nbs-iscc-s:green.164','green','green',[49,120,115],'317873',3242099],
['nbs-iscc-s:greenishblack.149','greenishblack','greenish black',[141,163,153],'8da399',9282457],
['nbs-iscc-s:greenishblue.164','greenishblue','greenish blue',[49,120,115],'317873',3242099],
['nbs-iscc-s:greenishblue.169','greenishblue','greenish blue',[0,119,145],'007791',30609],
['nbs-iscc-s:greenishblue.173','greenishblue','greenish blue',[54,117,136],'367588',3569032],
['nbs-iscc-s:hennabrown.13','hennabrown','henna brown ',[132,27,45],'841b2d',8657709],
['nbs-iscc-s:indigo.179','indigo','indigo',[0,65,106],'00416a',16746],
['nbs-iscc-s:indigo.204','indigo','indigo',[76,81,109],'4c516d',5001581],
['nbs-iscc-s:lake.15','lake','lake',[171,78,82],'ab4e52',11226706],
['nbs-iscc-s:lake.16','lake','lake',[114,47,55],'722f37',7483191],
['nbs-iscc-s:lake.19','lake','lake',[144,93,93],'905d5d',9461085],
['nbs-iscc-s:lake.39','lake','lake',[180,116,94],'b4745e',11826270],
['nbs-iscc-s:lake.258','lake','lake',[168,81,110],'a8516e',11030894],
['nbs-iscc-s:lemon.82','lemon','lemon',[243,195,0],'f3c300',15975168],
['nbs-iscc-s:lemon.83','lemon','lemon',[250,218,94],'fada5e',16439902],
['nbs-iscc-s:lemon.84','lemon','lemon',[212,175,55],'d4af37',13938487],
['nbs-iscc-s:lemon.85','lemon','lemon',[175,141,19],'af8d13',11504915],
['nbs-iscc-s:lemon.97','lemon','lemon',[220,211,0],'dcd300',14471936],
['nbs-iscc-s:lemonyellow.85','lemonyellow','lemon yellow',[175,141,19],'af8d13',11504915],
['nbs-iscc-s:lemonyellow.100','lemonyellow','lemon yellow',[155,148,0],'9b9400',10195968],
['nbs-iscc-s:lightblue.162','lightblue','light blue',[150,222,209],'96ded1',9887441],
['nbs-iscc-s:lightblue.163','lightblue','light blue',[102,173,164],'66ada4',6729124],
['nbs-iscc-s:lightblue.164','lightblue','light blue',[49,120,115],'317873',3242099],
['nbs-iscc-s:lightblue.168','lightblue','light blue',[35,158,186],'239eba',2334394],
['nbs-iscc-s:lightblue.172','lightblue','light blue',[102,170,188],'66aabc',6728380],
['nbs-iscc-s:lightblue.173','lightblue','light blue',[54,117,136],'367588',3569032],
['nbs-iscc-s:lightblue.178','lightblue','light blue',[0,103,165],'0067a5',26533],
['nbs-iscc-s:lightblue.181','lightblue','light blue',[112,163,204],'70a3cc',7381964],
['nbs-iscc-s:lightblue.184','lightblue','light blue',[188,212,230],'bcd4e6',12375270],
['nbs-iscc-s:lightbrown.32','lightbrown','light brown',[199,173,163],'c7ada3',13086115],
['nbs-iscc-s:lightgreen.131','lightgreen','light green',[68,148,74],'44944a',4494410],
['nbs-iscc-s:lightgreen.135','lightgreen','light green',[147,197,146],'93c592',9684370],
['nbs-iscc-s:lightgreen.140','lightgreen','light green',[62,180,137],'3eb489',4109449],
['nbs-iscc-s:lightolivegreen.106','lightolivegreen','light olive green',[134,126,54],'867e36',8814134],
['nbs-iscc-s:lightolivegreen.117','lightolivegreen','light olive green',[126,159,46],'7e9f2e',8298286],
['nbs-iscc-s:lightolivegreen.119','lightolivegreen','light olive green',[201,220,137],'c9dc89',13229193],
['nbs-iscc-s:lightredbrown.42','lightredbrown','light red brown',[168,124,109],'a87c6d',11041901],
['nbs-iscc-s:lilac.10','lilac','lilac',[193,182,179],'c1b6b3',12695219],
['nbs-iscc-s:lilac.209','lilac','lilac',[220,208,255],'dcd0ff',14471423],
['nbs-iscc-s:lilac.210','lilac','lilac',[140,130,182],'8c82b6',9208502],
['nbs-iscc-s:lilac.221','lilac','lilac',[213,186,219],'d5badb',14007003],
['nbs-iscc-s:lilac.226','lilac','lilac',[214,202,221],'d6cadd',14076637],
['nbs-iscc-s:lilac.227','lilac','lilac',[170,152,169],'aa98a9',11180201],
['nbs-iscc-s:lilac.228','lilac','lilac',[121,104,120],'796878',7956600],
['nbs-iscc-s:lilac.231','lilac','lilac',[232,227,229],'e8e3e5',15262693],
['nbs-iscc-s:lilac.232','lilac','lilac',[191,185,189],'bfb9bd',12564925],
['nbs-iscc-s:lilac.233','lilac','lilac',[139,133,137],'8b8589',9143689],
['nbs-iscc-s:lilac.241','lilac','lilac',[145,92,131],'915c83',9526403],
['nbs-iscc-s:lilac.244','lilac','lilac',[170,138,158],'aa8a9e',11176606],
['nbs-iscc-s:lilac.252','lilac','lilac',[232,204,215],'e8ccd7',15256791],
['nbs-iscc-s:lilac.253','lilac','lilac',[195,166,177],'c3a6b1',12822193],
['nbs-iscc-s:lilac.261','lilac','lilac',[175,134,142],'af868e',11503246],
['nbs-iscc-s:lilac.262','lilac','lilac',[145,95,109],'915f6d',9527149],
['nbs-iscc-s:lilacbrown.10','lilacbrown','lilac brown',[193,182,179],'c1b6b3',12695219],
['nbs-iscc-s:lilacbrown.22','lilacbrown','lilac brown',[143,129,127],'8f817f',9404799],
['nbs-iscc-s:lilacbrown.63','lilacbrown','lilac brown',[142,130,121],'8e8279',9339513],
['nbs-iscc-s:lilacgray.232','lilacgray','lilac gray',[191,185,189],'bfb9bd',12564925],
['nbs-iscc-s:lilacrose.3','lilacrose','lilac rose',[228,113,122],'e4717a',14971258],
['nbs-iscc-s:lilacrose.18','lilacrose','lilac rose',[173,136,132],'ad8884',11372676],
['nbs-iscc-s:lilacrose.237','lilacrose','lilac rose',[158,79,136],'9e4f88',10375048],
['nbs-iscc-s:lilacrose.248','lilacrose','lilac rose',[222,111,161],'de6fa1',14577569],
['nbs-iscc-s:lilacrose.249','lilacrose','lilac rose',[239,187,204],'efbbcc',15711180],
['nbs-iscc-s:lilacrose.258','lilacrose','lilac rose',[168,81,110],'a8516e',11030894],
['nbs-iscc-s:magenta.205','magenta','magenta',[144,101,202],'9065ca',9463242],
['nbs-iscc-s:magenta.218','magenta','magenta',[135,86,146],'875692',8869522],
['nbs-iscc-s:magenta.237','magenta','magenta',[158,79,136],'9e4f88',10375048],
['nbs-iscc-s:magenta.242','magenta','magenta',[93,57,84],'5d3954',6109524],
['nbs-iscc-s:magenta.255','magenta','magenta',[179,68,108],'b3446c',11748460],
['nbs-iscc-s:magenta.258','magenta','magenta',[168,81,110],'a8516e',11030894],
['nbs-iscc-s:magenta.259','magenta','magenta',[103,49,71],'673147',6762823],
['nbs-iscc-s:magenta.262','magenta','magenta',[145,95,109],'915f6d',9527149],
['nbs-iscc-s:maroon.16','maroon','maroon',[114,47,55],'722f37',7483191],
['nbs-iscc-s:maroon.259','maroon','maroon',[103,49,71],'673147',6762823],
['nbs-iscc-s:mauve.241','mauve','mauve',[145,92,131],'915c83',9526403],
['nbs-iscc-s:milkyblue.172','milkyblue','milky blue ',[102,170,188],'66aabc',6728380],
['nbs-iscc-s:myrtlegreen.164','myrtlegreen','myrtle green ',[49,120,115],'317873',3242099],
['nbs-iscc-s:myrtlegreen.165','myrtlegreen','myrtle green ',[0,75,73],'004b49',19273],
['nbs-iscc-s:ochre.51','ochre','ochre',[190,101,22],'be6516',12477718],
['nbs-iscc-s:ochre.53','ochre','ochre',[217,144,88],'d99058',14258264],
['nbs-iscc-s:ochre.57','ochre','ochre',[166,123,91],'a67b5b',10910555],
['nbs-iscc-s:ochre.67','ochre','ochre',[255,193,79],'ffc14f',16761167],
['nbs-iscc-s:ochre.71','ochre','ochre',[227,168,87],'e3a857',14919767],
['nbs-iscc-s:ochre.73','ochre','ochre',[250,214,165],'fad6a5',16438949],
['nbs-iscc-s:ochre.74','ochre','ochre',[153,101,21],'996515',10052885],
['nbs-iscc-s:ochre.83','ochre','ochre',[250,218,94],'fada5e',16439902],
['nbs-iscc-s:ochre.84','ochre','ochre',[212,175,55],'d4af37',13938487],
['nbs-iscc-s:ochre.86','ochre','ochre',[248,222,126],'f8de7e',16309886],
['nbs-iscc-s:ochre.87','ochre','ochre',[201,174,93],'c9ae5d',13217373],
['nbs-iscc-s:olive.95','olive','olive',[108,84,30],'6c541e',7099422],
['nbs-iscc-s:olive.110','olive','olive',[91,88,66],'5b5842',5986370],
['nbs-iscc-s:olive.122','olive','olive',[143,151,121],'8f9779',9410425],
['nbs-iscc-s:olivebistre.69','olivebistre','olive bistre',[201,133,0],'c98500',13206784],
['nbs-iscc-s:olivebistre.72','olivebistre','olive bistre',[190,138,61],'be8a3d',12487229],
['nbs-iscc-s:olivebistre.74','olivebistre','olive bistre',[153,101,21],'996515',10052885],
['nbs-iscc-s:olivebistre.77','olivebistre','olive bistre',[130,102,68],'826644',8545860],
['nbs-iscc-s:olivebistre.85','olivebistre','olive bistre',[175,141,19],'af8d13',11504915],
['nbs-iscc-s:olivebistre.87','olivebistre','olive bistre',[201,174,93],'c9ae5d',13217373],
['nbs-iscc-s:olivebistre.88','olivebistre','olive bistre',[171,145,68],'ab9144',11243844],
['nbs-iscc-s:olivebistre.89','olivebistre','olive bistre',[243,229,171],'f3e5ab',15984043],
['nbs-iscc-s:olivebistre.90','olivebistre','olive bistre',[194,178,128],'c2b280',12759680],
['nbs-iscc-s:olivebistre.91','olivebistre','olive bistre',[161,143,96],'a18f60',10588000],
['nbs-iscc-s:olivebistre.93','olivebistre','olive bistre',[191,184,165],'bfb8a5',12564645],
['nbs-iscc-s:olivebistre.94','olivebistre','olive bistre',[150,113,23],'967117',9859351],
['nbs-iscc-s:olivebistre.106','olivebistre','olive bistre',[134,126,54],'867e36',8814134],
['nbs-iscc-s:olivebistre.112','olivebistre','olive bistre',[138,135,118],'8a8776',9078646],
['nbs-iscc-s:olivebistre.122','olivebistre','olive bistre',[143,151,121],'8f9779',9410425],
['nbs-iscc-s:olivebistre.264','olivebistre','olive bistre',[185,184,181],'b9b8b5',12171445],
['nbs-iscc-s:oliveblack.110','oliveblack','olive black',[91,88,66],'5b5842',5986370],
['nbs-iscc-s:oliveblack.113','oliveblack','olive black',[87,85,76],'57554c',5723468],
['nbs-iscc-s:olivebrown.58','olivebrown','olive brown',[111,78,55],'6f4e37',7294519],
['nbs-iscc-s:olivebrown.63','olivebrown','olive brown',[142,130,121],'8e8279',9339513],
['nbs-iscc-s:olivebrown.78','olivebrown','olive brown',[75,54,33],'4b3621',4929057],
['nbs-iscc-s:olivebrown.80','olivebrown','olive brown',[126,109,90],'7e6d5a',8285530],
['nbs-iscc-s:olivebrown.81','olivebrown','olive brown',[72,60,50],'483c32',4734002],
['nbs-iscc-s:olivebrown.95','olivebrown','olive brown',[108,84,30],'6c541e',7099422],
['nbs-iscc-s:olivebrown.110','olivebrown','olive brown',[91,88,66],'5b5842',5986370],
['nbs-iscc-s:olivebrown.113','olivebrown','olive brown',[87,85,76],'57554c',5723468],
['nbs-iscc-s:olivegray.107','olivegray','olive gray',[102,93,30],'665d1e',6708510],
['nbs-iscc-s:olivegray.112','olivegray','olive gray',[138,135,118],'8a8776',9078646],
['nbs-iscc-s:olivegray.113','olivegray','olive gray',[87,85,76],'57554c',5723468],
['nbs-iscc-s:olivegray.122','olivegray','olive gray',[143,151,121],'8f9779',9410425],
['nbs-iscc-s:olivegray.155','olivegray','olive gray',[125,137,132],'7d8984',8227204],
['nbs-iscc-s:olivegray.156','olivegray','olive gray',[78,87,85],'4e5755',5134165],
['nbs-iscc-s:olivegreen.88','olivegreen','olive green',[171,145,68],'ab9144',11243844],
['nbs-iscc-s:olivegreen.89','olivegreen','olive green',[243,229,171],'f3e5ab',15984043],
['nbs-iscc-s:olivegreen.90','olivegreen','olive green',[194,178,128],'c2b280',12759680],
['nbs-iscc-s:olivegreen.91','olivegreen','olive green',[161,143,96],'a18f60',10588000],
['nbs-iscc-s:olivegreen.93','olivegreen','olive green',[191,184,165],'bfb8a5',12564645],
['nbs-iscc-s:olivegreen.100','olivegreen','olive green',[155,148,0],'9b9400',10195968],
['nbs-iscc-s:olivegreen.106','olivegreen','olive green',[134,126,54],'867e36',8814134],
['nbs-iscc-s:olivegreen.107','olivegreen','olive green',[102,93,30],'665d1e',6708510],
['nbs-iscc-s:olivegreen.109','olivegreen','olive green',[140,135,103],'8c8767',9209703],
['nbs-iscc-s:olivegreen.110','olivegreen','olive green',[91,88,66],'5b5842',5986370],
['nbs-iscc-s:olivegreen.112','olivegreen','olive green',[138,135,118],'8a8776',9078646],
['nbs-iscc-s:olivegreen.113','olivegreen','olive green',[87,85,76],'57554c',5723468],
['nbs-iscc-s:olivegreen.117','olivegreen','olive green',[126,159,46],'7e9f2e',8298286],
['nbs-iscc-s:olivegreen.120','olivegreen','olive green',[138,154,91],'8a9a5b',9083483],
['nbs-iscc-s:olivegreen.122','olivegreen','olive green',[143,151,121],'8f9779',9410425],
['nbs-iscc-s:olivegreen.123','olivegreen','olive green',[64,79,0],'404f00',4214528],
['nbs-iscc-s:olivegreen.125','olivegreen','olive green',[74,93,35],'4a5d23',4873507],
['nbs-iscc-s:olivegreen.127','olivegreen','olive green',[81,87,68],'515744',5330756],
['nbs-iscc-s:olivegreen.150','olivegreen','olive green',[94,113,106],'5e716a',6189418],
['nbs-iscc-s:olivegreen.151','olivegreen','olive green',[58,75,71],'3a4b47',3820359],
['nbs-iscc-s:olivegreen.155','olivegreen','olive green',[125,137,132],'7d8984',8227204],
['nbs-iscc-s:olivegreen.156','olivegreen','olive green',[78,87,85],'4e5755',5134165],
['nbs-iscc-s:oliveyellow.31','oliveyellow','olive yellow',[236,213,197],'ecd5c5',15521221],
['nbs-iscc-s:oliveyellow.33','oliveyellow','olive yellow',[194,172,153],'c2ac99',12758169],
['nbs-iscc-s:oliveyellow.73','oliveyellow','olive yellow',[250,214,165],'fad6a5',16438949],
['nbs-iscc-s:oliveyellow.86','oliveyellow','olive yellow',[248,222,126],'f8de7e',16309886],
['nbs-iscc-s:oliveyellow.87','oliveyellow','olive yellow',[201,174,93],'c9ae5d',13217373],
['nbs-iscc-s:oliveyellow.89','oliveyellow','olive yellow',[243,229,171],'f3e5ab',15984043],
['nbs-iscc-s:oliveyellow.93','oliveyellow','olive yellow',[191,184,165],'bfb8a5',12564645],
['nbs-iscc-s:orange.25','orange','orange',[255,183,165],'ffb7a5',16758693],
['nbs-iscc-s:orange.26','orange','orange',[248,131,121],'f88379',16286585],
['nbs-iscc-s:orange.28','orange','orange',[244,194,194],'f4c2c2',16040642],
['nbs-iscc-s:orange.29','orange','orange',[217,166,169],'d9a6a9',14263977],
['nbs-iscc-s:orange.34','orange','orange',[226,88,34],'e25822',14833698],
['nbs-iscc-s:orange.35','orange','orange',[217,96,59],'d9603b',14245947],
['nbs-iscc-s:orange.37','orange','orange',[203,109,81],'cb6d51',13331793],
['nbs-iscc-s:orange.48','orange','orange',[243,132,0],'f38400',15959040],
['nbs-iscc-s:orange.49','orange','orange',[253,148,63],'fd943f',16618559],
['nbs-iscc-s:orange.50','orange','orange',[237,135,45],'ed872d',15566637],
['nbs-iscc-s:orange.51','orange','orange',[190,101,22],'be6516',12477718],
['nbs-iscc-s:orange.52','orange','orange',[250,181,127],'fab57f',16430463],
['nbs-iscc-s:orange.53','orange','orange',[217,144,88],'d99058',14258264],
['nbs-iscc-s:orange.66','orange','orange',[246,166,0],'f6a600',16164352],
['nbs-iscc-s:orange.67','orange','orange',[255,193,79],'ffc14f',16761167],
['nbs-iscc-s:orange.68','orange','orange',[234,162,33],'eaa221',15376929],
['nbs-iscc-s:orange.69','orange','orange',[201,133,0],'c98500',13206784],
['nbs-iscc-s:orange.70','orange','orange',[251,201,127],'fbc97f',16501119],
['nbs-iscc-s:orange.71','orange','orange',[227,168,87],'e3a857',14919767],
['nbs-iscc-s:orange.72','orange','orange',[190,138,61],'be8a3d',12487229],
['nbs-iscc-s:orange.73','orange','orange',[250,214,165],'fad6a5',16438949],
['nbs-iscc-s:orange.76','orange','orange',[193,154,107],'c19a6b',12687979],
['nbs-iscc-s:orange.83','orange','orange',[250,218,94],'fada5e',16439902],
['nbs-iscc-s:orange.84','orange','orange',[212,175,55],'d4af37',13938487],
['nbs-iscc-s:orange.86','orange','orange',[248,222,126],'f8de7e',16309886],
['nbs-iscc-s:orange.87','orange','orange',[201,174,93],'c9ae5d',13217373],
['nbs-iscc-s:orange.89','orange','orange',[243,229,171],'f3e5ab',15984043],
['nbs-iscc-s:orangebrown.16','orangebrown','orange brown',[114,47,55],'722f37',7483191],
['nbs-iscc-s:orangebrown.37','orangebrown','orange brown',[203,109,81],'cb6d51',13331793],
['nbs-iscc-s:orangebrown.39','orangebrown','orange brown',[180,116,94],'b4745e',11826270],
['nbs-iscc-s:orangebrown.40','orangebrown','orange brown',[136,45,23],'882d17',8924439],
['nbs-iscc-s:orangebrown.51','orangebrown','orange brown',[190,101,22],'be6516',12477718],
['nbs-iscc-s:orangebrown.53','orangebrown','orange brown',[217,144,88],'d99058',14258264],
['nbs-iscc-s:orangebrown.54','orangebrown','orange brown',[174,105,56],'ae6938',11430200],
['nbs-iscc-s:orangebrown.55','orangebrown','orange brown',[128,70,27],'80461b',8406555],
['nbs-iscc-s:orangebrown.57','orangebrown','orange brown',[166,123,91],'a67b5b',10910555],
['nbs-iscc-s:orangebrown.58','orangebrown','orange brown',[111,78,55],'6f4e37',7294519],
['nbs-iscc-s:orangebrown.60','orangebrown','orange brown',[149,128,112],'958070',9797744],
['nbs-iscc-s:orangebrown.79','orangebrown','orange brown',[174,155,130],'ae9b82',11443074],
['nbs-iscc-s:orangebuff.29','orangebuff','orange buff',[217,166,169],'d9a6a9',14263977],
['nbs-iscc-s:orangered.6','orangered','orange red',[192,128,129],'c08081',12615809],
['nbs-iscc-s:orangered.35','orangered','orange red',[217,96,59],'d9603b',14245947],
['nbs-iscc-s:orangered.37','orangered','orange red',[203,109,81],'cb6d51',13331793],
['nbs-iscc-s:orangevermilion.37','orangevermilion','orange vermilion',[203,109,81],'cb6d51',13331793],
['nbs-iscc-s:orangevermilion.51','orangevermilion','orange vermilion',[190,101,22],'be6516',12477718],
['nbs-iscc-s:orangeyellow.48','orangeyellow','orange yellow',[243,132,0],'f38400',15959040],
['nbs-iscc-s:orangeyellow.49','orangeyellow','orange yellow',[253,148,63],'fd943f',16618559],
['nbs-iscc-s:orangeyellow.50','orangeyellow','orange yellow',[237,135,45],'ed872d',15566637],
['nbs-iscc-s:orangeyellow.52','orangeyellow','orange yellow',[250,181,127],'fab57f',16430463],
['nbs-iscc-s:orangeyellow.67','orangeyellow','orange yellow',[255,193,79],'ffc14f',16761167],
['nbs-iscc-s:orangeyellow.68','orangeyellow','orange yellow',[234,162,33],'eaa221',15376929],
['nbs-iscc-s:orangeyellow.69','orangeyellow','orange yellow',[201,133,0],'c98500',13206784],
['nbs-iscc-s:orangeyellow.70','orangeyellow','orange yellow',[251,201,127],'fbc97f',16501119],
['nbs-iscc-s:orangeyellow.71','orangeyellow','orange yellow',[227,168,87],'e3a857',14919767],
['nbs-iscc-s:paleblue.186','paleblue','pale blue',[83,104,120],'536878',5466232],
['nbs-iscc-s:paleblue.232','paleblue','pale blue',[191,185,189],'bfb9bd',12564925],
['nbs-iscc-s:palebrown.53','palebrown','pale brown',[217,144,88],'d99058',14258264],
['nbs-iscc-s:palebrown.57','palebrown','pale brown',[166,123,91],'a67b5b',10910555],
['nbs-iscc-s:palegraygreen.121','palegraygreen','pale gray green',[218,223,183],'dadfb7',14344119],
['nbs-iscc-s:palegreen.119','palegreen','pale green',[201,220,137],'c9dc89',13229193],
['nbs-iscc-s:palegreen.120','palegreen','pale green',[138,154,91],'8a9a5b',9083483],
['nbs-iscc-s:palegreen.143','palegreen','pale green',[142,209,178],'8ed1b2',9359794],
['nbs-iscc-s:palelilac.252','palelilac','pale lilac',[232,204,215],'e8ccd7',15256791],
['nbs-iscc-s:paleolivegreen.120','paleolivegreen','pale olive green',[138,154,91],'8a9a5b',9083483],
['nbs-iscc-s:paleyellowgreen.116','paleyellowgreen','pale yellow-green',[189,218,87],'bdda57',12442199],
['nbs-iscc-s:paleyellowgreen.119','paleyellowgreen','pale yellow-green',[201,220,137],'c9dc89',13229193],
['nbs-iscc-s:paleyellowgreen.130','paleyellowgreen','pale yellow-green',[131,211,125],'83d37d',8639357],
['nbs-iscc-s:peacockblue.169','peacockblue','peacock blue',[0,119,145],'007791',30609],
['nbs-iscc-s:pink.2','pink','pink',[234,147,153],'ea9399',15373209],
['nbs-iscc-s:pink.3','pink','pink',[228,113,122],'e4717a',14971258],
['nbs-iscc-s:pink.26','pink','pink',[248,131,121],'f88379',16286585],
['nbs-iscc-s:pink.246','pink','pink',[255,200,214],'ffc8d6',16763094],
['nbs-iscc-s:pink.247','pink','pink',[230,143,172],'e68fac',15110060],
['nbs-iscc-s:pink.248','pink','pink',[222,111,161],'de6fa1',14577569],
['nbs-iscc-s:pink.258','pink','pink',[168,81,110],'a8516e',11030894],
['nbs-iscc-s:plum.236','plum','plum',[135,0,116],'870074',8847476],
['nbs-iscc-s:plum.238','plum','plum',[112,41,99],'702963',7350627],
['nbs-iscc-s:plum.256','plum','plum',[120,24,74],'78184a',7870538],
['nbs-iscc-s:prussianblue.170','prussianblue','prussian blue',[46,132,149],'2e8495',3048597],
['nbs-iscc-s:prussianblue.174','prussianblue','prussian blue',[0,73,88],'004958',18776],
['nbs-iscc-s:prussiangreen.160','prussiangreen','prussian green',[0,122,116],'007a74',31348],
['nbs-iscc-s:prussiangreen.161','prussiangreen','prussian green',[0,68,63],'00443f',17471],
['nbs-iscc-s:prussiangreen.164','prussiangreen','prussian green',[49,120,115],'317873',3242099],
['nbs-iscc-s:prussiangreen.170','prussiangreen','prussian green',[46,132,149],'2e8495',3048597],
['nbs-iscc-s:purple.197','purple','purple',[39,36,88],'272458',2565208],
['nbs-iscc-s:purple.200','purple','purple',[78,81,128],'4e5180',5132672],
['nbs-iscc-s:purple.207','purple','purple',[96,78,151],'604e97',6311575],
['nbs-iscc-s:purple.211','purple','purple',[96,78,129],'604e81',6311553],
['nbs-iscc-s:purple.215','purple','purple',[85,76,105],'554c69',5590121],
['nbs-iscc-s:purple.216','purple','purple',[154,78,174],'9a4eae',10112686],
['nbs-iscc-s:purple.218','purple','purple',[135,86,146],'875692',8869522],
['nbs-iscc-s:purple.219','purple','purple',[96,47,107],'602f6b',6303595],
['nbs-iscc-s:purple.223','purple','purple',[134,96,142],'86608e',8806542],
['nbs-iscc-s:purple.241','purple','purple',[145,92,131],'915c83',9526403],
['nbs-iscc-s:purple.242','purple','purple',[93,57,84],'5d3954',6109524],
['nbs-iscc-s:purple.245','purple','purple',[131,100,121],'836479',8610937],
['nbs-iscc-s:purplebrown.228','purplebrown','purple brown',[121,104,120],'796878',7956600],
['nbs-iscc-s:purplishbistre.33','purplishbistre','purplish bistre',[194,172,153],'c2ac99',12758169],
['nbs-iscc-s:red.3','red','red',[228,113,122],'e4717a',14971258],
['nbs-iscc-s:red.5','red','red',[222,165,164],'dea5a4',14591396],
['nbs-iscc-s:red.8','red','red',[196,174,173],'c4aead',12889773],
['nbs-iscc-s:red.12','red','red',[188,63,74],'bc3f4a',12336970],
['nbs-iscc-s:red.19','red','red',[144,93,93],'905d5d',9461085],
['nbs-iscc-s:red.26','red','red',[248,131,121],'f88379',16286585],
['nbs-iscc-s:red.27','red','red',[230,103,97],'e66761',15099745],
['nbs-iscc-s:red.28','red','red',[244,194,194],'f4c2c2',16040642],
['nbs-iscc-s:red.29','red','red',[217,166,169],'d9a6a9',14263977],
['nbs-iscc-s:red.35','red','red',[217,96,59],'d9603b',14245947],
['nbs-iscc-s:red.36','red','red',[170,56,30],'aa381e',11155486],
['nbs-iscc-s:red.37','red','red',[203,109,81],'cb6d51',13331793],
['nbs-iscc-s:redbrown.8','redbrown','red brown',[196,174,173],'c4aead',12889773],
['nbs-iscc-s:redbrown.16','redbrown','red brown',[114,47,55],'722f37',7483191],
['nbs-iscc-s:redbrown.18','redbrown','red brown',[173,136,132],'ad8884',11372676],
['nbs-iscc-s:redbrown.19','redbrown','red brown',[144,93,93],'905d5d',9461085],
['nbs-iscc-s:redbrown.22','redbrown','red brown',[143,129,127],'8f817f',9404799],
['nbs-iscc-s:redbrown.31','redbrown','red brown',[236,213,197],'ecd5c5',15521221],
['nbs-iscc-s:redbrown.32','redbrown','red brown',[199,173,163],'c7ada3',13086115],
['nbs-iscc-s:redbrown.37','redbrown','red brown',[203,109,81],'cb6d51',13331793],
['nbs-iscc-s:redbrown.38','redbrown','red brown',[158,71,50],'9e4732',10372914],
['nbs-iscc-s:redbrown.42','redbrown','red brown',[168,124,109],'a87c6d',11041901],
['nbs-iscc-s:redbrown.43','redbrown','red brown',[121,68,59],'79443b',7947323],
['nbs-iscc-s:redbrown.51','redbrown','red brown',[190,101,22],'be6516',12477718],
['nbs-iscc-s:redbrown.53','redbrown','red brown',[217,144,88],'d99058',14258264],
['nbs-iscc-s:redbrown.54','redbrown','red brown',[174,105,56],'ae6938',11430200],
['nbs-iscc-s:redbrown.57','redbrown','red brown',[166,123,91],'a67b5b',10910555],
['nbs-iscc-s:redbrown.58','redbrown','red brown',[111,78,55],'6f4e37',7294519],
['nbs-iscc-s:redbrown.74','redbrown','red brown',[153,101,21],'996515',10052885],
['nbs-iscc-s:redbrown.77','redbrown','red brown',[130,102,68],'826644',8545860],
['nbs-iscc-s:redbrown.95','redbrown','red brown',[108,84,30],'6c541e',7099422],
['nbs-iscc-s:reddishbrown.46','reddishbrown','reddish brown',[103,76,71],'674c47',6769735],
['nbs-iscc-s:redlilac.8','redlilac','red lilac',[196,174,173],'c4aead',12889773],
['nbs-iscc-s:redlilac.258','redlilac','red lilac',[168,81,110],'a8516e',11030894],
['nbs-iscc-s:redlilac.262','redlilac','red lilac',[145,95,109],'915f6d',9527149],
['nbs-iscc-s:redorange.11','redorange','red orange',[190,0,50],'be0032',12451890],
['nbs-iscc-s:redorange.25','redorange','red orange',[255,183,165],'ffb7a5',16758693],
['nbs-iscc-s:redorange.34','redorange','red orange',[226,88,34],'e25822',14833698],
['nbs-iscc-s:redorange.35','redorange','red orange',[217,96,59],'d9603b',14245947],
['nbs-iscc-s:redorange.50','redorange','red orange',[237,135,45],'ed872d',15566637],
['nbs-iscc-s:redpurple.238','redpurple','red purple',[112,41,99],'702963',7350627],
['nbs-iscc-s:redpurple.259','redpurple','red purple',[103,49,71],'673147',6762823],
['nbs-iscc-s:redviolet.216','redviolet','red violet',[154,78,174],'9a4eae',10112686],
['nbs-iscc-s:redviolet.218','redviolet','red violet',[135,86,146],'875692',8869522],
['nbs-iscc-s:redviolet.219','redviolet','red violet',[96,47,107],'602f6b',6303595],
['nbs-iscc-s:redviolet.222','redviolet','red violet',[182,149,192],'b695c0',11965888],
['nbs-iscc-s:redviolet.223','redviolet','red violet',[134,96,142],'86608e',8806542],
['nbs-iscc-s:redviolet.237','redviolet','red violet',[158,79,136],'9e4f88',10375048],
['nbs-iscc-s:redviolet.238','redviolet','red violet',[112,41,99],'702963',7350627],
['nbs-iscc-s:redviolet.240','redviolet','red violet',[183,132,167],'b784a7',12027047],
['nbs-iscc-s:redviolet.241','redviolet','red violet',[145,92,131],'915c83',9526403],
['nbs-iscc-s:redviolet.249','redviolet','red violet',[239,187,204],'efbbcc',15711180],
['nbs-iscc-s:redviolet.250','redviolet','red violet',[213,151,174],'d597ae',13997998],
['nbs-iscc-s:redviolet.253','redviolet','red violet',[195,166,177],'c3a6b1',12822193],
['nbs-iscc-s:redviolet.262','redviolet','red violet',[145,95,109],'915f6d',9527149],
['nbs-iscc-s:rose.1','rose','rose',[255,181,186],'ffb5ba',16758202],
['nbs-iscc-s:rose.2','rose','rose',[234,147,153],'ea9399',15373209],
['nbs-iscc-s:rose.3','rose','rose',[228,113,122],'e4717a',14971258],
['nbs-iscc-s:rose.4','rose','rose',[249,204,202],'f9ccca',16370890],
['nbs-iscc-s:rose.5','rose','rose',[222,165,164],'dea5a4',14591396],
['nbs-iscc-s:rose.6','rose','rose',[192,128,129],'c08081',12615809],
['nbs-iscc-s:rose.12','rose','rose',[188,63,74],'bc3f4a',12336970],
['nbs-iscc-s:rose.13','rose','rose',[132,27,45],'841b2d',8657709],
['nbs-iscc-s:rose.15','rose','rose',[171,78,82],'ab4e52',11226706],
['nbs-iscc-s:rose.19','rose','rose',[144,93,93],'905d5d',9461085],
['nbs-iscc-s:rose.25','rose','rose',[255,183,165],'ffb7a5',16758693],
['nbs-iscc-s:rose.26','rose','rose',[248,131,121],'f88379',16286585],
['nbs-iscc-s:rose.27','rose','rose',[230,103,97],'e66761',15099745],
['nbs-iscc-s:rose.28','rose','rose',[244,194,194],'f4c2c2',16040642],
['nbs-iscc-s:rose.29','rose','rose',[217,166,169],'d9a6a9',14263977],
['nbs-iscc-s:rose.30','rose','rose',[196,131,121],'c48379',12878713],
['nbs-iscc-s:rose.31','rose','rose',[236,213,197],'ecd5c5',15521221],
['nbs-iscc-s:rose.32','rose','rose',[199,173,163],'c7ada3',13086115],
['nbs-iscc-s:rose.37','rose','rose',[203,109,81],'cb6d51',13331793],
['nbs-iscc-s:rose.227','rose','rose',[170,152,169],'aa98a9',11180201],
['nbs-iscc-s:rose.247','rose','rose',[230,143,172],'e68fac',15110060],
['nbs-iscc-s:rose.248','rose','rose',[222,111,161],'de6fa1',14577569],
['nbs-iscc-s:rose.249','rose','rose',[239,187,204],'efbbcc',15711180],
['nbs-iscc-s:rose.250','rose','rose',[213,151,174],'d597ae',13997998],
['nbs-iscc-s:rose.251','rose','rose',[193,126,145],'c17e91',12680849],
['nbs-iscc-s:rose.254','rose','rose',[206,70,118],'ce4676',13518454],
['nbs-iscc-s:rose.255','rose','rose',[179,68,108],'b3446c',11748460],
['nbs-iscc-s:rose.258','rose','rose',[168,81,110],'a8516e',11030894],
['nbs-iscc-s:rose.262','rose','rose',[145,95,109],'915f6d',9527149],
['nbs-iscc-s:rosebrown.11','rosebrown','rose brown',[190,0,50],'be0032',12451890],
['nbs-iscc-s:rosebrown.13','rosebrown','rose brown',[132,27,45],'841b2d',8657709],
['nbs-iscc-s:rosebrown.14','rosebrown','rose brown',[92,9,35],'5c0923',6031651],
['nbs-iscc-s:rosebrown.256','rosebrown','rose brown',[120,24,74],'78184a',7870538],
['nbs-iscc-s:rosebrown.257','rosebrown','rose brown',[84,19,59],'54133b',5509947],
['nbs-iscc-s:rosecarmine.3','rosecarmine','rose carmine',[228,113,122],'e4717a',14971258],
['nbs-iscc-s:rosecarmine.254','rosecarmine','rose carmine',[206,70,118],'ce4676',13518454],
['nbs-iscc-s:rosecarmine.255','rosecarmine','rose carmine',[179,68,108],'b3446c',11748460],
['nbs-iscc-s:rosecarmine.258','rosecarmine','rose carmine',[168,81,110],'a8516e',11030894],
['nbs-iscc-s:roselake.5','roselake','rose lake',[222,165,164],'dea5a4',14591396],
['nbs-iscc-s:roselake.259','roselake','rose lake',[103,49,71],'673147',6762823],
['nbs-iscc-s:roselake.262','roselake','rose lake',[145,95,109],'915f6d',9527149],
['nbs-iscc-s:roselilac.237','roselilac','rose lilac',[158,79,136],'9e4f88',10375048],
['nbs-iscc-s:roselilac.240','roselilac','rose lilac',[183,132,167],'b784a7',12027047],
['nbs-iscc-s:roselilac.250','roselilac','rose lilac',[213,151,174],'d597ae',13997998],
['nbs-iscc-s:roselilac.251','roselilac','rose lilac',[193,126,145],'c17e91',12680849],
['nbs-iscc-s:roselilac.258','roselilac','rose lilac',[168,81,110],'a8516e',11030894],
['nbs-iscc-s:roselilac.262','roselilac','rose lilac',[145,95,109],'915f6d',9527149],
['nbs-iscc-s:rosepink.254','rosepink','rose pink',[206,70,118],'ce4676',13518454],
['nbs-iscc-s:rosered.3','rosered','rose red',[228,113,122],'e4717a',14971258],
['nbs-iscc-s:rosered.15','rosered','rose red',[171,78,82],'ab4e52',11226706],
['nbs-iscc-s:rosered.16','rosered','rose red',[114,47,55],'722f37',7483191],
['nbs-iscc-s:roseviolet.237','roseviolet','rose violet',[158,79,136],'9e4f88',10375048],
['nbs-iscc-s:roseviolet.238','roseviolet','rose violet',[112,41,99],'702963',7350627],
['nbs-iscc-s:roseviolet.241','roseviolet','rose violet',[145,92,131],'915c83',9526403],
['nbs-iscc-s:roseviolet.256','roseviolet','rose violet',[120,24,74],'78184a',7870538],
['nbs-iscc-s:roseviolet.258','roseviolet','rose violet',[168,81,110],'a8516e',11030894],
['nbs-iscc-s:royalblue.179','royalblue','royal blue',[0,65,106],'00416a',16746],
['nbs-iscc-s:salmon.26','salmon','salmon',[248,131,121],'f88379',16286585],
['nbs-iscc-s:salmon.28','salmon','salmon',[244,194,194],'f4c2c2',16040642],
['nbs-iscc-s:salmon.29','salmon','salmon',[217,166,169],'d9a6a9',14263977],
['nbs-iscc-s:salmonrose.2','salmonrose','salmon rose',[234,147,153],'ea9399',15373209],
['nbs-iscc-s:salmonrose.4','salmonrose','salmon rose',[249,204,202],'f9ccca',16370890],
['nbs-iscc-s:salmonrose.26','salmonrose','salmon rose',[248,131,121],'f88379',16286585],
['nbs-iscc-s:salmonrose.28','salmonrose','salmon rose',[244,194,194],'f4c2c2',16040642],
['nbs-iscc-s:salmonrose.29','salmonrose','salmon rose',[217,166,169],'d9a6a9',14263977],
['nbs-iscc-s:scarlet.3','scarlet','scarlet',[228,113,122],'e4717a',14971258],
['nbs-iscc-s:scarlet.11','scarlet','scarlet',[190,0,50],'be0032',12451890],
['nbs-iscc-s:scarlet.12','scarlet','scarlet',[188,63,74],'bc3f4a',12336970],
['nbs-iscc-s:scarlet.15','scarlet','scarlet',[171,78,82],'ab4e52',11226706],
['nbs-iscc-s:scarlet.34','scarlet','scarlet',[226,88,34],'e25822',14833698],
['nbs-iscc-s:sepia.63','sepia','sepia',[142,130,121],'8e8279',9339513],
['nbs-iscc-s:slate.149','slate','slate',[141,163,153],'8da399',9282457],
['nbs-iscc-s:slate.151','slate','slate',[58,75,71],'3a4b47',3820359],
['nbs-iscc-s:slate.155','slate','slate',[125,137,132],'7d8984',8227204],
['nbs-iscc-s:slate.156','slate','slate',[78,87,85],'4e5755',5134165],
['nbs-iscc-s:slate.187','slate','slate',[54,69,79],'36454f',3556687],
['nbs-iscc-s:slate.188','slate','slate',[32,40,48],'202830',2107440],
['nbs-iscc-s:slate.191','slate','slate',[129,135,139],'81878b',8488843],
['nbs-iscc-s:slate.192','slate','slate',[81,88,94],'51585e',5331038],
['nbs-iscc-s:slate.201','slate','slate',[37,36,64],'252440',2434112],
['nbs-iscc-s:slate.204','slate','slate',[76,81,109],'4c516d',5001581],
['nbs-iscc-s:slate.266','slate','slate',[85,85,85],'555555',5592405],
['nbs-iscc-s:slateblue.164','slateblue','slate blue',[49,120,115],'317873',3242099],
['nbs-iscc-s:slateblue.186','slateblue','slate blue',[83,104,120],'536878',5466232],
['nbs-iscc-s:slategreen.109','slategreen','slate green ',[140,135,103],'8c8767',9209703],
['nbs-iscc-s:slategreen.112','slategreen','slate green ',[138,135,118],'8a8776',9078646],
['nbs-iscc-s:slategreen.123','slategreen','slate green ',[64,79,0],'404f00',4214528],
['nbs-iscc-s:slategreen.125','slategreen','slate green ',[74,93,35],'4a5d23',4873507],
['nbs-iscc-s:slategreen.145','slategreen','slate green ',[59,120,97],'3b7861',3897441],
['nbs-iscc-s:slategreen.151','slategreen','slate green ',[58,75,71],'3a4b47',3820359],
['nbs-iscc-s:slategreen.155','slategreen','slate green ',[125,137,132],'7d8984',8227204],
['nbs-iscc-s:straw.87','straw','straw',[201,174,93],'c9ae5d',13217373],
['nbs-iscc-s:turquoiseblue.172','turquoiseblue','turquoise blue',[102,170,188],'66aabc',6728380],
['nbs-iscc-s:turquoisegreen.158','turquoisegreen','turquoise green',[0,136,130],'008882',34946],
['nbs-iscc-s:turquoisegreen.159','turquoisegreen','turquoise green',[0,166,147],'00a693',42643],
['nbs-iscc-s:ultramarine.176','ultramarine','ultramarine',[0,161,194],'00a1c2',41410],
['nbs-iscc-s:ultramarine.177','ultramarine','ultramarine',[73,151,208],'4997d0',4822992],
['nbs-iscc-s:ultramarine.178','ultramarine','ultramarine',[0,103,165],'0067a5',26533],
['nbs-iscc-s:ultramarine.179','ultramarine','ultramarine',[0,65,106],'00416a',16746],
['nbs-iscc-s:ultramarine.181','ultramarine','ultramarine',[112,163,204],'70a3cc',7381964],
['nbs-iscc-s:ultramarine.182','ultramarine','ultramarine',[67,107,149],'436b95',4418453],
['nbs-iscc-s:ultramarine.185','ultramarine','ultramarine',[145,163,176],'91a3b0',9544624],
['nbs-iscc-s:ultramarine.186','ultramarine','ultramarine',[83,104,120],'536878',5466232],
['nbs-iscc-s:ultramarine.194','ultramarine','ultramarine',[48,38,122],'30267a',3155578],
['nbs-iscc-s:ultramarine.195','ultramarine','ultramarine',[108,121,184],'6c79b8',7109048],
['nbs-iscc-s:ultramarine.196','ultramarine','ultramarine',[84,90,167],'545aa7',5528231],
['nbs-iscc-s:ultramarine.198','ultramarine','ultramarine',[179,188,226],'b3bce2',11779298],
['nbs-iscc-s:ultramarine.199','ultramarine','ultramarine',[135,145,191],'8791bf',8884671],
['nbs-iscc-s:ultramarine.200','ultramarine','ultramarine',[78,81,128],'4e5180',5132672],
['nbs-iscc-s:ultramarine.202','ultramarine','ultramarine',[192,200,225],'c0c8e1',12634337],
['nbs-iscc-s:ultramarine.203','ultramarine','ultramarine',[140,146,172],'8c92ac',9212588],
['nbs-iscc-s:ultramarine.204','ultramarine','ultramarine',[76,81,109],'4c516d',5001581],
['nbs-iscc-s:ultramarine.210','ultramarine','ultramarine',[140,130,182],'8c82b6',9208502],
['nbs-iscc-s:umberbrown.61','umberbrown','umber brown',[99,81,71],'635147',6508871],
['nbs-iscc-s:vermilion.6','vermilion','vermilion',[192,128,129],'c08081',12615809],
['nbs-iscc-s:vermilion.12','vermilion','vermilion',[188,63,74],'bc3f4a',12336970],
['nbs-iscc-s:vermilion.15','vermilion','vermilion',[171,78,82],'ab4e52',11226706],
['nbs-iscc-s:vermilion.18','vermilion','vermilion',[173,136,132],'ad8884',11372676],
['nbs-iscc-s:vermilion.19','vermilion','vermilion',[144,93,93],'905d5d',9461085],
['nbs-iscc-s:vermilion.26','vermilion','vermilion',[248,131,121],'f88379',16286585],
['nbs-iscc-s:vermilion.27','vermilion','vermilion',[230,103,97],'e66761',15099745],
['nbs-iscc-s:vermilion.30','vermilion','vermilion',[196,131,121],'c48379',12878713],
['nbs-iscc-s:vermilion.35','vermilion','vermilion',[217,96,59],'d9603b',14245947],
['nbs-iscc-s:vermilion.36','vermilion','vermilion',[170,56,30],'aa381e',11155486],
['nbs-iscc-s:vermilion.37','vermilion','vermilion',[203,109,81],'cb6d51',13331793],
['nbs-iscc-s:vermilion.38','vermilion','vermilion',[158,71,50],'9e4732',10372914],
['nbs-iscc-s:violet.16','violet','violet',[114,47,55],'722f37',7483191],
['nbs-iscc-s:violet.200','violet','violet',[78,81,128],'4e5180',5132672],
['nbs-iscc-s:violet.206','violet','violet',[126,115,184],'7e73b8',8287160],
['nbs-iscc-s:violet.210','violet','violet',[140,130,182],'8c82b6',9208502],
['nbs-iscc-s:violet.211','violet','violet',[96,78,129],'604e81',6311553],
['nbs-iscc-s:violet.214','violet','violet',[150,144,171],'9690ab',9867435],
['nbs-iscc-s:violet.217','violet','violet',[211,153,230],'d399e6',13867494],
['nbs-iscc-s:violet.218','violet','violet',[135,86,146],'875692',8869522],
['nbs-iscc-s:violet.222','violet','violet',[182,149,192],'b695c0',11965888],
['nbs-iscc-s:violet.223','violet','violet',[134,96,142],'86608e',8806542],
['nbs-iscc-s:violet.226','violet','violet',[214,202,221],'d6cadd',14076637],
['nbs-iscc-s:violet.227','violet','violet',[170,152,169],'aa98a9',11180201],
['nbs-iscc-s:violet.228','violet','violet',[121,104,120],'796878',7956600],
['nbs-iscc-s:violet.229','violet','violet',[80,64,77],'50404d',5259341],
['nbs-iscc-s:violet.238','violet','violet',[112,41,99],'702963',7350627],
['nbs-iscc-s:violet.240','violet','violet',[183,132,167],'b784a7',12027047],
['nbs-iscc-s:violet.244','violet','violet',[170,138,158],'aa8a9e',11176606],
['nbs-iscc-s:violet.245','violet','violet',[131,100,121],'836479',8610937],
['nbs-iscc-s:violet.250','violet','violet',[213,151,174],'d597ae',13997998],
['nbs-iscc-s:violet.252','violet','violet',[232,204,215],'e8ccd7',15256791],
['nbs-iscc-s:violet.253','violet','violet',[195,166,177],'c3a6b1',12822193],
['nbs-iscc-s:violet.261','violet','violet',[175,134,142],'af868e',11503246],
['nbs-iscc-s:violetblack.20','violetblack','violet black',[84,61,63],'543d3f',5520703],
['nbs-iscc-s:violetblack.234','violetblack','violet black',[93,85,91],'5d555b',6116699],
['nbs-iscc-s:violetblue.196','violetblue','violet blue',[84,90,167],'545aa7',5528231],
['nbs-iscc-s:violetbrown.19','violetbrown','violet brown',[144,93,93],'905d5d',9461085],
['nbs-iscc-s:violetbrown.228','violetbrown','violet brown',[121,104,120],'796878',7956600],
['nbs-iscc-s:violetbrown.259','violetbrown','violet brown',[103,49,71],'673147',6762823],
['nbs-iscc-s:yellow.28','yellow','yellow',[244,194,194],'f4c2c2',16040642],
['nbs-iscc-s:yellow.29','yellow','yellow',[217,166,169],'d9a6a9',14263977],
['nbs-iscc-s:yellow.48','yellow','yellow',[243,132,0],'f38400',15959040],
['nbs-iscc-s:yellow.49','yellow','yellow',[253,148,63],'fd943f',16618559],
['nbs-iscc-s:yellow.50','yellow','yellow',[237,135,45],'ed872d',15566637],
['nbs-iscc-s:yellow.52','yellow','yellow',[250,181,127],'fab57f',16430463],
['nbs-iscc-s:yellow.53','yellow','yellow',[217,144,88],'d99058',14258264],
['nbs-iscc-s:yellow.57','yellow','yellow',[166,123,91],'a67b5b',10910555],
['nbs-iscc-s:yellow.66','yellow','yellow',[246,166,0],'f6a600',16164352],
['nbs-iscc-s:yellow.67','yellow','yellow',[255,193,79],'ffc14f',16761167],
['nbs-iscc-s:yellow.68','yellow','yellow',[234,162,33],'eaa221',15376929],
['nbs-iscc-s:yellow.69','yellow','yellow',[201,133,0],'c98500',13206784],
['nbs-iscc-s:yellow.70','yellow','yellow',[251,201,127],'fbc97f',16501119],
['nbs-iscc-s:yellow.71','yellow','yellow',[227,168,87],'e3a857',14919767],
['nbs-iscc-s:yellow.72','yellow','yellow',[190,138,61],'be8a3d',12487229],
['nbs-iscc-s:yellow.73','yellow','yellow',[250,214,165],'fad6a5',16438949],
['nbs-iscc-s:yellow.76','yellow','yellow',[193,154,107],'c19a6b',12687979],
['nbs-iscc-s:yellow.82','yellow','yellow',[243,195,0],'f3c300',15975168],
['nbs-iscc-s:yellow.83','yellow','yellow',[250,218,94],'fada5e',16439902],
['nbs-iscc-s:yellow.84','yellow','yellow',[212,175,55],'d4af37',13938487],
['nbs-iscc-s:yellow.85','yellow','yellow',[175,141,19],'af8d13',11504915],
['nbs-iscc-s:yellow.86','yellow','yellow',[248,222,126],'f8de7e',16309886],
['nbs-iscc-s:yellow.87','yellow','yellow',[201,174,93],'c9ae5d',13217373],
['nbs-iscc-s:yellow.88','yellow','yellow',[171,145,68],'ab9144',11243844],
['nbs-iscc-s:yellow.89','yellow','yellow',[243,229,171],'f3e5ab',15984043],
['nbs-iscc-s:yellow.90','yellow','yellow',[194,178,128],'c2b280',12759680],
['nbs-iscc-s:yellow.92','yellow','yellow',[240,234,214],'f0ead6',15788758],
['nbs-iscc-s:yellowbrown.33','yellowbrown','yellow brown',[194,172,153],'c2ac99',12758169],
['nbs-iscc-s:yellowbrown.38','yellowbrown','yellow brown',[158,71,50],'9e4732',10372914],
['nbs-iscc-s:yellowbrown.42','yellowbrown','yellow brown',[168,124,109],'a87c6d',11041901],
['nbs-iscc-s:yellowbrown.45','yellowbrown','yellow brown',[151,127,115],'977f73',9928563],
['nbs-iscc-s:yellowbrown.53','yellowbrown','yellow brown',[217,144,88],'d99058',14258264],
['nbs-iscc-s:yellowbrown.54','yellowbrown','yellow brown',[174,105,56],'ae6938',11430200],
['nbs-iscc-s:yellowbrown.55','yellowbrown','yellow brown',[128,70,27],'80461b',8406555],
['nbs-iscc-s:yellowbrown.57','yellowbrown','yellow brown',[166,123,91],'a67b5b',10910555],
['nbs-iscc-s:yellowbrown.72','yellowbrown','yellow brown',[190,138,61],'be8a3d',12487229],
['nbs-iscc-s:yellowbrown.74','yellowbrown','yellow brown',[153,101,21],'996515',10052885],
['nbs-iscc-s:yellowbrown.106','yellowbrown','yellow brown',[134,126,54],'867e36',8814134],
['nbs-iscc-s:yellowbuff.29','yellowbuff','yellow buff',[217,166,169],'d9a6a9',14263977],
['nbs-iscc-s:yellowbuff.53','yellowbuff','yellow buff',[217,144,88],'d99058',14258264],
['nbs-iscc-s:yellowbuff.71','yellowbuff','yellow buff',[227,168,87],'e3a857',14919767],
['nbs-iscc-s:yellowbuff.73','yellowbuff','yellow buff',[250,214,165],'fad6a5',16438949],
['nbs-iscc-s:yellowbuff.76','yellowbuff','yellow buff',[193,154,107],'c19a6b',12687979],
['nbs-iscc-s:yellowbuff.89','yellowbuff','yellow buff',[243,229,171],'f3e5ab',15984043],
['nbs-iscc-s:yellowbuff.90','yellowbuff','yellow buff',[194,178,128],'c2b280',12759680],
['nbs-iscc-s:yellowgreen.117','yellowgreen','yellow green',[126,159,46],'7e9f2e',8298286],
['nbs-iscc-s:yellowgreen.120','yellowgreen','yellow green',[138,154,91],'8a9a5b',9083483],
['nbs-iscc-s:yellowgreen.129','yellowgreen','yellow green',[39,166,76],'27a64c',2598476],
['nbs-iscc-s:yellowgreen.130','yellowgreen','yellow green',[131,211,125],'83d37d',8639357],
['nbs-iscc-s:yellowgreen.131','yellowgreen','yellow green',[68,148,74],'44944a',4494410],
['nbs-iscc-s:yellowgreen.132','yellowgreen','yellow green',[0,98,45],'00622d',25133],
['nbs-iscc-s:yellowgreen.134','yellowgreen','yellow green',[182,229,175],'b6e5af',11986351],
['nbs-iscc-s:yellowgreen.135','yellowgreen','yellow green',[147,197,146],'93c592',9684370],
['nbs-iscc-s:yellowgreen.136','yellowgreen','yellow green',[103,146,103],'679267',6787687],
['nbs-iscc-s:yellowgreen.137','yellowgreen','yellow green',[53,94,59],'355e3b',3497531],
['nbs-iscc-s:yellowgreen.139','yellowgreen','yellow green',[0,136,86],'008856',34902],
['nbs-iscc-s:yellowgreen.140','yellowgreen','yellow green',[62,180,137],'3eb489',4109449],
['nbs-iscc-s:yellowgreen.141','yellowgreen','yellow green',[0,121,89],'007959',31065],
['nbs-iscc-s:yellowgreen.143','yellowgreen','yellow green',[142,209,178],'8ed1b2',9359794],
['nbs-iscc-s:yellowgreen.144','yellowgreen','yellow green',[106,171,142],'6aab8e',6990734],
['nbs-iscc-s:yellowgreen.145','yellowgreen','yellow green',[59,120,97],'3b7861',3897441],
['nbs-iscc-s:yelloworange.50','yelloworange','yellow orange',[237,135,45],'ed872d',15566637],
['nbs-iscc-s:yelloworange.53','yelloworange','yellow orange',[217,144,88],'d99058',14258264]
    ];
}

sub _description {
    return {
          'subtitle' => 'Postage-Stamp Color Names',
          'title' => 'NBS/ISCC S',
          'description' => '    Kenneth L. Kelly and Deanne B. Judd.
    "Color: Universal Language and Dictionary of Names",
    National Bureau of Standards,
    Spec. Publ. 440, Dec. 1976, 189 pages.


    Color Name Dictionary: (S) Postage-Stamp Color Names

    Postage-Stamp Color Names,
    William H. Beck 

This dictionary is a hybrid of combinatorial and idiosyncratic styles. 178 names map to 905 colors, using 223 of 267 centroids. Despite covering 84% of the centroids, averaging over 5 colors per name makes this dictionary uselessly unspecific.

[http://swissnet.ai.mit.edu/~jaffer/Color/Dictionaries#S]
'
        }

}

1;
