/*	$NetBSD: nouveau_nvkm_subdev_instmem_nv40.c,v 1.7 2018/08/29 15:52:27 riastradh Exp $	*/

/*
 * Copyright 2012 Red Hat Inc.
 *
 * Permission is hereby granted, free of charge, to any person obtaining a
 * copy of this software and associated documentation files (the "Software"),
 * to deal in the Software without restriction, including without limitation
 * the rights to use, copy, modify, merge, publish, distribute, sublicense,
 * and/or sell copies of the Software, and to permit persons to whom the
 * Software is furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.  IN NO EVENT SHALL
 * THE COPYRIGHT HOLDER(S) OR AUTHOR(S) BE LIABLE FOR ANY CLAIM, DAMAGES OR
 * OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE,
 * ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR
 * OTHER DEALINGS IN THE SOFTWARE.
 *
 * Authors: Ben Skeggs
 */
#include <sys/cdefs.h>
__KERNEL_RCSID(0, "$NetBSD: nouveau_nvkm_subdev_instmem_nv40.c,v 1.7 2018/08/29 15:52:27 riastradh Exp $");

#define nv40_instmem(p) container_of((p), struct nv40_instmem, base)
#include "priv.h"

#include <core/memory.h>
#include <core/ramht.h>
#include <engine/gr/nv40.h>

#ifdef __NetBSD__
#  define	__iomem	__nvkm_memory_iomem
#endif

struct nv40_instmem {
	struct nvkm_instmem base;
	struct nvkm_mm heap;
#ifdef __NetBSD__
	bus_space_tag_t iomemt;
	bus_space_handle_t iomemh;
	bus_size_t iomemsz;
#endif
	void __iomem *iomem;
};

/******************************************************************************
 * instmem object implementation
 *****************************************************************************/
#define nv40_instobj(p) container_of((p), struct nv40_instobj, memory)

struct nv40_instobj {
	struct nvkm_memory memory;
	struct nv40_instmem *imem;
	struct nvkm_mm_node *node;
};

static enum nvkm_memory_target
nv40_instobj_target(struct nvkm_memory *memory)
{
	return NVKM_MEM_TARGET_INST;
}

static u64
nv40_instobj_addr(struct nvkm_memory *memory)
{
	return nv40_instobj(memory)->node->offset;
}

static u64
nv40_instobj_size(struct nvkm_memory *memory)
{
	return nv40_instobj(memory)->node->length;
}

static void __iomem *
nv40_instobj_acquire(struct nvkm_memory *memory)
{
	struct nv40_instobj *iobj = nv40_instobj(memory);
	return (char __iomem *)iobj->imem->iomem + iobj->node->offset;
}

static void
nv40_instobj_release(struct nvkm_memory *memory)
{
}

static u32
nv40_instobj_rd32(struct nvkm_memory *memory, u64 offset)
{
	struct nv40_instobj *iobj = nv40_instobj(memory);
#ifdef __NetBSD__
	offset += iobj->node->offset;
	return bus_space_read_stream_4(iobj->imem->iomemt, iobj->imem->iomemh,
	    offset);
#else
	return ioread32_native(iobj->imem->iomem + iobj->node->offset + offset);
#endif
}

static void
nv40_instobj_wr32(struct nvkm_memory *memory, u64 offset, u32 data)
{
	struct nv40_instobj *iobj = nv40_instobj(memory);
#ifdef __NetBSD__
	offset += iobj->node->offset;
	bus_space_write_stream_4(iobj->imem->iomemt, iobj->imem->iomemh,
	    offset, data);
#else
	iowrite32_native(data, iobj->imem->iomem + iobj->node->offset + offset);
#endif
}

static void *
nv40_instobj_dtor(struct nvkm_memory *memory)
{
	struct nv40_instobj *iobj = nv40_instobj(memory);
	mutex_lock(&iobj->imem->base.subdev.mutex);
	nvkm_mm_free(&iobj->imem->heap, &iobj->node);
	mutex_unlock(&iobj->imem->base.subdev.mutex);
	return iobj;
}

static const struct nvkm_memory_func
nv40_instobj_func = {
	.dtor = nv40_instobj_dtor,
	.target = nv40_instobj_target,
	.size = nv40_instobj_size,
	.addr = nv40_instobj_addr,
	.acquire = nv40_instobj_acquire,
	.release = nv40_instobj_release,
	.rd32 = nv40_instobj_rd32,
	.wr32 = nv40_instobj_wr32,
};

static int
nv40_instobj_new(struct nvkm_instmem *base, u32 size, u32 align, bool zero,
		 struct nvkm_memory **pmemory)
{
	struct nv40_instmem *imem = nv40_instmem(base);
	struct nv40_instobj *iobj;
	int ret;

	if (!(iobj = kzalloc(sizeof(*iobj), GFP_KERNEL)))
		return -ENOMEM;
	*pmemory = &iobj->memory;

	nvkm_memory_ctor(&nv40_instobj_func, &iobj->memory);
	iobj->imem = imem;

	mutex_lock(&imem->base.subdev.mutex);
	ret = nvkm_mm_head(&imem->heap, 0, 1, size, size,
			   align ? align : 1, &iobj->node);
	mutex_unlock(&imem->base.subdev.mutex);
	return ret;
}

/******************************************************************************
 * instmem subdev implementation
 *****************************************************************************/

static u32
nv40_instmem_rd32(struct nvkm_instmem *base, u32 addr)
{
#ifdef __NetBSD__
	struct nv40_instmem *imem = nv40_instmem(base);
	return bus_space_read_stream_4(imem->iomemt, imem->iomemh, addr);
#else
	return ioread32_native(nv40_instmem(base)->iomem + addr);
#endif
}

static void
nv40_instmem_wr32(struct nvkm_instmem *base, u32 addr, u32 data)
{
#ifdef __NetBSD__
	struct nv40_instmem *imem = nv40_instmem(base);
	bus_space_write_stream_4(imem->iomemt, imem->iomemh, addr, data);
#else
	iowrite32_native(data, nv40_instmem(base)->iomem + addr);
#endif
}

static int
nv40_instmem_oneinit(struct nvkm_instmem *base)
{
	struct nv40_instmem *imem = nv40_instmem(base);
	struct nvkm_device *device = imem->base.subdev.device;
	int ret, vs;

	/* PRAMIN aperture maps over the end of vram, reserve enough space
	 * to fit graphics contexts for every channel, the magics come
	 * from engine/gr/nv40.c
	 */
	vs = hweight8((nvkm_rd32(device, 0x001540) & 0x0000ff00) >> 8);
	if      (device->chipset == 0x40) imem->base.reserved = 0x6aa0 * vs;
	else if (device->chipset  < 0x43) imem->base.reserved = 0x4f00 * vs;
	else if (nv44_gr_class(device))   imem->base.reserved = 0x4980 * vs;
	else				  imem->base.reserved = 0x4a40 * vs;
	imem->base.reserved += 16 * 1024;
	imem->base.reserved *= 32;		/* per-channel */
	imem->base.reserved += 512 * 1024;	/* pci(e)gart table */
	imem->base.reserved += 512 * 1024;	/* object storage */
	imem->base.reserved = round_up(imem->base.reserved, 4096);

	ret = nvkm_mm_init(&imem->heap, 0, imem->base.reserved, 1);
	if (ret)
		return ret;

	/* 0x00000-0x10000: reserve for probable vbios image */
	ret = nvkm_memory_new(device, NVKM_MEM_TARGET_INST, 0x10000, 0, false,
			      &imem->base.vbios);
	if (ret)
		return ret;

	/* 0x10000-0x18000: reserve for RAMHT */
	ret = nvkm_ramht_new(device, 0x08000, 0, NULL, &imem->base.ramht);
	if (ret)
		return ret;

	/* 0x18000-0x18200: reserve for RAMRO
	 * 0x18200-0x20000: padding
	 */
	ret = nvkm_memory_new(device, NVKM_MEM_TARGET_INST, 0x08000, 0, false,
			      &imem->base.ramro);
	if (ret)
		return ret;

	/* 0x20000-0x21000: reserve for RAMFC
	 * 0x21000-0x40000: padding and some unknown crap
	 */
	ret = nvkm_memory_new(device, NVKM_MEM_TARGET_INST, 0x20000, 0, true,
			      &imem->base.ramfc);
	if (ret)
		return ret;

	return 0;
}

static void *
nv40_instmem_dtor(struct nvkm_instmem *base)
{
	struct nv40_instmem *imem = nv40_instmem(base);
	nvkm_memory_del(&imem->base.ramfc);
	nvkm_memory_del(&imem->base.ramro);
	nvkm_ramht_del(&imem->base.ramht);
	nvkm_memory_del(&imem->base.vbios);
	nvkm_mm_fini(&imem->heap);
	if (imem->iomem)
#ifdef __NetBSD__
		bus_space_unmap(imem->iomemt, imem->iomemh, imem->iomemsz);
#else
		iounmap(imem->iomem);
#endif
	return imem;
}

static const struct nvkm_instmem_func
nv40_instmem = {
	.dtor = nv40_instmem_dtor,
	.oneinit = nv40_instmem_oneinit,
	.rd32 = nv40_instmem_rd32,
	.wr32 = nv40_instmem_wr32,
	.memory_new = nv40_instobj_new,
	.persistent = false,
	.zero = false,
};

int
nv40_instmem_new(struct nvkm_device *device, int index,
		 struct nvkm_instmem **pimem)
{
	struct nv40_instmem *imem;
	int bar;

	if (!(imem = kzalloc(sizeof(*imem), GFP_KERNEL)))
		return -ENOMEM;
	nvkm_instmem_ctor(&nv40_instmem, device, index, &imem->base);
	*pimem = &imem->base;

	/* map bar */
	if (device->func->resource_size(device, 2))
		bar = 2;
	else
		bar = 3;

#ifdef __NetBSD__
    {
	bus_addr_t iomembase;
	bus_size_t iomemsz;
	int ret;

	imem->iomemt = device->func->resource_tag(device, bar);
	iomembase = device->func->resource_addr(device, bar);
	iomemsz = device->func->resource_size(device, bar);
	/* XXX errno NetBSD->Linux */
	ret = -bus_space_map(imem->iomemt, iomembase, iomemsz,
	    BUS_SPACE_MAP_LINEAR, &imem->iomemh);
	if (ret) {
		nvkm_error(&imem->base.subdev, "unable to map PRAMIN BAR %d"
		    ": %d\n", bar, ret);
		return ret;
	}
	imem->iomemsz = iomemsz;
	imem->iomem = bus_space_vaddr(imem->iomemt, imem->iomemh);
    }
#else
	imem->iomem = ioremap(device->func->resource_addr(device, bar),
			      device->func->resource_size(device, bar));
	if (!imem->iomem) {
		nvkm_error(&imem->base.subdev, "unable to map PRAMIN BAR\n");
		return -EFAULT;
	}
#endif

	return 0;
}
