;;; iiimcf-sc.el --- IIIMCF server control input method.
;;; 

;; Copyright (C) 2000 MIYASHITA Hisashi

;; Author: MIYASHITA Hisashi <himi@m17n.org>

;; This program is free software; you can redistribute it and/or modify
;; it under the terms of the GNU General Public License as published by
;; the Free Software Foundation; either version 2, or (at your option)
;; any later version.

;; This program is distributed in the hope that it will be useful,
;; but WITHOUT ANY WARRANTY; without even the implied warranty of
;; MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
;; GNU General Public License for more details.

;; You should have received a copy of the GNU General Public License
;; along with GNU Emacs; see the file COPYING.  If not, write to the
;; Free Software Foundation, Inc., 59 Temple Place - Suite 330,
;; Boston, MA 02111-1307, USA.

;;; Commentary:
;; This module provides a siple input method
;; under the server-side control via IIIM.

;;; Code:

(require 'iiimcf)

;;; version.

(defconst iiimcf-server-control-version "0.1 (Kamo)")

;;; Customizable options.

(defgroup iiimcf-server-control nil
  "*IIIMCF server control input method."
  :tag "IIIMCF Server Control Input Method"
  :group 'applications
  :group 'i18n)

(defcustom iiimcf-server-control-default-port 9010
  "*Default port number for IIIM server."
  :tag "Default port for IIIMCF Server Control"
  :group 'iiimcf-server-control :type 'integer)

(defcustom iiimcf-server-control-hostlist '("localhost")
  "*A list of IIIM Server hosts.
If each hostname has the form of \"HOSTNAME:N\", use N-th port."
  :tag "Hostlist for IIIMCF Server Control"
  :group 'iiimcf-server-control :type '(repeat string))

(defcustom iiimcf-server-control-username
  (concat (user-login-name) "@" (system-name))
  "*Username passed to IIIM server."
  :tag "Username for IIIMCF Server Control"
  :group 'iiimcf-server-control :type 'string)

(defcustom iiimcf-server-control-default-language "ja"
  "*Use this language by default."
  :tag "Default language for IIIMCF Server Control"
  :group 'iiimcf-server-control :type 'string)

(defcustom iiimcf-server-control-async-invocation-p t
  "*Receives and responds incomming messages asynchronously, if non-nil."
  :tag "Aaync invocation flag for IIIMCF Server Control"
  :group 'iiimcf-server-control :type 'boolean)

;;; preedit text

(defcustom iiimcf-server-control-preedit-open-string "|"
  "*Put this string before preedit string."
  :tag "Preedit open string for IIIMCF Server Control"
  :group 'iiimcf-server-control :type 'string)

(defcustom iiimcf-server-control-preedit-close-string "|"
  "*Put this string after preedit string."
  :tag "Preedit close string for IIIMCF Server Control"
  :group 'iiimcf-server-control :type 'string)

(defcustom iiimcf-server-control-preedit-face-alist
  '((reverse region ("[" . "]"))
    (underline underline ("_" . "_"))
    (highlight bold ("+" . "+")))
  "*A-list used for decorating preedit text."
  :tag "Preedit decoration settings for IIIMCF Server Control"
  :group 'iiimcf-server-control
  :type '(list (list :tag "Used for REVERSE text."
		     (const reverse) face (cons string string))
	       (list :tag "Used for UNDERLINE text."
		     (const underline) face (cons string string))
	       (list :tag "Used for HIGHLIGHT text."
		     (const highlight) face (cons string string))))

(defcustom iiimcf-server-control-preedit-use-face-p
  (if window-system t nil)
  "*Use faces when drawing preedit text, if non-nil."
  :tag "Flag whether faces are used for drawing preedit text."
  :group 'iiimcf-server-control :type 'boolean)

;;; internal variables

(defvar iiimcf-server-control-com-id nil)
(defvar iiimcf-server-control-im-id nil)
(defvar iiimcf-server-control-ic-id nil)
(defvar iiimcf-server-control-dynamic-event-flow-p nil)
(defvar iiimcf-server-control-editing-buffer nil
  "The buffer where a user is inputting currently.")
(defvar iiimcf-server-control-preedit "")
(defvar iiimcf-server-control-preedit-marker (make-marker))
(defvar iiimcf-server-control-caret-overlay
  (make-overlay 0 0))
(overlay-put iiimcf-server-control-caret-overlay
	     'face 'modeline)
(defvar iiimcf-server-control-maintaining-preedit-text nil)

(defvar iiimcf-server-control-mode-line "")
(make-variable-buffer-local 'iiimcf-server-control-mode-line)
(if (not (memq 'iiimcf-server-control-mode-line mode-line-format))
    (let ((ll mode-line-format)
	  gen elem slot)
      (setq-default
       mode-line-format
       (catch 'tag
	 (while (setq elem (car ll))
	   (if (or (eq elem 'mode-line-mule-info)
		   (equal elem "-%-"))
	       (throw 'tag (nconc (nreverse gen)
				  (list 'iiimcf-server-control-mode-line)
				  ll)))
	   (setq gen (cons elem gen)
		 ll (cdr ll)))))))

;;; lookup choice
(defvar iiimcf-server-control-current-lookup-choice nil)
(defvar iiimcf-server-control-lookup-choice-configuration nil)

;;; keymap
(defvar iiimcf-server-control-keymap
  (let ((map (make-keymap))
	(i 0))
    (while (< i 127)
      (define-key map (char-to-string i)
	'iiimcf-server-control-keyforward)
      (setq i (1+ i)))
    (setq i 128)
    (while (< i 256)
      (define-key map (vector i)
	'iiimcf-server-control-keyforward)
      (setq i (1+ i)))
    (define-key map "\C-m" 'iiimcf-server-control-keyforward)
    (define-key map "\C-p" 'iiimcf-server-control-keyforward)
    (define-key map "\C-n" 'iiimcf-server-control-keyforward)
    (define-key map "\C-b" 'iiimcf-server-control-keyforward)
    (define-key map "\C-f" 'iiimcf-server-control-keyforward)
    (define-key map "\C-a" 'iiimcf-server-control-keyforward)
    (define-key map "\C-e" 'iiimcf-server-control-keyforward)
    (define-key map "\C-d" 'iiimcf-server-control-keyforward)
    (define-key map "\C-k" 'iiimcf-server-control-keyforward)
    (define-key map "\C-h" 'iiimcf-server-control-keyforward)
    (define-key map "\177" 'iiimcf-server-control-keyforward)
    (define-key map [delete] 'iiimcf-server-control-keyforward)
    (define-key map [backspace] 'iiimcf-server-control-keyforward)
    (define-key map [return] 'iiimcf-server-control-keyforward)
    (define-key map [up] 'iiimcf-server-control-keyforward)
    (define-key map [down] 'iiimcf-server-control-keyforward)
    (define-key map [left] 'iiimcf-server-control-keyforward)
    (define-key map [right] 'iiimcf-server-control-keyforward)
    (define-key map [S-up] 'iiimcf-server-control-keyforward)
    (define-key map [S-down] 'iiimcf-server-control-keyforward)
    (define-key map [S-left] 'iiimcf-server-control-keyforward)
    (define-key map [S-right] 'iiimcf-server-control-keyforward)
    (define-key map [C-up] 'iiimcf-server-control-keyforward)
    (define-key map [C-down] 'iiimcf-server-control-keyforward)
    (define-key map [C-left] 'iiimcf-server-control-keyforward)
    (define-key map [C-right] 'iiimcf-server-control-keyforward)
    map)
  "Keymap used for forwarding keyevents to IIIM server side.")

(defun iiimcf-server-control-parse-hostname (hostname)
  (if (string-match ":" hostname)
      (list
       (substring hostname 0 (match-beginning 0))
       (string-to-number (substring hostname (match-end 0))))
    (list
     hostname
     iiimcf-server-control-default-port)))

(defun iiimcf-server-control-convert-iiim-feedback
  (feedbacks type)
  (if (eq type 'face)
      (let (slot result)
	(while feedbacks
	  (setq slot (assq (car feedbacks)
			   iiimcf-server-control-preedit-face-alist)
		feedbacks (cdr feedbacks))
	  (if slot
	      (setq result (cons (nth 1 slot) result))))
	result)
    nil))

(defun iiimcf-server-control-setup-event-flow-mode (mes)
  (setq iiimcf-server-control-dynamic-event-flow-p t))

(defun iiimcf-server-control-async-handler (com-id)
  (condition-case err
      (iiimcf-message-manager com-id nil)
    (iiimp-fatal
     (iiimcf-server-control-shutdown)
     (signal (car err) (cdr err)))))

(defun iiimcf-server-control-setup ()
  (if (and iiimcf-server-control-com-id
	   (not
	    (iiimp-check-channel-connection
	     iiimcf-server-control-com-id)))
      (iiimcf-server-control-shutdown))
  (if (null iiimcf-server-control-com-id)
      (setq iiimcf-server-control-com-id
	    (apply 
	     (function iiimp-create-network-channel)
	     (iiimcf-server-control-parse-hostname
	      (car iiimcf-server-control-hostlist)))))
  (let ((com-id iiimcf-server-control-com-id)
	mes)
    (iiimcf-register-handler
     (function
      iiimcf-server-control-setup-event-flow-mode)
     (list 'iiimp-im-register-trigger-keys
	   iiimcf-server-control-com-id))
    (if (null iiimcf-server-control-im-id)
	(progn
	  (iiimp-send-message
	   'iiimp-im-connect
	   com-id
	   iiimcf-server-control-username)
	  (setq mes (iiimcf-message-manager
		     com-id
		     (list
		      (list 'iiimp-im-connect-reply com-id))))
	  (setq iiimcf-server-control-im-id
		(nth 2 mes))
	  (iiimcf-send-client-descriptor
	   com-id iiimcf-server-control-im-id
	   (format "IIIMCF-Server-Control/%s"
		   iiimcf-server-control-version))))
    (if (null iiimcf-server-control-ic-id)
	(progn
	  (iiimp-send-message
	   'iiimp-im-createic
	   com-id iiimcf-server-control-im-id
	   (if iiimcf-server-control-default-language
	       (list (cons 'input-language
			   iiimcf-server-control-default-language))
	     nil))
	  (setq mes
		(iiimcf-message-manager
		 com-id
		 (list
		  (list 'iiimp-im-createic-reply
			com-id iiimcf-server-control-im-id))))
	  (setq iiimcf-server-control-ic-id
		(nth 3 mes))))
    (iiimcf-register-handler
     (function iiimcf-server-control-trigger-notify)
     (list 'iiimp-im-trigger-notify
	   com-id
	   iiimcf-server-control-im-id
	   iiimcf-server-control-ic-id))
    (iiimcf-register-handler
     (function iiimcf-server-control-maintain-preedit-text)
     (list 'iiimp-im-preedit-draw
	   com-id
	   iiimcf-server-control-im-id
	   iiimcf-server-control-ic-id))
    (iiimcf-register-handler
     (function iiimcf-server-control-commit-string)
     (list 'iiimp-im-commit-string
	   com-id
	   iiimcf-server-control-im-id
	   iiimcf-server-control-ic-id))
    (iiimcf-register-handler
     (function iiimcf-server-control-forward-event)
     (list 'iiimp-im-forward-event
	   com-id
	   iiimcf-server-control-im-id
	   iiimcf-server-control-ic-id))
    (iiimcf-register-handler
     (function iiimcf-server-control-forward-event)
     (list 'iiimp-im-forward-event-with-operations
	   com-id
	   iiimcf-server-control-im-id
	   iiimcf-server-control-ic-id))
    (iiimcf-register-handler
     (function iiimcf-server-control-prepare-lookup-choice)
     (list 'iiimp-im-lookup-choice-start
	   com-id
	   iiimcf-server-control-im-id
	   iiimcf-server-control-ic-id))
    (iiimcf-register-handler
     (function iiimcf-server-control-draw-lookup-choice)
     (list 'iiimp-im-lookup-choice-draw
	   com-id
	   iiimcf-server-control-im-id
	   iiimcf-server-control-ic-id))
    (iiimcf-register-handler
     (function iiimcf-server-control-redraw-lookup-choice)
     (list 'iiimp-im-lookup-choice-process
	   com-id
	   iiimcf-server-control-im-id
	   iiimcf-server-control-ic-id))
    (iiimcf-register-handler
     (function iiimcf-server-control-clear-lookup-choice)
     (list 'iiimp-im-lookup-choice-done
	   com-id
	   iiimcf-server-control-im-id
	   iiimcf-server-control-ic-id))
    (iiimcf-register-handler
     (function iiimcf-server-control-draw-status)
     (list 'iiimp-im-status-draw
	   com-id
	   iiimcf-server-control-im-id
	   iiimcf-server-control-ic-id))
    (iiimcf-register-handler
     (function iiimcf-server-control-draw-aux-data)
     (list 'iiimp-im-aux-draw
	   com-id
	   iiimcf-server-control-im-id
	   iiimcf-server-control-ic-id))

    (if iiimcf-server-control-async-invocation-p
	(iiimp-enable-async-invocation
	 com-id
	 (function iiimcf-server-control-async-handler)))))

(defun iiimcf-server-control-shutdown ()
  (if (or (null iiimcf-server-control-com-id)
	  (not (iiimp-check-channel-connection
		iiimcf-server-control-com-id)))
      (setq iiimcf-server-control-ic-id nil
	    iiimcf-server-control-im-id nil))
  (if iiimcf-server-control-ic-id
      (progn
	(iiimp-send-message
	 'iiimp-im-destroyic
	 iiimcf-server-control-com-id
	 iiimcf-server-control-im-id
	 iiimcf-server-control-ic-id)
	(iiimcf-message-manager
	 iiimcf-server-control-com-id
	 (list
	  (list
	   'iiimp-im-destroyic-reply
	   iiimcf-server-control-com-id
	   iiimcf-server-control-im-id
	   iiimcf-server-control-ic-id)))
	(setq iiimcf-server-control-ic-id nil)))
  (if iiimcf-server-control-im-id
      (progn
	(iiimp-send-message
	 'iiimp-im-disconnect
	 iiimcf-server-control-com-id
	 iiimcf-server-control-im-id)
	(iiimcf-message-manager
	 iiimcf-server-control-com-id
	 (list
	  (list
	   'iiimp-im-disconnect-reply
	   iiimcf-server-control-com-id
	   iiimcf-server-control-im-id)))
	(setq iiimcf-server-control-im-id nil)))
  (if iiimcf-server-control-com-id
      (progn
	(while (not (iiimp-destroy-network-channel
		     iiimcf-server-control-com-id))
	  (sleep-for 1))
	(setq iiimcf-server-control-com-id nil)))
  (setq iiimcf-server-control-dynamic-event-flow-p nil)
  (inactivate-input-method)

  (iiimcf-remove-handler
   (function
    iiimcf-server-control-setup-event-flow-mode)
   nil)
  (iiimcf-remove-handler
   (function iiimcf-server-control-maintain-preedit-text)
   nil)
  (iiimcf-remove-handler
   (function iiimcf-server-control-commit-string)
   nil)
  (iiimcf-remove-handler
   (function iiimcf-server-control-forward-event)
   nil)
  (iiimcf-remove-handler
   (function iiimcf-server-control-prepare-lookup-choice)
   nil)
  (iiimcf-remove-handler
   (function iiimcf-server-control-draw-lookup-choice)
   nil)
  (iiimcf-remove-handler
   (function iiimcf-server-control-redraw-lookup-choice)
   nil)
  (iiimcf-remove-handler
   (function iiimcf-server-control-clear-lookup-choice)
   nil)
  (iiimcf-remove-handler
   (function iiimcf-server-control-draw-status)
   nil)
  (iiimcf-remove-handler
   (function iiimcf-server-control-draw-aux-data)
   nil))

(defun iiimcf-server-control-inactivate ()
  "Inactivate IIIMCF server control input method."
  (interactive)
  (iiimcf-server-control-activate -1))

(defun iiimcf-server-control-notify-trigger (flag)
  (iiimp-send-message
   'iiimp-im-trigger-notify
   iiimcf-server-control-com-id
   iiimcf-server-control-im-id
   iiimcf-server-control-ic-id
   flag)
  (iiimcf-message-manager
   iiimcf-server-control-com-id
   (list
    (list 'iiimp-im-trigger-notify-reply
	  iiimcf-server-control-com-id
	  iiimcf-server-control-im-id
	  iiimcf-server-control-ic-id))))

(defun iiimcf-server-control-activate (&optional arg)
  (if (and arg
	  (< (prefix-numeric-value arg) 0))
      ;; inactivate
      (unwind-protect
	  (progn
	    (setq describe-current-input-method-function nil)
	    (iiimcf-server-control-notify-trigger nil)
	    (iiimcf-server-control-clear-preedit-text)
	    ;;(run-hooks 'iiimcf-server-control-inactivate-hook)
	    )
	(kill-local-variable 'input-method-function))
    ;; activate
    (iiimcf-server-control-setup)
    (iiimcf-server-control-notify-trigger t)
    (setq inactivate-current-input-method-function
	  'iiimcf-server-control-inactivate)
    (setq describe-current-input-method-function
	  'iiimcf-server-control-help)
    ;; inactivate the current input method also in minibuffers
    ;; before exiting.
    (if (eq (selected-window) (minibuffer-window))
	(add-hook 'minibuffer-exit-hook
		  (function
		   iiimcf-server-control-exit-from-minibuffer)))
    ;;(run-hooks 'iiimcf-server-control-activate-hook)
    (make-local-variable 'input-method-function)
    (setq input-method-function
	  (function iiimcf-server-control-keyforward))))

(defun iiimcf-server-control-exit-from-minibuffer ()
  (inactivate-input-method)
  (if (<= (minibuffer-depth) 1)
      (remove-hook 'minibuffer-exit-hook
		   (function
		    iiimcf-server-control-exit-from-minibuffer))))

(defun iiimcf-server-control-dispatch-event ()
  (let* ((input-method-function nil)
	 (keyseq (read-key-sequence
		  nil nil nil t))
	 (cmd (key-binding keyseq t)))
    (if (and (commandp cmd)
	     (not (eq cmd 'iiimcf-server-control-keyforward)))
	(progn
	  (setq last-command-event (aref keyseq
					 (1- (length keyseq)))
		last-command this-command
		this-command cmd)
	  (call-interactively cmd)))))

(defun iiimcf-server-control-keyforward (&optional ev)
  (interactive)
  (setq iiimcf-server-control-editing-buffer
	(current-buffer))
  (let* ((event (or ev
		    last-command-event))
	 (keyevent (iiimcf-translate-key event)))
    ;;(if (eq event ? ) (setq event 'convert))
    (setq keyevent (iiimcf-translate-key event))
    (if keyevent
	(condition-case err
	    (progn
	      (iiimp-send-message
	       'iiimp-im-forward-event
	       iiimcf-server-control-com-id
	       iiimcf-server-control-im-id
	       iiimcf-server-control-ic-id
	       (list 'keyevent
		     (list keyevent)))
	      ;; I have no idea why Solaris 8 IIIM server
	      ;; does not respond to IM_FORWARD_EVENT sometimes.
	      (iiimcf-message-manager
	       iiimcf-server-control-com-id
	       (list
		(list 'iiimp-im-forward-event-reply
		      iiimcf-server-control-com-id
		      iiimcf-server-control-im-id
		      iiimcf-server-control-ic-id)
		(list 'iiimp-im-forward-event
		      iiimcf-server-control-com-id
		      iiimcf-server-control-im-id
		      iiimcf-server-control-ic-id))))
	  (iiimp-fatal
	   (iiimcf-server-control-shutdown)
	   (signal (car err) (cdr err))))
      (setq unread-command-events
	    (nconc
	     unread-command-events
	     (list event)))
      (iiimcf-server-control-dispatch-event))
    nil))

(defun iiimcf-server-control-trigger-notify (mes)
  (let ((flag (nth 4 mes)))
    (if flag
	nil
      (inactivate-input-method))))

(defun iiimcf-server-control-forward-event (mes)
  (let ((con (nth 4 mes))
	(oldbuf (current-buffer))
	kevslot1 kevslot2 kev)
    (iiimp-add-debug-log
     (format "FWD:%S\n" con))
    (if (eq (car con) 'keyevent)
	(progn
	  (setq kevslot1 (nth 1 con))
	  (while kevslot1
	    (if (setq kev
		      (iiimcf-translate-iiim-keyevent
		       (car kevslot1)))
		(setq kevslot2
		      (cons kev kevslot2)))
	    (setq kevslot1 (cdr kevslot1)))
	  (if kevslot2
	      (progn
		(setq unread-command-events
		      (nconc (nreverse kevslot2)
			     unread-command-events))
		(if iiimcf-server-control-editing-buffer
		    (set-buffer iiimcf-server-control-editing-buffer))
		(iiimcf-server-control-dispatch-event)
		(set-buffer oldbuf)))))))

(defun iiimcf-server-control-put-face-to-preedit (str)
  (let ((len (length str))
	(pts 0)
	pte cprop)
    (while pts
      (setq cprop (get-text-property pts 'iiim-feedback str)
	    pte (next-single-property-change
		 pts 'iiim-feedback str))
      (if cprop
	  (put-text-property
	   pts (or pte len)
	   'face (iiimcf-server-control-convert-iiim-feedback
		  cprop 'face)
	   str))
      (setq pts pte))
    str))

(defun iiimcf-server-control-enclose-preedit-with-string (str caret)
  (let ((result "")
	(pts 0)
	(caretdiff 0)
	pte cprop cprop2 pprop
	elem slot hstr tstr)
    (while pts
      (setq cprop (get-text-property pts 'iiim-feedback str)
	    cprop2 cprop
	    pte (next-single-property-change
		 pts 'iiim-feedback str))
      (setq hstr "")
      (while (setq elem (car cprop))
	(if (memq elem pprop)
	    (setq pprop (delq elem pprop))
	  (if (setq slot (assq elem
			       iiimcf-server-control-preedit-face-alist))
	      (setq hstr (concat hstr (car (nth 2 slot))))))
	(setq cprop (cdr cprop)))
      (setq tstr "")
      (while pprop
	(if (setq slot (assq (car pprop)
			     iiimcf-server-control-preedit-face-alist))
	    (setq tstr (concat tstr (cdr (nth 2 slot)))))
	(setq pprop (cdr pprop)))
      (setq result (concat result tstr hstr (substring str pts pte)))
      (if (> caret pts) (setq caretdiff
			      (+ caretdiff
				 (length tstr)
				 (length hstr))))
      (setq pts pte
	    pprop cprop2))
    (while pprop
      (if (setq slot (assq (car pprop)
			   iiimcf-server-control-preedit-face-alist))
	  (setq result (concat result (cdr (nth 2 slot)))))
      (setq pprop (cdr pprop)))
    (cons result (+ caret caretdiff))))

(defun iiimcf-server-control-format-preedit (str caret)
  (if iiimcf-server-control-preedit-use-face-p
      (cons (iiimcf-server-control-put-face-to-preedit str)
	    caret)
    (iiimcf-server-control-enclose-preedit-with-string
     str caret)))

(defun iiimcf-server-control-setup-preedit-text ()
  (let ((prop-begin
	 '(iiimcf-server-control-preedit-start t intangible t))
	(prop-end
	 '(iiimcf-server-control-preedit-end t intangible t
	   rear-nonsticky t))
	(oldbuf (current-buffer))
	pss pse)
    ;; (if (string= " *IIIMP" (buffer-name oldbuf))
    ;;    (debug))
    (set-buffer iiimcf-server-control-editing-buffer)
    (setq pss (point))
    (let ((buffer-undo-list t))
      (insert iiimcf-server-control-preedit-open-string)
      (setq pse (point))
      (add-text-properties pss pse prop-begin) 
      (insert iiimcf-server-control-preedit-close-string)
      (add-text-properties pse (point) prop-end)
      (put-text-property pss (point) 'read-only t)
      (set-marker iiimcf-server-control-preedit-marker
		  pss)
      (set-marker-insertion-type
       iiimcf-server-control-preedit-marker t))
    (set-buffer oldbuf)))

(defun iiimcf-server-control-insert-preedit (text caret)
  (if (null (marker-position
	     iiimcf-server-control-preedit-marker))
      (iiimcf-server-control-setup-preedit-text))
  (let ((inhibit-read-only t)
	(inhibit-point-motion-hooks t)
	(oldbuf (current-buffer))
	(pts (marker-position
	      iiimcf-server-control-preedit-marker))
	pte)
    (set-buffer (marker-buffer
		 iiimcf-server-control-preedit-marker))
    (setq pte
	  (next-single-property-change
	   pts 'iiimcf-server-control-preedit-end))
    (if (null pte)
	(error "PREEDIT TEXT IS BROKEN!!"))
    (setq pts (1+ pts))
    (let ((buffer-undo-list t))
      (delete-region pts pte)
      (goto-char pts)
      (insert text)
      (put-text-property pts (point) 'read-only t)
      (goto-char (+ pts caret)))
    (set-buffer oldbuf)))

(defun iiimcf-server-control-maintain-preedit-text-internal
  (mes &optional init)
  (let* ((caret (nth 4 mes))
	 (ch-first (nth 5 mes))
	 (ch-len (nth 6 mes))
	 (ch-second (+ ch-first ch-len))
	 (contents (nth 7 mes))
	 (str (nth 1 contents))
	 (len (length iiimcf-server-control-preedit))
	 disptext head tail mbuf pt)
    (if (and (>= len ch-first)
	     (> ch-first 0))
	(setq head (substring
		    iiimcf-server-control-preedit
		    0 ch-first)))
    (if (> len ch-second)
	(setq tail (substring
		    iiimcf-server-control-preedit
		    ch-second)))
    (setq iiimcf-server-control-preedit (concat head str tail)
	  disptext (iiimcf-server-control-format-preedit
		    iiimcf-server-control-preedit caret)
	  caret (cdr disptext)
	  disptext (car disptext))
    (if (= 0 (length iiimcf-server-control-preedit))
	(iiimcf-server-control-clear-preedit-text)
      (if (and input-method-use-echo-area
	       (null
		(marker-position
		 iiimcf-server-control-preedit-marker)))
	  ;; show preedit string in minibuffer.
	  (save-excursion
	    ;; clear echo area.
	    (message nil)
	    (setq mbuf (window-buffer (minibuffer-window)))
	    (set-buffer mbuf)
	    (erase-buffer)
	    (insert "Preedit:")
	    (setq pt (point))
	    (insert disptext " ")
	    (move-overlay
	     iiimcf-server-control-caret-overlay
	     (+ pt caret) (+ 1 pt caret) mbuf))
	(if init
	    (iiimcf-server-control-setup-preedit-text))
	(iiimcf-server-control-insert-preedit disptext caret))
      )))

(defun iiimcf-server-control-maintain-preedit-text (mes)
  (if iiimcf-server-control-maintaining-preedit-text
      (iiimcf-server-control-maintain-preedit-text-internal mes)
    (iiimcf-server-control-maintain-preedit-text-internal mes t)
    (let ((iiimcf-server-control-maintaining-preedit-text t)
	  ;; disable the font-lock function invoked by
	  ;; after-change-functions during preedit.
	  (font-lock-fontify-region-function (function ignore))
	  ;; Even read-only text-property cannot protect inserted preedit
	  ;; text from modifications of the face text-property by font-lock.
	  keyseq cmd)
      (while (> (length iiimcf-server-control-preedit) 0)
	(let ((overriding-terminal-local-map
	       iiimcf-server-control-keymap)
	      (input-method-function nil))
	  (setq keyseq (read-key-sequence
			nil nil nil t)
		cmd (key-binding keyseq t)))
	;; (message "KS:%S, %S" keyseq cmd)
	(if (commandp cmd)
	    (progn
	      (setq last-command-event (aref keyseq
					     (1- (length keyseq)))
		    last-command this-command
		    this-command cmd)
	      (condition-case err
		  (if (or (eq cmd
			      (function
			       iiimcf-server-control-keyforward))
			  (null (marker-position
				 iiimcf-server-control-preedit-marker)))
		      (call-interactively cmd)
		    (save-excursion
		      (set-buffer
		       (marker-buffer
			iiimcf-server-control-preedit-marker))
		      (goto-char
		       (marker-position
			iiimcf-server-control-preedit-marker))
		      (call-interactively cmd)))
		(error
		 (debug)
		 (iiimcf-prevent-error
		  (message "%s" (cdr err)) (beep))))))))))

(defun iiimcf-server-control-clear-preedit-text ()
  (setq iiimcf-server-control-preedit "")
  (if (marker-position
       iiimcf-server-control-preedit-marker)
      (let ((pts (marker-position
		  iiimcf-server-control-preedit-marker))
	    (inhibit-read-only t)
	    pte)
	(save-excursion
	  (set-buffer (marker-buffer
		       iiimcf-server-control-preedit-marker))
	  (setq pte
		(next-single-property-change
		 pts 'iiimcf-server-control-preedit-end))
	  (if (null pte)
	      (error "PREEDIT TEXT BROKEN!!"))
	  (let ((buffer-undo-list t))
	    (delete-region pts (1+ pte))))
	(goto-char pts)
	(set-marker iiimcf-server-control-preedit-marker nil))
    (save-excursion
      (let ((mbuf (window-buffer (minibuffer-window))))
	(set-buffer mbuf)
	(erase-buffer)
	(move-overlay
	 iiimcf-server-control-caret-overlay
	 0 0 mbuf)))))

(defun iiimcf-server-control-prepare-lookup-choice (mes)
  (setq iiimcf-server-control-lookup-choice-configuration mes))

(defun iiimcf-server-control-draw-lookup-choice (mes)
  (if (null iiimcf-server-control-lookup-choice-configuration)
      (error "IM_LOOKUP_CHOICE_START is missing."))
  (with-output-to-temp-buffer "*Lookup choice*"
    (save-excursion
     (set-buffer standard-output)
     (let ((i 0)
	   (k 0)
	   (num (nth 5 iiimcf-server-control-lookup-choice-configuration))
	   (rows (nth 6 iiimcf-server-control-lookup-choice-configuration))
	   (cols (nth 7 iiimcf-server-control-lookup-choice-configuration))
	   (idxfirst (nth 4 mes))
	   (idxlast (nth 5 mes))
	   (idxcur (nth 6 mes))
	   (candlist (nth 7 mes))
	   (lblist (nth 8 mes))
	   (title (nth 9 mes))
	   curcand curlb)
       (setq iiimcf-server-control-current-lookup-choice
	     (list idxfirst candlist lblist title idxfirst idxlast))
       (insert (format "Title:%s\n" title))
       (while (and (> num i)
		   (setq curcand (car candlist))
		   (setq curlb (car lblist)))
	 (insert (format "%s : %s    " curlb curcand))
	 (setq i (1+ i)
	       k (1+ k))
	 (if (>= k rows)
	     (progn
	       (setq k 0)
	       (insert "\n")))
	 (setq candlist (cdr candlist)
	       lblist (cdr lblist)))))))

(defun iiimcf-server-control-redraw-lookup-choice (mes)
  (if (or (null iiimcf-server-control-lookup-choice-configuration)
	  (null iiimcf-server-control-current-lookup-choice))
      (error "IM_LOOKUP_CHOICE_START is missing."))
  (with-output-to-temp-buffer "*Lookup choice*"
    (save-excursion
     (set-buffer standard-output)
     (let ((i 0)
	   (k 0)
	   (num (nth 5 iiimcf-server-control-lookup-choice-configuration))
	   (rows (nth 6 iiimcf-server-control-lookup-choice-configuration))
	   (cols (nth 7 iiimcf-server-control-lookup-choice-configuration))
	   (idx (car iiimcf-server-control-current-lookup-choice))
	   (lblist (nth 1 iiimcf-server-control-current-lookup-choice))
	   (candlist (nth 2 iiimcf-server-control-current-lookup-choice))
	   (title (nth 3 iiimcf-server-control-current-lookup-choice))
	   (idxfirst (nth 4 iiimcf-server-control-current-lookup-choice))
	   (idxlast (nth 5 iiimcf-server-control-current-lookup-choice))
	   (type (nth 4 mes))
	   (val (nth 5 mes))
	   curcand curlb)
       (cond ((eq type 'index)
	      (setq idx val))
	     ((eq type 'page)
	      (cond ((eq val 'next)
		     (setq idx (+ idx num)))
		    ((eq val 'prev)
		     (setq idx (- idx num)))
		    ((eq val 'first)
		     (setq idx idxfirst))
		    ((eq val 'last)
		     (setq idx (- idxlast num))))))
       (setcar iiimcf-server-control-current-lookup-choice idx)
       (setq lblist (nthcdr (- idx idxfirst) lblist))
       (setq candlist (nthcdr (- idx idxfirst) candlist))
       (while (and (> num i)
		   (setq curcand (car candlist))
		   (setq curlb (car lblist)))
	 (insert (format "%s : %s    " curlb curcand))
	 (setq i (1+ i)
	       k (1+ k))
	 (if (>= k rows)
	     (progn
	       (setq k 0)
	       (insert "\n")))
	 (setq candlist (cdr candlist)
	       lblist (cdr lblist)))))))

(defun iiimcf-server-control-clear-lookup-choice (mes)
;;  (setq iiimcf-server-control-lookup-choice-configuration nil
;;	  iiimcf-server-control-current-lookup-choice nil)
  (let* ((buf (get-buffer "*Lookup choice*"))
	 (win (get-buffer-window buf)))
    (if win (delete-window win))))

(defun iiimcf-server-control-draw-status (mes)
  (setq iiimcf-server-control-mode-line
	(format "[%s]" (nth 1 (nth 4 mes))))
  (force-mode-line-update))

(defun iiimcf-server-control-draw-aux-data (mes)
  (let ((name (nth 5 mes))
	(strs (nth 7 mes)))
    (cond ((string-match "^jp\\.co\\.justsystem\\..*LookupAux$"
			 name)
	   (if strs
	       (message "%s:  %s"
			(car strs)
			(mapconcat 
			 (function identity)
			 (cdr strs) " "))
	     (message ""))))))
  

(defun iiimcf-server-control-commit-string (mes)
  (let ((str (copy-sequence (nth 1 (nth 4 mes))))
	(oldbuf (current-buffer))
	(buf (marker-buffer
	      iiimcf-server-control-preedit-marker))
	(pt (marker-position
	     iiimcf-server-control-preedit-marker))
	generated-events)
    (set-text-properties 0 (length str) nil str)
    (if pt
	(progn
	  (set-buffer buf)
	  (goto-char pt))
      (set-buffer iiimcf-server-control-editing-buffer))
    (setq generated-events
	  (append (string-to-list
		   (if enable-multibyte-characters
		       str
		     (string-make-unibyte str)))
		  generated-events))
    (setq unread-command-events
	  (nconc
	   (if (and input-method-exit-on-first-char generated-events)
	       (list (car generated-events))
	     generated-events)
	   unread-command-events))
    ;; (insert str)
    (if pt (set-marker
	    iiimcf-server-control-preedit-marker
	    (point)))
    (set-buffer oldbuf)
    (run-hooks 'input-method-after-insert-chunk-hook)))

(register-input-method
 "iiim-server-control" "Japanese"
 'iiimcf-server-control-activate
 ""  "IIIM server control input method")

(provide 'iiimcf-sc)

;; iiimcf-sc.el ends here.
