;;; mew-draft.el --- Draft mode for Mew

;; Author:  Kazu Yamamoto <Kazu@Mew.org>
;; Created: Oct  2, 1996
;; Revised: Jul  8, 2001

;;; Code:

(require 'mew)

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;
;;; Draft info
;;;

(defvar mew-tinfo-list
  '("header-keymap" "attach-keymap" "case" "encrypted-p" "privacy-err"
    "encode-err" "privacy-type" "hdr-file" "field-del"))

(mew-blinfo-defun 'mew-tinfo mew-tinfo-list)


(defvar mew-draft-mode-syntax-table nil
  "*Syntax table used while in Draft mode.")

(unless mew-draft-mode-syntax-table
  (setq mew-draft-mode-syntax-table (make-syntax-table text-mode-syntax-table))
  (modify-syntax-entry ?% "." mew-draft-mode-syntax-table))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;
;;; Draft mode
;;;

(defun mew-draft-set-local-variables ()
  (auto-save-mode mew-draft-mode-auto-save)
  (make-local-variable 'paragraph-start)
  (setq paragraph-start (concat mew-eoh "\\|[ \t]*$\\|" page-delimiter))
  (make-local-variable 'paragraph-separate)
  (setq paragraph-separate paragraph-start)
  (make-local-variable 'mail-header-separator)
  (setq mail-header-separator mew-header-separator)
  (make-local-variable 'comment-start)
  (setq comment-start ">")
  (make-local-hook 'after-change-functions)
  (add-hook 'after-change-functions 'mew-draft-dynamic-highlight nil 'local))

(defun mew-draft-mode (&optional encrypted)
  "\\<mew-draft-mode-map>
Mew Draft mode:: major mode for composing a MIME message.
Key actions are different in each region: Header, Body, and Attachment.

To send a draft, type \\[mew-draft-make-message] and \\[mew-draft-send-message].  To make multipart, type
\\[mew-draft-prepare-attachments], edit attachments, type \\[mew-draft-make-message] and \\[mew-draft-send-message].

*Whole buffer key assignment:

\\[mew-draft-make-message]	Compose a MIME message then put it into a queue folder.
\\[mew-draft-send-message]	Compose a MIME message then send it.

\\[mew-draft-prepare-attachments]	Prepare an attachment region in the bottom of the draft.
	To compose a multipart message, you should execute this 
	command first.

\\[mew-draft-rehighlight]  Highlight header and body again.

\\[mew-draft-kill]	Kill this draft.

\\[mew-pgp-sign-message]	Sign the entire draft with PGP. Input your passphrase.
\\[mew-pgp-encrypt-message]	Encrypt the entire draft with PGP.
\\[mew-pgp-sign-encrypt-message]	Sign then encrypt the entire draft with PGP.
	Input your passphrase.
\\[mew-pgp-encrypt-sign-message]	Encrypt then sign the entire draft with PGP.
	Input your passphrase.

\\[mew-draft-toggle-privacy-always]	Toggle whether or not all drafts are protected.
\\[mew-draft-toggle-privacy-encrypted]	Toggle whether or not drafts replying to encrypted messages 
		are protected.
\\[mew-draft-set-privacy-type]	Set privacy service which will be effective when \\[mew-draft-make-message].
\\<mew-draft-header-map>
*Header region key assignment:

\\[mew-draft-header-comp]	Complete field keys.
	Complete and expand an address short name.
	Complete folder names.
\\[mew-draft-circular-comp]	Complete your mail domain.
\\[mew-draft-expand]	Replace an address with 'NAME <address>'.

*Body region key assignment:

\\<mew-draft-body-map>\\[mew-draft-insert-signature]	Insert '~/.signature' on the cursor point.
\\<mew-draft-mode-map>\\[mew-draft-cite]	Copy and paste a part of message from Message mode with
	citation prefix and label.
	1. Roughly speaking, it copies the body in Message mode. 
	   For example, if text/plain is displayed, the entire Message 
	   mode is copied. If message/rfc822 is displayed, the body 
	   without the header is copied.
	2. If called with '\\[universal-argument]', the header is also copied if exists.
	3. If an Emacs mark exists, the target is the region between 
	   the mark and the cursor.
\\[mew-draft-yank]	Copy and paste a part of message from Message mode WITHOUT
	citation prefix and label.

*Attachments region Key assignment:
\\<mew-draft-attach-map>
\\[mew-attach-forward]	Go to the first subdirectory.
\\[mew-attach-backforward]	Go to the parent directory.
\\[mew-attach-next]	Go to the next file in the current directory.
\\[mew-attach-previous]	Go to the previous file in the current directory.

\\[mew-attach-copy]	Copy a file (via networks) on '.'.
	To copy a remote file, use the '/[user@]hostname:/filepath' syntax.
\\[mew-attach-link]	Link a file with a symbolic link on '.'.
\\[mew-attach-delete]	Delete this file or this directory.
\\[mew-attach-multipart]	Create a subdirectory(i.e. multipart) on '.'.
\\[mew-attach-find-file]	Open this file into a buffer.
\\[mew-attach-find-new-file]	Open a new file into a buffer on '.'.
\\[mew-attach-external-body]	Input external-body on '.'.
\\[mew-attach-audio]	Sampling voice and insert as audio file on '.'.
\\[mew-attach-pgp-public-key]	Extract the PGP key for the inputed user on '.'.
\\[mew-attach-description]	Input a description(Content-Description:).
\\[mew-attach-disposition]	Change the file name(Content-Disposition:).
\\[mew-attach-type]	Change the data type(Content-Type:).
\\[mew-attach-charset]	Specify the charset parameter for a Text/* object.
\\[mew-attach-icharset]	Specify a input coding-system for a text file.

\\[mew-attach-base64]	Put the 'B' mark to encode with Base64.
\\[mew-attach-quoted-printable]	Put the 'Q' mark to encode with Quoted-Printable.
\\[mew-attach-gzip64]	Put the 'G' mark to encode with Gzip64. This is applicable 
	only to Text/Plain and Application/Postscript since compression 
	is not effective other objects. For example, JPEG is already 
	compressed.
\\[mew-attach-pgp-sign]	Put the 'PS' mark to sign with PGP.
\\[mew-attach-pgp-enc]	Put the 'PE' mark to encrypt with PGP. 
	Input decryptors' addresses.
\\[mew-attach-undo]	Unmark. The original mark appears.

* Fill blanks
\\<mew-draft-mode-map>
Prepare '~/.mew-fib' like;

	name:  Kazuhiko Yamamoto
	email: Kazu@Mew.org

If you receive a message like;

	Your name : |>name<|
	Your e-mail address: |>email<|

Type \\<mew-summary-mode-map>\\[mew-summary-reply] in Summary mode to enter Draft mode. 
Then type \\<mew-draft-mode-map>\\[mew-draft-yank], \\[mew-fib-fill-default], and \\[mew-fib-delete-frame] makes following
draft.

	Your name : Kazuhiko Yamamoto
	Your e-mail address: Kazu@Mew.org

In this way, mew-fil fills up items quoted like |> <| from '~/.mew-fib'.
The fill functions described below.

\\[mew-fib-fill-default]	Fill |>item<| from '~/.mew-fib'.
\\[mew-fib-delete-frame]	Delete all quotations, i.e. |> <|.
\\[mew-fib-next-item]	Jump to the next fib item.
\\[mew-fib-previous-item]	Jump to the previous fib item.
\\[mew-fib-flush-input]	Flush input from '~/.mew-fib'.

Moreover, '~/.mew-fib' supports aliases like;

	email: Kazu@Mew.org
	e-mail:

"
  (interactive)
  (mew-draft-set-local-variables)
  (add-hook 'local-write-file-hooks (function mew-encode-make-backup))
  (setq major-mode 'mew-draft-mode)
  (use-local-map mew-draft-mode-map)
  (set-syntax-table mew-draft-mode-syntax-table)
  (mew-ainfo-set-icon (file-name-nondirectory (buffer-file-name)))
  (cd (expand-file-name mew-home))
  (when mew-require-final-newline
    (make-local-variable 'require-final-newline)
    (setq require-final-newline t))
  (mew-draft-setup-decoration)
  (mew-tinfo-set-encrypted-p encrypted)
  (mew-tinfo-set-privacy-err nil)
  (mew-tinfo-set-privacy-type nil)
  (setq mode-line-buffer-identification mew-mode-line-id)
  (mew-draft-mode-name)
  (run-hooks 'text-mode-hook 'mew-draft-mode-hook)
  ;; must be here for auto-fill
  (when auto-fill-function
    (make-local-variable 'auto-fill-function)
    (setq auto-fill-function (function mew-draft-auto-fill)))
  (setq buffer-undo-list nil))

(defun mew-draft-mode-name (&optional header)
  (let (pcdb sub)
    (cond
     ((mew-tinfo-get-privacy-type)
      (setq pcdb (mew-pcdb-by-service (mew-tinfo-get-privacy-type)))
      (setq sub (mew-pcdb-mark pcdb)))
     ((and (mew-tinfo-get-encrypted-p) mew-protect-privacy-encrypted)
      (setq pcdb (mew-pcdb-by-service mew-protect-privacy-encrypted-type))
      (setq sub (mew-pcdb-mark pcdb)))
     (mew-protect-privacy-always
      (setq pcdb (mew-pcdb-by-service mew-protect-privacy-always-type))
      (setq sub (mew-pcdb-mark pcdb))))
    (setq mode-name (if header mew-mode-name-header mew-mode-name-draft))
    (if sub (setq mode-name (concat mode-name " " sub)))
    (unless (mew-case-default-p (mew-tinfo-get-case))
      (setq mode-name (concat mode-name " " (mew-tinfo-get-case))))
    (force-mode-line-update)))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;
;;; Draft subfunctions
;;;

(defun mew-draft-dynamic-highlight (beg end len)
  (when (mew-in-header-p)
    (save-match-data
      (mew-highlight-header)
      (when (mew-draft-p)
	(mew-draft-header-keymap)))))

(defun mew-draft-auto-fill ()
  (let ((ret1 (do-auto-fill)) ret2)
    (when (mew-in-header-p)
      (save-excursion
	(beginning-of-line)
	(while (not (or (looking-at "[^ \t\n]+:\\|[ \t]") (bobp)))
	  (setq ret2 t)
	  (insert "\t")
	  (forward-line -1)
	  (beginning-of-line))))
    (or ret1 ret2))) ;; if modifies, return t.

(defun mew-draft-buffer-name (path)
  (let ((regex (format "^%s\\(.*\\)$"
		       (file-name-as-directory
			(expand-file-name mew-mail-path)))))
    (if (string-match regex path)
	(concat "+" (mew-match 1 path)))))

(defun mew-draft-find-and-switch (draft-path &optional switch-func)
  ;; switch-func = nil :: switch-to-buffer
  ;; switch-func = t   :: switch-to-buffer-other-window
  (let* ((special-display-buffer-names nil)
	 (special-display-regexps nil)
	 (same-window-buffer-names nil)
	 (same-window-regexps nil)
	 (draftname (mew-draft-buffer-name draft-path)))
    (when (get-buffer draftname)
      (save-excursion
	(set-buffer draftname)
	(clear-visited-file-modtime)
	(set-buffer-modified-p nil) ;; just in case
	(if (file-exists-p buffer-auto-save-file-name)
	    (delete-file buffer-auto-save-file-name))
	(mew-remove-buffer draftname)))
    (cond
     ((eq switch-func nil)
      (setq switch-func (function switch-to-buffer)))
     ((eq switch-func t)
      (setq switch-func (function switch-to-buffer-other-window))))
    (funcall switch-func (find-file-noselect draft-path))
    ;; draft buffer
    (mew-set-buffer-cs mew-cs-m17n)
    ;; copy config, first
    (mew-tinfo-set-case mew-case-output)
    (rename-buffer draftname)))

(defun mew-draft-to-attach (draft)
  "Converting draft to attach. E.g. +draft/1 -> +attach/1"
  (concat (file-name-as-directory mew-attach-folder)
	  (file-name-nondirectory draft)))

(defun mew-attachdir (&optional draft)
  (mew-expand-folder (mew-draft-to-attach (or draft (buffer-name)))))

(defun mew-draft-header-insert-alist (halist)
  "Insert field-body: and field-value. Return the value of
the Body: field."
  (let ((case-fold-search t)
	key val ret)
    (while halist
      (setq key (car (car halist)))
      (setq val (cdr (car halist)))
      (setq halist (cdr halist))
      (unless (string-match ":$" key)
	(setq key (concat key ":")))
      (if (string-match mew-body: key)
	  (setq ret val)
	(mew-draft-header-insert key val)))
    ret))

(defun mew-insert-address-list (field adrs del force-insert)
  (let ((cnt 0) (beg (point)) med adr)
    (while adrs
      (setq adr (car adrs) adrs (cdr adrs))
      (if (mew-is-my-address del adr)
	  ()
	(if (= cnt 0)
	    (insert adr)
	  (insert ", " adr))
	(setq del (cons (concat "^" (regexp-quote adr) "$") del))
	(setq cnt (1+ cnt))))
    (when (or force-insert (> cnt 0))
      (beginning-of-line)
      (insert field " ")
      (setq med (point))
      (end-of-line)
      (insert "\n")
      (mew-header-fold-region beg (point) med 'use-tab))
    del))

(defun mew-insert-address-list2 (field adrs)
  (when adrs
    (let ((beg (point)) med)
      (insert field " ")
      (setq med (point))
      (insert (car adrs))
      (setq adrs (cdr adrs))
      (while adrs
	(insert ", " (car adrs))
	(setq adrs (cdr adrs)))
      (insert "\n")
      (mew-header-fold-region beg (point) med 'use-tab))))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;
;;; Draft header
;;;

(defun mew-draft-header (&optional subject nl to cc newsgroups in-reply-to references other-headers fromme)
;; to -- string or list
;; cc -- string or list
;; nl -- one empty line under "----", which is necessary if
;;      attachment is prepared
  (let ((del mew-regex-my-address-list) ;; deleting list for Cc:
	case body)
    (goto-char (point-min))
    ;; Insert To: first.
    ;; All addresses inserted on To: are appended to del.
    (cond
     ((null to) (insert mew-to: " \n"))
     ((stringp to) ;; To: inputed from the mini-buffer.
      ;; Don't check to is mine. Cc: is also string
      ;; We believe that user never specifies the same address of To: to Cc:.
      (insert mew-to: " " to "\n"))
     ;; To: collected by reply
     ((listp to)
      (if fromme
	  (mew-insert-address-list2 mew-to: to)
	(setq del (mew-insert-address-list mew-to: to del t)))))
    (cond
     ((null cc) ()) ;; do nothing 
     ((stringp cc) ;; Cc: inputed from the mini-buffer.
      (insert mew-cc: " " cc "\n"))
     ((listp cc) ;; Cc: collected by reply.
      (if fromme
	  (mew-insert-address-list2 mew-cc: cc)
	(mew-insert-address-list mew-cc: cc del nil))))
    (mew-draft-header-insert mew-newsgroups:  newsgroups)
    (if mew-case-guess-when-prepared
	(mew-draft-set-case-by-guess))
    (setq case (mew-tinfo-get-case))
    (mew-draft-header-insert mew-cc:          (mew-cc case))
    (mew-draft-header-insert mew-subj:        (or subject ""))
    (mew-draft-header-insert mew-from:        (mew-from case))
    (mew-draft-header-insert mew-fcc:         (mew-fcc case))
    (mew-draft-header-insert mew-dcc:         (mew-dcc case))
    (mew-draft-header-insert mew-reply-to:    (mew-reply-to case))
    (mew-draft-header-insert mew-in-reply-to: in-reply-to)
    (mew-draft-header-insert mew-references:  references)
    (if (and mew-x-face-file
	     (file-exists-p (expand-file-name mew-x-face-file)))
	(let ((buf (generate-new-buffer mew-buffer-prefix))
	      xface)
	  (save-excursion
	    (set-buffer buf)
	    (mew-erase-buffer)
	    (insert-file-contents (expand-file-name mew-x-face-file))
	    (setq xface (mew-buffer-substring (point-min)
					      (max (buffer-size) 1))))
	  (mew-remove-buffer buf)
	  (mew-draft-header-insert mew-x-face: xface)))
    (mew-draft-header-insert mew-x-mailer: mew-x-mailer)
    (mew-draft-header-insert mew-organization: (mew-organization case))
    (setq body (mew-draft-header-insert-alist other-headers))
    ;; Deleting fields defined in mew-header-alist to replace them.
    (mew-header-delete-lines (mapcar (function car) (mew-header-alist case)))
    (mew-header-goto-end)
    (mew-draft-header-insert-alist (mew-header-alist case))
    (mew-header-prepared)
    ;; on the body
    (if nl (insert "\n"))
    (if body (save-excursion (insert body)))
    ;; move the cursor after "To: "
    (goto-char (point-min))
    (forward-char 4))) ;; Don't use (end-of-line) since the value may exist.

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;
;;; Citation
;;;

(defun mew-draft-auto-set-input-method ()
  (if (and (fboundp 'activate-input-method)
	   mew-charset-input-method-alist)
      (let* ((charset (mew-charset-guess-region
		       (mew-header-end) (or (mew-attach-begin) (point-max))))
	     (method (mew-charset-to-input-method charset)))
	(when (stringp method)
	  (activate-input-method method)
	  (message "Set input method to %s" method)))))

(defun mew-draft-yank (&optional arg force)
  "Copy and paste a part of message from Message mode WITHOUT
citation prefix and label.
1. Roughly speaking, it copies the body in Message mode. For example,
   if text/plain is displayed, the entire Message mode is copied.
   If message/rfc822 is displayed, the body without the header is copied.
2. If called with '\\[universal-argument]', the header is also copied if exists.
3. If an Emacs mark exists, the target is the region between the mark and 
   the cursor."
;; MUST take care of C-x C-x
;; MUST be able to cancel by C-x u
  (interactive "P")
  (if (and (not force) (or (mew-in-header-p) (mew-in-attach-p)))
      (message "You cannot cite a message here.")
    (let (cite beg end)
      (save-excursion
	(set-buffer (mew-buffer-message))
	(save-restriction
	  (widen)
	  (let ((mark-active t) (transient-mark-mode nil))
	    (cond
	     (arg 
	      (setq beg (point-min) end (point-max)))
	     ((mew-mark) 
	      (setq beg (region-beginning) end (region-end)))
	     ((mew-header-p)
	      ;; header exists in Message mode
	      (mew-header-goto-body)
	      (setq beg (point) end (point-max)))
	     (t
	      (setq beg (point-min) end (point-max)))))
	  (setq cite (mew-buffer-substring beg end))))
      (push-mark (point) t t) ;; for C-x C-x
      (insert cite)
      (mew-draft-auto-set-input-method))))

(defvar mew-message-citation-buffer nil
  "This value is used by mew-gnus.el to specify a buffer from where
you can cite.")

(defun mew-draft-cite (&optional arg force)
  "Copy and paste a part of message from Message mode with
citation prefix and label.
1. Roughly speaking, it copies the body in Message mode. For example,
   if text/plain is displayed, the entire Message mode is copied.
   If message/rfc822 is displayed, the body without the header is copied.
2. If called with '\\[universal-argument]', the header is also copied if exists.
3. If an Emacs mark exists, the target is the region between the mark and 
   the cursor."
;; MUST take care of C-x C-x
;; MUST be able to cancel by C-x u
  (interactive "P")
  (if (and (not force) (or (mew-in-header-p) (mew-in-attach-p)))
      (message "You cannot cite a message here.")
    (let ((nonmewbuf mew-message-citation-buffer) ;; buffer local, so copy here
	  (fld (mew-current-get-fld (mew-frame-id)))
	  (msg (mew-current-get-msg (mew-frame-id)))
	  cite beg end tbuf irt-msgid)
      (save-excursion
	;;
	;; extract the body without header
	;;
	(setq tbuf (or nonmewbuf (mew-buffer-message)))
	(if tbuf
	    (set-buffer tbuf)
	  (error "No buffer to be cited."))
	(save-restriction
	  ;; first prepare "cite"
	  (widen)
	  (let ((mark-active t) (transient-mark-mode nil))
	    (cond
	     ;; arg will be effect in mew-cite-original
	     ((and (mew-mark) (or (not mew-cite-ignore-mouse-region)
				  (not (mew-mouse-region-p))))
	      (setq beg (region-beginning) end (region-end)))
	     ((mew-header-p)
	      ;; header exists in Message mode. Skip the header
	      ;; because we will concatenate it to cite later.
	      (mew-header-goto-body)
	      (setq beg (point) end (point-max)))
	     (t
	      (setq beg (point-min) end (point-max)))))
	  (setq cite (mew-buffer-substring beg end)))
	;; concat the header
	;; see also mew-summary-reply
	(setq tbuf (or nonmewbuf
		       (save-excursion
			 (set-buffer (mew-buffer-message))
			 (if (mew-header-p) (current-buffer)))
		       ;; header exists only in cache if multipart
		       (mew-cache-hit fld msg)))
	(if tbuf
	    (set-buffer tbuf)
	  (error "No buffer to be cited."))
	(save-restriction
	  (widen)
	  (mew-header-goto-end)
	  (setq cite (concat (mew-buffer-substring (point-min) (point)) 
			     "\n" cite))
          (setq irt-msgid (mew-header-get-value mew-message-id:))))
      ;; 
      ;; Draft mode, insert the header and the body.
      ;;

      ;; Append message-id to In-Reply-To:
      (if (and irt-msgid (mew-header-p))
          (save-excursion
            (let ((irt (mew-header-get-value mew-in-reply-to:))
                  (irtl nil) rb)
              (when irt
                (setq rb irt)
                (while (string-match "<[^>]+>" rb)
                  (setq irtl (cons (mew-match 0 rb) irtl)) 
                  (setq rb (substring rb (match-end 0))))
                (if (member irt-msgid irtl)
                    (setq irt-msgid nil)))
              (when irt-msgid
                (setq irt (concat irt (if irt "\n\t") irt-msgid))
                (mew-header-delete-lines (list mew-in-reply-to:))
                (unless irtl (goto-char (mew-header-end)))
                (mew-draft-header-insert mew-in-reply-to: irt)))))
      (save-restriction
	;; this gets complicated due to supercite, please don't care
	(narrow-to-region (point)(point)) ;; for (goto-char (point-min))
	(insert cite)
	(push-mark (point) t t)
	(goto-char (point-min)))
      (cond
       (mew-cite-hook
	(run-hooks 'mew-cite-hook))
       (t (mew-cite-original arg)))
      ;; (mark-marker) indicates the point after label.
      ;; Should we include the label too?
      (or force (mew-highlight-body-region (mark-marker) (point) 'draft))
      (mew-draft-auto-set-input-method))))

(defun mew-cite-original (&optional arg)
  (if (< (marker-position (mark-marker)) (point))
      (exchange-point-and-mark))
  (let ((beg (point)) (end (marker-position (mark-marker)))
        label prefix)
    (save-restriction
      (narrow-to-region beg end)
      (condition-case nil
          (setq label (funcall mew-cite-strings-function))
        (error
	 (error "Syntax of mew-cite-format was changed. Read explanation of mew-cite-fields")))
      (if (null mew-cite-prefix-function)
          (setq prefix mew-cite-prefix)
        (setq prefix (funcall mew-cite-prefix-function)))
      (if mew-cite-prefix-confirmp
          (let ((ask (read-string 
                      (format "Prefix (\"%s\"): " prefix) "")))
            (if (not (string= ask "")) (setq prefix ask))))
      ;; C-u C-c C-y cites body with header.
      (if (eq arg nil) 
	  ;; header has been already cited. So, delete it.
	  (delete-region beg (progn (mew-header-goto-body) (point))))
      (insert label)
      (push-mark (point) t t) ;; for C-x C-x
      (and (bolp) (insert prefix))
      (while (= 0 (forward-line))
	(or (= (point) (point-max))
	    (insert prefix))))))

(defun mew-cite-get-value (field)
  (let ((value (mew-header-get-value field))
	repl func)
    (when (and (string= mew-from: field) value
	       (setq func (mew-addrbook-func mew-addrbook-for-cite-label)))
      (setq repl (funcall func (mew-addrstr-parse-address value)))
      (if repl (setq value repl)))
    (or value "")))

(defun mew-cite-strings ()
  "A function to create cite labels according to 
'mew-cite-format' and 'mew-cite-fields'."
  (if (null mew-cite-fields)
      ""
    (let* ((vals (mapcar (function mew-cite-get-value) mew-cite-fields))
	   (label (apply (function format) mew-cite-format vals))
	   (ellipses (if (stringp mew-draft-cite-ellipses)
			 mew-draft-cite-ellipses ""))
	   buf beg eol str)
      (if (not (or (eq mew-draft-cite-fill-mode 'truncate)
		   (eq mew-draft-cite-fill-mode 'wrap)))
	  label
	(setq buf (generate-new-buffer mew-buffer-prefix))
	(save-excursion
	  (set-buffer buf)
	  (mew-erase-buffer)
	  (let ((fill-column
		 (or mew-draft-cite-label-fill-column fill-column)))
	    (insert label)
	    (goto-char (point-min))
	    (while (not (eobp))
	      (cond
	       ((eq mew-draft-cite-fill-mode 'truncate)
		(end-of-line)            
		(if (>= fill-column (current-column))
		    ()
		  (setq eol (point))
		  (insert ellipses)
		  (goto-char eol)
		  (while (< fill-column (current-column))
		    (delete-backward-char 1))))
	       ((eq mew-draft-cite-fill-mode 'wrap)
		(setq beg (point))
		(end-of-line)
		(if (= (current-column) 0)
		    ()
		  (fill-region beg (point))
		  (if (= (current-column) 0) ;; for XEmacs
		      (delete-backward-char 1)))))
	      (forward-line)))
	  (setq str (buffer-string)))
	(mew-remove-buffer buf)
	str))))

(defun mew-cite-prefix-username ()
  "A good candidate for mew-cite-prefix-function.
The citation style is 'from_address> ', e.g. 'kazu> '"
  (let* ((from (mew-header-parse-address mew-from:))
	 (user (mew-addrstr-extract-user from))
	 (func (mew-addrbook-func mew-addrbook-for-cite-prefix))
	 nickname prefix)
    (if func (setq nickname (funcall func from)))
    (setq prefix (or nickname user))
    (if mew-ask-cite-prefix
	(setq prefix (read-string "Citation prefix: " prefix)))
    (concat prefix mew-cite-prefix)))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;
;;; Misc
;;;

(defun mew-draft-kill ()
  "Kill this draft."
  (interactive)
  (if (y-or-n-p "Kill draft message? ")
      (let* ((attachdir (mew-attachdir)) ;; attachdir must be here
	     (draft (buffer-file-name))
	     (mdi (concat draft mew-draft-info-suffix)))
	(mew-overlay-delete-buffer)
	(save-buffer)
	(mew-remove-buffer (current-buffer))
	(if (file-exists-p draft) (delete-file draft))
	(if (file-exists-p mdi) (delete-file mdi))
	(mew-current-get-window-config)
	(mew-delete-directory-recursively attachdir)
	(message "Draft was killed"))
    (message "Draft was not killed")))

(defun mew-draft-insert-signature ()
  "Insert the signature file specified by mew-signature-file.
If attachments exist and mew-signature-as-lastpart is *non-nil*,
the file is attached to the last part. Otherwise, the file is 
inserted into the body. If mew-signature-insert-last is *non-nil*,
the file is inserted to the end of the body. Otherwise, inserted
the cursor position."
  (interactive)
  (let ((sigfile
	 (expand-file-name (mew-signature-file (mew-tinfo-get-case)))))
    (if (not (file-exists-p sigfile))
	(message "No signature file %s" sigfile)
      (if (and (mew-attach-p) mew-signature-as-lastpart)
	  (progn
	    (goto-char (point-max))
	    (forward-line -2)
	    (mew-attach-forward)
	    (mew-attach-copy sigfile "Signature")
	    (let* ((nums (mew-syntax-nums))
		   (syntax (mew-syntax-get-entry mew-encode-syntax nums)))
	      (mew-syntax-set-cdp syntax nil)
	      (mew-syntax-set-cd  syntax mew-signature-description))
	    (mew-encode-syntax-print mew-encode-syntax))
	(when mew-signature-insert-last 
	  (if (null (mew-attach-p))
	      (goto-char (point-max))
	    (goto-char (1- (mew-attach-begin))))
	  (end-of-line)
	  (unless (bolp) (insert "\n")))
	(insert-file-contents sigfile)))))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;
;;; Re-highlight
;;; 

(defun mew-draft-rehighlight ()
  "Highlight header and body again."
  (interactive)
  (let ((mod (buffer-modified-p)))
    (mew-highlight-header)
    (mew-draft-header-keymap)
    (save-excursion
      (let ((beg (progn (goto-char (mew-header-end)) (forward-line) (point)))
            (end (or (mew-attach-begin) (point-max))))
        (mew-highlight-body-region beg end 'draft 'rehighlight)))
    (set-buffer-modified-p mod)))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;
;;; Privacy
;;;

(defun mew-draft-toggle-privacy-always ()
  "Toggle whether or not all drafts are protected."
  (interactive)
  (setq mew-protect-privacy-always (not mew-protect-privacy-always))
  (message "Set mew-protect-privacy-always to %s"
	   mew-protect-privacy-always)
  (mew-draft-mode-name))

(defun mew-draft-toggle-privacy-encrypted ()
  "Toggle whether or not drafts replying to encrypted messages are 
protected."
  (interactive)
  (setq mew-protect-privacy-encrypted (not mew-protect-privacy-encrypted))
  (message "Set mew-protect-privacy-encrypted to %s"
	   mew-protect-privacy-encrypted)
  (mew-draft-mode-name))

(defun mew-draft-set-privacy-type ()
  "\\<mew-draft-mode-map>
Set privacy service which will be effective when \\[mew-draft-make-message]."
  (interactive)
  (let* ((services (mew-pcdb-services))
	 (alist (mapcar (function (lambda (x) (cons (symbol-name x) x)))
			services))
	 str)
    (setq str (completing-read "Input privacy services : " alist nil t))
    (if (stringp str)
	(mew-tinfo-set-privacy-type (cdr (assoc str alist)))))
  (mew-draft-mode-name))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;
;;; Sending and Queuing
;;;

(defun mew-draft-process-message (action &optional privacy signer)
  (let* ((buf (current-buffer))
	 (case (mew-tinfo-get-case))
	 (server (mew-smtp-server case))
	 (ssh-server (mew-smtp-ssh-server case))
	 (pnm (mew-smtp-info-name server ssh-server))
	 (old-case (mew-tinfo-get-case))
	 sendit err)
    (if (mew-smtp-get-lock pnm)
	(message "Another message is being sent. Try later.")
      (run-hooks 'mew-send-hook)
      (if mew-case-guess-when-composed
	  (mew-draft-set-case-by-guess))
      (unless (string= old-case (mew-tinfo-get-case))
	(mew-draft-replace-fields old-case)
	(when (eq action 'send)
	  (mew-highlight-header)
	  (unless (mew-tinfo-get-hdr-file) (mew-draft-header-keymap)))
	(save-buffer))
      (if (and (eq action 'send) mew-ask-send)
	  (setq sendit (y-or-n-p "Really send this message? "))
	(setq sendit t))
      (when sendit
	(mew-smtp-set-case pnm case)
	(mew-smtp-set-server pnm server)
	(mew-smtp-set-port pnm (mew-smtp-port case))
	(mew-smtp-set-ssh-server pnm ssh-server)
	(mew-smtp-set-queue pnm (mew-queue-folder case))
	(mew-current-get-window-config)
	(delete-windows-on buf) ;; just in case
	(save-excursion
	  (save-window-excursion
	    (set-buffer buf)
	    (if (mew-encode pnm case privacy signer)
		(let ((mdi (concat (buffer-file-name) mew-draft-info-suffix)))
		  (if (file-exists-p mdi) (delete-file mdi))
		  (cond
		   ((eq action 'queue)
		    (mew-smtp-queue pnm "from Draft mode"))
		   ((eq action 'send)
		    (mew-smtp-send-message pnm))))
	      (setq err t)))))
      (if (not err)
	  (run-hooks 'mew-real-send-hook)
	(switch-to-buffer buf)
	(delete-other-windows)))))

(defun mew-draft-make-message (&optional privacy signer)
  "Compose a MIME message then put it into a queue folder."
  (interactive)
  (mew-draft-process-message 'queue privacy signer))

(defun mew-draft-send-message ()
  "Compose a MIME message then send it."
  (interactive)
  (mew-draft-process-message 'send))

;; backward-compatibility
(defalias 'mew-draft-send-letter 'mew-draft-send-message)

(provide 'mew-draft)

;;; Copyright Notice:

;; Copyright (C) 1996-2001 Mew developing team.
;; All rights reserved.

;; Redistribution and use in source and binary forms, with or without
;; modification, are permitted provided that the following conditions
;; are met:
;; 
;; 1. Redistributions of source code must retain the above copyright
;;    notice, this list of conditions and the following disclaimer.
;; 2. Redistributions in binary form must reproduce the above copyright
;;    notice, this list of conditions and the following disclaimer in the
;;    documentation and/or other materials provided with the distribution.
;; 3. Neither the name of the team nor the names of its contributors
;;    may be used to endorse or promote products derived from this software
;;    without specific prior written permission.
;; 
;; THIS SOFTWARE IS PROVIDED BY THE TEAM AND CONTRIBUTORS ``AS IS'' AND
;; ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
;; IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
;; PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL THE TEAM OR CONTRIBUTORS BE
;; LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
;; CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
;; SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR
;; BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
;; WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE
;; OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN
;; IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

;;; mew-draft.el ends here
