;;; mew-func.el --- Basic functions for Mew

;; Author:  Kazu Yamamoto <Kazu@Mew.org>
;; Created: Mar 23, 1997
;; Revised: Jul  8, 2001

;;; Code:

(require 'mew)

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;
;;; Mode
;;;

(defsubst mew-summary-p ()
  (eq major-mode 'mew-summary-mode))

(defsubst mew-virtual-p ()
  (eq major-mode 'mew-virtual-mode))

(defsubst mew-message-p ()
  (eq major-mode 'mew-message-mode))

(defsubst mew-draft-p ()
  (eq major-mode 'mew-draft-mode))

(defsubst mew-thread-p ()
  (mew-vinfo-get-thread-p))

(defsubst mew-summary-or-virtual-p ()
  (or (mew-summary-p) (mew-virtual-p)))

(defsubst mew-summary-or-thread-p ()
  (or (mew-summary-p) (mew-thread-p)))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;
;;; List functions
;;;

(defsubst mew-case-equal (str1 str2)
  (string= (downcase str1) (downcase str2)))

(defun mew-folder-recursive-match (key folder)
  "Initial substring match for folders.
If FOLDER is a sub-folder of KEY or KEY itself, t is returned."
  ;; file-name-as-directory should be first because
  ;; the path separator may be regex-non-safe.
  (let ((regex (if (string-match "^[+-=%]+$" key)
		   (mew-folder-regex key)
		 (mew-folder-regex (file-name-as-directory key)))))
    (string-match regex (file-name-as-directory folder))))

(defun mew-member-case-equal (str list)
  "Return the position equal to STR in LIST. Case is ignored."
  (let ((n 0))
    (catch 'member
      (while list
	(if (string= (downcase (car list)) (downcase str))
	    (throw 'member n))
	(setq list (cdr list))
	(setq n (1+ n))))))

(defun mew-member-match (str list &optional ignore-case)
  "Return the position matched to STR in LIST. If
IGNORE-CASE is t, matching is performed by ignoring case."
  (let ((n 0) (case-fold-search ignore-case))
    (catch 'member
      (while list
	(if (string-match (car list) str)
	    (throw 'member n))
	(setq list (cdr list))
	(setq n (1+ n))))))

(defun mew-uniq-list (lst)
  "Destructively uniqfy elements of LST.
This is O(N^2). So, don't use this function with a large LST."
  (let ((tmp lst))
    (while tmp (setq tmp (setcdr tmp (delete (car tmp) (cdr tmp))))))
  lst)

(defun mew-uniq-alist (alst)
  "Destructively uniqfy elements of ALST."
  (let ((vec (make-vector 511 0))
	ent str ret)
    (while alst
      (setq ent (car alst))
      (setq str (car ent))
      (setq alst (cdr alst))
      (cond
       ((not (stringp str))
	(setq ret (cons ent ret)))
       ((intern-soft str vec)
	())
       (t
	(setq ret (cons ent ret))
	(intern str vec))))
    (nreverse ret)))

(defun mew-delete (key alist)
  "Destructively delete elements whose first member is equal to key"
  (if (null key)
      alist
    (let (ret)
      (while (and (listp (nth 0 alist)) (equal (car (nth 0 alist)) key))
	(setq alist (cdr alist)))
      (setq ret alist)
      (while alist
	(if (and (listp (nth 1 alist)) (equal (car (nth 1 alist)) key))
	    (setcdr alist (cdr (cdr alist)))
	  (setq alist (cdr alist))))
      ret)))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;
;;; Associative list functions
;;;

(defun mew-assoc-equal (key alist nth)
  (let (a n)
    (catch 'loop
      (while alist
	(setq a (car alist))
	(setq n (nth nth a))
	(if (or (equal n key) (eq n t))
	    (throw 'loop a))
	(setq alist (cdr alist))))))

(defun mew-assoc-case-equal (key alist nth)
  (let ((skey (downcase key)) a n)
    (catch 'loop
      (while alist
	(setq a (car alist))
	(setq n (nth nth a))
	(if (or (and (stringp n) (string= (downcase n) skey))
		(eq n t))
	    (throw 'loop a))
	(setq alist (cdr alist))))))

(defun mew-assoc-match (key alist nth)
  "Return list in ALIST that KEY regex is matched to its NTH element.
Case is ignored. Note that the NTH element is 't', 
the list is always selected."
  (let ((case-fold-search t) a n)
    (catch 'loop
      (while alist
	(setq a (car alist))
	(setq n (nth nth a))
	(if (or (and (stringp n) (string-match key n))
		(equal n key) (eq n t))
	    (throw 'loop a))
	(setq alist (cdr alist))))))

(defun mew-assoc-match2 (key alist nth)
  "Return list in ALIST whose NTH regex is matched to KEY.
Case is ignored. Note that the NTH element is 't', 
the list is always selected."
  (let ((case-fold-search t) a n)
    (catch 'loop
      (while alist
	(setq a (car alist))
	(setq n (nth nth a))
	(if (or (and (stringp n) (string-match n key))
		(equal n key) (eq n t))
	    (throw 'loop a))
	(setq alist (cdr alist))))))

(defun mew-assoc-match3 (key alist nth)
  "Return list in ALIST whose NTH regex is matched to KEY.
Case is ignored. Note that the NTH element is 't', 
the list is always selected. The deference from mew-assoc-match2
is that this returns the position of a selected list in addition
to the list itself."
  (let ((case-fold-search t) (i 0) a n )
    (catch 'loop
      (while alist
	(setq a (car alist))
	(setq n (nth nth a))
	(if (or (and (stringp n) (string-match n key))
		(equal n key) (eq n t))
	    (throw 'loop (cons i a)))
	(setq i (1+ i))
	(setq alist (cdr alist))))))

(defsubst mew-assoc-member (key lol nth)
  "Return a list member of LoL whose NTH list contains 
a member equal to KEY."
  (mew-assoc-member-base key lol nth (function member)))

(defsubst mew-assoc-member-case-equal (key lol nth)
  "Return a list member of LoL whose NTH list contains 
a member equal to KEY ignoring case."
  (mew-assoc-member-base key lol nth (function mew-member-case-equal)))

(defun mew-assoc-member-base (key lol nth func)
  "Return a list member of LoL whose NTH list contains KEY
in the context of FUNC."
  (let (l)
    (catch 'loop
      (while lol
	(setq l (car lol))
	(setq lol (cdr lol))
	(if (and (listp (nth nth l)) (funcall func key (nth nth l)))
	    (throw 'loop l))))))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;
;;; getting next
;;;

(defun mew-get-next (LIST MEM)
  "Return of a member in LIST which is the next member of MEM."
  (let (frst next crnt)
    (setq frst (car LIST))
    (setq LIST (cdr LIST))
    (setq next (car LIST))
    (if (equal frst MEM)
	(if next next frst)
    (catch 'loop
      (while LIST
	(setq crnt next)
	(setq LIST (cdr LIST))
	(setq next (car LIST))
	(if (equal crnt MEM)
	    (throw 'loop (if next next frst))))))))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;
;;; Modifying list
;;;

(defmacro mew-add-first (variable value)
  `(setq ,variable (cons ,value ,variable)))

(defmacro mew-insert-after (variable value key)
  `(let ((var ,variable))
     (catch 'loop
       (while var
	 (if (equal (nth 0 (car var)) ,key)
	     (throw 'loop (setcdr var (cons ,value (cdr var)))))
	 (setq var (cdr var))))))

(defmacro mew-replace-with (variable value key)
  `(let ((var ,variable))
     (catch 'loop
       (while var
	 (if (equal (nth 0 (car var)) ,key)
	     (throw 'loop (setcar var ,value)))
	 (setq var (cdr var))))))

(defmacro mew-remove-entry (variable key)
  `(let ((crn ,variable) prv)
     (if (equal (nth 0 (car crn)) ,key)
	 (setq ,variable (cdr crn))
       (setq prv crn)
       (setq crn (cdr crn))
       (catch 'loop
	 (while crn
	   (if (equal (nth 0 (car crn)) ,key)
	       (throw 'loop (setcdr prv (cdr crn))))
	   (setq prv crn)
	   (setq crn (cdr crn)))))))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;
;;; String
;;;

(defun mew-replace-character (string from to)
  "Replace characters equal to FROM to TO in STRING."
  (let ((len (length string))
	(cnt 0))
    (while (< cnt len)
      (if (char-equal (aref string cnt) from)
	  (aset string cnt to))
      (setq cnt (1+ cnt)))
    string))

(defun mew-replace-white-space (string)
  "Replace white characters to a space."
  (while (string-match "\t+" string)
    (setq string (replace-match " " nil t string)))
  (while (string-match "  +" string)
    (setq string (replace-match " " nil t string)))
  string)

(defsubst mew-match (pos &optional string)
  "Substring POSth matched from STRING or the current buffer."
  (cond 
   ((stringp string)
    (substring string (match-beginning pos) (match-end pos)))
   (t 
    (mew-buffer-substring (match-beginning pos) (match-end pos)))))

(defsubst mew-capitalize (ostr)
  "Syntax table independent version of capitalize.
Words are separated by '/' and '-'."
  (let* ((len (length ostr))
	 (nstr (make-string len ?a))
	 (i 0) (topp t) c)
    (while (< i len)
      (setq c (aref ostr i))
      (cond
       (topp
	(aset nstr i (upcase c))
	(setq topp nil))
       ((or (char-equal c ?/) (char-equal c ?-))
	(aset nstr i c)
	(setq topp t))
       (t
	(aset nstr i (downcase c))))
      (setq i (1+ i)))
    nstr))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;
;;; Insertion
;;;

(defsubst mew-insert (form str)
  (when str
    (if form
	(insert (format form str))
      (insert str))))

(defsubst mew-insert-message (fld msg rcs size)
  (let ((file (mew-expand-folder fld msg)))
    (cond
     ((or (mew-folder-newsp fld) (mew-folder-imapp fld))
      (error "%s isn't supported yet" fld))
     ((file-readable-p file)
      (mew-frwlet
       rcs mew-cs-dummy
       (insert-file-contents file nil 0 size))
      ;; return physical size
      (cons (mew-file-get-time file) (mew-file-get-size file)))
     (t
      (error "%s%s doesn't exist" (file-name-as-directory fld) msg)))))

(defsubst mew-insert-manual (&rest args)
  (insert (substitute-command-keys (apply (function concat) args))))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;
;;; Stolen form Perl
;;;

(defsubst mew-join (separator string-list)
  (mapconcat (function identity) string-list separator))

(defun mew-split (str sepchar)
  "Return a list of strings splitting STR with SEPCHAR."
 (let ((len (length str)) (start 0) (i 0) ret)
   (while (< i len)
     (when (char-equal (aref str i) sepchar)
       (setq ret (cons (substring str start i) ret))
       (setq start (1+ i)))
     (setq i (1+ i)))
   (if (/= start len)
       (setq ret (cons (substring str start) ret)))
   (nreverse ret)))

(defun mew-split-quoted (str sepchar &optional qopen qclose)
  "Return a list of strings splitting STR with SEPCHAR.
SEPCHARs in double-quoted strings are ignored.
If QUOTEDCHAR is provided, SEPCHARs between QOPEN and QCLOSE are
also ignored."
 (let ((qlevel 0) (len (length str)) (start 0) (i 0) dblq ret c)
   (if (and qopen (not qclose)) (setq qclose qopen))
   (while (< i len)
     (setq c (aref str i))
     (cond 
      ((char-equal ?\\ c)
       (setq i (1+ i)))
      ((char-equal ?\" c)
       (setq dblq (not dblq)))
      ((and qopen (char-equal c qopen))
       (setq qlevel (1+ qlevel)))
      ((and qclose (char-equal c qclose))
       (setq qlevel (1- qlevel)))
      ((char-equal c sepchar)
       (if (or dblq (>= qlevel 1))
	   ()
	 (setq ret (cons (substring str start i) ret))
	 (setq start (1+ i)))))
     (setq i (1+ i)))
   (if (/= start len)
       (setq ret (cons (substring str start) ret)))
   (nreverse ret)))

(defun mew-chop (str)
  "Split off preceding and following white spaces."
  (let ((i 0) (j (length str)) c)
    (catch 'loop
      (while (< i j)
	(setq c (aref str i))
	(if (or (char-equal c 32) (char-equal c ?\t))
	    (setq i (1+ i))
	  (throw 'loop nil))))
    (setq j (1- j))
    (catch 'loop
      (while (< i j)
	(setq c (aref str j))
	(if (or (char-equal c 32) (char-equal c ?\t))
	    (setq j (1- j))
	  (throw 'loop nil))))
    (substring str i (1+ j))))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;
;;; Folder
;;;

(defsubst mew-folder-regex (folder)
  (concat "^" (regexp-quote folder)))

(defsubst mew-folder-mailp (folder)
  (string-match "^+" folder))

(defsubst mew-folder-newsp (folder)
  (string-match "^-" folder))

(defsubst mew-folder-local-newsp (folder)
  (string-match "^=" folder))

(defsubst mew-folder-imapp (folder)
  (string-match "^%" folder))

(defsubst mew-folder-virtualp (folder)
  (string-match "^\\+[+=]" folder))

(defsubst mew-folder-to-dir (folder)
  (if (string-match "^[+=]" folder)
      (substring folder 1 nil)
    folder))

(defsubst mew-folder-localp (folder)
  (or (mew-folder-mailp folder) (mew-folder-local-newsp folder)))

(defsubst mew-folder-remotep (folder)
  (or (mew-folder-newsp folder) (mew-folder-imapp folder)))

(defsubst mew-folder-inboxp (folder)
  (member folder mew-inbox-folders))

(defsubst mew-folder-queuep (folder)
  (member folder mew-queue-folders))

(defsubst mew-folder-draftp (folder)
  (equal folder mew-draft-folder))

(defun mew-canonicalize-folder (folder)
  (if (or (mew-folder-mailp folder)
	  (mew-folder-local-newsp folder)
	  (mew-folder-imapp folder)
	  (file-name-absolute-p folder))
      folder
    (concat "+" folder)))

(defun mew-expand-folder (folder &optional message)
  (let ((subdir (substring folder 1 nil))
	dir)
    (cond
     ((mew-folder-mailp folder)
      (setq dir (expand-file-name subdir mew-mail-path)))
     ((mew-folder-local-newsp folder)
      (setq dir (expand-file-name subdir mew-news-path)))
     ((mew-folder-imapp folder)
      ()) ;; IMAP
     ((file-name-absolute-p folder)
      (setq dir (expand-file-name folder))))
    (if message
	(expand-file-name message dir)
      dir)))

(defun mew-folder-check (folder &optional force-to-create)
  "A function to see if FOLDER exists.
Return t if exists or created. Otherwise, return nil."
  (if (not (stringp folder))
      nil ;; wrong
    (let ((absdir (mew-expand-folder folder))  ;; /home/Mail/foo
	  (create-it force-to-create))
      (if (file-exists-p absdir)
	  (if (file-directory-p absdir)
	      (progn
		;; The folder doesn't exist in mew-folder-list if
		;; a user creates it by hand...
		(mew-folder-insert folder) ;; just in case
		t) ;; exists
	    (message "%s is a file" folder)
	    nil) ;; xxx exists but a file
	(unless create-it
	  (if (mew-folder-imapp folder)
	      (if (y-or-n-p (format "Maybe %s doesn't exist (cache-dir not found). Create it? " folder))
		  (setq create-it t))
	    (if (y-or-n-p (format "%s doesn't exist. Create it? " folder))
		(setq create-it t))))
	(if (not create-it)
	    nil ;; not created
	  (mew-make-directory absdir)
	  (mew-folder-insert folder)
	  (mew-lisp-save mew-folder-list-file mew-folder-list)
	  (mew-lisp-save mew-folder-alist-file mew-folder-alist)
	  (message "%s has been created" folder)
	  t))))) ;; created

(defsubst mew-dir-messages (dir)
  ;; (default-file-name-coding-system nil)
  ;; (file-name-coding-system nil)
  ;; This makes scan faster but makes non-ASCII directories
  ;; unavailable.
  (directory-files dir nil mew-regex-message-files 'no-sort))

(defun mew-folder-new-message (folder &optional num-only)
  (let* ((dir (mew-expand-folder folder))
	 (max 0) cur maxfile maxpath msgs)
    ;; xxx create if there is no directory?
    (if (not (file-directory-p dir))
	nil
      (setq msgs (mew-dir-messages dir))
      (while msgs
	(setq cur (string-to-int (car msgs)))
	(if (> cur max) (setq max cur))
	(setq msgs (cdr msgs)))
      (setq max (1+ max))
      (setq maxfile (int-to-string max))
      (setq maxpath (mew-expand-folder folder maxfile))
      (when (file-exists-p maxpath)
	;; If NFS is used, readdir() may fail. Emacs 20 doesn't
	;; rescan the directory again. So, we need to rescan in
	;; the Emacs level.
	(mew-touch-folder folder);; can clear cache?
	(setq msgs (mew-dir-messages dir))
	(setq max 0)
	(while msgs
	  (setq cur (string-to-int (car msgs)))
	  (if (> cur max) (setq max cur))
	  (setq msgs (cdr msgs)))
	(setq max (1+ max))
	(setq maxfile (int-to-string max))
	(setq maxpath (mew-expand-folder folder maxfile)))
      (when (get-file-buffer maxpath)
	;; file not exist but there is a buffer.
	(setq max (1+ max))
	(setq maxfile (int-to-string max))
	(setq maxpath (mew-expand-folder folder maxfile)))
      (while (file-exists-p maxpath)
	(setq maxfile (read-string (format "%s/%s exists. Input a message number : " max folder)))
	(while (not (string-match mew-regex-message-files maxfile))
	  (setq maxfile (read-string "Input NUMBER : ")))
	(setq maxpath (mew-expand-folder folder maxfile)))
      (if num-only
	  maxfile
	maxpath))))

(defun mew-touch-folder (fld)
  (let (file)
    (if (and mew-touch-folder-p
	     (stringp mew-summary-touch-file)
	     (setq file (mew-expand-folder fld mew-summary-touch-file))
	     (file-writable-p file))
	(write-region "touched by Mew." nil file nil 'no-msg))))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;
;;; Directory
;;;

(defun mew-make-directory (path)
  (let ((parent (directory-file-name (file-name-directory path))))
    (unless (file-directory-p parent)
      (mew-make-directory parent))
    (if (and (file-exists-p path) (not (file-directory-p path)))
	(delete-file path))
    (make-directory path)
    (if (/= mew-folder-mode (mew-file-get-mode path))
	(set-file-modes path mew-folder-mode))))

(defun mew-delete-directory-recursively (dir)
  (when (file-directory-p dir)
    (let ((files (directory-files dir 'full mew-regex-files 'no-sort)))
      (while files
	(cond
	 ((file-symlink-p (car files))
	  ;; never chase symlink which points a directory
	  (delete-file (car files)))
	 ((file-directory-p (car files))
	  (mew-delete-directory-recursively (car files)))
	 (t
	  (delete-file (car files))))
	(setq files (cdr files))))
    (unless (directory-files dir 'full mew-regex-files 'no-sort)
      (delete-directory dir))))

(defun mew-directory-dirs (dir)
  (let ((files (directory-files dir 'full "^[^.]" 'no-sort))
	file dirs)
    (while files
      (setq file (car files))
      (setq files (cdr files))
      (if (file-directory-p file)
	  (setq dirs (cons file dirs))))
    dirs))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;
;;; File existence
;;;

(defun mew-which (file path)
  (catch 'loop
    (while path
      (if (file-exists-p (expand-file-name file (car path)))
	  (throw 'loop (expand-file-name file (car path)))
	(setq path (cdr path))))))

(defun mew-which-el (elfile)
  (or (mew-which (concat elfile ".el") load-path)
      (mew-which (concat elfile ".elc") load-path)))

(defun mew-which-exec (execfile)
  (or (mew-which execfile exec-path)
      (mew-which (concat execfile ".exe") exec-path)))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;
;;; File attribute
;;;

;; Functions to get other attributes are implemented in C level.

(defsubst mew-file-get-links (file)
  (let ((w32-get-true-file-link-count t)) ;; for Meadow
    (nth 1 (file-attributes file))))

(defsubst mew-file-get-time (file)
  (nth 5 (file-attributes file)))

(defsubst mew-file-get-size (file)
  (nth 7 (file-attributes file)))

(defun mew-file-get-mode (file)
  (let* ((mode (nth 8 (file-attributes file)))
	 (len (length mode))
	 (i 1)
	 (dec 0))
    (while (< i len)
      (setq dec (* dec 2))
      (unless (char-equal (aref mode i) ?-)
	(setq dec (1+ dec)))
      (setq i (1+ i)))
    dec))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;
;;; File
;;;

(defsubst mew-file-chase-links (file)
  "Chase links in FILENAME until a name that is not a link.
Does not examine containing directories for links."
  (let ((ret file) exp)
    (while (setq exp (file-symlink-p ret))
      (setq ret (expand-file-name exp (file-name-directory ret))))
    ret))

(defun mew-file-from-home (str)
  (if (string-match (expand-file-name mew-home) str)
      (concat mew-home (substring str (match-end 0)))
    str))

(defun mew-prepend-prefix (file prefix)
  (if (file-name-absolute-p file)
      (concat (file-name-directory file) prefix (file-name-nondirectory file))
    (concat prefix file)))

(defun mew-rotate-log-files (file-name)
  (let ((i 8) (file (expand-file-name file-name mew-conf-path)))
    (when (and (file-exists-p file)
	       (>= (mew-file-get-size file) mew-log-max-size))
      (while (>= i 0)
	(if (file-exists-p (format "%s.%d" file i))
	    (rename-file (format "%s.%d" file i)
			 (format "%s.%d" file (1+ i)) t))
	(setq i (1- i)))
      (rename-file file (format "%s.0" file)))))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;
;;; temp name
;;;

(defun mew-make-temp-name (&optional fname)
  (unless (file-exists-p mew-temp-dir)
    (mew-make-directory mew-temp-dir)) ;; just in case
  (if fname
      ;; File name of a temporary file should be ASCII only.
      (if (string-match "^[ -~]+$" fname)
	  (expand-file-name fname mew-temp-dir)	
	(if (string-match "\\.[ -~]+$" fname)
	    (concat (make-temp-name mew-temp-file) (mew-match 0 fname))
	  (make-temp-name mew-temp-file)))
    (make-temp-name mew-temp-file)))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;
;;; Random
;;;

(defvar mew-random-last nil)
(defvar mew-random-base-09 "0123456789")
(defvar mew-random-base-az "abcdefghijklmnopqrstuvwxwz")

(defun mew-random ()
  (let* ((vec (recent-keys))
	 (len (length vec))
	 (i 0) (ran (+ (random) (emacs-pid)))
	 c)
    (while (< i len)
      (setq c (aref vec i))
      (cond
       ((and mew-temacs-p (numberp c))
	(setq ran (+ ran c)))
       ((and mew-xemacs-p (eventp c))
	(if (characterp (event-to-character c))
	    (setq ran (+ ran (char-to-int (event-to-character c)))))))
      (setq i (1+ i)))
    (abs ran)))

(defun mew-random-string (len nump)
  (let* ((base (if nump mew-random-base-09 mew-random-base-az))
	 (baselen (length base))
	 (ret (make-string len ?a))
	 (i 0))
    (while (< i len)
      (aset ret i (aref base (% (mew-random) baselen)))
      (setq i (1+ i)))
    (while (string= ret mew-random-last)
      (setq i 0)
      (while (< i len)
	(aset ret i (aref base (% (mew-random) baselen)))
	(setq i (1+ i))))
    ret))

(defun mew-random-filename (dir len nump &optional suffix)
  (let ((cnt 0) (max 20) ;; ad hoc
	file filepath)
    (setq file (concat (mew-random-string len nump) suffix))
    (setq filepath (expand-file-name file dir))
    (while (and (file-exists-p filepath) (< cnt max))
      (setq file (concat (mew-random-string len nump) suffix))
      (setq filepath (expand-file-name file dir))
      (setq cnt (1+ cnt)))
    (if (file-exists-p filepath) 
	nil
      filepath)))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;
;;; Buffer
;;;

(defsubst mew-erase-buffer ()
  (mew-elet
   (widen)
   (erase-buffer)
   (buffer-disable-undo)))

(defun mew-remove-buffer (buf)
  (if (get-buffer buf) (kill-buffer buf)))

(defmacro mew-elet (&rest body)
  `(let ((buffer-read-only nil)
	 (inhibit-read-only t)
	 (zmacs-regions nil)) ;; for XEmacs
     ,@body))

(defsubst mew-region-bytes (beg end buf)
  (if (fboundp 'string-bytes)
      (save-excursion
	(set-buffer buf)
	(string-bytes (mew-buffer-substring beg end)))
    (- end beg)))

(defun mew-count-lines (beg end &optional skip-regex)
  "Return number of lines between BEG and END."
  (save-excursion
    (save-restriction
      (narrow-to-region beg end)
      (goto-char (point-min))
      (if skip-regex
	  (let ((lines 0))
	    (while (not (eobp))
	      (forward-line)
	      (unless (looking-at skip-regex)
		(setq lines (1+ lines))))
	    lines)
	(- (buffer-size) (forward-line (buffer-size)))))))

(defalias 'mew-buffer-substring 'buffer-substring-no-properties)

(defsubst mew-insert-buffer-substring (buf beg end)
  (insert (save-excursion (set-buffer buf) (mew-buffer-substring beg end))))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;
;;; process
;;;

(defmacro mew-filter (&rest body)
  `(if (bufferp (process-buffer process)) ;; MUST use 'process'
       (let ((obuf (buffer-name)))
	 ;; must use buffer-name instead of current-buffer
	 ;; so that get-buffer can detect killed buffer.
	 (unwind-protect
	     (progn
	       ;; buffer surely exists.
	       (set-buffer (process-buffer process)) ;; necessary
	       ,@body)
	   (if (get-buffer obuf)
	       ;; the body sometimes kills obuf.
	       (set-buffer obuf))))))

(defun mew-start-process-disp (name buffer program &rest program-args)
  (let ((disp (if (and mew-xemacs-p (eq (device-type) 'x))
                  (device-connection)
		(cdr (assq 'display (frame-parameters)))))
        (process-environment (copy-sequence process-environment)))
    (setenv "DISPLAY" disp)
    (apply (function start-process) name buffer program program-args)))

(defun mew-start-process-lang (name buffer program &rest program-args)
  (let ((process-environment (copy-sequence process-environment)))
    (setenv "LANGUAGE" "en")
    (apply (function start-process) name buffer program program-args)))

(defun mew-call-process-lang (prog &optional infile buffer display &rest args)
  (let ((process-environment (copy-sequence process-environment)))
    (setenv "LANGUAGE" "en")
    (apply (function call-process) prog infile buffer display args)))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;
;;; Global info
;;;

(defsubst mew-info (name)
  (cond
   ((vectorp name) name) ;; just for .mqi
   ((stringp name)
    (if (or (not (intern-soft name))
	    (not (boundp (intern name))))
	(if (string-match "^mew-[^-]+-info-" name)
	    (let* ((sym (intern (concat (mew-match 0 name) "list")))
		   (lst (symbol-value sym))
		   (len (length lst)))
	      (set (intern name) (make-vector len nil)))))
    (symbol-value (intern-soft name)))))

(defun mew-info-defun (prefix lst)
  (let ((i 0) ent)
    (while lst
      (setq ent (car lst))
      (setq lst (cdr lst))
      (fset (intern (concat prefix "get-" ent))
	    `(lambda (arg)
	       (cond
		((stringp arg) (aref (mew-info arg) ,i))
		((vectorp arg) (aref arg ,i)))))
      (fset (intern (concat prefix "set-" ent))
	    `(lambda (arg value)
	       (cond
		((stringp arg) (aset (mew-info arg) ,i value))
		((vectorp arg) (aset arg ,i value)))))
      (setq i (1+ i)))))

(defun mew-info-clean-up (arg)
  (let ((i 0) vec len)
    (cond
     ((stringp arg) (setq vec (mew-info arg)))
     ((vectorp arg) (setq vec arg)))
    (setq len (length vec))
    (while (< i len)
      (aset vec i nil)
      (setq i (1+ i)))))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;
;;; Buffer local info
;;;

(defsubst mew-blinfo (sym)
  (let* ((name (symbol-name sym))
	 (lname (concat name "-list"))
	 (lsym (intern lname))
	 (lst (symbol-value lsym))
	 (len (length lst)))
    (set sym (make-vector len nil))
    (symbol-value sym)))

(defun mew-blinfo-defun (blv-sym lst)
  (let ((i 0) ent)
    (while lst
      (setq ent (car lst))
      (setq lst (cdr lst))
      (fset (intern (concat (symbol-name blv-sym) "-get-" ent))
	    `(lambda ()
	       (cond
		((null ,blv-sym) (aref (mew-blinfo (quote ,blv-sym)) ,i))
		((vectorp ,blv-sym) (aref ,blv-sym ,i)))))
      (fset (intern (concat (symbol-name blv-sym) "-set-" ent))
	    `(lambda (value)
	       (cond
		((null ,blv-sym) (aset (mew-blinfo (quote ,blv-sym)) ,i value))
		((vectorp ,blv-sym) (aset ,blv-sym ,i value)))))
      (setq i (1+ i)))))


(defvar mew-ainfo-list '("icon" "win-cfg"))
(mew-blinfo-defun 'mew-ainfo mew-ainfo-list)

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;
;;; Address
;;;

(defsubst mew-get-my-address ()
  (or (mew-header-parse-address mew-from:) (mew-mail-address)))

;;

(defsubst mew-get-my-address-regex-list ()
  "This creates a list of regular expression used to tell
whether or not a given address is mine. The list is created
from (mew-user), (mew-mail-address), and 'mew-mail-address-list'."
  (cons (concat "^" (regexp-quote (mew-user)) "$")
	(cons (concat "^" (regexp-quote (mew-mail-address)) "$")
	      mew-mail-address-list)))
  
(defsubst mew-is-my-address (addrs from)
  (and from
       (let ((case-fold-search t))
	 (catch (quote match)
	   (car (mapcar (function (lambda (arg)
				    (and (string-match arg from)
					 (throw (quote match) t))))
			addrs))))))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;
;;; Lisp load/save
;;;

(defun mew-lisp-load (filename)
  "Load lisp from FILENAME"
  (let* ((fullname (if (file-name-absolute-p filename)
		       filename
		     (expand-file-name filename mew-conf-path)))
	 (tmp-buf (create-file-buffer fullname))
	 lisp)
    (if (file-readable-p fullname)
	(save-excursion
	  (set-buffer tmp-buf)
	  (mew-erase-buffer)
	  (mew-frwlet
	   mew-cs-m17n mew-cs-dummy
	   (insert-file-contents fullname))
	  (setq lisp 
		(condition-case nil
		    (read (current-buffer))
		  (error ())))))
    (mew-remove-buffer tmp-buf)
    lisp))

(defun mew-lisp-save (filename lisp &optional nobackup)
  "Save LISP to FILENAME. LISP is truncated to mew-lisp-max-length
by side-effect."
  (let* ((fullname (if (file-name-absolute-p filename)
		       filename
		     (expand-file-name filename mew-conf-path)))
	 (backname (concat fullname mew-backup-suffix))
	 (tmp-buf (create-file-buffer fullname))
	 print-length print-level) ;; for Emacs 21
    (if (file-writable-p fullname)
	(save-excursion
	  (if nobackup
	      (if (file-exists-p fullname)
		  (delete-file fullname))
	    (if (file-exists-p fullname)
		(rename-file fullname backname 'override)))
	  (set-buffer tmp-buf)
	  (mew-erase-buffer)
	  (if (> (length lisp) mew-lisp-max-length)
	      (setcdr (nthcdr (1- mew-lisp-max-length) lisp) nil))
	  (pp lisp tmp-buf)
	  (mew-frwlet
	   mew-cs-dummy mew-cs-m17n
	   (write-region (point-min) (point-max) fullname nil 'no-msg))))
    (mew-remove-buffer tmp-buf)))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;; 
;;; Month
;;;

(defvar mew-time-mon-alist
  '(("Jan" .  1) ("Feb" .  2) ("Mar" .  3) ("Apr" .  4)
    ("May" .  5) ("Jun" .  6) ("Jul" .  7) ("Aug" .  8)
    ("Sep" .  9) ("Oct" . 10) ("Nov" . 11) ("Dec" . 12)))

(defsubst mew-time-mon-str-to-int (str)
  (or (cdr (assoc (capitalize str) mew-time-mon-alist)) 0))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;; 
;;; Time Zone
;;;

(defvar mew-time-tmzn-alist
  '(("PST" . -8) ("PDT" . -7) ("MST" . -7) ("MDT" . -6)
    ("CST" . -6) ("CDT" . -5) ("EST" . -5) ("EDT" . -4)
    ("AST" . -4) ("NST" . -3) ("UT"  . +0) ("GMT" . +0)
    ("BST" . +1) ("MET" . +1) ("EET" . +2) ("JST" . +9)))

(defsubst mew-time-tmzn-str-to-int (str)
  (cdr (assoc (upcase str) mew-time-tmzn-alist)))

(defsubst mew-time-tmzn-int ()
  (let ((tz (car (current-time-zone))))
    (if (< tz 0)
	(format "-%02d%02d" (/ (- tz) 3600) (/ (% (- tz) 3600) 60))
      (format "+%02d%02d" (/ tz 3600) (/ (% tz 3600) 60)))))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;; RFC 822
;;; Date: Wed, 26 Jul 2000 21:18:35 +0900 (JST)
;;;

(defvar mew-time-rfc-regex
  "\\([0-9]+\\)[ \t]+\\([a-zA-Z]+\\)[ \t]+\\([0-9]+\\)[ \t]+\\([0-9]+\\):\\([0-9]+\\)\\(:\\([0-9]+\\)\\)?\\([ \t]+\\([-+0-9a-zA-Z]+\\)\\)?")

(defmacro mew-time-rfc-day  ()
  '(string-to-int (substring s (match-beginning 1) (match-end 1))))

(defmacro mew-time-rfc-mon  ()
  '(substring s (match-beginning 2) (match-end 2)))

(defmacro mew-time-rfc-year ()
  '(string-to-int (substring s (match-beginning 3) (match-end 3))))

(defmacro mew-time-rfc-hour ()
  '(string-to-int (substring s (match-beginning 4) (match-end 4))))

(defmacro mew-time-rfc-min  ()
  '(substring s (match-beginning 5) (match-end 5)))

(defmacro mew-time-rfc-sec  ()
  '(if (match-beginning 7)
       (substring s (match-beginning 7) (match-end 7))
     "00"))

(defmacro mew-time-rfc-tmzn ()
  '(if (match-beginning 9)
       (let ((tmzn (substring s (match-beginning 9) (match-end 9)))
	     int)
	 (cond
	  ((string-match "^[-+][0-9]+$" tmzn)
	   (/ (string-to-int tmzn) 100))
	  ((setq int (mew-time-tmzn-str-to-int tmzn))
	   int)
	  (t 0)))
     0))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;
;;;
;;;

;; "20000726231835"
(defsubst mew-time-rfc-to-sortkey (s)
  (if (string-match mew-time-rfc-regex s)
      (let ((year (mew-time-rfc-year))
	    (mon  (mew-time-mon-str-to-int (mew-time-rfc-mon)))
	    (day  (mew-time-rfc-day))
	    (hour (mew-time-rfc-hour))
	    (min  (mew-time-rfc-min))
	    (sec  (mew-time-rfc-sec))
	    (tmzn (mew-time-rfc-tmzn)))
	(cond
	 ((< year 50)
	  (setq year (+ year 2000)))
	 ((< year 100)
	  (setq year (+ year 1900))))
	(setq hour (- hour tmzn))
	(cond
	 ((< hour 0)
	  (setq hour (+ hour 24) day (1- day))
	  (when (< day 1)
	    (if (< (setq mon (1- mon)) 1)
		(setq year (1- year) mon (+ mon 12)))
	    (setq day (mew-last-day year mon))))
	 ((> hour 23)
	  (setq hour (- hour 24) day (1+ day))
	  (if (> day (mew-last-day year mon))
		(if (> (setq day 1 mon (1+ mon)) 12)
		    (setq year (1+ year) mon (- mon 12))))))
	(format "%4d%02d%02d%02d%s%s" year mon day hour min sec))))

(defvar mew-last-day-list '(31 nil 31 30 31 30 31 31 30 31 30 31))

(defsubst mew-last-day (year mon)
  (let ((day (nth (1- mon) mew-last-day-list)))
    (cond
     (day day)
     ((and (= (% year 4) 0)
	   (or (= (% year 400) 0)
	       (/= (% year 100) 0)))
      29)
     (t 28))))

(defsubst mew-time-ctz-to-sortkey (time)
  (let ((system-time-locale "C"))
    (format-time-string "%Y%m%d%H%M%S" time)))

;; Wed, 26 Jul 2000 21:18:35 +0900 (JST)
(defsubst mew-time-ctz-to-rfc (time)
  (let ((system-time-locale "C")
	(tmzn-int (mew-time-tmzn-int)))
    ;; XEmacs doesn't have %z
    (format (format-time-string "%a, %d %b %Y %T %%s (%Z)" time) tmzn-int)))

;; 2000/07/12 16:22:30
(defsubst mew-time-ctz-to-logtime (time)
  (let ((system-time-locale "C"))
    (format-time-string "%Y/%m/%d %H:%M:%S" time)))

;; 20000712.155559
(defsubst mew-time-ctz-to-msgid (time)
  (let ((system-time-locale "C"))
    (format-time-string "%Y%m%d.%H%M%S" time)))

;;

(defsubst mew-time-calc (new old)
  (let ((million 1000000)
	(sec (+ (* 65536 (- (nth 0 new) (nth 0 old)))
		(- (nth 1 new) (nth 1 old))))
	(usec (- (nth 2 new) (nth 2 old))))
    (if (< usec 0)
        (setq sec (1- sec)
              usec (+ usec million))
      (if (>= usec million)
          (setq sec (1+ sec)
                usec (- usec million))))
    (+ sec (/ usec (float million)))))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;; 
;;; Multibyte
;;;

(defsubst mew-set-buffer-multibyte (arg)
  (if (fboundp 'set-buffer-multibyte)
      (if mew-unibyte-p () (set-buffer-multibyte arg))))

(defvar mew-unibyte-p 
  (and (boundp 'enable-multibyte-characters)
       (null enable-multibyte-characters)))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;; 
;;; Network
;;;

(defsubst mew-port-sanity-check (port)
  (if (string-match "^[0-9]+$" port)
      (string-to-int port)
    port))

(provide 'mew-func)

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;; 
;;; sit-for
;;;

(defsubst mew-redraw (&optional time)
  (sit-for (or time 0)))

(defmacro mew-rendezvous (who)
  ;; Wait for the termination of WHO.
  ;; Emacs doesn't provide synchronize mechanism with
  ;; an asynchronous process. So, take this way. 
  `(while ,who
     (if mew-xemacs-p
	 (accept-process-output)
       (sit-for 0.1)
       ;; accept-process-output or sleep-for is not enough
       (discard-input))))

(defsubst mew-let-user-read ()
  (sit-for 1.5))

(defsubst mew-warn (&rest msg)
  (apply (function message) msg)
  (ding)
  (mew-let-user-read))

(defsubst mew-timing ()
  (sit-for 0.01))

;;; Copyright Notice:

;; Copyright (C) 1997-2001 Mew developing team.
;; All rights reserved.

;; Redistribution and use in source and binary forms, with or without
;; modification, are permitted provided that the following conditions
;; are met:
;; 
;; 1. Redistributions of source code must retain the above copyright
;;    notice, this list of conditions and the following disclaimer.
;; 2. Redistributions in binary form must reproduce the above copyright
;;    notice, this list of conditions and the following disclaimer in the
;;    documentation and/or other materials provided with the distribution.
;; 3. Neither the name of the team nor the names of its contributors
;;    may be used to endorse or promote products derived from this software
;;    without specific prior written permission.
;; 
;; THIS SOFTWARE IS PROVIDED BY THE TEAM AND CONTRIBUTORS ``AS IS'' AND
;; ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
;; IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
;; PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL THE TEAM OR CONTRIBUTORS BE
;; LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
;; CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
;; SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR
;; BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
;; WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE
;; OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN
;; IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

;;; mew-func.el ends here
