;;; tbl-mg.el --- Table Manager

;; Copyright (C) 1997-1999 Miyashita Hisashi

;; Keywords: mule, multilingual, table, CCL

;; This file is part of MULE-UCS

;; MULE-UCS is free software; you can redistribute it and/or modify
;; it under the terms of the GNU General Public License as published by
;; the Free Software Foundation; either version 2, or (at your option)
;; any later version.

;; MULE-UCS is distributed in the hope that it will be useful,
;; but WITHOUT ANY WARRANTY; without even the implied warranty of
;; MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
;; GNU General Public License for more details.

;; You should have received a copy of the GNU General Public License
;; along with this program; see the file COPYING.  If not, write to the
;; Free Software Foundation, Inc., 59 Temple Place - Suite 330,
;; Boston, MA 02111-1307, USA.

;;; This module manages tables for translation.
;;; This combines some tables to a table set that a unit of translation.

(require 'mucs)
(require 'trans-util)

(defun get-table-set-table-name (table-set number)
  "Return a name of a symbol used to register a ccl translation table,
which is managed by TABLE-SET."
  (format "%s-table-%d" (symbol-name table-set) number))

(defvar tbl-mg-temporary-table-postfix "tmp-table-"
  "Use this postfix to make a new symbol
for specifying a temporary table.")

(defvar tbl-mg-temporary-table-set-postfix "tmp-table-set-"
  "Use this postfix to make a new symbol
for specifying a temporary table set.")

(defvar default-max-codepoint-table-gap 256
  "Default max length of gap in a code point table.")

;;;
;;; Table package registration manager.
;;;

(defun mucs-table-registered-p (table)
  (memq table (get mucs-current-package
		   'mucs-registered-tables)))

(defun mucs-add-table-registration (table)
  (if (and mucs-current-package
	   (symbolp mucs-current-package))
      (or
       (mucs-table-registered-p table)
       (put mucs-current-package
	    'mucs-registered-tables
	    (cons table
		  (get mucs-current-package
		       'mucs-registered-tables))))))

(defun mucs-delete-table-registration (table)
  (if (and mucs-current-package
	   (symbolp mucs-current-package))
      (put mucs-current-package
	   'mucs-registered-tables
	   (delq
	    (get mucs-current-package
		 'mucs-registered-tables)))))
  
;;;
;;; Table set manager.
;;;
;; table-set is used for combining some tables that means certain unit
;; for translation.
;; the entity of table-set is a symbol that has the following keys.
;;     table-set-definition:
;;     table-set-symbol-tables:
;;     table-set-nested:
;;     table-set-reference-count:

(defvar table-set-internal-properties
  '(table-set-definition
    table-set-symbol-tables
    table-set-nested
    table-set-reference-count))

(defun define-table-set (table-set definition)
  "Define table-set.
Definition must be a (nested) list of tables."
  (if (table-set-p table-set)
      (clear-table-set table-set))
  (put table-set 'table-set-definition definition)
  (put table-set 'table-set-reference-count 1))

(defun table-set-nested-p (table-set)
  "Inspect whether this table-set is nested or not.
Nested table-set have at least one list that is consists of tables
or lists of tables."
  (let ((flag (get table-set 'table-set-nested))
	definition)
    (if flag
	(eq flag 'nested)
      (setq definition (get table-set 'table-set-definition))
      (while (if (consp (car definition))
		 nil
	       (setq definition (cdr definition))))
      (if definition
	  (progn
	    (put table-set 'table-set-nested 'nested)
	    t)
	(put table-set 'table-set-nested 'not-nested)
	nil))))

(defun clear-table-set (table-set)
  "Clear TABLE-SET.
You must call this to free memory occupied by table-set."
  (let ((symbols (get table-set 'table-set-symbol-tables))
	func)
    (setq func (lambda (syms)
		 (while syms
		   (cond ((symbolp (car syms))
			  (unintern (car syms)))
			 ((listp (car syms))
			  (funcall func (car syms))))
		   (setq syms (cdr syms)))))
    (funcall func symbols)
    (mapcar
     (lambda (x)
       (put table-set x nil))
     table-set-internal-properties)
    nil))

(defun table-set-add-reference (table-set)
  "Add reference count of table-set.
If you want to persist the table-set, call this."
  (put table-set 'table-set-reference-count
       (1+ (get table-set 'table-set-reference-count))))

(defun table-set-remove-reference (table-set)
  "Remove reference count of table-set.
If you don't need the table-set, call this."
  (put table-set 'table-set-reference-count
       (1- (get table-set 'table-set-reference-count)))
  (if (<= (get table-set 'table-set-reference-count) 0)
      (clear-table-set table-set)))

(defsubst table-set-p (table-set)
  (get table-set 'table-set-definition))

(defsubst get-table-set-definition (table-set)
  (get table-set 'table-set-definition))


(defun get-table-set-symbol-tables (table-set)
  "Retuen (nested) symbol list of the tables
used by the specified table-set."
  (and (table-set-p table-set)
       (or (get table-set 'table-set-symbol-tables)
	   (let ((i 1)
		 symbol-tables
		 table-sym
		 func)
	     (setq func (lambda (x result)
			  (while x
			    (if (listp (car x))
				(setq result (cons
					      (funcall func (car x) nil)
					      result))
			      (setq table-sym
				    (intern (get-table-set-table-name
					     table-set i)))
			      (register-code-conversion-map table-sym (car x))
			      (setq result (cons table-sym result)
				    i (1+ i)))
			    (setq x (cdr x)))
			  (nreverse result)))
	     (put table-set 'table-set-symbol-tables
		  (funcall func (get-table-set-definition table-set) nil))))))

(defun get-registration-required-tables (tables)
  "Return elisp program to register the table-set.
The generated elisp program is managed per package defined by
`mucs-current-package', thus this function does not generate
unnecessary duplicated elisp program."
  (let (table table-alist)
    (while (setq table (car tables))
      (if (listp table)
	  (get-registration-required-tables table)
	(if (mucs-table-registered-p table)
	    nil
	  (setq table-alist
		(cons
		 (cons table (get table 'code-conversion-map))
		 table-alist))
	  (mucs-add-table-registration table))
	(setq tables (cdr tables))))
    table-alist))

(defun generate-tables-registration-program (tables)
  (let ((table-alist (get-registration-required-tables
		      tables)))
    (if (null table-alist)
	nil
      `(let ((tbls (quote ,table-alist))
	     tbel)
	 (while (setq tbel (car tbls))
	   (register-code-conversion-map
	    (car tbel)
	    (cdr tbel))
	   (setq tbls (cdr tbls)))))))

(defsubst generate-table-set-registration-program (table-set)
  (and (table-set-p table-set)
       (generate-tables-registration-program
	(get-table-set-symbol-tables table-set))))

;;;
;;; Table creater
;;;

(defmacro define-ccl-codepoint-translation-table (symbol &rest args)
  `(let ((vector ,(apply 'make-codepoint-vector args)))
     (register-code-conversion-map ,symbol vector)
     vector))

(defmacro define-ccl-identity-translation-table (symbol start len)
  `(let ((vector ,(make-identity-code-conversion-vector start len)))
     (register-code-conversion-map ,symbol vector)
     vector))

(defmacro define-ccl-slide-translation-table (symbol start-s start-d len)
  `(let ((vector ,(make-slide-code-conversion-vector start-s start-d len)))
     (register-code-conversion-map ,symbol vector)
     vector))

(defmacro define-ccl-constant-translation-table (symbol start-s constant len)
  `(let ((vector ,(make-constant-code-conversion-vector start-s constant len)))
     (register-code-conversion-map ,symbol vector)
     vector))

(defun make-codepoint-vector (&rest args)
  "Return a vector of codepoints of given characters.
Each argument is a character or t or nil or lambda or string.
String must be an expression that is evaled into number."
  (let ((arg args) elem elem2
	table len vector)
    (while arg
      (setq elem (car arg))
      (cond ((numberp elem)
	     (setq table (cons (char-codepoint elem) table)))
	    ((or (eq elem t)
		 (eq elem 'lambda)
		 (null elem))
	     (setq table (cons elem table)))
	    ((stringp elem)
	     (setq elem2 (read elem))
	     (if (numberp elem2)
		 (setq table (cons elem2 table))
	       (error "Invalid argument %s" elem)))
	    (t
	     (error "Invalid argument %s" elem)))
      (setq arg (cdr arg)))
    (setq len (length table)
	  vector (make-vector len nil)
	  arg table)
    (while (> len 0)
      (setq len (1- len))
      (aset vector len (car arg))
      (setq arg (cdr arg)))
    vector))

(defun make-identity-code-conversion-vector (start len)
  (vector t t start (+ start len -1)))

(defun make-slide-code-conversion-vector (start-s start-d len)
  (setq len (1+ len))
  (let ((vector (make-vector len 0))
	(i 1))
    (aset vector 0 start-s)
    (while (< i len)
      (aset vector i start-d)
      (setq start-s (1+ start-s)
	    start-d (1+ start-d)
	    i (1+ i)))
    vector))

(defun make-constant-code-conversion-vector (start-s constant len)
  (vector t constant start-s (+ start-s len -1)))

(defsubst get-table-key (cell decodep)
  (if decodep (cdr cell) (car cell)))

(defsubst get-table-val (cell decodep)
  (if decodep (car cell) (cdr cell)))

(defun make-code-conversion-tables (alist conv &optional max)
  "Make code conversion tables.
When CONV is non-nil, convert all elements of alist with CONV."
  (if (null max) (setq max default-max-codepoint-table-gap))
  (let* ((alist-copy (sort
		      (cond ((null conv)
			     (copy-sequence alist))
			    ((eq conv 'decode)
			     (let* ((curalist alist)
				    elem result)
			       (while curalist
				 (setq elem (car curalist)
				       curalist (cdr curalist)
				       result (cons
					       (cons (cdr elem)
						     (car elem))
					       result)))
			       result))
			    ((functionp conv)
			     (mapcar conv alist))
			    (t
			     (error "Invalid CONV:%S" conv)))
		      (lambda (x y) (< (car x) (car y)))))
	 (curll alist-copy)
	 (stll alist-copy)
	 (ctll alist-copy)
	 (stp (car (car stll)))
	 (ctp stp)
	 stle
	 veclist
	 vec
	 curp
	 curle)
    (while ctll
      (setq curle (car curll)
	    curp (car curle))
      (if (and curll
	       (or
		(eq max t)
		(<= (- curp ctp) max)))
	  (setq ctp curp
		ctll curll)
	(setq vec (make-vector (- ctp stp -2) nil))
	(aset vec 0 stp)

	(setq stle (car stll))
	(while 
	    (prog2
		(aset vec 
		      (- (car stle) stp -1)
		      (if (eq (cdr stle) 'invalid)
			  mucs-invalid-code
			(cdr stle)))
		(not (eq stll ctll))
	      (setq stll (cdr stll)
		    stle (car stll))))

	(setq veclist (cons vec veclist)
	      ctll curll
	      stll curll
	      stp (car (car stll))
	      ctp stp))
      (setq curll (cdr curll)))
    (nreverse veclist)))

;;;
;;; Supplemental functions.
;;;    These functions may be removed in the future.
;;;

(defun compile-codepoint-alist-to-vector (table &optional cont func)
)

(defun merge-codepoint-vector (table11 table2)
)

(defun generate-meta-table-registration-program (symbol table)
  (register-code-conversion-map symbol table)
  (if (mucs-table-registered-p symbol)
      nil
    (mucs-add-table-registration symbol)
    `(let ((tbl ,table)
	   (i 1)
	   (j (length tbl))
	   elem id)
       (while (< i j)
	 (setq elem (aref tbl i))
	 (if (setq id (get 'code-conversion-map-id elem))
	     (aset tbl i id))
	 (setq i (1+ i)))
       (register-code-conversion-map ,symbol tbl))))

(provide 'tbl-mg)
