;;; rail-user-agent.el --- Replace Agent-string Internal Library

;; Copyright (C) 1999 by Free Software Foundation, Inc.

;; Author: SHIMADA Mitsunobu <simm-emacs@fan.gr.jp>
;; Keywords: User-Agent, MIME-Version, FLIM, SEMI, Rail

;; This file is free software; you can redistribute it and/or modify
;; it under the terms of the GNU General Public License as published by
;; the Free Software Foundation; either version 2, or (at your option)
;; any later version.

;; This file is distributed in the hope that it will be useful,
;; but WITHOUT ANY WARRANTY; without even the implied warranty of
;; MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
;; GNU General Public License for more details.

;; You should have received a copy of the GNU General Public License
;; along with GNU Emacs; see the file COPYING.  If not, write to
;; the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
;; Boston, MA 02111-1307, USA.

;;; Commentary:

;; Current version supports only MULE, FLIM, and SEMI.

;;; Code:

(require 'rail-vars)
(require 'rail-common)

;; replace codename
(defun rail-user-agent-replace-region (beg end pattern &optional pattern2)
  "Replace codename in region."
  (save-excursion
    (save-restriction
      (narrow-to-region beg end)
      (goto-char (point-min))
      (while (not (eobp))
	(skip-chars-forward " \t\r\n")
	(if (not (looking-at pattern))
	    (skip-chars-forward "^ \t\r\n")
	  (let* ((last (match-end 0))
		 (b1 (match-beginning 1))
		 (e1 (match-end 1))
		 (b2 (match-beginning 2))
		 (e2 (match-end 2))
		 (kind (rail-assoc (buffer-substring b1 e1) rail-product-name-alist t))
		 alist-a alist-b ccode)
	    (if (not kind)
		(goto-char last)
	      (setq ccode   (prin1-to-string kind)
		    alist-a (symbol-value
			     (intern (format "rail-additional-%s-codename-alist" ccode)))
		    alist-b (symbol-value
			     (intern (format "rail-%s-codename-alist" ccode))))
	      (setq ccode (rail-assoc (rail-replace-into-iso8859-4 (buffer-substring b2 e2))
				      (append alist-a alist-b)
				      rail-convert-direction))
	      (and pattern2
		   (eq 'xmas kind)
		   (eq nil ccode)
		   (looking-at pattern2)
		   (setq b2    (match-beginning 3)
			 e2    (match-end 3)
			 ccode (rail-assoc (rail-replace-into-iso8859-4
					    (buffer-substring b2 e2))
					   (append rail-additional-xmas-codename-alist
						   rail-xmas-codename-alist)
					   rail-convert-direction)))
	      (if (not ccode)
		  (goto-char last)
		(goto-char b2)
		(delete-region b2 e2)
		(insert ccode)
		(forward-char 1))))))
      (buffer-substring (point-min) (point-max)))))

;; search header
(defun rail-user-agent-search-header (head)
  "Search header string and return region."
  (save-excursion
    (goto-char (point-min))
    (and (re-search-forward (format "^%s:\\s +" head) nil t)
	 (let ((pt (point)))
	   (if (catch 'header-end
		 (while (not (eobp))
		   (forward-line 1)
		   (or (looking-at "\\s ")
		       (throw 'header-end t))))
	       (progn
		 (forward-char -1)
		 (cons pt (point))))))))

;; replace MIME-Version: header dinamically
(defun rail-user-agent-replace-mime-version-region (beg end)
  "Replace codename in MIME-Version: region."
  (rail-user-agent-replace-region beg end rail-mime-version-header-format))

;; replace User-Agent: header dinamically
(defun rail-user-agent-replace-user-agent-region (beg end)
  "Replace codename in User-Agent: region."
  (rail-user-agent-replace-region
   beg end rail-user-agent-header-format rail-user-agent-header-xmas-format))

;; replace header with rail-user-agent-replace-{mime-version|user-agent}-region
(defun rail-user-agent-translate-body (&optional buf)
  "Translate message body. It executes after mime-edit-translate-body."
  (and rail-user-agent-convert-dynamically
       (let (region)
	 (save-excursion
	   (and buf (set-buffer buf))
	   (setq region (rail-user-agent-search-header "MIME-Version"))
	   (rail-user-agent-replace-mime-version-region (car region) (cdr region))
	   (mapcar
	    '(lambda (item)
	       (and (setq region (rail-user-agent-search-header item))
		    (rail-user-agent-replace-user-agent-region (car region) (cdr region))))
	    rail-user-agent-header-item-list)))))

;; replace statically
(defun rail-user-agent-replace-string ()
  "Replace mime-edit-user-agent-value and mime-edit-mime-version-value."
  (and rail-user-agent-convert-statically
       (let (buf codename)
	 (save-excursion
	   (setq buf (get-buffer-create rail-temporary-buffer-name))
	   (if (set-buffer buf)
	       (progn
		 ;; mime-edit-mime-version-value
		 (erase-buffer)
		 (insert mime-edit-mime-version-value)
		 (setq mime-edit-mime-version-value
		       (rail-user-agent-replace-mime-version-region (point-min) (point-max)))
		 ;; mime-edit-user-agent-value
		 (erase-buffer)
		 (insert mime-edit-user-agent-value)
		 (setq mime-edit-user-agent-value
		       (rail-user-agent-replace-user-agent-region (point-min) (point-max)))
		 ;; end
		 (kill-buffer buf))))))
  ;; mime-library-product
  (and rail-user-agent-replace-mime-library-product
       (boundp 'mime-library-product)
       (setq codename (mime-product-code-name mime-library-product))
       (aset mime-library-product 2
	     (or (rail-assoc codename
			     (append rail-additional-flim-codename-alist rail-flim-codename-alist)
			     rail-convert-direction)
		 codename)))
  ;; mime-user-interface-product
  (and rail-user-agent-replace-mime-user-interface-product
       (boundp 'mime-user-interface-product)
       (setq codename (mime-product-code-name mime-user-interface-product))
       (aset mime-user-interface-product 2
	     (or (rail-assoc codename
			     (append rail-additional-semi-codename-alist rail-semi-codename-alist)
			     rail-convert-direction)
		 codename))))

;; add mime-edit-translate-buffer-hook
(defun rail-user-agent-add-hook ()
  "Add hook for rail-user-agent"
  (let ((list (memq 'mime-edit-translate-body mime-edit-translate-buffer-hook)))
    (or (memq 'rail-user-agent-translate-body mime-edit-translate-buffer-hook)
	(setcdr list (cons 'rail-user-agent-translate-body (cdr list))))))

;; add mime-edit-load-hook
(if (not (featurep 'mime-edit))
    (add-hook 'mime-edit-load-hook
	      '(lambda ()
		 (rail-user-agent-add-hook)
		 (rail-user-agent-replace-string)))
  (rail-user-agent-add-hook)
  (rail-user-agent-replace-string))

(provide 'rail-user-agent)

;;; rail-user-agent.el ends here
