/*                                                                         */
/*             OS/2 Font Driver using FreeType library                     */
/*     Copyright (C) 1997, 1998 by Michal Necasek <mike@mendelu.cz>        */
/*                                                                         */
/*       Version: 0.9 (Beta)                                               */
/*                                                                         */
/* This source is to be compiled with IBM VisualAge C++ 3.0 or alternately */
/* Watcom C/C++ 10.0 or higher (Wactom doesn't quite work yet).            */
/* Other compilers may actually work too but don't forget this is NOT a    */
/* normal DLL but rather a subsystem DLL. That means it shouldn't use      */
/* the usual C library as it has to run without runtime environment.       */
/* VisualAge provides a special subsystem version of the run-time library. */
/* All this is of course very compiler-dependent. See makefiles for        */
/* discussion of switches used.                                            */
/*                                                                         */
/*  Implemantation Notes:                                                  */
/* Note #1: Some local variables are declared static and you might wonder  */
/*   why. The reason is to conserve stack space. It might not be really    */
/*   necessary, but as they say, it's better to be safe than sorry.        */
/*   Even so, stack usage is roughly 100-200 bytes, perhaps more.          */
/*   I really hope I won't have to perform any stack-switching magic :)    */
/*                                                                         */
/* Note #2: As a consequence of this being a subsystem librarary, I had to */
/*   slightly modify the FreeType source, namely ttmemory.c and ttfile.c.  */
/*   FreeType/2 now allocates several chunks of memory and uses them as a  */
/*   heap. Note that memory allocation should use TTAlloc(), possibly      */
/*   directly SSAllocMem(). malloc() is unusable here and it doesn't work  */
/*   at all (runtime library isn't even initialized). See ttmemory.c for   */
/*   more info.                                                            */
/*    In ttfile.c I had to change all fopen(), fseek()... calls            */
/*   to OS/2 API calls (DosOpen, DosSetFilePtr...) because without proper  */
/*   runtime environment a subsystem DLL cannot use std C library calls.   */
/*                                                                         */
/* Note #3: On exit of each function reading from font file the API        */
/*   TT_Flush_Stream() must be called. This is because file handles opened */
/*   by this DLL actually belong to the calling process. As a consequence  */
/*    a) it's easy to run out of file handles, which results in really     */
/*       very nasty behavior and/or crashes. This could be solved by       */
/*       increased file handles limit, but cannot because                  */
/*    b) it is impossible to close files open by another process and       */
/*       therefore the fonts cannot be properly uninstalled (you can't     */
/*       delete them while the're open by other process)                   */
/*   The only solution I found is very simple - just close the file before */
/*   exiting a DLL function. This ensures files are not left open across   */
/*   processes and other problems.                                         */
/*                                                                         */
/* For Intelligent Font Interface (IFI) specification please see IFI32.TXT */
/* It is available free of charge from IBM DDK.                            */

#ifndef  __IBMC__
   #ifndef __WATCOMC__
      #error "This source requires IBM VisualAge C++ or Watcom C/C++"
   #endif
#endif

#define INCL_DEV
#define INCL_DDIDEFS
#define INCL_GRE_ALL
#include <os2.h>
#include <pmddi.h>          /* SSAllocmem(), SSFreemem() and more */

#include <string.h>
#include <stdlib.h>         /* min and max macros */
#include "32pmifi.h"        /* IFI header         */
#include "freetype.h"       /* FreeType header    */
#include "ftxkern.h"        /* kerning extension  */
#include "ftifi.h"          /* xlate table        */

/* (indirectly) exported functions */
LONG _System ConvertFontFile(PSZ pszSrc, PSZ pszDestDir, PSZ pszNewName, PVOID pfn);
HFF  _System LoadFontFile(PSZ pszFileName, PVOID pfn);
LONG _System UnloadFontFile(HFF hff, PVOID pfn);
LONG _System QueryFaces(HFF hff, PIFIMETRICS pifiMetrics, ULONG cMetricLen,
                        ULONG cFountCount, ULONG cStart, PVOID pfn);
HFC  _System OpenFontContext(HFF hff, ULONG ulFont, PVOID pfn);
LONG _System SetFontContext(HFC hfc, PCONTEXTINFO pci, PVOID pfn);
LONG _System CloseFontContext(HFC hfc, PVOID pfn);
LONG _System QueryFaceAttr(HFC hfc, ULONG iQuery, PBYTE pBuffer,
                           ULONG cb, PGLYPH pagi, GLYPH giStart, PVOID pfn);
LONG _System QueryCharAttr(HFC hfc, PCHARATTR pCharAttr,
                           PBITMAPMETRICS pbmm, PVOID pfn);
LONG _System QueryFullFaces(HFF hff, PVOID pBuff, PULONG buflen,
                            PULONG cFontCount, ULONG cStart, PVOID pfn);
LONG _System ClaimFontFile(PSZ pszFileName, PVOID pfn);

FDDISPATCH fdisp = {        /* Font driver dispatch table */
   LoadFontFile,
   QueryFaces,
   UnloadFontFile,
   OpenFontContext,
   SetFontContext,
   CloseFontContext,
   QueryFaceAttr,
   QueryCharAttr,
   ClaimFontFile,      /* the spec is VERY quiet on this one */
   ConvertFontFile,
   QueryFullFaces
};

/* the single exported entry point; this way is faster than exporting every */
/* single function */
#pragma export (fdhdr, "FONT_DRIVER_DISPATCH_TABLE", 1)
FDHEADER   fdhdr = {        /* Font driver Header */
   sizeof(FDHEADER),
   "OS/2 FONT DRIVER",      /* do not change */
   "TrueType (Using FreeType Engine)", /* desc. up to 40 chars */
   20,       /* version */
   0,        /* reserved */
   &fdisp
};


/* some debug macros & funcs */
/* the debug version logs system requests to a file */
#ifdef DEBUG
  HFILE LogHandle = NULLHANDLE;
  ULONG Written   = 0;
  char  log[2048] = "";
  char  buf[2048] = "";


char*  itoa10( int i, char* buffer ) {
    char*  ptr  = buffer;
    char*  rptr = buffer;
    char   digit;

    if (i == 0) {
      buffer[0] = '0';
      buffer[1] =  0;
      return buffer;
    }

    if (i < 0) {
      *ptr = '-';
       ptr++; rptr++;
       i   = -i;
    }

    while (i != 0) {
      *ptr = (char) (i % 10 + '0');
       ptr++;
       i  /= 10;
    }

    *ptr = 0;  ptr--;

    while (ptr > rptr) {
      digit = *ptr;
      *ptr  = *rptr;
      *rptr = digit;
       ptr--;
       rptr++;
    }

    return buffer;
}

  #define  COPY(s)     strcpy(log, s)
  #define  CAT(s)      strcat(log, s)
  #define  CATI(v)     strcat(log, itoa10( (int)v, buf ))
  #define  WRITE       DosWrite(LogHandle, log, strlen(log), &Written)
  #define  ERRRET(e)   { COPY("Error at ");  \
                          CATI(__LINE__);    \
                          CAT("\r\n");       \
                          WRITE;             \
                          return(e);         \
                       }


#else

  #define  COPY(s)
  #define  CAT(s)
  #define  CATI(v)
  #define  WRITE
  #define  ERRRET(e)  return(e);

#endif /* DEBUG */

/* FreeType engine instance; although this is a DLL, it isn't supposed to be actually */
/* shared by apps - it is only called by the OS/2 GRE. This means there's no need to  */
/* bother with reentrancy/thread safety. At least the IFI spec states it and it seems */
/* to work, too. */
TT_Engine engine;

/* structure containing data relevant to an open font file handle */
/* it is kept in a simple singly-linked list */

typedef struct _FFHANDLE {
   struct _FFHANDLE  *next;    /* ptr to next entry in linked list */
   HFF        hff;             /* HFF used from outside */
   CHAR       achName[260];    /* font file name */
   TT_Face    face;            /* handle to font face used by FreeType */
   TT_CharMap charMap;         /* handle to Unicode map for this font */
   TT_Kerning directory;       /* kerning directory */
   LONG       cOpen;           /* number of times this font file is open */
   LONG       upem;            /* points per em square */
   BOOL       scaling;         /* TRUE = scaling/rotation used; FALSE otherwise */
   LONG       transMode;       /* character translation mode :  */
                               /*   0 = Unicode                 */
                               /*   1 = Symbol (no translation) */
} FFHANDLE, *PFFHANDLE;

/* structure defining data relevant to an open font context */
typedef struct _FCHANDLE {
   HFC          hfc;           /* HFC used from outside */
   TT_Instance  instance;      /* handle to face instance used by FreeType */
   TT_Glyph     glyph;         /* handle to glyph container used by FreeType */
   BOOL         scaling;       /* TRUE = scaling/rotation used (rare) */
                               /* FALSE otherwise (normal) */
   TT_Matrix    matrix;        /* transformation matrix */
   PFFHANDLE    pffh;          /* HFF this context belongs to */
} FCHANDLE, *PFCHANDLE;

PFFHANDLE startffh = NULL;      /* head of linked list */

/* return pointer to FFHANDLE structure or NULL if invalid handle */
PFFHANDLE getFFH(HFF hff) {
   PFFHANDLE pffh = startffh;

   while (pffh) {
      if (pffh->hff == hff)
         break;
      pffh = pffh->next;
   }

   return pffh;     /* return pointer to structure (or NULL)*/
}

#define MAXHFC  5
/* array of font context handle structures */
/* Note that there isn't more than one font context open at any time anyway */
/* (but I want to be safe) */
FCHANDLE aFCH[MAXHFC]; /* this is rather too much */

/* return pointer to FCHANDLE structure or NULL if invalid handle */
PFCHANDLE getFCH(HFC hfc) {
   int   i = 0;

   while ((aFCH[i].hfc != hfc) && (i < MAXHFC))
      i++;
   if (i == MAXHFC)
      return NULL;     /* invalid handle */

   return &(aFCH[i]);     /* return pointer to structure */
}

/* a function to convert PM codepoint to TT glyph index */
/* mode = 0 - translate from UGL to Unicode */
/* mode = 1 - no translation (used for symbol fonts) */
int PM2TT(TT_CharMap charMap, ULONG mode, int index) {
   /* convert from PM383 to Unicode */
   if (index > 504) /* no idea if this is correct */
      return 0;
   switch (mode) {
      case 0:
         return TT_Char_Index(charMap, UGL2Uni[index]);
      case 1:
         return TT_Char_Index(charMap, index);
      default:
         return 0;
   }
}

/* A simple function for comparing strings without case sensitivity. Just */
/* returns zero if strings match, one otherwise. I wrote this because     */
/* stricmp is not available in the subsystem run-time library (probably   */
/* because it uses locales). toupper() is unfortunately unavailable too.  */

#define toupper( c ) ( ((c) >= 'a') && ((c) <= 'z') ? (c) - 'a' + 'A' : (c) )

int mystricmp(const char *s1, const char *s2) {
   int i = 0;
   int match = 0;

   if (strlen(s1) != strlen(s2))
      return 1;   /* no match */

   while (i < strlen(s1)) {
      if (toupper(s1[i]) != toupper(s2[i])) {
         match = 1;
         break;
      }
      i++;
   }

   return match;
}

/* perform translation on a point */
void Xform(POINTFX *pt, TT_Matrix *mat) {
   FIXED x = pt->x, y = pt->y;

   pt->x = mat->xx * x + mat->xy * y;
   pt->y = mat->yx * x + mat->yy * y;
}

/* this function tries to find M$ English name for a face */
/* length is limited to FACESIZE (defined by OS/2); returns NULL if unsuccessful */
/* warning: the string gets overwritten on the next invocation */
char*  LookupName(TT_Face face,  int index );

/* get suitable charmap from font */
ULONG GetCharmap(TT_Face face);

/* get # of bytes needed for glyph outline */
int GetOutlineLen(TT_Outline *ol);

/* get glyph outline in PM format */
int GetOutline(TT_Outline *ol, PBYTE pb);

/* The following two functions are declared here because including */
/* the entire ttmemory.h creates more problems than it solves */
TT_Error  TT_Alloc( long  Size, void**  P );
TT_Error  TT_Free( void**  P );


TT_Error error = 0;

/* install/delete font file */
LONG _System ConvertFontFile(PSZ pszSrc, PSZ pszDestDir, PSZ pszNewName, PVOID pfn) {
   PSZ  pszSrcName;

   COPY("ConvertFontFile: Src = ");
   CAT(pszSrc);
   if (pszDestDir) {
      CAT(", DestDir = ");
      CAT(pszDestDir);
   }
   CAT("\r\n");
   WRITE;

   if (pszDestDir && pszNewName) {  /* install font file */
      pszSrcName = strrchr(pszSrc, '\\');  /* find the last backslash */
      if (pszSrcName == NULL)
         ERRRET(-1)   /* something went wrong */
      pszSrcName++;   /* skip the backslash */
      strcpy(pszNewName, pszSrcName);
      if (DosCopy(pszSrc, pszDestDir, DCPY_EXISTING)) /* overwrite file */
         ERRRET(-1)   /* !!!! just return error */
                      /* should handle errors better */
      COPY(" -> Name: ");
      CAT(pszNewName);
      CAT("\r\n");
      WRITE;
      return 0;       /* OK */
   }
   else
      COPY("Delete file ");
      CAT(pszSrc);
      CAT("\r\n");
      WRITE;
      if (DosDelete(pszSrc))
         ERRRET(-1) /* couldn't delete file */
      return 0;     /* uninstall (i.e. deletion) OK */

}

/* open font file, return handle to it */
HFF _System LoadFontFile(PSZ pszFileName, PVOID pfn) {
   TT_Face     face;
   PSZ         pszExt;
   ULONG       encoding;
   PFFHANDLE   pffh, tempffh;
   TT_CharMap  cmap;

   COPY( "LoadFontFile " );
   CAT( pszFileName );
   CAT( "\r\n" );
   WRITE;

   /* first check if the file extension is supported */
   pszExt = strrchr(pszFileName, '.');  /* find the last dot */
   if (pszExt == NULL)
      return((HFF)-1);             /* this check is important! */
   if (mystricmp(pszExt, ".TTF"))  /* must be case insensitive! */
      return((HFF)-1);             /* unsupported file extension */

   /* then check if already open */
   pffh = startffh;
   while (pffh) {
      if (strcmp(pffh->achName, pszFileName) == 0) { /* does the filename match ? */
         break;
      }
      pffh = pffh->next;
   }
   if (pffh) {            /* yes it's already open */
      pffh->cOpen++;      /* increment open counter */

      COPY( " -> (duplicate) hff = " );
      CATI( pffh->hff );
      CAT( "\r\n" );
      WRITE;

      return pffh->hff;   /* no sense going on */
   }

   /* OK, this file isn't open yet, so give it a shot */
   error = TT_Open_Face(engine, pszFileName, &face);
   if (error) {
      COPY( "Error while opening " );
      CAT( pszFileName );
      CAT( ", error code = " );
      CATI( error );
      CAT( "\r\n" );
      WRITE;
      return (HFF)-1; /* error, can't open file */
                      /* !!!! should set error condition here! */
   }
   /*  now get Unicode/Apple Roman charmap for this font */
   encoding = GetCharmap(face);
   error = TT_Get_CharMap(face, encoding & 0xFFFF, &cmap);
   if (error) {
      COPY( "Error: No char map in " );
      CAT(  pszFileName );
      CAT(  "\r\n" );
      WRITE;
      TT_Flush_Face(face);
      ERRRET((HFF)-1) /* error, can't get charmap file */
   }

   /* find last entry */
   pffh = startffh;
   while (pffh)
      pffh = pffh->next;

   /* allocate new entry; don't even think of using malloc() */
   TT_Alloc(sizeof(FFHANDLE), (void**)&pffh);
   if (pffh == NULL) {
      TT_Flush_Face(face);
      ERRRET((HFF)-1);  /* couldn't allocate memory */
   }
   /* insert into list */
   if (startffh == NULL)
      startffh = pffh;  /* this is the only open font */
   else {
      tempffh = startffh;
      while(tempffh->next)
         tempffh = tempffh->next;
      tempffh->next = pffh;
   }
   pffh->cOpen     = 1;       /* initialize table entries */
   pffh->next      = NULL;
   pffh->face      = face;
   pffh->charMap   = cmap;
   pffh->transMode = encoding >> 16; /* zero means Unicode, one is no translation */
   error = TT_Get_Kerning_Directory(face, &(pffh->directory)); /* load kern dir */
   if (error)
      pffh->directory.nTables = 0; /* indicates no kerning in font */
   strcpy(pffh->achName, pszFileName);
   pffh->hff = (HFF)(pffh);  /* zero is invalid */

   COPY( "opened " );
   CAT(  pszFileName );
   CAT(  " successfully, hff = " );
   CATI( pffh->hff );
   CAT(  "\r\n" );
   WRITE;

   TT_Flush_Face(face);     /* this is important !*/
   return pffh->hff; /* everything is in order, return valid HFF */
}

/* destroy resurces associated with this HFF */
LONG _System UnloadFontFile(HFF hff, PVOID pfn) {
   PFFHANDLE   pFFH, tempffh;

   COPY("UnloadFontFile: hff = ");
   CATI((int) hff);
   CAT("\r\n");
   WRITE;

   pFFH = getFFH(hff);
   if (!pFFH)
      ERRRET(-1) /* error, invalid handle */

   pFFH->cOpen--;        /* decrement use counter */
   if (pFFH->cOpen > 0)  /* counter still not zero */
      return 0;          /* don't really close, return OK */

   error = TT_Close_Face(pFFH->face);  /* actually close face */
   if (error)
      ERRRET(-1)     /* something went wrong */

   /* remove entry from list */
   tempffh = pFFH;
   if (pFFH == startffh) {
      startffh = startffh->next; /* it's the first entry */
   }
   else {
      pFFH = startffh;
      while (pFFH->next != tempffh)
         pFFH = pFFH->next;
      pFFH->next = pFFH->next->next;  /* remove from list */
   }
   TT_Free((void**)&tempffh);

   return 0;         /* we're finished with this file */

}

/* return font metrics */
/* this routine has to do a LOT of (not very hard) work */
LONG _System QueryFaces(HFF hff, PIFIMETRICS pim, ULONG cMetricLen,
                ULONG cFontCount, ULONG cStart, PVOID pfn) {
   TT_Face_Properties   properties;
   TT_Header            *phead;
   TT_Horizontal_Header *phhea;
   TT_OS2               *pOS2;
   TT_Postscript        *ppost;
   PIFIMETRICS          xpim;   /* temporary structure */
   PFFHANDLE            pffh;

   COPY( "QueryFaces: hff = " ); CATI( hff );
   CAT(  ", cFontCount = " );    CATI( cFontCount );
   CAT(  ", cStart = " );        CATI( cStart );
   CAT(  ", cMetricLen = " );    CATI( cMetricLen );
   CAT( "\r\n");
   WRITE;

   pffh = getFFH(hff);
   if (!pffh)
      ERRRET(-1) /* error, invalid handle */

   if (cMetricLen == 0)    /* only number of faces is requested */
      return 1;            /* return 1 */

   error = TT_Get_Face_Properties(pffh->face, &properties);
   if (error) {
      TT_Flush_Face(pffh->face);
      ERRRET(-1); /* engine error */
   }
   pOS2  = properties.os2;
   phead = properties.header;
   phhea = properties.horizontal;
   ppost = properties.postscript;

   /* ??? maybe we should check for NULL strings here */
   if (cMetricLen == 64) {   /* only names are requested */
      strncpy(pim->szFamilyname,
              LookupName(pffh->face, 1), FACESIZE);    /* family name */
      strncpy(pim->szFacename,
              LookupName(pffh->face, 4), FACESIZE);    /* face name */
      TT_Flush_Face(pffh->face);
      return 1; /* more if TTCs are supported */
   }

   TT_Alloc(sizeof(IFIMETRICS), (void**)&xpim); /* allocate temp structure */
   if (xpim == NULL) {
      TT_Flush_Face(pffh->face);
      ERRRET(-1);   /* allocation failed */
   }
   strncpy(xpim->szFamilyname,
           LookupName(pffh->face, 1), FACESIZE);    /* family name */
   strncpy(xpim->szFacename,
           LookupName(pffh->face, 4), FACESIZE);    /* face name */
   strcpy(xpim->szGlyphlistName, "PM383"); /* !!!! should be different for symbol fonts */
   xpim->idRegistry         = 0;
   xpim->lCapEmHeight       = phead->Units_Per_EM; /* ??? probably correct */
   xpim->lXHeight           = phead->yMax /2; /* ??? IBM TRUETYPE.DLL seems to do this */
   xpim->lMaxAscender       = pOS2->usWinAscent;
   xpim->lMaxDescender      = pOS2->usWinDescent;
   xpim->lLowerCaseAscent   = pOS2->usWinAscent;
   xpim->lLowerCaseDescent  = pOS2->usWinDescent;
   xpim->lInternalLeading   = pOS2->usWinAscent + pOS2->usWinDescent - xpim->lCapEmHeight;
   xpim->lExternalLeading   = 0; /* seems OK, could put something here though */
   xpim->lAveCharWidth      = pOS2->xAvgCharWidth;
   xpim->lMaxCharInc        = phhea->advance_Width_Max;
   xpim->lEmInc             = phead->Units_Per_EM;
   xpim->lMaxBaselineExt    = pOS2->usWinAscent + pOS2->usWinDescent;
   xpim->fxCharSlope        = -ppost->italicAngle; /* is this correct ?*/
   xpim->fxInlineDir        = 0;  /* ??? seems normal for most typefaces */
   xpim->fxCharRot          = 0;  /* ??? seems normal for most typefaces */
   xpim->usWeightClass      = pOS2->usWeightClass; /* hopefully OK */
   xpim->usWidthClass       = pOS2->usWidthClass;
   xpim->lEmSquareSizeX     = phead->Units_Per_EM;
   xpim->lEmSquareSizeY     = phead->Units_Per_EM; /* probably correct */
   xpim->giFirstChar        = 0;     /* following values should work */
   xpim->giLastChar         = 503;   /* either 383 or 503 */
   xpim->giDefaultChar      = 0;
   xpim->giBreakChar        = 32;
   xpim->usNominalPointSize = 120;   /* these are simply constants */
   xpim->usMinimumPointSize = 10;
   xpim->usMaximumPointSize = 10000; /* limit to 1000 pt (like the ATM fonts) */
   xpim->fsType             = pOS2->fsType & IFIMETRICS_LICENSED; /* perhaps correct */
   xpim->fsDefn             = IFIMETRICS_OUTLINE;  /* this is always the case */
   xpim->fsSelection        = pOS2->fsSelection;
   xpim->fsCapabilities     = 0; /* must be zero according to the IFI spec */
   xpim->lSubscriptXSize    = pOS2->ySubscriptXSize;
   xpim->lSubscriptYSize    = pOS2->ySubscriptYSize;
   xpim->lSubscriptXOffset  = pOS2->ySubscriptXOffset;
   xpim->lSubscriptYOffset  = pOS2->ySubscriptYOffset;
   xpim->lSuperscriptXSize  = pOS2->ySuperscriptXSize;
   xpim->lSuperscriptYSize  = pOS2->ySuperscriptYSize;
   xpim->lSuperscriptXOffset= pOS2->ySuperscriptXOffset;
   xpim->lSuperscriptYOffset= pOS2->ySuperscriptYOffset;
   xpim->lUnderscoreSize    = ppost->underlineThickness; /* IBM's TT does this */
   xpim->lUnderscorePosition= -ppost->underlinePosition;
   xpim->lStrikeoutSize     = pOS2->yStrikeoutSize;
   xpim->lStrikeoutPosition = pOS2->yStrikeoutPosition;
   xpim->cKerningPairs      = 100;   /* !!!! not supported yet! */
                             /* just returning non-zero to see if anyone asks */
                             /* for kerning data */
   xpim->ulFontClass        = pOS2->sFamilyClass;

   /* the following adjustment are needed because the TT spec defines */
   /* usWeightClass and fsType differently */
   if (xpim->usWeightClass >= 100)
      xpim->usWeightClass /= 100;
   if (ppost->isFixedPitch)
      xpim->fsType |= IFIMETRICS_FIXED;

   xpim->fsType |= IFIMETRICS_KERNING; /* !!! for testing only! */

   pffh->upem = phead->Units_Per_EM; /* we'll need this later */

   /* copy to output buffer then deallocate temporary storage */
   memcpy(pim, xpim, cMetricLen);
   TT_Free((void**)&xpim);

   TT_Flush_Face(pffh->face);
   return 1;     /* one metric returned, it can't be more anyway; */
                 /* will change if TTC support is added */
}

/* open new font context */
HFC _System OpenFontContext(HFF hff, ULONG ulFont, PVOID pfn) {
   int                  i = 0;
   TT_Instance          instance;
   PFFHANDLE            pffh;

   COPY("OpenFontContext: hff = ");
   CATI((int) hff);
   CAT("\r\n");
   WRITE;

   pffh = getFFH(hff);
   if (!pffh)
      ERRRET((HFC)-1) /* error, invalid font handle */

   /* OK, create new instance with defaults */
   error = TT_New_Instance(pffh->face, &instance);
   if (error)
      ERRRET((HFC)-1) /* error, engine failure */

   error = TT_Set_Instance_Resolutions(instance, 96, 96);
   if (error)
      ERRRET((HFC)-1) /* error, engine failure */
   error = TT_Set_Instance_CharSize(instance, 10 * 64);
   if (error)
      ERRRET((HFC)-1) /* error, engine failure */

   /* find first unused index */
   i = 0;
   while ((aFCH[i].hfc != 0) && (i < MAXHFC))
      i++;

   if (i == MAXHFC)
      ERRRET((HFC)-1) /* error, no free slot in table */

   error = TT_New_Glyph(pffh->face, &(aFCH[i].glyph));
   if (error)
      ERRRET((HFC)-1) /* error, engine failure */

   aFCH[i].hfc = (HFC)(i + 0x100);    /* initialize table entries; zero = empty! */
   aFCH[i].instance = instance;
   aFCH[i].scaling  = FALSE;  /* no scaling/rotation assumed */
   aFCH[i].pffh = pffh;

   COPY("-> hfc ");
   CATI((int) aFCH[i].hfc);
   CAT("\r\n");
   WRITE;

   TT_Flush_Face(pffh->face);
   return aFCH[i].hfc; /* everything OK */
}

/* set font context parameters */
LONG _System SetFontContext(HFC hfc, PCONTEXTINFO pci, PVOID pfn) {
   ULONG                ptsize, temp;
   PFCHANDLE            pfch;

   COPY("SetFontContext: hfc = ");
   CATI((int) hfc);
   CAT(", sizlPPM.cx = ");
   CATI((int) pci->sizlPPM.cx);
   CAT(", sizlPPM.cy = ");
   CATI((int) pci->sizlPPM.cy);
   CAT("\r\n                pfxSpot.x = ");
   CATI((int) pci->pfxSpot.x);
   CAT(", pfxSpot.y = ");
   CATI((int) pci->pfxSpot.y);
   CAT("\r\n                eM11 = ");
   CATI((int) pci->matXform.eM11);
   CAT(", eM12 = ");
   CATI((int) pci->matXform.eM12);
   CAT(", eM21 = ");
   CATI((int) pci->matXform.eM21);
   CAT(", eM22 = ");
   CATI((int) pci->matXform.eM22);
   CAT("\r\n");
   WRITE;

   pfch = getFCH(hfc);
   if (!pfch)
      ERRRET(-1) /* error, invalid context handle */

   /* set target dpi */
   /* The 254 / 10000  is used for conversion from dots per meter to dpi. */
   error = TT_Set_Instance_Resolutions(pfch->instance,
                                      pci->sizlPPM.cx * 254 / 10000,
                                      pci->sizlPPM.cy * 254 / 10000);
   if (error)
      ERRRET(-1)  /* engine problem */

   /* set scaling/rotation mode */
   /* I hope it's correct */
   if ((pci->matXform.eM11 == pci->matXform.eM22) &&       /* M11 == M22 */
       ((pci->matXform.eM12 | pci->matXform.eM21) == 0) && /* M12 == M21 == 0 */
       (pci->matXform.eM11 > 0))                           /* M11 positive; */
      pfch->scaling = FALSE;                               /* important */
   else {
      pfch->scaling = TRUE;
      /* note that eM21 and eM12 are swapped; I have no idea why, but */
      /* it seems to be correct */
      pfch->matrix.xx = pci->matXform.eM11 * 64;
      pfch->matrix.xy = pci->matXform.eM21 * 64;
      pfch->matrix.yx = pci->matXform.eM12 * 64;
      pfch->matrix.yy = pci->matXform.eM22 * 64;
      return 0;
   }

   /* calculate & set  point size */
   /* this only works for no scaling/rotation. But that's what is */
   /* used about 99% of the time (maybe more). */
   /* If rotation/scaling is used, this is disregarded. */
   ptsize = 72 * pfch->pffh->upem;
   ptsize >>= 6; /* necessary to prevent overflow */
   temp = ptsize * pci->matXform.eM11;
   ptsize = temp / pci->sizlPPM.cx;
   ptsize = ptsize * 10000 / 254;  /* convert from ppm to dpi */
   ptsize >>= 4;     /* final conversion to 26.6 format */
   if (ptsize == 0)  /* must not allow zero point size ! */
      ptsize = 1;
   error = TT_Set_Instance_CharSize(pfch->instance,
                                     ptsize);
   if (error)
      ERRRET(-1)  /* engine problem */

   return 0;      /* pretend everything is OK */
}

/* destroy font context */
LONG _System CloseFontContext(HFC hfc, PVOID pfn) {
   PFCHANDLE            pfch;

   COPY("CloseFontContext: hfc = ");
   CATI((int)hfc);
   CAT("\r\n");
   WRITE;

   pfch = getFCH(hfc);
   if (!pfch)
      ERRRET(-1) /* error, invalid context handle */

   error = TT_Done_Instance(pfch->instance);
   if (error)
      ERRRET(-1)  /* engine error */

   error = TT_Done_Glyph(pfch->glyph);
   if (error)
      ERRRET(-1)  /* engine error */

   /* mark table entry as free */
   pfch->hfc = 0;
   return 0; /* success */
}

/* return various info about font face */
LONG _System QueryFaceAttr(HFC hfc, ULONG iQuery, PBYTE pBuffer,
                   ULONG cb, PGLYPH pagi, GLYPH giStart, PVOID pfn) {
   int                  count, i = 0;
   TT_Glyph_Metrics     metrics;
   PFCHANDLE            pfch;

   COPY("QueryFaceAttr: hfc = ");
   CATI((int) hfc);
   CAT("\r\n");
   WRITE;

   pfch = getFCH(hfc);
   if (!pfch)
      ERRRET(-1) /* error, invalid context handle */
   if (iQuery == FD_QUERY_KERNINGPAIRS)
      /* !!!! no support for kerning yet */
      ERRRET(-1)

   if (iQuery == FD_QUERY_ABC_WIDTHS) {
      count = cb / sizeof(ABC_TRIPLETS);
      COPY("QUERY_ABC_WIDTHS, ");
      CATI((int) count);
      CAT(" items, giStart = ");
      CATI((int) giStart);
      if (pBuffer == NULL) {
         CAT(" NULL buffer");
      }
      CAT("\r\n");
      WRITE;
      /* return 'advance widths' or 'ABC widths' for a range of glyphs */
      for (i = giStart; i < giStart + count; i++) {
         error = TT_Load_Glyph(pfch->instance, pfch->glyph, /* no scaling and/or hinting */
                    PM2TT(pfch->pffh->charMap, pfch->pffh->transMode, i),
                    0);
         if (error) {
            TT_Flush_Face(pfch->pffh->face);
            ERRRET(-1);
         }
         error = TT_Get_Glyph_Metrics(pfch->glyph, &metrics);
         if (error) {
            TT_Flush_Face(pfch->pffh->face);
            ERRRET(-1);
         }
         ((ABC_TRIPLETS*)pBuffer)[i - giStart].lA  = metrics.bearingX;
/* Although I believe my calculation was correct, Netscape/2 didn't quite   */
/* work. Therefore I will return the same values as IBM's TRUETYPE.DLL.     */
/* Well, another little mystery. */
/*       ((ABC_TRIPLETS*)pBuffer)[i - giStart].ulB = metrics.xMax - metrics.xMin;
         ((ABC_TRIPLETS*)pBuffer)[i - giStart].lC  = metrics.advance
                                                     - (metrics.bearingX
                                                     + metrics.xMax - metrics.xMin);*/
         ((ABC_TRIPLETS*)pBuffer)[i - giStart].ulB = metrics.advance -
                                                     metrics.bearingX;
         ((ABC_TRIPLETS*)pBuffer)[i - giStart].lC = 0;
      }
   }
   TT_Flush_Face(pfch->pffh->face);
   return count; /* number of entries filled in */
}

/* return glyph attributes, basically glyph's bit-map or outline */
/* some variables are declared static to conserve stack space */
LONG _System QueryCharAttr(HFC hfc, PCHARATTR pCharAttr, PBITMAPMETRICS pbmm, PVOID pfn) {
   PFCHANDLE            pfch;
static   TT_Glyph_Metrics     metrics;
static   TT_Raster_Map        bitmap;
static   TT_Outline     outline;
static   POINTFX              center, minp, maxp, pt1, pt2;
   LONG                 temp;
   PBYTE                pb;
   int                  i, j;
   ULONG                cb;

   pfch = getFCH(hfc);
   if (!pfch)
      ERRRET(-1) /* error, invalid context handle */

   error = TT_Load_Glyph(pfch->instance, pfch->glyph,
              PM2TT(pfch->pffh->charMap, pfch->pffh->transMode, pCharAttr->gi),
              pfch->scaling ? 0 : TTLOAD_DEFAULT);
              /* do scaling and hinting only if no rotation/scaling requested */
   if (error) {
      TT_Flush_Face(pfch->pffh->face);
      ERRRET(-1);
   }

   /* --- Outline processing --- */
   if (pCharAttr->iQuery & FD_QUERY_OUTLINE) {
      error = TT_Get_Glyph_Outline(pfch->glyph, &outline);
      TT_Flush_Face(pfch->pffh->face);
      if (error) {
         ERRRET(-1);
      }
      if (pCharAttr->cbLen == 0)  { /* only number of bytes is requested */
         cb = GetOutlineLen(&outline);
         return cb;
      }
      if (pfch->scaling) {   /* rotation/scaling enabled */
         error = TT_Get_Glyph_Metrics(pfch->glyph, &metrics);
         TT_Flush_Face(pfch->pffh->face);
         if (error) {
            ERRRET(-1);
         }
         /* !! This code is not properly tested */
         center.x = (metrics.bbox.xMax - metrics.bbox.xMin) / 2;
         center.y = (metrics.bbox.yMax - metrics.bbox.yMin) / 2;
         TT_Translate_Outline( &outline, -center.x, -center.y );
         TT_Transform_Outline( &outline, &pfch->matrix );
         Xform(&center, &pfch->matrix);
         center.x += 32768; center.y += 32768;
         center.x >>= 16; center.y >>= 16;
         TT_Translate_Outline( &outline, center.x, center.y );
      }

      return GetOutline(&outline, pCharAttr->pBuffer);
   }

   /* --- Bitmap processing --- */
   /* metrics are needed for all following operations */
   error = TT_Get_Glyph_Metrics(pfch->glyph, &metrics);
   if (error) {
      TT_Flush_Face(pfch->pffh->face);
      ERRRET(-1);
   }

   if (pfch->scaling) {   /* scaling enabled */
      TT_Get_Glyph_Outline( pfch->glyph, &outline );
      center.x = (metrics.bbox.xMax - metrics.bbox.xMin) / 2;
      center.y = (metrics.bbox.yMax - metrics.bbox.yMin) / 2;
      TT_Translate_Outline( &outline, -center.x, -center.y );
      TT_Transform_Outline( &outline, &pfch->matrix );
      Xform(&center, &pfch->matrix);
      center.x += 32768; center.y += 32768;
      center.x >>= 16; center.y >>= 16;
      TT_Translate_Outline( &outline, center.x, center.y );
   }

   minp.x = metrics.bbox.xMin;
   minp.y = metrics.bbox.yMin;
   maxp.x = metrics.bbox.xMax;
   maxp.y = metrics.bbox.yMax;

   if (pfch->scaling) {
      /* transform also min/max values */
      pt1.x = minp.x; pt1.y = maxp.y;
      pt2.x = maxp.x; pt2.y = minp.y;
      Xform(&minp, &pfch->matrix);
      Xform(&maxp, &pfch->matrix);
      Xform(&pt1, &pfch->matrix);
      Xform(&pt2, &pfch->matrix);
      /* find real min/max values */
      minp.x = min(minp.x, min(maxp.x, min(pt1.x, pt2.x)));
      minp.y = min(minp.y, min(maxp.y, min(pt1.y, pt2.y)));
      maxp.y = max(minp.y, max(maxp.y, max(pt1.y, pt2.y)));
      maxp.x = max(minp.x, max(maxp.x, max(pt1.x, pt2.x)));
      minp.x -= 32768; minp.y -= 32768;
      minp.x >>= 16; minp.y >>= 16;
      maxp.x += 32768; maxp.y += 32768;
      maxp.x >>= 16; maxp.y >>= 16;
   }
   minp.x &= -64; minp.y &= -64;
   maxp.x = (maxp.x + 63) & -64; maxp.y = (maxp.y + 63) & -64;

   if (pCharAttr->iQuery & FD_QUERY_BITMAPMETRICS) {
      /* fill in bitmap metrics */
      /* metrics values are in 26.6 format ! (but not imetrics !!)*/
      pbmm->sizlExtent.cx = (maxp.x - minp.x) >> 6;
      pbmm->sizlExtent.cy = (maxp.y - minp.y) >> 6;
      pbmm->cyAscent = 0;
      pbmm->pfxOrigin.x = MAKEFIXED(minp.x >> 6, 0);
      pbmm->pfxOrigin.y = MAKEFIXED(maxp.y >> 6, 0);
      if (!(pCharAttr->iQuery & FD_QUERY_BITMAPMETRICS)) {
         TT_Flush_Face(pfch->pffh->face);
         return sizeof(*pbmm);
      }
   }
   /* --- actual bitmap processing --- */
   if (pCharAttr->iQuery & FD_QUERY_CHARIMAGE) {
      /* values in 26.6 format ?!? */
      bitmap.width = (maxp.x - minp.x) >> 6;
      bitmap.rows  = (maxp.y - minp.y) >> 6;
      /* width rounded up to nearest multiple of 4 */
      bitmap.cols = ((bitmap.width + 31) / 8) & -4;
      bitmap.flow = TT_Flow_Down;
      bitmap.bitmap = pCharAttr->pBuffer;
      bitmap.size = bitmap.rows * bitmap.cols;
      TT_Flush_Face(pfch->pffh->face);

      if (pCharAttr->cbLen == 0) {
         return bitmap.size;
      }
      if (bitmap.size > pCharAttr->cbLen) {
         ERRRET(-1)     /* otherwise we might overwrite something */
      } /* endif */
      /* clean provided buffer (unfortunately necessary) */
      memset(bitmap.bitmap, 0, pCharAttr->cbLen);

      error = TT_Get_Glyph_Bitmap(pfch->glyph, &bitmap,
                                  -minp.x, -minp.y);
      if (error) {
         ERRRET(-1); /* engine error */
      }
      return bitmap.size; /* return # of bytes */
   }
   ERRRET(-1) /* error */
}

/* query names of all faces in this file */
LONG _System QueryFullFaces(HFF hff, PVOID pBuff, PULONG buflen,
                    PULONG cFontCount, ULONG cStart, PVOID pfn) {
   COPY("!QueryFullFaces: hff = ");
   CATI((int) hff);
   CAT("\r\n");
   WRITE;
   ERRRET(-1) /* error ? */
}

/* I have no idea whatsoever what this function might or should do. */
/* It isn't mentioned in the spec apart from the headers. And it    */
/* doesn't seem to be called anyway. Perhaps I'll someday squeeze   */
/* some info out of IBM. Perhaps not.                               */
LONG _System ClaimFontFile(PSZ pszFileName, PVOID pfn)
{
   /* just log this func has been called */
   COPY("!ClaimFontFile: ");
   CAT(pszFileName);
   CAT("\r\n");
   WRITE;

  return 0; /* act like everything is OK */
}

/* this is the DLL Initialization/termination function. It initializes */
/* the FreeType engine and some internal structures at startup. It     */
/* seems that the termination isn't actually called at all.            */
/* Slightly modified for use with Watcom C/C++                         */
/* Termination function is here although it never gets called anyway   */
#ifdef __IBMC__
ULONG _DLL_InitTerm(ULONG hModule, ULONG ulFlag) {
   #ifdef DEBUG
    ULONG Action;
   #endif /* DEBUG */
   switch (ulFlag) {
      case 0:             /* initializing */
#endif
#ifdef __WATCOMC__
unsigned __dll_initialize(void) {
   #ifdef DEBUG
    ULONG Action;
   #endif /* DEBUG */
#endif
         #ifdef DEBUG
            DosOpen("C:\\FTIFI.LOG", &LogHandle, &Action, 0, FILE_NORMAL,
                    OPEN_ACTION_CREATE_IF_NEW | OPEN_ACTION_REPLACE_IF_EXISTS,
                    OPEN_FLAGS_NO_CACHE | OPEN_FLAGS_WRITE_THROUGH |
                    OPEN_FLAGS_SEQUENTIAL | OPEN_SHARE_DENYWRITE | OPEN_ACCESS_WRITEONLY,
                    NULL);

            COPY("FTIFI loaded.\r\n");
            WRITE;
         #endif /* DEBUG */
         error = TT_Init_FreeType(&engine);   /* turn on the FT engine */
         if (error)
            return 0;     /* exit immediately */
         error = TT_Init_Kerning_Extension(engine); /* load kerning support */
         COPY("FreeType Init called\r\n");
         WRITE;
         if (error)
            return 0;     /* exit immediately */
         COPY("Initialization successful.\r\n");
         WRITE;
         return 1;

#ifdef __IBMC__

      case 1:            /* terminating */
#else /* Watcom */
}
unsigned __dll_terminate(void) {
#endif
         TT_Done_FreeType(engine);
         COPY("PMTTF terminated.\r\n");
         WRITE;
         #ifdef DEBUG
            DosClose(LogHandle);
         #endif
         return 1;
#ifdef __IBMC__
         break;

      default:
         return 0;
   }

   return 1;
#endif
}

/* the following function was ripped directly from a FreeType example program */
/* and subsequently modified to suit my purposes  */
/* It looks for a suitable name in the font file. */
char*  LookupName(TT_Face face,  int index )
{
   static char name_buffer[FACESIZE + 2];
   int    name_len = 0;
   int i, j, n;

   short  platform, encoding, language, id;
   char*  string;
   int    string_len;

   int    found;

   n = TT_Get_Name_Count( face );
   if ( n < 0 )
      return NULL;

   for ( i = 0; i < n; i++ )
   {
      TT_Get_Name_ID( face, i, &platform, &encoding, &language, &id );
      TT_Get_Name_String( face, i, &string, &string_len );

      if ( id == index )
      {

   /* The following code was inspired from Mark Leisher's ttf2bdf package */

        found = 0;

        /* Try to find a Microsoft English name */

        if ( platform == 3 )
          for ( j = 1; j >= 0; j-- )
            if ( encoding == j )  /* Microsoft ? */
              switch (language)
              {
                case 0x409:
                case 0x809:
                case 0xc09:
                case 0x1009:
                case 0x1409:
                case 0x1809: found = 1;
                             break;
              }

        if ( !found && platform == 0 && language == 0 )
          found = 1;

        /*
         * Found a Unicode Name.
         */
        if (found)
        {
          if ( string_len > FACESIZE * 2)  /* we must multiply by two - it's a Unicode string! */
            string_len = FACESIZE * 2;

          name_len = 0;

          for ( i = 1; i < string_len; i += 2 )
            name_buffer[name_len++] = string[i];

          name_buffer[name_len] = '\0';

          return name_buffer;
        }
      }
   }

   /* Not found */
   return NULL;
}

/* func to find suitable charmap, searching in this order of importance:    */
/*   1) Windows Unicode        */
/*   2) Apple Unicode          */
/*   3) Apple Roman            */
/*   4) Windows Symbol - not really supported  */
/*   5) other (perhaps WGL4?)  */
/* it maybe isn't very efficient, but does its job well */
/* High word of returned ULONG contains type of encoding */
ULONG GetCharmap(TT_Face face) {
   int    n;      /* # of encodings (charmaps) available */
   short  platform, encoding;
   int    i;

   n = TT_Get_CharMap_Count(face);

   if (n < 0) {    /* no encodings at all; don't yet know what the best course of action would be */
      ERRRET(-1)   /* such font should probably be rejected */
   } /* endif */

   /* look for Windoze Unicode */
   for (i = 0; i < n; i++) {
      TT_Get_CharMap_ID(face, i, &platform, &encoding);
      if (platform == 3 && encoding == 1) {
         return i;
      } /* endif */
   }
   /* then look for Apple Unicode */
   for (i = 0; i < n; i++) {
      TT_Get_CharMap_ID(face, i, &platform, &encoding);
      if (platform == 0) {
         return i;
      } /* endif */
   }
   /* then look for Apple Roman */
   for (i = 0; i < n; i++) {
      TT_Get_CharMap_ID(face, i, &platform, &encoding);
      if (platform == 1 && encoding == 0) {
         return i | (1 << 16); /* means no translation should be performed */
      } /* endif */
   }
   /* then look for Windows Symbol */
   /* !!! not really supported; so far there's no need */
   for (i = 0; i < n; i++) {
      TT_Get_CharMap_ID(face, i, &platform, &encoding);
      if (platform == 3 && encoding == 0) {
         return i;
      } /* endif */
   }
   /* no supported encoding found, just return the first one */
   return 0;
}

/* func to convert glyph outline from TT format to the format required by PM */
/* returns the number of bytes in buffer */
int GetOutlineLen(TT_Outline *ol) {

   int    index;     /* current point's index */
   BOOL   on_curve;  /* current point's state */
   int    i, start = 0;
   int    first, last;
   ULONG  cb = 0;

   /* loop thru all contours in a glyph */
   for ( i = 0; i < ol->contours; i++ ) {

      cb += sizeof(POLYGONHEADER);

      first = start;
      last = ol->conEnds[i];

      on_curve = (ol->flag[first] & 1);
      index    = first;

      /* process each contour point individually */
      while ( index < last ) {
         index++;

         if ( on_curve ) {
            /* the previous point was on the curve */
            on_curve = ( ol->flag[index] & 1 );
            if ( on_curve ) {
               /* two successive on points => emit segment */
               cb += sizeof(PRIMLINE);
            }
         }
         else {
            /* the previous point was off the curve */
            on_curve = ( ol->flag[index] & 1 );
            if ( on_curve ) {
               /* reaching an `on' point */
               cb += sizeof(PRIMSPLINE);
            }
            else {
               /* two successive `off' points => create middle point */
               cb += sizeof(PRIMSPLINE);
            }
         }
      }

      /* end of contour, close curve cleanly */
      if ( ol->flag[first] & 1 )
      {
        if ( on_curve )
           cb += sizeof(PRIMLINE);
        else
           cb += sizeof(PRIMSPLINE);
      }
      else
        if (!on_curve)
           cb += sizeof(PRIMSPLINE);

      start = ol->conEnds[i] + 1;

   }
   return cb; /* return # bytes used */
}

/* define a few global variables used in the following functions */
ULONG          cb = 0, polycb;
LONG           lastX, lastY;
PBYTE          pb;
POINTFX        Q, R;
POLYGONHEADER  hdr = {0, FD_POLYGON_TYPE};
PRIMLINE       line = {FD_PRIM_LINE};
PRIMSPLINE     spline = {FD_PRIM_SPLINE};

void Line_From(LONG x, LONG y) {
   line.pte.x = x << 10;
   line.pte.y = y << 10;
   /* store to output buffer */
   memcpy(&(pb[cb]), &line, sizeof(line));
   cb += sizeof(PRIMLINE);
   polycb += sizeof(PRIMLINE);
}

void Bezier_From( LONG x0, LONG y0, LONG x2, LONG y2, LONG x1, LONG y1 ) {
   spline.pte[0].x = x0 << 10;
   spline.pte[0].y = y0 << 10;
   /* convert from second-order to cubic Bezier spline */
   Q.x = (x0 + 2 * x1) / 3;
   Q.y = (y0 + 2 * y1) / 3;
   R.x = (x2 + 2 * x1) / 3;
   R.y = (y2 + 2 * y1) / 3;
   spline.pte[1].x = Q.x << 10;
   spline.pte[1].y = Q.y << 10;
   spline.pte[2].x = R.x << 10;
   spline.pte[2].y = R.y << 10;
   /* store to output buffer */
   memcpy(&(pb[cb]), &spline, sizeof(spline));
   cb += sizeof(PRIMSPLINE);
   polycb += sizeof(PRIMSPLINE);
}

/* Fills buffer pb with glyph outline converted to PM format; buffer is */
/* expected to be of size returned previously by GetOutlineLen().       */
/* The code is taken right from ttraster.c and subsequently modified.   */
int GetOutline(TT_Outline *ol, PBYTE pbuf) {
   LONG   x,  y;   /* current point                */
   LONG   cx, cy;  /* current Bezier control point */
   LONG   mx, my;  /* current middle point         */
   LONG   x_first, y_first;  /* first point's coordinates */
   LONG   x_last,  y_last;   /* last point's coordinates  */

   int    index;     /* current point's index */
   BOOL   on_curve;  /* current point's state */
   int    i, start = 0;
   int    first, last;
   ULONG  polystart;

   pb = pbuf;
   cb = 0;

   /* loop thru all contours in a glyph */
   for ( i = 0; i < ol->contours; i++ ) {

      polystart = cb;  /* save this polygon's start offset */
      polycb = sizeof(POLYGONHEADER); /* size of this polygon */
      cb += sizeof(POLYGONHEADER);

      first = start;
      last = ol->conEnds[i];

      x_first = ol->xCoord[first];
      y_first = ol->yCoord[first];

      x_last  = ol->xCoord[last];
      y_last  = ol->yCoord[last];

      lastX = cx = x_first;
      lastY = cy = y_first;

      on_curve = (ol->flag[first] & 1);
      index    = first;

      /* check first point to determine origin */
      if ( !on_curve ) {
         /* first point is off the curve.  Yes, this happens... */
         if ( ol->flag[last] & 1 ) {
            lastX = x_last;  /* start at last point if it */
            lastY = y_last;  /* is on the curve           */
         }
         else {
            /* if both first and last points are off the curve, */
            /* start at their middle and record its position    */
            /* for closure                                      */
            lastX = (lastX + x_last)/2;
            lastY = (lastY + y_last)/2;

            x_last = lastX;
            y_last = lastY;
         }
      }

      /* now process each contour point individually */
      while ( index < last ) {
         index++;
         x = ( ol->xCoord[index] );
         y = ( ol->yCoord[index] );

         if ( on_curve ) {
            /* the previous point was on the curve */
            on_curve = ( ol->flag[index] & 1 );
            if ( on_curve ) {
               /* two successive on points => emit segment */
               Line_From( lastX, lastY ); /*x, y*/
               lastX = x;
               lastY = y;
            }
            else {
               /* else, keep current control point for next bezier */
               cx = x;
               cy = y;
            }
         }
         else {
            /* the previous point was off the curve */
            on_curve = ( ol->flag[index] & 1 );
            if ( on_curve ) {
               /* reaching an `on' point */
               Bezier_From(lastX, lastY, x, y, cx, cy );
               lastX = x;
               lastY = y;
            }
            else {
               /* two successive `off' points => create middle point */
               mx = (cx + x) / 2;
               my = (cy + y)/2;

               Bezier_From( lastX, lastY, mx, my, cx, cy );
               lastX = mx;
               lastY = my;

               cx = x;
               cy = y;
            }
         }
      }

      /* end of contour, close curve cleanly */
      if ( ol->flag[first] & 1 ) {
         if ( on_curve )
            Line_From( lastX, lastY); /* x_first, y_first );*/
         else
            Bezier_From( lastX, lastY, x_first, y_first, cx, cy );
      }
      else
        if (!on_curve)
          Bezier_From( lastX, lastY, x_last, y_last, cx, cy );

      start = ol->conEnds[i] + 1;

      hdr.cb = polycb;
      memcpy(&(pb[polystart]), &hdr, sizeof(hdr));

   }
   return cb; /* return # bytes used */
}

