/*
 *   ttf2pk.c
 *
 *   This file is part of the ttf2pk package.
 *
 * Copyright 1997-1998 by
 *      Frederic Loyer <loyer@ensta.fr>
 *      Werner Lemberg <a7971428@unet.univie.ac.at>
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <math.h>       /* libc ANSI */
#include <ctype.h>


#include "pklib.h"
#include "ttfenc.h"
#include "ttflib.h"
#include "errormsg.h"
#include "filesrch.h"


char ident[] = "ttf2pk version 1.0";
char progname[] = "ttf2pk";             /* for error/warning messages */

int dpi;
int ptsize;

#define BUF_SIZE 255

char buffer[BUF_SIZE];  /* input buffer (modified while parsing) */
char obuffer[BUF_SIZE];
char *param;

FILE *inenc;            /* The input encoding file */
FILE *config_file;      /* The ttfonts.map file */

char quiet;

long inenc_array[256];



/*
 *   Either allocate storage or fail with explanation.
 */

char *
mymalloc(unsigned size,
         const char *why)
{
  char *mem = (char *)malloc(size);


  if (mem == NULL)
    oops("Cannot allocate %u bytes for %s.", size, why);
  return mem;
}


/*
 *   Either reallocate storage or fail with explanation.
 */

char *
myrealloc(char *oldp,
          unsigned size,
          const char *why)
{
  char *mem;


  mem = oldp == NULL ? (char *)malloc(size)
                     : (char *)realloc(oldp, size);
  if (mem == NULL)
    oops("Cannot reallocate %u bytes for %s.", size, why);
  return mem;
}


struct encoding
{
  char *name;
  char *vec[256];
};



/*
 *   Here is the encoding parser taken from ttf2tfm.
 */

/* the following function handles errors in the encoding file */

void
print_error(register char *s)
{
  (void)fprintf(stderr, "%s: ERROR: ", progname);
  (void)fprintf(stderr, "%s\n", s);

  if (obuffer[0])
  {
    (void)fprintf(stderr, "%s\n", obuffer);
    while (param > buffer)
    {
      (void)fprintf(stderr, " ");
      param-- ;
    }
    (void)fprintf(stderr, "^\n");
  }

  exit(1);
}


char *
newstring(char *s)
{
  char *q = mymalloc((unsigned long)(strlen(s) + 1), "newstring");
  (void)strcpy(q, s);
  return q;
}


/* getline() will truncate input lines after BUF_SIZE characters */

int
getline(void)
{
  register char *p;
  register int c;
  register int count;


  param = buffer;
  for (p = buffer, count = 0;
       (c = getc(inenc)) != EOF && c != '\n' && count < BUF_SIZE -1; count++)
    *p++ = c;
  *p = '\0';

  (void)strcpy(obuffer, buffer);

  if (p == buffer && c == EOF)
    return 0;
  else
    return 1;
}


/*
 *   Here we get a token from the AFM file.  We parse just as much PostScript
 *   as we expect to find in an encoding file.  We allow commented lines and
 *   names like 0, .notdef, _foo_.  We do not allow //abc.
 */

char smbuffer[100];    /* for tokens */


char *
gettoken(void)
{
  char *p, *q;


  while (1)
  {
    while (param == NULL || *param == '\0')
    {
      if (getline() == 0)
        print_error("Premature end in encoding file");
      for (p = buffer; *p; p++)
        if (*p == '%')
        {
          *p = '\0';
          break;
        }
    }

    while (*param && *param <= ' ')
      param++;

    if (*param)
    {
      if (*param == '[' || *param == ']' ||
          *param == '{' || *param == '}')
      {
        smbuffer[0] = *param++;
        smbuffer[1] = '\0';
        return smbuffer;
      }
      else if (*param == '/' || *param == '-' || *param == '_' ||
               *param == '.' ||
               ('0' <= *param && *param <= '9') ||
               ('a' <= *param && *param <= 'z') ||
               ('A' <= *param && *param <= 'Z'))
      {
        smbuffer[0] = *param;
        for (p = param+1, q = smbuffer+1;
             *p == '-' || *p == '_' || *p == '.' ||
             ('0' <= *p && *p <= '9') ||
             ('a' <= *p && *p <= 'z') ||
             ('A' <= *p && *p <= 'Z'); p++, q++)
          *q = *p;

        *q = '\0';
        param = p;
        return smbuffer;
      }
    }
  }
}


char *
strip_equal(char *p)
{
  while (*p && isspace(*p))
    p++;
  if (!*p == '=')
    oops("Missing `=' in config line for the selected font");
  if (*p)
    p++;
  while (*p && isspace(*p))
    p++;
  return p;
}


/*
 *   This routine reads in an encoding file, given the name.  It returns
 *   the final total structure.  It performs a number of consistency checks.
 */

struct encoding *
readencoding(char *enc)
{
  char *real_enname;
  char *p, c;
  int i;
  long l;
  struct encoding *e =
    (struct encoding *)mymalloc((unsigned long)sizeof (struct encoding),
                                "reading encoding");


  if (inenc)
    oops("Internal inenc error");

  if (enc)
  {
    real_enname = TeX_search_encoding_file(enc);
    if (!real_enname)
      oops("Cannot find encoding file `%s'", enc);

    inenc = fopen(real_enname, "r");
    if (inenc == NULL)
      oops("Cannot open encoding file `%s'", enc);

    param = NULL;
    p = gettoken();
    if (*p != '/' || p[1] == '\0')
      print_error(
        "First token in encoding must be literal encoding name");
    e->name = newstring(p + 1);
    p = gettoken();
    if (strcmp(p, "["))
      print_error("Second token in encoding must be mark ([) token");

    for (i = 0; i < 256; i++)
    {
      p = gettoken();
      if (*p != '/' || p[1] == 0)
        print_error(
          "Tokens 3 to 257 in encoding must be literal names");

      /* now we test for a generic code point resp. glyph index value */

      c = p[2];
      if (p[1] == '.' && (c == 'c' || c == 'g') && '0' <= p[3] && p[3] <= '9')
      {
        l = strtol(p + 3, &p, 0);
        if (*p != '\0' || l < 0 || l > 0xFFFF)
          print_error("Invalid encoding token");
        sprintf(p, ".%c0x%x", c, (int)l);
        e->vec[i] = newstring(p);
      }
      else
        e->vec[i] = newstring(p + 1);
    }

    p = gettoken();
    if (strcmp(p, "]"))
      print_error("Token 258 in encoding must be make-array (])");

    while (getline())
    {
      for (p = buffer; *p; p++)
        if (*p == '%')
        {
          *p = '\0';
          break;
        }
    }

    fclose(inenc);
    inenc = NULL;
  }
  else
    e = NULL;

  param = NULL;
  return e;
}


#define USAGE "\
  Convert a TrueType font to TeX's PK format.\n\
\n\
-q                  suppress informational output\n\
--help              print this message and exit\n\
--version           print version number and exit\n\
"

static void
usage(void)
{
  fputs("Usage: ttf2pk [-q] <font> <dpi>\n", stdout);
  fputs(USAGE, stdout);
  exit(0);
}


#define VERSION "\
Copyright (C) 1997-1998 Frederic Loyer and Werner Lemberg.\n\
There is NO warranty.  You may redistribute this software\n\
under the terms of the GNU General Public License\n\
and the gsftopk copyright.\n\
\n\
For more information about these matters, see the files\n\
named COPYING and pklib.c.\n\
\n\
Primary authors of ttf2pk: F. Loyer and W. Lemberg.\n\
\n\
ttf2pk is partially based on gsftopk from P. Vojta\n\
and the FreeType project from\n\
David Turner, Robert Wilhelm, and Werner Lemberg\n\
"

static void
version(void)
{
  fputs(ident, stdout);
  fprintf(stdout, " (%s)\n", TeX_search_version());
  fputs(VERSION, stdout);
  exit(0);
}


void 
main(int argc, char** argv)
{
  int i;
  long code;
  char *configline, *p;
  unsigned int cflinelen;
  struct encoding *enc;
  char *fontname;
  int fontname_len;
  char *ttf_filename, *pk_filename, *tfm_filename, *enc_filename;
  char *real_ttf_filename;
  int slant, extend;
  int pid = 3, eid = 1;


  TeX_search_init(argv[0], "TTF2PK");

  if (argc == 1)
    oops("Need at least two arguments.\n"
         "Try `ttf2pk --help' for more information.");

  if (argc == 2)
  {
    if (strcmp(argv[1], "--help") == 0)
      usage();
    else if (strcmp(argv[1], "--version") == 0)
      version();
  }

  if (argv[1][0] == '-')
  {
    if (argv[1][1] == 'q')
      quiet = 1;
    else
      oops("Unknown option `%s'.\n"
           "Try `ttf2pk --help' for more information.", argv[1]);

    argv++;
    argc--;
  }

  if (argc != 3)
    oops("Need at most two arguments.\n"
         "Try `ttf2pk --help' for more information.");

  if ((dpi = atoi(argv[2])) <= 50)
    oops("dpi value must be larger than 50");

  fontname = argv[1];
  fontname_len = strlen(fontname);
  enc_filename = NULL;
  ptsize = 10;

  slant = 0;
  extend = 1024;
  config_file = fopen(TeX_search_config_file("ttfonts.map"), "r");
  
  if (config_file == NULL) 
    oops("Cannot find file ttfonts.map");

  configline = (char *)mymalloc(cflinelen = 80, "Config file line");

  do
  {
    int len = 0;

    
    if (fgets(configline, cflinelen, config_file) == NULL)
    {
      fprintf(stderr,
              "%s: ERROR: Cannot find font %s in ttfonts.map\n",
              progname, fontname);
      exit(2);
    }

    /* we read in a line of the config file dynamically */

    for (;;)
    {
      i = strlen(configline + len);
      
      len += i;
      if (len > 0 && configline[len - 1] == '\n')
      {
        configline[--len] = '\0';
        break;
      }
      if (len < cflinelen - 1)
        break;
      configline = myrealloc(configline, cflinelen += 80,
                             "config file line");
      fgets(configline + len, cflinelen - len, config_file);
    }
  } while (memcmp(configline, fontname, fontname_len) != 0 ||
           (configline[fontname_len] &&
            !isspace(configline[fontname_len])));

  fclose(config_file);

  /*
   *   Parse the line from the config file.  We split the config line buffer
   *   into substrings according to the given options.
   */

  p = configline + fontname_len;

  while (*p && isspace(*p))
    p++;
  if (!*p)
    oops("No TTF file given in config line for selected font");

  ttf_filename = p;

  while (*p && !isspace(*p))
    p++;
  if (*p)
    *p++ = '\0';      

  for (; *p; p++)
  {
    if (isspace(*p))
      continue;

    if (!strncmp(p, "Slant", 5))
    {
      float s;


      p = strip_equal(p + 5);
      if (sscanf(p, "%f", &s) == 0)
        oops("Bad `Slant' parameter in config line for the selected font");
      slant = floor(s * 1024);
    }
    else if (!strncmp(p, "Encoding", 8))
    {
      p = strip_equal(p + 8);
      if (!*p)
        oops("Bad `Encoding' parameter in config line for the selected font");
      enc_filename = p;
    }
    else if (!strncmp(p, "Extend", 6))
    {
      float e;


      p = strip_equal(p + 6);
      if (sscanf(p, "%f", &e) == 0)
        oops("Bad `Extend' parameter in config line for the selected font");
      extend = floor(e * 1024);
    }
    else if (!strncmp(p, "Pid", 3))
    {
      p = strip_equal(p + 3);
      if (sscanf(p, "%d", &pid) < 0)
        oops("Bad `Pid' parameter in config line for the selected font");
    }
    else if (!strncmp(p, "Eid", 3))
    {
      p = strip_equal(p + 3);
      if (sscanf(p, "%d", &eid) < 0)
        oops("Bad `Eid' parameter in config line for the selected font");
    }

    while (*p && !isspace(*p))
      p++;
    if (*p)
      *p++ = '\0';
  }

  tfm_filename = mymalloc(fontname_len + 5, "tfm filename");
  sprintf(tfm_filename, "%s.tfm", fontname);
  TFMopen(tfm_filename);

  pk_filename = mymalloc(fontname_len + 10, "pk filename");
  sprintf(pk_filename, "%s.%dpk", fontname, dpi);
  PKopen(pk_filename, fontname);

  real_ttf_filename = TeX_search_ttf_file(ttf_filename);
  if (!real_ttf_filename)
    oops("Cannot find %s\n", ttf_filename);
  TTFopen(real_ttf_filename, dpi, ptsize, extend, slant, pid, eid);

  enc = readencoding(enc_filename);
  if (enc)
  {
    char *name;


    for (i = 0; i <= 0xFF; i++)
    {
      name = enc->vec[i];
      code = adobename_to_code(name);
      if (code < 0 && strcmp(name, ".notdef") != 0)
        warning("Cannot map character `%s'", name);
      inenc_array[i] = code;
    }
  }
  else
    TTFget_first_glyphs(inenc_array);

  for (i = 0; i <= 0xFF; i++)
  {
    byte *bitmap;
    int w, h, hoff, voff;


    if ((code = inenc_array[i]) >= 0)
    {
      if (!quiet)
      {
        printf("Processing %3d %s %04x %s\n",
               i, (code >= 0x10000) ? "glyph" : "code",
               (int)(code & 0xFFFF), enc ? enc->vec[i] : "");
        fflush(stdout);
      }

      TTFprocess(code, &bitmap, &w, &h, &hoff, &voff);

      PKputglyph(i,
                 -hoff, -voff, w - hoff, h - voff,
                 w, h, bitmap);
    }
  }

  PKclose();
  exit(0);
}


/* end */
