(****************************************************************************)
(*                                                                          *)
(* FTBitmap.pas                                                         1.0 *)
(*                                                                          *)
(*   The FreeType library uses the bitmap and pixmap structures defined     *)
(*   in this file. You can freely change them to suit your graphics         *)
(*   environment ( Windows, X11, OS/2, wathever.. ) as long as you provide  *)
(*   the functions that will be invoked by the library.                     *)
(*                                                                          *)
(*                                                                          *)
(*                                                                          *)
(*                                                                          *)
(*                                                                          *)
(****************************************************************************)

unit FTBitmap;

interface

uses
{$IFDEF VIRTUAL}
  Use32,
{$ENDIF}
  FTLib,
  FreeType;

  (******************************************************************)
  (*  Creates a new empty bitmap object                             *)
  (*                                                                *)
  function FT_New_Bitmap : FT_Bitmap;

  (**********************************************************)
  (*  convert a TrueType bitmap (i.e. a TT_Raster_Map)      *)
  (*  in an FT_Bitmap.                                      *)
  (*                                                        *)
  (*  the function should return the following values :     *)
  (*                                                        *)
  (*     0 : conversion performed. the tt bitmap buffer     *)
  (*         is referenced by the new 'bitmap' and should   *)
  (*         not be destroyed with the tt bitmap            *)
  (*                                                        *)
  (*     1 : conversion performed. the tt bitmap buffer     *)
  (*         has been copied to the new 'bitmap' and can    *)
  (*         be destroyed with the tt bitmap                *)
  (*                                                        *)
  (*     2 : or any other value, indicates the bitmap       *)
  (*         could not be converted, in which case          *)
  (*         'bitmap' should be set to nil                  *)
  (*                                                        *)
  function  FT_Convert_TT_Bitmap( var ttmap  : TT_Raster_Map;
                                  var bitmap : FT_Bitmap ) : integer;

  function  FT_Get_Bitmap_Width ( bitmap : FT_Bitmap ) : integer;
  function  FT_Get_Bitmap_Height( bitmap : FT_Bitmap ) : integer;

  (******************************************************************)
  (*  Delete a bitmap                                               *)
  (*                                                                *)
  function  FT_Done_Bitmap( bitmap : FT_Bitmap ) : FT_Error;

  (******************************************************************)
  (*  Create a new empty pixmap object                              *)
  (*                                                                *)
  function  FT_New_Pixmap : FT_Pixmap;

  (**********************************************************)
  (*  convert a TT_Pixmap to an equivalent FT_Pixmap        *)
  (*                                                        *)
  (*  the function should return the following values :     *)
  (*                                                        *)
  (*     0 : conversion performed. the tt pixmap buffer     *)
  (*         is referenced by the new 'pixmap' and should   *)
  (*         not be destroyed with the tt pixmap            *)
  (*                                                        *)
  (*     1 : conversion performed. the tt pixmap buffer     *)
  (*         has been copied to the new 'pixmap' and can    *)
  (*         be destroyed with the tt pixmap                *)
  (*                                                        *)
  (*     2 : or any other value, indicates the pixmap       *)
  (*         could not be converted                         *)
  (*                                                        *)
  function  FT_Convert_TT_Pixmap( var ttmap  : TT_Raster_Map;
                                  var pixmap : FT_Pixmap ) : integer;

  function  FT_Get_Pixmap_Width ( pixmap : FT_Pixmap ) : integer;
  function  FT_Get_Pixmap_Height( pixmap : FT_Pixmap ) : integer;

  (******************************************************************)
  (*  Delete a pixmap                                               *)
  (*                                                                *)
  function  FT_Done_Pixmap( pixmap : FT_Pixmap ) : FT_Error;


(*
  XXX : we don't have a Type1 engine yet..

  function  FT_Convert_T1_Bitmap( var t1map  : T1_Bitmap;
                                  var bitmap : FT_Bitmap ) : integer;

  function  FT_Convert_T1_Bitmap( var t1map  : T1_Pixmap;
                                  var pixmap : FT_Pixmap ) : integer;
*)

  (* The following definitions aren't used by the library directly *)

type
  PBitmap = ^TBitmap;
  TBitmap = record
    width     : integer;   (* width in pixels          *)
    height    : integer;   (* height in pixels         *)
    linewidth : integer;   (* row width in bytes       *)
    totalsize : integer;   (* total bitmap buffer size *)
    buffer    : Pointer;   (* bitmap buffer            *)
  end;

  PPixmap = ^TPixmap;
  TPixmap = TBitmap;

implementation

const
  empty_bitmap : TBitmap =
    ( width     : 0;
      height    : 0;
      linewidth : 0;
      totalsize : 0;
      buffer    : nil );

  (******************************************************************)
  (*  Creates a new empty bitmap object                             *)
  (*                                                                *)
  function FT_New_Bitmap : FT_Bitmap;
  var
    b : PBitmap;
  begin
    New( b );
    b^ := empty_bitmap;
  end;

  function FT_Get_Bitmap_Width( bitmap : FT_Bitmap ) : integer;
  var
    b : PBitmap;
  begin
    b := PBitmap(bitmap);
    if b <> nil then
      FT_Get_Bitmap_Width := b^.width
    else
      FT_Get_Bitmap_Width := 0;
  end;

  function FT_Get_Bitmap_Height( bitmap : FT_Bitmap ) : integer;
  var
    b : PBitmap;
  begin
    b := PBitmap(bitmap);
    if b <> nil then
      FT_Get_Bitmap_Height := b^.height
    else
      FT_Get_Bitmap_Height := 0;
  end;

  (**********************************************************)
  (*  convert a TT_Bitmap to an equivalent FT_Bitmap        *)
  (*                                                        *)
  (*  the function should return the following values :     *)
  (*                                                        *)
  (*     0 : conversion performed. the tt bitmap buffer     *)
  (*         is referenced by the new 'bitmap' and should   *)
  (*         not be destroyed with the tt bitmap            *)
  (*                                                        *)
  (*     1 : conversion performed. the tt bitmap buffer     *)
  (*         has been copied to the new 'bitmap' and can    *)
  (*         be destroyed with the tt bitmap                *)
  (*                                                        *)
  (*     2 : or any other value, indicates the bitmap       *)
  (*         could not be converted                         *)
  (*                                                        *)
  function  FT_Convert_TT_Bitmap( var ttmap  : TT_Raster_Map;
                                  var bitmap : FT_Bitmap ) : integer;
  var
    b : PBitmap;
  begin
    bitmap := FT_New_Bitmap;
    b := PBitmap(bitmap);
    if b <> nil then
    begin
      b^.height    := ttmap.rows;
      b^.width     := ttmap.width;
      b^.linewidth := ttmap.cols;
      b^.totalsize := ttmap.size;
      b^.buffer    := ttmap.buffer;

      FT_Convert_TT_Bitmap := 0;
    end
    else
      FT_Convert_TT_Bitmap := 2;
  end;

  function  FT_Done_Bitmap( bitmap : FT_Bitmap ) : FT_Error;
  var
    b : PBitmap;
  begin
    b := PBitmap(bitmap);
    if b <> nil then
      begin
        if (b^.buffer <> nil) and (b^.totalSize <> 0) then
          FreeMem( b^.buffer, b^.totalSize );
        Dispose( b );
      end;
  end;

  function FT_New_Pixmap : FT_Pixmap;
  var
    p : PPixmap;
  begin
    New(p);
    FT_New_Pixmap := FT_Pixmap(p);
  end;

  function FT_Get_Pixmap_Width( pixmap : FT_Pixmap ) : integer;
  var
    b : PPixmap;
  begin
    b := PPixmap(pixmap);
    if b <> nil then
      FT_Get_Pixmap_Width := b^.width
    else
      FT_Get_Pixmap_Width := 0;
  end;

  function FT_Get_Pixmap_Height( pixmap : FT_Pixmap ) : integer;
  var
    b : PPixmap;
  begin
    b := PPixmap(pixmap);
    if b <> nil then
      FT_Get_Pixmap_Height := b^.height
    else
      FT_Get_Pixmap_Height := 0;
  end;

  (**********************************************************)
  (*  convert a TT_Pixmap to an equivalent FT_Pixmap        *)
  (*                                                        *)
  (*  the function should return the following values :     *)
  (*                                                        *)
  (*     0 : conversion performed. the tt pixmap buffer     *)
  (*         is referenced by the new 'pixmap' and should   *)
  (*         not be destroyed with the tt pixmap            *)
  (*                                                        *)
  (*     1 : conversion performed. the tt pixmap buffer     *)
  (*         has been copied to the new 'pixmap' and can    *)
  (*         be destroyed with the tt pixmap                *)
  (*                                                        *)
  (*     2 : or any other value, indicates the pixmap       *)
  (*         could not be converted                         *)
  (*                                                        *)
  function  FT_Convert_TT_Pixmap( var ttmap  : TT_Raster_Map;
                                  var pixmap : FT_Pixmap ) : integer;
  var
    p : PPixmap;
  begin
    pixmap := FT_New_Pixmap;
    p := PPixmap(pixmap);
    if p <> nil then
      begin
        p^.height    := ttmap.rows;
        p^.width     := ttmap.width;
        p^.linewidth := ttmap.cols;
        p^.totalsize := ttmap.size;
        p^.buffer    := ttmap.buffer;

        FT_Convert_TT_Pixmap := 0;
      end
    else
      FT_Convert_TT_Pixmap := 2;
  end;

  function FT_Done_Pixmap( pixmap : FT_Pixmap ) : FT_Error;
  var
    p : PPixmap;
  begin
    p := PPixmap(pixmap);
    if p <> nil then
      Dispose(p);
    FT_Done_Pixmap := FT_Err_Ok;
  end;

end.
