/* The GIMP -- an image manipulation program
 * Copyright (C) 1995-1998 Spencer Kimball and Peter Mattis
 *
 * guash.c -- This is a plug-in for the GIMP 1.0
 * Time-stamp: <1998/01/09 23:33:28 narazaki@InetQ.or.jp>
 * Copyright (C) 1997-1998 Shuji Narazaki <narazaki@InetQ.or.jp>
 * guash uses internally for load xv's thumbnail:
 * xvpict (Copyright (C) 1997 Marco Lamberto <ml568366@silab.dsi.unimi.it>)
 * (slighly modified xvpict is embedded into guash for reducing communication
 * overhead)
 * Version: 0.99.2
 * Guash: Gimp Users' Another SHell (pronounced like 'gouache')
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 * Comment:
 * Thanks:
 *  Marco Lamberto <ml568366@silab.dsi.unimi.it>
 *
 * Implementation Priority:
 *    1. BUG FIX
 *    8. Clean up code: function name, layer violation, unnecessary precondition
 *    9. core cleaner? Who dumps core? And where?
 * 9990. DnD support after solving a desing issue.
 * 9999. Display non-image files (but I don't think it is necessary for guash)
 */

#include "guash-banner.h"
#include "guash-directory.h"
#include "gtk/gtk.h"
#include "gdk/gdkkeysyms.h"
#include "libgimp/gimp.h"
#include <sys/types.h>
#include <sys/stat.h>
#include <dirent.h>
#include <errno.h>
#include <fcntl.h>
#include <math.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <pwd.h>

#define	PLUG_IN_NAME	"extension_guash"
#define SHORT_NAME	"Guash"
#define MENU_PATH	"<Toolbox>/Xtns/Guash"
#define INTERFACE	guash_interface
#define	DIALOG		guash_dialog
#define VALS		guash_vals
#define THUMBNAIL_DIR	"/.xvpics"
#ifdef THUMBNAIL_FORMAT_IS_XCF
#undef THUMBNAIL_SUFFIX	".xcf"
#endif
#define GUASH_SELECTION_SCM	"#guash-selection.scm"
#define NEW_DIRECTORY_MODE	00777
#define NEW_FILE_MODE	00644
#define DIRECTORY_LABEL_MAXLEN	55
#define JUMP_BUTTON_WIDTH	40
#define	LINE_BUF_SIZE		1024
#define	SCROLLBAR_WIDTH		15
#define THUMBNAIL_INFO_MAXLEN	55
#define THUMBNAIL_WIDTH		80		/* Max width of XV thumbnail */
#define THUMBNAIL_HEIGHT	60		/* Max height of XV thumbnail */
#define THUMBNAIL_FONT_SIZE	10
#define THUMBNAIL_THEIGHT	(THUMBNAIL_FONT_SIZE + 6)
#define THUMBNAIL_FONT_FAMILY	"helvetica"
#define THUMBNAIL_FONT_SLANT	"r"
#define THUMBNAIL_SEPARATOR	5
#define INDEX_TO_X(i)	((i % ncol_of_thumbnail) \
			 * (THUMBNAIL_WIDTH + THUMBNAIL_SEPARATOR) \
			 + THUMBNAIL_SEPARATOR)
#define INDEX_TO_Y(i)	((i / ncol_of_thumbnail) \
			 * (THUMBNAIL_HEIGHT + THUMBNAIL_THEIGHT + THUMBNAIL_SEPARATOR) \
			 + THUMBNAIL_THEIGHT)
#define POS_TO_INDEX(x,y)	(cwd_cache->display_page * nthumbails_in_page \
				 + (y - THUMBNAIL_THEIGHT) \
				 / (THUMBNAIL_HEIGHT + THUMBNAIL_THEIGHT + THUMBNAIL_SEPARATOR) \
				 * ncol_of_thumbnail \
				 + x / (THUMBNAIL_WIDTH + THUMBNAIL_SEPARATOR))
#define EVENT_MASK	GDK_EXPOSURE_MASK | \
     			GDK_ENTER_NOTIFY_MASK | GDK_LEAVE_NOTIFY_MASK | \
			GDK_BUTTON_PRESS_MASK | GDK_BUTTON_RELEASE_MASK
/* gtkWrapper */
/* gtkW is the abbreviation of gtk Wrapper */
#define GTKW_ENTRY_BUFFER_SIZE	3
#define GTKW_ENTRY_WIDTH	20
#define GTKW_SCALE_WIDTH	200
#define	GTKW_BORDER_WIDTH	0
/* gtkW type */
typedef struct
{
  GtkWidget	*widget;
  gpointer	value;
  void	 	(*updater)();
} gtkW_widget_table;
gtkW_widget_table	widget_pointer[1];
/* gtkW global variables */
gint	gtkW_border_width = GTKW_BORDER_WIDTH;
gint	gtkW_border_height = 0;
gint	gtkW_homogeneous_layout	= FALSE;
gint	gtkW_frame_shadow_type = GTK_SHADOW_ETCHED_IN;
gint	gtkW_align_x = GTK_FILL|GTK_EXPAND;
gint	gtkW_align_y = GTK_FILL;
/* gtkW callback */
static void	gtkW_close_callback (GtkWidget *widget, gpointer data);
static void	gtkW_message_dialog (gint gtk_was_initialized, guchar *message);
static GtkWidget	*gtkW_message_dialog_new (guchar *name);
static gint	gtkW_confirmor_dialog (gint gtk_was_initialized, guchar *message,
				       gint default_value);
static GtkWidget 	*gtkW_confirmor_new (guchar *name, gint default_value,
					     gtkW_widget_table *table);
static void	gtkW_confirmor_yes (GtkWidget *widget, gpointer data);
static void	gtkW_confirmor_no (GtkWidget *widget, gpointer data);
static GtkWidget	*gtkW_frame_new (GtkWidget *parent, guchar *name);
static GtkWidget	*gtkW_hbox_new (GtkWidget *parent);
static GtkWidget	*gtkW_table_new (GtkWidget *parent, gint col, gint row);
static GtkWidget	*gtkW_table_add_button (GtkWidget	*table,
						gchar		*name,
						gint		x0,
						gint		x1,
						gint		y,
						GtkSignalFunc	callback,
						gpointer value);
static void	gtkW_iscroll_entry_change_value (gtkW_widget_table *wtable);
static void	gtkW_iscroll_update (GtkAdjustment *adjustment, gpointer data);
static void	gtkW_ivscroll_entry_new (GtkWidget **scroll,
					 GtkWidget **entry,
					 GtkSignalFunc	scale_update,
					 GtkSignalFunc	entry_update,
					 gint		*value,
					 gdouble	min,
					 gdouble	max,
					 gdouble	step,
					 gpointer	widget_entry);
static void	gtkW_ientry_update (GtkWidget *widget, gpointer data);
static GtkWidget	*gtkW_table_add_label (GtkWidget	*table,
					       gchar	*text,
					       gint	x0,
					       gint	x1,
					       gint	y,
					       gint	flush_left);
static void	gtkW_query_box (guchar *title, guchar *message,
				guchar *initial, guchar *data);
static void	gtkW_preview_force_to_update (GtkWidget *widget);
static GtkWidget *	gtkW_query_string_box_new (char        *title,
						   char        *message,
						   char        *initial,
						   gpointer     data);
static void gtkW_query_box_cancel_callback (GtkWidget *, gpointer);
static void gtkW_query_box_ok_callback (GtkWidget *, gpointer);
static gint gtkW_query_box_delete_callback (GtkWidget *, GdkEvent *, gpointer);
/* end of GtkW */

/* define enums */
/* for VAL.sort_kind (SORT_BY_NAME, SORT_BY_DATE */
#define SORT_BY_NAME	0
#define SORT_BY_DATE	1
/* for file_kind { NOT_EXIST, REGFILE, DIRECTORY, SLNKFILE, SLNKDIR } */
#define NOT_EXIST	0
#define REGFILE		2
#define DIRECTORY	4
#define SLNKFILE	8
#define SLNKDIR		16
/* for directory_cache.savable { FALSE, TRUE, UNKNOWN } */
#define UNKNOWN		2
/* for selection_kind { FALSE, IMAGE, DIRECTORY } */
#define	IMAGE		1
/* for sensives { SELECTION, SCROLL, HIDDEN } */
#define SELECTION	1
#define SCROLL		2
#define HIDDEN		4
/* for directory_cache_set_to_cwd() ( FASLE, TRUE, GC ) */
#define	GC		(TRUE + TRUE)

typedef struct
{
  gint	sort_type;
  guchar last_dir_name[LINE_BUF_SIZE];
  guchar mapped_command[LINE_BUF_SIZE];
} VALS;

typedef struct
{
  gint  deleted;
  gint	gc_flag;		/* 1 if it is a garbage */
  gint	directory_p;
  gint	width;
  gint	height;
  guchar	*data;
  guchar	name[256];
  guchar	info[256];
} cache_entry;

typedef struct
{
  guchar	name[LINE_BUF_SIZE];
#ifdef STRICT_MM
  gint		birth_index;
#endif
  gint		filed;
  gint		savable;
  gint		purged;	/* TRUE if move/copy/delete was executed */
  cache_entry	*image;
  cache_entry	*dir;
  gint		ndir;
  gint		nimage;
  gint		npage;
  gint		max_ndir;
  gint		max_nimage;
  gint		display_page;
} directory_cache;

VALS		VAL;
guchar		*directory_icon = NULL;
GHashTable	*directory_cache_table;
#ifdef STRICT_MM
/*
	If your machine does not have enough physical memory, you might want
	to set the upper boundary of memory that guash uses. It can do:

	(guash-directory-cache-size "M")	# M is an integer in [8:100]

	It defines how long history of directory traverse must hold.
	Since each thumbnail requires 80x60x3 byte = 14KB, guash requires at
	least:
		total_memory = M * N * 14 [KB],
	where N is the average number of image files in a directory.
	If no declaration, upper boundary is not set.
*/
gint		directory_cache_table_max_size = 0; /* this is not default */
gint		directory_cache_table_size = 0;
#endif
directory_cache *cwd_cache = NULL;
GtkWidget	*dlg = NULL;
GtkWidget	*cwd_label = NULL;
GtkWidget	*thumbnail_panel = NULL;
GtkWidget	*file_property = NULL;
GtkWidget	*thumbnail_panel_menu = NULL;
gint		ncol_of_thumbnail = 5;
gint		nrow_of_thumbnail = 3;
gint		nthumbails_in_page = 15;
gint		thumbnail_panel_last_index = -1;
gint		thumbnail_panel_page1 = 1;
gint		thumbnail_panel_width;
gint		thumbnail_panel_height;
gint32		text_image = -1;
gint32		text_bg = -1;
GList		*selection_list;
gint		selection_kind;
guchar		thumbnail_panel_info_default[256];
guchar		thumbnail_panel_directory_str[256];
guchar		fileselector_last_pathname[LINE_BUF_SIZE];

gint	binding_style = 0;
gint	use_confirmor = TRUE;
gint	hidden_features = FALSE;

#define NSCROLLER	2
GtkWidget	*widget_for_scroll[NSCROLLER];
#ifdef USE_DND
gchar		*sel_ops_keys[] = {"selection_operation"};
#endif

#define HEADER_PIXEL(data,pixel) \
  pixel[0] = (((data[0] - 33) << 2) | ((data[1] - 33) >> 4)); \
  pixel[1] = ((((data[1] - 33) & 0xF) << 4) | ((data[2] - 33) >> 2)); \
  pixel[2] = ((((data[2] - 33) & 0x3) << 6) | ((data[3] - 33))); \
  data += 4;

static void	query	(void);
static void	run	(char	*name,
			 int	nparams,
			 GParam	*param,
			 int	*nreturn_vals,
			 GParam **return_vals);
static gint	DIALOG ();
static gint	timer_initialize_thumbnail_panel (gpointer data);
static gint	about_dialog ();
static gint	cache_get_image (guchar *name, gint f_kind);
static gint	cache_get_image_from_file (guchar *name, guchar *suffix,
					   gint f_kind);
static void	cache_set_to_symlink_color ();
static void	cache_set_to_normal_color ();
static void	cache_render_text (guchar *str, cache_entry *cache);
static gint	cache_save_to_file (guchar *file_name, gint32 i_id);
static gint	cache_open_image_file (cache_entry *cache);
static guchar *	file_get_canonical_name (guchar *name);
static gint	file_get_last_slash_index (guchar *pathname);
static gint	file_get_last_period_index (guchar *pathname);
static guchar *	file_get_parent_directory (guchar *pathname);
static guchar *	file_get_filename (guchar *pathname);
static guchar *	file_build_cache_directory_name (guchar *filename);
static guchar *	file_build_cache_file_name (guchar *filename);
static gint	os_copy_file (guchar *filename, guchar *newname);
static gint	os_file_kind (guchar *filename, gint shrink);
static gint	os_file_mtimesort (struct dirent **a, struct dirent **b);
static gint	os_file_size (guchar *filename);
static gint	os_make_directory (guchar *pathname, gint mode);
static gint	os_rename_file (guchar *filename, guchar *newname);
static gint	os_scandir_selector (const struct dirent *dentry);
static gint	file_confirm_operation (guchar *operation_phrase);
static gint	cache_file_is_valid (guchar *file_name, guchar *suffix);
static gint	image_file_copy (guchar *filename, guchar *newname);
static gint	image_file_delete (guchar *filename);
static gint	image_file_move (guchar *filename, guchar *newname);
static void	fileselector_set_last_value (guchar *pathname);
directory_cache	*directory_cache_set_to_cwd (gint purge);
#ifdef STRICT_MM
static void	directory_cache_table_recycler (gpointer key,
						gpointer value,
						gpointer user_data);
#endif
directory_cache	*directory_cache_new (guchar *name);
static gint	directory_cache_set_npage ();
cache_entry *	directory_cache_get_nth (gint index);
static gint	directory_cache_add_directory (guchar *name, gint f_kind);
static gint	directory_cache_add_image (guchar *name, gint i_id, gint f_kind);
static gint	directory_cache_force_to_purge (guchar *name);
static gint	directory_cache_make_cache_directory (directory_cache *dentry);
static void	directory_cache_create_history_menu_foreach (gpointer key,
							     gpointer value,
							     gpointer user_data);
GtkWidget *	directory_cache_create_histroy_menu ();
GtkWidget *	directory_cache_create_parents_menu ();
static gint	directory_cache_delete_invalid_cache_files (gint force);
GtkWidget *	directory_cache_create_parents_menu ();
static gint	directory_cache_garbage_collect ();
static gint	directory_cache_update ();
static void	guash_parse_gimprc ();
GList *		selection_reset ();
static GList *	selection_add (gint index);
static GList *	selection_delete (gint index);
static gint	selection_is_active ();
static gint	selection_length ();
static gint	selection_member_p (gint index);
static void	selection_map_script ();
static void	selection_map_unix_command ();
static guchar *	generate_unix_command (guchar *template, guchar *directory, guchar *filename);
static void	selection_open_files ();
static gint	selection_reverse_member (gint index);
static void	selection_open_files ();
static void	selection_copy_files_to (guchar *pathname);
static void	selection_delete_files ();
static void	selection_move_files_to (guchar *pathname);
static gint	thumbnail_panel_show_cache (cache_entry *cache);
static void	thumbnail_panel_banner ();
static void	thumbnail_panel_clear ();
static void	thumbnail_panel_set_directory_info ();
static void	thumbnail_panel_set_info (guchar *name);
static void	thumbnail_panel_set_info_default ();
static void	thumbnail_panel_draw_frame (gint index, guchar val);
static void	thumbnail_panel_draw_selection_frame ();
static void	thumbnail_panel_clear_selection_frame ();
static void	thumbnail_panel_update_selection_buttons ();
static void	thumbnail_panel_update_scroller ();
static void	thumbnail_panel_update_sensitive_menu ();
static GtkWidget *thumbnail_panel_create_menu ();
static gint	thumbnail_panel_move_focus (gint offset);
static void	menu_show_callback (GtkWidget *widget, gpointer data);
static void	menu_change_directory_callback (GtkWidget *widget, gpointer data);
static void	directory_jump_callback (GtkWidget *widget, gpointer data);
static void	forward_callback (GtkWidget *widget, gpointer data);
static void	select_and_forward_callback (GtkWidget *widget, gpointer data);
static void	backward_callback (GtkWidget *widget, gpointer data);
static void	next_callback (GtkWidget *widget, gpointer data);
static void	prev_callback (GtkWidget *widget, gpointer data);
static void	next_page_callback (GtkWidget *widget, gpointer data);
static void	prev_page_callback (GtkWidget *widget, gpointer data);
static void	open_callback (GtkWidget *widget, gpointer data);
static void	update_callback (GtkWidget *widget, gpointer data);
static void	toggle_sort_mode_callback (GtkWidget *widget, gpointer data);
static void	purge_cache_file_callback (GtkWidget *widget, gpointer data);
static void	help_callback (GtkWidget *widget, gpointer data);
static void	fileselector_for_copy_callback (GtkWidget *widget, gpointer client_data);
static void	fileselector_for_move_callback (GtkWidget *widget, gpointer client_data);
static void	fileselector_for_chdir_callback (GtkWidget *widget, gpointer client_data);
static void	fileselector_for_mkdir_callback (GtkWidget *widget, gpointer client_data);
static void	copy_callback (GtkWidget *widget, gpointer data);
static void	move_callback (GtkWidget *widget, gpointer data);
static void	delete_callback (GtkWidget *widget, gpointer data);
static void	select_all_callback (GtkWidget *widget, gpointer data);
static void	select_none_callback (GtkWidget *widget, gpointer data);
static void	chdir_callback (GtkWidget *widget, gpointer data);
static void	mkdir_callback (GtkWidget *widget, gpointer data);
static void	selection_map_script_callback (GtkWidget *widget, gpointer data);
static void	selection_map_unix_command_callback (GtkWidget *widget, gpointer data);
static gint	preview_event_handler (GtkWidget *widget, GdkEvent *event);
static gint	cursor_event_handler (GtkWidget *widget, GdkEvent *event);
#ifdef USE_DND
static void	dnd_drag_copy_request (GtkWidget *button, GdkEvent *event);
static void	dnd_drag_move_request (GtkWidget *button, GdkEvent *event);
static void	dnd_drop (GtkWidget *button, GdkEvent *event);
#endif
static gint	gtkW_parse_gimprc_gint (guchar *name, gint default_value);
static guchar *	gtkW_parse_gimprc_string (guchar *name, guchar *buffer);
static gint	save_xvpict_image (char *filename, gint32 image_ID, gint32 drawable_ID);
static gint32	load_xvpict_image (char *filename);

typedef struct
{
  guchar	key;
  guint		mod;
} Gtk_binding;

#define NKEYSTYLES	2
typedef struct
{
  guchar	*label;
  void		(* command);
  gint		sensitive;
  GtkWidget	*widget;
  Gtk_binding	binding[NKEYSTYLES];
} image_command_table;

#define MENU_SEPARATOR   { NULL, NULL, 0, NULL, \
  				{ { 0, 0 }, \
				  { 0, 0 } \
			  } }
image_command_table
image_commands [] =
{
  { "Forward image", forward_callback, HIDDEN, NULL,
    { { 'F', 0 },
      { 'F', GDK_CONTROL_MASK }
    } },
  { "Backward image", backward_callback, HIDDEN, NULL,
    { { 'B', 0 },
      { 'B', GDK_CONTROL_MASK }
    } },
  { "Down image", next_callback, HIDDEN, NULL,
    { { 'D', 0 },
      { 'N', GDK_CONTROL_MASK }
    } },
  { "Up image", prev_callback, HIDDEN, NULL,
    { { 'U', 0 },
      { 'P', GDK_CONTROL_MASK }
    } },
  { "Next page", next_page_callback, SCROLL, NULL,
    { { 'N', 0 },
      { 'N', GDK_MOD1_MASK }
    } },
  { "Prev page", prev_page_callback, SCROLL, NULL,
    { { 'P', 0 },
      { 'P', GDK_MOD1_MASK }
    } },
  MENU_SEPARATOR,
  { "Selection All", select_all_callback, FALSE, NULL,
    { { 'A', GDK_CONTROL_MASK },
      { 'A', GDK_CONTROL_MASK }
    } },
  { "Selection None", select_none_callback, FALSE, NULL,
    { { 'A', GDK_CONTROL_MASK|GDK_SHIFT_MASK },
      { 'A', GDK_CONTROL_MASK|GDK_SHIFT_MASK }
    } },
  { "Select then go forward", select_and_forward_callback, HIDDEN, NULL,
    { { ' ', GDK_CONTROL_MASK },
      { ' ', GDK_CONTROL_MASK }
    } },
  MENU_SEPARATOR,
  { "Open", open_callback, SELECTION, NULL,
    { { 'O', GDK_CONTROL_MASK },
      { 'O', GDK_CONTROL_MASK }
    } },
  { "Copy to...", fileselector_for_copy_callback, SELECTION, NULL,
    { { 'C', GDK_CONTROL_MASK },
      { 'C', GDK_CONTROL_MASK }
    } },
  { "Move to...", fileselector_for_move_callback, SELECTION, NULL,
    { { 'V', GDK_CONTROL_MASK },
      { 'R', GDK_CONTROL_MASK }
    } },
  { "Delete", delete_callback, SELECTION, NULL,
    { { 'X', GDK_CONTROL_MASK },
      { 'D', GDK_MOD1_MASK }
    } },
  { "Map script on...", selection_map_script_callback, SELECTION, NULL,
    { { 'X', 0 },
      { '!', GDK_SHIFT_MASK }
    } },
  { "Map unix command on...", selection_map_unix_command_callback, SELECTION, NULL,
    { { 'X', GDK_SHIFT_MASK },
      { '|', GDK_SHIFT_MASK }
    } },
  MENU_SEPARATOR,
  { "Make new directory", fileselector_for_mkdir_callback, FALSE, NULL,
    { { 'D', GDK_SHIFT_MASK },
      { '+', GDK_SHIFT_MASK }
    } },
  { "Change directory", fileselector_for_chdir_callback, FALSE, NULL,
    { { 'C', 0 },
      { 'D', GDK_CONTROL_MASK }
    } },
  { "Update", update_callback, FALSE, NULL,
    { { 'R', 0 },
      { 'G', 0 }
    } },
  { "Change sort mode", toggle_sort_mode_callback, FALSE, NULL,
    { { 'S', 0 },
      { 'S', 0 }
    } },
  { "Remove all thumbnail files", purge_cache_file_callback, FALSE, NULL,
    { { 'R', GDK_MOD1_MASK },
      { 'P', GDK_MOD1_MASK }
    } },
  MENU_SEPARATOR,
  { "Help", help_callback, HIDDEN, NULL,
    { { '?', 0 },
      { 'H', GDK_CONTROL_MASK }
    } },
  { "Quit", gtkW_close_callback, FALSE, NULL,
    { { 'Q', 0 },
      { 'Q', GDK_CONTROL_MASK }
    } },
  /* Don't insert new commands after this line */
  { "Open menu", menu_show_callback, FALSE, NULL,
    { { 0, GDK_MOD1_MASK },
      { 'X', GDK_MOD1_MASK }
    } },
};

typedef struct
{
  guchar	*action;
  guchar	*condition;
  guchar	*behavior;
  gint		flush_left;
} help_table;

help_table
help_document [] =
{
  { NULL, "Gimp Users' Another SHell\nJan 10 1998", NULL, FALSE},
  { NULL, NULL, NULL, FALSE },
  { "ACTION", "CONDITION",
    "COMMAND", FALSE },
  { "Left-click", "on unselected image file",
    "Select only the image file", TRUE },
  { "Left-click", "on selected image file",
    "Open the selected image files", TRUE },
  { "Left-click", "on directory",
    "Jump to the directory", TRUE },
  { "Shift + Left-click", "on image file",
    "Add/delete the file to/from selection", TRUE },
  { "Control + Left-click", "on directory & active selection",
    "Copy selected files to the directory", TRUE },
  { "Shift + Left-click", "on directory & active selection",
    "Move selected files to the directory", TRUE },
  { "Middle-click", NULL,
    "Go to the next page of panel", TRUE },
  { "Shift + Middle-click", NULL,
    "Go to the pevious page of panel", TRUE },
  { "Right-click", "",
    "Open menu", TRUE },
  { NULL, NULL, NULL, FALSE },
  { NULL, "written by Shuji Narazaki <narazaki@inetq.or.jp>", NULL, FALSE},
};

GPlugInInfo PLUG_IN_INFO =
{
  NULL,				/* init_proc  */
  NULL,				/* quit_proc */
  query,			/* query_proc */
  run,				/* run_proc */
};

MAIN ()

static void
query ()
{
  static GParamDef args [] =
  {
    { PARAM_INT32, "run_mode", "Interactive, non-interactive"},
    { PARAM_STRING, "directory_name", "directory name used as the default directory" },
  };
  static GParamDef *return_vals = NULL;
  static int nargs = sizeof (args) / sizeof (args[0]);
  static int nreturn_vals = 0;

  gimp_install_procedure (PLUG_IN_NAME,
			  "Thumbnail-based directory browser",
			  "Thumbnail-based directory browser",
			  "Shuji Narazaki <narazaki@inetq.or.jp>",
			  "Shuji Narazaki",
			  "1997",
			  MENU_PATH,
			  "",
			  PROC_EXTENSION,
			  nargs, nreturn_vals,
			  args, return_vals);
}

static void
run (char	*name,
     int	nparams,
     GParam	*param,
     int	*nreturn_vals,
     GParam	**return_vals)
{
  GParam	*values;
  GStatusType	status = STATUS_EXECUTION_ERROR;
  GRunModeType	run_mode;

  values = g_new (GParam, 1);
  run_mode = param[0].data.d_int32;

  *nreturn_vals = 1;
  *return_vals = values;

  values[0].type = PARAM_STATUS;
  values[0].data.d_status = status;

  switch (run_mode)
    {
    case RUN_INTERACTIVE:
    case RUN_WITH_LAST_VALS:
      VAL.sort_type = SORT_BY_NAME;
      VAL.last_dir_name[0] = 0;
      strcpy (VAL.mapped_command, "xv -root -maxpect -quit {}");
      gimp_get_data (PLUG_IN_NAME, &VAL);

      directory_cache_table = g_hash_table_new (g_string_hash, g_string_equal);
      selection_list = g_list_alloc ();
      if (0 < strlen (VAL.last_dir_name))
	chdir (VAL.last_dir_name);

      text_image = gimp_image_new (THUMBNAIL_WIDTH, THUMBNAIL_THEIGHT, RGB);
      text_bg = gimp_layer_new (text_image, "Background",
				THUMBNAIL_WIDTH, THUMBNAIL_THEIGHT,
				RGB_IMAGE, 100, NORMAL_MODE);
      gimp_image_disable_undo (text_image);
      gimp_image_add_layer (text_image, text_bg, 0);
      {
	guchar	*val = directory_icon_data;
	gint	x, y;

	directory_icon =
	  (guchar *) g_malloc (directory_icon_width * directory_icon_height * 3);

	for (y = 0; y < directory_icon_height ; y++)
	  for (x = 0; x < directory_icon_width; x++)
	    {				/*  don't delete this blaket! */
	      HEADER_PIXEL (val,
			    (directory_icon + (y * directory_icon_width + x) * 3));
	    }
      }
      DIALOG ();
      gimp_displays_flush();
      gimp_set_data (PLUG_IN_NAME, &VAL, sizeof (VALS));
      status = STATUS_SUCCESS;
      break;
    case RUN_NONINTERACTIVE:
      break;
    }
  {
    GDrawable	*drw;

    text_bg = gimp_image_get_active_layer (text_image);
    if (0 < (drw = gimp_drawable_get (text_bg)))
      {
	gimp_drawable_detach (drw);
      }
    gimp_image_delete (text_image);
  }
  g_hash_table_destroy (directory_cache_table);
  g_list_free (selection_list);
  g_free (directory_icon);

  values[0].type = PARAM_STATUS;
  values[0].data.d_status = status;
}

/* guash classes */

static void
guash_parse_gimprc ()
{
  guchar buf[256];

  gtkW_parse_gimprc_string ("guash-confirmor", buf);
  if ((strcmp (buf, "no") == 0) ||
      (strcmp (buf, "No") == 0) ||
      (strcmp (buf, "NO") == 0))
    use_confirmor = 0;
  else
    use_confirmor = 1;

  gtkW_parse_gimprc_string ("guash-command", buf);
  if ((strcmp (buf, "all") == 0) ||
      (strcmp (buf, "All") == 0) ||
      (strcmp (buf, "ALL") == 0))
    hidden_features = 1;
  else
    hidden_features = 0;

  gtkW_parse_gimprc_string ("guash-keybindings", buf);
  if (((strcmp (buf, "emacs") == 0) ||
       (strcmp (buf, "Emacs") == 0) ||
       (strcmp (buf, "EMACS") == 0)))
    binding_style = 1;
  else
    binding_style = 0;

#ifdef STRICT_MM
  {
    gint	tmp = 0;

    tmp = gtkW_parse_gimprc_gint ("guash-directory-cache-size",
				  directory_cache_table_max_size);
    if (0 < tmp)
      directory_cache_table_max_size = CLAMP (tmp, 8, 100);
    else
      directory_cache_table_max_size = -1;
  }
#endif
}

GList *
selection_reset ()
{
  if (! cwd_cache)
    return NULL;
  thumbnail_panel_clear_selection_frame ();
  g_list_free (selection_list);
  selection_list = NULL;
  thumbnail_panel_update_selection_buttons ();
  gtk_widget_queue_draw (thumbnail_panel);
  selection_kind = FALSE;

  return selection_list;
}

GList *
selection_add (gint index)
{
  switch (selection_kind)
    {
    case FALSE:
      selection_kind = IMAGE;
      break;
    case IMAGE:
      break;
    default:
      selection_reset ();
      selection_kind = IMAGE;
      break;
    }

  if (! selection_member_p (index))
    {
      selection_list = g_list_append (selection_list, (gpointer) index);
      thumbnail_panel_draw_selection_frame ();
      thumbnail_panel_update_selection_buttons ();
    }
  return selection_list;
}

static GList *
selection_delete (gint index)
{
  switch (selection_kind)
    {
    case FALSE:
      selection_kind = IMAGE;
      break;
    case IMAGE:
      break;
    default:
      selection_reset ();
      selection_kind = IMAGE;
      break;
    }

  if (selection_member_p (index))
    {
      thumbnail_panel_clear_selection_frame ();
      selection_list = g_list_remove (selection_list, (gpointer) index);
      if (selection_is_active ())
	thumbnail_panel_draw_selection_frame ();
      else
	thumbnail_panel_set_info (NULL);
      thumbnail_panel_update_selection_buttons ();
    }
  return selection_list;
}

static gint
selection_is_active ()
{
  return (selection_list != NULL);
}

static gint
selection_length ()
{
  return g_list_length (selection_list);
}

static gint
selection_member_p (gint index)
{
  return (g_list_find (selection_list, (gpointer) index) != NULL);
}

static gint
selection_reverse_member (gint index)
{
  gint	flag = selection_member_p (index);

  switch (selection_kind)
    {
    case FALSE:
      selection_kind = IMAGE;
      break;
    case IMAGE:
      break;
    default:
      selection_reset ();
      selection_kind = IMAGE;
      break;
    }

  if (flag)
    selection_delete (index);
  else
    selection_add (index);
  return (! flag);
}

static void
selection_map_script ()
{
  gint		retvals;
  FILE		*file;
  GString	*script_file = NULL;
  GList		*l;
  guchar	buf[LINE_BUF_SIZE];

  if (getenv ("HOME") != NULL)
    {
      script_file = g_string_new (g_strdup (getenv ("HOME")));
      g_string_append (script_file, "/.gimp/scripts/");
    }
  else
    {
      struct passwd *pwd_ptr;

      /* the loser doesn't have $HOME set */
      setpwent ();

      pwd_ptr = getpwuid (getuid ());
      if(pwd_ptr)
	{
	  script_file = g_string_new (g_strdup (pwd_ptr->pw_dir));
	  g_string_append (script_file, "/.gimp/scripts/");
	}
    }
  if ((script_file == NULL) || (os_file_kind (script_file->str, TRUE) != DIRECTORY))
    {
      gtkW_message_dialog (TRUE, "Abort execution: you don't have ~/.gimp/script.");
      return;
    }

  g_string_append (script_file, GUASH_SELECTION_SCM);

  if ((file = fopen (script_file->str, "w")) == NULL)
    {
      sprintf (buf, " Fail to save to %s ", script_file->str);
      gtkW_message_dialog (TRUE, buf); /* info is too short to display filename */
      g_string_free (script_file, TRUE);
    }
  fprintf (file, ";; This file was generated by guash automatically.\n");
  fprintf (file, ";; You can delete me without trouble.\n\n");
  fprintf (file, "(define %%guash-selection\n");
  fprintf (file, "  '(\n");
  l = selection_list;
  while (l)
    {
      gint	selected_id = (gint) l->data;
      cache_entry	*selected;
      gint	ppos, i;

      selected = directory_cache_get_nth (selected_id);
      fprintf (file,
	       "    (\"%s/%s\" \"%s\" \"",
	       cwd_cache->name, selected->name, /* full path */
	       cwd_cache->name /* directory that does not end with slash */
	       );

      ppos = file_get_last_period_index (selected->name);

      for (i = 0; i < ppos; i++)
	fprintf (file, "%c", selected->name[i]);

      fprintf (file, "\" \"");

      for (i = ppos + 1; i < strlen (selected->name); i++)
	fprintf (file, "%c", selected->name[i]);

      fprintf (file, "\")\n");

      l = l->next;
    }
  fprintf (file, "    ))\n");
  fclose (file);

  /* script_fu_refresh is a temp_proc which returns NULL always */
  gimp_run_procedure ("script_fu_refresh",
		      &retvals,
		      PARAM_INT32, 0,
		      PARAM_END);
  /* And all script-fus return NULL always */
  gimp_run_procedure ("script-fu-map-on-guash-selection",
		      &retvals,
		      PARAM_INT32, 0,
		      PARAM_STRING, "",
		      PARAM_INT32, TRUE,
		      PARAM_INT32, FALSE,
		      PARAM_INT32, TRUE,
		      PARAM_END);
  g_string_free (script_file, TRUE);
}

static void
selection_map_unix_command ()
{
  GList		*l;
  guchar	template[LINE_BUF_SIZE];

  gtkW_query_box ("Guash: mapping unix command on selection",
		  "Type command to map on the selected image files. \n\
 \"{}\" in the command string is replaced to each filename without directory. \n\
 \"%%\" in the command string is replaced to the directory part of each file. \n\
 - SAMPLES -\n\
 rm .xvpics/{} \n\
 uuencode {} {} > /tmp/{}.uu; chmod go-r /tmp/{}.uu",
		  VAL.mapped_command, template);
  if (strlen (template) == 0)
    return;
  l = selection_list;
  while (l)
    {
      guchar *command;
      gint	selected_id = (gint) l->data;
      cache_entry	*selected;

      selected = directory_cache_get_nth (selected_id);
      command = generate_unix_command (template, cwd_cache->name, selected->name);
      if (command)
	{
	  system (command);
	  g_free (command);
	}
      l = l->next;
    }
  strcpy (VAL.mapped_command, template);
}

static guchar *
generate_unix_command (guchar *template, guchar *directory, guchar *filename)
{
  guchar	*new = NULL, *tmp;
  gint		index = -1;
  gint		directory_p = FALSE;
  gint		i = 0;

  for (tmp = template; *tmp; tmp++, i++)
    {
      if ((*tmp == '{') && (*(tmp + 1) == '}'))
	{
	  directory_p = FALSE;
	  index = i;
	  break;
	}
      if ((*tmp == '%') && (*(tmp + 1) == '%'))
	{
	  directory_p = TRUE;
	  index = i;
	  break;
	}
    }
  if (-1 < index)
    {
      if (directory_p)
	new = g_malloc (strlen (template) + strlen (directory) + 1);
      else
	new = g_malloc (strlen (template)
			+ strlen (directory) + strlen (filename) + 1);
      i = index;
      memcpy (new, template, i);
      if (directory_p)
	{
	  memcpy (new + i, directory, strlen (directory));
	  i += strlen (directory);
	  /* new[i++] = '/'; */
	}
      else
	{
	  memcpy (new + i, filename, strlen (filename));
	  i += strlen (filename);
	}
      memcpy (new + i, template + index + 2, strlen (template) - index - 2);
      i += strlen (template) - index - 2;
      new[i] = 0;

      tmp = generate_unix_command (new, directory, filename);
      g_free (new);
      return tmp;
    }
  else
    {
      return g_strdup (template);
    }
}

static void
selection_open_files ()
{
  GList	*list = selection_list;
  cache_entry	*selected;

  if (selection_kind != IMAGE)
    return;

  while (list)
    {
      gint	selected_id = (gint) list->data;

      selected = directory_cache_get_nth (selected_id);
      cache_open_image_file (selected);

      list = list->next;
    }
}

static void
selection_copy_files_to (guchar *pathname)
{
  guchar	info[256];
  gint		kind = NOT_EXIST;
  gint		success = FALSE;
  GList		*list = selection_list;

  if (selection_kind != IMAGE)
    return;

  while (list)
    {
      gint selected_id = (gint) list->data;

      kind = os_file_kind (pathname, TRUE);

      if ((kind == NOT_EXIST) || (kind == DIRECTORY))
	{
	  cache_entry	*selected;
	  GString	*name, *new;

	  selected = directory_cache_get_nth (selected_id);
	  name = g_string_new (cwd_cache->name);
	  g_string_append (name, "/");
	  g_string_append (name, selected->name);
	  new = g_string_new (pathname);
	  if (kind == DIRECTORY)
	    {
	      if (pathname [strlen (pathname) - 1] != '/')
		g_string_append (new, "/");
	      g_string_append (new, selected->name);
	    }
	  if (os_file_kind (new->str, TRUE) != NOT_EXIST)
	    {
	      sprintf (info, "%s already exists.", new->str);
	      gtkW_message_dialog (TRUE, info);
	    }
	  else if (image_file_copy (name->str, new->str) == TRUE)
	    success = TRUE;

	  g_string_free (name, TRUE);
	  g_string_free (new, TRUE);
	}
      else if (kind == REGFILE)
	{
	  sprintf (info, "%s already exists.", pathname);
	  gtkW_message_dialog (TRUE, info);
	}
      list = list->next;
    }
  if ((success == TRUE) && (kind == DIRECTORY))
    directory_cache_force_to_purge (pathname);

  selection_reset ();

  if (success)
    directory_cache_set_to_cwd (TRUE);

  if (success)
    thumbnail_panel_set_info ("Selected files are copied.");
  else
    thumbnail_panel_set_info ("Fail to copy.");
}

static void
selection_delete_files ()
{
  GList		*list = selection_list;

  if (selection_kind != IMAGE)
    return;

  while (list)
    {
      gint	selected_id = (gint) list->data;
      cache_entry	*selected;
      GString	*name;

      selected = directory_cache_get_nth (selected_id);
      name = g_string_new (g_strdup (cwd_cache->name));
      g_string_append (name, "/");
      g_string_append (name, selected->name);
      if (unlink (name->str) == 0)
	{
	  guchar *cache_name;

	  cache_name = file_build_cache_file_name (name->str);
	  unlink (cache_name);
	  g_free (cache_name);
	}
      g_string_free (name, TRUE);
      selected->deleted = TRUE;

      list = list->next;
    }
  selection_reset ();
  directory_cache_set_to_cwd (GC);
  thumbnail_panel_set_info ("Selected files are deleted.");
}

static void
selection_move_files_to (guchar *pathname)
{
  guchar	info[256];
  gint		kind = NOT_EXIST;
  gint		success = FALSE;
  GList		*list = selection_list;

  if (selection_kind != IMAGE)
    return;
  kind = os_file_kind (pathname, TRUE);
  if ((1 < selection_length ()) && (kind != DIRECTORY))
    {
      gtkW_message_dialog (TRUE, "The destination should be a directory");
      return;
    }

  while (list)
    {
      gint	selected_id = (gint) list->data;

      kind = os_file_kind (pathname, TRUE); /* might be overwrited */
      if ((kind == NOT_EXIST) || (kind == DIRECTORY))
	{
	  cache_entry	*selected;
	  GString	*name, *new;

	  selected = directory_cache_get_nth (selected_id);
	  name = g_string_new (g_strdup (cwd_cache->name));
	  g_string_append (name, "/");
	  g_string_append (name, selected->name);
	  new = g_string_new (g_strdup (pathname));

	  if (kind == DIRECTORY)
	    {
	      if (pathname [strlen (pathname) - 1] != '/')
		g_string_append (new, "/");
	      g_string_append (new, selected->name);
	    }
	  if (image_file_move (name->str, new->str))
	    {
	      selected->deleted = TRUE;
	      success = TRUE;
	    }
	  g_string_free (name, TRUE);
	  g_string_free (new, TRUE);
	}
      else if (kind == REGFILE)
	{
	  sprintf (info, "%s already exists.", pathname);
	  gtkW_message_dialog (TRUE, info);
	}
      list = list->next;
    }
  selection_reset ();

  if (success == TRUE)
    {
      if ((kind == DIRECTORY) && (strcmp (cwd_cache->name, pathname) != 0))
	{
	  directory_cache_force_to_purge (pathname);
	  directory_cache_set_to_cwd (GC);
	}
      else
	directory_cache_set_to_cwd (TRUE);
    }

  if (success == TRUE)
    thumbnail_panel_set_info ("Selected files are moved.");
  else
    thumbnail_panel_set_info ("Fail to move.");
}

static gint
thumbnail_panel_show_cache (cache_entry *cache)
{
  gint	y;
  gint	sx, sy;
  gint	ox, oy;

  thumbnail_panel_last_index++;
  if (nthumbails_in_page <= thumbnail_panel_last_index)
    return FALSE;
  sx = INDEX_TO_X (thumbnail_panel_last_index);
  sy = INDEX_TO_Y (thumbnail_panel_last_index);
  ox = CLAMP ((THUMBNAIL_WIDTH - cache->width) / 2, 0, thumbnail_panel_width);

  if (cache->directory_p)
    {
      ox = (THUMBNAIL_WIDTH - directory_icon_width) / 2;
      oy = (THUMBNAIL_HEIGHT + THUMBNAIL_THEIGHT - directory_icon_height) / 2;
      /* reder from directory_icon */
      for (y = 0; y < THUMBNAIL_HEIGHT; y++)
	gtk_preview_draw_row (GTK_PREVIEW (thumbnail_panel),
			      directory_icon + (y * directory_icon_width * 3),
			      sx + ox, sy + y + oy,
			      directory_icon_width);
      /* render its name */
      ox = CLAMP ((THUMBNAIL_WIDTH - cache->width) / 2, 0, thumbnail_panel_width);
      oy = THUMBNAIL_HEIGHT;
      for (y = 0; y < THUMBNAIL_THEIGHT; y++)
	gtk_preview_draw_row (GTK_PREVIEW (thumbnail_panel),
			      cache->data + (y * cache->width * 3),
			      sx + ox, sy + y + oy,
			      CLAMP (cache->width, 0, THUMBNAIL_WIDTH));
    }
  else
    {
      oy = (THUMBNAIL_HEIGHT + THUMBNAIL_THEIGHT - cache->height) / 2;
      for (y = 0; y < cache->height; y++)
	gtk_preview_draw_row (GTK_PREVIEW (thumbnail_panel),
			      cache->data + (y * cache->width * 3),
			      sx + ox, sy + y + oy,
			      /* CLAMP (cache->width, 0, THUMBNAIL_WIDTH) */
			      cache->width);
    }
  {
    GdkRectangle current_row;

    current_row.x = sx;
    current_row.y = sy;
    current_row.width = THUMBNAIL_WIDTH;
    current_row.height = THUMBNAIL_HEIGHT + THUMBNAIL_THEIGHT;
    gtk_widget_draw (thumbnail_panel, &current_row);
  }
  return TRUE;
}

static void
thumbnail_panel_banner ()
{
  guchar	data[3] = { 255, 255, 255 };
  guchar	*val = banner_data;
  guchar	*banner;
  gint		x, y;
  gint		sx, sy;
  gint		ey;

  for (y = 0; y < thumbnail_panel_height; y++)
    for (x = 0; x < thumbnail_panel_width; x++)
      gtk_preview_draw_row (GTK_PREVIEW (thumbnail_panel), data, x, y, 1);

  banner = g_malloc (banner_height * banner_width * 3);
  for (y = 0; y < banner_height ; y++)
    for (x = 0; x < banner_width; x++)
      {				/*  don't delete this blaket! */
	HEADER_PIXEL (val, (banner + (y * banner_width + x) * 3));
      }
  sx = MAX (0, ((thumbnail_panel_width - banner_width) / 2));
  sy = MAX (0, ((thumbnail_panel_height - banner_height) * 2 / 3));
  ey = MIN (thumbnail_panel_height - sy, banner_height);
  for (y = 0; y < banner_height; y++)
    gtk_preview_draw_row (GTK_PREVIEW (thumbnail_panel),
			  banner + (y * banner_width * 3),
			  sx,  sy + y,
			  banner_width);
  g_free (banner);
}

static void
thumbnail_panel_clear ()
{
  gint	x, y;
  guchar	data[3] = { 255, 255, 255 };

  for (y = 0; y < thumbnail_panel_height; y++)
    for (x = 0; x < thumbnail_panel_width; x++)
      gtk_preview_draw_row (GTK_PREVIEW (thumbnail_panel), data, x, y, 1);
  gtk_widget_draw (thumbnail_panel, NULL);
}

static void
thumbnail_panel_set_directory_info ()
{
  gint	maxlen = DIRECTORY_LABEL_MAXLEN;
  gint	from = 0;

  cwd_cache->display_page = CLAMP (cwd_cache->display_page, 0, cwd_cache->npage - 1);
  thumbnail_panel_page1 = cwd_cache->display_page + 1;

  if (1 < cwd_cache->npage)
    maxlen -= 7;
  if (maxlen < strlen (cwd_cache->name))
    from = strlen (cwd_cache->name) - maxlen;

  if (1 < cwd_cache->npage)
    sprintf (thumbnail_panel_directory_str,
	     "%s%s (%d/%d by %s) ",
	     (from == 0) ? " " : "*",
	     cwd_cache->name + from,
	     cwd_cache->display_page + 1,
	     cwd_cache->npage,
	     ((VAL.sort_type == SORT_BY_NAME) ? "name" : "date"));
  else
    sprintf (thumbnail_panel_directory_str,
	     "%s%s (by %s)",
	     (from == 0) ? " " : "*",
	     cwd_cache->name + from,
	     ((VAL.sort_type == SORT_BY_NAME) ? "name" : "date"));
  gtk_label_set (GTK_LABEL (cwd_label), thumbnail_panel_directory_str);
}

static void
thumbnail_panel_set_info (guchar *name)
{
  if (name == NULL)
    gtk_label_set (GTK_LABEL (file_property), thumbnail_panel_info_default);
  else
    {
      if (THUMBNAIL_INFO_MAXLEN < strlen (name))
	{
	  guchar	*str = NULL;

	  str = (guchar *) g_malloc (THUMBNAIL_INFO_MAXLEN + 1);
	  memcpy (str, name, THUMBNAIL_INFO_MAXLEN);
	  str[THUMBNAIL_INFO_MAXLEN] = 0;
	  gtk_label_set (GTK_LABEL (file_property), str);
	  g_free (str);
	}
      else
	gtk_label_set (GTK_LABEL (file_property), name);
    }
  /* Since info field should be updated immediately, I use gtk_widget_draw ()
     instead of gtk_widget_queue_draw () here. */
  gtk_widget_draw (file_property, NULL);
}

static void
thumbnail_panel_set_info_default ()
{
  if (! cwd_cache)
    strcpy (thumbnail_panel_info_default, "");

  if (1 == cwd_cache->ndir)
    sprintf (thumbnail_panel_info_default, "%d %s",
	     cwd_cache->nimage,
	     ((1 < cwd_cache->nimage) ? "images" : "image"));
  else
    sprintf (thumbnail_panel_info_default, "%d %s and %d %s",
	     cwd_cache->nimage,
	     ((1 < cwd_cache->nimage) ? "images" : "image"),
	     (cwd_cache->ndir - 1),
	     ((2 < cwd_cache->ndir) ? "subdirectories" : "subdirectory"));
}

static void
thumbnail_panel_draw_frame (gint index, guchar val)
{
  gint	x, y;
  gint	xi, yi;
  guchar data[3];
  gint	border = 2;

  index -= cwd_cache->display_page * nthumbails_in_page;
  x = INDEX_TO_X (index);
  y = INDEX_TO_Y (index);
  data[0] = data[1] = data[2] = val;
  data[0] = 255;

  /* draw top */
  for (yi = y - border; yi < y; yi++)
    for (xi = x - border; xi < x + THUMBNAIL_WIDTH + border; xi++)
      gtk_preview_draw_row (GTK_PREVIEW (thumbnail_panel), data, xi, yi, 1);

  /* draw left */
  for (yi = y - border;
       yi < y + THUMBNAIL_HEIGHT + THUMBNAIL_THEIGHT + border;
       yi++)
    for (xi = x - border; xi < x; xi++)
      gtk_preview_draw_row (GTK_PREVIEW (thumbnail_panel), data, xi, yi, 1);
  /* draw right */
  for (yi = y - border;
       yi < y + THUMBNAIL_HEIGHT + THUMBNAIL_THEIGHT + border;
       yi++)
    for (xi = x + THUMBNAIL_WIDTH; xi < x + THUMBNAIL_WIDTH + border; xi++)
      gtk_preview_draw_row (GTK_PREVIEW (thumbnail_panel), data, xi, yi, 1);
  /* draw bottom */
  for (yi = y + THUMBNAIL_HEIGHT + THUMBNAIL_THEIGHT;
       yi < y + THUMBNAIL_HEIGHT + THUMBNAIL_THEIGHT + border;
       yi++)
    for (xi = x - border; xi < x + THUMBNAIL_WIDTH + border; xi++)
      gtk_preview_draw_row (GTK_PREVIEW (thumbnail_panel), data, xi, yi, 1);
}

static void
thumbnail_panel_draw_selection_frame ()
{
  GList	*tmp = selection_list;
  gint	id;

  while (tmp)
    {
      id = (gint) tmp->data;
      if ((cwd_cache->display_page * nthumbails_in_page <= id)
	  && (id < (cwd_cache->display_page + 1) * nthumbails_in_page))
	thumbnail_panel_draw_frame (id, 0);
      tmp = tmp->next;
    }
}

static void
thumbnail_panel_clear_selection_frame ()
{
  GList	*tmp = selection_list;
  gint	id;

  while (tmp)
    {
      id = (gint) tmp->data;
      if ((cwd_cache->display_page * nthumbails_in_page <= id)
	  && (id < (cwd_cache->display_page + 1) * nthumbails_in_page))
	thumbnail_panel_draw_frame (id, 255);
      tmp = tmp->next;
    }
}

static void
thumbnail_panel_update_selection_buttons ()
{
  gint	ncommand = sizeof (image_commands) / sizeof (image_commands[0]);
  gint	flag = selection_is_active ();
  gint	i;

  for (i = 0; i < ncommand; i++)
    if (image_commands[i].sensitive & SELECTION)
      gtk_widget_set_sensitive (image_commands[i].widget, flag);
}

static void
thumbnail_panel_update_scroller ()
{
  gint	ncommand = sizeof (image_commands) / sizeof (image_commands[0]);
  gint	i;
  gint	flags[2];

  if (! cwd_cache)
    return;
  if ((widget_pointer[0].widget == NULL) || (widget_pointer[0].updater == NULL))
    return;

  flags[0] = flags[1] = TRUE;	/* 0 for back(prev), 1 for next */
  if (cwd_cache->display_page == 0)
    flags[0] = FALSE;
  if (cwd_cache->npage <= 1 + cwd_cache->display_page)
    flags[1] = FALSE;

  if (((GtkAdjustment *) widget_pointer[0].widget)->upper != cwd_cache->npage + 1)
    {
      ((GtkAdjustment *) widget_pointer[0].widget)->upper = cwd_cache->npage + 1;
    }
  /* Changes of thumbnail_panel_page1 propagates here only if changed */

  /* Condition:
     (gint) (((GtkAdjustment *) widget_pointer[0].widget)->value)
       != cwd_cache->display_page
     is not sufficent condition. Sometimes fail to update.
  */
  (widget_pointer[0].updater) (widget_pointer);

  for (i = 0; i < NSCROLLER; i++)
    {
      gtk_widget_set_sensitive (widget_for_scroll[i],
				(1 < cwd_cache->npage));
      gtk_widget_queue_draw (widget_for_scroll[i]);
    }
  {
    gint	j = 1;

    for (i = 0; i < ncommand; i++)
      if (image_commands[i].sensitive & SCROLL)
	{
	  gtk_widget_set_sensitive (image_commands[i].widget, flags[j]);
	  j = 0;
	}
  }
}

static gint
thumbnail_panel_update ()
{
  gint	index, max;

  if (! cwd_cache)
    return FALSE;

  thumbnail_panel_set_directory_info ();
  thumbnail_panel_clear ();
  gtkW_preview_force_to_update (thumbnail_panel);

  max = MIN ((1 + cwd_cache->display_page)*nthumbails_in_page,
	     cwd_cache->ndir + cwd_cache->nimage);
  thumbnail_panel_last_index = -1;

  for (index = cwd_cache->display_page * nthumbails_in_page; index < max; index++)
    thumbnail_panel_show_cache (directory_cache_get_nth (index));

  gtk_window_set_title (GTK_WINDOW (dlg), SHORT_NAME);
  thumbnail_panel_draw_selection_frame ();
  thumbnail_panel_set_info_default ();
  if (! selection_is_active ())
    thumbnail_panel_set_info (NULL);

  thumbnail_panel_update_sensitive_menu ();

  gdk_flush ();
  return TRUE;
}

static void
thumbnail_panel_update_sensitive_menu ()
{
  thumbnail_panel_update_selection_buttons ();
  thumbnail_panel_update_scroller ();
}

static GtkWidget *
thumbnail_panel_create_menu ()
{
  GtkWidget	*menu_item;
  gint	ncommand = sizeof (image_commands) / sizeof (image_commands[0]);
  gint	i;

  if (! thumbnail_panel_menu)
    {
      GtkAcceleratorTable *accelerator_table;

      thumbnail_panel_menu = gtk_menu_new ();
      accelerator_table = gtk_accelerator_table_new ();
      gtk_menu_set_accelerator_table (GTK_MENU (thumbnail_panel_menu),
				      accelerator_table);
      for (i = 0; i < ncommand; i++)
	{
	  if ((image_commands[i].sensitive == HIDDEN) && (! hidden_features))
	    continue;
	  if (image_commands[i].label)
	    menu_item = gtk_menu_item_new_with_label (image_commands[i].label);
	  else
	    menu_item = gtk_menu_item_new ();
	  image_commands[i].widget = menu_item;
	  gtk_menu_append (GTK_MENU (thumbnail_panel_menu), menu_item);
	  gtk_signal_connect (GTK_OBJECT (menu_item), "activate",
			      (GtkSignalFunc) image_commands[i].command,
			      (gpointer) image_commands[i].label);
	  if (image_commands[i].label
	      && (image_commands[i].binding[binding_style].key != 0))
	    gtk_widget_install_accelerator (menu_item,
					    accelerator_table,
					    "activate",
					    image_commands[i].binding[binding_style].key,
					    image_commands[i].binding[binding_style].mod);

	  if (i != ncommand - 1)
	    gtk_widget_show (menu_item);
	  if (! image_commands[i].label) /* separator should not focus on */
	    gtk_widget_set_sensitive (menu_item, FALSE);
	}
      gtk_window_add_accelerator_table (GTK_WINDOW (dlg),accelerator_table);
    }
  /* Don't invoke thumbnail_panel_update_sensitive_menu () here; some
     widgets are not created yet. */
  return thumbnail_panel_menu;
}

static gint
thumbnail_panel_move_focus (gint offset)
{
  gint	index = -1;
  gint	new_index = -1;
  gint	new_page = cwd_cache->display_page;
  cache_entry *cache;

  if (selection_is_active ())
    {
      GList *last = g_list_last (selection_list);

      index = (gint) last->data;
      new_index = index + offset;
    }
  else
    if (0 < cwd_cache->nimage)
      {
	if (0 < offset)
	  new_index = cwd_cache->ndir;
	else
	  new_index = cwd_cache->ndir + cwd_cache->nimage - 1;
      }
  if ((cwd_cache->ndir <= new_index)
      && (new_index < cwd_cache->ndir + cwd_cache->nimage))
    {
      if (0 < index)
	selection_delete (index);
      new_page = new_index / (ncol_of_thumbnail * nrow_of_thumbnail);
      if (new_page != cwd_cache->display_page)
	{
	  cwd_cache->display_page = new_page;
	  thumbnail_panel_update ();
	}
      selection_add (new_index);
      cache = directory_cache_get_nth (new_index);
      thumbnail_panel_set_info (cache->info);
    }
  else
    thumbnail_panel_set_info (NULL);

  thumbnail_panel_update_sensitive_menu ();
  return new_index;
}

directory_cache *
directory_cache_set_to_cwd (gint purge)
{
  gpointer	ptr = NULL;
#ifdef __USE_GNU
  guchar	*dir = get_current_dir_name ();
#else
  guchar	dir[LINE_BUF_SIZE];

  getcwd (dir, LINE_BUF_SIZE - 1);
#endif

  if ((ptr = g_hash_table_lookup (directory_cache_table, (gpointer) dir)))
    {
      cwd_cache = (directory_cache *) ptr;

      if ((purge == TRUE) || (cwd_cache->purged))
	{
	  cwd_cache->purged = FALSE;
	  directory_cache_update ();
	}
      else if (purge == GC)
	{
	  directory_cache_garbage_collect ();
	}
      else
	{
	  selection_reset ();
	  thumbnail_panel_update ();
	}
    }
  else
    {
#ifdef STRICT_MM
      if ((0 < directory_cache_table_max_size) &&
	  (directory_cache_table_max_size <= directory_cache_table_size))
	{
	  directory_cache	*tmp = NULL;

	  /* First, set cwd_cache to NULL. This means 1st entry is not found */
	  cwd_cache = NULL;
	  /* Second, set the 1st entry to cwd_cache by recycler */
	  g_hash_table_foreach (directory_cache_table,
				directory_cache_table_recycler,
				(gpointer) &tmp);
	  if (cwd_cache == NULL)
	    {
	      printf ("Warning: cache recycle failed (%s)!\n", tmp->name);
	      cwd_cache = tmp;
	    }
	  /* Then, delete it from directory_cache_table */
	  g_hash_table_remove (directory_cache_table, cwd_cache->name);
	  /* Finally, set the name as the new key */
	  strcpy (cwd_cache->name, dir);
	  cwd_cache->birth_index = directory_cache_table_max_size;
	}
      else
	{
	  if (0 < directory_cache_table_max_size)
	    directory_cache_table_size++;

	  cwd_cache = directory_cache_new (dir);
	}
#else
      cwd_cache = directory_cache_new (dir);
#endif
      g_hash_table_insert (directory_cache_table,
			   cwd_cache->name,
			   (gpointer) cwd_cache);

      directory_cache_update ();
    }
  strcpy (VAL.last_dir_name, cwd_cache->name);
  return cwd_cache;
}

#ifdef STRICT_MM
static void
directory_cache_table_recycler (gpointer key,
				gpointer value,
				gpointer user_data)
{
  directory_cache *cache, **fail_safe;

  cache = (directory_cache *) value;
  fail_safe = (directory_cache **)user_data;

 /* This is for fail safe */
  if ((cwd_cache == NULL) && (*fail_safe == NULL))
    *fail_safe = cache;

  if ((--(cache->birth_index) == 0) && (cwd_cache == NULL))
    cwd_cache = (directory_cache *) value;
}
#endif

directory_cache *
directory_cache_new (guchar *name)
{
  guchar	*cache_dir;
  directory_cache *p;

  p = (directory_cache *) g_malloc (sizeof (directory_cache));
  strcpy (p->name, name);
#ifdef STRICT_MM
  p->birth_index = directory_cache_table_size;
#endif

  cache_dir = file_build_cache_directory_name (name);
  switch (os_file_kind (cache_dir, TRUE))
    {
    case NOT_EXIST:
      p->savable = UNKNOWN;
      p->filed = FALSE;
      break;
    case DIRECTORY:
      p->savable = TRUE;
      p->filed = TRUE;
      break;
    default:
      p->savable = FALSE;
      p->filed = FALSE;
      break;
    }
  g_free (cache_dir);

  p->max_nimage = 16;
  p->image = (cache_entry *) g_malloc (p->max_nimage * sizeof (cache_entry));
  memset (p->image, 0, p->max_nimage * sizeof (cache_entry));
  p->nimage = 0;

  p->max_ndir = 2;
  p->dir = (cache_entry *) g_malloc (p->max_ndir * sizeof (cache_entry));
  memset (p->dir, 0, p->max_ndir * sizeof (cache_entry));
  p->ndir = 0;

  p->purged = FALSE;
  p->display_page = 0;
  return p;
}

static gint
directory_cache_set_npage ()
{
  gint	page;

  page = (cwd_cache->ndir + cwd_cache->nimage) / nthumbails_in_page;
  if ((cwd_cache->ndir + cwd_cache->nimage) % nthumbails_in_page != 0)
    page++;
  return cwd_cache->npage = page;
}

cache_entry *
directory_cache_get_nth (gint index)
{
  if (index < cwd_cache->ndir)
    return (cwd_cache->dir) + index;
  else
    return (cwd_cache->image) + (index - cwd_cache->ndir);
}

static gint
directory_cache_update ()
{
  /* side-effect: selection is reset  */
  struct dirent *dentry;
  struct dirent **dentry_list;
  guchar	old_fg[3], old_bg[3], counter[256], path[LINE_BUF_SIZE];
  gint		nentry, index, index_to_update;
#ifdef NO_CORE
  gint		no_core;
#endif

  gimp_palette_get_foreground (old_fg, old_fg + 1, old_fg + 2);
  gimp_palette_get_background (old_bg, old_bg + 1, old_bg + 2);

  cache_set_to_normal_color ();

  if (VAL.last_dir_name[0] != 0)
    {
      thumbnail_panel_clear ();
      gtkW_preview_force_to_update (thumbnail_panel);
    }
  if (cwd_cache->filed)
    gtk_window_set_title (GTK_WINDOW (dlg),
			  "Guash: making with cache files...");
  else
    gtk_window_set_title (GTK_WINDOW (dlg),
			  "Guash: wait a minute...");

  selection_reset ();
  cwd_cache->nimage = cwd_cache->ndir = 0;
  index_to_update = (1 + cwd_cache->display_page) * nthumbails_in_page;

  directory_cache_add_directory ("..", DIRECTORY);

  nentry = scandir (cwd_cache->name, &dentry_list, &os_scandir_selector,
		    ((VAL.sort_type == SORT_BY_NAME)
		     ? &alphasort : &os_file_mtimesort));
  /* Are you a linux user? Did your compiler complain that:
     guash.c:1856: warning: pointer type mismatch in conditional expression
     I could not find error/warning-free declaration for my linux box (libc1).
     But guash works well with current declaration...

     typedef int (*__dir_compar_fn_t) __P ((
		__const struct dirent * __const *,
		__const struct dirent * __const *
		));
     What the hell is it? Declaration in the man page is not valid.
  */
  /* For eagar display, I use two pass scanning. */
  thumbnail_panel_set_info ("Scanning subdirectories...");
  gdk_flush ();

#ifdef NO_CORE
  /* before the traverse, check the existance of core */
  no_core = (os_file_kind ("core", TRUE) == NOT_EXIST);
#endif

  /* 1st path: for collecting directories */
  for (index = 0; index < nentry; index++)
    {
      gint	f_kind = NOT_EXIST;

      dentry = dentry_list[index];
      if (dentry->d_name[0] == '.')
	continue;

      f_kind = os_file_kind (dentry->d_name, FALSE);

      switch (f_kind)
	{
	case DIRECTORY:
	case SLNKDIR:
	  directory_cache_add_directory (dentry->d_name, f_kind);
	  break;
	default:
	  break;
	}
    }
  if (index_to_update < cwd_cache->ndir)
    {
      thumbnail_panel_update ();
      gtkW_preview_force_to_update (thumbnail_panel);
      /* stop further update */
      index_to_update = -1;
    }
 /* 2nd path: for loading image files */
  for (index = 0; index < nentry; free (dentry_list[index]), index++)
    {
      gint	f_kind = NOT_EXIST;

      sprintf (counter, "Reading the directory (%d/%d)... ", index, nentry);
      thumbnail_panel_set_info (counter);
      gdk_flush ();

      dentry = dentry_list[index];
      if (dentry->d_name[0] == '.')
	continue;

      sprintf (path, "%s/%s", cwd_cache->name, dentry->d_name);
      f_kind = os_file_kind (path, FALSE);

      switch (f_kind)
	{
	case REGFILE:
	case SLNKFILE:
	  {
	    gint32	i_id;

	    if ((i_id = cache_get_image (dentry->d_name, f_kind)) < 0)
	      continue;

	    directory_cache_add_image (dentry->d_name, i_id, f_kind);
	    gimp_image_delete (i_id);
	  }
	  break;
	default:
	  break;
	}
      if (index_to_update == cwd_cache->ndir + cwd_cache->nimage)
	{
	  thumbnail_panel_update ();
	  gtkW_preview_force_to_update (thumbnail_panel);
	  /* if the next entry is non-direcoty/image, another update is
	     occurred. */
	  index_to_update = -1;
	}
    }
  free (dentry_list);

#ifdef NO_CORE
  if (no_core && (os_file_kind ("core", TRUE) == REGFILE))
    unlink ("core");
#endif

  gimp_palette_set_foreground (old_fg[0], old_fg[1], old_fg[2]);
  gimp_palette_set_background (old_bg[0], old_bg[1], old_bg[2]);

  if (cwd_cache->npage <= cwd_cache->display_page)
    {
      cwd_cache->display_page = 0;
      thumbnail_panel_update ();
    }
  else if (cwd_cache->ndir + cwd_cache->nimage < index_to_update)
    thumbnail_panel_update ();
  else
    {
      /* Then thumbnail_panel doesn't know the true number of dir & image. */
      thumbnail_panel_set_directory_info ();
      thumbnail_panel_set_info_default ();
      thumbnail_panel_update_sensitive_menu ();
      thumbnail_panel_set_info (NULL);
    }

  return TRUE;
}

static gint
directory_cache_garbage_collect ()
{
  /* side-effect: selection is reset  */
  gint		old_nimage, new_nimage, index;
  cache_entry	*chunk;
  guchar	*tmp;

  new_nimage = old_nimage = cwd_cache->nimage;
  chunk = cwd_cache->image;
  index = 0;
  while (index < new_nimage)
    {
      if (chunk[index].deleted == TRUE)
	{
	  new_nimage--;
	  tmp = chunk[index].data;
	  memcpy (chunk + index,
		  chunk + (index + 1),
		  sizeof (cache_entry) * (new_nimage - index));
	  chunk[new_nimage].data = tmp;
	}
      else
	index++;
    }
  cwd_cache->nimage = new_nimage;
  directory_cache_set_npage ();

  selection_reset ();
  if (cwd_cache->npage <= cwd_cache->display_page)
    cwd_cache->display_page = 0;
  thumbnail_panel_update ();
  return TRUE;
}

static gint
directory_cache_add_directory (guchar *name, gint f_kind)
{
  cache_entry	*cache;
  gint	x, y;

  if (cwd_cache->max_ndir <= cwd_cache->ndir)
    {
      cwd_cache->max_ndir *= 2;
      cwd_cache->dir =
	(cache_entry *) g_realloc ((gpointer) cwd_cache->dir,
				   cwd_cache->max_ndir * sizeof (cache_entry));
      memset (cwd_cache->dir + cwd_cache->ndir, 0,
	      (cwd_cache->max_ndir - cwd_cache->ndir) * sizeof (cache_entry));
    }

  cache = &cwd_cache->dir[cwd_cache->ndir++];
  cache->deleted = FALSE;
  if (cache->data == 0)
    cache->data = (guchar *) g_malloc (THUMBNAIL_WIDTH * THUMBNAIL_THEIGHT * 3);

  cache->directory_p = TRUE;
  strcpy (cache->name, name);
  if (f_kind == SLNKDIR)
#ifdef VERBOSE_LINK
    sprintf (cache->info, "(link) %s", name);
#else
    strcpy (cache->info, name);
#endif
  else
    strcpy (cache->info, name);
  cache->width = THUMBNAIL_WIDTH;
  cache->height = THUMBNAIL_THEIGHT;

  for (y = 0; y < cache->height; y++)
    for (x = 0; x < cache->width; x++)
      {
	cache->data[(y * THUMBNAIL_WIDTH + x) * 3    ] = 255;
	cache->data[(y * THUMBNAIL_WIDTH + x) * 3 + 1] = 255;
	cache->data[(y * THUMBNAIL_WIDTH + x) * 3 + 2] = 255;
      }

  if ((strlen (cache->name) == 2) && (strcmp ("..", cache->name) == 0))
    cache_render_text (".. (parent)", cache);
  else
    {
      if (f_kind == SLNKDIR)
	cache_set_to_symlink_color ();
      cache_render_text (cache->info, cache);
      if (f_kind == SLNKDIR)
	cache_set_to_normal_color ();
    }
  directory_cache_set_npage ();
  return TRUE;
}

static gint
directory_cache_add_image (guchar *name, gint i_id, gint f_kind)
{
  gint		x, y;
  gint		i_width, i_height;
  gint		drawable_id;
  GDrawable	*drawable;
  GPixelRgn	src_rgn;
  guchar	*src;
  gpointer	pr;
  gint		gap = 0;
  cache_entry	*cache;

  if (cwd_cache->max_nimage <= cwd_cache->nimage)
    {
      cwd_cache->max_nimage *= 2;
      cwd_cache->image
	= (cache_entry *) g_realloc ((gpointer) cwd_cache->image,
				     cwd_cache->max_nimage * sizeof (cache_entry));
      memset (cwd_cache->image + cwd_cache->nimage, 0,
	      (cwd_cache->max_nimage - cwd_cache->nimage) * sizeof (cache_entry));
    }

  cache = &cwd_cache->image[cwd_cache->nimage++];
  cache->deleted = FALSE;
  if (cache->data == 0)
    cache->data = (guchar *) g_malloc (THUMBNAIL_WIDTH * (THUMBNAIL_HEIGHT + THUMBNAIL_THEIGHT) * 3);

  drawable_id = gimp_image_get_active_layer (i_id);
  drawable = gimp_drawable_get (drawable_id);
  i_width = gimp_layer_width (drawable_id);
  i_height = gimp_layer_height (drawable_id);

  gap = (gimp_drawable_has_alpha (drawable_id)) ? 1 : 0;
  gimp_pixel_rgn_init (&src_rgn, drawable, 0, 0, i_width, i_height, FALSE, FALSE);
  pr = gimp_pixel_rgns_register (1, &src_rgn);

  for (; pr != NULL; pr = gimp_pixel_rgns_process (pr))
    {
      gint	gx = src_rgn.x;
      gint	gy = src_rgn.y;

      for (y = 0; y < src_rgn.h; y++)
	{
	  src = src_rgn.data + y * src_rgn.rowstride;

	  for (x = 0; x < src_rgn.w; x++)
	    {
	      guchar	*ch = src + (x * (3 + gap));
	      gint	offset = (gy + y) * i_width * 3 + (gx + x) * 3;

	      cache->data[offset    ] = *ch;
	      cache->data[offset + 1] = *(ch + 1);
	      cache->data[offset + 2] = *(ch + 2);
	    }
	}
    }
  gimp_drawable_detach (drawable);

  cache->directory_p = FALSE;
  strcpy (cache->name, name);
  if (f_kind == SLNKFILE)
#ifdef VERBOSE_LINK
    sprintf (cache->info, "(link) %s",
	     gimp_layer_get_name (gimp_image_get_active_layer (i_id)));
#else
    strcpy (cache->info,
	    gimp_layer_get_name (gimp_image_get_active_layer (i_id)));
#endif
  else
    strcpy (cache->info,
	    gimp_layer_get_name (gimp_image_get_active_layer (i_id)));
  cache->width = i_width;
  cache->height = i_height;

  directory_cache_set_npage ();
  return TRUE;
}

static gint
directory_cache_force_to_purge (guchar *name)
{
  gpointer	ptr = NULL;
  guchar	cwd[LINE_BUF_SIZE];
  guchar	dir[LINE_BUF_SIZE];

  getcwd (cwd, LINE_BUF_SIZE - 1);
  if (chdir (name) == -1)
    return FALSE;
  getcwd (dir, LINE_BUF_SIZE - 1);
  chdir (cwd);

  if ((ptr = g_hash_table_lookup (directory_cache_table, (gpointer) dir)))
    {
      directory_cache	*dcache = (directory_cache *) ptr;

      dcache->purged = TRUE;
      return TRUE;
    }
  else
    return FALSE;
}

static gint
directory_cache_make_cache_directory (directory_cache *dentry)
{
  guchar	*cache_dir;
  mode_t	mode = NEW_DIRECTORY_MODE;

  cache_dir = file_build_cache_directory_name (dentry->name);

  if (os_make_directory (cache_dir, mode) == -1)
    {
      switch (errno)
	{
	case EEXIST:
	  dentry->savable = TRUE;
	  dentry->filed = TRUE;
	  break;
	default:
	  dentry->savable = FALSE;
	  dentry->filed = FALSE;
	  break;
	}
      g_free (cache_dir);
      return FALSE;
    }
  else
    {
      dentry->savable = TRUE;
      dentry->filed = TRUE;

      g_free (cache_dir);
      return TRUE;
    }
}

static gint
directory_cache_delete_invalid_cache_files (gint force)
{
  guchar	*path;
  DIR	*dir;
  struct dirent *dentry;
  gint	index = 0;

  path = file_build_cache_directory_name (cwd_cache->name);
  dir = opendir (path);
  rewinddir (dir);

  gtk_window_set_title (GTK_WINDOW (dlg), "Guash: clear cache files...");
  for (; (dentry = readdir (dir)); index++)
    {
      GString	*file;

      file = g_string_new (g_strdup (path));
      g_string_append (file, "/");
      g_string_append (file, dentry->d_name);

      if (os_file_kind (file->str, TRUE) == REGFILE)
	{
#ifdef THUMBNAIL_FORMAT_IS_XCF
	  if (force ||
	      ((cache_file_is_valid (file->str, THUMBNAIL_SUFFIX) == FALSE)
	       && (cache_file_is_valid (file->str, "") == FALSE)))
#else
	  if (force || (cache_file_is_valid (file->str, "") == FALSE))
#endif
	    unlink (file->str);
#ifdef THUMBNAIL_FORMAT_IS_XCF
	  g_string_append (file, THUMBNAIL_SUFFIX);
	  if (os_file_kind (file->str, TRUE) == REGFILE)
	    unlink (file->str);
#endif
	}
      g_string_free (file, TRUE);
    }
  g_free (path);
  closedir (dir);
  if (! force)
    {
      cwd_cache->ndir = cwd_cache->nimage = 0;
      directory_cache_set_to_cwd (TRUE);
    }
  return TRUE;
}

GtkWidget *
directory_cache_create_parents_menu ()
{
  GtkWidget	*menu;
  gint		count, index;
  guchar	*cwd;
  GtkWidget	*menuitem;
  guchar	*label = NULL;

  menu = gtk_menu_new ();
  cwd = cwd_cache->name;

  label = (guchar *) g_malloc (2);
  label[0] = '/';
  label[1] = 0;
  menuitem = gtk_menu_item_new_with_label (label);
  gtk_menu_append (GTK_MENU (menu), menuitem);
  gtk_signal_connect (GTK_OBJECT (menuitem), "activate",
		      (GtkSignalFunc) menu_change_directory_callback,
		      (gpointer) label);
  gtk_widget_show (menuitem);

  for (count = index = 1; index < strlen (cwd); index++)
    if (cwd[index] == '/')
      {
	label = (guchar *) g_malloc (index);
	memcpy (label, cwd, index);
	label[index] = 0;

	menuitem = gtk_menu_item_new_with_label (label);
	gtk_menu_prepend (GTK_MENU (menu), menuitem);
	gtk_signal_connect (GTK_OBJECT (menuitem), "activate",
			    (GtkSignalFunc) menu_change_directory_callback,
			  (gpointer) label);
	gtk_widget_show (menuitem);
    }

  return menu;
}

static void
directory_cache_create_history_menu_foreach (gpointer key,
					     gpointer value,
					     gpointer user_data)
{
  GtkWidget	*menu_item;
  GtkWidget	*menu;

  menu = (GtkWidget *) user_data;

  menu_item = gtk_menu_item_new_with_label ((guchar *)key);
  gtk_menu_append (GTK_MENU (menu), menu_item);
  gtk_signal_connect (GTK_OBJECT (menu_item), "activate",
		      (GtkSignalFunc) menu_change_directory_callback,
		      key);
  gtk_widget_show (menu_item);
}

GtkWidget *
directory_cache_create_history_menu ()
{
  GtkWidget *menu;

  menu = gtk_menu_new ();

  g_hash_table_foreach (directory_cache_table,
			directory_cache_create_history_menu_foreach,
			menu);

  return menu;
}

static gint
cache_get_image (guchar *name, gint f_kind)
{
  gint	id = -1;

  if (cwd_cache->filed)			/* check cache */
    {
      id = cache_get_image_from_file (name, "", f_kind);
#ifdef THUMBNAIL_FORMAT_IS_XCF
      if (id < 0)
	id = cache_get_image_from_file (name, THUMBNAIL_SUFFIX, f_kind);
#endif
    }
  if (id < 0)
    {
      id = cache_get_image_from_file (name, NULL, f_kind);
    }
  return id;
}

static gint
cache_get_image_from_file (guchar *name, guchar *suffix, gint f_kind)
{
  GParam*	return_vals;
  GDrawable	*text_layer = NULL;
  gint		retvals;
  gint		xv_thumbnail_p = FALSE;
  gint		f_size = 0;
  gint32	i_id, d_id, width, height, new_width, new_height;
  gint32	bg_layer;
  GString	*path;
  guchar	info[256];
  guchar	image_type[16];

  image_type[0] = 0;

  if (suffix == NULL)		/* original image file */
    path = g_string_new (cwd_cache->name);
  else				/* thumbnail file */
    path = g_string_new (file_build_cache_directory_name (cwd_cache->name));

  g_string_append (path, "/");
  g_string_append (path, name);
  if ((suffix != NULL) && (0 < strlen (suffix)))
    g_string_append (path, suffix);

  if ((os_file_kind (path->str, TRUE)) == NOT_EXIST)
    {
      g_string_free (path, TRUE);
      return -1;
    }

  if ((suffix != NULL) && (strlen (suffix) == 0))
    {
      /* invoke embbed xvpict loader */
      i_id = load_xvpict_image (path->str);
      if (i_id < 0)
	{
	  g_string_free (path, TRUE);
	  return i_id;
	}
      else
	{
	  xv_thumbnail_p = TRUE;
	  strcpy (info, gimp_layer_get_name (gimp_image_get_active_layer (i_id)));
	}
    }
  else
    {
      return_vals = gimp_run_procedure ("gimp_file_load",
					&retvals,
					PARAM_INT32, 0,
					PARAM_STRING, path->str,
					PARAM_STRING, path->str,
					PARAM_END);
      f_size = os_file_size (path->str);
      if (return_vals[0].data.d_status != STATUS_SUCCESS)
	{
	  g_string_free (path, TRUE);
	  gimp_destroy_params (return_vals, retvals);
	  return -1;
	}
      else
	gimp_destroy_params (return_vals, retvals);
      i_id = return_vals[1].data.d_image;
    }
  g_string_free (path, TRUE);

  if (gimp_image_base_type (i_id) != RGB)
    {
      if (gimp_image_base_type (i_id) == INDEXED)
	strcpy (image_type, "Indexed file");
      else
	strcpy (image_type, "Gray file");

      return_vals = gimp_run_procedure ("gimp_convert_rgb",
					&retvals,
					PARAM_IMAGE, i_id,
					PARAM_END);
      gimp_destroy_params (return_vals, retvals);
    }
  else
    strcpy (image_type, "RGB file");

#ifdef THUMBNAIL_FORMAT_IS_XCF
  /* The Image gotten from xvpict or original file requires some
     transformations. */
  if ((suffix != NULL) && (0 < strlen (suffix)))
    {
      return i_id;			/* the case of xcf cache */
    }
#endif

  new_width = width = gimp_image_width (i_id);
  new_height = height = gimp_image_height (i_id);

  if (THUMBNAIL_HEIGHT < new_height)
    {
      new_width = new_width * THUMBNAIL_HEIGHT / new_height;
      new_height = THUMBNAIL_HEIGHT;
    }
  if (THUMBNAIL_WIDTH < new_width)
    {
      new_height = new_height * THUMBNAIL_WIDTH / new_width;
      new_width = THUMBNAIL_WIDTH;
    }
  if (new_width < 2) new_width = 2;
  if (new_height < 2) new_height = 2;
  gimp_image_flatten (i_id);
  d_id = gimp_image_get_active_layer (i_id);
  gimp_layer_scale (d_id, new_width, new_height, 0);
  gimp_image_resize (i_id, new_width, new_height, 0, 0);

 /* Now we can save image as an XV thumbnail.*/
  if (! xv_thumbnail_p)
    {
      sprintf (info, "%dx%d %s (%d bytes)", width, height, image_type, f_size);
      gimp_layer_set_name (d_id, info);

      if (cwd_cache->savable == UNKNOWN)
	directory_cache_make_cache_directory (cwd_cache);
      if (cwd_cache->savable == TRUE)
	{
	  if (cache_save_to_file (name, i_id) == TRUE)
	    cwd_cache->filed = TRUE;
	}
    }
  /* OK, let's make the compound thumbnail image! */

  /* First make background layer for both of thumbnail and filename.
     WARNING:
     To set white BG, a new layer should be inserted as the bottom layer.
     But don't copy the current buttom layer by gimp_layer_copy.
     It leaves 60 or 100 tiles internally.
     So gimp quits with a complaint message.
   */
  gimp_layer_add_alpha (d_id);
  bg_layer = gimp_layer_new (i_id, "White BG", new_width, new_height,
			     RGBA_IMAGE, 100, NORMAL_MODE);
  gimp_image_add_layer (i_id, bg_layer, 0);
  gimp_layer_set_offsets (bg_layer, 0, 0);
  gimp_layer_scale (bg_layer,
		    THUMBNAIL_WIDTH,
		    THUMBNAIL_HEIGHT + THUMBNAIL_THEIGHT - 6,
		    0);

  gimp_drawable_fill (bg_layer, TRANS_IMAGE_FILL);

  gimp_layer_set_offsets (d_id,
			  (THUMBNAIL_WIDTH - new_width) / 2,
			  (THUMBNAIL_HEIGHT - new_height) / 2);

  if (f_kind == SLNKFILE)
    cache_set_to_symlink_color ();

  return_vals = gimp_run_procedure ("gimp_text",
				    &retvals,
				    PARAM_IMAGE, i_id,
				    PARAM_DRAWABLE, d_id,
				    PARAM_FLOAT, (gfloat) 0,
				    PARAM_FLOAT, (gfloat) THUMBNAIL_HEIGHT,
				    PARAM_STRING, name,
				    PARAM_INT32, (gint32) 1,
				    PARAM_INT32, (gint32) 1,
				    PARAM_FLOAT, (gfloat) THUMBNAIL_FONT_SIZE,
				    PARAM_INT32, (gint32) 0,
				    PARAM_STRING, "*", /* foundary */
				    PARAM_STRING, THUMBNAIL_FONT_FAMILY,
				    PARAM_STRING, "*", /* weight */
				    PARAM_STRING, THUMBNAIL_FONT_SLANT,
				    PARAM_STRING, "*", /* set_width */
				    PARAM_STRING, "*", /* spacing */
				    PARAM_END);
    if (return_vals[0].data.d_status == STATUS_SUCCESS)
    {
      text_layer = gimp_drawable_get (return_vals[1].data.d_layer);

      if (text_layer->width < THUMBNAIL_WIDTH)
	gimp_layer_set_offsets (return_vals[1].data.d_layer,
				CLAMP (((THUMBNAIL_WIDTH - text_layer->width) / 2),
				       0, THUMBNAIL_WIDTH),
				THUMBNAIL_HEIGHT);
    }
  gimp_destroy_params (return_vals, retvals);
  if (f_kind == SLNKFILE)
    cache_set_to_normal_color ();

  gimp_image_resize (i_id,
		     THUMBNAIL_WIDTH,
		     THUMBNAIL_HEIGHT + THUMBNAIL_THEIGHT - 6,
		     0, 0);
  gimp_image_flatten (i_id);

#ifdef BALANCE
  gimp_drawable_detach (gimp_drawable_get (bg_layer));
  gimp_drawable_detach (gimp_drawable_get (d_id));
#endif
  if (text_layer) gimp_drawable_detach (text_layer);
  d_id = gimp_image_get_active_layer (i_id);
  if (! xv_thumbnail_p)
    sprintf (info, "%s %dx%d %s (%d bytes)",
	     name, width, height,
	     image_type, f_size);
  gimp_layer_set_name (d_id, info);

  return i_id;
}

static void
cache_set_to_symlink_color ()
{
  gimp_palette_set_foreground (46, 139, 87); /* This is Sea grean. */
  gimp_palette_set_background (255, 255, 255);
}

static void
cache_set_to_normal_color ()
{
  gimp_palette_set_foreground (0, 0, 0);
  gimp_palette_set_background (255, 255, 255);
}

static void
cache_render_text (guchar *str, cache_entry *cache)
{
  GParam*	return_vals;
  gint		retvals;
  gint	x, y;
  gint	i_width, i_height;
  gint		drawable_id;
  GDrawable	*drawable;
  GPixelRgn	src_rgn;
  guchar	*src;
  gpointer	pr;
  gint		gap = 0;

  return_vals = gimp_run_procedure ("gimp_selection_none",
				    &retvals,
				    PARAM_IMAGE, text_image,
				    PARAM_END);
  gimp_destroy_params (return_vals, retvals);
  gimp_drawable_fill (text_bg, BG_IMAGE_FILL);
  return_vals = gimp_run_procedure ("gimp_text",
				    &retvals,
				    PARAM_IMAGE, text_image,
				    PARAM_DRAWABLE, text_bg,
				    PARAM_FLOAT, (gfloat) 0,
				    PARAM_FLOAT, (gfloat) 0,
				    PARAM_STRING, str,
				    PARAM_INT32, (gint32) 1,
				    PARAM_INT32, (gint32) 1,
				    PARAM_FLOAT, (gfloat) THUMBNAIL_FONT_SIZE,
				    PARAM_INT32, (gint32) 0,
				    PARAM_STRING, "*", /* foundary */
				    PARAM_STRING, THUMBNAIL_FONT_FAMILY,
				    PARAM_STRING, "*", /* weight */
				    PARAM_STRING, THUMBNAIL_FONT_SLANT,
				    PARAM_STRING, "*", /* set_width */
				    PARAM_STRING, "*", /* spacing */
				    PARAM_END);

  if (return_vals[0].data.d_status == STATUS_SUCCESS)
    {
      GDrawable *drw = gimp_drawable_get (return_vals[1].data.d_layer);

      if (drw->width < THUMBNAIL_WIDTH)
	gimp_layer_set_offsets (return_vals[1].data.d_layer,
				(THUMBNAIL_WIDTH - drw->width) / 2,
				0);
      gimp_destroy_params (return_vals, retvals);
      gimp_image_flatten (text_image);
      gimp_drawable_detach (drw);
    }
  else
    {
      gimp_destroy_params (return_vals, retvals);
      return;
    }

  text_bg = drawable_id = gimp_image_get_active_layer (text_image);
  drawable = gimp_drawable_get (drawable_id);
  i_width = gimp_layer_width (drawable_id);
  i_height = gimp_layer_height (drawable_id);

  gap = (gimp_drawable_has_alpha (drawable_id)) ? 1 : 0;
  gimp_pixel_rgn_init (&src_rgn, drawable, 0, 0, i_width, i_height, FALSE, FALSE);
  pr = gimp_pixel_rgns_register (1, &src_rgn);
  for (; pr != NULL; pr = gimp_pixel_rgns_process (pr))
    {
      gint	gx = src_rgn.x;
      gint	gy = src_rgn.y;

      for (y = 0; y < src_rgn.h; y++)
	{
	  src = src_rgn.data + y * src_rgn.rowstride;

	  for (x = 0; (x < src_rgn.w) && (gx + x < THUMBNAIL_WIDTH); x++)
	    {
	      guchar	*ch = src + (x * (3 + gap));
	      gint	offset = (gy + y) * THUMBNAIL_WIDTH * 3 + (gx + x) * 3;
	      cache->data[offset    ] = *ch;
	      cache->data[offset + 1] = *(ch + 1);
	      cache->data[offset + 2] = *(ch + 2);
	    }
	}
    }
  gimp_drawable_detach (drawable);
}

static gint
cache_save_to_file (guchar *file_name, gint32 i_id)
{
  GParam	*return_vals = NULL;
  GString	*dir, *s;
  gint	retvals;

  if (cwd_cache->savable == TRUE)
    {
      dir = g_string_new (file_build_cache_directory_name (cwd_cache->name));
      g_string_append (dir, "/");

      s = g_string_new (dir->str);
      g_string_append (s, file_name);
#ifdef THUMBNAIL_FORMAT_IS_XCF
      g_string_append (s, THUMBNAIL_SUFFIX);
#endif
      gimp_image_set_filename (i_id, s->str);
      if (gimp_image_base_type (i_id) != RGB)
	{
	  return_vals = gimp_run_procedure ("gimp_convert_rgb",
					    &retvals,
					    PARAM_IMAGE, i_id,
					    PARAM_END);
	  gimp_destroy_params (return_vals, retvals);
	}
#ifdef THUMBNAIL_FORMAT_IS_XCF
      return_vals = gimp_run_procedure ("gimp_file_save",
					&retvals,
					PARAM_INT32, 0,
					PARAM_IMAGE, i_id,
					PARAM_DRAWABLE, gimp_image_get_active_layer (i_id),
					PARAM_STRING, s->str,
					PARAM_STRING, s->str,
					PARAM_END);
      gimp_destroy_params (return_vals, retvals);
#else
      save_xvpict_image (s->str, i_id, gimp_image_get_active_layer (i_id));
#endif
      g_string_free (s, TRUE);
      g_string_free (dir, TRUE);
      return TRUE;
    }
  else
    return FALSE;
}

static gint
cache_open_image_file (cache_entry *cache)
{
  GParam*	return_vals;
  gint		retvals;
  gint		i_id;
  GString *path;

  path = g_string_new (cwd_cache->name);
  g_string_append (path, "/");
  g_string_append (path, cache->name);

  return_vals = gimp_run_procedure ("gimp_file_load",
				    &retvals,
				    PARAM_INT32, 0,
				    PARAM_STRING, path->str,
				    PARAM_STRING, path->str,
				    PARAM_END);
  g_string_free (path, TRUE);

  if (return_vals[0].data.d_status != STATUS_SUCCESS)
    {
      gimp_destroy_params (return_vals, retvals);
      return -1;
    }
  else
    gimp_destroy_params (return_vals, retvals);

  i_id = return_vals[1].data.d_image;
  /* the following sequence is copied form app/fileops.c */
  /*  enable & clear all undo steps  */
  gimp_image_enable_undo (i_id);
  /*  set the image to clean  */
  gimp_image_clean_all (i_id);
  /*  display the image */
  gimp_display_new (i_id);
  return i_id;
}

static guchar *
file_get_canonical_name (guchar *name)
{
  guchar	cwd[LINE_BUF_SIZE];
  guchar	*dir;

  dir = g_malloc (LINE_BUF_SIZE);

  getcwd (cwd, LINE_BUF_SIZE - 1);
  if (chdir (name) == -1)
    return NULL;
  getcwd (dir, LINE_BUF_SIZE - 1);
  chdir (cwd);

  return dir;
}

static gint
file_get_last_slash_index (guchar *pathname)
{
  gint	index;
  gint	pname_len;

  pname_len = strlen (pathname);

  for (index = pname_len - 2; 0 <= index; index--)
    if (pathname[index] == '/') break;

  return index;
}

static gint
file_get_last_period_index (guchar *pathname)
{
  gint	index;
  gint	pname_len;

  pname_len = strlen (pathname);

  for (index = pname_len - 2; 0 <= index; index--)
    if (pathname[index] == '.') break;

  return index;
}

static guchar *
file_get_parent_directory (guchar *pathname)
{
  gint	pname_len, slash_pos;
  guchar	*dirname;

  pname_len = strlen (pathname);
  slash_pos = file_get_last_slash_index (pathname);

  dirname = (guchar *) g_malloc (slash_pos + 1);
  memcpy (dirname, pathname, slash_pos);
  dirname[slash_pos] = 0;

  return dirname;
}

static guchar *
file_get_filename (guchar *pathname)
{
  gint	pname_len, slash_pos;
  guchar	*filename;

  pname_len = strlen (pathname);
  slash_pos = file_get_last_slash_index (pathname);

  filename = (guchar *) g_malloc (pname_len - slash_pos);
  strcpy (filename, pathname + (slash_pos + 1));

  return filename;
}

/* foo/bar.baz => foo/bar.baz/.xvpics */
static guchar *
file_build_cache_directory_name (guchar *dir_name)
{
  gint	dir_len, cd_len, cp_len;
  guchar	*cache_path;
  guchar	*cache_dir = THUMBNAIL_DIR;

  dir_len = strlen (dir_name);
  cd_len = strlen (THUMBNAIL_DIR);
  cp_len = dir_len + cd_len;
  cache_path = (guchar *) g_malloc (cp_len + 1);

  memcpy (cache_path, dir_name, dir_len);
  memcpy (cache_path + dir_len, cache_dir, cd_len);
  cache_path[cp_len] = 0;

  return cache_path;
}

static guchar *
file_build_cache_file_name (guchar *filename)
{
  gint	index;
  gint	fname_len, cname_len;
  guchar	*cache_name;
  guchar	*cache_dir = THUMBNAIL_DIR;
#ifdef THUMBNAIL_FORMAT_IS_XCF
  guchar	*cache_suffix = THUMBNAIL_SUFFIX;
#endif

  fname_len = strlen (filename);
#ifdef THUMBNAIL_FORMAT_IS_XCF
  cname_len = fname_len + strlen (cache_dir) + strlen (cache_suffix);
#else
  cname_len = fname_len + strlen (cache_dir);
#endif
  index = file_get_last_slash_index (filename);

  cache_name = (guchar *) g_malloc (cname_len + 1);
  memcpy (cache_name, filename, index);
  memcpy (cache_name + index, cache_dir, strlen (cache_dir));
  memcpy (cache_name + (index + strlen (cache_dir)),
	  filename + index,
	  fname_len - index);
#ifdef THUMBNAIL_FORMAT_IS_XCF
  memcpy (cache_name + (cname_len - strlen (cache_suffix)),
	  cache_suffix,
	  strlen (cache_suffix));
#endif
  cache_name [cname_len] = 0;
  return cache_name;
}

static gint
file_mkdir (guchar *pathname)
{
  guchar	info[256];
  mode_t	mode = NEW_DIRECTORY_MODE;
  gint		kind;

  kind = os_file_kind (pathname, TRUE);

  if (NOT_EXIST == kind)
    {
      if (os_make_directory (pathname, mode) != -1)
	{
	  guchar	*canonical;

	  canonical = file_get_canonical_name (pathname);
	  if (canonical != NULL)
	    {
	      guchar	*parent;
	      guchar	*filename;

	      parent = file_get_parent_directory (canonical);
	      filename = file_get_filename (canonical);

	      if (strcmp (cwd_cache->name, parent) == 0)
		{
		  guchar	old_fg[3], old_bg[3];

		  gimp_palette_get_foreground (old_fg, old_fg + 1, old_fg + 2);
		  gimp_palette_get_background (old_bg, old_bg + 1, old_bg + 2);
		  directory_cache_add_directory (filename, DIRECTORY);
		  gimp_palette_set_foreground (old_fg[0], old_fg[1], old_fg[2]);
		  gimp_palette_set_background (old_bg[0], old_bg[1], old_bg[2]);
		  thumbnail_panel_update ();
		}
	      else
		directory_cache_force_to_purge (parent);

	      g_free (parent);
	      g_free (filename);
	    }
	  else
	    printf ("file_mkdir: can't move to the directory\n");
	  g_free (canonical);
	}
    }
  else
    {
      sprintf (info, "%s already exists.", pathname);
      thumbnail_panel_set_info (info);
    }
  return (NOT_EXIST == kind);
}

static gint
os_copy_file (guchar *filename, guchar *newname)
{
  gint	from;
  gint	to;
  gint	size;
  guchar	buffer[LINE_BUF_SIZE];
  mode_t	mode = NEW_FILE_MODE;

  if ((from = open (filename, O_RDONLY)) == -1)
    {
      perror ("os_copy_file: can't open source file.");
      return FALSE;
    }
  if ((to = creat (newname, mode)) == -1)
    {
      close (from);
      perror ("os_copy_file: can't create destination file.");
      return FALSE;
    }
  while (0 < (size = read (from, buffer, LINE_BUF_SIZE - 1)))
    write (to, buffer, size);

  close (from);
  close (to);

  return TRUE;
}

static gint
os_file_kind (guchar *filename, gint shrink)
{
  struct stat ent_sbuf;
  gint	flag = TRUE;			/*  fail safe */
  gint	symlink = FALSE;
  gint	size = 0;
  guchar	fname[LINE_BUF_SIZE], tmp[LINE_BUF_SIZE];

  strcpy (fname, filename);
  while ((size = readlink (fname, tmp, LINE_BUF_SIZE - 1)) != -1)
    {
      /* fname is a symbolic link. */
      symlink = TRUE;
      tmp[size] = 0;
      strcpy (fname, tmp);
    }

  if (stat (fname, &ent_sbuf) == 0)
    {
      /* something exists */
      if (S_ISDIR (ent_sbuf.st_mode))
	flag = DIRECTORY;
      else if (S_ISREG (ent_sbuf.st_mode))
	flag = REGFILE;

      if ((! shrink) && symlink)
	{
	  switch (flag)
	    {
	    case REGFILE:
	      flag = SLNKFILE;
	      break;
	    case DIRECTORY:
	      flag = SLNKDIR;
	      break;
	    default:
	      break;
	    }
	}
    }
  else
    {
      /* error was occured */
      if (errno == ENOENT)
	flag = NOT_EXIST;
    }
  return flag;
}

static gint
os_file_mtimesort (struct dirent **a, struct dirent **b)
{
  /* if b is newer than a, return 1 */
  struct stat a_stat;
  struct stat b_stat;

  /* Warning: file existance check is omitted */
  stat ((*a)->d_name, &a_stat);
  stat ((*b)->d_name, &b_stat);

  if (a_stat.st_mtime < b_stat.st_mtime)
    return 1;
  else if (a_stat.st_mtime == b_stat.st_mtime)
    return 0;
  else
    return -1;
}

static gint
os_file_size (guchar *filename)
{
  struct stat ent_sbuf;
  gint	size = 0;
  gint	f_size = -1;
  guchar	fname[LINE_BUF_SIZE], tmp[LINE_BUF_SIZE];

  strcpy (fname, filename);
  while ((size = readlink (fname, tmp, LINE_BUF_SIZE - 1)) != -1)
    {
      /* fname is a symbolic link. */
      tmp[size] = 0;
      strcpy (fname, tmp);
    }
  if (stat (fname, &ent_sbuf) == 0)
    f_size = ent_sbuf.st_size;

  return f_size;
}


static gint
os_make_directory (guchar *pathname, gint mode)
{
  return mkdir (pathname, mode);
}

static gint
os_rename_file (guchar *filename, guchar *newname)
{
  return rename (filename, newname);
}

static int
os_scandir_selector (const struct dirent *dentry)
{
  return 1;
}

static gint
file_confirm_operation (guchar *operation_phrase)
{
  gint	flag = TRUE;

  if (! selection_is_active ())
    return flag;

  if (use_confirmor)
    {
      guchar buffer [LINE_BUF_SIZE];

      if (selection_length () == 1)
	{
	  cache_entry	*selected;

	  selected = directory_cache_get_nth ((gint) selection_list->data);
	  sprintf (buffer, " %s the selected image: %s? \n\
 This is NO WARRANTY program. ",
		   operation_phrase,
		   selected->name);
	}
      else
	{
	  sprintf (buffer, " %s the selected %d images? \n\
 This is NO WARRANTY program. ",
		   operation_phrase,
		   selection_length ());
	}

      if (! gtkW_confirmor_dialog (TRUE, buffer, FALSE))
	flag = FALSE;
    }

  return flag;
}

static gint
cache_file_is_valid (guchar *cache_name, guchar *suffix)
{
  guchar	*file_name;
  struct stat	image, cache;
  gint	stripper;
  gint	last_slash, parent_slash;

  stripper = strlen (THUMBNAIL_DIR) + strlen (suffix);
  file_name = (guchar *) g_malloc (strlen (cache_name) - stripper + 1);
  last_slash = file_get_last_slash_index (cache_name);
  parent_slash = last_slash - strlen (THUMBNAIL_DIR);
  /* /.../dir/  */
  memcpy (file_name, cache_name, parent_slash);
  /* /.../dir/image.file */
  memcpy (file_name + parent_slash,
	  cache_name + last_slash,
	  strlen (cache_name) - last_slash - strlen (suffix));
  /* /.../dir/image.file + TERMINATOR */
  file_name [strlen (cache_name) - stripper] = 0;

  if (stat (file_name, &image) != 0)
    return FALSE;
  if (stat (cache_name, &cache) != 0)
    return FALSE;

  return (image.st_mtime <= cache.st_mtime);
}

static gint
image_file_copy (guchar *filename, guchar *newname)
{
  if (os_file_kind (newname, TRUE) == NOT_EXIST)
    {
      guchar	*from_cache_name;
      guchar	*to_cache_name;
      mode_t	mode = NEW_DIRECTORY_MODE;

      if (os_copy_file (filename, newname) == FALSE)
	return FALSE;

      /* copy image cache */
      from_cache_name = file_build_cache_file_name (filename);
      if (os_file_kind (from_cache_name, TRUE) != REGFILE)
	{
	  g_free (from_cache_name);
	  return TRUE;
	}
      to_cache_name = file_build_cache_file_name (newname);
      {
	guchar	*to_dir_name;

	to_dir_name = file_get_parent_directory (to_cache_name);

	if (os_make_directory (to_dir_name, mode) == -1)
	  if (errno != EEXIST)
	    {
	      g_free (to_dir_name);
	      g_free (to_cache_name);
	      g_free (from_cache_name);
	      return TRUE;
	    }
	if (os_file_kind (to_dir_name, TRUE) != DIRECTORY)
	  {
	    g_free (to_dir_name);
	    g_free (to_cache_name);
	    g_free (from_cache_name);
	    return TRUE;
	  }
	g_free (to_dir_name);
      }
      os_copy_file (from_cache_name, to_cache_name);

      g_free (from_cache_name);
      g_free (to_cache_name);
      return TRUE;
    }
  else
    return FALSE;
}

static gint
image_file_delete (guchar *filename)
{
  return (unlink (filename) == 0);
}

static gint
image_file_move (guchar *filename, guchar *newname)
{
  if (os_file_kind (newname, TRUE) == NOT_EXIST)
    {
      if (os_rename_file (filename, newname) == 0)
	{
	  guchar	*from_cache_name = NULL;
	  guchar	*to_cache_name = NULL;
	  guchar	*cache_dir = NULL;
	  mode_t	mode = NEW_DIRECTORY_MODE;

	  from_cache_name = file_build_cache_file_name (filename);

	  if (os_file_kind (from_cache_name, TRUE) != REGFILE)
	    {
	      g_free (from_cache_name);
	      return TRUE;
	    }
	  to_cache_name = file_build_cache_file_name (newname);
	  cache_dir = file_get_parent_directory (to_cache_name);

	  if (os_make_directory (cache_dir, mode) == -1)
	    if (errno != EEXIST)
	      {
		g_free (cache_dir);
		g_free (to_cache_name);
		g_free (from_cache_name);
		return TRUE;
	      }
	  if (os_file_kind (cache_dir, TRUE) != DIRECTORY)
	    {
	      g_free (cache_dir);
	      g_free (to_cache_name);
	      g_free (from_cache_name);
	      return TRUE;
	    }
	  g_free (cache_dir);

	  if (os_rename_file (from_cache_name, to_cache_name) == -1)
	    {
	      perror ("rename cache file");
	      printf ("from %s to %s\n", from_cache_name, to_cache_name);
	    }

	  g_free (from_cache_name);
	  g_free (to_cache_name);
	  return TRUE;
	}
      else
	{
	  if (errno == EXDEV)
	    {
	      if (image_file_copy (filename, newname))
		{
		  image_file_delete (filename);
		  return TRUE;
		}
	      else
		return FALSE;
	    }
	  else
	    {
	      perror ("file_move");
	      return FALSE;
	    }
	}
    }
  else
    {
      GString *message;

      message = g_string_new ("NOP: ");
      g_string_append (message, newname);
      g_string_append (message, " already exists.");
      gtkW_message_dialog (TRUE, message->str);
      g_string_free (message, TRUE);

      return FALSE;
    }
}

static void
fileselector_set_last_value (guchar *pathname)
{
  if (os_file_kind (pathname, TRUE) == DIRECTORY)
    strcpy (fileselector_last_pathname, pathname);
  else
    strcpy (fileselector_last_pathname, file_get_parent_directory (pathname));
}

/* dialog stuff */
static int
DIALOG ()
{
  GtkWidget	*frame;
  GtkWidget	*hbox;
  GtkWidget	*ps_box;
  GtkWidget	*table;
  GtkWidget	*preview_frame;
  GtkWidget	*preview;
  GtkWidget	*button;
  gchar	**argv;
  gint	argc;

  argc = 1;
  argv = g_new (gchar *, 1);
  argv[0] = g_strdup (PLUG_IN_NAME);
  gtk_init (&argc, &argv);
  gtk_rc_parse (gimp_gtkrc ());

  ncol_of_thumbnail = gtkW_parse_gimprc_gint ("guash-ncol", ncol_of_thumbnail);
  ncol_of_thumbnail = CLAMP (ncol_of_thumbnail, 4, 10);
  nrow_of_thumbnail = gtkW_parse_gimprc_gint ("guash-nrow", nrow_of_thumbnail);
  nrow_of_thumbnail = CLAMP (nrow_of_thumbnail, 2, 10);
  nthumbails_in_page = ncol_of_thumbnail * nrow_of_thumbnail;
  thumbnail_panel_width = (ncol_of_thumbnail * (THUMBNAIL_WIDTH + THUMBNAIL_SEPARATOR) + THUMBNAIL_SEPARATOR);
  thumbnail_panel_height =(nrow_of_thumbnail * (THUMBNAIL_HEIGHT + THUMBNAIL_THEIGHT + THUMBNAIL_SEPARATOR) + THUMBNAIL_THEIGHT);

  dlg = gtk_dialog_new ();
  gtk_window_set_title (GTK_WINDOW (dlg), "Guash: initializing...");
  gtk_window_position (GTK_WINDOW (dlg), GTK_WIN_POS_MOUSE);
  gtk_signal_connect (GTK_OBJECT (dlg), "destroy",
		      (GtkSignalFunc) gtkW_close_callback, NULL);
  gtk_signal_connect (GTK_OBJECT (dlg), "key_press_event",
		      (GtkSignalFunc) cursor_event_handler,
		      NULL);

  /* Action Area */
  gtk_container_border_width (GTK_CONTAINER (GTK_DIALOG (dlg)->action_area), 0);
  button = gtk_button_new_with_label ("Help");
  GTK_WIDGET_SET_FLAGS (button, GTK_CAN_DEFAULT);
  gtk_signal_connect (GTK_OBJECT (button), "clicked",
		      (GtkSignalFunc) help_callback, dlg);
  gtk_box_pack_start (GTK_BOX (GTK_DIALOG (dlg)->action_area), button,
		      TRUE, TRUE, 0);
  gtk_widget_show (button);

  button = gtk_button_new_with_label ("Close");
  GTK_WIDGET_SET_FLAGS (button, GTK_CAN_DEFAULT);
  gtk_signal_connect_object (GTK_OBJECT (button), "clicked",
			     (GtkSignalFunc) gtkW_close_callback,
			     GTK_OBJECT(dlg));
  gtk_box_pack_start (GTK_BOX (GTK_DIALOG (dlg)->action_area), button,
		      TRUE, TRUE, 0);
  gtk_widget_grab_default (button);
  gtk_widget_show (button);

  hbox = gtkW_hbox_new ((GTK_DIALOG (dlg)->vbox));
  frame = gtkW_frame_new (hbox, NULL);
  table = gtkW_table_new (frame, 3, 6);
  gtk_container_border_width (GTK_CONTAINER (table), 0);

  button = gtkW_table_add_button (table, "Jump", 5, 6, 0,
			     (GtkSignalFunc) directory_jump_callback,
			     NULL);
  gtk_widget_set_usize (button, JUMP_BUTTON_WIDTH, 0);
  gdk_set_use_xshm (gimp_use_xshm ());
  gtk_preview_set_gamma (gimp_gamma ());
  gtk_preview_set_install_cmap (gimp_install_cmap ());
  {
    guchar *color_cube;
    color_cube = gimp_color_cube ();
    gtk_preview_set_color_cube (color_cube[0], color_cube[1],
				color_cube[2], color_cube[3]);
  }
  gtk_widget_set_default_visual (gtk_preview_get_visual ());
  gtk_widget_set_default_colormap (gtk_preview_get_cmap ());

  preview = gtk_preview_new (GTK_PREVIEW_COLOR);
  gtk_preview_size (GTK_PREVIEW (preview), thumbnail_panel_width, thumbnail_panel_height);
  gtk_widget_set_events (preview, EVENT_MASK);
  gtk_signal_connect (GTK_OBJECT (preview), "event",
		      (GtkSignalFunc) preview_event_handler,
		      NULL);
  preview_frame = gtkW_frame_new (NULL, NULL);
  gtkW_frame_shadow_type = GTK_SHADOW_ETCHED_IN;
  gtk_widget_set_usize (preview_frame, 0, 0);

  gtk_container_border_width (GTK_CONTAINER (preview_frame), 0);
  gtk_container_add (GTK_CONTAINER (preview_frame), preview);

  gtkW_ivscroll_entry_new (&widget_for_scroll[0],
			   &widget_for_scroll[1],
			   (GtkSignalFunc) gtkW_iscroll_update,
			   (GtkSignalFunc) gtkW_ientry_update,
			   &thumbnail_panel_page1,
			   1, 5, 1, &widget_pointer[0]);
  gtk_widget_set_usize (widget_for_scroll[0], SCROLLBAR_WIDTH, thumbnail_panel_height);
  gtk_widget_set_usize (widget_for_scroll[1], JUMP_BUTTON_WIDTH, 0);
  gtk_widget_show (widget_for_scroll[0]);
  gtk_widget_show (widget_for_scroll[1]);

  ps_box = gtkW_hbox_new (NULL);
  gtk_box_pack_start (GTK_BOX (ps_box), preview_frame, TRUE, TRUE, 0);
  gtk_box_pack_start (GTK_BOX (ps_box), widget_for_scroll[0], TRUE, TRUE, 0);
  gtk_table_attach (GTK_TABLE (table), ps_box, 0, 6, 1, 2,
		    GTK_FILL, GTK_FILL|GTK_EXPAND, 0, 0);

#ifdef USE_DND
  gtk_signal_connect (GTK_OBJECT (preview),
		      "drag_request_event",
		      (GtkSignalFunc) dnd_drag_move_request,
		      preview);
  gtk_widget_dnd_drag_set (preview, TRUE, sel_ops_keys, 1);
  gtk_signal_connect (GTK_OBJECT (preview),
		      "drop_data_available_event",
		      (GtkSignalFunc) dnd_drop,
		      preview);
  gtk_widget_dnd_drop_set (preview, TRUE, sel_ops_keys, 1, FALSE);
#endif USE_DND
  gtk_widget_show (preview);
  thumbnail_panel = preview;

  file_property = gtk_label_new ("Gimp Users' Another SHell: initializing...");
  gtk_widget_set_usize (file_property, thumbnail_panel_width - JUMP_BUTTON_WIDTH, 0);
  gtk_misc_set_alignment (GTK_MISC (file_property), 0.1, 1.0);
  gtk_table_attach (GTK_TABLE (table), file_property, 0, 5, 2, 3,
		    GTK_FILL|GTK_EXPAND, 0, 0, 0);
  gtk_widget_show (file_property);
  gtk_table_attach (GTK_TABLE (table), widget_for_scroll[1], 5, 6, 2, 3,
		    GTK_FILL|GTK_EXPAND, 0, 0, 0);
  if (VAL.last_dir_name[0] == 0)
    thumbnail_panel_banner ();
  else
    thumbnail_panel_clear ();

  cwd_label = gtk_label_new ("");
  gtk_misc_set_alignment (GTK_MISC (cwd_label), 0.1, 0.5);
  gtk_table_attach (GTK_TABLE (table), cwd_label, 0, 5, 0, 1,
		    GTK_FILL|GTK_EXPAND, 0, 0, 0);
  gtk_widget_set_usize (cwd_label, thumbnail_panel_width - JUMP_BUTTON_WIDTH, 0);
  gtk_widget_show (cwd_label);
  gtk_widget_show (dlg);

  gtk_window_set_title (GTK_WINDOW (dlg), SHORT_NAME);

  gtk_timeout_add (((VAL.last_dir_name[0] == 0) ? 1000: 100),
		   timer_initialize_thumbnail_panel,
		   NULL);

  gtk_main ();
  gdk_flush ();
  return 1;
}

static gint
timer_initialize_thumbnail_panel (gpointer data)
{
  guash_parse_gimprc ();
  thumbnail_panel_create_menu ();
  directory_cache_set_to_cwd (FALSE);
  return FALSE;
}

static gint
about_dialog ()
{
  GtkWidget	*a_dlg;
  GtkWidget	*button;
  GtkWidget	*frame;
  GtkWidget	*hbox;
  GtkWidget	*table;
  GtkWidget	*hseparator;
  gint		index, i, nhelp;
  gint		align;

  nhelp = sizeof (help_document) / sizeof (help_document[0]);

  a_dlg = gtk_dialog_new ();
  gtk_window_set_title (GTK_WINDOW (a_dlg), "Guash help");
  gtk_window_position (GTK_WINDOW (a_dlg), GTK_WIN_POS_MOUSE);
  /*
  gtk_signal_connect (GTK_OBJECT (a_dlg), "destroy",
		      (GtkSignalFunc) gtkW_close_callback, NULL);
  */
  gtk_container_border_width (GTK_CONTAINER (GTK_DIALOG (a_dlg)->action_area),
			      gtkW_border_width);

  button = gtk_button_new_with_label ("Close");
  GTK_WIDGET_SET_FLAGS (button, GTK_CAN_DEFAULT);
  gtk_signal_connect_object (GTK_OBJECT (button), "clicked",
			     (GtkSignalFunc) gtk_widget_destroy,
			     GTK_OBJECT(a_dlg));
  gtk_box_pack_start (GTK_BOX (GTK_DIALOG (a_dlg)->action_area), button,
		      TRUE, TRUE, 0);
  gtk_widget_grab_default (button);
  gtk_widget_show (button);

  hbox = gtkW_hbox_new ((GTK_DIALOG (a_dlg)->vbox));
  frame = gtkW_frame_new (hbox, NULL);

  table = gtkW_table_new (frame, nhelp + 2, 3);
  index = 0;
  align = gtkW_align_x;
  gtkW_align_x = GTK_EXPAND|GTK_FILL;

  for (i = 0; i < nhelp; i++, index++)
    {
      if (help_document[i].action)
	{
	  gtkW_table_add_label (table, help_document[i].action,
				0, 1, index, help_document[i].flush_left);
	  if (help_document[i].condition)
	    gtkW_table_add_label (table, help_document[i].condition,
				  1, 2, index, help_document[i].flush_left);
	  gtkW_table_add_label (table, help_document[i].behavior,
				2, 3, index, help_document[i].flush_left);
	}
      else
	{
	  if (help_document[i].condition)
	    {
	      gtkW_table_add_label (table, help_document[i].condition,
				    0, 3, index, FALSE);
	    }
	  else			/* hseparator */
	    {
	      hseparator = gtk_hseparator_new ();
	      gtk_table_attach (GTK_TABLE (table), hseparator,
				0, 3, index, index + 1,
				GTK_FILL, GTK_FILL,
				gtkW_border_width, 2 * gtkW_border_width);
	      gtk_widget_show (hseparator);
	    }
	}
    }
  gtkW_align_x = align;
  gtk_widget_show (a_dlg);

  gdk_flush ();
  return 1;
}

static void
menu_show_callback (GtkWidget *widget, gpointer data)
{
  gtk_menu_popup (GTK_MENU (thumbnail_panel_menu), NULL, NULL, NULL, NULL, 1, 0);
}

static void
menu_change_directory_callback (GtkWidget *widget, gpointer data)
{
  if (data != NULL)
    {
      chdir ((guchar *) data);
      directory_cache_set_to_cwd (FALSE);
    }
}

static void
directory_jump_callback (GtkWidget *widget, gpointer data)
{
  GtkWidget	*jump_menu, *parents_menu, *history_menu, *menu_item;

  jump_menu = gtk_menu_new ();

  parents_menu = gtk_menu_item_new_with_label ("Parents");
  gtk_menu_append (GTK_MENU (jump_menu), parents_menu);
  gtk_menu_item_set_submenu (GTK_MENU_ITEM (parents_menu),
			     directory_cache_create_parents_menu ());
  gtk_widget_show (parents_menu);

  history_menu = gtk_menu_item_new_with_label ("History");
  gtk_menu_append (GTK_MENU (jump_menu), history_menu);
  gtk_menu_item_set_submenu (GTK_MENU_ITEM (history_menu),
			     directory_cache_create_history_menu ());
  gtk_widget_show (history_menu);

  menu_item = gtk_menu_item_new_with_label ("To ...");
  gtk_menu_append (GTK_MENU (jump_menu), menu_item);
  gtk_signal_connect (GTK_OBJECT (menu_item), "activate",
		      (GtkSignalFunc) fileselector_for_chdir_callback,
		      NULL);
  gtk_widget_show (menu_item);

  menu_item = gtk_menu_item_new_with_label ("Close menu");
  gtk_menu_append (GTK_MENU (jump_menu), menu_item);
  gtk_signal_connect (GTK_OBJECT (menu_item), "activate",
		      (GtkSignalFunc) NULL,
		      NULL);
  gtk_widget_show (menu_item);

  gtk_menu_popup (GTK_MENU (jump_menu), NULL, NULL, NULL, NULL, 1, 0);
}

static void
forward_callback (GtkWidget *widget, gpointer data)
{
  thumbnail_panel_move_focus (1);
}

static void
select_and_forward_callback (GtkWidget *widget, gpointer data)
{
  gint	index = -1;
  cache_entry *cache;

  if (selection_is_active ())
    {
      GList *last = g_list_last (selection_list);

      index = (gint) last->data;

      if (index < cwd_cache->ndir + cwd_cache->nimage)
	index++;
      else
	index = -1;
    }
  else
    if (0 < cwd_cache->nimage)
      index = cwd_cache->ndir;

  if ((cwd_cache->ndir <= index)
      && (index < cwd_cache->ndir + cwd_cache->nimage))
    {
      selection_add (index);
      cache = directory_cache_get_nth (index);
    thumbnail_panel_set_info (cache->info);
    }
  else
    thumbnail_panel_set_info (NULL);

  thumbnail_panel_update_sensitive_menu ();
}

static void
backward_callback (GtkWidget *widget, gpointer data)
{
  thumbnail_panel_move_focus (-1);
}

static void
next_callback (GtkWidget *widget, gpointer data)
{
  thumbnail_panel_move_focus (ncol_of_thumbnail);
}

static void
prev_callback (GtkWidget *widget, gpointer data)
{
  thumbnail_panel_move_focus (- ncol_of_thumbnail);
}

static void
next_page_callback (GtkWidget *widget, gpointer data)
{
  if (cwd_cache->display_page + 1 < cwd_cache->npage)
    {
      cwd_cache->display_page++;
      thumbnail_panel_update ();
    }
}

static void
prev_page_callback (GtkWidget *widget, gpointer data)
{
  if (0 < cwd_cache->display_page)
    {
      cwd_cache->display_page--;
      thumbnail_panel_update ();
    }
}

static void
open_callback (GtkWidget *widget, gpointer data)
{
  selection_open_files ();
}

static void
update_callback (GtkWidget *widget, gpointer data)
{
  directory_cache_delete_invalid_cache_files (FALSE);
}

static void
toggle_sort_mode_callback (GtkWidget *widget, gpointer data)
{
  VAL.sort_type = (VAL.sort_type == SORT_BY_NAME) ? SORT_BY_DATE : SORT_BY_NAME;
  directory_cache_set_to_cwd (TRUE);
  thumbnail_panel_set_info ((VAL.sort_type == SORT_BY_NAME)
			    ? "Changed to `sort by name' mode"
			    : "Changed to `sort by date' mode");
}

static void
purge_cache_file_callback (GtkWidget *widget, gpointer data)
{
  directory_cache_delete_invalid_cache_files (TRUE);
}

static void
help_callback (GtkWidget *widget, gpointer client_data)
{
  about_dialog ();
}

static void
fileselector_for_copy_callback (GtkWidget *widget, gpointer client_data)
{
  GtkWidget *filesel;

  filesel = gtk_file_selection_new ("Copy selected images to");
  gtk_window_position (GTK_WINDOW (filesel), GTK_WIN_POS_MOUSE);

  gtk_signal_connect (GTK_OBJECT (GTK_FILE_SELECTION (filesel)->ok_button),
		      "clicked", (GtkSignalFunc) copy_callback,
		      filesel);

  gtk_signal_connect_object (GTK_OBJECT (GTK_FILE_SELECTION (filesel)->cancel_button),
			     "clicked", (GtkSignalFunc) gtk_widget_destroy,
			     GTK_OBJECT (GTK_WINDOW (filesel)));

  if (selection_length ())
    {
      cache_entry	*selected;
      guchar	tmp[LINE_BUF_SIZE];

      selected = directory_cache_get_nth ((gint) selection_list->data);

      sprintf (tmp, "%s/%s", cwd_cache->name, selected->name);
      gtk_file_selection_set_filename (GTK_FILE_SELECTION (filesel), tmp);
    }
  else if (strlen (VAL.last_dir_name) > 0)
    {
      gtk_file_selection_set_filename (GTK_FILE_SELECTION (filesel),
				       VAL.last_dir_name);
    }
  else
    {
      gtk_file_selection_set_filename (GTK_FILE_SELECTION (filesel),
				       cwd_cache->name);
    }
  gtk_widget_show (filesel);
}

static void
fileselector_for_move_callback (GtkWidget *widget, gpointer client_data)
{
  GtkWidget *filesel;

  if (! selection_is_active ())
    return;
  if (selection_kind != IMAGE)
    return;
  if (! file_confirm_operation ("Move"))
    return;

  filesel = gtk_file_selection_new ("Move selected images to");
  gtk_window_position (GTK_WINDOW (filesel), GTK_WIN_POS_MOUSE);

  gtk_signal_connect (GTK_OBJECT (GTK_FILE_SELECTION (filesel)->ok_button),
		      "clicked", (GtkSignalFunc) move_callback,
		      filesel);

  gtk_signal_connect_object (GTK_OBJECT (GTK_FILE_SELECTION (filesel)->cancel_button),
			     "clicked", (GtkSignalFunc) gtk_widget_destroy,
			     GTK_OBJECT (GTK_WINDOW (filesel)));

  if (fileselector_last_pathname)
    gtk_file_selection_set_filename (GTK_FILE_SELECTION (filesel),
				     fileselector_last_pathname);
  gtk_widget_show (filesel);
}

static void
fileselector_for_chdir_callback (GtkWidget *widget, gpointer client_data)
{
  GtkWidget *filesel;

  filesel = gtk_file_selection_new ("Change directory");
  gtk_window_position (GTK_WINDOW (filesel), GTK_WIN_POS_MOUSE);

  gtk_signal_connect (GTK_OBJECT (GTK_FILE_SELECTION (filesel)->ok_button),
		      "clicked", (GtkSignalFunc) chdir_callback,
		      filesel);

  gtk_signal_connect_object (GTK_OBJECT (GTK_FILE_SELECTION (filesel)->cancel_button),
			     "clicked", (GtkSignalFunc) gtk_widget_destroy,
			     GTK_OBJECT (GTK_WINDOW (filesel)));
  if (fileselector_last_pathname)
    gtk_file_selection_set_filename (GTK_FILE_SELECTION (filesel),
				     fileselector_last_pathname);

  gtk_widget_show (filesel);
}

static void
fileselector_for_mkdir_callback (GtkWidget *widget, gpointer client_data)
{
  GtkWidget *filesel;

  filesel = gtk_file_selection_new ("New directory");
  gtk_window_position (GTK_WINDOW (filesel), GTK_WIN_POS_MOUSE);

  gtk_signal_connect (GTK_OBJECT (GTK_FILE_SELECTION (filesel)->ok_button),
		      "clicked", (GtkSignalFunc) mkdir_callback,
		      filesel);

  gtk_signal_connect_object (GTK_OBJECT (GTK_FILE_SELECTION (filesel)->cancel_button),
			     "clicked", (GtkSignalFunc) gtk_widget_destroy,
			     GTK_OBJECT (GTK_WINDOW (filesel)));
  if (fileselector_last_pathname != NULL)
    gtk_file_selection_set_filename (GTK_FILE_SELECTION (filesel),
				     fileselector_last_pathname);

  gtk_widget_show (filesel);
}

static void
copy_callback (GtkWidget *widget, gpointer data)
{
  guchar	*pathname, *tmp;

  tmp = gtk_file_selection_get_filename (GTK_FILE_SELECTION (data));
  pathname = (guchar *) g_malloc (strlen (tmp) + 1);
  strcpy (pathname, tmp);
  fileselector_set_last_value (pathname);

  gtk_widget_destroy (GTK_WIDGET (data));

  selection_copy_files_to (pathname);

  g_free (pathname);
}

static void
move_callback (GtkWidget *widget, gpointer data)
{
  guchar	*pathname, *tmp;

  tmp = gtk_file_selection_get_filename (GTK_FILE_SELECTION (data));
  pathname = (guchar *) g_malloc (strlen (tmp) + 1);
  strcpy (pathname, tmp);
  fileselector_set_last_value (pathname);

  gtk_widget_destroy (GTK_WIDGET (data));

  selection_move_files_to (pathname);

  g_free (pathname);
}

static void
delete_callback (GtkWidget *widget, gpointer data)
{
  if (! selection_is_active ())
    return;

  if (use_confirmor)
    {
      guchar buffer [LINE_BUF_SIZE];

      if (selection_length () == 1)
	{
	  cache_entry	*selected;

	  selected = directory_cache_get_nth ((gint) selection_list->data);
	  sprintf (buffer, " Do you want to delete the selected image: %s? ",
		   selected->name);
	}
      else
	{
	  sprintf (buffer,  " Do you want to delete the selected %d images? ",
		   selection_length ());
	}
      if (! gtkW_confirmor_dialog (TRUE, buffer, FALSE))
	return;
    }
  selection_delete_files ();
}

static void
select_all_callback (GtkWidget *widget, gpointer data)
{
  gint	index;

  for (index = cwd_cache->ndir;
       index < cwd_cache->ndir + cwd_cache->nimage;
       index++)
    selection_add (index);
  thumbnail_panel_set_info (NULL);
}

static void
select_none_callback (GtkWidget *widget, gpointer data)
{
  selection_reset ();
  thumbnail_panel_set_info (NULL);
}

static void
chdir_callback (GtkWidget *widget, gpointer data)
{
  guchar	*pathname, *tmp;
  gint		f_kind;

  tmp = gtk_file_selection_get_filename (GTK_FILE_SELECTION (data));
  pathname = (guchar *) g_malloc (strlen (tmp) + 1);
  strcpy (pathname, tmp);
  fileselector_set_last_value (pathname);

  gtk_widget_destroy (GTK_WIDGET (data));

  tmp = pathname;
  f_kind = os_file_kind (tmp, TRUE);
  while ((f_kind == NOT_EXIST) || (f_kind == REGFILE))
    {
      tmp = file_get_parent_directory (tmp);
      f_kind = os_file_kind (tmp, TRUE);
    }

  switch (os_file_kind (tmp, TRUE))
    {
    case DIRECTORY:
      chdir (tmp);
      directory_cache_set_to_cwd (FALSE);
      break;
    default:
      thumbnail_panel_set_info ("Can't move there.");
      break;
    }
  g_free (pathname);
}

static void
mkdir_callback (GtkWidget *widget, gpointer data)
{
  guchar	*pathname, *tmp;

  tmp = gtk_file_selection_get_filename (GTK_FILE_SELECTION (data));
  pathname = (guchar *) g_malloc (strlen (tmp) + 1);
  strcpy (pathname, tmp);
  fileselector_set_last_value (pathname);

  gtk_widget_destroy (GTK_WIDGET (data));

  file_mkdir (pathname);

  g_free (pathname);
}

static void
selection_map_script_callback (GtkWidget *widget, gpointer data)
{
  selection_map_script ();
}

static void
selection_map_unix_command_callback (GtkWidget *widget, gpointer data)
{
  selection_map_unix_command ();
}

static void
gtkW_close_callback (GtkWidget *widget,
		     gpointer   data)
{
  gtk_main_quit ();
}

static void
gtkW_message_dialog (gint gtk_was_initialized, guchar *message)
{
  GtkWidget *dlg;
  GtkWidget *table;
  GtkWidget *label;
  gchar	**argv;
  gint	argc;

  if (! gtk_was_initialized)
    {
      argc = 1;
      argv = g_new (gchar *, 1);
      argv[0] = g_strdup (PLUG_IN_NAME);
      gtk_init (&argc, &argv);
      gtk_rc_parse (gimp_gtkrc ());
    }

  dlg = gtkW_message_dialog_new (PLUG_IN_NAME);

  table = gtkW_table_new (GTK_DIALOG (dlg)->vbox, 1, 1);

  label = gtk_label_new (message);
  gtk_table_attach (GTK_TABLE (table), label, 0, 1, 0, 1, GTK_FILL|GTK_EXPAND,
		    0, 0, 0);

  gtk_widget_show (label);
  gtk_widget_show (dlg);

  gtk_main ();
  gdk_flush ();
}

static GtkWidget *
gtkW_message_dialog_new (guchar *name)
{
  GtkWidget *dlg, *button;

  dlg = gtk_dialog_new ();
  gtk_window_set_title (GTK_WINDOW (dlg), name);
  gtk_window_position (GTK_WINDOW (dlg), GTK_WIN_POS_MOUSE);
  gtk_signal_connect (GTK_OBJECT (dlg), "destroy",
		      (GtkSignalFunc) gtkW_close_callback, NULL);
  gtk_container_border_width (GTK_CONTAINER (GTK_DIALOG (dlg)->action_area),
			      gtkW_border_width);

  /* Action Area */
  button = gtk_button_new_with_label ("OK");
  GTK_WIDGET_SET_FLAGS (button, GTK_CAN_DEFAULT);
  gtk_signal_connect_object (GTK_OBJECT (button), "clicked",
			     (GtkSignalFunc) gtk_widget_destroy,
			     GTK_OBJECT (dlg));
  gtk_box_pack_start (GTK_BOX (GTK_DIALOG (dlg)->action_area), button,
		      TRUE, TRUE, 0);
  gtk_widget_grab_default (button);
  gtk_widget_show (button);

  return dlg;
}

static gint
gtkW_confirmor_dialog (gint gtk_was_initialized, guchar *message, gint default_value)
{
  GtkWidget *dlg;
  GtkWidget *table;
  GtkWidget *label;
  gtkW_widget_table	wtable;
  gchar	**argv;
  gint	argc;

  if (! gtk_was_initialized)
    {
      argc = 1;
      argv = g_new (gchar *, 1);
      argv[0] = g_strdup (PLUG_IN_NAME);
      gtk_init (&argc, &argv);
      gtk_rc_parse (gimp_gtkrc ());
    }

  dlg = gtkW_confirmor_new (PLUG_IN_NAME, default_value, &wtable);

  table = gtkW_table_new (GTK_DIALOG (dlg)->vbox, 1, 1);

  label = gtk_label_new (message);
  gtk_table_attach (GTK_TABLE (table), label, 0, 1, 0, 1, GTK_FILL|GTK_EXPAND,
		    0, 0, 0);

  gtk_widget_show (label);
  gtk_widget_show (dlg);

  gtk_main ();
  gdk_flush ();
  return (gint) wtable.value;
}

static GtkWidget *
gtkW_confirmor_new (guchar *name, gint default_value, gtkW_widget_table *table)
{
  GtkWidget	*dlg, *button;

  dlg = gtk_dialog_new ();
  gtk_window_set_title (GTK_WINDOW (dlg), name);
  gtk_window_position (GTK_WINDOW (dlg), GTK_WIN_POS_MOUSE);
  gtk_signal_connect (GTK_OBJECT (dlg), "destroy",
		      (GtkSignalFunc) gtkW_close_callback, NULL);
  gtk_container_border_width (GTK_CONTAINER (GTK_DIALOG (dlg)->action_area),
			      gtkW_border_width);
  table->widget = dlg;

  /* Action Area */
  button = gtk_button_new_with_label ("Yes");
  gtk_signal_connect (GTK_OBJECT (button), "clicked",
		      (GtkSignalFunc) gtkW_confirmor_yes,
		      (gpointer) table);
  gtk_box_pack_start (GTK_BOX (GTK_DIALOG (dlg)->action_area), button,
		      TRUE, TRUE, 0);
  GTK_WIDGET_SET_FLAGS (button, GTK_CAN_DEFAULT);
  if (default_value == TRUE)
    {
      gtk_widget_grab_default (button);
    }
  gtk_widget_show (button);

  button = gtk_button_new_with_label ("No");
  gtk_signal_connect (GTK_OBJECT (button), "clicked",
		      (GtkSignalFunc) gtkW_confirmor_no,
		      (gpointer) table);
  gtk_box_pack_start (GTK_BOX (GTK_DIALOG (dlg)->action_area), button,
		      TRUE, TRUE, 0);
  GTK_WIDGET_SET_FLAGS (button, GTK_CAN_DEFAULT);
  if (default_value == FALSE)
    {
      gtk_widget_grab_default (button);
    }
  gtk_widget_show (button);

  return dlg;
}

static void
gtkW_confirmor_yes (GtkWidget *widget, gpointer data)
{
  gtkW_widget_table *table;

  table = (gtkW_widget_table *)data;
  table->value = (gpointer) TRUE;
  gtk_widget_destroy ((GtkWidget *) (table->widget));
}

static void
gtkW_confirmor_no (GtkWidget *widget, gpointer data)
{
  gtkW_widget_table *table;

  table = (gtkW_widget_table *)data;
  table->value = (gpointer) FALSE;
  gtk_widget_destroy ((GtkWidget *) (table->widget));
}

static GtkWidget *
gtkW_frame_new (GtkWidget *parent,
		guchar *name)
{
  GtkWidget *frame;

  frame = gtk_frame_new (name);
  gtk_frame_set_shadow_type (GTK_FRAME (frame), GTK_SHADOW_ETCHED_IN);
  gtk_container_border_width (GTK_CONTAINER (frame), gtkW_border_width);
  if (parent != NULL)
    gtk_box_pack_start (GTK_BOX(parent), frame, FALSE, FALSE, 0);
  gtk_widget_show (frame);

  return frame;
}

static GtkWidget *
gtkW_hbox_new (GtkWidget *parent)
{
  GtkWidget	*hbox;

  hbox = gtk_hbox_new (FALSE, 2);
  gtk_container_border_width (GTK_CONTAINER (hbox), gtkW_border_width);
  /* gtk_box_pack_start (GTK_BOX (parent), hbox, FALSE, TRUE, 0); */
  if (parent)
    gtk_container_add (GTK_CONTAINER (parent), hbox);
  gtk_widget_show (hbox);

  return hbox;
}

static GtkWidget *
gtkW_table_new (GtkWidget *parent, gint col, gint row)
{
  GtkWidget	*table;

  table = gtk_table_new (col,row, FALSE);
  gtk_container_border_width (GTK_CONTAINER (table), gtkW_border_width);
  gtk_container_add (GTK_CONTAINER (parent), table);
  gtk_widget_show (table);

  return table;
}

static gint
preview_event_handler (GtkWidget *widget, GdkEvent *event)
{
  GdkEventButton *bevent;
  gint		x, y, index;

  gtk_widget_get_pointer (widget, &x, &y);

  bevent = (GdkEventButton *) event;
  index = cwd_cache ? (POS_TO_INDEX (x, y)) : -1;

  switch (event->type)
    {
    case GDK_EXPOSE:
      gtk_widget_show (dlg);
      gdk_flush ();
      break;
    case GDK_BUTTON_PRESS:
      switch (bevent->button)
	{
	case 1:
	  if ((0 <= index) && (index < cwd_cache->ndir + cwd_cache->nimage))
	    {
	      cache_entry *cache = directory_cache_get_nth (index);

	      if (cache->directory_p)
		{
		  if (selection_is_active ())
		    {
		      if (bevent->state & GDK_CONTROL_MASK)
			{
			  selection_copy_files_to (cache->name);
			}
		      else if (bevent->state & GDK_SHIFT_MASK)
			{
			  if (file_confirm_operation ("Move"))
			    selection_move_files_to (cache->name);
			}
		      else
			{
			  selection_reset ();
			  chdir (cache->name);
			  directory_cache_set_to_cwd (FALSE);
			}
		    }
		  else
		    {
		      selection_reset ();
		      chdir (cache->name);
		      directory_cache_set_to_cwd (FALSE);
		    }
		}
	      else
		{
		  if (bevent->state & GDK_SHIFT_MASK)
		    {
		      selection_reverse_member (index);
		      if (selection_is_active ())
			thumbnail_panel_set_info (cache->info);
		    }
		  else
		    {
		      if (selection_is_active () && selection_member_p (index))
			selection_open_files ();
		      else
			{
			  selection_reset ();
			  selection_add (index);
			  thumbnail_panel_set_info (cache->info);
			}
		    }
		}
	    }
	  else
	    {
	      selection_reset ();
	      thumbnail_panel_set_info (NULL);
	    }
	  break;
	case 2:
	  if (bevent->state & (GDK_SHIFT_MASK | GDK_CONTROL_MASK))
	    {
	      if (0 < cwd_cache->display_page)
		cwd_cache->display_page--;
	      else
		cwd_cache->display_page = cwd_cache->npage - 1;
	    }
	  else
	    {
	      if (cwd_cache->display_page + 1 < cwd_cache->npage)
		cwd_cache->display_page++;
	      else if (0 < cwd_cache->display_page)
		cwd_cache->display_page = 0;
	    }
	  thumbnail_panel_update ();
	  break;
	case 3:
	  gtk_menu_popup (GTK_MENU (thumbnail_panel_menu),
			  NULL, NULL, NULL, NULL, 3, bevent->time);
	  break;
	default:
	  thumbnail_panel_update ();
	  break;
	}
      break;
    case GDK_BUTTON_RELEASE:
    default:
      break;
    }
  return FALSE;
}

static gint
cursor_event_handler (GtkWidget *widget, GdkEvent *event)
{
  GdkEventKey	*kevent;

  kevent = (GdkEventKey *) event;

  gtk_widget_grab_focus (thumbnail_panel);
  switch (kevent->keyval)
    {
      /* Please explain why the following code does not work!
    case GDK_Up:
      thumbnail_panel_move_focus (- ncol_of_thumbnail);
      gtk_widget_grab_focus (file_property);
      break;
    case GDK_Down:
      thumbnail_panel_move_focus (ncol_of_thumbnail);
      gtk_widget_grab_focus (cwd_label);
      break;
      */
      /* Please explain why the following focus moving is required!
	 I found this code after a long try-and-error, by chance.
      */
    case GDK_Left:
      thumbnail_panel_move_focus (-1);
      gtk_widget_grab_focus (file_property);
      break;
    case GDK_Right:
      thumbnail_panel_move_focus (1);
      gtk_widget_grab_focus (cwd_label);
      break;
    default:
      break;
    }
  return FALSE;
}

static GtkWidget *
gtkW_table_add_button (GtkWidget	*table,
		       gchar		*name,
		       gint		x0,
		       gint		x1,
		       gint		y,
		       GtkSignalFunc	callback,
		       gpointer		value)
{
  GtkWidget *button;

  button = gtk_button_new_with_label (name);
  gtk_table_attach (GTK_TABLE(table), button, x0, x1, y, y+1,
		    gtkW_align_x, gtkW_align_y, 0, 0);
  if (callback != NULL)
    gtk_signal_connect (GTK_OBJECT (button), "clicked",
			(GtkSignalFunc) callback, value);
  gtk_widget_show (button);

  return button;
}

static GtkWidget *
gtkW_table_add_label (GtkWidget	*table,
		      gchar	*text,
		      gint	x0,
		      gint	x1,
		      gint	y,
		      gint	flush_left)
{
  GtkWidget *label;

  label = gtk_label_new (text);
  if (flush_left)
    gtk_misc_set_alignment (GTK_MISC (label), 0.0, 1.0);
  gtk_table_attach (GTK_TABLE(table), label, x0, x1, y, y+1,
		    gtkW_align_x, gtkW_align_y, 5, 0);
  gtk_widget_show (label);

  return label;
}

static void
gtkW_iscroll_entry_change_value (gtkW_widget_table *wtable)
{
  GtkWidget *entry;
  gchar buffer[32];
  GtkAdjustment *adjustment = (GtkAdjustment *) (wtable->widget);

  if (! cwd_cache)
    return;
  /*  adustment->value is double, that is not precise to hold long interger. */
  adjustment->value = * (gint *) (wtable->value);
  gtk_signal_emit_by_name (GTK_OBJECT (adjustment), "value_changed");
  entry = gtk_object_get_user_data (GTK_OBJECT (adjustment));
  if (entry)
    {
      sprintf (buffer, "%d", (gint) adjustment->value);
      gtk_entry_set_text (GTK_ENTRY (entry), buffer);
    }
}

static void
gtkW_iscroll_update (GtkAdjustment *adjustment,
		    gpointer       data)
{
  GtkWidget *entry;
  gchar	buffer[32];
  gint	*val;

  if (! cwd_cache)
    return;
  val = data;
  if (*val != (gint) adjustment->value)
    {
      *val = adjustment->value;

      entry = gtk_object_get_user_data (GTK_OBJECT (adjustment));
      if (entry)
	{
	  sprintf (buffer, "%d", (int) adjustment->value);
	  gtk_entry_set_text (GTK_ENTRY (entry), buffer);
	}
      cwd_cache->display_page = (gint) adjustment->value - 1;
      thumbnail_panel_update ();
    }
}

static void
gtkW_ivscroll_entry_new (GtkWidget	**scroll,
			 GtkWidget	**entry,
			 GtkSignalFunc	scroll_update,
			 GtkSignalFunc	entry_update,
			 gint		*value,
			 gdouble	min,
			 gdouble	max,
			 gdouble	step,
			 gpointer	widget_entry)
{
  GtkObject	*adjustment;
  guchar	*buffer;

  adjustment = gtk_adjustment_new (*value, min, max, step, step, step);
  gtk_signal_connect (GTK_OBJECT (adjustment), "value_changed",
		      (GtkSignalFunc) scroll_update, value);

  *scroll = gtk_vscrollbar_new (GTK_ADJUSTMENT (adjustment));
  gtk_widget_set_usize (*scroll, thumbnail_panel_width - GTKW_ENTRY_WIDTH - 40, 0);
  gtk_range_set_update_policy (GTK_RANGE (*scroll), GTK_UPDATE_CONTINUOUS);

  *entry = gtk_entry_new ();
  gtk_object_set_user_data (GTK_OBJECT (*entry), adjustment);
  gtk_object_set_user_data (GTK_OBJECT (adjustment), *entry);
  gtk_widget_set_usize (*entry, GTKW_ENTRY_WIDTH, 0);
  buffer = (guchar *) g_malloc (GTKW_ENTRY_BUFFER_SIZE);
  sprintf (buffer, "%d", *value);
  gtk_entry_set_text (GTK_ENTRY (*entry), buffer);
  gtk_signal_connect (GTK_OBJECT (*entry), "changed",
		      (GtkSignalFunc) entry_update, value);

  if (widget_entry)
    {
      gtkW_widget_table *tentry = (gtkW_widget_table *) widget_entry;

      tentry->widget = (GtkWidget *) adjustment;
      tentry->updater = gtkW_iscroll_entry_change_value;
      tentry->value = value;
    }
}

static void
gtkW_ientry_update (GtkWidget *widget,
		    gpointer   data)
{
  GtkAdjustment *adjustment;
  gint	new_val;
  gint	val;

  if (! cwd_cache)
    return;

  val = cwd_cache->display_page + 1;
  new_val = atoi (gtk_entry_get_text (GTK_ENTRY (widget)));

  if (val != new_val)
    {
      adjustment = gtk_object_get_user_data (GTK_OBJECT (widget));

      if ((new_val >= adjustment->lower) &&
	  (new_val <= adjustment->upper))
	{
	  adjustment->value = new_val;
	  cwd_cache->display_page = (gint) new_val - 1;
	  gtk_signal_emit_by_name (GTK_OBJECT (adjustment), "value_changed");
	}
    }
}

#ifdef USE_DND
static void
dnd_drag_move_request (GtkWidget *button, GdkEvent *event)
{
  gtk_widget_dnd_data_set (button, event, "move",
			   strlen ("move") + 1);
}

static void
dnd_drag_copy_request (GtkWidget *button, GdkEvent *event)
{
  gtk_widget_dnd_data_set (button, event, "copy",
			   strlen ("copy") + 1);
}

static void
dnd_drop (GtkWidget *button, GdkEvent *event)
{
  gint            x, y, index;
  GdkEventButton *bevent;

  gtk_widget_get_pointer (button, &x, &y);

  bevent = (GdkEventButton *) event;
  index = POS_TO_INDEX (x, y);

  if (0 < selection_length () && (0 <= index)
      && (index < cwd_cache->ndir + cwd_cache->nimage))
    {
      cache_entry	*cache = directory_cache_get_nth (index);

      if (cache->directory_p)
	{
	  if (strcmp (event->dropdataavailable.data, "move") == 0)
	    {
	      selection_move_files_to (cwd_cache->name);
	      directory_cache_set_to_cwd (TRUE);
	    }
	  else if (strcmp (event->dropdataavailable.data, "copy") == 0)
	    {
	      selection_copy_files_to (cache->name);
	      selection_reset ();
	    }
	}
      else
	{
	  thumbnail_panel_set_info ("NOP: please drop into a directory!");
	}
      g_free (event->dropdataavailable.data);
      g_free (event->dropdataavailable.data_type);
    }
}
#endif

static void
gtkW_query_box (guchar *title, guchar *message, guchar *initial, guchar *data)
{
  GtkWidget *qbox;

  qbox = gtkW_query_string_box_new (title, message, initial, data);
  gtk_main ();
  gdk_flush ();
}

static void
gtkW_preview_force_to_update (GtkWidget *widget)
{
  GtkPreview	*preview = GTK_PREVIEW (widget);

  gtk_preview_put (GTK_PREVIEW (widget),
		   widget->window, widget->style->black_gc,
		   (widget->allocation.width - preview->buffer_width) / 2,
		   (widget->allocation.height - preview->buffer_height) / 2,
		   0, 0,
		   widget->allocation.width, widget->allocation.height);
  gdk_flush ();
}

/* copied from gimp/app/interface.c */
/*
 *  A text string query box
 */
typedef struct _QueryBox QueryBox;

struct _QueryBox
{
  GtkWidget *qbox;
  GtkWidget *entry;
  gpointer data;
};

static GtkWidget *
gtkW_query_string_box_new (char        *title,
			   char        *message,
			   char        *initial,
			   gpointer     data)
{
  QueryBox  *query_box;
  GtkWidget *qbox;
  GtkWidget *vbox;
  GtkWidget *label;
  GtkWidget *entry;
  GtkWidget *button;

  query_box = (QueryBox *) g_malloc (sizeof (QueryBox));

  qbox = gtk_dialog_new ();
  gtk_window_set_title (GTK_WINDOW (qbox), title);
  gtk_window_position (GTK_WINDOW (qbox), GTK_WIN_POS_MOUSE);
  gtk_signal_connect (GTK_OBJECT (qbox), "delete_event",
		      (GtkSignalFunc) gtkW_query_box_delete_callback,
		      query_box);
  gtk_container_border_width (GTK_CONTAINER (GTK_DIALOG (qbox)->action_area),
			      gtkW_border_height);

  button = gtk_button_new_with_label ("OK");
  GTK_WIDGET_SET_FLAGS (button, GTK_CAN_DEFAULT);
  gtk_signal_connect (GTK_OBJECT (button), "clicked",
                      (GtkSignalFunc) gtkW_query_box_ok_callback,
                      query_box);
  gtk_box_pack_start (GTK_BOX (GTK_DIALOG (qbox)->action_area), button, TRUE, TRUE, 0);
  gtk_widget_grab_default (button);
  gtk_widget_show (button);

  button = gtk_button_new_with_label ("Cancel");
  GTK_WIDGET_SET_FLAGS (button, GTK_CAN_DEFAULT);
  gtk_signal_connect (GTK_OBJECT (button), "clicked",
                      (GtkSignalFunc) gtkW_query_box_cancel_callback,
                      query_box);
  gtk_box_pack_start (GTK_BOX (GTK_DIALOG (qbox)->action_area), button, TRUE, TRUE, 0);
  gtk_widget_show (button);

  vbox = gtk_vbox_new (FALSE, 1);
  gtk_container_border_width (GTK_CONTAINER (vbox), gtkW_border_width);
  gtk_container_add (GTK_CONTAINER (GTK_DIALOG (qbox)->vbox), vbox);
  gtk_widget_show (vbox);

  label = gtk_label_new (message);
  gtk_box_pack_start (GTK_BOX (vbox), label, TRUE, FALSE, 0);
  gtk_widget_show (label);

  entry = gtk_entry_new ();
  gtk_box_pack_start (GTK_BOX (vbox), entry, TRUE, TRUE, 0);
  if (initial)
    gtk_entry_set_text (GTK_ENTRY (entry), initial);
  gtk_widget_show (entry);

  query_box->qbox = qbox;
  query_box->entry = entry;
  query_box->data = data;

  gtk_widget_show (qbox);

  return qbox;
}

static gint
gtkW_query_box_delete_callback (GtkWidget *w,
				GdkEvent  *e,
				gpointer   client_data)
{
  gtkW_query_box_cancel_callback (w, client_data);
  return FALSE;
}

static void
gtkW_query_box_cancel_callback (GtkWidget *w,
				gpointer   client_data)
{
  QueryBox *query_box;

  query_box = (QueryBox *) client_data;

  *((guchar *) query_box->data) = 0;
  /*  Destroy the box  */
  gtk_widget_destroy (query_box->qbox);
  g_free (query_box);
  gtk_main_quit ();
}

static void
gtkW_query_box_ok_callback (GtkWidget *w,
			    gpointer   client_data)
{
  QueryBox *query_box;
  char *string;

  query_box = (QueryBox *) client_data;

  /*  Get the entry data  */
  string = g_strdup (gtk_entry_get_text (GTK_ENTRY (query_box->entry)));

  /*  Call the user defined callback  */
  /* (* query_box->callback) (w, query_box->data, (gpointer) string); */
  strcpy ((guchar *)query_box->data, string);
  /*  Destroy the box  */
  gtk_widget_destroy (query_box->qbox);
  g_free (query_box);
  gtk_main_quit ();
}

static gint
gtkW_parse_gimprc_gint (guchar *name, gint default_value)
{
  GParam	*return_vals;
  gint		nreturn_vals;
  gint		val = default_value;

  return_vals = gimp_run_procedure ("gimp_gimprc_query",
				    &nreturn_vals,
				    PARAM_STRING, name,
				    PARAM_END);
  if (return_vals[0].data.d_status == STATUS_SUCCESS)
    val = atoi (return_vals[1].data.d_string);
  gimp_destroy_params (return_vals, nreturn_vals);

  return val;
}

static guchar *
gtkW_parse_gimprc_string (guchar *name, guchar *result)
{
  GParam	*return_vals;
  gint		nreturn_vals;

  return_vals = gimp_run_procedure ("gimp_gimprc_query",
				    &nreturn_vals,
				    PARAM_STRING, name,
				    PARAM_END);
  if (return_vals[0].data.d_status == STATUS_SUCCESS)
    strcpy (result, return_vals[1].data.d_string);
  else
    *result = 0;
  gimp_destroy_params (return_vals, nreturn_vals);

  return result;
}
/* gtkW ends here */

static gint
save_xvpict_image (char *filename, gint32 image_ID, gint32 drawable_ID)
{
  FILE		*dest;
  GPixelRgn	src_rgn;
  GDrawable	*drawable = NULL;
  gint		i_width;
  gint		i_height;
  gint		gap;
  gint		x, y;
  guchar	*buf;

  drawable = gimp_drawable_get (drawable_ID);
  i_width = gimp_layer_width (drawable_ID);
  i_height = gimp_layer_height (drawable_ID);
  gap = (gimp_drawable_has_alpha (drawable_ID)) ? 1 : 0;

  if ((dest = fopen (filename, "w")) == NULL)
    {
      gimp_drawable_detach (drawable);
      return FALSE;
    }
  fprintf (dest, "P7 332\n");
  /* fprintf (dest, "#XVVERSION:Version 3.10a  Rev: 12/29/94\n"); */
  fprintf (dest, "#XVVERSION:Version 3.10a  Rev: 12/29/94 (faked by The GIMP/GUASH 1.0)\n");
  fprintf (dest, "#IMGINFO:%s\n",
	   gimp_layer_get_name (gimp_image_get_active_layer (image_ID)));
  fprintf (dest, "#END_OF_COMMENTS\n");
  fprintf (dest, "%d %d 255\n", i_width, i_height);

  buf = (guchar *) g_malloc ((3 + gap) * i_width);

  gimp_pixel_rgn_init (&src_rgn, drawable, 0, 0, i_width, i_height,
		       FALSE, FALSE);

  for (y = 0; y < i_height; y++)
    {
      guchar	*ch;
      guchar	val;

      gimp_pixel_rgn_get_row (&src_rgn, buf, 0, y, i_width);
      ch = buf;

      for (x = 0; x < i_width; x++, ch += 3)
	{
	  val = (*ch & 0xE0)
	    + ((*(ch + 1) & 0xE0) >> 3)
	    + ((*(ch + 2) & 0xC0) >> 6);
	  fprintf (dest, "%c", val);
	}
    }
  gimp_drawable_detach (drawable);
  fclose (dest);
  g_free (buf);
  return TRUE;
}
/* end of my part of guash.c */
#define EMBEDDED_IN_GUASH	1

static gint32
load_xvpict_image (char *filename) {
	FILE *fd;
	char *name_buf, w_buf[4], h_buf[4];
	int i, j, w, h;
	unsigned char cmap[3*256];
	gint32 image_ID = -1, layer_ID;
	GPixelRgn pixel_rgn;
	GDrawable *drawable;
	guchar *dest;
#ifdef	EMBEDDED_IN_GUASH
	guchar comment[LINE_BUF_SIZE];
#endif

#ifndef	EMBEDDED_IN_GUASH
	name_buf = g_malloc(strlen(filename) + 11);
	sprintf(name_buf, "Loading %s:", filename);
	gimp_progress_init(name_buf);
	g_free(name_buf);
	gimp_progress_update(1);
#endif
	fd = fopen(filename, "rb");
	if (!fd) {
#ifdef EMBEDDED_IN_GUASH
	  return -1;
#else
		printf("XVPics: can't open \"%s\"\n", filename);
		/* FIXME: quit the plug-in */
#endif
	}

	/* calculate the rgb 332 8bit palette          */
	/* NOTE: this palette should be pre-calculated */
	for (i = 0, j = 0; i < 256; i++, j+=3) {
		cmap[j+0] = (((i >> 5) & 0x07)*255) / 7;
		cmap[j+1] = (((i >> 2) & 0x07)*255) / 7;
		cmap[j+2] = (((i >> 0) & 0x03)*255) / 3;
	}

#ifdef	EMBEDDED_IN_GUASH
	fgets (comment, LINE_BUF_SIZE - 1, fd);
	fgets (comment, LINE_BUF_SIZE - 1, fd);
	fgets (comment, LINE_BUF_SIZE - 1, fd);
	i = 1;
#else
	/* skips the first three lines of comment */
	i = 4;
#endif
	while (i) if (fgetc(fd) == 0xA) i--;

	/* read the last line of comment that contains info on image size */
	fscanf(fd, "%s %s %*s\n", w_buf, h_buf);
#ifdef	EMBEDDED_IN_GUASH
	w = atoi(w_buf);
	h = atoi(h_buf);
#else
	printf("XVPict %s[%i|%i]\n", filename, w = atoi(w_buf), h = atoi(h_buf));
#endif
	/* create the image */
	image_ID = gimp_image_new(w, h, INDEXED);
	layer_ID = gimp_layer_new(image_ID, "Background", w, h, INDEXED_IMAGE, 100, NORMAL_MODE);
	gimp_image_add_layer(image_ID,layer_ID,0);
#ifdef EMBEDDED_IN_GUASH
	{
	  guchar	*basename = file_get_filename (filename);

	  name_buf = (guchar*) g_malloc (strlen (basename)
					 + strlen (comment) - 8);
	  strcpy (name_buf, basename);
	  name_buf [strlen (basename)] = ' ';
	  memcpy (name_buf + strlen (basename) + 1,
		  comment + 9,
		  strlen (comment) - 10);
	  name_buf[strlen (basename) + strlen (comment) - 9] = 0;
	  gimp_layer_set_name (layer_ID, name_buf);
	  g_free (basename);
	}
#endif
	drawable = gimp_drawable_get(layer_ID);
	dest = g_malloc(drawable->width*drawable->height);
	fread(dest, drawable->width*drawable->height, 1, fd);
	fclose (fd);
	gimp_image_set_filename(image_ID, filename);
	gimp_pixel_rgn_init(&pixel_rgn, drawable, 0, 0, drawable->width, drawable->height, TRUE, FALSE);
	gimp_pixel_rgn_set_rect(&pixel_rgn, dest, 0, 0, drawable->width, drawable->height);
	gimp_image_set_cmap(image_ID, cmap, 256);
	gimp_drawable_flush(drawable);
	gimp_drawable_detach(drawable);
	g_free(dest);
	return image_ID;
}
/* guash.c ends here */
