/*
 * Author:      William Chia-Wei Cheng (william@cs.ucla.edu)
 *
 * Copyright (C) 1990-1994, William Cheng.
 *
 * Permission limited to the use, copy, modify, and distribute this software
 * and its documentation for any purpose is hereby granted by the Author without
 * fee, provided that the above copyright notice appear in all copies and
 * that both the copyright notice and this permission notice appear in
 * supporting documentation, and that the name of the Author not be used
 * in advertising or publicity pertaining to distribution of the software
 * without specific, written prior permission.  The Author makes no
 * representations about the suitability of this software for any purpose.
 * It is provided "as is" without express or implied warranty.  All other
 * rights (including the right to sell "tgif" and the right to sell derivative
 * works of tgif) are reserved by the Author.
 *
 * THE AUTHOR DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS SOFTWARE,
 * INCLUDING ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS, IN NO
 * EVENT SHALL THE AUTHOR BE LIABLE FOR ANY SPECIAL, INDIRECT OR
 * CONSEQUENTIAL DAMAGES OR ANY DAMAGES WHATSOEVER RESULTING FROM LOSS OF
 * USE, DATA OR PROFITS, WHETHER IN AN ACTION OF CONTRACT, NEGLIGENCE OR
 * OTHER TORTIOUS ACTION, ARISING OUT OF OR IN CONNECTION WITH THE USE OR
 * PERFORMANCE OF THIS SOFTWARE.
 */
#ifndef lint
static char RCSid[] =
      "@(#)$Header: /export/bourbon/multimedia/william/X11/TGIF2/RCS/remote.c,v 2.46 1995/01/05 06:30:23 william Exp $";
#endif

#include <sys/types.h>
#include <sys/stat.h>
#include <sys/socket.h>
#include <sys/time.h>
#include <sys/ioctl.h>
#include <netinet/in.h>
#include <netdb.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <malloc.h>
#include <errno.h>
#include <fcntl.h>

#include <X11/Xlib.h>

#include "const.h"
#include "types.h"
#include "patchlevel.h"

#include "copypaste.e"
#include "dialog.e"
#include "drawing.e"
#include "file.e"
#include "http.e"
#include "menu.e"
#include "msg.e"
#include "names.e"
#include "navigate.e"
#ifndef _NO_EXTERN
#include "remote.e"
#endif /* !_NO_EXTERN */
#include "setup.e"
#include "util.e"
#include "version.e"

extern int fork ARGS_DECL((void));

static char DEF_FILE[]="index.html";
static int debugRemote=FALSE;

int GetClientID(psz_buf, buf_sz)
   char *psz_buf;
   int buf_sz;
{
   char agent_name[128];

   sprintf(agent_name, "%s/%s patchlevel/%1d", TOOL_NAME, version_string,
         TGIF_PATCHLEVEL);
   return UtilStrCpy(psz_buf, buf_sz, agent_name);
}

int GetUserID(psz_buf, buf_sz)
   char *psz_buf;
   int buf_sz;
{
   char user_name[MAXSTRING+1];
   int total=0;

   sprintf(user_name, "%s@", TOOL_NAME);
   total = strlen(user_name);
   if (gethostname(&user_name[total], sizeof(user_name)-1-total) < 0) {
      sprintf(&user_name[total], "UNKNOWN");
   } else {
      struct hostent *p_hostent=gethostbyname(&user_name[total]);

      if (p_hostent != NULL && p_hostent->h_name != NULL &&
            *p_hostent->h_name != '\0') {
         strcpy(&user_name[total], p_hostent->h_name);
      }
   }
   return UtilStrCpy(psz_buf, buf_sz, user_name);
}

int UserAbortComm()
{
   return CheckInterrupt();
}

void FreeRemoteBuf(psz_buf)
   char *psz_buf;
{
   TcpFreeBuf(psz_buf);
}

static
int ParseURL(url, protocol, host, port, path)
   char *url, **protocol, **host, **path;
   unsigned short *port;
{
   char *c_ptr=strchr(url, ':');

   *protocol = *host = *path = 0;
   if (c_ptr == NULL) return TG_REMOTE_STATUS_FORMAT;

   *c_ptr = '\0';
   *protocol = UtilStrDup(url);
   *c_ptr++ = ':';
   if (strncmp(c_ptr, "//", 2) == 0) {
      char *tmp_host=(&c_ptr[2]), *port_ptr;

      if ((c_ptr=strchr(tmp_host, '/')) == NULL) {
         *path = UtilStrDup("");
      } else {
         *path = UtilStrDup(c_ptr);
         *c_ptr = '\0';
      }
      if ((port_ptr=strchr(tmp_host, ':')) != NULL) {
         *port_ptr = '\0';
         *port = (unsigned short)atoi(&port_ptr[1]);
         *host = UtilStrDup(tmp_host);
         *port_ptr = ':';
         if (*port <= 0) {
            return TG_REMOTE_STATUS_FORMAT;
         }
      } else {
         *host = UtilStrDup(tmp_host);
      }
      if (c_ptr != NULL) *c_ptr = '/';
   } else {
      *host = UtilStrDup("localhost");
      *path = UtilStrDup(c_ptr);
   }
   return TG_REMOTE_STATUS_OK;
}

static
void DumpURL(host, port, path)
   char *host, *path;
   unsigned short port;
{
   fprintf(stderr, "\thost = %s\n", host);
   fprintf(stderr, "\tport = %1d\n", port);
   fprintf(stderr, "\tpath = %s\n", path);
   fprintf(stderr, "\n");
}

int DirIsRemote(psz_dir)
   char *psz_dir;
{
   unsigned short port=0;
   char *protocol=NULL, *host=NULL, *path=NULL;
   int status=ParseURL(psz_dir, &protocol, &host, &port, &path), rc=FALSE;

   if (status == TG_REMOTE_STATUS_OK && protocol != NULL) {
      if (UtilStrICmp(protocol, "http") == 0 ||
            UtilStrICmp(protocol, "ftp") == 0) {
         rc = TRUE;
      }
   }
   if (protocol != NULL) cfree(protocol);
   if (host != NULL) cfree(host);
   if (path != NULL) cfree(path);
   return rc;
}

int FileIsRemote(psz_file)
   char *psz_file;
{
   return DirIsRemote(psz_file);
}

int UrlIsHtml(psz_url)
   char *psz_url;
{
   char *ext_str=UtilStrRChr(psz_url, '/');

   if (ext_str == NULL) return FALSE;
   if ((ext_str=strchr(ext_str, '.')) == NULL) return FALSE;
   ext_str++;
   return (UtilStrICmp(ext_str,"html")==0 || UtilStrICmp(ext_str,"htm")==0);
}

static
int FormLocalName(psz_file, psz_def_ext, psz_return)
   char *psz_file, *psz_def_ext, *psz_return;
{
   if (psz_def_ext != NULL) {
      char *slash_ptr=UtilStrRChr(psz_file, '/'), *dot_ptr=NULL;

      if (slash_ptr == NULL) {
         dot_ptr = strchr(psz_file, '.');
      } else {
         dot_ptr = strchr(slash_ptr, '.');
      }
      if (dot_ptr == NULL) {
         if (slash_ptr != NULL && slash_ptr[1] == '\0') {
            sprintf(psz_return, "%s/index.%s", psz_file, psz_def_ext);
         } else {
            sprintf(psz_return, "%s.%s", psz_file, psz_def_ext);
         }
      } else {
         strcpy(psz_return, psz_file);
      }
   } else {
      strcpy(psz_return, psz_file);
   }
   return UtilShrinkName(psz_return);
}

int FormRemoteName(psz_file, psz_def_ext, psz_return)
   char *psz_file, *psz_def_ext, *psz_return;
{
   char *c_ptr=strstr(psz_file, "//"), *path;

   if (c_ptr == NULL) return FALSE;
   if ((c_ptr=strchr(&c_ptr[2], '/')) == NULL) return FALSE;
   if (!FormLocalName(c_ptr, psz_def_ext, psz_return)) return FALSE;
   path = UtilStrDup(psz_return);
   *c_ptr = '\0';
   sprintf(psz_return, "%s%s", psz_file, path);
   cfree(path);
   *c_ptr = '/';
   return TRUE;
}

int FormNewFileName(psz_dir, psz_file, psz_def_ext, psz_return)
   char *psz_dir, *psz_file, *psz_def_ext, *psz_return;
{
   int rc=TRUE;

   if (DirIsRemote(psz_dir)) {
      if (*psz_file == '/') {
         int len;
         char *c_ptr=strstr(psz_dir, "//"), *fname=NULL;

         if (c_ptr == NULL) return FALSE;
         if ((c_ptr=strchr(&c_ptr[2], '/')) == NULL) return FALSE;
         *c_ptr = '\0';
         len = strlen(psz_dir)+strlen(psz_file);
         fname = (char*)calloc(len+1, sizeof(char));
         if (fname == NULL) {
            *c_ptr = '/';
            fprintf(stderr, "Can not calloc().\n");
            return FALSE;
         }
         sprintf(fname, "%s%s", psz_dir, psz_file);
         *c_ptr = '/';
         if (!FormRemoteName(fname, psz_def_ext, psz_return)) rc = FALSE;
         cfree(fname);
      } else if (FileIsRemote(psz_file)) {
         if (!FormRemoteName(psz_file, psz_def_ext, psz_return)) rc = FALSE;
      } else {
         int len=strlen(psz_dir)+1+strlen(psz_file);
         char *fname=(char*)calloc(len+1, sizeof(char));

         if (fname == NULL) {
            fprintf(stderr, "Can not calloc().\n");
            return FALSE;
         }
         sprintf(fname, "%s/%s", psz_dir, psz_file);
         if (!FormRemoteName(fname, psz_def_ext, psz_return)) rc = FALSE;
         cfree(fname);
      }
   } else {
      if (*psz_file == '/') {
         if (!FormLocalName(psz_file, psz_def_ext, psz_return)) rc = FALSE;
      } else if (FileIsRemote(psz_file)) {
         if (!FormRemoteName(psz_file, psz_def_ext, psz_return)) rc = FALSE;
      } else {
         int len=strlen(psz_dir)+1+strlen(psz_file);
         char *fname=(char*)calloc(len+1, sizeof(char));

         if (fname == NULL) {
            fprintf(stderr, "Can not calloc().\n");
            return FALSE;
         }
         sprintf(fname, "%s/%s", psz_dir, psz_file);
         if (!FormLocalName(fname, psz_def_ext, psz_return)) rc = FALSE;
         cfree(fname);
      }
   }
   return rc;
}

void ShowRemoteStatus(psz_msg)
   char *psz_msg;
{
   SetStringStatus(psz_msg);
   XSync(mainDisplay, False);
}

#define FTP_LOGGED_IN 0x10000

int LoadRemoteFileInMem(url, ppsz_buf, ppsz_content_type, pn_buf_sz, pn_html)
   char *url, **ppsz_buf, **ppsz_content_type;
   int *pn_buf_sz, *pn_html;
   /* if the remote type is http, the url has an extension and it */
   /*       is not html but the retrieved file is html, then it's */
   /*       probably an error */
{
   int status=0, n_socket=0, buf_sz=0;
   unsigned short port=0;
   char *protocol=NULL, *host=NULL, *path=NULL, *buf=NULL;

   if (pn_buf_sz != NULL) *pn_buf_sz = 0;
   if (pn_html != NULL) *pn_html = FALSE;
   *ppsz_buf = NULL;
   status = ParseURL(url, &protocol, &host, &port, &path);
   if (status != TG_REMOTE_STATUS_OK) {
      if (protocol != NULL) cfree(protocol);
      if (host != NULL) cfree(host);
      if (path != NULL) cfree(path);
      fprintf(stderr, "Fail to parse HTTP-URL '%s'.\n", url);
      return FALSE;
   }
   if (UtilStrICmp(protocol, "http") == 0) {
      int len=strlen(path);

      if (path[len-1] == '/') {
         char *tmp_path=(char*)malloc(len+strlen(DEF_FILE)+1);

         if (tmp_path == NULL) {
            fprintf(stderr, "Memory allocation failed.\n");
            return FALSE;
         }
         sprintf(tmp_path, "%s%s", path, DEF_FILE);
         cfree(path);
         path = tmp_path;
      }
      if (port == 0) port = 80;
      if (debugRemote) DumpURL(host, port, path);

      ShowInterrupt(1);
      status = HttpDoConnect(host, port, &n_socket);
      if (status == TG_REMOTE_STATUS_INTR) {
         sprintf(gszMsgBox, "HTTP: connection interrupted.");
         MsgBox(gszMsgBox, TOOL_NAME, INFO_MB);
      } else if (status < 0) {
         sprintf(gszMsgBox, "HTTP: unable to connect to remote host.");
         MsgBox(gszMsgBox, TOOL_NAME, INFO_MB);
         sprintf(gszMsgBox, "Can not connect to %s.", host);
         Msg(gszMsgBox);
      } else if (status == TG_REMOTE_STATUS_OK) {
         if (HttpDoWrite(n_socket, path) == TG_REMOTE_STATUS_OK) {
            if ((status=HttpDoRead(n_socket, &buf, &buf_sz)) ==
                  TG_REMOTE_STATUS_OK && buf != NULL && *buf != '\0') {
               *ppsz_buf = HttpExtractText(buf, pn_buf_sz, pn_html,
                     ppsz_content_type);
               FreeRemoteBuf(buf);
            } else if (status == TG_REMOTE_STATUS_INTR) {
               sprintf(gszMsgBox, "HTTP: connection interrupted.");
               MsgBox(gszMsgBox, TOOL_NAME, INFO_MB);
            } else if (buf == NULL || *buf == '\0') {
               sprintf(gszMsgBox,
                     "HTTP: fail to retrieve file from the server.");
               MsgBox(gszMsgBox, TOOL_NAME, INFO_MB);
               sprintf(gszMsgBox, "Can not get %s.", url);
               Msg(gszMsgBox);
            } else {
               sprintf(gszMsgBox,
                     "HTTP: network error occurred when retrieving file.");
               MsgBox(gszMsgBox, TOOL_NAME, INFO_MB);
               sprintf(gszMsgBox, "Network error while talking to %s.", host);
               Msg(gszMsgBox);
            }
         }
         close(n_socket);
      } else {
         sprintf(gszMsgBox, "HTTP: unable to connect to remote host.");
         MsgBox(gszMsgBox, TOOL_NAME, INFO_MB);
         sprintf(gszMsgBox, "Can not connect to %s.", host);
         Msg(gszMsgBox);
      }
      HideInterrupt();
   } else if (UtilStrICmp(protocol, "ftp") == 0) {
      if (port == 0) port = 21;
      if (debugRemote) DumpURL(host, port, path);

      ShowInterrupt(1);
      status = FtpDoConnect(host, port, &n_socket);
      if (status == TG_REMOTE_STATUS_INTR) {
         sprintf(gszMsgBox, "FTP: connection interrupted.");
         MsgBox(gszMsgBox, TOOL_NAME, INFO_MB);
      } else if (status < 0) {
         sprintf(gszMsgBox, "FTP: unable to connect to remote host.");
         MsgBox(gszMsgBox, TOOL_NAME, INFO_MB);
         sprintf(gszMsgBox, "Can not connect to %s.", host);
         Msg(gszMsgBox);
      } else if (status == TG_REMOTE_STATUS_OK) {
         sprintf(gszMsgBox, "FTP: sending requests...");
         ShowRemoteStatus(gszMsgBox);
         if ((status=FtpDoTalk(n_socket, path, &buf, &buf_sz)) ==
               TG_REMOTE_STATUS_OK && buf != NULL && *buf != '\0') {
            *ppsz_buf = buf;
            if (pn_buf_sz != NULL) *pn_buf_sz = buf_sz;
            sprintf(gszMsgBox, "FTP: data received.");
            ShowRemoteStatus(gszMsgBox);
         } else if ((status & (~FTP_LOGGED_IN)) == TG_REMOTE_STATUS_INTR) {
            sprintf(gszMsgBox, "FTP: connection interrupted.");
            MsgBox(gszMsgBox, TOOL_NAME, INFO_MB);
         } else if (status & FTP_LOGGED_IN) {
            sprintf(gszMsgBox, "FTP: fail to receive data.");
            MsgBox(gszMsgBox, TOOL_NAME, INFO_MB);
            sprintf(gszMsgBox, "Can not get %s.", url);
            Msg(gszMsgBox);
         } else {
            sprintf(gszMsgBox, "FTP: fail to login.");
            MsgBox(gszMsgBox, TOOL_NAME, INFO_MB);
            sprintf(gszMsgBox, "Can not login to %s.", host);
            Msg(gszMsgBox);
         }
         close(n_socket);
      } else {
         sprintf(gszMsgBox, "FTP: unable to connect to remote host.");
         MsgBox(gszMsgBox, TOOL_NAME, INFO_MB);
         sprintf(gszMsgBox, "Can not connect to %s.", host);
         Msg(gszMsgBox);
      }
      HideInterrupt();
   } else {
      if (debugRemote) DumpURL(host, port, path);
      fprintf(stderr, "Do not how to talk the '%s' protocol.\n", url);
   }
   if (protocol != NULL) cfree(protocol);
   if (host != NULL) cfree(host);
   if (path != NULL) cfree(path);
   return TRUE;
}

extern char *mktemp ARGS_DECL((char *Template));

#ifndef _NO_EXTERN
#if defined(IRIX) || defined(linux)
extern int	system ARGS_DECL((const char *));
#else /* ~(defined(IRIX) || defined(linux)) */
extern int	system ARGS_DECL((char *));
#endif /* defined(IRIX) || defined(linux) */
#endif /* !_NO_EXTERN */

char *WriteRemoteFileIntoTemp(psz_buf, buf_sz)
   char *psz_buf;
   int buf_sz;
{
   char *tmp_fname;
   int fd;

   tmp_fname = (char*)malloc(strlen(TMP_DIR)+20);
   if (tmp_fname == NULL) {
      FailAllocMessage();
      return NULL;
   }
   sprintf(tmp_fname, "%sTgifXXXXXX", TMP_DIR);
   mktemp(tmp_fname);
   unlink(tmp_fname);
   if ((fd=open(tmp_fname, O_WRONLY|O_CREAT|O_TRUNC)) == NULL) {
      sprintf(gszMsgBox, "Fail to create temporary file.");
      MsgBox(gszMsgBox, TOOL_NAME, INFO_MB);
      return NULL;
   } else {
      int ok=TRUE;

      if (write(fd, psz_buf, buf_sz-1) != buf_sz-1) {
         sprintf(gszMsgBox, "%s.  %s.",
               "Fail to write to temporary file",
               "File system may be full.");
         MsgBox(gszMsgBox, TOOL_NAME, INFO_MB);
         ok = FALSE;
      }
      close(fd);
      if (!ok) {
         unlink(tmp_fname);
         return NULL;
      }
      if (chmod(tmp_fname, 0777)) {
         sprintf(gszMsgBox, "Can not chmod '%s' to 0777.", tmp_fname);
         Msg(gszMsgBox);
      }
      return tmp_fname;
   }
}

static
void LaunchViewer(launch_remote_file, psz_viewer, psz_cmd, psz_url,
      psz_new_fname)
   int launch_remote_file;
   char *psz_viewer, *psz_cmd, *psz_url, *psz_new_fname;
{
   if (strcmp(psz_viewer, "NONE") != 0) {
#ifdef _BACKGROUND_DONT_FORK
      int prefix_len;
#else /* ~_BACKGROUND_DONT_FORK */
      int prefix_len, pid;
#endif /* _BACKGROUND_DONT_FORK */

      strcpy(psz_cmd, "Launch viewer: ");
      prefix_len = strlen(psz_cmd);
      if (launch_remote_file) {
         sprintf(&psz_cmd[prefix_len], psz_viewer, psz_url);
      } else {
         sprintf(&psz_cmd[prefix_len], psz_viewer, psz_new_fname);
      }
      ShowRemoteStatus(psz_cmd);
      Msg(psz_cmd);

#ifdef _BACKGROUND_DONT_FORK
      strcat(&psz_cmd[prefix_len], " &");
      system(&psz_cmd[prefix_len]);
#else /* ~_BACKGROUND_DONT_FORK */
      pid = fork();
      if (pid == 0) {
         system(&psz_cmd[prefix_len]);
         exit(0);
      }
#endif /* _BACKGROUND_DONT_FORK */
   } else {
      sprintf(psz_cmd, "%s saved.", psz_new_fname);
      Msg(psz_cmd);
   }
}

struct MimeTypeRec {
   char *main_type, *sub_type;
   struct MimeTypeRec *next;
};

struct MailCapRec {
   char *main_type, *sub_type;
   char *cmd, *params;
   struct MailCapRec *next, *prev;
};

static struct MimeTypeRec *topMimeTypesInfo=NULL;
static struct MailCapRec *topMailCapInfo=NULL, *botMailCapInfo=NULL;

static
void AddAMimeType(main_type, sub_type)
   char *main_type, *sub_type;
{
   struct MimeTypeRec *mime_type_ptr=(struct MimeTypeRec *)calloc(1,
         sizeof(struct MimeTypeRec));

   if (mime_type_ptr == NULL) { FailAllocMessage(); return; }
   mime_type_ptr->main_type = UtilStrDup(main_type);
   mime_type_ptr->sub_type = UtilStrDup(sub_type);
   mime_type_ptr->next = topMimeTypesInfo;
   topMimeTypesInfo = mime_type_ptr;
}

static
void AddAMailCap(main_type, sub_type, cmd, params)
   char *main_type, *sub_type, *cmd, *params;
{
   struct MailCapRec *mail_cap_ptr;

   mail_cap_ptr = (struct MailCapRec *)calloc(1, sizeof(struct MailCapRec));
   if (mail_cap_ptr == NULL) { FailAllocMessage(); return; }
   mail_cap_ptr->main_type = UtilStrDup(main_type);
   mail_cap_ptr->sub_type = UtilStrDup(sub_type);
   mail_cap_ptr->cmd = UtilStrDup(cmd);
   mail_cap_ptr->params = (params==NULL ? NULL : UtilStrDup(params));
   mail_cap_ptr->prev = botMailCapInfo;
   mail_cap_ptr->next = NULL;
   if (botMailCapInfo == NULL) {
      topMailCapInfo = mail_cap_ptr;
   } else {
      botMailCapInfo->next = mail_cap_ptr;
   }
   botMailCapInfo = mail_cap_ptr;
}

static
void FreeMimeTypesInfo()
{
   struct MimeTypeRec *next_mime_type_ptr;

   for ( ; topMimeTypesInfo != NULL; topMimeTypesInfo=next_mime_type_ptr) {
      next_mime_type_ptr = topMimeTypesInfo->next;
      if (topMimeTypesInfo->main_type!=NULL) cfree(topMimeTypesInfo->main_type);
      if (topMimeTypesInfo->sub_type!=NULL) cfree(topMimeTypesInfo->sub_type);
      cfree(topMimeTypesInfo);
   }
}

static
void FreeMailCapInfo()
{
   struct MailCapRec *next_mail_cap_ptr;

   for ( ; topMailCapInfo != NULL; topMailCapInfo=next_mail_cap_ptr) {
      next_mail_cap_ptr = topMailCapInfo->next;
      if (topMailCapInfo->main_type != NULL) cfree(topMailCapInfo->main_type);
      if (topMailCapInfo->sub_type != NULL) cfree(topMailCapInfo->sub_type);
      if (topMailCapInfo->cmd != NULL) cfree(topMailCapInfo->cmd);
      if (topMailCapInfo->params != NULL) cfree(topMailCapInfo->params);
      cfree(topMailCapInfo);
   }
   botMailCapInfo = NULL;
}

static
void DumpMimeTypesInfo()
{
   struct MimeTypeRec *mime_type_ptr;

   fprintf(stderr, "***  Mime Types  ***\n");
   for (mime_type_ptr=topMimeTypesInfo; mime_type_ptr != NULL;
         mime_type_ptr=mime_type_ptr->next) {
      fprintf(stderr, "%s/%s\n",
            mime_type_ptr->main_type != NULL ? mime_type_ptr->main_type : "",
            mime_type_ptr->sub_type != NULL ? mime_type_ptr->sub_type : "");
   }
   fprintf(stderr, "\n");
}

static
void DumpMailCapInfo()
{
   struct MailCapRec *mail_cap_ptr;

   fprintf(stderr, "***  Mail Capabilities  ***\n");
   for (mail_cap_ptr=topMailCapInfo; mail_cap_ptr != NULL;
         mail_cap_ptr=mail_cap_ptr->next) {
      fprintf(stderr, "%s/%s; %s; %s\n",
            mail_cap_ptr->main_type != NULL ? mail_cap_ptr->main_type : "",
            mail_cap_ptr->sub_type != NULL ? mail_cap_ptr->sub_type : "",
            mail_cap_ptr->cmd != NULL ? mail_cap_ptr->cmd : "",
            mail_cap_ptr->params != NULL ? mail_cap_ptr->params : "");
   }
   fprintf(stderr, "\n");
}

static
void FillMimeTypesInfo()
{
   char *c_ptr, *buf, fname[MAXPATHLENGTH+1];
   FILE *fp;
   int line_num=0;

   if ((c_ptr=XGetDefault(mainDisplay,TOOL_NAME,"MimeTypesFile")) != NULL) {
      strcpy(fname, c_ptr);
   } else {
      sprintf(fname, "%s/.mime.types", homeDir);
   }
   if ((fp=fopen(fname, "r")) == NULL) return;
   while ((buf=UtilGetALine(fp)) != NULL) {
      line_num++;
      if (*buf != '#') {
         char *c_ptr1=strtok(buf, " \t\n\r");

         if (c_ptr1 != NULL && *c_ptr1 != '\0') {
            int ok=TRUE;
            char *sub_type=NULL;

            if ((sub_type=strchr(c_ptr1, '/')) == NULL) {
               ok = FALSE;
            } else {
               *sub_type++ = '\0';
               if (*sub_type == '\0') ok = FALSE;
               if (*c_ptr1 == '*') ok = FALSE;
            }
            if (!ok) {
               sprintf(gszMsgBox, "Malformed line %1d in '%s'.", line_num,
                     fname);
               Msg(gszMsgBox);
            } else {
               AddAMimeType(c_ptr1, sub_type);
            }
         }
      }
      cfree(buf);
   }
   fclose(fp);
   if (debugRemote) DumpMimeTypesInfo();
}

static
void FillMailCapInfo()
{
   char *spec, *fname, *colon_ptr, *c_ptr;

   if ((c_ptr=getenv("MAILCAPS")) != NULL) {
      spec = UtilStrDup(c_ptr);
   } else {
      sprintf(gszMsgBox, "%s/.mailcap", homeDir);
      spec = UtilStrDup(gszMsgBox);
   }
   if (spec == NULL) { FailAllocMessage(); return; }

   fname = spec;
   colon_ptr = strchr(fname, ':');
   while (fname != NULL) {
      if (colon_ptr != NULL) *colon_ptr = '\0';
      if (*fname != '\0') {
         FILE *fp;

         if ((fp=fopen(fname, "r")) == NULL) {
            sprintf(gszMsgBox, "Can not open mailcap file '%s'.", fname);
            Msg(gszMsgBox);
         } else {
            int line_num=0;
            char *buf;

            while ((buf=UtilGetAContinuedLine(fp)) != NULL) {
               line_num++;
               UtilTrimBlanks(buf);
               if (*buf != '\0' && *buf != '#') {
                  char *first_semi_ptr=strchr(buf, ';');
                  char *second_semi_ptr=NULL, *sub_type=NULL;
                  int ok=TRUE;

                  if (first_semi_ptr == NULL) {
                     ok = FALSE;
                  } else {
                     *first_semi_ptr++ = '\0';
                     while (*first_semi_ptr == ' ' && *first_semi_ptr == '\t') {
                        first_semi_ptr++;
                     }
                     second_semi_ptr = strchr(first_semi_ptr, ';');
                     if (second_semi_ptr != NULL) {
                        *second_semi_ptr++ = '\0';
                     }
                     if ((sub_type=strchr(buf, '/')) == NULL) {
                        ok = FALSE;
                     } else {
                        *sub_type++ = '\0';
                     }
                  }
                  if (!ok) {
                     sprintf(gszMsgBox, "Malformed line %1d in '%s'.",
                           line_num, fname);
                     Msg(gszMsgBox);
                  } else {
                     if (first_semi_ptr!=NULL) UtilTrimBlanks(first_semi_ptr);
                     if (second_semi_ptr!=NULL) UtilTrimBlanks(second_semi_ptr);
                     AddAMailCap(buf, sub_type, first_semi_ptr,
                           second_semi_ptr);
                  }
               }
               cfree(buf);
            }
         }
         fclose(fp);
      }
      if (colon_ptr == NULL) break;
      *colon_ptr++ = ':';
      fname = colon_ptr;
      colon_ptr = strchr(fname, ':');
   }
   if (debugRemote) DumpMailCapInfo();
}

static
void MalformedMailCapEntryMessage(mail_cap_ptr)
   struct MailCapRec *mail_cap_ptr;
{
   if (mail_cap_ptr->params == NULL) {
      sprintf(gszMsgBox, "%s:\n\n%s/%s; %s",
            "Malformed mailcap entry", mail_cap_ptr->main_type,
            mail_cap_ptr->sub_type, mail_cap_ptr->cmd);
   } else {
      sprintf(gszMsgBox, "%s:\n\n%s/%s; %s; %s",
            "Malformed mailcap entry", mail_cap_ptr->main_type,
            mail_cap_ptr->sub_type, mail_cap_ptr->cmd,
            mail_cap_ptr->params);
   }
   MsgBox(gszMsgBox, TOOL_NAME, INFO_MB);
}

static
char *GetContentValue(name, params)
   char *name, *params;
{
   char *buf, *c_ptr;

   if (params == NULL) return NULL;

   buf = UtilStrDup(params);
   if (buf == NULL) { FailAllocMessage(); return NULL; }
   for (c_ptr=strtok(buf, ";"); c_ptr != NULL; c_ptr=strtok(NULL, ";")) {
      char *equal_ptr=strchr(c_ptr, '-');

      if (equal_ptr != NULL) {
         char *c_ptr1=c_ptr;

         while (*c_ptr1 == ' ' || *c_ptr1 == '\t') c_ptr1++;
         *equal_ptr = '\0';
         if (UtilStrICmp(c_ptr1, name) == 0) {
            char *return_buf=UtilStrDup(&equal_ptr[1]);

            cfree(buf);
            return return_buf;
         }
         *equal_ptr = '=';
      }
   }
   cfree(buf);
   return NULL;
}

static int mimeViewerInitialized=FALSE;

static
char *GetMimeViewer(psz_content_type)
   char *psz_content_type;
{
   struct MailCapRec *mail_cap_ptr;
   struct MimeTypeRec *mime_type_ptr;
   char *main_type=UtilStrDup(psz_content_type), *sub_type=NULL;
   char *params=NULL, *buf=NULL, *c_ptr;
   int buf_sz=0, buf_index=0;

   if (main_type == NULL) { FailAllocMessage(); return NULL; }
   UtilTrimBlanks(main_type);
   params = strchr(main_type, ';');
   if (params != NULL) *params++ = '\0';

   if (UtilStrICmp(main_type, "application/x-tgif") == 0) {
      cfree(main_type);
      return NULL;
   }
   sub_type = strchr(main_type, '/');
   if (sub_type == NULL) { cfree(main_type); return NULL; }
   *sub_type++ = '\0';
   if (UtilStrICmp(main_type, "multipart") == 0) {
      cfree(main_type);
      sprintf(gszMsgBox, "Multipart MIME not supported.");
      MsgBox(gszMsgBox, TOOL_NAME, INFO_MB);
      return NULL;
   }

   if (!mimeViewerInitialized) {
      mimeViewerInitialized = TRUE;
      FillMimeTypesInfo();
      FillMailCapInfo();
   }
   for (mime_type_ptr=topMimeTypesInfo; mime_type_ptr != NULL;
         mime_type_ptr=mime_type_ptr->next) {
      if (UtilStrICmp(main_type, mime_type_ptr->main_type) == 0) {
         if (strcmp(mime_type_ptr->sub_type, "*") == 0) {
            break;
         } else if (UtilStrICmp(sub_type, mime_type_ptr->sub_type) == 0) {
            break;
         }
      }
   }
   if (mime_type_ptr == NULL) { cfree(main_type); return NULL; }

   for (mail_cap_ptr=topMailCapInfo; mail_cap_ptr != NULL;
         mail_cap_ptr=mail_cap_ptr->next) {
      if (UtilStrICmp(main_type, mail_cap_ptr->main_type) == 0) {
         if (strcmp(mail_cap_ptr->sub_type, "*") == 0) {
            break;
         } else if (UtilStrICmp(sub_type, mail_cap_ptr->sub_type) == 0) {
            break;
         }
      }
   }
   cfree(main_type);
   if (mail_cap_ptr == NULL) return NULL;

   buf_sz = 0x100;
   buf = (char*)calloc(buf_sz+1, sizeof(char));
   if (buf == NULL) { FailAllocMessage(); return NULL; }
   c_ptr = mail_cap_ptr->cmd;
   while (*c_ptr != '\0') {
      if (*c_ptr == '\\' && c_ptr[1] == '\0') {
         MalformedMailCapEntryMessage(mail_cap_ptr);
         cfree(buf);
         return NULL;
      }
      if (*c_ptr == '%') {
         char *tmp_buf=NULL, *right_bracket;
         int num_ch_to_skip=(-1), len;

         switch (c_ptr[1]) {
         case 's': tmp_buf=UtilStrDup("%s"); num_ch_to_skip=2; break;
         case 'S': tmp_buf=UtilStrDup("%S"); num_ch_to_skip=2; break;
         case 't':
            len = strlen(main_type)+strlen(sub_type)+1;
            tmp_buf = (char*)calloc(len+1, sizeof(char));
            if (tmp_buf != NULL) sprintf(tmp_buf, "%s/%s", main_type, sub_type);
            num_ch_to_skip = 2;
            break;
         case '{':
            if ((right_bracket=strchr(&c_ptr[2], '}')) == NULL) {
               if (mail_cap_ptr->params == NULL) {
                  sprintf(gszMsgBox, "%s:\n\n%s/%s; %s",
                        "Can not find matching '}' for mailcap entry",
                        mail_cap_ptr->main_type, mail_cap_ptr->sub_type,
                        mail_cap_ptr->cmd);
               } else {
                  sprintf(gszMsgBox, "%s:\n\n%s/%s; %s; %s",
                        "Can not find matching '}' for mailcap entry",
                        mail_cap_ptr->main_type, mail_cap_ptr->sub_type,
                        mail_cap_ptr->cmd, mail_cap_ptr->params);
               }
               MsgBox(gszMsgBox, TOOL_NAME, INFO_MB);
               cfree(buf);
               return NULL;
            }
            num_ch_to_skip = right_bracket-c_ptr+1;
            *right_bracket = '\0';
            if ((tmp_buf=GetContentValue(&c_ptr[2], params)) == NULL) {
               sprintf(gszMsgBox, "%s '%s' %s.",
                     "HTTP: can not find the", &c_ptr[2],
                     "information in the message Content-Type");
               MsgBox(gszMsgBox, TOOL_NAME, INFO_MB);
               *right_bracket = '}';
               cfree(buf);
               return NULL;
            }
            *right_bracket = '}';
            break;
         default:
            MalformedMailCapEntryMessage(mail_cap_ptr);
            cfree(buf);
            return NULL;
         }
         if (num_ch_to_skip < 0 || tmp_buf == NULL) {
            cfree(buf);
            FailAllocMessage();
            return NULL;
         }
         len = strlen(tmp_buf);
         while (buf_index+len+1 >= buf_sz) {
            buf_sz += 0x100;
            if ((buf=(char*)realloc(buf, buf_sz+1)) == NULL) return NULL;
         }
         strcpy(&buf[buf_index], tmp_buf);
         buf_index += len;

         cfree(tmp_buf);
         c_ptr += num_ch_to_skip;
      } else {
         if (buf_index+1 >= buf_sz) {
            buf_sz += 0x100;
            if ((buf=(char*)realloc(buf, buf_sz+1)) == NULL) return NULL;
         }
         if (*c_ptr == '\\') c_ptr++;
         buf[buf_index++] = *c_ptr++;
      }
   }
   buf[buf_index] = '\0';
   return buf;
}

static
int SaveTmpURLToFile(psz_url, psz_msg, tmp_fname, new_file_name)
   char *psz_url, *psz_msg, *tmp_fname, *new_file_name;
{
   int first_msg_len=(psz_msg == NULL ? 0 : strlen(psz_msg));
   char *msg=(char*)malloc(first_msg_len+10+strlen(psz_url)+30);
   char *msg1=(char*)malloc(strlen(curDirIsLocal ? curDir : curLocalDir)+30);
   int rc=TG_REMOTE_STATUS_OK;

   if (msg == NULL || msg1 == NULL) { FailAllocMessage(); return INVALID; }

   if (psz_msg == NULL) {
      sprintf(msg, "Save %s as:", psz_url);
   } else {
      sprintf(msg, "%s  Save %s as:", psz_msg, psz_url);
   }
   sprintf(msg1, "( working directory: %s )", curDirIsLocal ? curDir :
         curLocalDir);
   Dialog(msg, msg1, new_file_name);
   cfree(msg);
   cfree(msg1);
   UtilTrimBlanks(new_file_name);

   if (*new_file_name == '\0') return INVALID;
   if (!OkayToCreateFile(new_file_name)) return INVALID;

   rc = UtilCopyFile(tmp_fname, new_file_name);
   if (rc != TG_REMOTE_STATUS_OK) {
      switch (rc) {
      case TG_REMOTE_STATUS_READ:
         sprintf(gszMsgBox, "Can not open '%s' for read.", tmp_fname);
         break;
      case TG_REMOTE_STATUS_WRITE:
         sprintf(gszMsgBox, "Can not open '%s' for write.", new_file_name);
         break;
      case TG_REMOTE_STATUS_FILE:
         sprintf(gszMsgBox,
               "Fail to write to '%s'.  File system may be full.",
               new_file_name);
         break;
      default: sprintf(gszMsgBox, "Unknown error in copying files"); break;
      }
      MsgBox(gszMsgBox, TOOL_NAME, INFO_MB);
      return INVALID;
   }
   return rc;
}

static
int UseMimeViewer(psz_url, psz_content_type, tmp_fname)
   char *psz_url, *psz_content_type, *tmp_fname;
   /* returns FALSE if psz_url is to be interpreted as a tgif file or text */
   /* returns TRUE if external viewer is launched */
   /* returns INVALID if there's an error or if the user canceled */
{
   char new_file_name[MAXPATHLENGTH+1];
   char *viewer=NULL, *cmd=NULL, *big_s=NULL;
   int launch_remote_file=FALSE;

   if (psz_content_type == NULL ||
         UtilStrICmp(psz_content_type, "application/x-tgif") == 0 ||
         strncmp(psz_url, "http:", 5) != 0) {
      return FALSE;
   }
   sprintf(gszMsgBox, "%s", psz_url);
   Msg(gszMsgBox);
   sprintf(gszMsgBox, "    Content-type: %s", psz_content_type);
   Msg(gszMsgBox);
   /*
    * fprintf(stderr, "URL: %s\n", psz_url);
    * fprintf(stderr, "Content-Type: %s\n", psz_content_type);
    * fprintf(stderr, "tmp_fname: %s\n", tmp_fname);
    */
   viewer = GetMimeViewer(psz_content_type);

   /* if content_type is application/*, we should try to just save it */
   if (viewer == NULL) {
      char *slash_ptr=(psz_content_type == NULL ? NULL :
            strchr(psz_content_type, '/'));

      if (slash_ptr != NULL) {
         *slash_ptr = '\0';
         if (UtilStrICmp(psz_content_type, "application") == 0 ||
               UtilStrICmp(psz_content_type, "audio") == 0 ||
               UtilStrICmp(psz_content_type, "image") == 0 ||
               UtilStrICmp(psz_content_type, "video") == 0 ||
               ((*psz_content_type == 'X' || *psz_content_type == 'x') &&
               psz_content_type[1] == '-')) {
            *slash_ptr = '/';
            SaveTmpURLToFile(psz_url, "No viewer found.", tmp_fname,
                  new_file_name);
            return TRUE;
         }
         *slash_ptr = '/';
      }
      return FALSE;
   }
   /*
    * fprintf(stderr, "Launch viewer: %s\n", viewer);
    */
   if ((big_s=strstr(viewer, "%S")) == NULL) {
      if (SaveTmpURLToFile(psz_url, NULL, tmp_fname, new_file_name) ==
            INVALID) {
         cfree(viewer);
         return INVALID;
      }
      cmd = (char*)malloc(strlen(viewer)+strlen(new_file_name)+40);
   } else {
      big_s[1] = 's';
      launch_remote_file = TRUE;
      cmd = (char*)malloc(strlen(viewer)+strlen(psz_url)+40);
   }
   if (cmd == NULL) {
      cfree(viewer);
      return INVALID;
   }
   LaunchViewer(launch_remote_file, viewer, cmd, psz_url, new_file_name);
   cfree(viewer);
   cfree(cmd);
   return TRUE;
}

static char *gpszViewerInfo=NULL;

static
char *GetViewer(ext_str)
   char *ext_str;
   /* caller needs to call cfree() if not return NULL */
{
   int ext_len=strlen(ext_str), value_len, orig_len;
   char resource_str[MAXSTRING+1];
   char *value=NULL, *c_ptr;

   for (c_ptr=gpszViewerInfo; c_ptr != NULL && *c_ptr != '\0'; ) {
      char *cr=strchr(c_ptr, '\n'), *next_cr=NULL;
      int cmp_result;

      if (cr == NULL) {
         cfree(gpszViewerInfo);
         gpszViewerInfo = NULL;
         return NULL;
      }
      *cr = '\0';
      cmp_result = UtilStrICmp(c_ptr, ext_str);
      *cr++ = '\n';
      next_cr = strchr(cr, '\n');
      if (next_cr == NULL) {
         cfree(gpszViewerInfo);
         gpszViewerInfo = NULL;
         return NULL;
      }
      if (cmp_result == 0) {
         char *return_str;

         *next_cr = '\0';
         return_str = (*cr == '\0' ? NULL : UtilStrDup(cr));
         *next_cr = '\n';
         return return_str;
      }
      c_ptr = &next_cr[1];
   }
   sprintf(resource_str, "%sViewer", ext_str);
   if ((c_ptr=XGetDefault(mainDisplay,TOOL_NAME,resource_str)) != NULL) {
      int count=0;

      value = (char*)malloc(strlen(c_ptr)+2);
      if (value == NULL) return NULL;
      sprintf(value, "%s\n", c_ptr);
      for (c_ptr=strstr(value,"%s"); c_ptr != NULL;
            c_ptr=strstr(&c_ptr[2],"%s")) {
         count++;
      }
      for (c_ptr=strstr(value,"%S"); c_ptr != NULL;
            c_ptr=strstr(&c_ptr[2],"%S")) {
         count++;
      }
      if (count > 1) {
         sprintf(gszMsgBox, "Invalid %s*%s: '%s'.", TOOL_NAME, resource_str,
               value);
         MsgBox(gszMsgBox, TOOL_NAME, INFO_MB);
         strcpy(value, "\n");
      }
   } else {
      if ((value=UtilStrDup("\n")) == NULL) return NULL;
   }
   value_len = strlen(value);
   orig_len = (gpszViewerInfo == NULL ? 0 : strlen(gpszViewerInfo));
   if ((gpszViewerInfo=(gpszViewerInfo == NULL ?
         (char*)malloc(orig_len+ext_len+1+value_len+2) :
         (char*)realloc(gpszViewerInfo, orig_len+ext_len+1+value_len+2))) ==
         NULL) {
      return NULL;
   }
   sprintf(&gpszViewerInfo[orig_len], "%s\n%s", ext_str, value);
   if (value[value_len-1] == '\n') value[--value_len] = '\0';
   if (*value == '\0') {
      cfree(value);
      return NULL;
   }
   return value;
}

int UseViewer(psz_url, psz_content_type, tmp_fname)
   char *psz_url, *psz_content_type, *tmp_fname;
   /* returns FALSE if psz_url is to be interpreted as a tgif file or text */
   /* returns TRUE if external viewer is launched */
   /* returns INVALID if there's an error or if the user canceled */
{
   char *ext_str=UtilStrRChr(psz_url, '/');
   char new_file_name[MAXPATHLENGTH+1];
   char *viewer=NULL, *cmd=NULL, *big_s=NULL, *c_ptr, saved_ch='\0';
   int rc, launch_remote_file=FALSE;

   if (ext_str == NULL) {
      return UseMimeViewer(psz_url, psz_content_type, tmp_fname);
   }
   for (c_ptr=(&ext_str[1]); *c_ptr != '\0'; c_ptr++) {
      if (*c_ptr == '#' || *c_ptr == '?') {
         saved_ch = *c_ptr;
         *c_ptr = '\0';
         break;
      }
   }
   if ((ext_str=strchr(ext_str, '.')) == NULL) {
      if (c_ptr != NULL) *c_ptr = saved_ch;
      return UseMimeViewer(psz_url, psz_content_type, tmp_fname);
   }
   ext_str++;
   viewer = GetViewer(ext_str);
   if (c_ptr != NULL) *c_ptr = saved_ch;
   if (viewer == NULL) {
      return UseMimeViewer(psz_url, psz_content_type, tmp_fname);
   }
   if ((big_s=strstr(viewer, "%S")) == NULL) {
      if (SaveTmpURLToFile(psz_url, NULL, tmp_fname, new_file_name) ==
            INVALID) {
         cfree(viewer);
         return INVALID;
      }
      cmd = (char*)malloc(strlen(viewer)+strlen(new_file_name)+40);
   } else {
      big_s[1] = 's';
      launch_remote_file = TRUE;
      cmd = (char*)malloc(strlen(viewer)+strlen(psz_url)+40);
   }
   if (cmd == NULL) {
      cfree(viewer);
      return INVALID;
   }
   LaunchViewer(launch_remote_file, viewer, cmd, psz_url, new_file_name);
   cfree(viewer);
   cfree(cmd);
   return TRUE;
}

int LoadRemoteFileFromMem(psz_url, psz_buf, psz_content_type, buf_sz, is_html)
   char *psz_url, *psz_buf, *psz_content_type;
   int buf_sz, is_html;
{
   char *tmp_fname=WriteRemoteFileIntoTemp(psz_buf, buf_sz);
   int navigating=navigatingBackAndForth, rc_loadfile;
   int url_is_html=UrlIsHtml(psz_url);

   if (tmp_fname == NULL) return FALSE;

   if ((url_is_html || !is_html) && (UseViewer(psz_url, psz_content_type,
         tmp_fname) != FALSE)) {
      unlink(tmp_fname);
      cfree(tmp_fname);
      return FALSE;
   }
   if (!navigating) navigatingBackAndForth = TRUE;
   rc_loadfile = LoadFile(tmp_fname, (-1));
   if (!navigating) navigatingBackAndForth = FALSE;

   if (!rc_loadfile) {
      NewProc();
      PasteString(psz_buf);
   }
   SetCurDir(psz_url);
   curFileDefined = TRUE;
   RedrawTitleWindow();
   sprintf(gszMsgBox, "Current file is '%s'.", psz_url);
   Msg(gszMsgBox);
   unlink(tmp_fname);
   cfree(tmp_fname);
   if (!navigating) CommitNavigate();
   return TRUE;
}

void InitRemote()
{
   char *c_ptr;
   int val;

   autoHyperSpaceOnRemote = TRUE;
   if ((c_ptr=XGetDefault(mainDisplay,TOOL_NAME,"AutoHyperSpaceOnRemote")) !=
         NULL && (strcmp(c_ptr, "false") == 0 || strcmp(c_ptr, "False") == 0)) {
      autoHyperSpaceOnRemote = FALSE;
   }
   allowLaunchInHyperSpace = FALSE;
   if ((c_ptr=XGetDefault(mainDisplay,TOOL_NAME,"AllowLaunchInHyperSpace")) !=
         NULL && (strcmp(c_ptr, "true") == 0 || strcmp(c_ptr, "True") == 0)) {
      allowLaunchInHyperSpace = TRUE;
   }
   if ((c_ptr=XGetDefault(mainDisplay,TOOL_NAME,"DebugHttp")) != NULL &&
         sscanf(c_ptr, "%d", &val) == 1 && val >= 0) {
      HttpDebug(val);
   }
   if ((c_ptr=XGetDefault(mainDisplay,TOOL_NAME,"DebugFtp")) != NULL &&
         sscanf(c_ptr, "%d", &val) == 1 && val >= 0) {
      FtpDebug(val);
   }
}

void CleanUpRemote()
{
   if (gpszViewerInfo != NULL) {
      cfree(gpszViewerInfo);
      gpszViewerInfo = NULL;
   }
   FreeMimeTypesInfo();
   FreeMailCapInfo();
}
