/*
 * Author:      William Chia-Wei Cheng (william@cs.ucla.edu)
 *
 * Copyright (C) 1990-1994, William Cheng.
 *
 * Permission limited to the use, copy, modify, and distribute this software
 * and its documentation for any purpose is hereby granted by the Author without
 * fee, provided that the above copyright notice appear in all copies and
 * that both the copyright notice and this permission notice appear in
 * supporting documentation, and that the name of the Author not be used
 * in advertising or publicity pertaining to distribution of the software
 * without specific, written prior permission.  The Author makes no
 * representations about the suitability of this software for any purpose.
 * It is provided "as is" without express or implied warranty.  All other
 * rights (including the right to sell "tgif" and the right to sell derivative
 * works of tgif) are reserved by the Author.
 *
 * THE AUTHOR DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS SOFTWARE,
 * INCLUDING ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS, IN NO
 * EVENT SHALL THE AUTHOR BE LIABLE FOR ANY SPECIAL, INDIRECT OR
 * CONSEQUENTIAL DAMAGES OR ANY DAMAGES WHATSOEVER RESULTING FROM LOSS OF
 * USE, DATA OR PROFITS, WHETHER IN AN ACTION OF CONTRACT, NEGLIGENCE OR
 * OTHER TORTIOUS ACTION, ARISING OUT OF OR IN CONNECTION WITH THE USE OR
 * PERFORMANCE OF THIS SOFTWARE.
 */
#ifndef lint
static char RCSid[] =
      "@(#)$Header: /export/bourbon/multimedia/william/X11/TGIF2/RCS/ruler.c,v 2.24 1994/04/04 00:49:06 william Exp $";
#endif

#include <stdio.h>
#include <X11/Xlib.h>
#include "const.h"
#include "types.h"

#include "font.e"
#include "grid.e"
#include "mainmenu.e"
#include "raster.e"
#ifndef _NO_EXTERN
#include "ruler.e"
#endif
#include "setup.e"

#define INCH_H 8
#define HALF_INCH_H 6
#define QUARTER_INCH_H 4
#define MIN_INCH_H 2

int	showMeasurement=FALSE;

static GC	rulerGC;

static int	oldXOff=0, oldYOff=0;
static int	rulerLen=0;
static int	hRulerJustRedrawn=TRUE, justUnFrozen=FALSE;
static int	freezeMarkRulerText=FALSE, frozenXOff=0, frozenYOff;

void InitRuler ()
{
   XGCValues	values;
   char		* c_ptr;

   values.foreground = myFgPixel;
   values.background = myBgPixel;
   values.fill_style = FillSolid;
   values.font = rulerFontPtr->fid;

   rulerGC = XCreateGC (mainDisplay, mainWindow,
         GCForeground | GCBackground | GCFillStyle | GCFont, &values);

   showMeasurement = FALSE;
   if ((c_ptr = XGetDefault (mainDisplay, TOOL_NAME, "ShowMeasurement")) !=
         NULL && (strcmp ("True", c_ptr) == 0 || strcmp ("true", c_ptr) == 0))
      showMeasurement = TRUE;

   rulerLen = rulerW-1;
}

void CleanUpRuler ()
{
   XFreeGC (mainDisplay, rulerGC);
}

void RedrawHRuler ()
{
   register int	i, pos, len, index;
   int		inc, abs_inc;
   char		s[5];
   XEvent	ev;

   XClearWindow (mainDisplay, hRuleWindow);

   XSync (mainDisplay, False);
   while (XCheckWindowEvent (mainDisplay, hRuleWindow, ExposureMask, &ev)) ;

   switch (gridSystem)
   {
      case ENGLISH_GRID:
         inc = (zoomedIn ? (xyEnglishGrid<<zoomScale) : xyEnglishGrid);
         abs_inc = ABS_SIZE(inc);

         if (drawOrigX % abs_inc == 0)
         {
            i = 0;
            pos = 0;
         }
         else
         {
            i = ((int)(drawOrigX / abs_inc) + 1) * abs_inc - drawOrigX;
            pos = ZOOMED_SIZE(i);
         }
         for ( ; i<drawWinW; i+=abs_inc, pos+=inc)
         {
            if ((GRID_ZOOMED_SIZE(i+drawOrigX)) % ONE_INCH == 0)
            {
               index = (int)((GRID_ZOOMED_SIZE(i+drawOrigX)) / ONE_INCH);
               sprintf (s, "%1d", GRID_ABS_SIZE(index));
               len = strlen (s);
               XDrawString (mainDisplay, hRuleWindow, rulerGC,
                     pos-((len*rulerFontWidth)>>1),
                     rulerW-INCH_H-rulerFontDes, s, len);
               XDrawLine (mainDisplay, hRuleWindow, defaultGC, pos, rulerW,
                     pos, rulerW-INCH_H);
            }
            else if ((GRID_ZOOMED_SIZE(i+drawOrigX)) % HALF_INCH == 0)
               XDrawLine (mainDisplay, hRuleWindow, defaultGC, pos, rulerW,
                     pos, rulerW-HALF_INCH_H);
            else if ((GRID_ZOOMED_SIZE(i+drawOrigX)) % QUARTER_INCH == 0)
               XDrawLine (mainDisplay, hRuleWindow, defaultGC, pos, rulerW,
                     pos, rulerW-QUARTER_INCH_H);
            else
               XDrawLine (mainDisplay, hRuleWindow, defaultGC, pos, rulerW,
                     pos, rulerW-MIN_INCH_H);
         }
         break;
      case METRIC_GRID:
         inc = (zoomedIn ? (xyMetricGrid<<zoomScale) : xyMetricGrid);
         abs_inc = ABS_SIZE(inc);

         if (drawOrigX % abs_inc == 0)
         {
            i = 0;
            pos = 0;
         }
         else
         {
            i = ((int)(drawOrigX / abs_inc) + 1) * abs_inc - drawOrigX;
            pos = ZOOMED_SIZE(i);
         }
         for ( ; i<drawWinW; i+=abs_inc, pos+=inc)
         {
            if ((GRID_ZOOMED_SIZE(i+drawOrigX)) % ONE_CM == 0)
            {
               index = (int)((GRID_ZOOMED_SIZE(i+drawOrigX)) / ONE_CM);
               sprintf (s, "%1d", GRID_ABS_SIZE(index));
               len = strlen (s);
               XDrawString (mainDisplay, hRuleWindow, rulerGC,
                     pos-rulerFontWidth/2, rulerW-HALF_INCH_H-rulerFontDes,
                     s, len);
               XDrawLine (mainDisplay, hRuleWindow, defaultGC, pos, rulerW,
                     pos, rulerW-HALF_INCH_H);
            }
            else if ((GRID_ZOOMED_SIZE(i+drawOrigX)) % FIVE_MM == 0)
               XDrawLine (mainDisplay, hRuleWindow, defaultGC, pos, rulerW,
                     pos, rulerW-QUARTER_INCH_H);
            else
               XDrawLine (mainDisplay, hRuleWindow, defaultGC, pos, rulerW,
                     pos, rulerW-MIN_INCH_H);
         }
         break;
   }
   oldXOff = 0;
   XDrawLine (mainDisplay, hRuleWindow, revDefaultGC, oldXOff, 0,
         oldXOff, rulerLen);
   hRulerJustRedrawn = TRUE;
   justUnFrozen = FALSE;
}

void RedrawVRuler ()
{
   register int	i, pos, len, index;
   int		inc, abs_inc;
   char		s[5];
   XEvent	ev;

   XClearWindow (mainDisplay, vRuleWindow);

   XSync (mainDisplay, False);
   while (XCheckWindowEvent (mainDisplay, vRuleWindow, ExposureMask, &ev)) ;

   switch (gridSystem)
   {
      case ENGLISH_GRID:
         inc = (zoomedIn ? (xyEnglishGrid<<zoomScale) : xyEnglishGrid);
         abs_inc = ABS_SIZE(inc);

         if (drawOrigY % abs_inc == 0)
         {
            i = 0;
            pos = 0;
         }
         else
         {
            i = ((int)(drawOrigY / abs_inc) + 1) * abs_inc - drawOrigY;
            pos = ZOOMED_SIZE(i);
         }
         for ( ; i<drawWinH; i+=abs_inc, pos+=inc)
         {
            if ((GRID_ZOOMED_SIZE(i+drawOrigY)) % ONE_INCH == 0)
            {
               index = (int)((GRID_ZOOMED_SIZE(i+drawOrigY)) / ONE_INCH);
               sprintf (s, "%1d", GRID_ABS_SIZE(index));
               len = strlen (s);
               XDrawString (mainDisplay, vRuleWindow, rulerGC, 1,
                     pos-rulerFontHeight/2+rulerFontAsc, s, len);
               XDrawLine (mainDisplay, vRuleWindow, defaultGC, rulerW, pos,
                     rulerW-INCH_H, pos);
            }
            else if ((GRID_ZOOMED_SIZE(i+drawOrigY)) % HALF_INCH == 0)
               XDrawLine (mainDisplay, vRuleWindow, defaultGC, rulerW, pos,
                     rulerW-HALF_INCH_H, pos);
            else if ((GRID_ZOOMED_SIZE(i+drawOrigY)) % QUARTER_INCH == 0)
               XDrawLine (mainDisplay, vRuleWindow, defaultGC, rulerW, pos,
                     rulerW-QUARTER_INCH_H, pos);
            else
               XDrawLine (mainDisplay, vRuleWindow, defaultGC, rulerW, pos,
                     rulerW-MIN_INCH_H, pos);
         }
         break;
      case METRIC_GRID:
         inc = (zoomedIn ? (xyMetricGrid<<zoomScale) : xyMetricGrid);
         abs_inc = ABS_SIZE(inc);

         if (drawOrigY % abs_inc == 0)
         {
            i = 0;
            pos = 0;
         }
         else
         {
            i = ((int)(drawOrigY / abs_inc) + 1) * abs_inc - drawOrigY;
            pos = ZOOMED_SIZE(i);
         }
         for ( ; i<drawWinH; i+=abs_inc, pos+=inc)
         {
            if ((GRID_ZOOMED_SIZE(i+drawOrigY)) % ONE_CM == 0)
            {
               index = (int)((GRID_ZOOMED_SIZE(i+drawOrigY)) / ONE_CM);
               sprintf (s, "%1d", GRID_ABS_SIZE(index));
               len = strlen (s);
               XDrawString (mainDisplay, vRuleWindow, rulerGC, 1,
                     pos-rulerFontHeight/2+rulerFontAsc, s, len);
               XDrawLine (mainDisplay, vRuleWindow, defaultGC, rulerW, pos,
                     rulerW-HALF_INCH_H, pos);
            }
            else if ((GRID_ZOOMED_SIZE(i+drawOrigY)) % FIVE_MM == 0)
               XDrawLine (mainDisplay, vRuleWindow, defaultGC, rulerW, pos,
                     rulerW-QUARTER_INCH_H, pos);
            else
               XDrawLine (mainDisplay, vRuleWindow, defaultGC, rulerW, pos,
                     rulerW-MIN_INCH_H, pos);
         }
         break;
   }
   if (!freezeMarkRulerText && showMeasurement && !hRulerJustRedrawn &&
         oldYOff != 0)
   {
      char	buf[80];

      sprintf (buf, "[%1d,%1d]", ABS_X(oldXOff), ABS_Y(oldYOff));
      XDrawString (mainDisplay, hRuleWindow, revDefaultGC, 8, 2+defaultFontAsc,
            buf, strlen(buf));
      hRulerJustRedrawn = TRUE;
   }
   oldYOff = 0;
   XDrawLine (mainDisplay, vRuleWindow, revDefaultGC, 0, oldYOff,
         rulerLen, oldYOff);
}

void MarkRulers (XOff, YOff)
   int	XOff, YOff;
{
   char buf[80];

   XDrawLine (mainDisplay,hRuleWindow,revDefaultGC,oldXOff,0,oldXOff,rulerLen);
   XDrawLine (mainDisplay,vRuleWindow,revDefaultGC,0,oldYOff,rulerLen,oldYOff);

   if (hRulerJustRedrawn)
      hRulerJustRedrawn= FALSE;
   else if (!freezeMarkRulerText && showMeasurement)
   {
      if (justUnFrozen)
      {
         justUnFrozen = FALSE;
         sprintf (buf, "[%1d,%1d]", ABS_X(frozenXOff), ABS_Y(frozenYOff));
      }
      else
         sprintf (buf, "[%1d,%1d]", ABS_X(oldXOff), ABS_Y(oldYOff));
      XDrawString (mainDisplay, hRuleWindow, revDefaultGC, 8, 2+defaultFontAsc,
            buf, strlen(buf));
   }

   XDrawLine (mainDisplay,hRuleWindow,revDefaultGC,XOff,0,XOff,rulerLen);
   XDrawLine (mainDisplay,vRuleWindow,revDefaultGC,0,YOff,rulerLen,YOff);

   if (!freezeMarkRulerText && showMeasurement)
   {
      sprintf (buf, "[%1d,%1d]", ABS_X(XOff), ABS_Y(YOff));
      XDrawString (mainDisplay, hRuleWindow, revDefaultGC, 8, 2+defaultFontAsc,
            buf, strlen(buf));
   }

   oldXOff = XOff;
   oldYOff = YOff;
}

void RedrawRulers ()
{
   RedrawHRuler ();
   RedrawVRuler ();
}

void RulersEventHandler (input)
   XEvent       * input;
{
   if (input->xany.window == vRuleWindow)
   {
      if (input->type == Expose)
         RedrawVRuler ();
      else if (input->type == EnterNotify)
         SetMouseStatus ("(none)", "(none)", "(none)");
   }
   else if (input->xany.window == hRuleWindow)
   {
      if (input->type == Expose)
         RedrawHRuler ();
      else if (input->type == EnterNotify)
         SetMouseStatus ("(none)", "(none)", "(none)");
   }
}

void FreezeMarkRulerText ()
{
   freezeMarkRulerText = TRUE;
   frozenXOff = oldXOff;
   frozenYOff = oldYOff;
}

void UnFreezeMarkRulerText ()
{
   freezeMarkRulerText = FALSE;
   justUnFrozen = TRUE;
}

static int oldLtX, oldLtY, oldRbX, oldRbY, oldMdX, oldMdY;

static
void DoIntervalRulers ()
{
   if (oldLtX == oldRbX)
      XDrawLine(mainDisplay,hRuleWindow,revDefaultGC,oldLtX,0,oldLtX,rulerLen);
   else if (oldLtX==oldMdX || oldRbX==oldMdX)
   {
      XDrawLine(mainDisplay,hRuleWindow,revDefaultGC,oldLtX,0,oldLtX,rulerLen);
      XDrawLine(mainDisplay,hRuleWindow,revDefaultGC,oldRbX,0,oldRbX,rulerLen);
   }
   else
   {
      XDrawLine(mainDisplay,hRuleWindow,revDefaultGC,oldLtX,0,oldLtX,rulerLen);
      XDrawLine(mainDisplay,hRuleWindow,revDefaultGC,oldMdX,0,oldMdX,rulerLen);
      XDrawLine(mainDisplay,hRuleWindow,revDefaultGC,oldRbX,0,oldRbX,rulerLen);
   }
   if (oldLtY == oldRbY)
      XDrawLine(mainDisplay,vRuleWindow,revDefaultGC,0,oldLtY,rulerLen,oldLtY);
   else if (oldLtY==oldMdY || oldRbY==oldMdY)
   {
      XDrawLine(mainDisplay,vRuleWindow,revDefaultGC,0,oldLtY,rulerLen,oldLtY);
      XDrawLine(mainDisplay,vRuleWindow,revDefaultGC,0,oldRbY,rulerLen,oldRbY);
   }
   else
   {
      XDrawLine(mainDisplay,vRuleWindow,revDefaultGC,0,oldLtY,rulerLen,oldLtY);
      XDrawLine(mainDisplay,vRuleWindow,revDefaultGC,0,oldMdY,rulerLen,oldMdY);
      XDrawLine(mainDisplay,vRuleWindow,revDefaultGC,0,oldRbY,rulerLen,oldRbY);
   }
}

void BeginIntervalRulers (ltx, lty, rbx, rby)
   int	ltx, lty, rbx, rby;
{
   XDrawLine(mainDisplay,hRuleWindow,revDefaultGC,oldXOff,0,oldXOff,rulerLen);
   XDrawLine(mainDisplay,vRuleWindow,revDefaultGC,0,oldYOff,rulerLen,oldYOff);

   oldLtX = ltx; oldLtY = lty;
   oldRbX = rbx; oldRbY = rby;
   oldMdX = (oldLtX+oldRbX)>>1;
   oldMdY = (oldLtY+oldRbY)>>1;
   if (showMeasurement) FreezeMarkRulerText ();
   DoIntervalRulers ();
}

void DrawIntervalRulers (ltx, lty, rbx, rby)
   int	ltx, lty, rbx, rby;
{
   DoIntervalRulers ();
   oldLtX = ltx; oldLtY = lty;
   oldRbX = rbx; oldRbY = rby;
   oldMdX = (oldLtX+oldRbX)>>1; oldMdY = (oldLtY+oldRbY)>>1;
   DoIntervalRulers ();
   if (showMeasurement) UnFreezeMarkRulerText ();
}

void EndIntervalRulers (x, y)
   int	x, y;
{
   DoIntervalRulers ();
   oldXOff = x;
   oldYOff = y;
   XDrawLine(mainDisplay,hRuleWindow,revDefaultGC,oldXOff,0,oldXOff,rulerLen);
   XDrawLine(mainDisplay,vRuleWindow,revDefaultGC,0,oldYOff,rulerLen,oldYOff);
   MarkRulers (x, y);
   frozenXOff = oldXOff;
   frozenYOff = oldYOff;
}

void StartShowMeasureCursor (XOff, YOff, Str, ExtraSpace)
   int	XOff, YOff, ExtraSpace;
   char	*Str;
{
   if (!showMeasurement) return;
   MarkRulers (XOff, YOff);
   FreezeMarkRulerText ();
   if (Str != NULL && *Str != '\0')
   {
      if (ExtraSpace)
         XDrawString (mainDisplay, drawWindow, revDefaultGC, XOff+18,
               YOff+defaultFontAsc, Str, strlen (Str));
      else
         XDrawString (mainDisplay, drawWindow, revDefaultGC, XOff+4,
               YOff+defaultFontAsc, Str, strlen (Str));
   }
}

void ShowMeasureCursor (XOff, YOff, Str, ExtraSpace)
   int	XOff, YOff, ExtraSpace;
   char	*Str;
{
   if (!showMeasurement) return;
   if (Str != NULL && *Str != '\0')
   {
      if (ExtraSpace)
         XDrawString (mainDisplay, drawWindow, revDefaultGC, XOff+18,
               YOff+defaultFontAsc, Str, strlen (Str));
      else
         XDrawString (mainDisplay, drawWindow, revDefaultGC, XOff+4,
               YOff+defaultFontAsc, Str, strlen (Str));
   }
}

void EndShowMeasureCursor (XOff, YOff, Str, ExtraSpace)
   int	XOff, YOff, ExtraSpace;
   char	*Str;
{
   if (!showMeasurement) return;
   if (Str != NULL && *Str != '\0')
   {
      if (ExtraSpace)
         XDrawString (mainDisplay, drawWindow, revDefaultGC, XOff+18,
               YOff+defaultFontAsc, Str, strlen (Str));
      else
         XDrawString (mainDisplay, drawWindow, revDefaultGC, XOff+4,
               YOff+defaultFontAsc, Str, strlen (Str));
   }
   UnFreezeMarkRulerText ();
}

void ToggleShowMeasurement ()
{
   showMeasurement = !showMeasurement;
   RedrawHRuler ();
   if (showMeasurement)
      Msg ("Showing measurement enabled.");
   else
      Msg ("Showing measurement disabled.");
   UpdateSubMenu (MENU_LAYOUT);
}

