#!perl

use strict;

use Test::More tests => 1 + 20 * 97 * 4;
use Test::Number::Delta within => 0.01;

use Astro::Telescope;
my $jcmt = Astro::Telescope->new('JCMT');

require_ok('App::SourcePlot::Source');

my $mw = TestMainWindow->new();
my $n = 0;

while (1) {
    $n ++;
    my $line = <DATA>;
    last unless $line;
    chomp $line;
    my ($date, $time, $ra, $dec, $sys, $num_points) = split ' ', $line;
    $line = <DATA>;
    chomp $line;
    my @time_el_ref = split ' ', $line;
    $line = <DATA>;
    chomp $line;
    my @az_pa_ref = split ' ', $line;

    unless ((scalar @time_el_ref) == (2 * $num_points)
            and (scalar @az_pa_ref) == (2 * $num_points)) {
        fail('Wrong number of points read from data block.');
        next;
    }

    my $s = App::SourcePlot::Source->new('test', $ra, $dec, $sys);
    $s->calcPoints($date, $time, $num_points, $mw, $jcmt);
    my @time_el_calc = $s->time_ele_points();
    my @az_pa_calc = $s->az_pa_points();

    unless ((scalar @time_el_calc) == (2 * $num_points)
            and (scalar @az_pa_calc) == (2 * $num_points)) {
        fail('Wrong number of points returned by calculation.');
        next;
    }

    for (my $i = 0; $i < $num_points; $i ++) {
        delta_ok($time_el_calc[2 * $i],     $time_el_ref[2 * $i],     "Test $n point $i LST");
        delta_ok($time_el_calc[2 * $i + 1], $time_el_ref[2 * $i + 1], "Test $n point $i EL");
        delta_ok(  $az_pa_calc[2 * $i],       $az_pa_ref[2 * $i],     "Test $n point $i AZ");
        delta_ok(  $az_pa_calc[2 * $i + 1],   $az_pa_ref[2 * $i + 1], "Test $n point $i PA");
    }
}

# Dummy class to give to calcPoints as a "Main Window".

package TestMainWindow;

sub new {
    my $class = shift;
    return bless {}, $class;
}

sub update {
}

package main;

# The following test data was created from App::SourcePlot::Source before
# conversion to using Astro::Coords.  However it was necessary first to fix
# a bug in the original Source module's handling of negative declinations.
# This data should help check that the calculations have not inadvertently
# been changed.  The times, dates and coordinates are all randomly selected.

__DATA__
2076/12/14 10:35:48 23:06:35 +32:35:06 RB 97
1.00147389350005 27.7228279971194 1.06710293978669 24.6057215610769 1.13273198607333 21.5034209738897 1.19836103235998 18.419069587524 1.26399007864662 15.3558756095167 1.32961912493326 12.3171743148455 1.3952481712199 9.30648644421915 1.46087721750654 6.32757514914965 1.52650626379318 3.38450325829839 1.59213531007982 0.48169218454437 1.65776435636647 -2.37601660834471 1.72339340265311 -5.18329707098652 1.78902244893975 -7.93427093846716 1.85465149522639 -10.6224375935943 1.92028054151303 -13.2406028296018 1.98590958779967 -15.7808090109892 2.05153863408632 -18.2342705669703 2.11716768037296 -20.5913205808346 2.1827967266596 -22.8413764613185 2.24842577294624 -24.9729352041218 2.31405481923288 -26.9736113173529 2.37968386551952 -28.8302326013775 2.44531291180617 -30.5290098498113 2.51094195809281 -32.0557950859429 2.57657100437945 -33.39643792479 2.64220005066609 -34.5372400196738 2.70782909695273 -35.4654930898591 2.77345814323937 -36.1700680405146 2.83908718952601 -36.6420045041764 2.90471623581266 -36.8750368426896 2.9703452820993 -36.8659897295939 3.03597432838594 -36.6149874687344 3.10160337467258 -36.1254456874936 3.16723242095922 -35.4038466724229 3.23286146724586 -34.459331907495 3.29849051353251 -33.3031689822135 3.36411955981915 -31.9481599892172 3.42974860610579 -30.4080546762874 3.49537765239243 -28.6970177661348 3.56100669867907 -26.8291815308762 3.62663574496571 -24.8182968966936 3.69226479125235 -22.6774822300273 3.757893837539 -20.4190596910493 3.82352288382564 -18.0544643238539 3.88915193011228 -15.5942098165643 3.95478097639892 -13.0478958676382 4.02041002268556 -10.4242442682277 4.0860390689722 -7.73115338646833 4.15166811525885 -4.97576324567944 4.21729716154549 -2.16452558360342 4.28292620783213 0.696724921541719 4.34855525411877 3.60270066433428 4.41418430040541 6.54859370144101 4.47981334669205 9.53001145326778 4.5454423929787 12.5429154511082 4.61107143926534 15.5835625890341 4.67670048555198 18.6484479321869 4.74232953183862 21.7342477308037 4.80795857812526 24.8377608273679 4.8735876244119 27.9558460449933 4.93921667069854 31.0853523044692 5.00484571698519 34.2230369835097 5.07047476327183 37.3654661705155 5.13610380955847 40.508887601658 5.20173285584511 43.6490625883756 5.26736190213175 46.7810361103755 5.33299094841839 49.8988127114515 5.39861999470504 52.9948868770514 5.46424904099168 56.0595450151691 5.52987808727832 59.0798032780332 5.59550713356496 62.0377574446516 5.6611361798516 64.9079800223141 5.72676522613824 67.6533991445982 5.79239427242489 70.2189142169906 5.85802331871153 72.522287276871 5.92365236499817 74.4441333627484 5.98928141128481 75.8263830982714 6.05491045757145 76.5008902605056 6.12053950385809 76.3616345139303 6.18616855014473 75.4320471249114 6.25179759643138 73.8487264774826 6.31742664271802 71.7825033618503 6.38305568900466 69.3795597488925 6.4486847352913 66.7458506239401 6.51431378157794 63.9533139596818 6.57994282786458 61.0500968911616 6.64557187415123 58.0687662555232 6.71120092043787 55.0318458508967 6.77682996672451 51.9553493933537 6.84245901301115 48.8510104755598 6.90808805929779 45.7277022839179 6.97371710558443 42.5923574648298 7.03934615187107 39.4505772780169 7.10497519815772 36.3070448732729 7.17060424444436 33.165813087642 7.236233290731 30.0305106090913 7.30186233701764 26.90449433981
297.997354016103 83.4926412443841 298.450771847945 81.6289671970724 299.002503690085 79.7814284452823 299.64860635561 77.9383980441721 300.386760135972 76.0892240918417 301.216039154327 74.2239394496556 302.136741547732 72.3330300240025 303.150263545754 70.4072475340099 304.259005199472 68.4374572966107 305.466297808518 66.4145150955046 306.776344341663 64.3291700952261 308.194164562899 62.1719933209629 309.725536292472 59.9333336670128 311.376923342513 57.6033058694469 313.15537926016 55.1718174562483 315.068414223038 52.6286443500237 317.123810504067 49.9635673838876 319.329370278264 47.1665841341741 321.692578878494 44.2282115203888 324.220167968407 41.1398935547276 326.917567917903 37.8945240332708 329.788248637217 34.4870841669449 332.832964842481 30.9153785717518 336.048945918722 27.1808289532352 339.429100919825 23.2892545320714 342.961341157816 19.2515364122088 346.628147519544 15.0840384603067 350.406515087924 10.8086518676503 354.268382099873 6.45235614293519 358.181588319878 2.04625126188923 2.11131658817757 -2.37589296244953 6.02187213416952 -6.77942504449319 9.87857756768835 -11.1307155966218 13.64953395597 -15.3989272691583 17.3070301790809 -19.5574332110082 20.8284621645815 -23.5847458175283 24.196723185735 -27.4649171282818 27.4001161133466 -31.1874619614171 30.4318978098182 -34.7469141948624 33.2895887817799 -38.142149565098 35.9741738930583 -41.3756010819756 38.4892943266245 -44.4524675989274 40.8404989349574 -47.3799840972897 43.0345930940306 -50.1667923062542 45.0790995290821 -52.8224267434244 46.98182940861 -55.3569152299907 48.7505524764687 -57.7804835879385 50.3927504892435 -60.1033499552853 51.9154370608461 -62.3355932889785 53.3250277863885 -64.4870817839097 54.6272462029429 -66.5674491310583 55.8270530741301 -68.5861091386039 56.9285882516389 -70.55230189887 57.9351157388106 -72.4751672531659 58.8489634287986 -74.3638437734384 59.6714492236005 -76.2275939247023 60.4027847671556 -78.075958648197 61.041946708506 -79.9189475358621 61.586503027479 -81.7672743650188 62.0323781460281 -83.632652471045 62.3735347411489 -85.528170898985 62.6015414692088 -87.4687814520067 62.7049827938621 -89.4719401046863 62.668647568216 -91.5584660206699 62.4724035011242 -93.753711090393 62.0896197898377 -96.0891778536267 61.484931957925 -98.6047919779866 60.6110398674679 -101.352138570566 59.4040783442682 -104.399123147711 57.7768927288536 -107.83672521904 55.6093233825136 -111.78874072246 52.73454965383 -116.425463062896 48.9213955212324 -121.981400786863 43.8565536218336 -128.773075418701 37.1439704624986 -137.199656886903 28.3714474633039 -147.676382103844 17.338308732086 -160.406906003878 4.47054542279445 -174.968175382572 351.055642548685 169.924380145002 338.702623766697 155.87687539888 328.423803005138 143.898703792149 320.385013787735 134.152749064222 314.275164831887 126.324922984909 309.67230573711 119.990212659556 306.205719737405 114.774754218053 303.591287627868 110.391176799944 301.621651935117 106.628745698447 300.147498888927 103.334622003704 299.061094097477 100.397374419428 298.284040295016 97.7347113817039 297.758680312622 95.2848563792378 297.442167332995 93.0005882747837 297.302379306384 90.8451232372107 297.31508525457 88.7892457428723 297.461960112653 86.8092850365608 297.729178511575 84.8856672185214 298.106407768223 83.0018630320542
2012/02/01 15:01:16 03:07:06 -29:19:07 RB 97
2.98441087966007 -38.26738522851 3.05003992420997 -41.5506024706398 3.11566896875987 -44.8363509289064 3.18129801330977 -48.1214245556853 3.24692705785967 -51.4019932482929 3.31255610240957 -54.6732688427641 3.37818514695947 -57.928981425743 3.44381419150937 -61.160524783692 3.50944323605927 -64.3555026681515 3.57507228060917 -67.4951451923274 3.64070132515906 -70.5495085495739 3.70633036970896 -73.4681981854477 3.77195941425886 -76.1621449824956 3.83758845880876 -78.4699851477397 3.90321750335866 -80.1151812962049 3.96884654790856 -80.7362103969982 4.03447559245846 -80.1372122201252 4.10010463700836 -78.5077656122081 4.16573368155826 -76.2093379520018 4.23136272610816 -73.5208159711321 4.29699177065806 -70.6053385904288 4.36262081520796 -67.5529580654827 4.42824985975786 -64.4145854408937 4.49387890430776 -61.2204416276035 4.55950794885765 -57.9894514196163 4.62513699340755 -54.7341014805009 4.69076603795745 -51.4630528896114 4.75639508250735 -48.182609942108 4.82202412705725 -44.8975828682105 4.88765317160715 -41.6118161268128 4.95328221615705 -38.3285251013243 5.01891126070695 -35.0505193317559 5.08454030525685 -31.7803566751382 5.15016934980675 -28.5204545368988 5.21579839435665 -25.2731741099126 5.28142743890655 -22.0408876769502 5.34705648345645 -18.8260355541885 5.41268552800635 -15.6311771514606 5.47831457255624 -12.4590393326763 5.54394361710614 -9.31256445486203 5.60957266165604 -6.19495995622553 5.67520170620594 -3.10975103464557 5.74083075075584 -0.0608377292689325 5.80645979530574 2.94744246992735 5.87208883985564 5.91024555947086 5.93771788440554 8.82214484867938 6.00334692895544 11.6770474176739 6.06897597350534 14.4681035674215 6.13460501805524 17.1876105621754 6.20023406260514 19.8269133573194 6.26586310715504 22.3763069994828 6.33149215170493 24.8249481275217 6.39712119625483 27.1607865907405 6.46275024080473 29.3705325975332 6.52837928535463 31.4396797019967 6.59400832990453 33.3526085519733 6.65963737445443 35.0927992181294 6.72526641900433 36.6431788940524 6.79089546355423 37.9866240391589 6.85652450810413 39.1066190867233 6.92215355265403 39.9880468276973 6.98778259720393 40.618051302758 7.05341164175383 40.986880224231 7.11904068630373 41.0885924111923 7.18466973085363 40.9215182304946 7.25029877540352 40.4883930522985 7.31592781995342 39.7961395430003 7.38155686450332 38.8553380094824 7.44718590905322 37.6794751968885 7.51281495360312 36.2840868317696 7.57844399815302 34.6859050044105 7.64407304270292 32.9020957740231 7.70970208725282 30.949637480469 7.77533113180272 28.8448572210147 7.84096017635262 26.6031183015399 7.90658922090252 24.2386369428931 7.97221826545242 21.76440069574 8.03784731000232 19.192161184572 8.10347635455222 16.5324772907073 8.16910539910211 13.7947896721864 8.23473444365201 10.9875123418941 8.30036348820191 8.11813124040577 8.36599253275181 5.19330311816532 8.43162157730171 2.21895058924588 8.49725062185161 -0.799648947713738 8.56287966640151 -3.85778155654013 8.62850871095141 -6.95122236320638 8.69413775550131 -10.0761681369234 8.75976680005121 -13.2291761201015 8.82539584460111 -16.4071081294434 8.89102488915101 -19.6070787584809 8.95665393370091 -22.8264063190259 9.02228297825081 -26.0625649118254 9.0879120228007 -29.3131356524597 9.1535410673506 -32.5757545110968 9.2191701119005 -35.8480533223945 9.2847991564504 -39.1275890690216
248.048509826645 86.8220207750591 248.221847027615 88.5392493472996 248.26602072193 90.3374838444024 248.158551891573 92.2439259989654 247.868530792601 94.2938910010081 247.352973219511 96.5344884362564 246.551116512576 99.030360094812 245.375242765535 101.872887602073 243.695490528333 105.195408956169 241.314049777972 109.199049548113 237.920605608913 114.197302534717 233.016616446271 120.691773057326 225.801499906682 129.486013455429 215.090371627796 141.767804409115 199.691440926211 158.731778389579 180.190188787988 179.795260139895 160.641735788285 -159.094083193492 145.151104743356 -142.038486871139 134.361904712908 -129.678754723253 127.093619281347 -120.831529385635 122.155024807978 -114.302692088375 118.738732329383 -109.281885463704 116.341715117849 -105.263021150704 114.65096185748 -101.929911919754 113.467048807536 -99.0798214316949 112.65912728124 -96.5784276863364 112.138857134439 -94.3337305588981 111.845071160882 -92.2806880001325 111.734514133045 -90.3719242903274 111.776089514798 -88.5719429950936 111.947186041894 -86.8534164102387 112.231272581739 -85.1947394693162 112.616287880089 -83.5783742585093 113.093541674238 -81.9897011386717 113.656952977886 -80.4162017979247 114.302515905573 -78.8468641429571 115.027922464687 -77.2717380568212 115.832295903922 -75.6815953280689 116.716003441889 -74.0676624907177 117.680526946553 -72.4214053827557 118.72837641111 -70.734351000613 119.863035071344 -68.9979369657687 121.088927444287 -67.2033824334821 122.411402851549 -65.3415771007895 123.836727347487 -63.4029874946783 125.372076502679 -61.3775822358888 127.025520216697 -59.2547807195406 128.805988626598 -57.0234328290817 130.723205213823 -54.6718410478604 132.787569430942 -52.1878406954154 135.009966774035 -49.5589588464825 137.401479738476 -46.7726773073438 139.972969571148 -43.8168287989103 142.734498015764 -40.6801564039597 145.694563187764 -37.3530615440131 148.859138137869 -33.828551443502 152.230528768723 -30.1033690230558 155.806112497024 -26.1792435058128 159.577079132177 -22.0641399970525 163.527361558151 -17.7733202249108 167.632995817978 -13.329974650971 171.862158981519 -8.76517740452449 176.176070619519 -4.11697807195831 180.530800324081 0.571412229487148 184.879825171459 5.25418450962955 189.176989007326 9.88589728862971 193.3794039193 14.424384180928 197.449850648417 18.8331605424898 201.358371840557 23.0830231971407 205.082952599879 27.1527378263918 208.609372435492 31.0288982477911 211.930437232873 34.7051663632094 215.044843152658 38.1811448510474 217.955902170877 41.4611125156488 220.670301090462 44.552794358876 223.196999785173 47.4662724158493 225.546317609992 50.2130866281266 227.72921633651 52.8055345263889 229.756763556296 55.2561541742618 231.639748772264 57.5773632239269 233.388421154845 59.781223848393 235.012319561616 61.8793051415044 236.52016931379 63.8826187385456 237.919824761205 65.801608300173 239.218240946267 67.646178247899 240.421461269796 69.4257513567307 241.534610816722 71.1493484101053 242.561886901695 72.8256861772117 243.506539473462 74.4632926423427 244.370834289627 76.0706409088801 245.155991204085 77.6563057684464 245.862088372142 79.2291498802709 246.487920409709 80.7985502604414 247.030794069575 82.3746809261739 247.486238011785 83.9688749729515 247.847592389066 85.5941005059932 248.105426949573 87.2656020426281
2061/10/04 22:11:11 07:52:16 -16:39:11 RB 97
2.8163049766926 34.7214278828823 2.88193402527143 31.9131965135547 2.94756307385026 29.011102634722 3.0131921224291 26.0280186227262 3.07882117100793 22.9748617796623 3.14445021958676 19.8608915027752 3.2100792681656 16.6939713796367 3.27570831674443 13.4807946360958 3.34133736532326 10.2270750088735 3.4069664139021 6.93770675299444 3.47259546248093 3.61689799469806 3.53822451105976 0.268281542461847 3.6038535596386 -3.10499310921246 3.66948260821743 -6.50018332747207 3.73511165679626 -9.91488966922334 3.8007407053751 -13.3470026628919 3.86636975395393 -16.7946581011684 3.93199880253276 -20.2561992201834 3.9976278511116 -23.7301443925842 4.06325689969043 -27.2151591487385 4.12888594826926 -30.7100314597676 4.1945149968481 -34.2136492595812 4.26014404542693 -37.724979129443 4.32577309400576 -41.2430448739824 4.3914021425846 -44.7669042949796 4.45703119116343 -48.2956216456308 4.52266023974226 -51.8282316699183 4.58828928832109 -55.3636880462142 4.65391833689993 -58.900782767963 4.71954738547876 -62.438009438943 4.78517643405759 -65.9733120512679 4.85080548263643 -69.5035811985777 4.91643453121526 -73.0235336133915 4.98206357979409 -76.5228669754254 5.04769262837292 -79.9775924735372 5.11332167695176 -83.3154963723532 5.17895072553059 -86.2106600121902 5.24457977410942 -86.9192146774633 5.31020882268826 -84.5153759002038 5.37583787126709 -81.2753319058721 5.44146691984592 -77.8514389697994 5.50709596842476 -74.3650059468744 5.57272501700359 -70.8512780055805 5.63835406558242 -67.3241499770482 5.70398311416125 -63.7903216155855 5.76961216274009 -60.2535456970121 5.83524121131892 -56.7161999948312 5.90087025989775 -53.1799620446477 5.96649930847659 -49.6461333777612 6.03212835705542 -46.1158103799371 6.09775740563425 -42.5899820297435 6.16338645421308 -39.0695904673383 6.22901550279192 -35.5555718263257 6.29464455137075 -32.0488863907432 6.36027359994958 -28.5505431119342 6.42590264852842 -25.0616214883137 6.49153169710725 -21.5832927555234 6.55716074568608 -18.1168417858761 6.62278979426491 -14.6636908278594 6.68841884284375 -11.2254261158356 6.75404789142258 -7.80382838983976 6.81967694000141 -4.40090845733828 6.88530598858025 -1.01894909033709 6.95093503715908 2.33944522171777 7.01656408573791 5.67128885009936 7.08219313431674 8.9731437637455 7.14782218289558 12.2410338301225 7.21345123147441 15.4703416122279 7.27908028005324 18.6556841048669 7.34470932863208 21.7907634211784 7.41033837721091 24.8681882067849 7.47596742578974 27.8792617809798 7.54159647436857 30.8137341255292 7.60722552294741 33.6595175905972 7.67285457152624 36.402371695855 7.73848362010507 39.0255723017749 7.80411266868391 41.5095967832714 7.86974171726274 43.8318816963127 7.93537076584157 45.9667434203291 8.00099981442041 47.8855917157394 8.06662886299924 49.5575985105466 8.13225791157807 50.95098364609 8.1978869601569 52.0350082156017 8.26351600873574 52.7825930491529 8.32914505731457 53.1732179559408 8.3947741058934 53.195502401548 8.46040315447224 52.8487828251099 8.52603220305107 52.1432078615324 8.5916612516299 51.0983212145645 8.65729030020873 49.7405665864772 8.72291934878757 48.1003910999563 8.7885483973664 46.2095691534843 8.85417744594523 44.0991249757513 8.91980649452407 41.7979641635018 8.9854355431029 39.3321386519058 9.05106459168173 36.7245874149042 9.11669364026056 33.9951882441829
231.189223456442 50.0246427564985 233.879626241603 52.5994010191415 236.345578236488 54.9464596563464 238.612270064426 57.0906031205308 240.702523148768 59.0542050428447 242.636768773646 60.8572001211882 244.433165477855 62.5171930004264 246.10778781044 64.0496368513699 247.674844737292 65.4680395270439 249.146903094938 66.7841721639244 250.535102626252 68.0082660770116 251.849356162415 69.1491906147579 253.098532839788 70.2146086744778 254.290624854692 71.2111087937629 255.432899847231 72.1443137640858 256.532042034535 73.0189659541843 257.594286015202 73.8389892030569 258.625547999092 74.607526331813 259.63156032249 75.3269499994918 260.618016772856 75.9988426327903 261.590738872626 76.6239381647587 262.555877498818 77.2020137235328 263.520171135305 77.7317121496743 264.49129361172 78.2102643554399 265.47834397861 78.6330604726708 266.492566143554 78.9929835262354 267.548450016372 79.2793550456904 268.66548871023 79.4762190584562 269.871113874329 79.5594432348207 271.205861395187 79.4915856726572 272.733040881401 79.2122545305658 274.558256155634 78.61861382492 276.87276592731 77.5220466866818 280.06266853419 75.5389926016793 285.035205341026 71.7646644112326 294.459820221103 63.532007974836 318.975955013405 40.2039249002277 21.1421931857808 -20.7758536136586 59.1604396082339 -57.6069294648728 72.157805036058 -69.4141052169848 78.3281938867869 -74.3889548853698 82.0427432647777 -76.9002456467388 84.6293083261223 -78.2734053088635 86.611999490978 -79.030040778484 88.2373674952967 -79.4141032306175 89.6365107126732 -79.5539740946256 90.8857690673783 -79.5231358284174 92.0328784783915 -79.3663010278053 93.1094963770665 -79.1119099676425 94.1377150755651 -78.7786128409625 95.1336813902189 -78.3788545108097 96.109722663892 -77.9209612348482 97.0756566940168 -77.4104060722667 98.0396344070831 -76.8506008816991 99.0087046125201 -76.2434031509598 99.9892084124319 -75.58944395401 100.987066951418 -74.8883392169964 102.008001679217 -74.1388217068539 103.057712147645 -73.3388167106393 104.142027965816 -72.4854756463274 105.267046444108 -71.575176393071 106.439264313538 -70.6034956130493 107.665709932729 -69.565156010136 108.954081113289 -68.4539499163656 110.312892787079 -67.2626396002037 111.75163795406 -65.9828341745149 113.280964441132 -64.6048429878079 114.912868674226 -63.1175060752385 116.660905510979 -61.5080039476481 118.540409609262 -59.761652241851 120.568717953381 -57.861692369921 122.765373787477 -55.7890985012392 125.152277627757 -53.5224356575385 127.753729145417 -51.0378254821822 130.596272385796 -48.309107496293 133.708214915689 -45.3083254707103 137.118641519538 -42.0067184699374 140.855695787975 -38.3764433673551 144.943884935367 -34.3932736113375 149.400224703161 -30.0404574809832 154.229255776672 -25.3137044831956 159.417399855577 -20.2268318175168 164.927769881525 -14.8169564621047 170.697194756096 -9.14747259683425 176.637404691298 -3.30686826472568 182.641541799551 2.59778329603233 188.595359130196 8.45202082022675 194.390445892947 14.147218389742 199.93584486799 19.5922009793865 205.165203432518 24.7203935697578 210.038565379253 29.4916109666504 214.539798009765 33.8894827710521 218.671565418449 37.916424372761 222.449666093243 41.5879720452766 225.897933959153 44.9276812122798 229.044239510436 47.9631244661664 231.917662303247 50.7230605328153
2059/02/18 02:11:16 11:35:01 +61:13:05 RB 97
6.20156531997265 -9.56535970947104 6.26719436130797 -9.50583259448884 6.33282340264329 -9.33114016465334 6.3984524439786 -9.04220564832008 6.46408148531392 -8.64054126313325 6.52971052664924 -8.12821926356142 6.59533956798455 -7.50783362305311 6.66096860931987 -6.78245425702158 6.72659765065519 -5.95557594722827 6.7922266919905 -5.03106422134962 6.85785573332582 -4.0131003848665 6.92348477466114 -2.90612772127348 6.98911381599645 -1.71480060502466 7.05474285733177 -0.443937946697645 7.12037189866709 0.901517953270094 7.1860009400024 2.31653639757343 7.25162998133772 3.79602668505971 7.31725902267304 5.33485970724421 7.38288806400836 6.92788274736745 7.44851710534367 8.56992762143124 7.51414614667899 10.255812389864 7.57977518801431 11.9803369136479 7.64540422934962 13.738272538948 7.71103327068494 15.5243461796356 7.77666231202026 17.3332190383806 7.84229135335557 19.1594601756089 7.90792039469089 20.997515114012 7.97354943602621 22.8416696686207 8.03917847736152 24.68600923555 8.10480751869684 26.524373877148 8.17043656003216 28.3503097332457 8.23606560136747 30.1570175993724 8.30169464270279 31.937299981165 8.36732368403811 33.6835086022935 8.43295272537343 35.3874952532668 8.49858176670874 37.0405700521728 8.56421080804406 38.6334726486522 8.62983984937938 40.1563635843484 8.69546889071469 41.5988447732714 8.76109793205001 42.95001958323 8.82672697338533 44.1986037931685 8.89235601472064 45.333098078854 8.95798505605596 46.3420298173607 9.02361409739128 47.2142661439407 9.08924313872659 47.9393910137736 9.15487218006191 48.5081270762889 9.22050122139723 48.9127702991346 9.28613026273255 49.1475946031727 9.35175930406786 49.2091790278866 9.41738834540318 49.0966141485078 9.4830173867385 48.811558343855 9.54864642807381 48.3581356302217 9.61427546940913 47.7426900190419 9.67990451074445 46.9734307867045 9.74553355207976 46.0600141934569 9.81116259341508 45.0131085650677 9.8767916347504 43.8439827948326 9.94242067608571 42.5641466242506 10.008049717421 41.1850581894098 10.0736787587563 39.7179030567365 10.1393078000917 38.1734407556528 10.204936841427 36.5619098980297 10.2705658827623 34.8929808554164 10.3361949240976 33.1757448395499 10.4018239654329 31.4187293006863 10.4674530067682 29.6299311859123 10.5330820481036 27.8168613534143 10.5987110894389 25.9865950662503 10.6643401307742 24.1458248707986 10.7299691721095 22.3009132665075 10.7955982134448 20.4579434086874 10.8612272547802 18.6227666911177 10.9268562961155 16.8010464732769 10.9924853374508 14.9982974895624 11.0581143787861 13.2199206412169 11.1237434201214 11.4712329558436 11.1893724614567 9.75749252859045 11.2550015027921 8.08391825278479 11.3206305441274 6.45570412196062 11.3862595854627 4.87802785338681 11.451888626798 3.35605355746073 11.5175176681333 1.89492816887865 11.5831467094686 0.499771374937704 11.648775750804 -0.82434116646066 11.7144047921393 -2.07240142295097 11.7800338334746 -3.23949934550747 11.8456628748099 -4.32085972354475 11.9112919161452 -5.31188500775187 11.9769209574805 -6.20820326141129 12.0425499988159 -7.00572006328305 12.1081790401512 -7.70067284396837 12.1738080814865 -8.28968582455895 12.2394371228218 -8.76982347749173 12.3050661641571 -9.13864028162377 12.3706952054924 -9.39422452872281 12.4363242468278 -9.53523407858928 12.5019532881631 -9.56092226121095
0.0289595370014974 -0.0555170196803221 1.89884151206035 -3.64196854365873 3.76065768836726 -7.22333002667711 5.60657783492694 -10.7947193507522 7.42898925074228 -14.351507157207 9.2205915889841 -17.8894295970177 10.9744772122953 -21.4046871501892 12.6841940478356 -24.8940266694128 14.3437891330524 -28.3548050324383 15.9478322750121 -31.7850340275627 17.4914203687442 -35.1834072283649 18.9701638442953 -38.5493105476604 20.3801573859311 -41.882818843769 21.7179374752079 -45.1846813685691 22.9804294690606 -48.456299011584 24.1648868737881 -51.6996962445683 25.268825267683 -54.9174904556221 26.2899530145968 -58.1128610322446 27.2261005505512 -61.2895201559222 28.0751496610822 -64.4516868446571 28.8349638351003 -67.6040653506519 29.5033205099717 -70.7518286025036 30.0778458338014 -73.9006069776261 30.5559524816503 -77.0564822937894 30.9347810880669 -80.2259865021187 31.2111460151549 -83.4161041231007 31.3814864828267 -86.6342769604008 31.4418245698641 -89.8884090166433 31.3877322797934 -93.1868687778223 31.2143107866034 -96.5384850834637 30.9161861628747 -99.9525316157215 30.4875273669985 -103.438693592648 29.9220940189754 -107.007008539026 29.2133234622944 -110.667771089874 28.3544686316167 -114.431389813405 27.338800006368 -118.308182334648 26.1598858875848 -122.308094137024 24.8119645548373 -126.440327150068 23.2904183820681 -130.712867755936 21.5923522993367 -135.131911581293 19.7172656711449 -139.701195795011 17.6677868759079 -144.421269451289 15.4504142353559 -149.288758082264 13.076178643444 -154.295707068084 10.5611188658314 -159.429112711835 7.92644973415837 -164.670760711027 5.19831690243185 -169.997478288566 2.40707635116637 -175.381861739301 359.58611022264 179.206532240667 356.770278386722 173.79963088449 353.994183057812 168.429102704418 351.290466428996 163.124647864284 348.68835363475 157.912533923435 346.212598147551 152.814534097459 343.882903092479 147.847341564487 341.713806177039 143.02244756611 339.714950062497 138.34640519053 337.891624865052 133.821365599927 336.245464103462 129.4457681075 334.775191275814 125.215081394187 333.477340664899 121.122519593805 332.346904012272 117.159685041802 331.377878817089 113.317113628787 330.563711884262 109.58471660278 329.897643287373 105.952124243919 329.372962214566 102.408943181555 328.983188685655 98.9449416913127 328.722195223456 95.550177455914 328.584281297533 92.2150810625936 328.564211476764 88.9305066830566 328.657226196289 85.6877594040929 328.859032128068 82.4786068083527 329.165777469705 79.2952807675509 329.574016076953 76.1304740415279 330.080663258665 72.9773351652374 330.682945204389 69.8294642171254 331.37834339229 66.6809113525639 332.16453489702 63.5261794100628 333.039329256156 60.3602314142407 334.000602437231 57.178503371756 335.04622845704 53.9769223549009 336.174009324908 50.7519294700677 337.381604196395 47.5005069008208 338.666458915309 44.2202077933365 340.025737466624 40.9091873216421 341.45625723027 37.566232849237 342.954430276468 34.1907907221709 344.516213229431 30.7829869273149 346.137068393846 27.3436386773062 347.81193883236 23.8742539920154 349.535239853177 20.377016581846 351.300868881238 16.854753832399 353.102234936187 13.3108864498698 354.932307950783 9.74935932555995 356.783686998069 6.17455435214839 358.648685253212 2.59118717449119 0.519428321443709 -0.995808941419134
1996/12/01 00:52:06 04:59:09 -38:42:42 RB 97
4.49650065093161 -71.059503223797 4.56212969402795 -70.4289541665725 4.62775873712428 -69.301369862365 4.69338778022062 -67.7532885696543 4.75901682331695 -65.8669793224612 4.82464586641328 -63.716977852839 4.89027490950962 -61.3647133962279 4.95590395260595 -58.8583251525737 5.02153299570229 -56.2346755150008 5.08716203879862 -53.5218172495813 5.15279108189495 -50.7411913850077 5.21842012499129 -47.9093666554047 5.28404916808762 -45.0393454449197 5.34967821118396 -42.1415222745397 5.41530725428029 -39.2243825476204 5.48093629737662 -36.2950132588393 5.54656534047296 -33.359479482977 5.61219438356929 -30.4231055257127 5.67782342666563 -27.4906883788028 5.74345246976196 -24.5666630431726 5.80908151285829 -21.6552335959289 5.87471055595463 -18.7604799029889 5.94033959905096 -15.8864470995994 6.0059686421473 -13.0372230027228 6.07159768524363 -10.2170072188812 6.13722672833996 -7.43017468384106 6.2028557714363 -4.68133558518592 6.26848481453263 -1.97539297938358 6.33411385762897 0.682401151871891 6.3997429007253 3.28639121447189 6.46537194382163 5.83046508174085 6.53100098691797 8.30800400787264 6.5966300300143 10.7118372247341 6.66225907311064 13.0342039930345 6.72788811620697 15.2667269410606 6.7935171593033 17.4004016962772 6.85914620239964 19.4256090133017 6.92477524549597 21.3321566676569 6.99040428859231 23.1093590653664 7.05603333168864 24.746162459056 7.12166237478497 26.2313224264616 7.18729141788131 27.5536374116994 7.25292046097764 28.7022373327009 7.31854950407398 29.6669195192495 7.38417854717031 30.4385160972615 7.44980759026664 31.0092685783951 7.51543663336298 31.3731786582279 7.58106567645931 31.5263011242541 7.64669471955565 31.466946984993 7.71232376265198 31.1957729987227 7.77795280574831 30.7157466769373 7.84358184884465 30.0319910635091 7.90921089194098 29.1515278912432 7.97483993503732 28.0829481413521 8.04046897813365 26.8360438488007 8.10609802122998 25.4214340865471 8.17172706432632 23.8502126727626 8.23735610742265 22.1336372460396 8.30298515051898 20.282870924368 8.36861419361531 18.3087802982751 8.43424323671165 16.2217878270276 8.49987227980798 14.0317730194431 8.56550132290431 11.7480148817016 8.63113036600065 9.37916759263751 8.69675940909698 6.93326177027073 8.76238845219331 4.41772462638745 8.82801749528965 1.83941347049831 8.89364653838598 -0.795341777630209 8.95927558148231 -3.48069029046596 9.02490462457865 -6.21120848905815 9.09053366767498 -8.9818460498368 9.15616271077131 -11.7878704429921 9.22179175386765 -14.6248100640216 9.28742079696398 -17.488395456537 9.35304984006031 -20.3744975702578 9.41867888315664 -23.2790613987822 9.48430792625298 -26.1980326310742 9.54993696934931 -29.1272740439648 9.61556601244564 -32.0624671465084 9.68119505554198 -34.9989929022701 9.74682409863831 -37.9317829775656 9.81245314173464 -40.8551295716885 9.87808218483098 -43.7624370285613 9.94371122792731 -46.6458914993424 10.0093402710236 -49.4960151742608 10.07496931412 -52.3010583211185 10.1405983572163 -55.0461655290557 10.2062274003126 -57.7122346479256 10.271856443409 -60.2743776420557 10.3374854865053 -62.6999196770903 10.4031145296016 -64.9460031666213 10.468743572698 -66.9572316055165 10.5343726157943 -68.6645835705994 10.6000016588906 -69.9880679619537 10.665630701987 -70.8464037878453 10.7312597450833 -71.174901468227 10.7968887881796 -70.9457892017334
174.127312407337 -172.920701278582 165.42169861192 -162.351463624061 157.536136684291 -152.597275311036 150.711264458737 -143.895707464967 144.989079648694 -136.285646933526 140.287292384583 -129.681631609779 136.470880330556 -123.945387817342 133.397101594441 -118.930824827153 130.936724860025 -114.505245788956 128.980940023139 -110.556238212199 127.441408635616 -106.991703154237 126.247695741581 -103.737264556951 125.344100925825 -100.733076414488 124.686708482693 -97.9308475549862 124.24091373454 -95.2913409586516 123.979446697825 -92.7823687327753 123.880830330264 -90.3772199432001 123.928191376958 -88.0534393490545 124.108347294409 -85.7918806375532 124.411105581141 -83.5759707289296 124.828725199193 -81.3911352461421 125.355501274396 -79.2243469759624 125.987443410472 -77.0637685998044 126.722024920543 -74.8984683430039 127.55798544063 -72.7181928757093 128.495173117169 -70.5131861989346 129.534415169402 -68.2740467104592 130.677407364097 -65.9916174419785 131.926613991904 -63.6569068002927 133.285170447296 -61.2610391817648 134.756780610425 -58.7952366564479 136.345601031142 -56.2508345753254 138.056103566569 -53.6193354153286 139.892907816416 -50.8925063368376 141.860574702637 -48.0625265834244 143.963353219102 -45.1221906839927 146.204874208545 -42.0651719828786 148.587788569071 -38.8863477891961 151.113353116872 -35.5821818567645 153.780975839212 -32.1511515915305 156.587743435339 -28.5941963727579 159.527967056715 -24.9151504819459 162.59279510411 -21.1211112839171 165.769951660598 -17.2226836597696 169.043661576636 -13.2340393151072 172.394814419628 -9.17273843853777 175.801397228075 -5.0592834948757 179.2391915157 -0.916409460299674 182.682688921429 3.23184412342779 186.106141612918 7.36119559687728 189.484638255356 11.4482091509674 192.79509142222 15.4712943967351 196.017039237007 19.4115205354159 199.13319776537 23.2531818685231 202.12974179211 26.9840925198363 204.996329963298 30.5956266116145 207.725918353595 34.082548250092 210.314421165651 37.4426903649766 212.760279367904 40.6765436018853 215.06399104685 43.7868095057346 217.227645270465 46.77796034097 219.25448796465 49.6558347095298 221.148536165169 52.4272861260965 222.914247286102 55.0998921628557 224.556243068164 57.6817250183829 226.079083383664 60.1811801580681 227.487082558757 62.6068575134327 228.784159750341 64.967489060408 229.973714665408 67.2719069341926 231.058520119605 69.5290472069866 232.040623289815 71.7479858014792 232.921247800163 73.9380045910076 233.700688830332 76.1086874880889 234.378193120777 78.2700482584822 234.951814951884 80.4326939893857 235.418237760146 82.6080307138369 235.772548857327 84.8085208312722 236.007951519878 87.0480059113977 236.115394231193 89.3421135390563 236.08309073739 91.7087734319841 235.895896432442 94.1688765529479 235.534496120588 96.7471216653753 234.974345524608 99.4731066542418 234.184295343234 102.382735641393 233.124816613344 105.520023075106 231.74575125452 108.939370925238 229.983559834195 112.70834700601 227.758192118409 116.910838956658 224.970095076851 121.650069329591 221.498743395028 127.050086865951 217.205805510232 133.252621082959 211.948918247737 140.403327241924 205.615120710555 148.618371190257 198.181491994298 157.923809652465 189.7937988221 168.176960020442 180.816960011273 179.015952875214 171.788266631882 -170.09347147767
2028/06/04 00:41:41 01:48:39 +49:06:52 RJ 97
1.35878528621576 40.5117147832563 1.42441432943603 38.1556290919725 1.4900433726563 35.7635728686211 1.55567241587657 33.3446483240201 1.62130145909684 30.9069613401695 1.68693050231711 28.4578188623142 1.75255954553738 26.0038971686476 1.81818858875765 23.5513848540304 1.88381763197792 21.1061044456721 1.94944667519819 18.6736162517508 2.01507571841846 16.2593075561898 2.08070476163872 13.8684697345091 2.14633380485899 11.5063653344228 2.21196284807926 9.17828666323899 2.27759189129953 6.88960695620452 2.3432209345198 4.64582476105839 2.40884997774007 2.45260175662366 2.47447902096034 0.315793820483039 2.54010806418061 -1.75852523062351 2.60573710740088 -3.76404818150389 2.67136615062115 -5.69422854698585 2.73699519384142 -7.54228590209025 2.80262423706169 -9.30122425655728 2.86825328028196 -10.9638654208154 2.93388232350223 -12.5228992335248 2.9995113667225 -13.9709521986251 3.06514040994277 -15.300675451034 3.13076945316304 -16.5048519904474 3.19639849638331 -17.576521783977 3.26202753960358 -18.5091216898616 3.32765658282385 -19.2966353204006 3.39328562604411 -19.9337461479758 3.45891466926438 -20.4159856368145 3.52454371248465 -20.7398672578656 3.59017275570492 -20.9029971861544 3.65580179892519 -20.9041534568197 3.72143084214546 -20.7433273691794 3.78705988536573 -20.4217237839647 3.852688928586 -19.9417202893805 3.91831797180627 -19.3067885456551 3.98394701502654 -18.5213839832418 4.04957605824681 -17.5908120568216 4.11520510146708 -16.5210802493067 4.18083414468735 -15.3187449759519 4.24646318790762 -13.9907616240164 4.31209223112789 -12.54434444857 4.37772127434816 -10.9868412328863 4.44335031756843 -9.32562578642606 4.5089793607887 -7.56800970120533 4.57460840400897 -5.72117344370136 4.64023744722924 -3.79211587458203 4.70586649044951 -1.78762065500592 4.77149553366977 0.285762327921782 4.83712457689004 2.42172246395225 4.90275362011031 4.61418269589227 4.96838266333058 6.85728418704137 5.03401170655085 9.14536269129619 5.09964074977112 11.4729175849873 5.16526979299139 13.8345741423057 5.23089883621166 16.2250392449321 5.29652787943193 18.6390503138731 5.3621569226522 21.0713168343251 5.42778596587247 23.5164534058665 5.49341500909274 25.9689027828306 5.55904405231301 28.4228468685695 5.62467309553328 30.8721030964039 5.69030213875355 33.3100030915547 5.75593118197382 35.7292500192196 5.82156022519409 38.1217507032784 5.88718926841436 40.478418673675 5.95281831163463 42.7889451698325 6.0184473548549 45.0415374686858 6.08407639807517 47.2226287937621 6.14970544129543 49.3165730790232 6.2153344845157 51.3053530755448 6.28096352773597 53.1683538199956 6.34659257095624 54.8822861482166 6.41222161417651 56.4213832946662 6.47785065739678 57.7580251731116 6.54347970061705 58.8639433758093 6.60910874383732 59.712086966048 6.67473778705759 60.2790539383215 6.74036683027786 60.5477339293022 6.80599587349813 60.5095697172039 6.8716249167184 60.1657951525476 6.93725395993867 59.5272522162741 7.00288300315894 58.612847822675 7.06851204637921 57.4471351796807 7.13414108959948 56.0576755727078 7.19977013281975 54.4727249068851 7.26539917604002 52.7195314601043 7.33102821926029 50.8232867067192 7.39665726248056 48.8066222656843 7.46228630570083 46.6894944835371 7.52791534892109 44.4893080711934 7.59354439214136 42.2211653177754 7.65917343536163 39.8981650701527
318.699003134214 107.961281237873 317.810919318338 104.549032383112 317.124365323398 101.286923451535 316.62133207621 98.1523220442667 316.286346926478 95.1249189180966 316.106249552566 92.1864917426143 316.069952556648 89.3206552396388 316.168207408996 86.5126190457639 316.393386151087 83.7489645645763 316.739283013779 81.0174460298045 317.20093641803 78.3068175544986 317.774469733037 75.6066861261166 318.456948039107 72.9073896891463 319.246247581369 70.1998992292058 320.140934362448 67.4757438773123 321.14014826693 64.7269583247175 322.243489166162 61.9460521668503 323.450901593654 59.1260010986365 324.762554818136 56.2602600950338 326.178715507612 53.3427987650344 327.699610727072 50.3681588971115 329.325279810846 47.3315337488107 331.055414766262 44.228867805137 332.889190354968 41.0569744863514 334.825086885752 37.8136676063676 336.860711002334 34.4979003085296 338.992622236327 31.1099028656578 341.21617557928 27.651308370179 343.525392442491 24.1252533274432 345.912873646494 20.5364389766276 348.369767996753 16.8911393187602 350.885808105621 13.197143784934 353.449421169867 9.46362647480319 356.047916518414 5.70093982703247 358.667744461562 1.92033789544955 1.29481328398533 -1.86635788486813 3.9148443840613 -5.64713070274768 6.51374082316837 -9.41015366081903 9.07794283802676 -13.1441621501542 11.5947455324468 -16.8387887757951 14.052558665977 -20.4848410091063 16.4410952751017 -24.0745085758404 18.7514835487217 -27.6014952962549 20.9763036765879 -31.0610774191797 23.1095573088322 -34.4500964481938 25.1465812449744 -37.7668984844273 27.0839188955118 -41.0112340921431 28.9191631675616 -44.1841328711507 30.6507831685101 -47.2877657469115 32.2779450153501 -50.3253059846937 33.800334556071 -53.3007975751352 35.2179873207777 -56.2190372963697 36.5311287660909 -59.0854746785141 37.7400259833039 -61.9061324108057 38.8448505454723 -64.6875484814536 39.8455510479256 -67.4367405066122 40.7417330943457 -70.1611922348915 41.5325439281469 -72.8688620398381 42.216558540417 -75.5682132645105 42.7916638473453 -78.2682664920608 43.2549373870937 -80.9786741191102 43.6025169290896 -83.7098179370193 43.8294574420011 -86.472930699351 43.9295721012875 -89.2802427619418 43.8952545723844 -92.1451546605965 43.7172809227143 -95.0824356840122 43.3845915907855 -98.1084467023331 42.8840575007928 -101.241382094702 42.2002406191839 -104.501519619731 41.3151694300957 -107.911457068636 40.2081659578176 -111.496298536539 38.855785654396 -115.283728578847 37.2319674697648 -119.303876619555 35.3085404701265 -123.588825010467 33.0562931911772 -128.171554384269 30.4468702813493 -133.084061627631 27.4557861836982 -138.354360648228 24.0667757981366 -144.002145948988 20.2774450071718 -150.033156106903 16.1056524497983 -156.432805771986 11.5952675331419 -163.160441212284 6.81918696299026 -170.14633715081 1.87734731231035 -177.293697410878 356.888558988403 175.513165949155 351.977266979787 168.395580347101 347.258684563862 161.465633367221 342.82656625385 154.813932659495 338.746571884431 148.502956899739 335.055695413037 142.566471342667 331.766183513906 137.013433004613 328.871621064078 131.834062541405 326.353224571907 127.006123847529 324.185197780512 122.500265596881 322.338740289965 118.284015573118 320.784758763121 114.324477379759 319.495531462301 110.589980329567 318.445615492698 107.050971975902
2061/07/09 01:07:01 09:16:42 +73:24:39 RJ 97
2.07177777519788 35.2660111784522 2.13740681444383 35.698613285259 2.20303585368978 36.0545155831139 2.26866489293574 36.3311517974379 2.33429393218169 36.5264892748016 2.39992297142764 36.6390721739363 2.4655520106736 36.6680535390581 2.53118104991955 36.6132144014602 2.5968100891655 36.4749687712756 2.66243912841145 36.2543542178896 2.72806816765741 35.953008606562 2.79369720690336 35.5731343669022 2.85932624614931 35.1174523324983 2.92495528539526 34.5891476501282 2.99058432464122 33.9918104825625 3.05621336388717 33.329374224406 3.12184240313312 32.6060537466677 3.18747144237907 31.8262858323336 3.25310048162503 30.9946735198112 3.31872952087098 30.1159355898682 3.38435856011693 29.1948619621861 3.44998759936288 28.2362753449176 3.51561663860884 27.2449991262208 3.58124567785479 26.2258312199238 3.64687471710074 25.1835233778274 3.7125037563467 24.1227653516205 3.77813279559265 23.0481732170777 3.8437618348386 21.9642811498275 3.90939087408455 20.8755359534778 3.97501991333051 19.7862936765418 4.04064895257646 18.7008177055903 4.10627799182241 17.6232777815369 4.17190703106836 16.5577494490505 4.23753607031432 15.5082135125638 4.30316510956027 14.4785551343396 4.36879414880622 13.4725622697618 4.43442318805217 12.4939231923366 4.50005222729813 11.5462229162283 4.56568126654408 10.6329383781131 4.63131030579003 9.75743229339171 4.69693934503599 8.92294565488499 4.76256838428194 8.13258889533561 4.82819742352789 7.38933178827286 4.89382646277384 6.69599221457697 4.9594555020198 6.05522397344178 5.02508454126575 5.46950386498411 5.0907135805117 4.9411183156905 5.15634261975765 4.47214985515366 5.22197165900361 4.06446378092521 5.28760069824956 3.71969536565382 5.35322973749551 3.43923796512541 5.41885877674146 3.22423237601555 5.48448781598742 3.07555776745142 5.55011685523337 2.9938244710769 5.61574589447932 2.97936886138252 5.68137493372527 3.03225049371205 5.74700397297123 3.15225159454378 5.81263301221718 3.33887892096975 5.87826205146313 3.59136792774595 5.94389109070909 3.90868910491603 6.00952012995504 4.28955628062284 6.07514916920099 4.73243662557878 6.14077820844694 5.23556205025629 6.2064072476929 5.79694165476097 6.27203628693885 6.41437487517763 6.3376653261848 7.08546496863799 6.40329436543075 7.80763249137108 6.46892340467671 8.57812844788737 6.53455244392266 9.39404682316685 6.60018148316861 10.2523362510611 6.66581052241456 11.1498106189317 6.73143956166052 12.0831584588963 6.79706860090647 13.0489510283726 6.86269764015242 14.043649035731 6.92832667939838 15.0636080201197 6.99395571864433 16.1050824476375 7.05958475789028 17.164228639162 7.12521379713623 18.2371066987202 7.19084283638219 19.3196816659534 7.25647187562814 20.4078241726301 7.32210091487409 21.4973109418197 7.38772995412004 22.5838255293994 7.453358993366 23.6629597705693 7.51898803261195 24.7302164576057 7.5846170718579 25.7810138366142 7.65024611110385 26.8106925663865 7.71587515034981 27.8145258256694 7.78150418959576 28.7877332782299 7.84713322884171 29.725499598106 7.91276226808766 30.6229982087944 7.97839130733362 31.4754207874749 8.04402034657957 32.2780129169694 8.10964938582552 33.0261160249491 8.17527842507148 33.7152154282878 8.24090746431743 34.3409939051744 8.30653650356338 34.8993897646752 8.37216554280933 35.3866579020788
7.63468117070479 -154.451744055422 6.40661056710009 -158.763610341198 5.13568702937578 -163.107454818134 3.82983426015161 -167.47759860535 2.49758984799858 -171.867748674761 1.14797277067932 -176.271130215127 359.790327272335 179.319357197403 358.434150185957 174.910966757923 357.088910327842 170.510923610432 355.763869355396 166.126245498507 354.467913337583 161.763574200957 353.209403260188 157.429026038127 351.996050930737 153.128067978482 350.834824514608 148.86542364385 349.731885525946 144.645011106251 348.692556806132 140.469912081371 347.721319082802 136.342370187325 346.821832262657 132.263814501172 345.996976727492 128.234903765432 345.248909550332 124.255586247787 344.579130644563 120.325170356133 343.9885542895 116.442401543994 343.477582119778 112.605541686305 343.046174413153 108.812447852275 342.693917272716 105.060648160722 342.420084011895 101.347413110196 342.223689674843 97.6698213914287 342.103538142598 94.0248196943091 342.058261683917 90.4092764118733 342.086353116464 86.8200294266216 342.186190962717 83.2539283526107 342.356058131547 79.7078717156412 342.594154747284 76.178839599314 342.89860579807 72.6639222814191 343.26746429703 69.1603453459439 343.698710653591 65.6654916917945 344.19024894552 62.1769207790172 344.739900770364 58.6923853641677 345.345397340813 55.2098458844549 346.004370473507 51.7274825603905 346.714343104559 48.2437052032399 347.472719946205 44.7571606404812 348.276778875203 41.266737613264 349.123663612129 37.7715689577221 350.010378208595 34.2710308596787 350.933783804151 30.7647389719394 351.890598044052 27.2525412062646 352.877397462466 23.7345070584207 353.89062303325 20.2109133932259 354.926588974212 16.6822267045172 355.981494764328 13.1490819681876 357.051440201736 9.61225831914183 358.132443199979 6.07265189829408 359.220459897807 2.53124632599705 0.311406551432595 -1.01091864415441 1.40118259409913 -4.55277965945874 2.48569419186565 -8.09328280641843 3.5608776005272 -11.6314164470524 4.62272163821358 -15.1662431560366 5.66728863080118 -18.6969301012088 6.69073325984123 -22.2227772773208 7.68931884032013 -25.7432430927846 8.65943067183393 -29.2579669148933 9.59758623449954 -32.7667882930719 10.5004421328554 -36.2697626944173 11.3647978203801 -39.7671736938569 12.1875962584034 -43.2595416564055 12.9659217718595 -46.7476290263123 13.696995457884 -50.2324423939607 14.378168580666 -53.7152315443839 15.0069144476115 -57.1974857007924 15.5808193093542 -60.6809271634076 16.0975728618353 -64.167502509961 16.5549589554132 -67.6593714719615 16.9508471366231 -71.1588935332888 17.2831856653322 -74.6686122182178 17.5499966654806 -78.1912369483617 17.7493740821587 -81.7296222563363 17.879485130915 -85.2867440529135 17.9385759346381 -88.865672559522 17.9249820448892 -92.4695414458599 17.8371445316135 -96.1015126612548 17.6736322887054 -99.764736428248 17.4331711313058 -103.462305889689 17.1146801399689 -107.197205980398 16.7173155211262 -110.972256246601 16.2405219861859 -114.790047576737 15.6840912881242 -118.652873150133 15.0482270839141 -122.562654365336 14.3336147113701 -126.520863078974 13.5414937910014 -130.528442157123 12.6737308168057 -134.585727084003 11.7328881372134 -138.692372134142 10.7222850267141 -142.847285315315 9.64604601200341 -147.048576828012 8.50913136203449 -151.293526044429 7.31734479880518 -155.578571865788
2030/01/23 07:35:11 16:05:09 +37:52:48 RJ 97
0.88887540501046 -31.5205394993778 0.954504443812333 -32.0205776164343 1.02013348261421 -32.3076816962153 1.08576252141608 -32.378608073906 1.15139156021795 -32.2325456109484 1.21702059901983 -31.8711604578405 1.2826496378217 -31.2985044988551 1.34827867662357 -30.5207990418569 1.41390771542545 -29.5461200152142 1.47953675422732 -28.3840201221779 1.54516579302919 -27.0451258585601 1.61079483183107 -25.5407436679795 1.67642387063294 -23.8825017477418 1.74205290943481 -22.0820445284311 1.80768194823669 -20.1507877134914 1.87331098703856 -18.0997343284768 1.93894002584043 -15.9393470504244 2.0045690646423 -13.6794691127057 2.07019810344418 -11.3292849049877 2.13582714224605 -8.89731148410348 2.20145618104792 -6.39141308118456 2.2670852198498 -3.81883193809814 2.33271425865167 -1.18623015825858 2.39834329745354 1.50026145239977 2.46397233625542 4.23499043197238 2.52960137505729 7.01272749831416 2.59523041385916 9.82861026118105 2.66085945266104 12.6780862162023 2.72648849146291 15.5568548370522 2.79211753026478 18.4608080396243 2.85774656906666 21.3859677438444 2.92337560786853 24.328418644571 2.9890046466704 27.2842335484245 3.05463368547228 30.2493876408385 3.12026272427415 33.2196566824076 3.18589176307602 36.1904922030316 3.2515208018779 39.1568639853255 3.31714984067977 42.1130560930156 3.38277887948164 45.0523968066029 3.44840791828352 47.9668942341156 3.51403695708539 50.8467369669664 3.57966599588726 53.6796017617904 3.64529503468914 56.4496873476096 3.71092407349101 59.1363677225864 3.77655311229288 61.7123422368784 3.84218215109475 64.1411922405883 3.90781118989663 66.3744343917867 3.9734402286985 68.3486955782 4.03906926750037 69.9848117951527 4.10469830630225 71.1924164368727 4.17032734510412 71.8842593107578 4.23595638390599 72.0000343987086 4.30158542270787 71.5287895144807 4.36721446150974 70.5135207376204 4.43284350031161 69.0340619408625 4.49847253911349 67.1818904868061 4.56410157791536 65.0415567950924 4.62973061671723 62.6827107710512 4.69535965551911 60.1593750152911 4.76098869432098 57.5123223744608 4.82661773312285 54.7720973869069 4.89224677192473 51.9616860215662 4.9578758107266 49.0985921513754 5.02350484952847 46.1963666062185 5.08913388833035 43.2657082302976 5.15476292713222 40.3152523174492 5.22039196593409 37.3521374146044 5.28602100473597 34.3824168453436 5.35165004353784 31.411361736374 5.41727908233971 28.4436881100221 5.48290812114158 25.483730666429 5.54853715994346 22.5355790441172 5.61416619874533 19.6031876685544 5.6797952375472 16.6904670868091 5.74542427634908 13.8013624642041 5.81105331515095 10.9399233594529 5.87668235395282 8.11036777507674 5.9423113927547 5.31714264547975 6.00794043155657 2.56498226580815 6.07356947035844 -0.141034401470595 6.13919850916032 -2.79542714351739 6.20482754796219 -5.39225880389625 6.27045658676406 -7.92508128582885 6.33608562556594 -10.386884816913 6.40171466436781 -12.7700529506393 6.46734370316968 -15.0663268606249 6.53297274197156 -17.2667837239815 6.59860178077343 -19.3618353328119 6.6642308195753 -21.3412543874913 6.72985985837718 -23.194236983243 6.79548889717905 -24.9095102549331 6.86111793598092 -26.4754935307032 6.9267469747828 -27.8805191334361 6.99237601358467 -29.1131146839025 7.05800505238654 -30.1623421790492 7.12363409118842 -31.0181805326509 7.18926312999029 -31.671928701374
350.154984471656 11.745061994187 353.60014085433 7.6257427457746 357.093729250116 3.46055811791766 0.609664127738686 -0.725815620773106 4.12089272970983 -4.90772545037203 7.60056752210497 -9.05972556724042 11.0231959894775 -13.1577387310249 14.3656439467191 -17.1800713620933 17.6078970068309 -21.1081872343972 20.7335286777513 -24.9271884558254 23.7298706486611 -28.6259995349495 26.5879201389507 -32.1972896729273 29.3020441161194 -35.6371933882962 31.8695494553766 -38.9448988952172 34.2901845272742 -42.1221701283031 36.5656260207281 -45.1728567027914 38.6989897943892 -48.1024311784579 40.6943897960074 -50.9175783607906 42.5565566945786 -53.6258491274911 44.2905186343532 -56.2353822216206 45.9013403693031 -58.7546915420627 47.3939134400034 -61.19251317109 48.7727883316315 -63.5577050472089 50.0420390548001 -65.8591921988173 51.2051507883604 -68.1059513047092 52.2649217157773 -70.3070296905927 53.223370690585 -72.47159548941 54.0816426835836 -74.6090174801113 54.8399039493636 -76.728975054195 55.4972183794027 -78.8416008911608 56.0513954556242 -80.9576613636931 56.4987984244176 -83.08878261291 56.8340985646545 -85.2477338752409 57.0499574320608 -87.4487843326963 57.1366133293062 -89.7081559294058 57.0813404578672 -92.0446027993961 56.8677386305464 -94.4801588344701 56.4747974448512 -97.0411091270025 55.8756611831034 -99.7592588230229 55.0360004862524 -102.673593259647 53.9118788304379 -105.832441363709 52.4470001740029 -109.296256008616 50.5692752003517 -113.141074008426 48.1868334974034 -117.462528538504 45.184127298932 -122.379768503287 41.4199941822347 -128.037417611276 36.7320692375274 -134.601182800327 30.9561910560382 -142.238467825542 23.9737344394837 -151.071059265261 15.7955074541627 -161.091247477963 6.66057639604747 -172.063019097291 357.072528902892 176.514189019647 347.687618899049 165.293617927113 339.091419873811 154.857823161649 331.629149768129 145.548982876497 325.390321181626 137.4535311039 320.295120228891 130.488519325572 316.189582155253 124.496774311212 312.907099166449 119.308387526826 310.297104732568 114.769380218163 308.234042171959 110.750651817526 306.617192078193 107.147786078012 305.367217105371 103.87757329141 304.422086664677 100.873910986089 303.733392772134 98.0840941949005 303.263332467329 95.4657705755958 302.982344608107 92.9845480728845 302.867297880146 90.6121518788392 302.900114139498 88.3250147781268 303.066725339155 86.1031991712815 303.356282558397 83.9295694417945 303.760554556062 81.7891523817462 304.273468689889 79.6686390130655 304.89075888681 77.5559932574432 305.609694153754 75.4401420784586 306.428867528398 73.3107286133307 307.348029949158 71.1579150235913 308.367956717177 68.9722258013735 309.490336360644 66.7444254368354 310.717673033682 64.4654269556139 312.053194264245 62.1262300710286 313.500756044348 59.7178896982129 315.064737053343 57.231517420334 316.749913354178 54.6583201890379 318.561304377291 51.9896820114284 320.503980653324 49.2172954530835 322.582823925195 46.3333501749303 324.802231441734 43.3307849821968 327.165759016653 40.2036074187901 329.675702510719 36.9472801299295 332.332625380962 33.5591654372382 335.134851136353 30.0390085386436 338.077953539747 26.3894258747254 341.154292653835 22.6163500481304 344.352658331639 18.7293692615758 347.658090108234 14.7418919485097 351.051938706165 10.6710710657453
2006/04/09 09:06:39 05:39:48 +29:35:59 RJ 97
2.59316421231112 32.1474459565591 2.65879325075424 28.9009574302959 2.72442228919736 25.6666530835741 2.79005132764049 22.446993250297 2.85568036608361 19.2444896058597 2.92130940452673 16.0617593101717 2.98693844296985 12.9015754664106 3.05256748141297 9.76691649719285 3.1181965198561 6.66101644918524 3.18382555829922 3.58741785017325 3.24945459674234 0.550028471664365 3.31508363518546 -2.44681686936152 3.38071267362859 -5.39828855941051 3.44634171207171 -8.29898786171404 3.51197075051483 -11.1428657960503 3.57759978895795 -13.9231356209404 3.64322882740108 -16.6321802317552 3.7088578658442 -19.261457130783 3.77448690428732 -21.8014055281428 3.84011594273044 -24.241362726295 3.90574498117357 -26.5695003080242 3.97137401961669 -28.7727947421319 4.03700305805981 -30.8370515422372 4.10263209650293 -32.7470063375924 4.16826113494606 -34.4865288163642 4.23389017338918 -36.038954475208 4.2995192118323 -37.3875619582942 4.36514825027542 -38.5161982016549 4.43077728871854 -39.4100288436118 4.49640632716167 -40.0563598614772 4.56203536560479 -40.4454449911349 4.62766440404791 -40.5711727396969 4.69329344249103 -40.4315274850585 4.75892248093415 -40.0287467303275 4.82455151937727 -39.3691466962782 4.8901805578204 -38.4626470654485 4.95580959626352 -37.3220749033295 5.02143863470664 -35.9623539567455 5.08706767314976 -34.3996848811785 5.15269671159288 -32.6508003547174 5.21832575003601 -30.732347387595 5.28395478847913 -28.6604178596941 5.34958382692225 -26.4502240201016 5.41521286536537 -24.1159005969908 5.48084190380849 -21.6704084003974 5.54647094225162 -19.1255135240654 5.61209998069474 -16.4918189833719 5.67772901913786 -13.7788298925504 5.74335805758098 -10.9950377967142 5.8089870960241 -8.14801383494293 5.87461613446723 -5.24450373678719 5.94024517291035 -2.29052020939152 6.00587421135347 0.708569857585885 6.07150324979659 3.74796462516384 6.13713228823971 6.82335333330427 6.20276132668284 9.93084888076995 6.26839036512596 13.0669263127549 6.33401940356908 16.228366293424 6.3996484420122 19.4122024185966 6.46527748045532 22.6156710030543 6.53090651889845 25.8361617016577 6.59653555734157 29.0711669282126 6.66216459578469 32.3182274315331 6.72779363422781 35.5748704387024 6.79342267267093 38.8385352608029 6.85905171111406 42.1064788025651 6.92468074955718 45.3756493686729 6.9903097880003 48.6425103296847 7.05593882644342 51.9027833883624 7.12156786488654 55.1510601196398 7.18719690332967 58.3801917013568 7.25282594177279 61.5802930248837 7.31845498021591 64.7370523833647 7.38408401865903 67.828744704998 7.44971305710215 70.8207469745533 7.51534209554528 73.6551801305097 7.5809711339884 76.2314861036762 7.64660017243152 78.3741455815645 7.71222921087464 79.8038976727054 7.77785824931776 80.2022536945412 7.84348728776088 79.4514198014086 7.90911632620401 77.7615768532607 7.97474536464713 75.457450267272 8.04037440309025 72.7853190034123 8.10600344153337 69.8930029426806 8.17163247997649 66.8647749700375 8.23726151841962 63.7496348645444 8.30289055686274 60.5773259922226 8.36851959530586 57.3668010245264 8.43414863374898 54.1307347888423 8.4997776721921 50.8780104603615 8.56540671063523 47.6151445233162 8.63103574907835 44.3471368933465 8.69666478752147 41.0779981931145 8.76229382596459 37.8110893710125 8.82792286440771 34.5493488501534 8.89355190285084 31.2954505159875
293.185124803295 84.0554772398043 293.631804393339 82.4242448244092 294.16794240508 80.8119150308964 294.788969120214 79.2069799857686 295.492103729985 77.5990992208674 296.276064889369 75.9787394040955 297.140865733074 74.3368925297331 298.087670442338 72.6648499596585 299.118696154671 70.954016902427 300.237148306388 69.1957569435899 301.447180148114 67.3812599453674 302.753868622813 65.5014295800642 304.1631992939 63.5467893460857 305.682052678978 61.5074084368779 307.318183221985 59.3728515297807 309.080180218168 57.1321596242833 310.977397286576 54.7738726193079 313.019833518734 52.286108408674 315.217945429419 49.6567177432918 317.582364801637 46.8735384948797 320.123494446511 43.9247762955204 322.850953496095 40.7995391356266 325.772848711023 37.4885487976133 328.894861864944 33.9850385456627 332.219169304654 30.285820551007 335.743250847658 26.3924655445653 339.458700059314 22.3124823673499 343.350208313873 18.0603247732407 347.394942985927 13.6580049328229 351.562549714354 9.13508351958638 355.815954564649 4.52786039764536 0.113014555101009 -0.122282693349331 4.40888548687804 -4.77125652356516 8.65879783965381 -9.3750467103541 12.8208211502898 -13.8924845417177 16.8582004882584 -18.287590016236 20.7409640131178 -22.5311862051606 24.4466805230201 -26.6016639735551 27.960423781818 -30.4849539876224 31.2741241420442 -34.173886681459 34.3855383025838 -37.6671711999558 37.2970557380227 -40.9682120613215 40.0145111448152 -44.0839331276217 42.5461118167563 -47.0237180860018 44.9015345764111 -49.7985224172034 47.0912066321438 -52.4201716471402 49.1257594307839 -54.9008354749748 51.0156315485156 -57.2526544684376 52.7707921548055 -59.4874916701232 54.4005571834525 -61.6167822661844 55.9134735020424 -63.6514578980206 57.3172504228418 -65.6019266068513 58.6187218819358 -67.4780938585993 59.8238260436573 -69.2894141598986 60.9375917654729 -71.0449663081233 61.9641232347309 -72.7535483577089 62.9065751700238 -74.4237910673445 63.7671112698521 -76.0642911110276 64.5468380472426 -77.6837679313592 65.2457046840937 -79.2912510885977 65.8623568154475 -80.8963087172474 66.393927752074 -82.5093328180192 66.8357437909824 -84.1419044539466 67.180909647374 -85.8072728520481 67.4197234891734 -87.5209991641126 67.5388448694504 -89.3018419448526 67.5200966489476 -91.1730036723258 67.338712539668 -93.1639271234173 66.960724992168 -95.3129473280962 66.3389868972941 -97.6713060657243 65.4069668274231 -100.309389591305 64.0688261839094 -103.326681612389 62.1831589461593 -106.868051181518 59.535835255237 -111.150934604745 55.7945555507438 -116.510804955597 50.4363762709118 -123.473668469506 42.6556798116282 -132.848115387509 31.3650600897949 -145.724454150763 15.7533423209496 -162.916710568356 356.976284938651 176.728055748218 338.876669740596 157.04982355649 324.636099945725 141.227390237735 314.5728757706 129.57622953884 307.661175201689 121.067654273109 302.883467678802 114.681211486961 299.531985738428 109.706134926507 297.153385761621 105.685985724506 295.458939836942 102.328823081538 294.261254768157 99.4439007189439 293.435556403124 96.9029270093137 292.896612038614 94.6169612472873 292.584844277452 92.5224991212118 292.457794147148 90.5729048530088 292.484740456242 88.7329966136288 292.643223315902 86.9755327458912 292.916745371512 85.2788719191999 293.293219915632 83.6253759484589
1999/08/07 18:24:59 01:52:05 -29:41:01 RJ 97
0.811886160361015 37.3810260362627 0.877515207794718 36.0459444137141 0.943144255228421 34.5063965257786 1.00877330266212 32.7790277229392 1.07440235009583 30.8804629098025 1.14003139752953 28.8268227767969 1.20566044496323 26.6333850803054 1.27128949239694 24.3143741267917 1.33691853983064 21.8828540504228 1.40254758726434 19.3507001210868 1.46817663469804 16.728624707691 1.53380568213175 14.0262386296491 1.59943472956545 11.2521331053641 1.66506377699915 8.41397159860144 1.73069282443286 5.51858425061223 1.79632187186656 2.57206020802472 1.86195091930026 -0.420164911763468 1.92757996673396 -3.45322780778435 1.99320901416767 -6.52276298659879 2.05883806160137 -9.6248347787686 2.12446710903507 -12.7558752812951 2.19009615646878 -15.9126273342634 2.25572520390248 -19.0920914107139 2.32135425133618 -22.291475075999 2.38698329876988 -25.5081434009122 2.45261234620359 -28.7395683273292 2.51824139363729 -31.9832743991806 2.58387044107099 -35.2367773545767 2.64949948850469 -38.49751061588 2.7151285359384 -41.7627323572454 2.7807575833721 -45.0294019512927 2.8463866308058 -48.2940080773074 2.9120156782395 -51.5523195326643 2.97764472567321 -54.79900984661 3.04327377310691 -58.0270702924719 3.10890282054061 -61.2268568400132 3.17453186797432 -64.3844816165233 3.24016091540802 -67.4789881920844 3.30578996284172 -70.4771986034795 3.37141901027542 -73.3240404033682 3.43704805770913 -75.9244531083337 3.50267710514283 -78.1129355451083 3.56830615257653 -79.6233283916308 3.63393520001023 -80.1376561730643 3.69956424744394 -79.5080131657916 3.76519329487764 -77.912262007666 3.83082234231134 -75.6705525062502 3.89645138974505 -73.0384252426878 3.96208043717875 -70.1723686172854 4.02770948461245 -67.1621060920475 4.09333853204615 -64.0597851066535 4.15896757947986 -60.8969702664779 4.22459662691356 -57.693698949648 4.29022567434726 -54.4633153518836 4.35585472178096 -51.2151302250206 4.42148376921467 -47.9559417032155 4.48711281664837 -44.6909400603345 4.55274186408207 -41.4242671568153 4.61837091151577 -38.1593755964839 4.68399995894948 -34.8992680735537 4.74962900638318 -31.6466631552693 4.81525805381688 -28.4041149797981 4.88088710125059 -25.1741037387555 4.94651614868429 -21.9591076430145 5.01214519611799 -18.7616633910195 5.07777424355169 -15.5844199176102 5.1434032909854 -12.4301888118343 5.2090323384191 -9.30199391756863 5.2746613858528 -6.20312207069922 5.3402904332865 -3.1371765563417 5.40591948072021 -0.108134605461154 5.47154852815391 2.87958997065151 5.53717757558761 5.82107814262443 5.60280662302131 8.71083045897369 5.66843567045502 11.5426851512903 5.73406471788872 14.3097300809831 5.79969376532242 17.0042100372779 5.86532281275613 19.6174323163976 5.93095186018983 22.1396755096468 5.99658090762353 24.5601091275561 6.06220995505723 26.8667351533217 6.12783900249094 29.0463667714287 6.19346804992464 31.0846639990462 6.25909709735834 32.9662499518437 6.32472614479204 34.6749335809519 6.39035519222575 36.1940628395112 6.45598423965945 37.5070238867619 6.52161328709315 38.5978850051237 6.58724233452685 39.4521580494976 6.65287138196056 40.0576185507581 6.71850042939426 40.4050955196519 6.78412947682796 40.4891242906373 6.84975852426167 40.3083605568303 6.91538757169537 39.8656851159273 6.98101661912907 39.1679808062889 7.04664566656277 38.2256207432085 7.11227471399648 37.0517527430031
200.302539252179 22.0689496013267 204.022143021936 26.1558599084471 207.551768833494 30.0575233698799 210.882635865375 33.7660614798445 214.01188137724 37.2795655650224 216.941386554237 40.6009328189422 219.676591473806 43.7366927317289 222.225413878146 46.6959388851554 224.597331572514 49.4894262785754 226.802646713684 52.1288528702191 228.851923568866 54.626317429729 230.755577153238 56.993931747948 232.523584838164 59.2435600914323 234.165293121701 61.3866590926719 235.689294644234 63.4341944243146 237.10335447417 65.396614897149 238.41436865658 67.2838690520593 239.628341482658 69.1054533986126 240.750370674621 70.8704850214452 241.784631624947 72.5877943514157 242.734352981801 74.2660366067833 243.601776237769 75.9138229366916 244.388091518608 77.5398748795033 245.093340365045 79.1532086765216 245.716273711056 80.7633596475632 246.254149051399 82.3806617912129 246.702444220868 84.0166048484608 247.054455045331 85.6843015646108 247.300728308627 87.3991139126739 247.428256521977 89.1795121267337 247.419320881776 91.0482805646072 247.24980302971 93.0342502221662 246.886675959907 95.1748479967221 246.2841954056 97.5199417988899 245.377982338326 100.137791292175 244.075599801979 103.12450142563 242.241182698539 106.619420503815 239.66988620225 110.830717451856 236.045274420313 116.078016966805 230.871262131486 122.860480010072 223.383450316449 131.945489192682 212.533763482514 144.38403129866 197.455013726262 161.046152236088 179.011789461104 -178.929913289775 160.787595840519 -159.124864199438 146.1536082639 -142.907068738245 135.704010150798 -130.867884433298 128.500772448975 -122.0664195694 123.517524857338 -115.473385957797 120.021339012797 -110.352863033333 117.538735179071 -106.2282845956 115.767509443468 -102.794244965586 114.511582765702 -99.8513256522951 113.640557174089 -97.2656275570782 113.065488627344 -94.9445179454552 112.724291548963 -92.8220068815874 112.572769482206 -90.8497472361837 112.578968098176 -88.9913541364604 112.719531604223 -87.2187245150622 112.977296839993 -85.5095906306537 113.339671161996 -83.8458532275812 113.797518785758 -82.2124185612754 114.344384666601 -80.5963679359873 114.975947422144 -78.9863508579505 115.689630942329 -77.3721311247794 116.484328119271 -75.7442390936847 117.360205228445 -74.0936986932687 118.318565205026 -72.4118078049307 119.361754338231 -70.6899574459188 120.493100921305 -68.9194799761721 121.716876845294 -67.0915201183911 123.038274421989 -65.1969254566689 124.463391092468 -63.2261556413091 125.999214230086 -61.1692120632143 127.653597012292 -59.0155924998009 129.435214308809 -56.754278356747 131.353484716689 -54.3737657530211 133.418441341331 -51.8621558270676 135.640529925936 -49.2073240876934 138.030309040683 -46.3971928639712 140.598024345883 -43.4201338641962 143.353029267485 -40.2655277284066 146.303030486222 -36.9245015494657 149.453151946677 -33.3908501518395 152.804839250088 -29.6621188527478 156.354669398326 -25.7407823995763 160.093186747957 -21.6353989432189 164.003945013598 -17.3615589716362 168.062978023459 -12.9424062993514 172.238922472234 -8.40850769365883 176.493950623434 -3.79691303218306 180.785534539885 0.850615724698626 185.068882275997 5.49004730715365 189.299717072633 10.0778668159302 193.436977417962 14.5737822897592 197.445037709152 18.9429520128818 201.295176868467 23.157460038486
2067/01/09 22:41:52 02:52:19 -20:07:35 RJ 97
4.61569123336471 -51.89825349901 4.68132027689066 -48.3915981904806 4.74694932041661 -44.8911270598318 4.81257836394257 -41.3975823715063 4.87820740746852 -37.9117699639617 4.94383645099447 -34.4345689057061 5.00946549452042 -30.96694258027 5.07509453804638 -27.5099514717751 5.14072358157233 -24.0647679761077 5.20635262509828 -20.6326936277522 5.27198166862423 -17.2151792114873 5.33761071215018 -13.8138483242008 5.40323975567614 -10.4305250682635 5.46886879920209 -7.0672666977344 5.53449784272804 -3.72640220569481 5.60012688625399 -0.410578038289608 5.66575592977994 2.87718764981895 5.7313849733059 6.13344052457302 5.79701401683185 9.35421343063896 5.8626430603578 12.5349353384156 5.92827210388375 15.6703233912511 5.99390114740971 18.7542554233082 6.05953019093566 21.7796204100568 6.12515923446161 24.738144844754 6.19078827798756 27.6201943364099 6.25641732151351 30.4145523019523 6.32204636503947 33.1081822024881 6.38767540856542 35.685987306398 6.45330445209137 38.1305935338692 6.51893349561732 40.4221973993534 6.58456253914327 42.5385422052196 6.65019158266923 44.4551086496747 6.71582062619518 46.1456233750941 6.78144966972113 47.5829866971572 6.84707871324708 48.7406795501789 6.91270775677303 49.5946131724995 6.97833680029899 50.1252366605159 7.04396584382494 50.3195587391402 7.10959488735089 50.1726497451043 7.17522393087684 49.6882452497548 7.2408529744028 48.8782869386927 7.30648201792875 47.7615263961778 7.3721110614547 46.3615491619294 7.43774010498065 44.70465533563 7.5033691485066 42.8179591098692 7.56899819203256 40.7279150721307 7.63462723555851 38.459325784013 7.70025627908446 36.0347805004407 7.76588532261041 33.4744267313279 7.83151436613636 30.7959698596591 7.89714340966232 28.014811651309 7.96277245318827 25.1442613009301 8.02840149671422 22.1957743067959 8.09403054024017 19.1791916325235 8.15965958376612 16.1029638280574 8.22528862729208 12.9743528124638 8.29091767081803 9.79960894777084 8.35654671434398 6.58412383990289 8.42217575786993 3.33256074762853 8.48780480139589 0.0489650959248162 8.55343384492184 -3.26314227181597 8.61906288844779 -6.60068658582158 8.68469193197374 -9.96097310813043 8.75032097549969 -13.3416315612892 8.81595001902565 -16.740569315384 8.8815790625516 -20.1559318843071 8.94720810607755 -23.5860695156921 9.0128371496035 -27.0295088606217 9.07846619312945 -30.4849288802864 9.14409523665541 -33.9511402901834 9.20972428018136 -37.427067961706 9.27535332370731 -40.9117357996283 9.34098236723326 -44.404253695314 9.40661141075921 -47.9038062223374 9.47224045428517 -51.409642796048 9.53786949781112 -54.9210690634483 9.60349854133707 -58.4374393261074 9.66912758486302 -61.9581498276564 9.73475662838898 -65.4826327586078 9.80038567191493 -69.0103508421361 9.86601471544088 -72.5407923541278 9.93164375896683 -76.0734663555179 9.99727280249278 -79.60789756468 10.0629018460187 -83.1436182680837 10.1285308895447 -86.6801330913515 10.1941599330706 -89.7800432466043 10.2597889765966 -86.245533573286 10.3254180201225 -82.7090847964233 10.3910470636485 -79.1734971874535 10.4566761071744 -75.6392566678516 10.5223051507004 -72.1068311267564 10.5879341942264 -68.5766974463535 10.6535632377523 -65.0493488604681 10.7191922812783 -61.5252999749916 10.7848213248042 -58.0050915431396 10.8504503683302 -54.4892954894449 10.9160794118561 -50.9785203879931
97.2060784053552 -82.891425848478 97.9236122021612 -82.1648585787445 98.6587967330705 -81.4219866302536 99.4134873337833 -80.6606613768962 100.189740526705 -79.8786122185622 100.989824791035 -79.0734107405782 101.816238916366 -78.2424362435159 102.671736693773 -77.3828403680686 103.55935752519 -76.4915092113427 104.482463027167 -75.5650216529615 105.444780031242 -74.5996027469234 106.450450618002 -73.591071072527 107.504090001255 -72.5347789231225 108.610853213946 -71.4255441787001 109.776511632183 -70.2575726888636 111.007540384234 -69.0243700247382 112.311217582348 -67.7186415956301 113.695736011437 -66.3321804491314 115.170327291674 -64.8557427034759 116.745397420701 -63.2789116814972 118.432670724578 -61.5899536985485 120.245336210465 -59.7756714975541 122.198185561683 -57.8212660800891 124.307724797854 -55.710224902758 126.592231003156 -53.4242650297089 129.071710486611 -50.9433748725331 131.767694508225 -48.2460183800037 134.702783511786 -45.3095907338493 137.899823351224 -42.111242061308 141.380574918769 -38.6292077543081 145.163737761342 -34.8447848073091 149.262235557881 -30.7450462998335 153.679802559738 -26.3262549205131 158.40715583037 -21.5976906955523 163.418389613509 -16.5852565917043 168.668588968707 -11.3338648542106 174.093821085176 -5.90744566825177 179.614371540909 -0.385710876421023 185.141247153148 5.14234889429507 190.584821593168 10.5871098694077 195.863632943381 15.8671127061138 200.911277921118 20.9159567395411 205.680126716115 25.6860148434965 210.141727777866 30.1488382423144 214.284683906714 34.2930326281815 218.111149671645 38.1207556074277 221.632987624693 41.6438729463124 224.86827292018 44.8804632252616 227.838474824322 47.851999387913 230.566381148766 50.5812732195753 233.074685864888 53.0909830131332 235.38510343684 55.4028479619143 237.517869405392 57.5371088133775 239.491507054967 59.5122946183462 241.322767309662 61.3451627244494 243.026674755118 63.0507449142603 244.616633654615 64.6424535611588 246.104563555613 66.1322174050722 247.501045257767 67.5306277268456 248.815465545294 68.847083334194 250.056154132727 70.0899278215741 251.230509510706 71.2665758083889 252.345112389561 72.3836268771973 253.405826637689 73.4469671415945 254.417888284983 74.4618590591555 255.385983500695 75.4330204620419 256.314316585926 76.3646939354083 257.206669023795 77.2607077167051 258.066450554086 78.1245292741299 258.896743110234 78.9593126896385 259.700338283563 79.7679409529676 260.479768754026 80.5530643008729 261.237333816686 81.3171358554741 261.975118668581 82.0624461046054 262.695006354875 82.791158375024 263.398679894794 83.505348688555 264.087609420662 84.207055957296 264.763013506678 84.898354026237 265.425770945601 85.5814699009641 266.07622680915 86.2590048447653 266.713745194804 86.934406398954 267.335560774796 87.6131396139779 267.933261381366 88.3062256167777 268.478350817366 89.0446986492372 268.815462035791 89.9864117297304 98.9141733768797 -98.8357614770379 91.195202492142 -89.8401029531621 91.5834337457528 -88.9490603261964 92.1382937783071 -88.2196050152733 92.7398108316328 -87.5292738800433 93.3638357550741 -86.851372043972 94.0030320829942 -86.1759440220401 94.6550074895225 -85.4978851826893 95.3192868629436 -84.8138934877076 95.9962975367115 -84.1214317191495 96.6869675377295 -83.4183006894737 97.3925497663928 -82.702435940021
2009/07/12 03:39:19 13:07:48 -84:59:11 RB 97
2.78283443226847 -16.1758746137882 2.8484634720641 -15.9922702060941 2.91409251185974 -15.8248515709378 2.97972155165537 -15.6742968749655 3.04535059145101 -15.5412151779884 3.11097963124664 -15.4261442569393 3.17660867104228 -15.3295486767351 3.24223771083791 -15.2518181172166 3.30786675063355 -15.1932659646707 3.37349579042918 -15.1541281755003 3.43912483022482 -15.1345624184185 3.50475387002045 -15.1346475001437 3.57038290981609 -15.1543830780061 3.63601194961172 -15.1936896611984 3.70164098940736 -15.252408900661 3.76727002920299 -15.3303041658579 3.83289906899863 -15.427061405016 3.89852810879426 -15.5422902838396 3.9641571485899 -15.6755255963115 4.02978618838553 -15.826228940007 4.09541522818116 -15.9937906474084 4.1610442679768 -16.1775319640431 4.22667330777243 -16.3767074638939 4.29230234756807 -16.5905076924467 4.3579313873637 -16.8180620279458 4.42356042715934 -17.0584417518848 4.48918946695497 -17.3106633204617 4.55481850675061 -17.5736918295966 4.62044754654624 -17.8464446671255 4.68607658634188 -18.1277953468551 4.75170562613751 -18.4165775202354 4.81733466593315 -18.7115891624011 4.88296370572878 -19.0115969301663 4.94859274552442 -19.3153406901487 5.01422178532005 -19.6215382154793 5.07985082511569 -19.9288900494326 5.14547986491132 -20.2360845337352 5.21110890470695 -20.5418029982059 5.27673794450259 -20.8447251067229 5.34236698429822 -21.1435343522481 5.40799602409386 -21.4369236907857 5.47362506388949 -21.723601300709 5.53925410368513 -22.0022964499017 5.60488314348076 -22.271765448709 5.6705121832764 -22.5307976618608 5.73614122307203 -22.7782215474669 5.80177026286767 -23.0129106860326 5.8673993026633 -23.2337897573965 5.93302834245894 -23.4398404187533 5.99865738225457 -23.6301070327087 6.06428642205021 -23.8037021908407 6.12991546184584 -23.9598119757296 6.19554450164148 -24.0977009030626 6.26117354143711 -24.2167164853781 6.32680258123275 -24.3162933604236 6.39243162102838 -24.3959569300185 6.45806066082401 -24.4553264597574 6.52368970061965 -24.4941175957974 6.58931874041528 -24.5121442622296 6.65494778021092 -24.5093199109391 6.72057682000655 -24.4856581051817 6.78620585980219 -24.4412724280349 6.85183489959782 -24.3763757171158 6.91746393939346 -24.2912786371358 6.98309297918909 -24.186387611666 7.04872201898473 -24.0622021445832 7.11435105878036 -23.9193115697853 7.179980098576 -23.7583912746714 7.24560913837163 -23.5801984484017 7.31123817816727 -23.3855674099886 7.3768672179629 -23.1754045737728 7.44249625775854 -22.9506831108473 7.50812529755417 -22.7124373645795 7.57375433734981 -22.4617570766961 7.63938337714544 -22.1997814776009 7.70501241694107 -21.9276932909025 7.77064145673671 -21.6467126977478 7.83627049653234 -21.3580913017138 7.90189953632798 -21.0631061299179 7.96752857612361 -20.7630537008593 8.03315761591925 -20.4592441844925 8.09878665571488 -20.1529956752891 8.16441569551052 -19.8456285947139 8.23004473530615 -19.5384602356909 8.29567377510179 -19.2327994583541 8.36130281489742 -18.9299415436756 8.42693185469306 -18.631163209474 8.49256089448869 -18.3377177917927 8.55818993428433 -18.0508305936866 8.62381897407996 -17.7716944030067 8.6894480138756 -17.5014651807656 8.75507705367123 -17.2412579220407 8.82070609346687 -16.9921426920345 8.8863351332625 -16.7551408408003 8.95196417305814 -16.53122140116 9.01759321285377 -16.3212976754223 9.0832222526494 -16.1262240175737
176.897517017449 -38.5071489154559 177.154166543982 -34.8305715734862 177.422301800125 -31.1563867640265 177.700819752197 -27.4843043461098 177.988582254106 -23.8140454406514 178.284419945042 -20.1453419546464 178.587136226074 -16.477935800408 178.895511311068 -12.8115778414235 179.208306345107 -9.14602660015668 179.524267581465 -5.48104676641541 179.842130606249 -1.81640754767535 180.160624598281 1.84811909504068 180.478476610572 5.51276028052009 180.794415859046 9.17774406377113 181.107178003889 12.8433012689783 181.415509409183 16.5096672528246 181.718171367184 20.1770835529845 182.013944274838 23.8457993782045 182.301631751647 27.5160728986018 182.580064689979 31.1881722975891 182.848105230985 34.8623765501271 183.10465066161 38.5389758957619 183.348637230424 42.2182719790777 183.579043882225 45.9005776347245 183.794895913262 49.5862162990248 183.995268550591 53.2755210352836 184.179290460172 56.968833165269 184.346147188919 60.6665005048985 184.495084545817 64.3688752079081 184.625411926381 68.0763112272016 184.73650558314 71.7891614096634 184.827811842333 75.507774246448 184.898850263735 79.2324903071358 184.949216736395 82.9636383926327 184.978586498187 86.7015314482586 184.98671706149 90.4464622850677 184.973451021203 94.1986991639818 184.938718714742 97.9584813037032 184.882540696955 101.726014379454 184.805029986217 105.501466085199 184.70639403158 109.284961836943 184.586936345117 113.076580698711 184.447057738809 116.87635161565 184.287257101815 120.684250040091 184.108131652051 124.500195036061 183.91037659612 128.324046945384 183.694784133819 132.15560569395 183.462241748247 135.994609809745 183.213729729747 139.840736214754 182.950317891859 143.693600840809 182.673161449879 147.55276010504 182.383496047374 151.417713263934 182.082631932772 155.28790564653 181.771947306386 159.162732747448 181.452880877437 163.041545139839 181.126923690029 166.923654147691 180.795610295965 170.808338196952 180.460509369833 174.694849746437 180.123213877348 178.5824226832 179.785330920685 -177.529719946227 179.448471393925 -173.642358007152 179.114239587323 -169.756266294835 178.784222880636 -165.872206645202 178.459981663101 -161.99092028122 178.143039611007 -158.113120531086 177.834874443445 -154.239486043562 177.536909263257 -150.370654611615 177.250504574049 -146.507217698589 176.976951046106 -142.649715742141 176.717463084952 -138.798634290958 176.473173236839 -134.954401008554 176.245127446507 -131.117383557962 176.034281164578 -127.287888361651 175.841496285761 -123.466160212939 175.667538884841 -119.652382699132 175.513077705681 -115.846679382784 175.378683349274 -112.04911567618 175.264828100296 -108.259701335283 175.171886327593 -104.478393493119 175.100135392366 -100.705100148606 175.049756998359 -96.9396840250565 175.020838920662 -93.1819667127354 175.013377053684 -89.4317330116701 175.027277723892 -85.688735394085 175.062360218887 -81.9526985101278 175.118359490857 -78.2233236656898 175.194928999165 -74.500293206883 175.291643663539 -70.7832747519115 175.408002905744 -67.0719252175161 175.543433763659 -63.3658945937358 175.697294067022 -59.6648294273214 175.868875668845 -55.9683759806929 176.057407730369 -52.2761830397906 176.262060060529 -48.5879043505037 176.481946513088 -44.903200669563 176.716128446055 -41.2217414218131 176.963618248575 -37.5432059616644
2016/11/19 23:39:09 01:59:05 -32:16:01 RJ 97
3.99074569151496 -69.1970444282445 4.05637473566116 -66.4124768060264 4.12200377980736 -63.4987103345312 4.18763282395356 -60.4962339314889 4.25326186809976 -57.4319644452626 4.31889091224596 -54.3243605941659 4.38451995639216 -51.1865155508299 4.45014900053837 -48.028046085834 4.51577804468457 -44.8562709509748 4.58140708883077 -41.6769648170888 4.64703613297697 -38.4948540154957 4.71266517712317 -35.3139519504294 4.77829422126937 -32.1377929606768 4.84392326541557 -28.9696007433784 4.90955230956177 -25.8124140644366 4.97518135370797 -22.6691844124924 5.04081039785417 -19.5428552956664 5.10643944200037 -16.4364297758298 5.17206848614657 -13.3530308559998 5.23769753029278 -10.2959580510274 5.30332657443898 -7.26874261729575 5.36895561858518 -4.27520332775375 5.43458466273138 -1.31950424369363 5.50021370687758 1.59378442590932 5.56584275102378 4.45962163821903 5.63147179516998 7.27242666758077 5.69710083931618 10.0260062764696 5.76272988346238 12.7134787937814 5.82835892760858 15.3271969585529 5.89398797175478 17.8586727094456 5.95961701590099 20.2985088549276 6.02524606004719 22.6363447983256 6.09087510419339 24.860826197611 6.15650414833959 26.9596114575785 6.22213319248579 28.9194309025137 6.28776223663199 30.726216658378 6.35339128077819 32.3653215950237 6.41902032492439 33.8218427086678 6.48464936907059 35.0810565526955 6.55027841321679 36.128960740211 6.61590745736299 36.9528964984268 6.68153650150919 37.5422053476319 6.7471655456554 37.8888533520658 6.8127945898016 37.9879458184651 6.8784236339478 37.8380598058405 6.944052678094 37.4413435052612 7.0096817222402 36.8033666797416 7.0753107663864 35.9327458561739 7.1409398105326 34.8406009478366 7.2065688546788 33.5399182685516 7.272197898825 32.044895968276 7.3378269429712 30.3703349411268 7.4034559871174 28.5311176418378 7.4690850312636 26.5417955854654 7.53471407540981 24.4162882493214 7.60034311955601 22.1676837284952 7.66597216370221 19.8081248145519 7.73160120784841 17.3487619954732 7.79723025199461 14.799755683505 7.86285929614081 12.1703123912554 7.92848834028701 9.46874258576449 7.99411738443321 6.70253092636695 8.05974642857941 3.87841220951392 8.12537547272561 1.00244848236149 8.19100451687181 -1.91989555243233 8.25663356101802 -4.8836804747898 8.32226260516422 -7.88442415278971 8.38789164931042 -10.9180378798145 8.45352069345662 -13.9807662177345 8.51914973760282 -17.0691297328311 8.58477878174902 -20.1798694627338 8.65040782589522 -23.309891582493 8.71603687004142 -26.4562102786709 8.78166591418762 -29.6158862071544 8.84729495833382 -32.7859569838117 8.91292400248002 -35.9633547555351 8.97855304662622 -39.1448037326031 9.04418209077243 -42.3266871524279 9.10981113491863 -45.504867677159 9.17544017906483 -48.6744362952325 9.24106922321103 -51.8293499125257 9.30669826735723 -54.9618925208406 9.37232731150343 -58.0618510532906 9.43795635564963 -61.1152203052896 9.50358539979583 -64.1021167886502 9.56921444394203 -66.993352522025 9.63484348808823 -69.7447715242866 9.70047253223443 -72.2881168263265 9.76610157638063 -74.5177759572191 9.83173062052684 -76.2776427744673 9.89735966467304 -77.3680595377555 9.96298870881924 -77.6100037895131 10.0286177529654 -76.9559806268968 10.0942467971116 -75.5274323164154 10.1598758412578 -73.5256235822502 10.225504885404 -71.1335423855973 10.2911339295502 -68.4826401464916
130.208576007499 -121.905448578882 126.122862733069 -116.117532309917 123.096088756105 -111.370733897056 120.847647654676 -107.381186011555 119.181244266273 -103.94919908428 117.959229801265 -100.933573285085 117.083995584266 -98.2329674426084 116.485355348124 -95.7732545652395 116.112141426378 -93.499089029841 115.926598629183 -91.3682689298362 115.900606850559 -89.3479243258107 116.013101620634 -87.4119003018251 116.248287458048 -85.5389293754348 116.594383262622 -83.7113322121363 117.042730341744 -81.9140769876774 117.587151590014 -80.134085739046 118.223487381287 -78.3597132016406 118.949257674755 -76.5803477546609 119.763415481785 -74.786100014617 120.666167149781 -72.9675553244674 121.658841730389 -71.115573765808 122.743796156477 -69.2211265771766 123.924345768493 -67.2751617804985 125.204711346133 -65.2684949313595 126.589974481174 -63.191723589 128.086033026003 -61.035166617737 129.699547558807 -58.7888319786563 131.437868384608 -56.4424193786682 133.308930624879 -53.9853670632536 135.321102598236 -51.4069551014493 137.482970277508 -48.6964804653791 139.803038699234 -45.8435215234458 142.289330750338 -42.8383103293911 144.948866187195 -39.6722288992279 147.787010956759 -36.3384386397515 150.806701050387 -32.832638076474 154.007568033343 -29.153921230077 157.385025340219 -25.3056771324334 160.929412516458 -21.2964329535281 164.625331128856 -17.1405067232032 168.451328168929 -12.858313573932 172.380074737661 -8.4761774999887 176.379137290568 -4.02555116377669 180.412345289918 0.458360268572769 184.441640501448 4.93845665709357 188.429183452373 9.37785744904499 192.339429620212 13.7419859855187 196.140895478141 18.0003438246488 199.807409667065 22.1277705018441 203.318759311593 26.1050989039982 206.66075731849 29.9192322707698 209.824841968919 33.5627543292131 212.807361400678 37.0332253803366 215.608696225676 40.3323178406948 218.232347227336 43.4649184871699 220.684077333592 46.4382869620815 222.971159775782 49.2613228841799 225.101754136965 51.9439637979844 227.084410962725 54.4967152828508 228.927693036919 56.9303021251667 230.639895360615 59.2554235888955 232.228844170851 61.4825943713189 233.701756184171 63.6220540083017 235.065141326123 65.6837300004841 236.324734648094 67.6772429400574 237.48544542396 69.6119449640399 238.551313283835 71.4969857370273 239.525462541078 73.3414028346811 240.410046526307 75.1542359271978 241.206173692698 76.94466669334 241.913806401305 78.7221891397092 242.531621460685 80.4968182437007 243.056818386737 82.279348980739 243.484856479978 84.0816834111962 243.809094402788 85.9172514467257 244.020294718086 87.8015625059827 244.105938799072 89.7529425733172 244.049271369352 91.7935374912173 243.827953385085 93.9507039713794 243.412138430893 96.2589734129004 242.76168748309 98.7628739685644 241.822078476583 101.521054731671 240.518320819858 104.612402208908 238.745821628412 108.145202611625 236.356693001996 112.270860930855 233.139723418691 117.203953999185 228.793455861826 123.249175184354 222.899150982562 130.828392657524 214.927154157084 140.474302609193 204.378164239888 152.689179830701 191.225677793991 167.502452089462 176.539550313127 -176.152843016668 162.353586048841 -160.307630144594 150.360913794037 -146.652154392152 141.060580576663 -135.682557350517 134.127576437556 -127.070884007931 129.015397619862 -120.267627640405
2004/05/09 18:13:09 23:07:06 -51:59:48 RJ 97
5.50861904362296 13.2108783930743 5.57424809202831 14.3097933784889 5.63987714043366 15.2825480922759 5.70550618883901 16.1235085428608 5.77113523724437 16.8275811688804 5.83676428564972 17.3903329290536 5.90239333405507 17.8081044918121 5.96802238246042 18.0781098631741 6.03365143086577 18.1985160479054 6.09928047927113 18.1684972771591 6.16490952767648 17.9882599123297 6.23053857608183 17.6590361938364 6.29616762448718 17.1830472971292 6.36179667289253 16.5634383921303 6.42742572129789 15.8041902942228 6.49305476970324 14.9100136293538 6.55868381810859 13.8862321008871 6.62431286651394 12.7386614429716 6.6899419149193 11.4734900729826 6.75557096332465 10.0971664759021 6.82120001173 8.61629715149267 6.88682906013535 7.03755770408246 6.9524581085407 5.36761849228281 7.01808715694606 3.61308527329986 7.08371620535141 1.78045451934628 7.14934525375676 -0.123917439856693 7.21497430216211 -2.09383260247142 7.28060335056746 -4.1232580872349 7.34623239897282 -6.20632279421444 7.41186144737817 -8.33730506903577 7.47749049578352 -10.5106123061245 7.54311954418887 -12.7207531774273 7.60874859259422 -14.9623028929955 7.67437764099958 -17.2298616006532 7.74000668940493 -19.5180057209004 7.80563573781028 -21.8212316945794 7.87126478621563 -24.1338912987279 7.93689383462099 -26.4501173675393 8.00252288302634 -28.7637384552473 8.06815193143169 -31.0681807255202 8.13378097983704 -33.3563552022117 8.19941002824239 -35.6205285645305 8.26503907664775 -37.8521760748277 8.3306681250531 -40.0418162424906 8.39629717345845 -42.17882883772 8.4619262218638 -44.251261425216 8.52755527026915 -46.2456354134911 8.59318431867451 -48.1467715275084 8.65881336707986 -49.9376672503331 8.72444241548521 -51.5994749844465 8.79007146389056 -53.1116473673548 8.85570051229591 -54.452329734381 8.92132956070127 -55.5990786869526 8.98695860910662 -56.5299555529607 9.05258765751197 -57.2249714275417 9.11821670591732 -57.6677478358279 9.18384575432268 -57.8471333892966 9.24947480272803 -57.7584406957995 9.31510385113338 -57.4039998669833 9.38073289953873 -56.7928808650735 9.44636194794408 -55.939859690011 9.51199099634944 -54.8638917452489 9.57762004475479 -53.5864315406827 9.64324909316014 -52.1298929059054 9.70887814156549 -50.5164288090777 9.77450718997084 -48.7670885219197 9.8401362383762 -46.9013227104044 9.90576528678155 -44.9367636862505 9.9713943351869 -42.8891988709221 10.0370233835923 -40.772665564008 10.1026524319976 -38.5996122461112 10.168281480403 -36.3810887268411 10.2339105288083 -34.1269413809253 10.2995395772137 -31.8459998414103 10.365168625619 -29.5462482998654 10.4307976740244 -27.234978800384 10.4964267224297 -24.9189263777739 10.5620557708351 -22.6043871920751 10.6276848192404 -20.2973213893637 10.6933138676458 -18.0034425661946 10.7589429160511 -15.7282956190777 10.8245719644565 -13.4773245351152 10.8902010128618 -11.2559313917526 10.9558300612672 -9.06952751911084 11.0214591096725 -6.9235774564198 11.0870881580779 -4.82363601374057 11.1527172064832 -2.77537843611968 11.2183462548886 -0.784623363502032 11.2839753032939 1.14265200788346 11.3496043516993 3.00030541823719 11.4152334001046 4.78203276559174 11.48086244851 6.48138228439077 11.5464914969154 8.09178096852979 11.6121205453207 9.60657445392176 11.6777495937261 11.019081333698 11.7433786421314 12.3226624316944 11.8090076905368 13.5108048825632
160.857356438563 -30.0500631698819 162.956620238521 -26.5882575152247 165.13083993401 -23.0705237467075 167.372740633727 -19.5012654891945 169.673901435135 -15.886152550327 172.024808573308 -12.2320460429393 174.414964306645 -8.54686918318997 176.833050548576 -4.83942445151693 179.267141304711 -1.11916277698709 181.704953072597 2.60408467442989 184.134118265124 6.32039741897939 186.542464123161 10.0200619907965 188.918278952352 13.6938557711108 191.250548968499 17.3333018410286 193.529152301939 20.9308816627305 195.745001219806 24.4801969166235 197.890128635039 27.9760768520174 199.957719745043 31.4146323093156 201.942093593051 34.793261555225 203.838642118924 38.1106158873869 205.643735761764 41.3665345105197 207.354605001951 44.5619585714348 208.969206628196 47.6988337096531 210.486082286063 50.7800093353621 211.904215300491 53.8091413809564 213.222890109086 56.7906037308288 214.441557070741 59.7294120928632 215.559704032542 62.6311628419524 216.57673489704 65.5019883900193 217.491854539912 68.348529923805 218.303958767404 71.177927877084 219.011527544745 73.9978302264019 219.612519441473 76.8164185673981 220.104265106599 79.6424518844505 220.483357601178 82.4853279048517 220.745537597612 85.355161853397 220.885571856397 88.2628821968679 220.897124110739 91.2203424587128 220.772618689678 94.2404472065999 220.503099141302 97.3372886036319 220.078087148136 100.526287089257 219.485451678759 103.824325291025 218.7113052946 107.249857459374 217.739954701905 110.822966691133 216.553946949336 114.565328020847 215.134271829359 118.500016406293 213.460804820215 122.651074943896 211.513100789994 127.042732838115 209.271669619085 131.698141774303 206.71986625714 136.637498033998 203.846484797318 141.875460664965 200.649025767088 147.417896434111 197.137379010706 153.258209141396 193.33733693298 159.373838498915 189.293009065375 165.723857603911 185.06703768117 172.248769281277 180.737752468519 178.873363298219 176.393162538326 -174.487263807619 172.122758638339 -167.919499135402 168.00896489075 -161.502653356157 164.12022590079 -155.302035997932 160.507049459883 -149.364986281065 157.201253638781 -143.720108279045 154.217757925713 -138.37905004592 151.557842624999 -133.339750920013 149.212858050158 -128.590138543506 147.167676227661 -124.111568334418 145.40352557538 -119.881645914234 143.900113632768 -115.876337631427 142.63709704186 -112.071428473109 141.595023245475 -108.443451961592 140.755878726312 -104.97022703592 140.103362202085 -101.631120566178 139.622975883105 -98.4071289081525 139.302002953597 -95.2808470434785 139.12941861023 -92.2363731215828 139.09576609005 -89.2591804356902 139.193017648853 -86.3359775314446 139.414432497602 -83.4545693518546 139.754418352992 -80.6037271508772 140.208399745199 -77.7730715947429 140.772693970504 -74.9529714301225 141.444394161772 -72.1344589064005 142.221258089243 -69.3091624955508 143.101600808565 -66.4692571478418 144.084189024686 -63.6074322073452 145.168134972582 -60.7168770808438 146.352787699233 -57.7912847248541 147.637619863264 -54.824872919961 149.022108565215 -51.812423079541 150.505609309933 -48.749335937413 152.087223013385 -45.6317028304021 153.765657022048 -42.4563904061268 155.539082416206 -39.2211354374097 157.404991384938 -35.9246450451639 159.360060104103 -32.5666961078428 161.400024167737 -29.1482261200874
2032/06/07 09:42:42 17:48:35 +18:52:06 RB 97
3.77800547190502 41.5055848415153 3.84363451896009 45.0118644332534 3.90926356601516 48.5245799315194 3.97489261307022 52.0429879869497 4.04052166012529 55.5663828983637 4.10615070718036 59.0940808966824 4.17177975423543 62.6253985945378 4.23740880129049 66.1596185512375 4.30303784834556 69.6959252793746 4.36866689540063 73.2332670927932 4.4342959424557 76.770004136775 4.49992498951077 80.3027969323582 4.56555403656583 83.8217089156482 4.6311830836209 87.2675902852179 4.69681213067597 88.6090804427428 4.76244117773104 85.3546065690745 4.8280702247861 81.8518818337246 4.89369927184117 78.3231090349675 4.95932831889624 74.7875065286041 5.02495736595131 71.2501696163619 5.09058641300638 67.7132560260962 5.15621546006144 64.1780092811676 5.22184450711651 60.6453241716033 5.28747355417158 57.1159505204005 5.35310260122665 53.5905812500987 5.41873164828171 50.0698967743903 5.48436069533678 46.5545909805143 5.54998974239185 43.0453889527997 5.61561878944692 39.54306099981 5.68124783650199 36.0484352646007 5.74687688355705 32.5624101961739 5.81250593061212 29.0859677042483 5.87813497766719 25.6201876164479 5.94376402472226 22.1662639824914 6.00939307177732 18.7255237684067 6.07502211883239 15.2994485301248 6.14065116588746 11.8896997392602 6.20628021294253 8.49814855094502 6.2719092599976 5.12691095454456 6.33753830705266 1.77838943517491 6.40316735410773 -1.5446775000575 6.4687964011628 -4.83915631056567 6.53442544821787 -8.10144799946757 6.60005449527293 -11.3274040178283 6.665683542328 -14.5122258913848 6.73131258938307 -17.6503456616241 6.79694163643814 -20.7352840569304 6.86257068349321 -23.7594834163415 6.92819973054827 -26.7141130406798 6.99382877760334 -29.588846267111 7.05945782465841 -32.3716118051002 7.12508687171348 -35.0483276751713 7.19071591876854 -37.6026356925161 7.25634496582361 -40.0156692012128 7.32197401287868 -42.2659076397902 7.38760305993375 -44.3291977928002 7.45323210698882 -46.1790486812598 7.51886115404388 -47.7873236487865 7.58449020109895 -49.1254394858986 7.65011924815402 -50.1661136666907 7.71574829520909 -50.8855616085364 7.78137734226415 -51.2658551017483 7.84700638931922 -51.2969824581957 7.91263543637429 -50.9781070524866 7.97826448342936 -50.3176760398763 8.04389353048443 -49.3323478197068 8.1095225775395 -48.0450389819593 8.17515162459456 -46.4825817297535 8.24078067164963 -44.6734712715938 8.3064097187047 -42.6460266133107 8.37203876575977 -40.4270942196787 8.43766781281484 -38.0412729442939 8.50329685986991 -35.5105578790283 8.56892590692498 -32.8542790957667 8.63455495398004 -30.0892241632441 8.70018400103511 -27.2298596214084 8.76581304809018 -24.2885935037139 8.83144209514525 -21.2760429577959 8.89707114220032 -18.2012868192027 8.96270018925539 -15.072093410933 9.02832923631046 -11.8951202206408 9.09395828336553 -8.67608572582391 9.15958733042059 -5.41991549077045 9.22521637747566 -2.13086544654033 9.29084542453073 1.18737455418472 9.3564744715858 4.5315939553505 9.42210351864087 7.89898882576056 9.48773256569594 11.2871009744657 9.55336161275101 14.6937668695239 9.61899065980607 18.1170746881432 9.68461970686114 21.5553281000037 9.75024875391621 25.0070156186822 9.81587780097128 28.470784549904 9.88150684802635 31.9454187225124 9.94713589508142 35.4298193110181 10.0127649421365 38.9229881481229 10.0783939891916 42.4240129790278
81.9735971042843 -79.8416255875085 82.8082585328714 -80.4846931693224 83.6397172240771 -81.093941108245 84.4742297678065 -81.6670402840368 85.3196772426009 -82.1997591291406 86.1865931330513 -82.6849722412509 87.0899723057778 -83.1108840404793 88.0526745026953 -83.457654608383 89.1123681405639 -83.6904824386597 90.3372379811756 -83.743920989444 91.8668138169773 -83.4810722932372 94.041681510659 -82.5638966429451 97.9735362208657 -79.8831788435071 110.39470602577 -68.7090231111473 225.820978443253 45.4719556058819 258.722293457957 77.1273072377339 264.581544628902 81.7375304599621 267.291919801635 83.1933960966169 269.041070953124 83.6800993417016 270.375616146796 83.7417435235631 271.496698589655 83.5768801388623 272.496096064197 83.2745981451642 273.422071803105 82.8803532715923 274.302802681891 82.4193728822478 275.156176477483 81.9064277599124 275.994396430667 81.3504086184329 276.82634809028 80.7566618326798 277.658907087722 80.1282624742544 278.497707483539 79.4667436492047 279.347618865302 78.7725295706842 280.213058809492 78.0451981144216 281.098209033365 77.28364017957 282.007173987615 76.4861534419382 282.944104907525 75.650492169805 283.913303635951 74.7738858611736 284.919315559996 73.8530342618675 285.967018091944 72.8840831700525 287.061709385934 71.8625834501045 288.209200920248 70.7834343974149 289.415916896757 69.6408117648313 290.689002919221 68.4280802586766 292.036445957226 67.1376901019621 293.467207030635 65.7610573806676 294.99136717741 64.2884284468109 296.620285845971 62.7087298528342 298.366768525629 61.0094074621749 300.24523668223 59.1762620138363 302.271887177705 57.1932942244677 304.464819344034 55.0425814586221 306.844094530108 52.7042213044756 309.431673924017 50.1563963740833 312.251154829441 47.3756402452592 315.327193871939 44.3374161452801 318.684471977197 41.0171535951703 322.346031712019 37.3919124728392 326.330825758049 33.4428357650239 330.650393697859 29.1584738611903 335.304780085226 24.5388674447969 340.278151143041 19.5999316613424 345.535022200691 14.3772294708151 351.018402151217 8.92782792109764 356.651188021024 3.32890426438483 2.34151364496437 -2.32760206278161 7.99149973330777 -7.94373887000272 13.5075019351009 -13.4257886467166 18.8092972805384 -18.6934543701443 23.8361179893212 -23.6858926627209 28.5487250861047 -28.36378672785 32.9280401549788 -32.7079773827368 36.9716097903155 -36.7159267310738 40.6892505240129 -40.3973622391576 44.0988679970407 -43.7700949643968 47.2229831446479 -46.8565446833721 50.0861283825022 -49.6811348803519 52.713055794125 -52.2684995649161 55.1276035657416 -54.6423481262787 57.3520494586175 -56.8248169671004 59.4068006977759 -58.8361572371855 61.3103027257274 -60.6946410426773 63.0790817679023 -62.416600985588 64.7278629647907 -64.0165446721662 66.2697259673758 -65.5073059377075 67.7162741543074 -66.9002087541151 69.077803312272 -68.2052294116661 70.3634619835629 -69.4311488506353 71.5813997513686 -70.5856909777598 72.7389022697404 -71.6756451906443 73.8425133976791 -72.7069726779307 74.8981457456074 -73.6848967170005 75.9111815527982 -74.6139773767299 76.8865662784452 -75.498170876086 77.8288977620137 -76.3408733989052 78.7425144410146 -77.1449483985569 79.6315870903884 -77.9127352385089 80.5002201381385 -78.646035182768 81.3525712654125 -79.3460678526665 82.1930025133846 -80.0133865211204
2007/10/18 23:06:42 08:59:11 +25:24:52 RB 97
3.28422828981448 41.4698590467815 3.34985733540421 38.0829204028977 3.41548638099394 34.7043702200544 3.48111542658367 31.3358198370055 3.5467444721734 27.9789006067679 3.61237351776314 24.6353050613066 3.67800256335287 21.3068232988214 3.7436316089426 17.9953774332741 3.80926065453233 14.7030561594601 3.87488970012206 11.4321510443526 3.9405187457118 8.18519591972846 4.00614779130153 4.96501064330881 4.07177683689126 1.77475046697471 4.13740588248099 -1.38203773337644 4.20303492807072 -4.50135108103439 4.26866397366046 -7.57865687707582 4.33429301925019 -10.608808086256 4.39992206483992 -13.5859470558352 4.46555111042965 -16.5033967818635 4.53118015601939 -19.3535397729163 4.59680920160912 -22.1276858277623 4.66243824719885 -24.815932087723 4.72806729278858 -27.4070218530638 4.79369633837832 -29.8882132004376 4.85932538396805 -32.2451746789797 4.92495442955778 -34.4619333205908 4.99058347514751 -36.5209093196797 5.05621252073725 -38.4030804014184 5.12184156632698 -40.0883238952787 5.18747061191671 -41.5559808164529 5.25309965750644 -42.7856676389884 5.31872870309618 -43.7583229896866 5.38435774868591 -44.4574188810282 5.44998679427564 -44.8702003733635 5.51561583986537 -44.9887662904564 5.5812448854551 -44.8107940593992 5.64687393104484 -44.3397609368546 5.71250297663457 -43.584614182811 5.7781320222243 -42.5589605137603 5.84376106781403 -41.2799372101814 5.90939011340377 -39.7669643418537 5.9750191589935 -38.0405577080697 6.04064820458323 -36.1213257093588 6.10627725017296 -34.0292077896585 6.1719062957627 -31.7829578227415 6.23753534135243 -29.3998415422261 6.30316438694216 -26.8955019740931 6.36879343253189 -24.2839453460504 6.43442247812163 -21.577606028757 6.50005152371136 -18.7874580186878 6.56568056930109 -15.9231494467428 6.63130961489082 -12.9931442299055 6.69693866048055 -10.0048608704291 6.76256770607029 -6.96480264409963 6.82819675166002 -3.87867630338894 6.89382579724975 -0.751498286654591 6.95945484283948 2.41231143322745 7.02508388842922 5.6088470744042 7.09071293401895 8.834645222713 7.15634197960868 12.0866221938103 7.22197102519841 15.3620192822242 7.28760007078815 18.6583546178914 7.35322911637788 21.9733803809651 7.41885816196761 25.3050441351518 7.48448720755734 28.6514529983455 7.55011625314708 32.0108392404823 7.61574529873681 35.3815256289923 7.68137434432654 38.7618883442272 7.74700338991627 42.1503144069974 7.81263243550601 45.5451490163479 7.87826148109574 48.9446254527308 7.94389052668547 52.3467652004085 8.0095195722752 55.7492264902317 8.07514861786493 59.149060781565 8.14077766345467 62.5422977472539 8.2064067090444 65.9231927769096 8.27203575463413 69.2827636987443 8.33766480022386 72.6057004808066 8.4032938458136 75.8631528612916 8.46892289140333 78.9937485668802 8.53455193699306 81.8468469997125 8.60018098258279 84.0088046794969 8.66581002817253 84.5761119514546 8.73143907376226 83.1406684501152 8.79706811935199 80.5777476834741 8.86269716494172 77.5670921905657 8.92832621053146 74.3665990702158 8.99395525612119 71.0738668750121 9.05958430171092 67.731459143877 9.12521334730065 64.3606024678401 9.19084239289039 60.9731223075322 9.25647143848012 57.5762490364163 9.32210048406985 54.1747799581784 9.38772952965958 50.7721448974345 9.45335857524931 47.3709729385807 9.51898762083905 43.9734143186256 9.58461666642878 40.5813345046929
286.567767425416 85.1534496504257 286.992886688822 83.8248210878082 287.495257524381 82.5222869196284 288.068499440973 81.2340657905391 288.708458935676 79.9501273988502 289.412724284845 78.6616449252262 290.180296688679 77.3605962043013 291.011368191522 76.039463179215 291.907174672196 74.6909970516241 292.869903174911 73.3080275768428 293.902639719424 71.8833019797867 295.009348058345 70.4093436050236 296.194872533495 68.8783236351581 297.464959725312 67.2819416306891 298.826294253265 65.6113126826762 300.286543976645 63.8568609504338 301.854408953427 62.008221564127 303.539666736444 60.0541555877365 305.353203741069 57.9824862587382 307.307018273645 55.780069358443 309.414175129192 53.4328166108995 311.688684313883 50.9257986314974 314.145267584836 48.2434630123776 316.798966931515 45.3700128537331 319.664540830171 42.2899994602922 322.755590943565 38.9891861716558 326.083370320919 35.4557319822367 329.655253119605 31.6817146941761 333.472905505728 27.6649537533698 337.53029407919 23.410996263375 341.811795822725 18.9350020440789 346.290802887865 14.2631343087584 350.929288965409 9.4329891142858 355.678749107248 4.49265163194302 0.482693819319354 -0.501801657684808 5.28049926857991 -5.49015675911489 10.0120146237942 -10.412673031965 14.6220809281584 -15.2146055992582 19.0641454836675 -19.8498233388136 23.3024464265726 -24.2829972042085 27.3126524371372 -28.4902438900625 31.0811989787698 -32.4584663150494 34.6037573260446 -36.1838272699727 37.8832963892084 -39.6698163390143 40.9281069604183 -42.925279836466 43.7500252821898 -45.9626507977855 46.3629699353666 -48.7964931879161 48.7818168585583 -51.4423853392405 51.021584561078 -53.9161149261182 53.0968775742356 -56.2331338228001 55.0215307678146 -58.4082158469014 56.8084017724894 -60.4552650995817 58.4692675359366 -62.3872315243811 60.0147905789061 -64.216100007627 61.4545291150171 -65.9529281556578 62.7969722150303 -67.6079151876549 64.0495865527175 -69.1904901250112 65.2188651336256 -70.7094118468707 66.3103710216613 -72.1728769159954 67.3287706666555 -73.588633669599 68.2778521650796 -74.9641032096808 69.1605237421385 -76.3065098963727 69.97878691603 -77.6230260343445 70.7336770613227 -78.9209379838443 71.4251611243656 -80.2078443753304 72.0519774930997 -81.4919021300725 72.6113954860158 -82.782143661459 73.0988598420155 -84.0889007610259 73.5074658276612 -85.4243904300861 73.8271773889289 -86.803551066698 74.0436433068876 -88.2452748276996 74.1363632670869 -89.7742849699409 74.075763481949 -91.4240991786594 73.8183661915603 -93.2418951404487 73.2984659509864 -95.2968659755602 72.4130434841202 -97.6953352179417 70.9927239222898 -100.609824944747 68.7417175381637 -104.33913932235 65.1028798180602 -109.443327066831 58.9276432397692 -117.073770762066 47.6440748993793 -129.805149176556 25.9222974622551 -152.970035367922 351.640941294695 171.307537033318 322.290465732238 140.515377080027 306.235126526503 123.015086117232 297.784197542745 113.11326158934 292.98752297058 106.857031081006 290.086388529767 102.484904841299 288.268659829148 99.1818947078436 287.123019534205 96.5337308314477 286.422862311032 94.310734896022 286.034519711224 92.3760217024504 285.874764110487 90.6429779053926 285.889535471718 89.0539613165122 286.042562945966 87.5688923911025 286.308931254351 86.1587873995621 286.671269695841 84.8019106357418
2010/01/18 12:39:41 06:42:09 -51:09:05 RJ 97
2.13243738402556 16.5279373519529 2.19806642452443 15.6093601510815 2.26369546502331 14.5589302955072 2.32932450552218 13.382717435846 2.39495354602105 12.0871593296431 2.46058258651992 10.6789425725253 2.52621162701879 9.16489533753282 2.59184066751767 7.55189471172143 2.65746970801654 5.8467899439961 2.72309874851541 4.05634186195775 2.78872778901428 2.1871779211615 2.85435682951316 0.245761818931795 2.91998587001203 -1.76162368961081 2.9856149105109 -3.82888223430126 3.05124395100977 -5.95009894027928 3.11687299150864 -8.11952645382187 3.18250203200752 -10.3315628969599 3.24813107250639 -12.5807224825731 3.31376011300526 -14.8615992112841 3.37938915350413 -17.1688237433472 3.445018194003 -19.4970131982942 3.51064723450188 -21.8407132843585 3.57627627500075 -24.1943317996556 3.64190531549962 -26.5520621811567 3.70753435599849 -28.9077954160412 3.77316339649737 -31.2550182976474 3.83879243699624 -33.5866957547596 3.90442147749511 -35.8951349006074 3.97005051799398 -38.1718286972653 4.03567955849285 -40.4072779782016 4.10130859899173 -42.5907924369559 4.1669376394906 -44.7102747018677 4.23256667998947 -46.7519976462594 4.29819572048834 -48.7003946969693 4.36382476098722 -50.5378971101757 4.42945380148609 -52.2448712610309 4.49508284198496 -53.7997311456584 4.56071188248383 -55.1793205514091 4.6263409229827 -56.3596634112524 4.69196996348158 -57.3171511702475 4.75759900398045 -58.0301542269045 4.82322804447932 -58.4809085471862 4.88885708497819 -58.6573726393078 4.95448612547707 -58.5546492747727 5.02011516597594 -58.1756021043446 5.08574420647481 -57.5304923759389 5.15137324697368 -56.6357397460735 5.21700228747255 -55.5121395656783 5.28263132797143 -54.1829485977761 5.3482603684703 -52.6721797274866 5.41388940896917 -51.0032963567005 5.47951844946804 -49.1983507538958 5.54514748996692 -47.2775135619538 5.61077653046579 -45.25889967381 5.67640557096466 -43.1585927960193 5.74203461146353 -40.9907877439297 5.8076636519624 -38.7679917062726 5.87329269246128 -36.5012459120199 5.93892173296015 -34.200344655535 6.00455077345902 -31.8740393805114 6.07017981395789 -29.5302223930139 6.13580885445676 -27.1760888576091 6.20143789495564 -24.818277960858 6.26706693545451 -22.4629951944962 6.33269597595338 -20.1161180868944 6.39832501645225 -17.7832876976305 6.46395405695113 -15.469987970588 6.52958309745 -13.1816147232828 6.59521213794887 -10.9235356931861 6.66084117844774 -8.70114269560784 6.72647021894661 -6.51989658558963 6.79209925944549 -4.38536536291696 6.85772829994436 -2.30325541609849 6.92335734044323 -0.279435569426605 6.9889863809421 1.68004671864847 7.05461542144098 3.56895794827365 7.12024446193985 5.38088115378738 7.18587350243872 7.10922983737303 7.25150254293759 8.74727472080843 7.31713158343646 10.2881847209834 7.38276062393534 11.7250833165188 7.44838966443421 13.0511210076188 7.51401870493308 14.2595638478148 7.57964774543195 15.3438970371452 7.64527678593083 16.2979413421623 7.7109058264297 17.1159787295763 7.77653486692857 17.7928822034464 7.84216390742744 18.3242436059926 7.90779294792631 18.706492292332 7.97342198842519 18.936997323258 8.03905102892406 19.0141462848975 8.10468006942293 18.9373950858786 8.1703091099218 18.7072850180827 8.23593815042067 18.3254257833416 8.30156719091954 17.7944457770999 8.36719623141841 17.1179133372243 8.43282527191729 16.3002346031507
193.915837119003 21.1479419673921 196.175708892926 24.7029550982854 198.361999574276 28.2014085759965 200.467842969549 31.6393090831987 202.487581592416 35.0140254261911 204.416695034519 38.3242435036332 206.251693499828 41.5698892262872 207.989986973798 44.7520303780344 209.629739653803 47.8727676407808 211.169717769006 50.935123599115 212.609137099521 53.9429368304013 213.947514634516 56.9007664424823 215.184527072232 59.8138108318372 216.319877368096 62.687843107975 217.353169322682 65.5291646081313 218.283789266473 68.3445772064725 219.110793214515 71.1413746710176 219.832797391386 73.9273530902905 220.447869726981 76.7108403235299 220.953419768191 79.5007444622693 221.346084430116 82.3066213613358 221.62160713792 85.1387613275192 221.774708236743 88.008294947595 221.798945171016 90.9273176614311 221.686562023952 93.9090318494791 221.42832982859 96.9679036263235 221.013382016483 100.119828801396 220.429054047263 103.382297991678 219.660743484406 106.774543825071 218.691817630524 110.317642476795 217.503611586655 114.034526158699 216.075581456159 117.949841433393 214.385705787835 122.08955994092 212.41126135577 126.480215196893 210.130130078072 131.147608476905 207.522806237825 136.114814490088 204.575237757492 141.399352996481 201.282509087433 147.009518745217 197.653112691664 152.940122713401 193.713154538926 159.168299220521 189.509388633491 165.650483882421 185.109713648712 172.321929352141 180.600014111415 179.099876418784 176.077166872809 -174.109440245788 171.63940622728 -167.401145024137 167.376347965357 -160.861733228862 163.361123643897 -154.561193167704 159.64617113889 -148.548784433171 156.262846113123 -142.852637022096 153.223919163186 -137.482236028511 150.527595529459 -132.432428780497 148.161851780143 -127.687748830169 146.108313354829 -123.226280694849 144.345325899657 -119.022718312774 142.850173420876 -115.050570295131 141.600554412626 -111.283623215273 140.575481956371 -107.696829054964 139.755770592675 -104.266779777313 139.124244872425 -100.971904159839 138.665771133847 -97.7924887305951 138.367184121941 -94.7105948060578 138.217156466642 -91.7099201321435 138.206041770962 -88.775636487748 138.32571004308 -85.8942227382812 138.569386139861 -83.0533049296356 138.931496629118 -80.2415099471516 139.407527133339 -77.4483361378312 139.993890142664 -74.6640424577879 140.687802032684 -71.879556706173 141.487167295189 -69.0864029306103 142.390467596952 -66.2766479261395 143.396653108111 -63.4428667542245 144.505033526418 -60.5781272760693 145.71516634346 -57.6759937447741 147.02674015847 -54.7305494657985 148.439451268836 -51.7364383528728 149.952872390366 -48.6889248188858 151.566313225251 -45.5839707966551 153.278673734425 -42.4183277445038 155.088292395024 -39.1896402433816 156.992793404197 -35.8965562648224 158.988938640039 -32.5388374715194 161.072492045215 -29.1174611702985 163.23810571222 -25.6347040196661 165.479238001733 -22.0941966223443 167.788114161825 -18.5009380645369 170.155738810518 -14.861260625175 172.571967091736 -11.18273747712 175.025637324258 -7.47403023022651 177.504762829503 -3.7446783297991 179.996774938073 -0.00483803278264134 182.488803753873 3.73501587821998 184.967978987377 7.46440820488832 187.421730815338 11.1731814692538 189.838070721105 14.8517942749228 192.205834618276 18.49158244639 194.514874823629 22.084969773416
2049/03/19 09:05:05 05:09:05 +46:05:42 RB 97
2.23521315395264 42.2415795080509 2.30084218960031 39.7274298287798 2.36647122524798 37.1796590044893 2.43210026089566 34.6072959584138 2.49772929654333 32.0182568236726 2.563358332191 29.4195886782207 2.62898736783868 26.8176696372679 2.69461640348635 24.2183736200001 2.76024543913402 21.6272069617984 2.82587447478169 19.049422799677 2.89150351042937 16.490117998084 2.95713254607704 13.954316366862 3.02276158172471 11.4470410582071 3.08839061737238 8.97337829497951 3.15401965302006 6.53853394991652 3.21964868866773 4.14788393427843 3.2852777243154 1.80701883794117 3.35090675996308 -0.478217231832397 3.41653579561075 -2.70169415551967 3.48216483125842 -4.85697068680909 3.54779386690609 -6.93727825167679 3.61342290255377 -8.93551567967046 3.67905193820144 -10.8442573283832 3.74468097384911 -12.6557774093659 3.81031000949678 -14.3620935008177 3.87593904514446 -15.9550321425395 3.94156808079213 -17.4263189406872 4.0071971164398 -18.7676946544739 4.07282615208748 -19.9710572137017 4.13845518773515 -21.0286275101354 4.20408422338282 -21.9331342093284 4.26971325903049 -22.6780099744442 4.33534229467817 -23.2575887595248 4.40097133032584 -23.6672917091588 4.46660036597351 -23.9037882138322 4.53222940162119 -23.9651192327016 4.59785843726886 -23.8507722854843 4.66348747291653 -23.5617013719848 4.7291165085642 -23.1002899917394 4.79474554421188 -22.4702606443294 4.86037457985955 -21.6765388574184 4.92600361550722 -20.7250832119067 4.99163265115489 -19.6226946022203 5.05726168680257 -18.3768180319721 5.12289072245024 -16.9953488640669 5.18851975809791 -15.4864530828213 5.25414879374559 -13.8584083070871 5.31977782939326 -12.1194694829243 5.38540686504093 -10.2777607185056 5.4510359006886 -8.34119278995288 5.51666493633628 -6.31740450036989 5.58229397198395 -4.21372527658514 5.64792300763162 -2.03715604637132 5.71355204327929 0.205634559053185 5.77918107892697 2.50830139467618 5.84481011457464 4.86480371809822 5.91043915022231 7.26937770745272 5.97606818586999 9.71650196433118 6.04169722151766 12.2008568884362 6.10732625716533 14.7172783184441 6.172955292813 17.2607053435945 6.23858432846068 19.8261216912486 6.30421336410835 22.4084895683372 6.36984239975602 25.0026742549659 6.4354714354037 27.6033570881321 6.50110047105137 30.2049337024582 6.56672950669904 32.8013934858427 6.63235854234671 35.3861751463901 6.69798757799439 37.9519920909178 6.76361661364206 40.4906200745899 6.82924564928973 42.9926385312208 6.8948746849374 45.4471166487391 6.96050372058508 47.8412366348876 7.02613275623275 50.1598516112475 7.09176179188042 52.384987424452 7.1573908275281 54.4953214487483 7.22301986317577 56.4657139536999 7.28864889882344 58.2669350741185 7.35427793447111 59.8658213689871 7.41990697011879 61.2261859075733 7.48553600576646 62.31082671946 7.55116504141413 63.0848154227717 7.6167940770618 63.5198028275664 7.68242311270948 63.5984273292366 7.74805214835715 63.3174324457884 7.81368118400482 62.6882875057199 7.8793102196525 61.7350521483892 7.94493925530017 60.4903506770964 8.01056829094784 58.9908710627502 8.07619732659551 57.2735643260737 8.14182636224319 55.3730811310238 8.20745539789086 53.3204214000132 8.27308443353853 51.1424930368595 8.3387134691862 48.8622322839248 8.40434250483388 46.4990096020172 8.46997154048155 44.0691405941369 8.53560057612922 41.5864002784074
315.168436420876 106.578548541707 314.280491614836 103.28110521239 313.60598976634 100.146605864596 313.123113402054 97.1484990542024 312.813580277569 94.2635233882571 312.662189297785 91.4712353405323 312.656402178864 88.7535736314233 312.785973027588 86.0944729395775 313.042627746907 83.4795296315997 313.419789974702 80.8957172232309 313.912347908658 78.3311472093074 314.516455543485 75.7748704064298 315.229361783944 73.2167142819635 316.049261169536 70.6471524675827 316.975160324033 68.0572035391833 318.006754612384 65.4383570590158 319.144309806235 62.7825257473544 320.388543829866 60.082023430906 321.7405039155 57.329569058358 323.201434802343 54.518317525805 324.772634053967 51.641918244887 326.455291253354 48.6946022221891 328.250308891857 45.6712977892001 330.158104331358 42.5677739096235 332.178394404994 39.3808080914528 334.309967094754 36.1083732927516 336.550448243023 32.7498348828688 338.896075214652 29.3061449178401 341.341493402715 25.7800171399067 343.879594799526 22.1760628869456 346.501419667411 18.5008663694175 349.196141705675 14.7629784792622 351.95115325069 10.9728122099105 354.752259660405 7.142430196179 357.583981559867 3.2852253917851 0.42995136659571 -0.584491814146789 3.27337851173439 -4.45197471890932 6.09754836318234 -8.30255492232946 8.88631505193616 -12.1221508829217 11.6245492809688 -15.897731738847 14.2985085576844 -19.617704087305 16.8961077237236 -23.2721998623382 19.4070799795746 -26.8532557952918 21.8230305079627 -30.3548868769795 24.1373944397371 -33.773065917399 26.345317234619 -37.1056276746559 28.4434783764188 -40.3521189110886 30.4298790860552 -43.5136156115608 32.3036123859541 -46.5925263194203 34.0646302414751 -49.5923970485439 35.7135184744295 -52.5177293351865 37.2512862835442 -55.3738192995045 38.6791738700482 -58.1666224535748 39.9984790026358 -60.9026465779534 41.2104013681166 -63.5888733055777 42.315902157573 -66.2327080276083 43.3155754074575 -68.8419572593941 44.2095270169549 -71.4248325583996 44.9972569739724 -73.989980363969 45.6775400392884 -76.5465376453934 46.2482998825152 -79.1042139343222 46.7064713807612 -81.6734011297262 47.0478454554153 -84.2653133537362 47.2668904436601 -86.8921600510015 47.3565436395374 -89.5673563770335 47.307966436864 -92.3057755579623 47.110256742509 -95.1240480422536 46.7501135157295 -98.0409113937962 46.211451331014 -101.077612096544 45.4749693068962 -104.258354211062 44.517691169426 -107.610777571475 43.3125157623661 -111.166425807892 41.8278562619848 -114.961125644882 40.0275104202458 -119.035134948506 37.871003389833 -123.432817833587 35.3147853913612 -128.201464489438 32.3148373084758 -133.388702598246 28.831379631612 -139.037804916826 24.8363417035868 -145.180236084097 20.3237479918629 -151.825281963047 15.3218831078338 -158.947899843969 9.9039787517215 -166.47804638738 4.19213677995207 -174.296769173987 358.349349920982 177.755799595047 352.558605963442 169.859532998801 346.994811938645 162.186219345762 341.799684841643 154.874438640254 337.067885222363 148.015681822634 332.846238698184 141.653557761002 329.142193673969 135.792236209548 325.935882721425 130.408494118795 323.191534023776 125.463111643232 320.866247746994 120.909632655803 318.91584235215 116.70019474378 317.298287163788 112.788944976896 315.975435611468 109.133755862072 314.913684002865 105.696866368419
2076/05/04 04:01:11 21:07:52 -09:59:16 RJ 97
1.70467163823783 -49.8243965755782 1.77030068182952 -53.3119755685374 1.8359297254212 -56.7747607623495 1.90155876901289 -60.2027365501945 1.96718781260458 -63.5810180895079 2.03281685619627 -66.886508101029 2.09844589978795 -70.0816700284306 2.16407494337964 -73.1025763914208 2.22970398697133 -75.8361233369552 2.29533303056301 -78.0818661018263 2.3609620741547 -79.5205314686224 2.42659111774639 -79.8040883941275 2.49222016133808 -78.8440312554947 2.55784920492976 -76.9114359451744 2.62347824852145 -74.3632536281085 2.68910729211314 -71.4520481793716 2.75473633570482 -68.3242653655167 2.82036537929651 -65.0620107676792 2.8859944228882 -61.7125849555349 2.95162346647988 -58.304486442563 3.01725251007157 -54.855712801977 3.08288155366326 -51.3781486367694 3.14851059725495 -47.8799736032441 3.21413964084663 -44.3670397782513 3.27976868443832 -40.8436920129143 3.34539772803001 -37.3132755219125 3.41102677162169 -33.7784614761554 3.47665581521338 -30.2414632818481 3.54228485880507 -26.7041854236195 3.60791390239676 -23.1683298478261 3.67354294598844 -19.6354752989868 3.73917198958013 -16.107139467066 3.80480103317182 -12.5848305119071 3.8704300767635 -9.07009256336812 3.93605912035519 -5.56454863248212 4.00168816394688 -2.06994371905974 4.06731720753857 1.41180939866818 4.13294625113025 4.87857924470325 4.19857529472194 8.3279569662591 4.26420433831363 11.7571868730577 4.32983338190531 15.1630825590642 4.395462425497 18.541924105887 4.46109146908869 21.8893307104365 4.52672051268037 25.2001015886018 4.59234955627206 28.4680161582408 4.65797859986375 31.6855823384133 4.72360764345544 34.8437195110028 4.78923668704712 37.9313608006893 4.85486573063881 40.9349590289274 4.9204947742305 43.8378844256251 4.98612381782218 46.6197145579554 5.05175286141387 49.2554459815711 5.11738190500556 51.7147151165055 5.18301094859725 53.9612175359761 5.24863999218893 55.9526674639137 5.31426903578062 57.6418157441416 5.37989807937231 58.9791377186125 5.44552712296399 59.9175961568259 5.51115616655568 60.4191507198004 5.57678521014737 60.4615095364852 5.64241425373906 60.0427223664813 5.70804329733074 59.181605138173 5.77367234092243 57.9137965285095 5.83930138451412 56.2852109297472 5.9049304281058 54.3453361104716 5.97055947169749 52.1420957734282 6.03618851528918 49.7187992753087 6.10181755888086 47.1128637760367 6.16744660247255 44.355704916467 6.23307564606424 41.4732495865104 6.29870468965593 38.4866967566106 6.36433373324761 35.4133129979048 6.4299627768393 32.2671602884016 6.49559182043099 29.0597185947002 6.56122086402267 25.8003990120392 6.62684990761436 22.4969579471109 6.69247895120605 19.1558277079053 6.75810799479774 15.7823790688104 6.82373703838942 12.3811296635022 6.88936608198111 8.95590978743426 6.9549951255728 5.50999498150152 7.02062416916448 2.04621285601126 7.08625321275617 -1.43296994138699 7.15188225634786 -4.92537592486024 7.21751129993955 -8.42905636260646 7.28314034353123 -11.9422391452062 7.34876938712292 -15.463282244281 7.41439843071461 -18.9906302962744 7.48002747430629 -22.522771850501 7.54565651789798 -26.0581945735582 7.61128556148967 -29.595335125691 7.67691460508135 -33.1325193722219 7.74254364867304 -36.6678868025468 7.80817269226473 -40.1992900337094 7.87380173585642 -43.7241552330618 7.9394307794481 -47.2392806502311 8.00505982303979 -50.740535250057
278.624929518614 70.6534496560178 280.64694049713 69.7010022523658 282.957810030173 68.4383858581978 285.659893569278 66.7658997179628 288.903575184482 64.5356537379876 292.91681414543 61.5220463721426 298.055276013557 57.3716556101857 304.883506708057 51.522085037508 314.278957541779 43.097957346325 327.423206785912 30.9197100720582 345.116478260971 14.1890933561291 5.82301188870135 -5.55618492601502 25.2584685901946 -24.0298488013247 40.4103966347977 -38.2175054599214 51.3201089479272 -48.1585074082089 59.1601599133139 -55.0234138079689 64.9642464341943 -59.8438755160224 69.42390214216 -63.309312448545 72.9755863548865 -65.8539845847496 75.8962439096331 -67.7525324204226 78.3661309314703 -69.1827815747753 80.5062404247525 -70.2631450338239 82.4003429866805 -71.0746378613096 84.1081788618879 -71.6740400504494 85.6735569936786 -72.1019625301691 87.1294679008333 -72.3879223937981 88.5014024698829 -72.5536185324144 89.8095651414831 -72.615095062688 91.0703890585713 -72.5841988045566 92.2976007582332 -72.4695768020581 93.5029887347587 -72.2773662844931 94.6969745775678 -72.0116734183582 95.8890514929675 -71.6749027822047 97.0881339804373 -71.2679778106509 98.3028491934132 -70.7904784135037 99.5417921257397 -70.2407126040428 100.813761478651 -69.6157324837123 102.127989819865 -68.9113002377099 103.494379788784 -68.1218062199918 104.923757202833 -67.2401383306012 106.428151670078 -66.2574994803789 108.021115430334 -65.1631679259404 109.718091281621 -63.9441937559014 111.536840055039 -62.5850242057458 113.497936234053 -61.0670516108352 115.625335298025 -59.3680822762946 117.947005259598 -57.461735242142 120.495592613896 -55.3168018628644 123.309051216438 -52.8966385845676 126.431088646175 -50.1587390811148 129.911160849801 -47.0547555300832 133.803554071386 -43.5314304919229 138.164830304796 -39.5331635367451 143.048632936068 -35.0072162664023 148.496745452153 -29.9126631176705 154.525796489756 -24.233694893694 161.110734549374 -17.9961518872538 168.169451385607 -11.2829077484803 175.556463732698 -4.24019458274632 183.073783210079 2.93326043523096 190.500538858598 10.0158512605747 197.631888303456 16.8019978712306 204.311354931214 23.1344771286962 210.445311826503 28.9189002372725 215.999288986992 34.1200132005036 220.983583280226 38.7472996097055 225.436501096781 42.8382155772082 229.410312910269 46.4441358650415 232.961531364241 49.6206226455128 236.145111871624 52.4216156649221 239.011497945576 54.8964657754009 241.605466864122 57.0887671329814 243.965998882699 59.0362109076385 246.126665827174 60.7709558876452 248.116240353168 62.320216709229 249.959362114529 63.7069052649915 251.677178715313 64.9502423023633 253.287925808588 66.0663024864174 254.807435772216 67.0684809725652 256.249577081542 67.967881834402 257.626632359552 68.7736340339547 258.949625373606 69.4931422142315 260.228607824078 70.1322792276543 261.472916682936 70.6955259369328 262.691412690126 71.1860618909498 263.892710776889 71.6058081431885 265.085413944358 71.9554207026701 266.278363781171 72.2342296843144 267.48092375356 72.4401147782383 268.7033162502 72.5693015648535 269.957042081098 72.6160544654057 271.255423308727 72.5722291080162 272.614329604967 72.4266269007552 274.053179355127 72.1640630311957 275.596357464875 71.7640079375975 277.275276538427 71.1985772290783 279.131452808291 70.4295000330074
2070/05/19 00:13:13 18:07:16 +16:48:05 RB 97
0.955999488326488 -37.4160521354709 1.02162852717707 -39.997207889459 1.08725756602766 -42.4317867675684 1.15288660487824 -44.6960607258378 1.21851564372883 -46.7631815907069 1.28414468257941 -48.6035290861621 1.34977372142999 -50.1855781043135 1.41540276028058 -51.4774359913728 1.48103179913116 -52.4490955932537 1.54666083798175 -53.0752373222455 1.61228987683233 -53.338132701265 1.67791891568292 -53.22998087148 1.7435479545335 -52.7540174671132 1.80917699338409 -51.9240486386514 1.87480603223467 -50.7625586444356 1.94043507108525 -49.2979555748426 2.00606410993584 -47.5616564660128 2.07169314878642 -45.585568948211 2.13732218763701 -43.4002505950647 2.20295122648759 -41.0337762986738 2.26858026533818 -38.5111943197409 2.33420930418876 -35.8544008987761 2.39983834303935 -33.0822745946897 2.46546738188993 -30.2109483990331 2.53109642074052 -27.2541373218501 2.5967254595911 -24.2234715418455 2.66235449844168 -21.1288081432137 2.72798353729227 -17.978509175595 2.79361257614285 -14.7796824693135 2.85924161499344 -11.5383863449192 2.92487065384402 -8.25980160391814 2.99049969269461 -4.94837501270951 3.05612873154519 -1.60793854899982 3.12175777039578 1.75819163831842 3.18738680924636 5.14713307478447 3.25301584809694 8.55636863099762 3.31864488694753 11.9836895080775 3.38427392579811 15.4271474406709 3.4499029646487 18.8850143662205 3.51553200349928 22.3557480869889 3.58116104234987 25.8379626618234 3.64679008120045 29.330402407013 3.71241912005104 32.8319184533978 3.77804815890162 36.3414467831048 3.8436771977522 39.8579865175513 3.90930623660279 43.3805768762891 3.97493527545337 46.9082705309181 4.04056431430396 50.4400997518632 4.10619335315454 53.9750291865203 4.17182239200513 57.5118839928955 4.23745143085571 61.0492312943361 4.3030804697063 64.585168748684 4.36870950855688 68.1169150682644 4.43433854740746 71.6399379525147 4.49996758625805 75.1458633927308 4.56559662510863 78.6166051917953 4.63122566395922 82.0037105617451 4.6968547028098 85.126534623079 4.76248374166039 86.9951740929404 4.82811278051097 85.5523179294985 4.89374181936156 82.516366945564 4.95937085821214 79.151861126801 5.02499989706272 75.6896085954489 5.09062893591331 72.187580872115 5.15625797476389 68.6665252112856 5.22188701361448 65.1357766350216 5.28751605246506 61.6002785510598 5.35314509131565 58.0630097461473 5.41877413016623 54.525975206061 5.48440316901682 50.9906637053739 5.5500322078674 47.4582808563987 5.61566124671798 43.9298781698198 5.68129028556857 40.4064303485516 5.74691932441915 36.8888854240822 5.81254836326974 33.3782001972262 5.87817740212032 29.8753677240371 5.94380644097091 26.3814407447321 6.00943547982149 22.8975534862189 6.07506451867208 19.4249434993524 6.14069355752266 15.9649747984018 6.20632259637325 12.519163393285 6.27195163522383 9.08920626445462 6.33758067407441 5.67701488534082 6.403209712925 2.28475452862724 6.46883875177558 -1.08510920733459 6.53446779062617 -4.42975494843204 6.60009682947675 -7.7459391321324 6.66572586832734 -11.0299161905707 6.73135490717792 -14.2773424251431 6.79698394602851 -17.483160169618 6.86261298487909 -20.6414583580304 6.92824202372967 -23.7453052602241 6.99387106258026 -26.7865491260687 7.05950010143084 -29.7555831497553 7.12512914028143 -32.6410731347742 7.19075817913201 -35.4296504768087 7.2563872179826 -38.1055810338181
311.550679409319 47.3522629505384 314.76589464943 44.2517151323195 318.290814724598 40.8391413544659 322.153021261205 37.0872646912659 326.375276364925 32.973608355086 330.971210976511 28.4848104893817 335.939959015598 23.6219928405985 341.260415920577 18.4065047662881 346.886499536483 12.8846646835169 352.745357543041 7.12955481424838 358.740367594016 1.23802309994242 4.75958382234565 -4.67776132955846 10.6882234122154 -10.5027927970591 16.4219438872419 -16.1325044425269 21.877275628082 -21.4831987890339 26.9969258019394 -26.4973496563471 31.7498059984238 -31.1436276783179 36.1272721369869 -35.4131379476771 40.1375935918947 -39.3138866878403 43.8003095184763 -42.8651348826947 47.1414334907503 -46.0925999132917 50.1898487227577 -49.0248474017215 52.974850385837 -51.6908296884224 55.5246198115521 -54.1183556288332 57.8653800205343 -56.3332410230816 60.0210122920003 -58.3589192377069 62.0129645417543 -60.216342593908 63.8603318921432 -61.9240546638758 65.5800298298215 -63.4983535659117 67.1870096023275 -64.9534955212734 68.6944856153961 -66.3019079370749 70.1141578289097 -67.5543943696052 71.4564205718593 -68.7203219486415 72.7305544098623 -69.8077867899665 73.944900857985 -70.8237557165891 75.1070216352295 -71.7741840046395 76.2238453488829 -72.6641093472435 77.3018053544702 -73.4977220689178 78.3469733424318 -74.2784109558043 79.3651942018933 -75.0087829026201 80.362229172008 -75.6906528027115 81.3439165689945 -76.3249974570165 82.3163630203948 -76.9118632541521 83.2861840611557 -77.4502110906541 84.2608227516446 -77.9376718679142 85.248991590665 -78.3701690002738 86.2613119645455 -78.7413351495269 87.3112776626228 -79.0415978962333 88.4167673090773 -79.2567105551466 89.6025245146631 -79.3653102333853 90.9044294224867 -79.3346802289045 92.3772893709323 -79.11298970864 94.1100664175003 -78.6140934939355 96.2583253566382 -77.6851088159767 99.1215228234143 -76.0291488828627 103.356695344704 -72.9916557791786 110.704580475616 -66.8342989017483 127.216560087695 -51.508050173877 174.643586429581 -5.26427570120786 228.28669834323 47.1957632482786 247.607344415089 65.3312164659161 255.775982571986 72.3102274879074 260.334022341656 75.6718587176419 263.353355630665 77.4856045852544 265.587878289537 78.5028947508809 267.372765791309 79.0563503972963 268.879090839536 79.3143811234706 270.203392316254 79.3707315137674 271.404521742588 79.2813031799496 272.520669752628 79.0811543517837 273.577971782388 78.7930757327676 274.595180045006 78.4322291335204 275.586353750655 78.0088001941242 276.562487832206 77.5295847031026 277.532543758559 76.9989712146117 278.504129360274 76.41956585717 279.4839656604 75.7925961166853 280.4782211729 75.1181726357187 281.492762448077 74.3954561424429 282.533351564998 73.6227582602281 283.605810643651 72.7975940002594 284.716167050878 71.9166969853948 285.870789048326 70.9760041492234 287.076519182974 69.9706138299456 288.340811160192 68.8947192898452 289.671874887174 67.7415184423987 291.078833555016 66.5030998267225 292.571895795284 65.170304624017 294.162544828811 63.7325648743971 295.863744762373 62.1777192792028 297.690161270179 60.4918105077504 299.658389069708 58.6588724751118 301.787170772232 56.6607236809363 304.09757932736 54.4767949076351 306.613117390923 52.0840383517823 309.359659254179 49.4569928673726 312.365122666675 46.5681182392897
