#!perl

use strict;

use Test::More tests => 1000 * 2;
use Test::Number::Delta within => 4;
use Astro::Coords;
use Astro::Telescope;
use DateTime;

my $i = 0;

while (<DATA>) {
  $i ++;
  chomp;
  my ($year, $month, $day, $hour, $minute, $second,
      $ra, $dec, $type, $telescope_name, $event, $horizon,
      $rise_ref, $set_ref) = split;

  my $c = new Astro::Coords(ra => $ra, dec => $dec,
                            type => $type, units => 'sexagesimal');
  my $dt = new DateTime(year => $year, month => $month, day => $day,
           hour => $hour, minute => $minute, second => $second, time_zone => 'UTC');

  $c->telescope(new Astro::Telescope($telescope_name));
  $c->datetime($dt);

  my $epoch = $dt->epoch();

  my $rise = $c->rise_time(event => $event, horizon => $horizon);
  my $set  =  $c->set_time(event => $event, horizon => $horizon);

  if ($rise_ref eq 'undef') {
    if (! defined $rise) {
      pass("Rise time $i not defined");
    }
    else {
      fail("Rise time $i should not have been defined");
    }
  }
  elsif (! defined $rise) {
    fail("Rise time $i not defined but should have been");
  }
  else {
    delta_ok($rise->epoch(), $rise_ref, "Rise time $i ($epoch $event)");
  }

  if ($set_ref eq 'undef') {
    if (! defined $set) {
      pass("Set time $i not defined");
    }
    else {
      fail("Set time $i should not have been defined");
    }
  }
  elsif (! defined $set) {
    fail("Set time $i not defined but should have been");
  }
  else {
    delta_ok($set->epoch(), $set_ref, "Set time $i ($epoch $event)");
  }
}


# Regression/timing test data generated with
# Astro::Coords version 0.13
# Astro::Coords::Equatorial version 0.06
__DATA__
2042 03 07 06 13 52 05:06:11 -29:13:41 B1950 TOLOLO1.5M 0 -0.174532777777778 2277815807 2277788816
1992 05 14 05 59 01 04:42:06 -38:11:11 B1950 BLOEMF 0 -0.174532777777778 705810942 705788666
2053 01 19 08 13 09 09:48:01 -37:11:16 B1950 CFHT 1 -0.104719666666667 2620968330 2620921452
2029 09 12 20 39 48 18:48:13 +32:13:39 J2000 TOLOLO4M 1 0.104719666666667 1884024171 1883966413
2066 11 12 11 52 48 21:13:07 -61:13:59 J2000 ESOSCHM 0 0 undef undef
2012 01 09 07 11 41 07:39:07 -61:01:59 J2000 DUNLAP74 1 0.0174532777777778 undef undef
1992 06 14 18 42 42 07:39:07 -18:48:07 B1950 SUBARU -1 0 708462115 708501802
2040 07 09 16 19 19 11:41:39 -29:52:24 J2000 SUGARGROVE 1 0.174532777777778 2225472660 2225493778
2070 01 04 03 09 35 21:24:41 +61:42:05 B1950 KPNO84 -1 -0.0174532777777778 undef undef
2008 04 04 11 13 01 10:01:48 +49:19:48 J2000 ESONTT -1 0.104719666666667 1207266058 1207282046
2044 03 07 12 48 35 17:13:42 +86:39:11 J2000 STEWARD90 0 0 undef undef
1986 12 25 15 01 52 02:09:39 -86:06:05 B1950 CSO 1 -0.0174532777777778 undef undef
2069 06 09 21 16 11 05:11:35 -01:16:48 B1950 LPO4.2 -1 0.0174532777777778 3137988192 3138030426
2070 08 07 06 01 41 00:05:42 -46:16:11 B1950 ESOSCHM 0 -0.104719666666667 3174590367 3174656425
2012 04 09 20 07 09 04:16:39 +03:59:42 J2000 ANU2.3 1 0.104719666666667 1334015078 1334053605
2055 07 04 23 07 05 19:07:01 -18:41:35 J2000 MAGELLAN1 -1 0 2698352302 2698314369
1981 03 25 04 01 39 11:13:41 +86:24:06 J2000 TIDBINBLA 1 0 undef undef
1999 11 14 18 42 59 16:01:05 +09:42:09 B1950 CSO 1 0.349065555555556 942688773 942637107
2008 05 19 15 07 35 14:01:16 +46:07:19 J2000 TOLOLO1.5M 0 0.174532777777778 1211245543 1211172174
1984 04 14 00 13 48 17:41:41 +25:19:59 B1950 GEMINIS 1 0.0174532777777778 450763519 450798300
2070 09 18 18 52 35 03:11:07 +16:13:52 B1950 MAGELLAN1 -1 0 3178234163 3178272668
2034 04 18 12 19 24 14:48:16 +01:07:52 J2000 TOLOLO1.5M -1 0 2028930652 2028973459
2020 08 14 00 09 01 11:09:06 -46:07:13 J2000 MTLEMMON60 -1 -0.174532777777778 1597336108 1597282745
2053 11 25 01 52 35 11:19:59 -58:16:11 B1950 TAUTSCHM -1 -0.349065555555556 2647576316 2647580055
1998 07 19 19 13 06 22:06:07 +32:41:11 J2000 ESOSCHM 0 0 900901352 900848190
2025 06 14 05 01 19 16:52:52 -12:24:05 J2000 NOBEYAMA -1 0.0349065555555556 1749805185 1749842630
2049 01 09 16 13 01 16:59:01 -03:35:11 B1950 SANPM83 1 0.104719666666667 2493892886 2493845356
2012 01 23 17 01 13 19:41:13 -38:16:16 J2000 TIDBINBLA 1 0 1327339200 1327398589
2048 01 12 22 09 59 11:13:24 +84:39:13 B1950 KECK2 -1 -0.0523598333333333 undef undef
2007 05 01 06 59 05 11:52:11 +58:39:06 J2000 LPO1 1 0 1178021489 1178009096
2035 11 04 23 13 35 00:06:09 -03:05:24 B1950 TAUTNBG 0 -0.0349065555555556 2077799352 2077842412
2079 01 07 14 13 13 06:41:19 +01:11:59 J2000 DUNLAP74 -1 0.0523598333333333 3440272119 3440313724
2036 10 18 16 13 41 02:09:59 -51:07:07 B1950 PARKES -1 -0.0523598333333333 2107916307 2107903464
2072 12 12 06 42 48 01:09:24 -29:01:07 J2000 DAO72 1 0.349065555555556 undef undef
2010 04 14 03 13 16 13:13:01 -49:52:19 J2000 ARECIBO 1 0.349065555555556 1271300288 1271222483
1987 08 04 10 48 48 19:52:42 -25:52:35 J2000 MOPRA -1 0.104719666666667 555056986 555018286
2075 09 19 21 48 13 21:13:59 -61:42:06 B1950 KISO -1 0 undef undef
1999 01 14 18 42 41 09:09:35 -58:41:19 J2000 IRTF 0 -0.174532777777778 916296793 916333300
2058 03 18 20 19 07 17:52:06 +32:13:01 J2000 IRTF 1 0.349065555555556 2783762031 2783713354
2047 12 07 01 16 07 23:42:35 -03:16:42 B1950 GEMINIS 0 -0.0174532777777778 2459265472 2459309872
2080 01 18 11 06 48 06:24:05 -84:01:52 J2000 LICK120 1 0.0174532777777778 undef undef
2058 10 25 00 59 13 23:01:24 +29:35:24 B1950 TOLOLO1.5M 1 0.0174532777777778 2802804997 2802751776
2030 08 07 12 48 39 09:24:07 +84:11:09 B1950 LPO4.2 1 -0.104719666666667 undef undef
2030 03 07 03 11 41 14:59:48 +46:11:19 J2000 ATCA 1 0 1899124239 1899149440
2005 12 18 13 01 59 22:06:16 +73:19:07 B1950 CAMB1MILE -1 0 undef undef
1985 01 01 03 11 52 12:07:48 -25:35:11 B1950 GBVA140 1 0.0523598333333333 473409264 473439314
2002 07 14 22 13 16 01:24:01 -25:07:11 B1950 DUPONT 0 0 1026704310 1026668349
1982 12 09 21 05 01 01:01:52 +03:59:13 J2000 SUGARGROVE 1 0 408394138 408352543
1987 03 14 03 41 42 17:59:07 -25:05:16 J2000 MCDONLD2.1 1 -0.174532777777778 542706082 542747662
2004 03 14 13 41 01 08:16:39 +46:19:48 B1950 VLT4 0 0 1079299489 1079242759
1995 11 04 23 35 01 02:41:42 +73:11:11 J2000 CAMB1MILE 0 0.0523598333333333 undef undef
2068 09 18 10 24 48 21:39:52 +84:13:52 J2000 KPNO158 0 0.0349065555555556 undef undef
2068 09 04 21 01 19 19:09:48 -20:19:07 B1950 TAUTSCHM 0 0 3113997547 3114027771
2000 12 01 23 39 13 06:05:48 -41:16:59 J2000 VLT4 1 -0.0349065555555556 975795082 975764910
2043 10 25 17 16 42 10:19:42 -73:05:19 B1950 TOLOLO4M 0 0 undef undef
2062 09 12 17 39 07 02:05:59 +49:41:06 J2000 MTEKAR -1 -0.174532777777778 undef undef
2055 09 07 08 39 59 09:41:48 +46:11:41 J2000 KECK2 0 0 2703936796 2703904184
2046 03 09 03 05 42 16:09:07 -16:09:16 J2000 MAGELLAN1 -1 0.0523598333333333 2404092457 2404138275
2049 04 19 00 59 13 15:39:19 -61:16:16 J2000 VLT4 -1 0.174532777777778 2502399013 2502368495
2076 01 07 06 09 11 17:07:19 +46:11:39 J2000 MTHOP1.5 0 -0.349065555555556 undef undef
2069 09 18 10 39 52 17:39:42 -46:39:05 B1950 UKST 0 0.0523598333333333 3146687249 3146746475
2034 03 04 23 09 48 09:35:24 +84:42:24 B1950 HPROV1.52 0 -0.174532777777778 undef undef
1994 02 23 09 01 09 18:41:09 -03:42:24 J2000 TOLOLO1.5M 1 0 762073183 762031137
2049 10 01 05 41 01 05:13:42 +58:59:06 J2000 JODRELL1 0 0.104719666666667 undef undef
1980 08 04 09 42 24 07:11:06 +03:24:07 J2000 ST.ANDREWS 0 0.104719666666667 334212740 334253108
2043 11 01 01 16 16 08:19:06 -73:19:19 B1950 TOLOLO4M 1 0.0174532777777778 undef undef
1985 08 18 20 01 39 15:07:01 -86:09:11 B1950 IRTF 1 0 undef undef
2042 03 01 04 07 39 00:59:48 -32:06:06 B1950 UKIRT 0 0 2277229336 2277266283
2035 09 25 19 01 16 11:59:39 +32:59:16 B1950 UKIRT -1 -0.174532777777778 2074342936 2074312616
2046 12 23 07 06 39 09:42:52 -38:07:11 B1950 BOSQALEGRE -1 0.349065555555556 2429143392 2429099678
2039 08 01 10 41 16 13:01:42 +18:16:35 J2000 LICK120 -1 -0.0349065555555556 2195745884 2195797182
2047 02 18 10 13 59 19:19:05 +61:41:16 J2000 SUGARGROVE -1 -0.349065555555556 undef undef
2022 01 23 06 39 41 06:48:52 +03:07:19 J2000 LPO4.2 1 0 1642959755 1643003652
2057 03 23 12 05 48 22:52:19 -84:19:13 J2000 MAUNAK88 -1 -0.0523598333333333 undef undef
2072 05 07 01 24 19 12:11:59 +46:05:48 J2000 KOSMA3M 1 0 undef undef
2044 11 07 04 16 16 04:01:06 -46:16:52 B1950 UKST 0 0.0174532777777778 2362113112 2362087679
2052 09 23 03 16 35 01:41:16 -86:19:09 B1950 JCMT 0 -0.0349065555555556 undef undef
2025 07 07 21 11 42 03:19:39 -51:35:16 B1950 KOTTAMIA 1 0.104719666666667 1751949128 1751960723
2059 09 19 21 52 06 17:07:48 +73:07:42 J2000 DUPONT -1 -0.0523598333333333 undef undef
1984 08 12 17 11 48 20:24:05 +41:59:52 J2000 KPNO36FT 0 0.0349065555555556 461197346 461168792
2043 08 07 12 01 09 11:07:59 -25:42:35 J2000 LPO4.2 -1 -0.174532777777778 2322552546 2322508182
2022 06 14 16 06 35 17:06:52 -03:41:06 J2000 ANU2.3 0 0 1655191946 1655236110
2074 03 09 19 11 13 21:05:13 -51:39:41 B1950 HARVARD -1 0 undef undef
2059 01 01 14 35 11 22:42:35 -12:48:48 J2000 MCDONLD2.7 0 0.0349065555555556 2808668164 2808620311
2051 09 07 10 59 01 15:11:13 +12:24:13 B1950 MTEKAR 0 0.0174532777777778 2577688768 2577737204
2016 02 12 17 07 19 21:24:01 -16:06:05 B1950 DUNLAP74 1 0 1455365795 1455315153
2043 12 12 06 42 16 07:16:42 -20:11:52 B1950 KPNO158 -1 -0.0174532777777778 2333506353 2333457476
1999 08 04 18 06 42 01:13:59 +16:48:01 B1950 GBVA140 -1 0 933735038 933784869
2038 11 04 07 07 01 15:06:09 +58:39:09 J2000 MTLEMMON60 -1 -0.0523598333333333 undef undef
1984 01 01 02 19 48 18:41:09 -29:42:06 J2000 PALOMAR60 0 0.0174532777777778 441732228 441764033
2072 05 04 14 59 39 13:11:42 -38:48:11 B1950 KPNO90 -1 0 3229552643 3229580938
2004 09 25 23 52 59 23:06:41 -20:13:35 B1950 KISO 1 -0.0174532777777778 1096187540 1096224017
1980 04 07 17 09 16 15:35:35 -58:09:13 J2000 GEMINIS 1 -0.0349065555555556 undef undef
2042 03 12 05 59 16 03:41:11 -20:07:11 J2000 LPO2.5 1 0.174532777777778 2278242612 2278274095
2033 12 19 01 41 41 18:59:41 -61:05:11 B1950 UKIRT 0 0.0523598333333333 2018552226 2018571471
2006 03 01 15 01 59 20:09:39 +41:05:42 J2000 LPO2.5 -1 -0.0523598333333333 1141180233 1141153394
2016 12 07 22 13 24 16:42:59 -38:42:35 J2000 VLT4 -1 -0.349065555555556 1481092019 1481076806
2067 07 19 07 13 19 03:07:41 -38:11:48 B1950 VLA 0 -0.174532777777778 3078293612 3078243355
2039 11 14 02 19 11 18:24:06 +84:48:05 B1950 VLT2 0 -0.0174532777777778 undef undef
2070 11 19 04 07 39 03:41:11 -12:42:07 J2000 BOSQALEGRE 0 -0.174532777777778 3183569341 3183622083
2008 03 14 12 13 52 06:42:35 -16:39:59 J2000 STEWARD90 1 0 1205529661 1205567586
2072 12 01 06 11 39 18:07:35 -12:42:07 B1950 PALOMAR200 -1 -0.174532777777778 3247743935 3247788806
2008 09 23 04 09 24 02:52:39 -12:19:13 J2000 SUGARGROVE -1 0 1222137647 1222089790
2046 05 14 08 09 52 05:52:52 +16:11:24 J2000 JCMT 1 -0.104719666666667 2409933356 2409982533
2047 04 12 10 05 09 17:39:42 +84:48:41 B1950 KECK1 -1 0.104719666666667 undef undef
2007 02 12 05 19 35 18:39:19 -49:07:42 J2000 VLT1 0 -0.0523598333333333 1171257682 1171232856
1988 03 19 01 35 06 03:19:16 +18:09:16 B1950 LICK120 0 -0.104719666666667 574704614 574758705
2058 10 12 20 11 41 04:11:05 +58:05:39 B1950 TOLOLO4M 0 0.0523598333333333 undef undef
2041 04 14 16 13 59 09:06:24 -09:41:09 B1950 TAUTSCHM 1 0.104719666666667 2249648323 2249594327
2040 09 18 20 52 01 13:48:19 -38:39:05 J2000 HARVARD 0 0 2231596473 2231616777
2044 08 14 15 05 06 17:11:24 +29:39:48 B1950 TAUTNBG 0 -0.349065555555556 undef undef
2065 12 18 10 05 13 08:06:24 -86:05:13 J2000 TOLOLO1.5M -1 -0.0349065555555556 undef undef
1987 11 14 12 39 52 12:41:42 -25:41:11 J2000 ESONTT 1 -0.174532777777778 563953892 563924836
1984 09 19 06 52 48 12:39:07 -49:52:41 J2000 LPO2.5 1 0 464438258 464461956
2015 02 04 10 41 05 12:09:35 -86:01:07 J2000 STEWARD90 0 0 undef undef
2035 10 19 23 06 24 08:09:35 +61:48:52 B1950 ESO3.6 0 0.174532777777778 undef undef
2058 11 23 04 05 52 10:16:59 -37:01:16 B1950 VLA -1 -0.174532777777778 2805179110 2805215098
2027 05 07 21 07 05 21:52:39 +12:41:24 J2000 UKST -1 0 1809703707 1809656824
2014 05 23 12 16 24 11:06:59 +41:11:09 J2000 LOWELL72 -1 0 1400781690 1400842877
2005 04 18 01 42 52 15:42:52 +03:01:35 J2000 TIDBINBLA -1 -0.174532777777778 1113729854 1113777789
2009 07 12 12 06 52 05:41:52 -09:59:06 J2000 MTHOP1.5 -1 0 1247314356 1247354454
2073 05 23 00 07 42 17:42:41 +25:19:24 J2000 QUEBEC1.6 0 -0.349065555555556 undef undef
2020 01 07 02 52 05 22:42:07 -49:35:42 J2000 AAT 0 -0.174532777777778 undef undef
2070 10 23 22 01 11 00:07:09 +86:24:09 B1950 ST.ANDREWS 1 0.104719666666667 undef undef
2063 11 18 06 09 19 14:42:41 -18:48:59 B1950 KPNO158 0 0 2962617323 2962568202
2016 04 23 21 07 52 09:16:13 +84:48:07 J2000 GEMINIS 0 0 undef undef
2021 11 14 07 41 07 08:52:13 +12:52:06 J2000 MTEKAR 0 0.349065555555556 1636846591 1636882183
2015 05 14 06 52 42 11:48:01 +37:48:39 J2000 SUGARGROVE 0 -0.174532777777778 1431616975 1431604383
2021 10 14 09 48 01 03:16:42 -86:42:52 B1950 VLT3 0 -0.174532777777778 undef undef
2000 01 01 10 52 16 15:39:16 +61:16:48 B1950 ESONTT 0 0.174532777777778 undef undef
2055 05 07 15 07 13 01:05:05 -03:42:05 J2000 VLT1 -1 -0.174532777777778 2693289500 2693252465
1984 07 04 05 09 52 13:11:13 -46:01:39 B1950 LICK120 1 0.104719666666667 457840101 457844698
2055 02 04 10 16 05 06:13:09 -58:42:59 J2000 JODRELL1 -1 0 undef undef
2078 01 23 18 01 19 19:05:07 -41:16:35 B1950 SANPM83 0 0 3410174938 3410202880
2068 06 23 17 42 11 20:11:42 +38:05:16 B1950 TOLOLO1.5M 1 -0.0349065555555556 3107730228 3107761725
2040 02 23 06 48 48 20:07:24 -20:01:48 J2000 MCDONLD2.1 -1 -0.349065555555556 2213518197 2213567029
2080 01 14 16 13 41 04:42:01 -01:16:35 B1950 AAT -1 0 3472434923 3472392145
2062 01 07 21 13 05 03:48:16 +03:05:59 B1950 MOPRA 0 -0.349065555555556 2903916021 2903883170
2038 05 07 06 59 05 04:48:01 -01:13:06 B1950 HARVARD -1 -0.174532777777778 2156759428 2156808549
1993 03 01 22 41 05 12:24:06 +58:42:59 B1950 KISO 0 0.349065555555556 731063301 731030586
2007 01 01 18 11 59 21:05:11 -20:11:16 J2000 MMT -1 -0.0174532777777778 1167669155 1167620415
2020 09 25 10 42 35 08:41:59 +58:41:13 J2000 GEMININ 0 0.174532777777778 1601034703 1600998167
2038 03 18 15 09 06 21:39:41 -41:41:16 B1950 FCRAO 1 0 2152613942 2152545357
1991 08 09 02 59 39 06:52:05 +58:13:19 B1950 SUGARGROVE 0 0.0523598333333333 undef undef
2037 03 14 11 52 16 02:07:41 -16:06:11 B1950 MCDONLD2.7 0 0.0523598333333333 2120661032 2120611588
2029 10 01 03 01 41 01:19:13 +37:48:16 B1950 ATCA 0 0 1885545209 1885489004
2036 11 09 17 39 05 22:59:24 +86:05:06 B1950 LOWELL72 -1 0 undef undef
1981 09 18 05 41 16 14:06:05 -51:06:13 B1950 CFHT 1 0 369692768 369723071
2027 02 12 08 01 42 07:19:24 -84:42:01 J2000 SUBARU 0 -0.0349065555555556 undef undef
2044 05 25 20 06 16 09:13:19 +51:09:42 B1950 JODRELL1 0 0 undef undef
2030 12 25 08 19 52 10:11:24 -51:11:06 B1950 LPO2.5 0 0.174532777777778 undef undef
2062 01 12 15 52 52 13:11:42 +03:48:05 B1950 SANPM83 0 0.0523598333333333 2904277029 2904319359
2048 08 14 23 48 42 00:07:09 +20:48:42 J2000 MMT -1 0 2480987114 2481036788
2042 07 23 06 42 24 00:52:35 -49:07:39 B1950 KPNO158 0 0 2289719914 2289741407
2074 11 25 09 59 06 23:35:07 +03:01:41 J2000 JCMT 0 0 3310328319 3310371996
2079 07 23 20 07 13 19:09:19 -51:59:11 J2000 STEWARD90 -1 -0.0174532777777778 3457310155 3457329514
2026 11 19 19 01 05 09:39:05 +09:16:01 B1950 CAMB5KM 0 0 1795129227 1795091724
2043 01 07 02 41 52 11:48:06 +38:05:35 J2000 QUEBEC1.6 1 -0.0523598333333333 2304285406 2304272066
2046 02 01 00 24 24 14:42:35 +29:39:48 B1950 SANPM83 0 0.174532777777778 2401082268 2401042067
2049 08 07 17 06 13 04:52:42 +84:01:06 J2000 UKST -1 0.0349065555555556 undef undef
1994 10 01 06 42 24 18:52:39 +25:06:11 B1950 PALOMAR60 0 0.0174532777777778 780951496 781002536
1992 03 09 18 16 52 12:59:05 +01:13:07 J2000 PALOMAR200 1 0 700198236 700241715
2027 07 07 04 48 35 06:13:09 -20:24:59 J2000 CAMB1MILE 1 0 1814944116 1814973486
2078 07 18 06 52 07 18:06:06 +51:35:05 B1950 PALOMAR60 -1 0 3425315257 3425299057
2005 05 12 05 35 24 18:06:13 -73:07:19 J2000 CATALINA61 -1 -0.0523598333333333 undef undef
1989 07 14 00 13 06 06:09:19 -01:59:59 J2000 BLOEMF 1 0.0174532777777778 616388128 616431192
2077 10 18 01 59 06 12:19:09 +03:09:07 J2000 SUBARU -1 -0.0523598333333333 3401707276 3401666195
1985 03 18 01 39 07 22:19:01 +61:41:07 J2000 KOSMA3M 0 0 undef undef
2059 02 19 11 19 41 20:07:16 +49:59:05 B1950 PALOMAR48 1 0 2812955205 2812937238
2030 07 19 07 42 39 03:16:24 -86:11:16 B1950 MAUNAK88 0 0.0174532777777778 undef undef
2007 07 14 16 01 48 01:11:39 -01:41:16 J2000 MOPRA -1 0 1184420606 1184378003
2011 07 12 08 09 09 15:59:05 -32:35:06 B1950 DUPONT 0 0.0523598333333333 1310494697 1310459557
2004 12 07 05 42 59 21:07:09 +03:42:19 B1950 UKST -1 0.104719666666667 1102380413 1102332805
1985 05 23 17 24 39 08:01:09 +51:11:39 B1950 MTEKAR 1 0.104719666666667 undef undef
1980 08 18 06 06 01 18:06:19 +20:06:39 J2000 DUPONT 1 0.174532777777778 335479009 335510370
2061 02 18 02 52 59 06:07:11 +41:24:42 B1950 USSR600 1 0 2875938149 2875922712
2062 07 18 13 16 07 17:05:41 +09:11:05 J2000 DUNLAP74 0 0 2920478596 2920439754
2007 11 14 13 24 19 12:48:52 +38:42:06 B1950 CAMB5KM 0 -0.0349065555555556 undef undef
2004 12 04 13 39 39 06:24:52 +09:35:19 J2000 MMT 0 -0.0174532777777778 1102127217 1102173739
2040 03 07 08 48 52 12:11:42 +84:59:52 B1950 KISO 1 0.104719666666667 undef undef
1988 06 14 17 16 59 07:16:48 -73:24:42 J2000 GEMINIS 1 0.0349065555555556 undef undef
2005 11 18 23 05 19 00:06:41 +86:35:39 B1950 MTLEMMON60 -1 -0.349065555555556 undef undef
2053 06 23 02 41 05 15:39:11 -25:42:35 J2000 TOLOLO1.5M 1 0.0174532777777778 2634318949 2634283067
2009 04 18 21 59 11 02:24:19 -32:01:41 J2000 MAGELLAN1 0 0 1240134951 1240101571
2065 02 12 16 11 13 17:41:42 +73:48:06 J2000 CAMB5KM -1 -0.349065555555556 undef undef
2007 02 07 17 24 35 16:01:35 +46:59:52 B1950 ESONTT 0 0.174532777777778 1170841694 1170854785
2044 05 01 02 01 16 09:48:01 +12:09:01 J2000 KPNO90 1 -0.349065555555556 2345739820 2345712515
2044 01 09 17 42 05 09:41:24 -51:42:39 J2000 CAMB1MILE 0 0 undef undef
2020 12 09 01 01 05 10:24:05 +61:06:09 J2000 ESO3.6 0 -0.349065555555556 1607488886 1607440227
1984 04 25 10 01 01 06:19:11 -18:42:09 J2000 MTEKAR -1 0 451651393 451684709
2053 09 23 18 07 05 11:41:24 -86:41:52 B1950 GBVA140 -1 0.349065555555556 undef undef
2008 07 04 21 59 05 22:52:09 -18:41:52 B1950 MAUNAK88 0 0.174532777777778 1215250525 1215198674
1993 06 19 07 19 06 21:48:59 -73:39:07 J2000 HPROV1.52 0 -0.174532777777778 undef undef
2021 11 23 09 48 35 22:39:59 +25:16:19 J2000 CATALINA61 0 0.104719666666667 1637694852 1637656297
2066 11 12 18 24 59 21:35:39 +01:05:52 J2000 AAT 0 0 3056840165 3056796676
1996 12 25 14 11 48 18:19:48 -49:42:48 J2000 DAO72 0 0.0349065555555556 undef undef
2030 05 18 01 41 39 16:06:41 +16:01:07 J2000 KISO 1 -0.174532777777778 1905319915 1905375145
2069 12 25 13 19 16 01:48:59 +01:19:11 J2000 MAUNAK88 -1 0.0174532777777778 3155155196 3155198056
1986 12 01 08 06 41 16:41:24 +12:19:11 J2000 SANPM83 -1 0.174532777777778 533743404 533784406
1980 12 19 18 48 05 12:11:09 -41:01:41 J2000 NOBEYAMA 0 0.0523598333333333 346097066 346118345
1985 11 04 21 42 42 17:07:16 -32:05:09 B1950 ST.ANDREWS 1 0 500043856 500053155
2059 08 01 11 11 09 23:59:13 +58:24:19 J2000 MCDONLD2.1 -1 -0.104719666666667 undef undef
1989 04 23 22 16 05 12:48:07 -86:48:42 B1950 BLOEMF -1 0 undef undef
2010 01 01 07 41 05 19:11:13 +37:48:01 J2000 TAUTNBG -1 0.104719666666667 1262312915 1262292994
2068 02 18 12 59 05 08:01:16 +29:01:05 B1950 SANPM83 -1 0.349065555555556 3096750484 3096790106
2050 05 09 06 52 59 23:16:59 +20:41:59 J2000 UKIRT 1 0 2535710374 2535757259
2058 04 23 08 01 09 04:35:07 -29:48:16 J2000 UKIRT 1 0 2786816297 2786853702
2079 07 07 13 39 42 19:06:52 -32:52:39 B1950 PARKES 0 0.0174532777777778 3455938102 3455992219
2045 05 25 14 59 48 11:48:07 +84:11:09 B1950 DUPONT 0 -0.0174532777777778 undef undef
2066 07 04 09 48 06 16:42:16 +03:06:39 B1950 NOBEYAMA 0 -0.174532777777778 3045447939 3045497994
2009 03 18 03 16 59 02:59:13 +86:09:52 J2000 SANPM83 0 0.174532777777778 undef undef
2033 12 12 05 01 41 04:35:16 +46:07:39 B1950 MCDONLD2.1 0 0.0174532777777778 2017950535 2018010960
2070 12 25 06 09 07 23:35:06 +09:13:06 B1950 MCDONLD2.1 -1 0.174532777777778 3186672364 3186712657
2035 11 25 17 39 35 06:01:01 +18:48:19 J2000 PARKES -1 0.349065555555556 2079606939 2079544383
2053 12 23 04 42 41 07:59:19 -41:24:01 B1950 LICK120 0 0 2650085493 2650108081
1989 06 12 06 42 39 11:16:19 +41:11:42 J2000 MAGELLAN1 1 -0.349065555555556 613672012 613716021
2067 11 01 05 24 52 01:05:01 +37:41:07 J2000 LPO2.5 -1 0.349065555555556 3087309299 3087350575
2043 01 25 21 09 06 20:42:52 +16:13:05 B1950 KPNO84 -1 0.0349065555555556 2305804905 2305765747
2028 08 23 23 59 42 02:19:05 +51:05:52 J2000 FCRAO 1 0.104719666666667 1850769193 1850757499
2026 12 09 04 11 52 01:35:24 -32:42:41 J2000 TOLOLO4M 1 -0.0349065555555556 1796837213 1796805996
2027 07 04 13 19 13 16:59:05 +01:52:07 B1950 UKST 0 0 1814682170 1814724741
2044 07 12 13 19 39 06:19:07 +12:48:09 B1950 MOPRA 0 -0.174532777777778 2351961992 2351920855
2057 08 09 18 24 24 02:07:41 -49:41:09 J2000 FCRAO -1 0.0174532777777778 undef undef
2077 05 04 19 42 05 20:16:24 +46:52:48 B1950 EFFELSBERG -1 0.349065555555556 3387301048 3387359183
1985 09 07 11 09 09 11:16:07 -16:59:39 B1950 MCDONLD2.1 0 0 494948993 494900852
2039 12 12 10 06 42 14:06:07 -73:09:24 B1950 MAGELLAN1 0 0 undef undef
2067 02 23 10 41 39 16:01:42 -01:59:05 B1950 ESO3.6 0 0.349065555555556 3065666678 3065699370
2024 11 25 06 07 01 08:59:52 -20:52:42 J2000 DUPONT 1 0.349065555555556 1732594144 1732545364
2050 01 25 15 24 16 16:07:35 +03:05:09 B1950 PARKES -1 -0.174532777777778 2526736719 2526698468
2013 01 07 05 09 35 03:59:42 -03:09:09 J2000 CATALINA61 0 -0.174532777777778 1357508239 1357556046
2019 03 12 12 09 13 09:35:48 -03:52:07 B1950 JODRELL1 -1 0 1552323285 1552363677
2076 07 25 19 13 42 08:24:16 -32:05:05 B1950 MCDONLD2.7 -1 -0.0523598333333333 3362912392 3362860787
2067 11 18 12 41 35 02:16:42 -86:59:16 J2000 MAGELLAN1 0 -0.0523598333333333 undef undef
2007 01 12 04 06 48 17:07:06 -46:07:05 J2000 NOBEYAMA 1 0.0523598333333333 1168640233 1168655590
1985 07 14 02 09 41 15:13:07 +86:48:11 B1950 KOSMA3M 1 0 undef undef
2032 07 23 20 42 19 03:52:41 -12:42:13 B1950 FLAGSTF61 0 0.0523598333333333 1974190071 1974227031
1993 11 14 16 16 01 15:07:39 -32:07:06 B1950 HPROV1.52 0 -0.0174532777777778 753262536 753288649
2001 02 19 05 13 19 08:42:16 +16:42:11 J2000 CSO 0 0 982550575 982596629
2014 01 19 05 09 59 19:19:13 +58:48:13 J2000 SANPM83 1 -0.0523598333333333 undef undef
2002 12 12 04 16 35 12:24:35 -37:59:59 B1950 MAGELLAN1 -1 -0.349065555555556 1039655415 1039645556
1997 04 14 05 39 01 20:09:59 -46:41:48 B1950 VLA -1 -0.174532777777778 860924038 860955556
2062 04 19 06 13 05 09:05:42 +58:16:05 J2000 ATCA 1 -0.104719666666667 2912653311 2912674827
2000 09 01 12 52 07 15:19:48 +29:09:19 J2000 KISO 0 0.0349065555555556 967852847 967819663
2004 12 18 23 16 07 11:42:24 +46:11:35 J2000 BOSQALEGRE 1 0.174532777777778 1103445876 1103455819
2080 02 14 14 35 24 13:59:19 +61:07:05 J2000 VLT1 0 0 3475118757 3475135580
2019 11 18 08 39 19 15:16:05 -32:13:07 J2000 BLOEMF -1 0.0523598333333333 1574044750 1574009452
2006 06 04 17 52 06 22:11:07 +12:13:19 B1950 UKST 0 -0.104719666666667 1149427701 1149470526
2054 02 19 19 11 06 07:07:41 -49:52:13 J2000 MCDONLD2.7 1 -0.174532777777778 2655157018 2655189050
2023 02 01 08 09 11 11:39:35 +16:48:59 J2000 CSO 0 0.0174532777777778 1675234645 1675280163
1992 02 23 07 42 16 00:16:11 +86:01:11 B1950 ESOSCHM -1 0.349065555555556 undef undef
2022 03 09 08 48 19 22:13:59 +37:19:52 B1950 TAUTSCHM 0 0.174532777777778 1646790864 1646851761
2013 08 07 19 41 48 10:06:13 +38:59:42 B1950 VLT1 0 -0.104719666666667 1375879053 1375916061
1983 02 14 23 05 07 19:35:01 +09:48:35 J2000 BLOEMF 0 0 414124341 414078633
2076 08 09 04 59 24 20:35:16 -37:42:24 B1950 LICK120 -1 0 3364171062 3364111000
2004 08 07 00 52 01 08:42:24 -20:41:35 B1950 VLA 1 -0.104719666666667 1091884678 1091924376
2056 07 12 23 01 05 06:52:06 -29:13:07 B1950 KECK2 -1 -0.0523598333333333 2730644692 2730597787
2072 07 19 09 59 05 17:41:13 +41:48:06 B1950 MAGELLAN1 0 0 3236106972 3236135847
2057 03 09 14 16 13 03:16:06 -01:01:41 J2000 MTHOP1.5 0 0.174532777777778 2751387596 2751338629
2006 04 25 01 11 39 20:01:24 +51:39:09 J2000 JODRELL1 1 0.0349065555555556 undef undef
1990 04 25 11 35 09 07:24:35 -12:07:07 B1950 PALOMAR60 1 0 641071945 641111111
2050 12 04 11 35 05 08:42:07 -73:19:13 J2000 TAUTNBG 0 0.0523598333333333 undef undef
2030 08 04 21 01 09 18:39:05 -61:01:19 B1950 HPROV1.52 -1 0 undef undef
2047 02 25 14 01 24 15:42:01 +37:16:42 J2000 ANU2.3 -1 0.349065555555556 2434645535 2434653158
1994 12 19 09 13 16 22:09:24 +86:05:13 J2000 ST.ANDREWS 0 0 undef undef
2067 12 07 06 16 05 12:05:13 +58:05:09 B1950 CAMB1MILE 1 0.0523598333333333 undef undef
2007 08 07 11 01 59 15:52:42 +58:41:13 B1950 GEMININ 0 -0.349065555555556 undef undef
1988 10 09 09 11 07 16:05:35 -86:13:52 J2000 PALOMAR200 -1 -0.349065555555556 undef undef
2020 01 14 05 05 05 21:19:11 -01:13:59 J2000 HPROV1.52 -1 0.349065555555556 1578907590 1578936490
2080 12 14 14 07 59 02:19:41 -20:05:07 J2000 UKIRT -1 0.349065555555556 3501371673 3501399905
2055 06 14 02 42 05 14:52:05 +09:24:48 B1950 IRTF 1 0 2696635962 2696594446
2016 07 19 17 19 52 13:39:24 +38:19:24 B1950 EFFELSBERG 1 -0.174532777777778 undef undef
1983 01 01 17 11 01 04:39:13 -49:59:05 J2000 VLT2 -1 0.104719666666667 410210074 410263484
2043 11 12 00 09 13 05:07:24 +25:59:05 J2000 HPROV1.93 0 0.0174532777777778 2330876331 2330932024
2042 08 01 05 39 07 00:11:16 -03:35:59 J2000 PALOMAR60 1 0 2290569956 2290525815
1988 04 23 11 24 48 17:39:52 -09:16:42 B1950 ST.ANDREWS -1 0.174532777777778 577757121 577783791
2021 05 07 07 06 42 03:41:09 -73:24:59 J2000 VLA 0 -0.0174532777777778 undef undef
2008 05 18 05 07 13 02:41:41 +46:01:52 J2000 ST.ANDREWS -1 -0.174532777777778 undef undef
2071 09 09 01 16 07 23:24:11 -29:42:35 J2000 STEWARD90 1 -0.174532777777778 3208990079 3209029983
2023 11 12 15 19 59 14:52:35 -61:52:39 J2000 KOSMA3M 1 -0.104719666666667 undef undef
2003 02 23 15 16 16 18:24:35 +84:42:16 B1950 CFHT 0 0 undef undef
1983 05 18 06 35 48 17:09:11 -38:06:35 B1950 ESONTT 0 0 422058583 422114162
2068 02 14 18 01 42 14:06:24 -29:19:24 B1950 CAMB5KM -1 -0.174532777777778 3096404117 3096435511
2042 11 18 00 35 24 17:41:07 -09:39:59 J2000 KPNO90 1 -0.0349065555555556 2299937607 2299892747
2068 04 25 09 07 19 22:07:19 -58:05:16 B1950 HPROV1.93 0 0 undef undef
1993 04 01 16 19 59 15:01:07 -25:39:11 B1950 KPNO84 0 0.104719666666667 733642549 733673173
1989 08 01 15 01 59 12:13:11 -51:41:52 J2000 VLT3 -1 0 617975541 617949403
2012 06 01 07 24 24 11:06:24 +86:19:59 J2000 HARVARD 0 -0.174532777777778 undef undef
2043 11 12 10 01 48 14:42:41 -37:01:41 J2000 ESOSCHM 0 0 2330929327 2330898297
2013 04 14 02 19 59 02:24:48 +37:06:06 J2000 MOPRA 0 0.349065555555556 1365904588 1365912137
2041 02 25 11 24 52 13:42:35 +61:06:11 J2000 AAT 1 -0.0523598333333333 2245422207 2245429466
1994 08 19 23 59 19 16:39:24 +49:09:59 B1950 MMT -1 -0.0349065555555556 777314983 777296220
2077 03 12 20 07 13 09:35:01 -86:48:52 J2000 LICK120 1 0.0349065555555556 undef undef
2034 09 14 19 09 24 14:52:05 +41:39:06 J2000 HPROV1.52 -1 -0.0349065555555556 2041821075 2041809864
2006 01 01 01 11 19 07:11:48 -46:52:06 J2000 ARECIBO 0 0 1136074791 1136107949
2020 07 14 10 41 05 13:35:42 +58:35:05 J2000 JODRELL1 -1 0.0349065555555556 undef undef
2000 04 07 03 11 48 12:01:59 -29:48:19 B1950 SANPM83 0 0 955073125 955106439
2053 08 25 13 41 06 23:06:52 +73:06:16 B1950 HARVARD -1 0.174532777777778 undef undef
2017 04 25 04 48 06 21:24:19 +12:19:41 J2000 STROMLO74 -1 -0.174532777777778 1493046211 1493090999
1999 09 12 11 01 35 14:42:01 -58:16:09 J2000 CAMB5KM 0 0 undef undef
2079 09 23 17 16 07 12:48:35 +51:06:52 J2000 MTHOP1.5 0 0 3462691814 3462672139
2058 02 19 09 41 19 22:59:52 -38:19:41 B1950 BLOEMF 0 -0.0349065555555556 2781314880 2781371705
2020 01 09 18 09 16 16:24:11 +61:39:19 J2000 PALOMAR60 -1 -0.0349065555555556 undef undef
2034 06 25 15 52 05 23:39:16 +51:13:48 B1950 KOSMA3M 1 -0.349065555555556 undef undef
1986 06 18 11 42 52 11:42:35 +73:39:11 J2000 CFHT 0 -0.104719666666667 undef undef
2060 01 23 22 01 06 10:06:35 -25:13:59 B1950 ESOSCHM -1 -0.349065555555556 2842119781 2842098190
2057 02 25 05 41 48 22:06:52 +37:35:07 B1950 FLAGSTF61 0 0.0349065555555556 2750325330 2750296568
2026 04 07 09 41 07 14:16:05 -84:41:06 J2000 CATALINA61 0 0.174532777777778 undef undef
1991 07 23 03 41 01 18:48:01 +09:19:24 B1950 QUEBEC1.6 -1 -0.0349065555555556 680215430 680178380
1998 12 09 06 24 11 11:59:48 -61:48:42 B1950 DAO72 0 0 undef undef
2025 08 23 14 11 41 12:11:35 -32:13:07 B1950 OKAYAMA 1 0 1755996937 1756027481
2033 04 07 00 11 13 12:09:16 +46:11:41 B1950 KOTTAMIA 0 -0.349065555555556 undef undef
1989 10 18 22 35 01 00:11:05 +49:07:07 J2000 LPO1 0 -0.349065555555556 undef undef
2002 01 25 11 42 06 14:24:24 -37:41:16 B1950 KPNO84 0 -0.0523598333333333 1011949933 1011981495
1985 05 19 01 52 48 09:09:19 -01:09:48 J2000 KOTTAMIA 0 -0.0523598333333333 485341321 485299593
1990 03 12 04 11 16 01:59:01 +29:35:41 J2000 PARKES -1 0.104719666666667 637203046 637145240
2013 01 04 17 19 09 12:09:41 -61:16:19 J2000 VLT3 0 0.0349065555555556 1357346132 1357326669
2064 10 19 19 48 24 11:39:11 -16:05:01 J2000 GEMINIS 1 0 2991714447 2991676108
1999 11 04 04 11 41 16:07:35 +16:24:48 B1950 NOBEYAMA 0 0 941663773 941712709
2047 12 04 09 06 01 07:19:06 -61:52:35 J2000 LICK120 1 -0.0174532777777778 undef undef
1980 07 19 13 11 06 20:09:35 -84:16:35 B1950 STEWARD90 1 0 undef undef
2036 04 12 12 39 13 06:09:07 +38:59:48 J2000 USSR600 -1 0.104719666666667 2091591387 2091565629
2053 03 23 02 16 48 22:05:01 -25:41:19 J2000 CAMB1MILE -1 0 2626238270 2626263286
2056 08 25 23 35 41 13:16:01 -09:05:05 J2000 FLAGSTF61 1 -0.104719666666667 2734532417 2734489697
2066 09 07 19 39 09 09:52:59 +29:59:59 J2000 LPO1 0 0.0174532777777778 3051147036 3051112022
2075 05 04 20 05 06 01:16:41 -09:41:11 B1950 ARECIBO 1 0 3324272908 3324228380
2038 06 07 23 42 24 20:09:05 +38:06:52 J2000 CAMB5KM 0 0.104719666666667 2159544835 2159527067
1992 05 25 02 19 07 11:07:06 +12:13:16 B1950 KISO 1 0 706763504 706810803
2054 02 12 05 41 24 22:41:48 -03:16:16 B1950 CAMB5KM 0 0 2654494241 2654449478
2059 12 12 03 35 42 14:59:06 +46:11:05 B1950 KPNO36FT 0 0 2838443151 2838419129
2049 05 14 14 41 09 08:35:39 +29:09:11 J2000 NOBEYAMA 1 0 2504650639 2504618888
2015 07 09 11 41 19 03:42:39 -12:07:59 B1950 OKAYAMA -1 0 1436379557 1436418632
1990 11 07 06 52 07 18:35:07 +73:05:35 B1950 MAUNAK88 0 0 undef undef
2008 11 07 20 01 24 03:59:59 -12:07:09 J2000 ESOSCHM -1 -0.0523598333333333 1226012128 1226060211
2043 10 07 21 06 42 11:09:24 +25:13:48 J2000 ESONTT -1 -0.0174532777777778 2327823970 2327860430
2080 04 07 05 35 35 18:41:07 +84:06:59 B1950 MAGELLAN2 0 0.0174532777777778 undef undef
1997 12 09 17 41 06 19:19:39 +58:07:19 B1950 LPO2.5 0 0.174532777777778 881651825 881709578
1999 09 23 15 42 24 03:19:59 +41:11:06 J2000 HPROV1.52 1 0 938105937 938176517
2045 07 14 05 13 39 00:48:16 -03:09:35 J2000 KECK1 -1 0 2383552196 2383594776
2038 04 19 02 35 59 03:05:09 +37:52:09 B1950 KPNO158 1 -0.0174532777777778 2155293701 2155265529
2054 03 19 11 07 11 01:01:13 +73:52:16 J2000 TAUTSCHM -1 0 undef undef
2042 01 12 07 35 07 17:59:01 -32:41:16 B1950 USSR600 1 0.349065555555556 undef undef
2019 08 04 18 42 16 09:06:16 -01:06:59 J2000 MTHOP1.5 1 -0.174532777777778 1565009489 1564971685
2015 10 09 02 07 07 10:52:39 +58:13:09 J2000 ANU2.3 0 0.174532777777778 undef undef
2045 02 18 09 42 11 23:39:41 +25:19:19 J2000 SUGARGROVE -1 0 2370944281 2370998082
2002 04 14 23 52 59 20:48:07 -49:11:39 B1950 MCDONLD2.1 1 -0.0349065555555556 1018867652 1018892380
1995 06 23 20 48 52 16:01:09 +12:06:48 B1950 KPNO84 1 0.0349065555555556 803948543 803994109
2039 06 14 20 06 06 07:09:39 +37:06:01 B1950 KPNO158 -1 -0.0349065555555556 2191669541 2191641483
1994 09 12 16 52 35 18:13:24 +32:13:48 J2000 KECK1 0 0.104719666666667 779409975 779369530
2034 09 18 06 01 19 06:16:05 -46:13:09 B1950 LPO1 0 0.104719666666667 2042168048 2042188042
2065 03 14 02 13 41 05:19:19 +29:11:35 J2000 ESOSCHM 0 -0.0349065555555556 3004191653 3004227295
1988 12 18 19 52 16 01:59:39 -12:35:11 B1950 FLAGSTF61 -1 0 598400230 598439045
2057 01 23 14 48 09 08:19:24 +49:39:11 B1950 GEMINIS 1 0 2747526316 2747548970
2036 04 01 22 16 42 12:19:24 +32:41:13 B1950 ESOSCHM -1 -0.174532777777778 2090616846 2090656693
1992 09 19 23 07 41 15:41:09 +29:16:05 J2000 GEMINIS 0 0 716917498 716951442
2068 05 18 05 07 41 20:42:01 +18:39:11 J2000 FCRAO -1 0 3104534112 3104499748
2076 10 23 16 16 24 17:48:19 +29:41:41 B1950 KISO -1 0.104719666666667 3370635160 3370685527
2010 09 19 13 35 09 05:06:05 -16:35:13 J2000 CFHT 1 -0.0349065555555556 1284976047 1284931083
2070 10 09 20 59 24 10:16:41 -12:01:41 J2000 MCDONLD2.7 1 -0.0174532777777778 3180162230 3180116141
1996 11 01 02 11 24 01:48:41 +86:09:13 J2000 UKST 1 0 undef undef
2024 04 14 15 42 35 10:06:24 +41:09:42 J2000 CATALINA61 0 0 1713123760 1713096746
2024 04 14 05 42 19 15:13:05 +37:16:06 B1950 GBVA140 0 0.174532777777778 1713052199 1713104224
2009 08 18 10 13 01 23:06:06 +16:35:16 B1950 SUBARU 1 0.349065555555556 1250664163 1250613573
2062 07 04 17 48 42 15:52:52 -73:13:19 J2000 MMT 1 0.0523598333333333 undef undef
2030 11 23 12 39 06 00:42:01 -84:35:24 B1950 ATCA 1 -0.174532777777778 undef undef
2059 07 19 00 59 35 04:06:19 +86:19:16 B1950 FLAGSTF61 1 0.0523598333333333 undef undef
2064 07 19 15 11 52 12:01:59 -84:48:13 J2000 TOLOLO4M 0 -0.0349065555555556 undef undef
2060 06 18 05 13 59 18:05:01 +29:07:52 B1950 MTLEMMON60 0 -0.0174532777777778 2854743312 2854797032
2008 10 12 03 59 07 08:41:16 +86:48:05 J2000 MTLEMMON60 1 0.0523598333333333 undef undef
2052 03 23 11 59 59 18:41:06 -32:35:35 B1950 PALOMAR60 1 0 2594887433 2594832496
2046 09 14 02 42 59 21:48:39 -38:42:52 J2000 DAO72 1 0.0523598333333333 undef undef
2040 01 01 04 48 11 09:09:19 -51:09:41 B1950 KISO 0 0 2209045120 2208970854
2078 06 01 14 59 13 22:01:01 -25:13:41 J2000 STROMLO74 0 0 3421311072 3421363324
2039 07 19 12 52 09 23:35:11 -58:13:11 J2000 DUNLAP74 1 -0.349065555555556 2194753414 2194777347
1993 11 19 21 48 39 00:16:01 +25:39:19 B1950 ATCA 0 0 753773347 753722369
1984 10 14 09 09 01 16:16:01 +84:52:06 B1950 TAUTSCHM 0 -0.104719666666667 undef undef
1987 12 12 02 05 01 02:11:59 +86:52:11 J2000 TAUTNBG 0 0.349065555555556 undef undef
2060 12 14 11 59 52 02:06:05 -12:41:01 B1950 HPROV1.93 0 -0.0174532777777778 2870261675 2870213551
2038 01 09 07 06 06 07:01:09 +41:09:13 B1950 DAO72 0 0.349065555555556 2146611394 2146662818
2061 09 04 12 42 09 20:09:09 -61:48:52 J2000 APO3.5 -1 -0.0349065555555556 undef undef
2061 02 01 21 52 41 21:07:09 +84:52:42 B1950 TOLOLO4M 1 -0.0349065555555556 undef undef
2030 06 09 22 48 48 17:19:06 -03:24:41 B1950 KPNO84 -1 -0.104719666666667 1907198416 1907243845
1980 07 04 21 07 05 06:19:35 -25:05:52 J2000 VLA 0 0.0349065555555556 331567409 331600295
2032 08 04 08 07 39 01:06:11 -37:35:19 B1950 CFHT 1 0 1975225273 1975260769
2077 09 14 15 24 52 17:09:48 +86:39:01 J2000 LPO2.5 0 0.0523598333333333 undef undef
2062 03 18 02 52 48 10:39:39 -01:24:59 B1950 TOLOLO4M 1 0 2909943423 2909900899
1989 06 04 15 16 01 08:52:39 +25:52:05 J2000 LPO4.2 0 -0.0349065555555556 612957587 613009341
2021 10 23 01 42 11 00:42:41 +51:42:05 J2000 DUNLAP74 -1 0.104719666666667 1634920336 1634916032
2043 06 07 07 13 05 09:09:41 -09:48:41 J2000 TOLOLO4M 0 0.174532777777778 2317302881 2317257009
2039 01 04 00 42 16 06:35:42 -41:59:01 J2000 PARKES -1 0 2177645390 2177705614
1984 08 19 10 11 39 11:16:13 +73:09:42 B1950 KECK1 -1 0.174532777777778 461691535 461750491
2018 02 07 02 09 09 13:59:19 +18:16:07 J2000 LPO2.5 -1 0.349065555555556 1517964992 1517915534
1991 04 14 17 42 59 11:01:19 +18:39:48 B1950 ARECIBO 0 0 671657716 671617672
2066 08 07 10 39 42 16:19:01 +58:19:24 J2000 MCDONLD2.7 1 0 3048419949 3048411729
1992 06 07 07 01 48 10:01:24 -16:42:01 B1950 PALOMAR60 -1 -0.174532777777778 707856314 707899867
1989 09 18 00 09 01 05:41:41 +49:24:06 B1950 MAUNAK88 1 -0.104719666666667 622108360 622082642
2042 03 14 13 11 52 08:39:48 -20:24:39 J2000 ANU2.3 -1 0 2278383902 2278347126
2060 11 12 18 01 59 21:42:59 -32:52:13 J2000 QUEBEC1.6 0 -0.0523598333333333 2867512617 2867539363
2043 01 04 03 09 13 22:24:35 -20:01:35 B1950 LICK120 -1 -0.104719666666667 2303921600 2303874902
1990 06 19 22 09 06 17:19:16 +46:07:09 B1950 TOLOLO1.5M -1 0.0349065555555556 645757285 645780535
2004 06 14 22 06 01 01:13:52 +38:52:52 B1950 FLAGSTF61 0 0.0523598333333333 1087283428 1087254444
1996 04 23 02 13 11 13:42:06 +73:19:39 B1950 BLOEMF -1 0.0523598333333333 undef undef
1982 06 19 03 48 41 11:39:39 -03:06:52 J2000 EFFELSBERG 0 0.0523598333333333 393335721 393288609
2005 04 14 13 09 41 10:07:01 -41:59:24 J2000 AAT 1 0.0349065555555556 1113532772 1113503810
2030 09 18 03 11 16 06:06:35 +01:13:09 B1950 CFHT 1 0 1915958477 1916001768
1984 07 19 18 41 07 11:05:42 +84:06:05 J2000 TOLOLO4M 0 0.0349065555555556 undef undef
1990 10 18 05 19 11 04:59:42 -25:11:24 J2000 MAGELLAN2 -1 0.349065555555556 656217374 656169572
2013 06 09 08 19 19 06:07:48 +49:52:06 B1950 VLT4 -1 0.104719666666667 1370703029 1370723998
2039 02 25 09 39 41 08:16:35 +37:01:39 J2000 KOTTAMIA 0 0.0349065555555556 2182249264 2182216934
2023 05 12 01 48 59 01:19:24 +25:09:24 J2000 KPNO36FT 0 0 1683886731 1683851849
1999 05 23 13 39 41 10:09:05 -09:35:01 J2000 SAAO74 -1 0 927454731 927414591
2066 10 23 22 42 06 00:48:59 -38:07:19 J2000 KISO 1 0.104719666666667 3055142024 3055163003
2042 04 09 09 11 01 09:13:19 -20:09:35 B1950 APO3.5 -1 0 2280607650 2280644046
2010 07 18 00 11 42 14:39:07 +09:42:42 B1950 FLAGSTF61 1 -0.104719666666667 1279481151 1279444916
2054 08 23 01 59 11 10:07:06 +16:52:09 B1950 MTEKAR 1 -0.104719666666667 2671068638 2671124788
2077 07 12 18 09 39 01:19:39 +38:09:42 B1950 IRTF 1 0.174532777777778 3393396533 3393354911
2077 05 12 14 41 13 10:09:35 +01:09:41 B1950 TAUTSCHM 0 0.0174532777777778 3388046960 3388089595
1999 03 23 19 52 59 11:01:13 +01:59:07 J2000 MAGELLAN1 1 -0.104719666666667 922223835 922269674
1995 05 14 15 48 35 09:01:42 -46:06:05 J2000 APO3.5 -1 -0.0349065555555556 800399419 800424538
2054 08 18 09 39 07 04:35:19 +37:01:06 J2000 CFHT 0 -0.104719666666667 2670658531 2670627103
1987 05 12 00 01 39 04:06:19 +61:39:48 B1950 CAMB1MILE 1 0.0349065555555556 undef undef
2058 01 25 00 01 05 05:24:59 -84:24:13 J2000 IRTF 0 -0.104719666666667 undef undef
2003 12 18 23 05 19 11:01:11 -09:39:05 J2000 ESOSCHM 1 0 1071804784 1071850488
2022 04 09 02 42 06 02:05:59 +20:52:11 J2000 VLT2 1 0.0349065555555556 1649507277 1649544365
2007 11 14 07 09 06 06:59:13 +16:01:41 J2000 VLA -1 0 1195012430 1194974710
2023 04 12 01 52 59 15:19:13 +25:07:01 J2000 GEMINIS -1 0.0523598333333333 1681178777 1681212373
2003 12 04 01 41 19 01:11:16 +03:16:09 B1950 STEWARD90 -1 0.104719666666667 1070489435 1070444028
2011 02 14 21 07 24 05:52:35 +18:19:52 J2000 MAUNAK88 -1 -0.0349065555555556 1297641838 1297689286
2011 02 14 05 05 48 15:24:39 +25:41:05 B1950 KPNO84 1 -0.349065555555556 1297742532 1297722448
2057 01 09 11 39 48 08:24:42 +29:59:09 B1950 FLAGSTF61 1 0 2746314207 2746282441
2008 09 25 19 05 24 20:01:19 +32:16:41 B1950 SUGARGROVE 1 0 1222448138 1222419611
2016 04 07 16 24 59 02:07:13 -51:41:13 B1950 MTEKAR 1 0.349065555555556 undef undef
2009 10 14 05 42 09 13:48:16 +16:35:16 B1950 IRTF -1 -0.174532777777778 1255448066 1255413275
1994 11 01 15 11 19 09:01:42 +38:13:01 B1950 HPROV1.52 0 0 783722548 783702876
2079 07 12 10 01 35 04:01:24 +32:06:52 B1950 AAT 1 0.0349065555555556 3456412387 3456443133
2016 11 19 10 16 52 02:01:35 +51:39:16 B1950 GEMININ -1 0 1479516250 1479486308
2063 05 18 02 19 52 03:01:42 -61:48:16 B1950 ESOSCHM 0 -0.174532777777778 undef undef
2023 09 07 12 11 06 16:39:13 +20:52:16 B1950 MAGELLAN1 1 0 1694106453 1694143735
2071 06 12 17 16 16 20:07:01 -32:24:19 B1950 NOBEYAMA -1 0.349065555555556 3201351896 3201274271
2019 06 14 02 16 06 08:59:06 +51:07:41 B1950 GEMININ -1 -0.349065555555556 undef undef
2026 11 09 19 41 06 17:39:06 +73:16:07 J2000 CAMB5KM 0 -0.0174532777777778 undef undef
2044 08 23 06 11 16 13:06:39 +38:42:42 J2000 DUPONT -1 -0.174532777777778 2355488906 2355526830
1997 01 12 22 35 35 15:41:24 +73:35:42 B1950 VLT1 -1 -0.174532777777778 853066624 853080681
1995 11 19 03 07 35 03:48:41 -41:24:16 B1950 JODRELL1 -1 -0.0523598333333333 undef undef
2035 10 12 00 59 59 16:05:39 -01:19:48 J2000 SANPM83 0 0 2075733140 2075775814
2033 08 09 00 24 06 06:48:35 -09:42:52 B1950 MAUNAK88 1 0 2007209710 2007164916
2031 05 14 08 48 41 22:16:05 +01:06:59 J2000 MAGELLAN2 0 0 1936503370 1936546115
2045 06 04 02 42 48 05:06:13 +16:06:48 J2000 QUEBEC1.6 0 -0.349065555555556 2380174044 2380156502
2022 08 19 11 06 16 02:19:01 +01:19:42 J2000 LPO2.5 0 -0.0349065555555556 1660865411 1660909963
2033 09 14 17 42 05 09:39:39 +38:01:16 B1950 TOLOLO1.5M -1 0.0523598333333333 2010308375 2010250147
2031 01 09 22 07 39 18:09:42 +12:48:06 B1950 MOPRA 1 0 1925839698 1925792813
1990 06 07 09 59 19 15:05:19 +46:01:13 B1950 UKIRT -1 0 644720441 644687800
2026 12 18 02 52 19 11:06:01 -61:05:01 J2000 ATCA 0 -0.0349065555555556 undef undef
1992 06 25 07 13 52 17:01:16 -29:48:13 J2000 VLA 1 -0.349065555555556 709515215 709474563
2045 08 14 05 05 05 00:16:07 +03:41:39 J2000 DUPONT 1 -0.104719666666667 2386372029 2386331186
2068 04 18 16 05 41 09:19:42 +37:48:01 J2000 DUNLAP74 -1 -0.0523598333333333 3101987242 3101970970
2037 02 12 15 59 59 18:39:48 +61:05:24 B1950 ANU2.3 1 0 undef undef
2032 09 14 03 48 41 12:06:01 +51:48:39 B1950 VLA 0 -0.349065555555556 undef undef
2024 01 14 09 09 07 10:59:19 +25:05:09 J2000 KISO 0 -0.0349065555555556 1705229183 1705196882
2017 11 19 05 06 41 11:09:06 -12:52:41 J2000 PALOMAR60 -1 0 1510998106 1511037015
1991 02 23 08 19 39 21:01:48 +49:13:41 B1950 ATCA 0 0.174532777777778 667268663 667272364
2029 09 25 03 06 42 15:42:39 +38:13:13 J2000 CATALINA61 -1 0.0523598333333333 1884957146 1884925943
2073 08 14 11 35 52 07:07:39 +73:39:24 J2000 MAGELLAN2 1 0.0523598333333333 undef undef
2000 09 23 07 06 48 13:06:59 -18:13:09 J2000 ANU2.3 0 -0.174532777777778 969650708 969705491
2048 12 18 22 09 19 14:19:07 +01:06:35 B1950 STEWARD90 1 0.0523598333333333 2491985297 2492026884
2071 02 18 14 07 41 03:52:41 +51:13:35 B1950 ANU2.3 0 -0.349065555555556 3191452847 3191492583
2042 12 14 16 24 52 11:05:06 -73:41:06 J2000 KPNO84 0 0 undef undef
2066 01 23 07 19 48 19:19:06 -46:52:01 B1950 MAUNAK88 -1 -0.0174532777777778 3031405386 3031438524
2059 03 19 04 06 11 16:39:48 +09:07:13 J2000 MTLEMMON60 0 -0.0523598333333333 2815278037 2815239461
2024 12 12 03 06 11 06:19:06 +20:52:48 J2000 MAGELLAN2 -1 -0.0523598333333333 1733962351 1733915211
2016 08 23 06 39 59 06:11:19 +58:13:19 B1950 TOLOLO4M 0 -0.104719666666667 1471945926 1471967439
1982 09 04 05 24 16 14:41:09 +12:09:16 J2000 MCDONLD2.1 -1 0 399917816 399964435
1992 12 14 20 07 05 06:35:07 +01:52:35 J2000 EFFELSBERG -1 -0.349065555555556 724349498 724323325
2057 11 09 02 09 07 00:59:24 +84:09:06 B1950 SAAO74 0 -0.0174532777777778 undef undef
2062 04 25 00 48 35 18:52:07 +29:39:39 B1950 ANU2.3 0 0.0349065555555556 2913114099 2913146047
2020 03 14 11 59 16 05:35:05 +03:11:06 J2000 LPO1 0 -0.349065555555556 1584185865 1584154714
1985 10 09 20 06 59 03:24:52 +73:48:24 J2000 KECK2 -1 0.174532777777778 497679190 497652995
2020 04 09 19 19 07 01:35:41 -18:59:35 J2000 CAMB1MILE 0 -0.0174532777777778 1586419280 1586450757
2017 08 09 19 09 48 20:19:06 -03:52:48 B1950 ATCA 0 0 1502348405 1502306349
1998 04 04 13 16 11 22:11:01 +01:05:11 J2000 KECK2 -1 -0.104719666666667 891695706 891655865
2036 07 04 16 41 39 23:48:39 -46:07:09 J2000 MCDONLD2.1 1 0.174532777777778 2098865317 2098877530
2012 09 04 16 07 07 04:59:13 -51:24:05 J2000 VLA 1 -0.174532777777778 1346836401 1346778612
2025 07 23 20 19 39 13:35:09 +12:48:41 B1950 DAO72 -1 0.0349065555555556 1753297225 1753259535
2059 10 25 01 16 16 09:06:48 -16:11:24 B1950 CFHT 1 -0.104719666666667 2834306186 2834349495
2075 10 09 22 52 11 05:59:09 -16:35:24 B1950 ANU2.3 1 0.349065555555556 3337940761 3337891129
1982 09 09 09 48 06 12:35:11 -20:06:13 B1950 PALOMAR200 1 0.0523598333333333 400436580 400471017
1983 10 19 17 11 16 01:09:41 +03:39:05 J2000 LICK120 -1 -0.174532777777778 435371154 435421633
1986 11 01 06 59 48 11:01:39 +12:07:11 J2000 HARVARD 0 0 531210029 531172415
2065 03 23 06 39 24 19:59:07 +01:35:01 J2000 MTEKAR 0 -0.174532777777778 3004992397 3005043295
2076 07 14 14 48 39 02:39:07 -29:09:48 B1950 SANPM83 1 -0.0349065555555556 3362032524 3361981598
2012 02 14 05 06 16 17:07:52 -16:06:24 J2000 ESO3.6 1 -0.104719666666667 1329196215 1329247272
2018 05 18 20 24 16 21:24:13 -46:06:09 B1950 PARKES 1 0 1526727585 1526704560
2043 01 01 20 06 39 06:35:19 -01:06:59 B1950 MAGELLAN2 -1 -0.174532777777778 2303675548 2303724439
2027 06 01 22 24 13 18:11:01 -03:19:16 J2000 IRTF 1 0 1811915832 1811958342
2068 08 09 22 42 16 08:16:16 +29:11:41 J2000 MAGELLAN1 -1 0 3111735581 3111770101
2020 06 23 05 19 16 21:09:11 +51:05:07 B1950 CAMB5KM 0 0.174532777777778 undef undef
2058 09 25 02 39 48 01:06:39 +03:39:52 B1950 ST.ANDREWS 0 -0.0174532777777778 2800118116 2800165132
2035 11 12 12 24 09 07:09:11 +12:52:06 J2000 KPNO158 -1 -0.349065555555556 2078449101 2078422165
2078 05 25 02 48 09 20:13:39 +12:16:39 J2000 HPROV1.52 0 0 3420651219 3420700213
2044 12 01 19 41 48 11:01:42 +12:13:39 B1950 ESO3.6 -1 0.174532777777778 2364186063 2364220228
1998 06 09 07 13 52 22:42:09 +16:16:39 J2000 MAGELLAN1 -1 0 897367958 897320418
2030 08 23 07 19 42 17:19:13 +20:35:42 J2000 MOPRA 1 0 1913774813 1913725425
2070 06 19 14 39 35 16:07:42 +73:42:11 B1950 QUEBEC1.6 0 0.349065555555556 undef undef
2033 08 23 01 16 42 13:09:16 +25:07:35 B1950 MAUNAK88 1 0.349065555555556 2008441178 2008391605
1993 02 18 01 01 13 01:41:35 +25:09:35 J2000 TIDBINBLA 1 -0.104719666666667 730082014 730033643
2061 03 25 18 42 13 20:42:19 -03:11:05 J2000 JCMT 0 0 2878981056 2879023627
2042 05 18 17 52 24 14:42:16 -51:42:01 J2000 HPROV1.93 0 -0.349065555555556 2284050314 2284080334
1992 06 18 11 42 13 23:52:59 -32:09:05 J2000 SAAO74 -1 -0.0174532777777778 708815020 708783934
1985 05 18 05 24 41 08:09:42 -38:41:11 J2000 MMT 0 0.0174532777777778 485207769 485235857
2044 09 09 09 11 07 21:07:01 -38:41:13 J2000 KISO 0 0.0349065555555556 2357025532 2357049969
2037 12 19 12 11 13 16:09:52 -41:52:06 J2000 VLT3 0 0 2144820223 2144874957
2079 03 04 22 16 48 02:19:05 +58:16:52 B1950 CSO -1 0 3445176789 3445151246
2016 01 14 17 48 59 19:06:24 -61:48:52 J2000 AAT 0 -0.349065555555556 undef undef
1985 02 14 12 01 07 22:09:52 -29:24:16 B1950 OKAYAMA -1 -0.174532777777778 477180902 477220020
2013 05 18 08 39 39 08:05:19 +18:48:48 B1950 MCDONLD2.1 1 0 1368894770 1368943372
2019 06 12 05 06 35 11:09:59 -46:07:11 J2000 TAUTNBG 0 -0.0349065555555556 undef undef
1988 10 09 07 07 07 12:06:42 -73:48:52 J2000 VLT3 0 0.349065555555556 592389727 592352956
2053 07 23 10 19 59 05:41:39 +38:01:59 J2000 JODRELL1 -1 -0.0523598333333333 undef undef
2076 03 19 16 39 13 08:39:11 +61:41:35 B1950 SAAO74 -1 -0.104719666666667 3351779586 3351792089
2045 12 23 18 01 07 11:13:13 +51:24:52 J2000 SUGARGROVE -1 0.104719666666667 2397603694 2397584955
1982 04 19 12 24 42 01:11:41 +29:06:52 J2000 PALOMAR48 0 0 388064605 388031776
2056 06 18 12 09 09 13:59:09 +84:39:39 B1950 EFFELSBERG 1 0 undef undef
2044 08 12 17 16 09 23:01:19 +29:39:06 B1950 BOSQALEGRE 0 0.0174532777777778 2354664309 2354610501
2070 03 01 14 19 16 11:52:01 +37:06:48 J2000 ESO3.6 1 0 3160949793 3160981051
2011 12 12 09 19 19 10:16:01 -84:06:01 B1950 MCDONLD2.7 0 -0.0523598333333333 undef undef
2079 05 01 13 41 16 15:42:24 -37:19:41 J2000 USSR600 1 0 3450195124 3450215417
2045 12 12 09 11 01 20:59:39 +16:35:06 J2000 MTHOP1.5 1 0 2396708120 2396756332
2025 06 09 13 11 13 10:41:09 -51:11:19 B1950 LICK120 0 0 1749515590 1749436985
2021 08 12 15 16 41 05:07:48 -12:19:24 B1950 QUEBEC1.6 -1 -0.349065555555556 1628746046 1628710709
1989 09 23 06 16 35 06:48:11 +25:24:59 J2000 ANU2.3 -1 0.174532777777778 622486131 622514416
2078 11 09 13 13 24 09:07:35 +46:24:01 J2000 SUBARU 1 0.0174532777777778 3435296021 3435262675
2078 01 07 08 13 39 00:09:52 +61:13:48 B1950 MAGELLAN1 -1 0.104719666666667 undef undef
1995 09 07 14 59 48 17:05:09 +51:42:59 B1950 KISO 1 -0.0523598333333333 undef undef
2020 02 01 19 05 59 19:42:06 +37:52:06 B1950 BOSQALEGRE 0 0.174532777777778 1580560138 1580580367
2046 12 07 14 59 09 18:59:01 +32:39:06 B1950 CFHT 0 0.0523598333333333 2427817176 2427778671
2067 08 07 04 41 24 08:48:41 +09:16:39 J2000 KOSMA3M 1 0 3080003773 3079965219
2009 01 18 02 48 52 19:24:24 -12:52:13 B1950 ANU2.3 -1 0 1232219430 1232180130
2057 04 09 04 42 24 00:11:06 -29:41:16 B1950 AAT 1 -0.0349065555555556 2754063352 2754031102
2016 05 23 22 16 11 09:07:48 +16:01:09 J2000 VLA 1 0 1464110617 1464072872
2005 06 09 10 59 42 23:16:09 +18:42:06 J2000 CSO -1 0 1118311125 1118271403
2058 06 25 01 59 52 15:13:35 +49:35:24 B1950 ESONTT -1 0.0349065555555556 2792184452 2792119710
2002 06 14 13 09 07 02:52:41 -32:39:41 B1950 MAUNAK88 0 -0.0174532777777778 1024065220 1024102578
1996 03 04 20 16 35 02:16:52 +84:41:05 J2000 TAUTNBG -1 0 undef undef
2043 10 12 05 41 05 11:39:19 -01:19:05 J2000 TIDBINBLA 0 0 2328200414 2328244027
1986 09 01 22 19 01 18:01:06 +84:41:16 B1950 MCDONLD2.1 1 0 undef undef
2048 08 18 22 59 41 08:07:06 +73:16:42 J2000 VLA 0 0 undef undef
2029 07 09 00 06 06 09:16:19 -01:05:19 B1950 ANU2.3 0 0 1878243124 1878286620
2066 12 04 01 05 42 19:52:09 -18:42:42 B1950 TIDBINBLA 1 0.104719666666667 3058728208 3058687908
1997 08 18 15 06 39 07:13:42 +73:24:39 J2000 VLT4 -1 -0.0174532777777778 undef undef
2038 02 01 03 41 52 20:52:19 +86:41:41 J2000 KOSMA3M 0 0 undef undef
2078 09 18 17 48 48 15:11:42 +25:11:41 J2000 PARKES 0 -0.104719666666667 3430771593 3430723990
2069 11 04 23 13 11 20:24:41 +84:13:13 J2000 MAGELLAN1 1 0.349065555555556 undef undef
2037 03 23 20 48 52 05:07:24 +49:11:05 B1950 VLA -1 0 2121432687 2121414416
2055 05 25 08 11 13 20:11:59 -01:13:59 J2000 TAUTSCHM -1 0.0523598333333333 2694807747 2694761751
1986 06 19 13 48 19 09:19:59 +32:05:01 B1950 KOSMA3M 0 -0.104719666666667 519542472 519612029
2026 10 14 04 06 06 17:07:42 +61:13:16 B1950 MCDONLD2.1 1 -0.0174532777777778 undef undef
2022 05 14 22 52 06 12:01:39 +46:06:19 J2000 KISO 0 -0.0174532777777778 1652579826 1652561162
1993 08 09 03 19 42 14:16:42 +03:39:06 J2000 LICK120 1 0.0523598333333333 744923619 744880074
2005 12 19 00 11 52 11:24:09 -49:13:41 B1950 GEMINIS 1 0 1134955578 1135019294
2006 10 12 12 11 35 17:41:11 +58:48:39 B1950 SANPM83 1 0.0349065555555556 1160660111 1160734870
2050 01 12 10 06 41 05:42:42 +73:48:39 J2000 ESO3.6 -1 0.0523598333333333 undef undef
1981 01 19 17 39 52 22:16:09 +09:07:41 B1950 PALOMAR60 -1 0 348767076 348726951
2035 01 19 20 16 05 20:42:42 -32:09:39 B1950 LPO4.2 1 -0.0174532777777778 2052897308 2052931528
2052 11 25 07 01 24 22:42:05 -84:59:05 J2000 JCMT 1 0 undef undef
2040 02 07 02 24 52 16:11:06 +46:24:48 J2000 VLT4 0 0.0523598333333333 2212214562 2212155075
2051 04 01 07 16 06 00:13:05 -01:35:39 J2000 CSO -1 0.0349065555555556 2563891920 2563933758
1986 05 18 02 41 11 03:41:35 +18:19:59 J2000 MTEKAR 1 -0.349065555555556 516849157 516834339
2025 06 04 11 59 16 04:11:35 -29:16:19 B1950 UKST 1 0 1749060336 1749112869
2024 02 14 18 13 39 00:07:52 -61:07:05 B1950 AAT 1 -0.174532777777778 undef undef
2079 08 09 10 09 24 12:07:59 +51:52:39 J2000 CSO 0 -0.174532777777778 3458822807 3458802914
2027 03 25 03 59 42 02:35:39 -73:42:19 J2000 NOBEYAMA 1 0.0349065555555556 undef undef
2006 07 18 13 19 39 13:41:19 +84:09:11 J2000 LPO2.5 0 0 undef undef
2065 09 09 14 07 06 14:41:13 -46:07:59 J2000 USSR600 -1 0.0523598333333333 undef undef
2025 11 04 05 01 24 01:05:13 -25:05:59 J2000 CFHT 0 0 1762225950 1762264408
2080 01 01 06 11 07 16:01:16 +51:48:16 B1950 USSR6 -1 0.104719666666667 3471277072 3471269826
2032 06 04 06 19 19 10:42:06 +84:52:11 B1950 PARKES 1 0.0349065555555556 undef undef
2033 06 04 09 39 16 04:42:06 -18:11:48 B1950 TAUTNBG 1 0.104719666666667 2001569019 2001509077
2061 03 04 09 52 41 00:19:13 +41:16:41 B1950 ST.ANDREWS 0 0.174532777777778 2877131719 2877121108
2049 09 25 22 24 06 07:16:16 -37:05:39 B1950 LPO4.2 -1 -0.104719666666667 2516152554 2516188198
2039 06 14 06 07 48 11:52:24 -20:16:16 B1950 MOPRA 0 0 2191628303 2191677760
1981 08 01 04 24 35 13:59:42 -29:13:01 B1950 PALOMAR60 -1 0 365459903 365406415
1995 06 18 15 35 42 18:01:35 +38:05:13 J2000 STROMLO74 0 -0.349065555555556 803463971 803506429
2037 08 12 01 09 59 08:06:39 -03:06:16 B1950 GEMINIS -1 0.174532777777778 2133598292 2133636771
2056 01 14 13 35 19 15:06:07 -09:16:42 B1950 KPNO84 -1 0 2715067694 2715021689
2067 10 14 02 24 59 12:42:06 -09:01:42 B1950 JODRELL1 -1 0.349065555555556 3085722294 3085739584
2011 10 07 03 05 05 00:59:11 +49:39:35 B1950 CAMB1MILE 0 0 undef undef
2023 01 25 08 01 05 04:41:07 +41:09:24 J2000 AAT -1 0.0174532777777778 1674629106 1674569756
2079 06 18 21 59 07 02:05:05 +73:13:24 B1950 BOSQALEGRE -1 0.0349065555555556 undef undef
2022 11 07 06 07 42 12:52:01 +86:24:52 J2000 TAUTNBG 0 0.174532777777778 undef undef
2034 08 14 17 09 09 16:07:09 +29:52:06 J2000 BLOEMF 1 -0.0523598333333333 2039255082 2039205104
1985 02 25 05 01 24 04:19:09 -20:35:52 B1950 GEMININ 1 -0.349065555555556 478214470 478178411
2028 12 14 22 48 13 21:07:07 +12:48:39 B1950 HARVARD 1 0 1860499727 1860462557
1983 12 12 22 52 52 10:06:35 +51:41:13 B1950 CATALINA61 1 -0.0349065555555556 440129233 440200996
2041 06 18 11 05 24 02:07:24 +32:39:16 J2000 OKAYAMA 0 -0.0349065555555556 2255182059 2255153193
2030 03 25 10 11 41 21:24:24 -41:07:59 J2000 ATCA 0 0 1900682004 1900653544
1981 04 25 12 11 52 03:24:11 +32:06:42 B1950 OKAYAMA 1 0.0349065555555556 357079574 357133472
2016 12 12 09 42 35 15:24:16 +73:42:24 B1950 QUEBEC1.6 0 -0.0349065555555556 undef undef
2013 09 25 17 41 24 11:52:35 -86:35:01 B1950 CAMB1MILE 1 -0.349065555555556 undef undef
1982 09 25 11 42 09 00:48:39 -49:19:39 J2000 GEMINIS 0 0.0523598333333333 401834971 401809114
2012 01 04 12 01 01 04:07:07 -03:52:39 J2000 MOPRA 0 -0.104719666666667 1325652052 1325699636
2038 04 19 04 52 48 22:11:42 +25:01:41 B1950 VLT4 0 -0.0349065555555556 2155276067 2155228141
1984 05 04 16 19 13 12:48:42 -51:05:19 B1950 UKIRT 1 0 452577992 452608287
2012 06 04 23 39 19 01:24:39 +29:16:19 J2000 VLT4 0 0.174532777777778 1338887042 1338830332
2014 01 23 08 41 48 00:41:07 -58:39:01 B1950 KECK1 1 -0.174532777777778 1390513622 1390550268
1997 09 23 05 59 52 11:11:16 -84:13:39 B1950 TAUTNBG 1 0.349065555555556 undef undef
2066 04 14 01 41 19 20:06:48 -58:24:06 B1950 PALOMAR60 0 -0.104719666666667 3038472693 3038403306
2049 02 25 05 06 24 21:07:39 +18:19:48 J2000 UKST 0 -0.104719666666667 2497806620 2497847655
2027 06 19 09 11 59 12:06:35 +61:07:05 B1950 BOSQALEGRE 0 0 undef undef
2074 01 25 20 05 16 00:07:19 -20:11:11 B1950 TIDBINBLA 0 0.174532777777778 3284149749 3284107372
2070 04 23 19 39 59 08:11:52 +12:07:05 J2000 DUPONT -1 0.0349065555555556 3165499583 3165452161
2042 01 18 23 41 11 14:19:35 +25:07:06 J2000 TOLOLO4M -1 -0.0174532777777778 2273638257 2273674471
2011 09 04 16 19 24 12:16:24 +16:52:13 B1950 TOLOLO1.5M 0 -0.104719666666667 1315138817 1315180628
2035 12 25 04 01 05 23:13:52 -09:06:13 B1950 PALOMAR200 -1 0.104719666666667 2082138328 2082088978
2072 01 12 05 06 07 08:52:05 +25:24:42 B1950 JCMT 1 0.349065555555556 3219806914 3219843538
2075 01 19 00 24 11 10:13:01 -29:41:39 B1950 APO3.5 1 0.104719666666667 3315101547 3315129613
2044 04 25 17 59 07 14:06:35 -41:07:48 J2000 QUEBEC1.6 -1 0 2345165540 2345178286
1990 12 07 11 35 13 00:16:01 -32:05:59 B1950 MCDONLD2.1 -1 -0.0174532777777778 660519195 660552620
2077 07 23 00 59 13 06:59:07 +46:48:59 B1950 VLA -1 -0.0174532777777778 3394170161 3394150200
1992 06 23 22 05 24 11:06:16 +61:19:41 J2000 GEMINIS -1 0 undef undef
2013 06 07 12 06 09 22:19:41 +32:39:07 B1950 DAO72 1 -0.104719666666667 1370660446 1370649531
2022 08 23 21 41 06 10:06:52 +29:41:05 B1950 JODRELL1 -1 0 1661223573 1661290001
1990 09 14 04 01 52 12:52:59 -09:13:48 J2000 KECK2 1 0 653334837 653290156
2075 02 01 20 07 09 01:59:35 +51:19:06 J2000 JODRELL1 0 0.0174532777777778 undef undef
2068 06 19 12 59 39 10:16:52 +20:09:19 B1950 SUBARU 0 0 3107362815 3107323255
2044 12 25 04 07 52 20:13:06 +12:13:48 J2000 BOSQALEGRE 0 0.104719666666667 2366284499 2366234189
2057 04 01 02 09 16 21:13:35 +49:07:11 B1950 TAUTNBG -1 -0.0174532777777778 undef undef
2046 09 07 01 59 48 06:01:59 -09:19:06 J2000 BLOEMF -1 0.0174532777777778 2419887452 2419846315
2069 10 09 04 39 52 04:09:59 -41:01:06 J2000 ATCA 1 -0.104719666666667 3148531990 3148595038
2031 03 01 20 09 01 18:06:48 +49:35:06 B1950 AAT 1 0 1930242796 1930177917
2062 05 23 07 01 42 01:06:42 +84:05:39 J2000 HARVARD 0 0 undef undef
2040 12 25 05 01 41 19:09:48 +16:41:01 J2000 ESO3.6 -1 0 2239964326 2240002762
2076 08 19 07 42 48 02:11:05 +37:11:59 J2000 KPNO84 -1 0 3365034833 3365005471
2046 12 12 23 06 11 06:19:05 -41:42:35 J2000 KOTTAMIA 0 0 2428253419 2428281711
2047 04 25 06 06 35 22:39:42 -86:41:59 B1950 CATALINA61 0 -0.0349065555555556 undef undef
2069 01 04 07 13 09 05:16:11 -73:19:01 J2000 MCDONLD2.7 -1 0 undef undef
1992 05 23 09 09 35 02:35:35 -29:35:48 B1950 LICK120 1 0.349065555555556 706640866 706651629
1989 02 14 02 11 39 00:41:48 +20:06:41 J2000 UKIRT 0 -0.104719666666667 603397809 603447817
2024 09 04 09 39 05 22:16:24 -03:35:07 J2000 DUPONT 0 0.174532777777778 1725403462 1725441987
1981 09 07 01 01 05 08:35:48 -12:24:59 B1950 MOPRA -1 0 368644023 368604653
2040 12 18 21 06 52 19:59:24 +84:35:06 B1950 KECK2 0 -0.0523598333333333 undef undef
2013 04 09 09 42 42 04:16:42 +46:11:05 J2000 CFHT 0 0 1365530315 1365497810
2005 10 04 23 42 19 10:42:11 +84:05:59 B1950 BLOEMF 0 0 undef undef
2059 06 18 09 52 01 20:16:09 -73:05:48 B1950 AAT 1 0 undef undef
1989 05 14 18 19 59 08:52:35 +12:16:48 B1950 TAUTSCHM 1 0 611228039 611192320
2074 09 23 03 24 41 21:16:13 -84:52:24 B1950 UKIRT 1 -0.349065555555556 3304978693 3304936960
2058 01 18 04 24 48 23:19:24 -38:48:35 J2000 VLT2 1 -0.0349065555555556 2778582821 2778637599
2036 06 07 19 06 09 15:01:16 -46:41:24 B1950 APO3.5 0 0 2096503031 2096439021
2076 03 19 07 11 16 02:05:24 -49:06:06 B1950 LPO4.2 0 0 3351845003 3351869723
1990 11 14 06 13 01 03:35:01 +46:06:41 J2000 MAGELLAN1 -1 0.0174532777777778 658545269 658484373
2004 08 12 08 05 07 00:42:48 +09:24:42 J2000 MTEKAR 1 0.0523598333333333 1092341310 1092300839
1993 01 04 22 13 41 22:11:35 +51:01:59 J2000 DUNLAP74 -1 0.104719666666667 726140442 726132331
2063 11 04 13 16 59 19:41:59 -01:24:42 J2000 UKST -1 0 2961363157 2961406605
2004 11 09 21 11 13 09:35:06 +73:11:41 B1950 JCMT -1 -0.0174532777777778 undef undef
2011 01 14 04 24 16 20:59:19 +32:39:42 B1950 JODRELL1 1 0.0349065555555556 1295064165 1295046360
1988 05 19 08 06 39 16:48:16 -20:06:41 B1950 EFFELSBERG -1 0.174532777777778 579995151 580015497
2024 11 01 16 35 42 09:11:06 -16:48:19 J2000 GEMINIS 1 0.0349065555555556 1730522182 1730482813
2001 09 23 04 42 59 20:09:41 +86:07:52 B1950 ANU2.3 0 0.104719666666667 undef undef
2053 03 07 21 07 05 07:52:11 +86:42:05 J2000 STEWARD90 -1 0 undef undef
2058 06 01 17 09 39 02:35:01 +20:05:11 B1950 FLAGSTF61 1 0.0349065555555556 2790239574 2790202539
2009 11 12 17 48 19 12:39:24 +09:48:11 B1950 CSO -1 0.0174532777777778 1258032455 1257990508
2007 04 04 11 42 24 03:19:39 -73:24:11 B1950 ATCA 0 -0.0174532777777778 undef undef
2066 07 19 10 39 42 18:06:09 -18:05:01 B1950 JODRELL1 -1 0.0523598333333333 3046704132 3046731948
2025 01 09 16 06 16 21:39:06 -09:16:19 J2000 MAGELLAN1 0 0.0523598333333333 1736427558 1736471431
2014 01 09 07 52 19 05:41:48 +18:48:01 J2000 DUNLAP74 0 0 1389213041 1389265267
2038 11 12 11 24 24 09:42:59 -73:09:59 B1950 VLT4 0 0 undef undef
1991 07 25 05 24 41 12:05:24 +46:11:06 B1950 ESO3.6 0 0.174532777777778 680380936 680395320
2008 06 14 13 41 24 06:42:48 -12:06:16 B1950 STEWARD90 0 0.104719666666667 1213457945 1213493798
2030 11 18 21 13 05 21:13:41 +51:24:24 B1950 TAUTNBG 1 0 undef undef
2017 06 04 20 16 24 09:07:07 -29:42:48 B1950 ESONTT 1 -0.349065555555556 1496662376 1496643656
2050 12 12 11 06 41 09:07:39 -20:39:48 J2000 KISO -1 0 2554378884 2554414328
2002 03 09 02 42 06 00:24:59 -25:35:48 B1950 DUPONT 1 0 1015671703 1015722061
1990 09 18 16 48 39 18:59:24 -86:01:19 J2000 SAAO74 0 0 undef undef
2032 06 07 12 09 11 17:42:52 -01:35:59 J2000 CFHT -1 0.349065555555556 1970202564 1970148973
1982 05 18 08 52 07 10:11:07 +20:41:05 J2000 STROMLO74 -1 0.0349065555555556 390541524 390489666
2035 08 14 22 24 07 20:24:11 +61:07:41 J2000 LOWELL72 1 -0.0523598333333333 undef undef
2079 07 01 01 42 19 13:42:11 -01:07:09 J2000 DAO72 -1 0 3455386219 3455342316
2035 02 12 01 39 13 16:35:09 +16:13:01 J2000 CFHT -1 0.349065555555556 2054810195 2054845662
1996 05 12 13 35 06 18:16:39 +09:52:07 B1950 JCMT 1 -0.349065555555556 831965603 831934770
2043 09 04 09 42 06 00:24:42 +84:05:59 J2000 IRTF 0 0.174532777777778 undef undef
2012 03 14 21 19 09 20:48:39 +12:16:19 B1950 TIDBINBLA 1 0.0349065555555556 1331834832 1331786186
2043 06 23 08 11 24 10:48:48 -73:06:52 B1950 UKST 0 0.104719666666667 undef undef
2010 09 25 06 19 48 18:24:06 -18:07:11 B1950 KPNO158 -1 -0.0349065555555556 1285359406 1285311905
2004 01 01 07 48 09 10:48:39 -20:39:39 J2000 EFFELSBERG 1 0 1072999595 1072943456
2075 04 01 16 07 52 22:11:35 -58:59:42 J2000 DUNLAP74 1 0 undef undef
2018 07 19 00 01 09 19:39:05 -16:19:35 J2000 GEMININ -1 -0.0523598333333333 1531888147 1531929931
2052 08 25 03 39 05 16:35:41 -46:24:42 B1950 STROMLO74 -1 0 2608154234 2608134392
2077 08 18 20 52 11 18:48:01 -03:24:42 J2000 VLT1 0 -0.104719666666667 3396539395 3396586378
2026 10 04 03 01 48 18:39:13 -20:13:13 J2000 SANPM83 1 0 1791145181 1791095976
2004 11 25 21 11 06 12:16:59 -58:39:05 B1950 TIDBINBLA 0 0 undef undef
1984 02 19 10 42 24 09:13:42 +03:11:52 B1950 MTLEMMON60 0 0 445999071 446043085
2050 01 07 23 59 59 19:16:24 -03:06:01 B1950 LPO1 1 -0.0349065555555556 2525239092 2525282502
1990 12 25 12 13 13 11:39:48 -37:05:41 B1950 NOBEYAMA -1 0.174532777777778 662061430 662078032
2003 08 09 00 13 13 07:35:48 +25:42:41 B1950 TOLOLO1.5M 0 0 1060424216 1060373398
2030 08 18 10 35 41 12:07:59 -29:35:13 B1950 UKIRT -1 0.0349065555555556 1913226439 1913262554
2036 06 25 19 01 16 00:59:06 +61:11:16 B1950 MCDONLD2.7 1 0.0174532777777778 undef undef
2061 12 25 02 11 06 09:42:59 +20:07:41 B1950 JODRELL1 1 0.0349065555555556 2902766458 2902735110
2072 09 07 00 11 39 12:48:39 +73:52:16 J2000 SANPM83 -1 0 undef undef
2046 12 14 21 24 59 23:16:24 -18:48:41 B1950 MCDONLD2.1 -1 0 2428428495 2428380004
2049 10 01 08 41 05 19:01:06 -01:05:24 J2000 KOSMA3M 0 0 2516702045 2516658463
2072 09 01 07 07 52 05:48:39 -12:48:06 J2000 IRTF -1 -0.0174532777777778 3239869572 3239910932
2019 01 12 15 11 06 12:48:19 -09:35:19 J2000 JODRELL1 -1 -0.104719666666667 1547250333 1547292005
1989 08 12 03 06 42 10:06:35 -29:06:35 B1950 KECK1 0 -0.349065555555556 618855615 618904629
2067 07 25 05 48 11 15:48:41 +09:24:48 B1950 IRTF 1 -0.0174532777777778 3078863021 3078822034
2021 02 23 00 39 07 00:42:05 +09:41:48 J2000 MAUNAK88 1 -0.104719666666667 1614103757 1614065502
2065 12 12 01 09 48 17:35:19 +09:24:41 B1950 CFHT 0 0.104719666666667 3027775862 3027817483
2078 03 23 05 41 13 22:09:59 +25:16:39 B1950 SUBARU -1 0.104719666666667 3415184637 3415229136
2031 12 09 13 09 09 01:01:05 +09:07:11 J2000 QUEBEC1.6 0 0.349065555555556 1954612229 1954559987
2077 10 09 01 07 16 07:42:07 +73:24:41 B1950 GEMININ 0 0.349065555555556 3401003921 3400959078
2011 02 25 10 06 24 06:41:16 -41:07:48 B1950 LICK120 0 -0.104719666666667 1298593792 1298622729
2077 10 14 23 52 41 16:16:52 +29:48:48 J2000 LPO4.2 -1 -0.0349065555555556 3401426002 3401479126
2001 11 25 20 39 06 15:19:07 +18:01:35 B1950 LPO1 0 0 1006752683 1006714473
2043 09 01 01 07 06 09:59:52 -16:52:24 B1950 GEMINIS 0 0 2324712215 2324674149
2052 01 07 20 01 35 08:35:35 +03:05:09 J2000 VLA -1 -0.349065555555556 2588201827 2588257641
2057 05 19 01 19 11 22:52:41 -38:13:59 B1950 HARVARD 1 0.0174532777777778 2757488684 2757509032
2056 11 01 01 05 39 03:06:19 +16:35:39 B1950 JCMT 1 0 2740278247 2740324358
2073 10 09 03 06 19 23:05:11 -61:42:06 J2000 KOSMA3M 0 0.104719666666667 undef undef
2042 08 25 13 59 16 22:07:01 -61:24:07 B1950 IRTF -1 0.0349065555555556 2292564396 2292585221
2066 02 07 08 41 13 12:59:11 -51:52:41 B1950 MMT 0 0.0349065555555556 3032760062 3032774095
1986 03 12 21 09 07 09:01:01 +38:13:06 B1950 KECK1 1 0 511059486 511110423
2015 03 04 12 13 52 00:05:06 -32:06:41 B1950 KOTTAMIA -1 0.174532777777778 1425454754 1425394299
2011 02 25 08 35 48 12:42:16 +49:01:48 J2000 MAGELLAN1 1 -0.349065555555556 1298682774 1298638554
2044 08 09 22 09 06 21:09:19 -29:11:39 J2000 CFHT 1 0 2354418161 2354455723
2004 08 25 02 06 35 09:52:39 -73:07:59 J2000 CAMB1MILE 1 0 undef undef
2025 07 12 10 16 35 23:07:42 +51:07:07 B1950 USSR6 -1 -0.0523598333333333 undef undef
1982 02 14 00 06 01 01:09:11 -49:01:06 J2000 MAUNAK88 1 0 382570224 382515362
2033 05 04 09 48 48 07:59:48 +86:48:11 J2000 MCDONLD2.7 0 0 undef undef
2058 08 14 16 39 48 10:19:09 +16:16:24 B1950 APO3.5 1 0 2796642578 2796604498
1987 10 19 03 11 09 00:11:42 +38:24:06 J2000 KOTTAMIA 1 0.104719666666667 561647020 561612340
2074 10 23 08 48 01 07:09:16 -86:01:59 J2000 PARKES 0 0 undef undef
2030 07 07 23 11 42 08:09:11 -01:13:09 J2000 VLT4 1 -0.0349065555555556 1909740913 1909699173
1990 09 04 13 05 39 21:52:11 -37:09:09 B1950 QUEBEC1.6 0 0.349065555555556 undef undef
2060 12 19 13 48 24 10:06:05 -03:42:13 B1950 QUEBEC1.6 0 -0.174532777777778 2870648559 2870696405
2000 10 14 17 48 24 13:35:01 +61:01:39 B1950 HARVARD -1 0 undef undef
2000 10 18 18 59 11 21:11:59 -29:16:42 J2000 ESO3.6 1 0.174532777777778 971977272 971936550
2031 07 19 17 09 42 14:59:42 +37:01:24 J2000 ESO3.6 1 0 1942256025 1942287199
1999 12 18 00 39 39 18:06:01 -25:48:52 B1950 PALOMAR200 -1 0.174532777777778 945447981 945475117
2006 03 14 16 06 41 01:05:16 -29:41:09 J2000 PALOMAR60 -1 -0.349065555555556 1142348548 1142308116
2010 04 07 09 59 41 16:59:59 +09:01:16 B1950 VLT1 0 -0.0523598333333333 1270608392 1270651097
1982 09 12 07 52 16 05:24:52 -58:42:52 J2000 BLOEMF 1 0 400700947 400689487
2018 07 12 18 35 09 09:09:39 +09:09:07 B1950 STEWARD90 0 0.0174532777777778 1531407617 1531452802
1986 09 19 12 35 52 12:52:09 +61:35:42 J2000 PALOMAR60 1 0 undef undef
1986 12 07 02 13 41 10:05:19 +61:59:52 B1950 USSR600 1 -0.0349065555555556 undef undef
2018 01 09 18 24 41 16:52:35 -84:09:59 B1950 UKST -1 -0.104719666666667 undef undef
2051 02 18 19 24 16 15:24:16 +51:06:41 J2000 STEWARD90 -1 0 2560304341 2560285301
2015 01 09 23 19 41 18:16:16 -09:11:41 B1950 AAT 0 -0.349065555555556 1420823219 1420880768
2025 06 04 12 07 39 05:11:52 -25:24:16 J2000 MOPRA -1 0 1748978372 1749029467
2030 08 23 22 06 05 02:01:41 -12:07:07 J2000 DUNLAP74 0 -0.104719666666667 1913772226 1913727621
2037 03 04 16 01 24 11:06:01 +01:01:16 B1950 KECK1 -1 0 2119754546 2119711559
2041 10 19 00 24 59 11:05:48 -46:13:41 J2000 TAUTSCHM 0 0.0349065555555556 undef undef
2023 12 04 06 01 35 16:39:35 -58:48:09 B1950 BLOEMF 1 0 1701733045 1701722125
2051 11 19 10 05 41 06:35:35 -32:42:35 B1950 KPNO158 1 0 2584071829 2584017395
2026 08 19 09 09 48 06:42:11 +12:59:09 B1950 LPO1 0 0 1787110742 1787157284
1987 01 01 11 13 24 21:59:05 +61:13:07 J2000 CAMB1MILE 0 -0.174532777777778 undef undef
1992 03 01 20 48 06 12:24:06 +29:39:16 B1950 MMT 0 -0.0174532777777778 699500525 699467887
2017 06 23 22 59 42 08:52:35 -51:11:09 J2000 ARECIBO 1 -0.349065555555556 1498307476 1498268873
1989 05 23 00 01 01 20:09:01 +58:42:24 B1950 KPNO90 -1 0.349065555555556 611815114 611865195
2012 08 18 07 16 39 04:41:09 +20:35:06 J2000 PARKES 0 0 1345305343 1345255496
2042 11 19 16 06 07 08:24:24 +03:05:35 B1950 SUBARU 1 0.0174532777777778 2300086400 2300043290
2055 03 07 11 05 19 06:11:35 -03:24:39 B1950 FLAGSTF61 0 0.0523598333333333 2688066360 2688020347
1986 07 09 03 05 48 03:35:13 +51:16:11 J2000 VLA 1 0.0523598333333333 521274343 521340322
2000 12 04 22 16 52 06:07:09 -12:35:59 J2000 MAGELLAN2 1 0 975972311 976018802
2060 03 19 20 07 35 02:11:59 +49:01:52 B1950 KPNO158 0 -0.0349065555555556 2846924620 2846992910
2039 08 12 16 11 35 09:52:42 +41:39:24 J2000 TAUTSCHM 0 0 undef undef
2025 04 09 19 06 01 08:19:05 -01:11:35 J2000 JODRELL1 -1 0 1744205054 1744161156
2065 02 23 06 24 01 16:39:42 -51:16:52 B1950 LPO2.5 1 0.174532777777778 undef undef
2017 10 25 14 19 42 03:11:16 -25:01:39 B1950 GBVA140 0 0.174532777777778 1508899759 1508924831
2065 02 25 11 19 07 09:52:13 -49:24:13 B1950 CSO 1 0 3002852011 3002796772
2052 10 07 07 06 35 20:07:42 -61:07:13 B1950 PALOMAR48 1 -0.104719666666667 2611963381 2611974302
2005 03 12 05 09 16 03:59:16 -37:01:13 B1950 JCMT 1 0 1110665001 1110614411
1991 01 18 22 52 42 12:35:16 +38:06:05 B1950 SUGARGROVE 1 0 664248575 664309964
1999 07 09 01 19 13 05:05:13 -49:16:11 B1950 NOBEYAMA 0 -0.104719666666667 931469211 931493161
2057 02 18 10 24 48 09:09:19 +03:52:09 B1950 BLOEMF 0 0 2749736431 2749692438
2002 09 18 15 59 13 10:42:35 -38:59:41 J2000 STEWARD90 1 0.104719666666667 1032447724 1032384730
2046 03 18 14 41 05 17:52:07 +01:59:11 J2000 HPROV1.93 0 0.349065555555556 2405035766 2404980128
2075 01 01 07 05 35 19:19:24 -01:19:52 J2000 CATALINA61 -1 0.104719666666667 3313492563 3313531876
2069 03 01 23 07 59 23:09:42 -20:48:39 J2000 GEMININ 1 -0.0174532777777778 3129470230 3129424000
2010 01 07 06 35 13 12:24:39 +18:07:39 B1950 MCDONLD2.1 1 -0.0349065555555556 1262927955 1262891322
2003 08 14 13 39 13 06:05:09 +61:09:09 J2000 STEWARD90 1 0 undef undef
2013 06 18 05 41 13 01:07:19 +32:35:39 J2000 KPNO36FT 0 0 1371539607 1371507809
2070 02 19 06 01 06 00:16:42 -09:11:13 J2000 LPO1 -1 0.349065555555556 3159948679 3159978037
1980 08 09 23 07 07 11:01:07 +41:59:41 J2000 DAO72 0 0 undef undef
2067 01 07 17 39 07 16:11:07 +73:52:11 B1950 MTLEMMON60 1 -0.349065555555556 undef undef
2050 12 23 08 19 11 00:11:05 -29:24:01 J2000 KOTTAMIA 1 0.0523598333333333 2555407757 2555439857
2042 02 14 09 24 19 14:09:09 -20:42:35 B1950 MAUNAK88 0 0 2275983003 2276022250
2013 01 01 02 42 35 14:16:16 -29:39:48 B1950 AAT 1 -0.174532777777778 1357046112 1357020152
1987 12 19 16 52 48 08:06:42 -18:01:39 J2000 KPNO90 1 0 566972785 567010271
1987 06 01 11 41 09 00:01:13 -46:35:01 J2000 LICK120 0 0 549551093 549568361
1990 07 18 17 59 06 07:35:13 -20:06:01 J2000 KPNO158 -1 0.0523598333333333 648311097 648259839
2023 07 07 20 41 59 12:48:19 -16:09:59 J2000 IRTF 1 0.104719666666667 1688770796 1688807762
2045 03 07 06 42 07 21:19:42 +20:11:39 B1950 LPO2.5 0 0 2372474799 2372523582
2040 11 09 23 16 35 15:39:05 -25:19:13 B1950 ANU2.3 0 0 2236101735 2236152918
2069 12 19 13 01 35 16:59:59 +03:11:09 B1950 ARECIBO 0 0 3154671278 3154714841
2034 05 19 11 13 48 21:42:05 -18:39:11 J2000 TOLOLO4M 0 -0.349065555555556 2031617347 2031678476
1990 06 14 03 48 48 16:19:09 +58:52:11 J2000 VLT3 0 -0.0523598333333333 645322444 645346186
2007 12 01 11 16 11 13:42:52 +09:35:59 J2000 USSR600 0 0.0174532777777778 1196466450 1196513304
1986 11 19 09 13 11 00:01:11 -86:42:06 J2000 CSO 0 -0.0349065555555556 undef undef
2022 09 12 14 01 42 02:41:39 +20:13:06 B1950 OKAYAMA -1 0 1662981945 1662946022
2080 04 12 10 42 41 11:06:42 -41:35:35 B1950 HPROV1.52 1 0 3480175639 3480189398
2066 03 01 03 01 01 21:09:09 +01:59:35 B1950 BLOEMF -1 -0.0174532777777778 3034637600 3034594409
1991 01 04 14 42 16 12:24:39 -73:05:52 J2000 SUBARU -1 0 undef undef
1999 09 14 13 24 06 02:13:05 -73:24:42 B1950 TOLOLO1.5M 0 0.0174532777777778 undef undef
1993 03 18 20 07 01 16:48:24 +58:06:01 B1950 PARKES 1 -0.0523598333333333 732562563 732487282
2059 09 23 04 48 16 06:01:41 -01:07:01 B1950 KECK1 -1 0 2831451932 2831494821
1994 09 07 16 41 41 20:16:05 +38:19:52 J2000 UKIRT 1 0 778983852 779034858
2021 07 25 22 07 41 08:07:16 +16:42:19 B1950 FCRAO 1 0.0523598333333333 1627293486 1627255826
2001 04 09 14 19 11 00:16:59 +18:13:59 B1950 DAO72 0 -0.104719666666667 986814605 986787336
2007 07 23 21 16 24 17:09:41 -01:42:42 B1950 VLT4 0 0.104719666666667 1185221562 1185261875
1992 11 04 23 39 35 10:59:13 +58:42:05 J2000 KPNO158 -1 -0.0523598333333333 undef undef
2003 05 18 03 07 01 02:24:52 -20:06:39 J2000 USSR600 -1 0.174532777777778 1053145823 1053170999
2058 05 25 19 07 48 21:01:19 +84:42:01 J2000 VLT3 0 0 undef undef
2061 12 07 19 59 09 13:48:01 +12:42:11 B1950 JCMT -1 0.349065555555556 2901190578 2901139304
1993 03 23 11 13 16 02:48:16 -38:19:06 B1950 DAO72 0 -0.0349065555555556 732919147 732849635
2049 01 18 22 07 42 22:06:48 -51:05:11 B1950 HARVARD 0 0 undef undef
1997 03 12 09 24 24 02:09:19 -03:52:48 J2000 UKST -1 -0.0349065555555556 858119730 858078908
1982 09 09 03 42 35 05:59:41 -46:01:06 B1950 SAAO74 0 -0.174532777777778 400358274 400349832
2053 06 07 16 52 52 07:42:05 +01:06:59 B1950 MCDONLD2.7 -1 -0.174532777777778 2632920535 2632883279
2043 08 14 17 59 35 16:13:35 +32:41:41 J2000 PALOMAR48 -1 0.0174532777777778 2323105257 2323159506
2050 09 23 19 24 06 04:52:05 -46:59:42 B1950 JODRELL1 1 0 undef undef
2040 06 14 00 16 59 02:11:09 -73:16:52 B1950 FCRAO 1 0 undef undef
2037 09 18 20 06 06 20:13:48 -18:16:59 B1950 AAT -1 0 2136858440 2136906975
1985 09 14 13 24 09 01:52:39 -46:16:39 J2000 NOBEYAMA 1 0 495555713 495575129
2054 08 19 11 06 05 02:11:48 -61:07:52 J2000 PALOMAR60 1 0 undef undef
2073 09 14 13 59 19 07:05:52 +03:35:42 J2000 BOSQALEGRE -1 -0.0174532777777778 3272594184 3272550641
2046 11 23 19 48 39 20:09:59 -41:42:35 B1950 EFFELSBERG 1 -0.0523598333333333 2426682607 2426690266
2046 12 01 08 16 06 09:16:41 -20:16:24 B1950 MAUNAK88 -1 -0.349065555555556 2427264185 2427228105
2075 08 07 11 05 52 03:13:39 -09:16:39 B1950 DUNLAP74 0 -0.0349065555555556 3332383007 3332423339
2061 12 12 23 06 07 16:06:09 -25:48:13 J2000 CAMB5KM -1 0 2901597538 2901622028
2008 12 07 11 07 07 11:24:06 -73:52:09 J2000 HPROV1.93 1 0.0174532777777778 undef undef
2026 08 18 20 41 59 10:41:01 -37:59:16 B1950 LPO2.5 0 -0.349065555555556 1787125849 1787084493
2017 05 04 16 06 19 05:06:19 -32:48:42 J2000 QUEBEC1.6 0 0.349065555555556 undef undef
2016 08 14 12 09 05 03:52:39 -61:19:16 J2000 UKIRT 1 0 1471181139 1471204540
2072 04 01 10 06 19 00:06:01 -49:19:05 J2000 SANPM83 -1 -0.0174532777777778 3226665417 3226688750
2019 10 18 01 07 16 05:48:06 +09:05:11 J2000 USSR6 1 0 1571424026 1571385148
2075 12 14 19 13 24 00:48:07 -86:19:16 J2000 TOLOLO4M -1 0 undef undef
2057 01 01 07 52 11 04:59:05 +18:19:24 J2000 FLAGSTF61 1 0.174532777777778 2745617909 2745575119
2046 02 25 02 59 07 17:16:01 +09:06:41 J2000 KECK2 0 0 2403169500 2403127996
2029 12 12 20 06 35 16:39:16 -38:01:59 B1950 PARKES 1 0 1891876684 1891848302
2001 06 04 11 42 05 03:11:01 +20:24:24 B1950 UKST -1 0.349065555555556 991602721 991626188
2039 05 01 00 09 42 04:39:39 -49:11:13 J2000 VLA 0 0.0523598333333333 2187804141 2187817832
1983 05 12 13 52 35 13:05:11 -41:13:07 B1950 CSO -1 -0.349065555555556 421551209 421512727
2075 07 18 17 52 35 19:35:07 -16:13:48 J2000 EFFELSBERG -1 0 3330615046 3330648338
2027 06 25 22 48 48 09:16:11 +25:24:39 J2000 PARKES 1 -0.104719666666667 1813967286 1814005709
2033 03 12 21 05 41 10:42:48 -01:16:16 B1950 ANU2.3 -1 0 1994225136 1994268717
2021 12 14 12 09 35 09:16:35 -32:24:24 B1950 LPO4.2 0 0 1639441324 1639474528
2021 02 09 15 05 42 10:09:41 -12:09:09 J2000 DAO72 -1 0.104719666666667 1612845750 1612877330
1997 09 09 06 59 42 10:13:07 +29:11:09 B1950 TOLOLO1.5M 0 -0.0523598333333333 873801805 873751728
2014 08 14 12 42 09 06:24:42 -61:42:24 J2000 BOSQALEGRE 1 0.0523598333333333 undef undef
2073 01 01 11 19 05 00:13:35 +49:16:59 B1950 KECK2 0 0 3250527118 3250496203
1986 09 25 06 01 09 03:06:11 -16:05:13 J2000 HPROV1.93 1 0 528067790 528016963
2012 06 01 15 41 07 21:35:48 +32:06:52 J2000 STEWARD90 -1 0 1338526199 1338494179
2009 11 23 06 59 24 06:05:41 -58:09:05 B1950 KECK1 0 0.0349065555555556 1258966864 1258990510
2051 11 14 06 05 09 02:35:59 +01:42:42 B1950 LICK120 1 0.104719666666667 2583624901 2583578991
2049 12 12 12 16 39 02:01:59 -03:06:07 J2000 MTLEMMON60 0 0.0349065555555556 2522960157 2522915066
2029 10 09 22 41 07 07:59:05 +18:07:06 J2000 ESONTT -1 -0.174532777777778 1886217900 1886261722
2045 12 12 19 39 24 14:41:09 +51:16:16 J2000 EFFELSBERG 0 0.174532777777778 undef undef
2031 09 19 11 01 19 08:05:13 +41:11:13 B1950 VLT4 -1 -0.104719666666667 1947571021 1947521139
2078 12 01 04 52 13 05:24:13 +16:09:41 J2000 NOBEYAMA 1 -0.349065555555556 3437103011 3437165535
2061 02 12 19 06 59 19:52:24 -84:13:01 B1950 GBVA140 0 -0.0174532777777778 undef undef
2045 11 12 20 16 19 19:11:24 -32:39:42 J2000 ESONTT -1 0 2394104701 2394071663
2054 11 04 00 39 13 02:06:09 +12:41:39 B1950 MAGELLAN1 -1 0 2677357850 2677311193
1991 10 14 16 35 06 00:24:11 +84:16:07 B1950 IRTF 0 0 undef undef
1995 09 01 16 13 41 17:41:05 -12:16:48 B1950 STEWARD90 1 -0.0174532777777778 809988856 810028778
2025 11 12 23 41 07 00:05:13 +46:48:59 J2000 LOWELL72 0 -0.104719666666667 1762968253 1762958693
2037 11 04 17 24 35 18:13:19 +16:52:07 B1950 KECK2 1 0 2140974986 2141021080
2045 05 12 04 16 13 20:19:19 +03:35:16 J2000 FLAGSTF61 -1 -0.104719666666667 2378094643 2378142526
2023 11 14 07 48 13 20:59:09 -09:42:48 J2000 BLOEMF 0 0.104719666666667 1699955283 1699911457
2055 07 09 18 11 07 21:52:09 +37:13:41 B1950 DUNLAP74 0 0.104719666666667 2698789586 2698762814
2057 03 04 09 52 42 04:35:19 +32:19:52 B1950 TOLOLO4M -1 0 2750868269 2750900933
2031 05 18 03 06 35 03:06:48 +20:52:41 J2000 JCMT 1 0 1936883653 1936844378
2029 03 04 05 42 59 01:24:07 +41:35:07 J2000 STEWARD90 0 -0.174532777777778 1867321032 1867304854
2026 01 14 10 09 42 02:07:01 -25:59:39 B1950 SANPM83 -1 -0.174532777777778 1768336422 1768377746
2030 03 14 17 24 39 23:16:48 +38:05:05 J2000 EFFELSBERG -1 0 1899678677 1899670620
1980 02 07 02 48 16 11:19:39 +09:35:13 J2000 FCRAO -1 0.104719666666667 318733192 318690450
2000 02 04 11 42 42 09:06:05 +09:09:19 B1950 MTEKAR 0 0 949682908 949644296
2038 07 01 09 19 11 19:41:01 -51:35:09 J2000 NOBEYAMA 1 0 2161606422 2161618075
2068 12 14 01 09 59 21:07:06 +38:01:42 J2000 KOSMA3M -1 0.174532777777778 3122608058 3122665478
2038 08 01 22 05 06 04:39:59 +51:05:13 B1950 IRTF 1 0.0174532777777778 2164358695 2164327487
2079 09 01 09 07 16 02:11:42 +01:42:35 J2000 LPO2.5 -1 0.0523598333333333 3460748118 3460703944
2043 01 04 11 01 16 16:16:35 -37:59:48 B1950 DUNLAP74 0 -0.0523598333333333 2303983866 2304007020
2080 10 25 09 24 13 20:11:52 +32:19:35 J2000 TAUTNBG 1 0.349065555555556 3497078168 3497125209
2040 02 14 16 52 11 18:39:05 +01:35:24 B1950 LPO4.2 -1 -0.0174532777777778 2212805498 2212849568
2021 10 14 00 42 39 19:19:01 -73:59:05 J2000 MOPRA -1 0 undef undef
2064 08 25 20 52 24 13:48:39 -41:41:05 J2000 VLT3 -1 -0.0523598333333333 2986892306 2986863280
2061 12 18 22 19 06 05:42:11 +01:42:39 J2000 MCDONLD2.1 1 0.349065555555556 2902184275 2902216672
2006 06 14 09 42 01 19:42:09 +29:07:48 B1950 HARVARD 0 0.0523598333333333 1150240785 1150296123
2073 02 01 13 24 09 12:07:52 -01:06:48 J2000 VLT3 1 0 3253226333 3253183585
1994 01 01 11 11 59 07:59:52 -20:42:13 J2000 AAT 0 0.0349065555555556 757413447 757461654
2009 06 18 05 48 05 21:48:24 -03:05:24 J2000 KPNO158 0 -0.0349065555555556 1245302826 1245346128
2001 04 25 09 09 59 06:06:24 -41:41:41 B1950 JCMT 0 -0.349065555555556 988227373 988188858
2002 12 07 13 52 48 00:39:42 +18:59:48 J2000 DUNLAP74 -1 0.174532777777778 1039199910 1039245091
2059 02 01 02 52 13 18:19:48 +20:39:59 B1950 BLOEMF 0 0 2811292919 2811330198
2001 02 07 06 05 24 07:06:48 -58:35:48 J2000 MTEKAR 1 0 undef undef
2042 04 19 21 16 59 00:52:05 -25:24:01 B1950 LPO1 0 0.0349065555555556 2281591409 2281540009
2009 09 01 09 41 11 06:09:06 +32:09:35 B1950 USSR6 0 -0.0174532777777778 1251835629 1251811169
2027 10 09 03 42 41 03:52:52 +61:13:48 B1950 MCDONLD2.7 1 -0.0523598333333333 undef undef
2004 11 01 09 06 13 07:42:19 +61:01:24 J2000 EFFELSBERG 1 0.0174532777777778 undef undef
2038 08 01 22 41 19 05:16:13 -09:19:59 B1950 DUPONT 1 0.0349065555555556 2164345672 2164390120
2063 04 23 10 42 05 16:24:24 +49:07:24 J2000 LICK120 0 0.349065555555556 2944524994 2944574204
1980 10 14 09 06 35 06:24:52 +84:07:59 B1950 VLT3 0 -0.174532777777778 undef undef
2050 10 25 09 05 01 07:07:39 +16:07:39 J2000 ESONTT -1 0.0174532777777778 2550284419 2550236320
1984 10 07 21 59 41 03:48:06 +51:07:42 B1950 HARVARD -1 0 undef undef
2027 06 19 05 48 41 12:41:07 +84:19:41 J2000 JODRELL1 1 0.0349065555555556 undef undef
2056 10 01 06 52 01 03:05:13 -49:42:42 J2000 SAAO74 1 0.0174532777777778 2737641496 2737620083
1999 03 07 14 24 09 07:07:42 +38:48:19 J2000 MTHOP1.5 0 -0.0349065555555556 920834263 920807119
2015 01 19 09 06 52 09:35:16 -86:24:24 J2000 TOLOLO1.5M -1 0 undef undef
2011 03 14 07 06 16 23:41:13 -73:48:24 J2000 TOLOLO4M 1 0.0523598333333333 undef undef
2035 08 04 17 05 19 05:24:19 +03:01:42 B1950 CATALINA61 -1 0 2069833958 2069791818
2027 11 12 18 06 07 17:35:05 +58:24:06 B1950 PALOMAR48 -1 0.174532777777778 1826024940 1826002069
2031 07 12 11 19 39 07:11:07 -41:19:52 J2000 NOBEYAMA -1 0.174532777777778 1941584978 1941595693
2079 03 23 07 19 19 15:16:09 -84:41:24 J2000 SUGARGROVE 1 0 undef undef
2028 05 23 23 41 16 09:13:16 +58:01:07 J2000 CAMB5KM -1 -0.0523598333333333 undef undef
2002 11 09 05 11 16 13:24:05 +49:11:41 B1950 LPO2.5 0 0.0523598333333333 1036811690 1036784216
2029 10 23 15 42 09 10:35:07 -03:11:39 J2000 PARKES 0 0 1887467122 1887425083
2016 03 18 06 13 42 02:19:35 -46:05:19 J2000 TAUTNBG -1 -0.0174532777777778 undef undef
1995 05 01 20 42 24 18:13:01 -46:06:07 J2000 TOLOLO1.5M -1 0.349065555555556 799294446 799338446
2055 03 14 13 48 19 13:42:07 +84:06:24 B1950 HARVARD -1 0 undef undef
2010 06 23 02 39 09 12:48:39 +25:11:09 B1950 MCDONLD2.7 1 -0.0523598333333333 1277317231 1277283717
2046 10 14 06 16 19 09:11:09 +38:09:59 J2000 PALOMAR60 0 -0.0349065555555556 2423113887 2423087326
2003 01 14 03 13 24 17:06:13 -73:07:59 J2000 KECK1 0 -0.349065555555556 1042551558 1042510229
2041 08 19 20 41 19 19:13:13 -51:52:16 J2000 VLT4 -1 -0.174532777777778 2260540345 2260527443
1997 01 09 22 16 09 14:48:52 +01:13:05 J2000 KPNO36FT -1 0 852800180 852843630
2014 06 14 18 07 24 00:13:35 +86:24:16 B1950 DUNLAP74 0 0.0174532777777778 undef undef
2010 10 23 21 19 41 19:39:41 -20:59:48 J2000 MTEKAR -1 0.0174532777777778 1287836791 1287867937
2065 10 12 09 35 42 15:39:01 -25:16:19 B1950 MTHOP1.5 -1 0.104719666666667 3022507708 3022538527
2026 11 19 19 01 19 04:39:41 +61:41:59 B1950 TAUTNBG 1 0.0349065555555556 undef undef
2056 03 07 12 59 01 00:16:05 -37:42:24 B1950 CAMB1MILE 0 0.0523598333333333 undef undef
2067 11 04 16 59 09 01:13:07 -03:01:06 J2000 HPROV1.93 -1 0 3087648482 3087604174
2027 07 07 04 01 01 00:35:42 -49:35:11 B1950 MAGELLAN2 1 -0.0349065555555556 1815009629 1814987911
2005 10 09 22 42 39 06:01:35 -01:39:16 B1950 MTLEMMON60 0 0.349065555555556 1128930686 1128875602
1981 10 09 18 41 13 15:24:59 -41:19:05 J2000 ESO3.6 -1 -0.104719666666667 371470348 371446700
2025 11 19 00 13 16 02:11:16 +58:11:07 J2000 TOLOLO1.5M -1 -0.0174532777777778 1763429112 1763441291
2033 01 07 05 41 01 02:06:24 -86:06:11 B1950 TIDBINBLA 1 0.0523598333333333 undef undef
2024 01 19 14 42 01 15:16:59 +25:05:48 J2000 CSO 1 0.104719666666667 1705750323 1705708491
2062 11 14 18 09 01 01:52:05 -01:05:19 B1950 PALOMAR200 0 -0.174532777777778 2930772083 2930734572
2054 10 04 01 42 09 08:06:07 +25:09:42 J2000 KOTTAMIA 1 0 2674764271 2674728644
2045 06 19 20 41 16 03:01:16 +46:39:11 J2000 LOWELL72 -1 0 2381469883 2381450022
2021 02 25 23 05 48 00:35:39 +46:42:09 B1950 STROMLO74 1 0.349065555555556 undef undef
2021 04 01 10 05 19 05:35:52 +49:35:42 J2000 APO3.5 1 -0.174532777777778 undef undef
2057 03 04 10 11 24 03:52:07 +03:05:16 J2000 STROMLO74 1 0.0174532777777778 2750980801 2750936027
1996 07 18 04 19 16 08:05:35 +12:59:41 J2000 MOPRA 1 0.174532777777778 837726163 837673311
2059 06 07 16 39 16 23:52:52 -73:41:06 B1950 GEMININ 0 0.0349065555555556 undef undef
2069 04 14 03 41 35 07:24:05 +16:16:06 B1950 SUBARU 0 0 3133115900 3133161826
2050 07 14 16 09 41 23:16:42 -84:48:11 B1950 ESONTT -1 0.0174532777777778 undef undef
2034 05 12 15 11 07 13:52:39 +18:24:39 B1950 CAMB1MILE 1 -0.0349065555555556 2031143793 2031114385
2020 07 04 20 05 59 02:52:41 -38:48:11 B1950 EFFELSBERG 0 0 1593930802 1593851773
2052 12 14 01 06 13 22:16:09 -86:01:52 B1950 HPROV1.93 -1 0 undef undef
2039 01 07 17 09 41 01:09:05 -18:01:48 B1950 NOBEYAMA 0 0.0523598333333333 2178071901 2178020542
2013 03 09 10 48 09 07:13:09 +41:59:16 J2000 VLT4 0 -0.174532777777778 1362856853 1362809494
2019 10 23 15 41 09 08:48:01 +73:16:06 B1950 KECK1 0 0.0349065555555556 undef undef
2055 03 04 03 01 19 05:16:24 +49:13:13 B1950 AAT 1 -0.104719666666667 2687748331 2687776443
2023 08 23 05 13 48 09:11:48 -61:19:05 B1950 MTLEMMON60 0 0 undef undef
2028 07 07 22 52 19 14:42:11 +09:05:19 J2000 JCMT 1 -0.0349065555555556 1846625879 1846671556
2011 04 09 22 09 48 02:11:16 -29:06:07 B1950 BOSQALEGRE 0 0.174532777777778 1302346710 1302392698
2029 04 09 15 09 09 01:35:16 -03:39:11 B1950 FCRAO 1 0 1870514676 1870470172
2009 02 23 05 35 09 22:48:07 -41:01:48 B1950 STEWARD90 0 -0.349065555555556 1235397879 1235354657
2067 05 04 00 06 05 07:19:52 +16:35:42 J2000 SANPM83 -1 0.349065555555556 3071675620 3071625936
2080 11 25 13 05 48 06:06:41 -49:41:13 J2000 SUGARGROVE -1 0 3499739080 3499748745
2037 12 25 23 52 06 18:48:24 +61:13:24 B1950 GBVA140 0 -0.0174532777777778 undef undef
2029 12 09 05 16 42 12:59:48 -20:06:35 J2000 SANPM83 0 -0.0349065555555556 1891505441 1891457427
2007 11 09 22 09 11 05:19:41 -73:19:01 J2000 AAT -1 0 undef undef
2056 06 18 10 05 48 01:48:05 +86:09:09 B1950 MMT 0 0.0174532777777778 undef undef
2010 03 18 23 24 48 10:13:11 +86:39:13 J2000 PARKES 0 0 undef undef
1994 12 18 20 01 35 07:06:07 +18:41:01 J2000 ESO3.6 1 0.104719666666667 787799566 787833831
2026 05 25 10 01 07 21:07:19 +61:09:16 B1950 KPNO84 -1 0 undef undef
2031 02 04 12 05 35 13:05:24 -86:48:42 J2000 KPNO158 -1 0 undef undef
1993 09 14 11 52 48 15:13:11 +29:39:39 J2000 FLAGSTF61 0 0 748020635 747988899
2045 07 25 09 42 09 09:01:24 +73:39:01 B1950 JCMT 0 0.349065555555556 2384616742 2384571911
2005 02 23 18 11 16 15:13:06 +86:35:05 J2000 UKST 0 0.174532777777778 undef undef
1998 04 01 01 11 05 16:24:42 -38:52:42 B1950 VLT3 1 0.0523598333333333 891393820 891445169
2019 11 18 14 35 48 03:16:11 -61:19:52 B1950 KPNO36FT 1 -0.0523598333333333 undef undef
1994 03 09 03 13 06 23:41:41 -51:16:39 B1950 HPROV1.93 0 -0.174532777777778 763206552 763223848
2066 05 07 21 59 48 16:52:19 -46:11:24 B1950 TIDBINBLA -1 0.0523598333333333 3040442415 3040418828
1986 03 04 18 59 19 21:24:41 -20:59:13 J2000 IRTF 0 0 510334156 510373420
2020 10 07 05 48 41 08:05:59 +49:09:06 B1950 ESO3.6 1 0.104719666666667 1602063040 1602079483
2025 12 14 11 05 42 22:16:35 +03:48:13 B1950 IRTF -1 0.0349065555555556 1765660414 1765703199
1990 09 07 20 19 48 21:05:35 +37:48:11 J2000 ARECIBO 1 -0.0174532777777778 652821409 652786115
2024 06 01 20 16 05 04:24:19 -32:19:59 J2000 PALOMAR200 1 0.0174532777777778 1717341054 1717285462
2015 04 25 01 09 41 21:24:42 -09:01:41 B1950 TAUTSCHM 0 0.0174532777777778 1429924839 1429961899
2063 04 12 07 05 16 01:11:19 +84:24:01 J2000 MCDONLD2.7 0 0.104719666666667 undef undef
1990 03 04 17 42 19 20:05:13 +25:59:16 B1950 APO3.5 1 -0.104719666666667 636625852 636595692
2001 12 14 11 52 09 04:05:35 -01:16:42 B1950 SUBARU -1 0.0523598333333333 1008299585 1008254779
2057 02 19 13 48 06 14:16:35 +29:19:11 B1950 SUGARGROVE 1 -0.174532777777778 2749855296 2749833155
1987 06 07 16 05 07 08:19:01 +86:41:41 B1950 GEMININ -1 0 undef undef
2020 06 19 16 19 42 14:09:06 -46:11:24 B1950 VLT3 0 0 1592586364 1592557119
2020 06 25 08 59 05 18:11:07 -46:52:05 B1950 ST.ANDREWS 0 0 undef undef
2014 04 04 02 59 35 13:05:07 -29:35:48 J2000 KISO -1 -0.0349065555555556 1396521205 1396554147
2055 12 12 06 59 11 23:48:05 +86:19:39 J2000 VLT2 1 0 undef undef
2018 07 07 22 05 09 02:01:42 -16:24:41 J2000 KECK2 1 0.174532777777778 1531052889 1531001510
2045 04 09 06 52 52 14:13:06 +20:42:41 B1950 OKAYAMA -1 0.0523598333333333 2375256715 2375304949
2053 10 09 12 24 07 17:19:39 -41:05:35 J2000 SANPM83 -1 0.0174532777777778 2643566871 2643593885
2070 12 25 11 42 06 15:09:05 -84:11:19 B1950 APO3.5 -1 -0.104719666666667 undef undef
1982 11 18 16 13 11 14:09:05 +84:52:11 B1950 ATCA 0 -0.0174532777777778 undef undef
1987 03 14 08 07 05 00:16:35 -03:09:19 J2000 VLT2 1 0 542719523 542763315
2028 09 04 15 42 16 10:13:52 -09:59:24 B1950 DUPONT -1 0.0523598333333333 1851674101 1851632137
2057 08 14 23 06 09 05:01:35 +09:52:48 J2000 LOWELL72 -1 0.349065555555556 2765009257 2765043954
2055 06 19 04 59 19 14:16:35 +01:48:13 B1950 IRTF 0 -0.0174532777777778 2696978922 2697022742
1991 03 12 08 13 42 04:16:16 -37:01:42 B1950 BLOEMF -1 0.174532777777778 668680751 668728487
1986 03 25 19 09 52 17:19:06 -32:59:42 J2000 KPNO84 0 -0.0523598333333333 512121123 512154943
2022 03 23 03 39 13 16:39:05 -61:52:11 B1950 PALOMAR60 -1 0 undef undef
2061 06 12 20 35 42 06:59:59 +09:05:19 B1950 SUBARU 1 -0.0349065555555556 2885909766 2885869270
2054 03 09 10 13 01 07:39:39 -46:07:05 J2000 SAAO74 0 0 2656664566 2656641342
