
package Role::NotSoTiny::Creator;
$Role::NotSoTiny::Creator::VERSION = '0.1.0'; # TRIAL
use 5.010;
use strict;
use warnings;

use Role::NotSoTiny ();

use Carp 'croak';

sub create_role {
  my ( $class, $package, %args ) = @_;
  my $methods = $args{methods} // {};
  Role::NotSoTiny->make_role($package);
  for my $meth ( keys %$methods ) {
    my $code = $methods->{$meth};
    croak qq{Not a coderef: $meth ($code)} unless ref $code eq 'CODE';
    no strict 'refs';
    *{"${package}::${meth}"} = $code;
  }
  return $package;
}

1;

#pod =encoding utf8
#pod
#pod =head1 NAME
#pod
#pod Role::NotSoTiny::Creator - Experiment with Role::Tiny / Create roles programmatically
#pod
#pod =head1 SYNOPSIS
#pod
#pod   use Role::NotSoTiny::Creator ();
#pod
#pod   my $role = Role::NotSoTiny::Creator->create_role(
#pod     'Foo',
#pod     methods => {
#pod       foo => sub {...}
#pod     },
#pod   );
#pod
#pod   # runtime equivalent of
#pod   package Foo;
#pod   use Role::Tiny;
#pod   sub foo {...}
#pod
#pod =head1 DESCRIPTION
#pod
#pod This module is an experiment with L<Role::Tiny>.
#pod It illustrates how the change in L<Role::NotSoTiny> makes easier
#pod to build functionality such as a programmatic role creator.
#pod
#pod =head1 METHODS
#pod
#pod L<Role::NotSoTiny::Creator> implements the following methods.
#pod
#pod =head2 make_role
#pod
#pod   Role::NotSoTiny::Creator->create_role('Some::Package', methods => \%methods);
#pod
#pod Prepares a given package as a role.
#pod This is done by promoting the package to a L<Role::Tiny>
#pod and adding the given methods to its stash.
#pod
#pod =cut

__END__

=pod

=encoding UTF-8

=head1 NAME

Role::NotSoTiny::Creator

=head1 VERSION

version 0.1.0

=head1 SYNOPSIS

  use Role::NotSoTiny::Creator ();

  my $role = Role::NotSoTiny::Creator->create_role(
    'Foo',
    methods => {
      foo => sub {...}
    },
  );

  # runtime equivalent of
  package Foo;
  use Role::Tiny;
  sub foo {...}

=head1 DESCRIPTION

This module is an experiment with L<Role::Tiny>.
It illustrates how the change in L<Role::NotSoTiny> makes easier
to build functionality such as a programmatic role creator.

=head1 NAME

Role::NotSoTiny::Creator - Experiment with Role::Tiny / Create roles programmatically

=head1 METHODS

L<Role::NotSoTiny::Creator> implements the following methods.

=head2 make_role

  Role::NotSoTiny::Creator->create_role('Some::Package', methods => \%methods);

Prepares a given package as a role.
This is done by promoting the package to a L<Role::Tiny>
and adding the given methods to its stash.

=head1 AUTHOR

Adriano Ferreira <ferreira@cpan.org>

=head1 COPYRIGHT AND LICENSE

This software is copyright (c) 2018 by Adriano Ferreira.

This is free software; you can redistribute it and/or modify it under
the same terms as the Perl 5 programming language system itself.

=cut
