require "dpklib/error"

module Dpklib
  module CvsIgnore
    PATTERNS = %w( RCS CVS *~ *.bak *.BAK core *.core #* .#* *.orig *.rej *.old )
  end #/CvsIgnore

  class << CvsIgnore
    def match?(fpath)
      matched = false
      filename = File.basename(fpath)
      CvsIgnore::PATTERNS.each do |pattern|
        matched = Dpklib.glob_match(pattern, filename)
        break if matched
      end
      matched
    end

  end #/<< CvsIgnore

  ReadLimitExceedError = Dpklib.new_error_class(IOError) { |limit|
    "I/O read limitation exceeded. (#{limit} byte)"    
  }
end #/Dpklib

class << Dpklib
  # from install.rb
  def glob_match(pat, str)
    case pat[0]
    when nil
      not str[0]
    when ?*
      glob_match(pat[1..-1], str) || str[0] && glob_match(pat, str[1..-1])
    when ??
      str[0] && glob_match(pat[1..-1], str[1..-1])
    else
      pat[0] == str[0] && glob_match(pat[1..-1], str[1..-1])
    end
  end

  def each_files_in_dir(dir, &block)
    Dir.foreach(dir) { |file|
      next if file =~ %r"\A\.+\Z"
      yield File.expand_path(file, dir)
    }
  end

  def read_with_limit(io, maxbyte = nil)
    data = io.read(maxbyte)
    io.eof? || Dpklib::ReadLimitExceedError.raise(maxbyte)
    data
  end

  def read_file(file, maxbyte = nil)
    File.open(file, "rb") { |input|
      read_with_limit(input, maxbyte)
    }
  end

  def write_file(file, string)
    File.open(file, "wb") { |output|
      output.write(string)
    }
  end

  def gsub_fileext(source, replaced)
    if source =~ %r'^(.+)\..+?$'
      "#{$1}.#{replaced}"
    else
      "#{source}.#{replaced}"
    end
  end
end #/<< Dpklib
