require "dpklib/dpkstd"
require "dpklib/error"

module Dpklib
  EnvNotSetError = Dpklib.new_error_class { |envname|
    "environmental variable #{envname} is not set."
  }
  EnvEvalError = Dpklib.new_error_class { |envname, expr, error|
    "Invalid environmental variable #{envname} = \'#{expr}\' {\n#{Dpklib.stacktrace_message(error)}\n}"
  }
  class ShellCommandExecuteError < Dpklib.new_error_class
    attr_reader :status
    def init_message(status, command)
      @status = status
      "Command aborted (status: #{Dpklib.get_exit_status(status)}): '#{command}'"
    end
  end
end #/Dpklib

class << Dpklib
  def get_exit_status(status_from_wait)
    (status_from_wait >> 8 & 0xFF)
  end

  def getenv_surely(envname)
    ENV[envname] || raise(Dpklib::EnvNotSetError.new(envname))
  end  

  def eval_env_surely(envname)
    expr = ENV[envname] || raise(Dpklib::EnvNotSetError.new(envname))
    error = nil
    value = Dpklib.safe_eval(4) do
      Dpklib.protect_eval(expr) do |error|
      end
    end
    if error
      puts Dpklib.stacktrace_message(error)
      raise Dpklib::EnvEvalError.new(envname, expr, error)
    else
      expr
    end
  end

  def eval_env(envname)
    begin
      eval_env_surely(envname)
    rescue Dpklib::EnvNotSetError
      nil
    end
  end

  def quote_shell(unquoted)
    quoted = unquoted.dup
    # \ -> \\
    quoted.gsub! %r"\\" do '\\\\' end
    # " -> \"
    quoted.gsub! %r'"' do '\\"' end
    # ` -> \`
    quoted.gsub! %r'`' do '\\`' end
    # $ -> \$
    quoted.gsub! %r'\$' do '\\$' end
    # newline -> \newline
    quoted.gsub! %r'\n' do "\\\n" end
    quoted
  end

  def shell_command_string(command)
    output = `#{command}`
    if $? == 0
      output
    else
      Dpklib::ShellCommandExecuteError.raise($?, command)
    end
  end

  def shell_command_array(command)
    output = shell_command_string(command)
    array = []
    output.each_line do |line|
      line.strip!
      unless line.empty?
        array << line
      end
    end
    array
  end

  def system_popen(mode, *execargs, &block)
    IO.popen("-", mode) { |io|
      if io
        yield(io)
      else
        with_exit! {
          exec(*execargs)
        }
      end
    }
  end

  def system_to_array(*execargs)
    system_popen("r", *execargs) { |io|
      io.collect { |line|
        line.chomp!
      }
    }
  end

  # NOTE:
  # Forking without with_exit! may causes problems when
  # it is invoked in Ruby/GTK's main iteration.
  def with_exit!(&block)
    begin
      yield
    rescue Exception
      STDERR.puts($!.to_s)
    end
    exit!
  end

  def fork_with_exit!(&block)
    fork {
      with_exit!(&block)
    }
  end

  def terminate_stdio(&block)
    STDIN.reopen("/dev/null")
    STDOUT.reopen("/dev/null")
  end

  def launcher_fork(&block)
    pid = fork_with_exit! {
      exit! if fork
      terminate_stdio
      yield
    }
    Process.waitpid(pid)
  end

  def system_background(*execargs)
    launcher_fork {
      exec(*execargs)
    }
  end
end #/<< Dpklib
