require "dpklib/class"
require "dpklib/strict"

module Dpklib
  class AbstractUpdateMonitor
    include Dpklib::AbstractMethod

    attr_accessor :last_mtime

    abstract_method :current_mtime
    
    def initialize
      @last_mtime = nil
    end

    def update(mtime = nil)
      mtime ||= current_mtime
      @last_mtime = mtime
    end

    def up_to_date?
      older_to?(current_mtime)
    end

    def lost?
      current_mtime.nil?
    end

    def with_update(&proc_on_update)
      mtime = current_mtime
      if older_to?(mtime)
        nil
      else
        ret = yield
        update(mtime)
        ret
      end
    end

    private
    def older_to?(current_mtime)
      if @last_mtime && current_mtime
        @last_mtime >= current_mtime
      else
        false
      end
    end
  end #/AbstractUpdateMonitor

  class UpdateMonitor < AbstractUpdateMonitor
    include Dpklib::Strict
    def initialize(&mtime_proc)
      super()
      must_if(block_given?)
      @mtime_proc = mtime_proc
    end
    
    def current_mtime
      @mtime_proc.call
    end
  end #/UpdateMonitor

  class FileUpdateMonitor < AbstractUpdateMonitor
    attr_accessor :monitored_file, :error_on_lost
    def initialize(monitored_file = nil)
      super()
      @monitored_file = monitored_file
      @error_on_lost = false
    end

    def current_mtime
      if error_on_lost
        File.mtime(@monitored_file)
      else
        File.exist?(@monitored_file) ? File.mtime(@monitored_file) : nil
      end
    end
  end #/FileUpdateMonitor
end #/Dpklib
