require "date"
require "dpklib/hour"
require "dpklib/timeutils"
require "rubyunit"

class TESTCASE_Hour < RUNIT::TestCase
  include Dpklib

  def test_new
    [ Hour.new(1,2,3), Hour[1,2,3] ].each do |hour|
      assert_equals 1, hour.hour
      assert_equals 2, hour.min
      assert_equals 3, hour.sec
    end
  end

  def test_calc
    hour = Hour.new(12, 30)
    hour.hour += 5
    assert_equals 17, hour.hour
    assert_equals 30, hour.min

    hour = Hour.new(14, 45)
    hour.min += 30
    assert_equals 15, hour.hour
    assert_equals 15, hour.min

    hour = Hour.new(19, 22, 55)
    hour.sec += 15
    assert_equals 19, hour.hour
    assert_equals 23, hour.min
    assert_equals 10, hour.sec

  end

  def test_compare
    lhs = Hour.new(19, 22, 55)
    
    rhs = Hour.new(19, 22, 55)
    assert (lhs == rhs)
    assert (lhs === rhs)

    rhs = Hour.new(23, 31, 40)
    assert (rhs > lhs)

    rhs = Hour.new(15, 54, 29)
    assert (rhs < lhs)

    another = Object.new
    assert (lhs != another)
  end

  def test_from_time
    time = Time.now
    hour = Hour.from_time(time)

    assert_equals time.hour, hour.hour
    assert_equals time.min, hour.min
    assert_equals time.sec, hour.sec
  end

  def test_step
    hour_from = Hour[7, 30]
    hour_to = Hour[8, 00]

    hours = []
    Hour.step( (hour_from..hour_to), 0, 15) do |hour|
      hours << hour
    end
    assert_equals [ Hour[7,30], Hour[7,45], Hour[8,00] ], hours

    hours = []
    enum = Hour.step( (hour_from...hour_to), 0, 15)
    hours = enum.to_a
    assert_equals [ Hour[7,30], Hour[7,45] ], hours

    assert_exception ArgumentError do
      Hour.step (hour_from...hour_to)
    end
    
  end

  def test_to_time
    date = Date.new(1999, 6, 5)
    hour = Hour[12, 25, 30]

    time = hour.to_time(date)
    assert_equals Time.local(1999,6,5,12,25,30), time

    # checks hour over 24H works
    assert_equals( Dpklib.date_to_time(date + 1, 0, 0, 0),
                   Hour[24, 0, 0].to_time(date) )
  end

  def test_addition
    lhs = Hour[01, 30, 15]
    rhs = Hour[03, 15, 30]
    answer = lhs + rhs
    assert_equals Hour[04, 45, 45], answer
  end

  def test_parse
    assert_equals Hour[4, 59], Hour.parse("0459")
    assert_equals Hour[7, 12, 1], Hour.parse("071201")
    assert_equals Hour[4, 5], Hour.parse("4:5")
    assert_equals Hour[1, 2, 3], Hour.parse("1:2:3")
    
    assert_exception(HourParseError) do
      Hour.parse("")
    end
    assert_exception(HourParseError) do
      Hour.parse("1a:d")
    end
  end

end
