=begin
=Assertions.rb
Standard assertions for Lapidary
=end

#<standard_header>
#
# Copyright (C) 2000-2001 Nathaniel Talbott
#
#    This program is free software; you can redistribute it and/or modify
#    it under the terms of the GNU General Public License as published by
#    the Free Software Foundation; either version 2 of the License, or
#    (at your option) any later version.
#
#    This program is distributed in the hope that it will be useful,
#    but WITHOUT ANY WARRANTY; without even the implied warranty of
#    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#    GNU General Public License for more details.
#
#    You should have received a copy of the GNU General Public License
#    along with this program; if not, write to the Free Software
#    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
#
#</standard_header>

module Lapidary
	module Assertions
		def assertBlock(message="")
			_wrap_assertion(message) {
				if (! yield)
					addFailedAssertion(message)
					throw :assertionFailed
				end
			}
		end
		def assert(boolean, message="")
			_wrap_assertion {
				assertBlock("assert should not be called with a block!") {!block_given?}
				assertBlock(message) {boolean}
			}
		end
		def assertEqual(expected, actual, message=nil)
			_wrap_assertion {
				fullMessage = buildMessage(message, expected, actual) {
					| arg1, arg2 |
					"Expected <#{arg1}> but was <#{arg2}>"
				}
				assertBlock(fullMessage) {
					expected == actual
				}
			}
		end
		def assertRaises(expectedExceptionKlass, message="")
			_wrap_assertion {
				assertInstanceOf(Class, expectedExceptionKlass, "Should expect a type of exception")
				actualException = nil
				fullMessage = buildMessage(message, expectedExceptionKlass) {
					| arg |
					"Expected exception of type <#{arg}> but none was thrown"
				}
				assertBlock(fullMessage) {
					thrown = false
					begin
						yield
					rescue Exception => thrownException
						actualException = thrownException
						thrown = true
					end
					thrown
				}
				fullMessage = buildMessage(message, expectedExceptionKlass, actualException.class.name, actualException.message, actualException.backtrace.join("\n\t")) {
					| arg1, arg2, arg3, arg4 |
					"Expected exception to be of type <#{arg1}> but was <#{arg2}: #{arg3}\n\t#{arg4}>"
				}
				assertBlock(fullMessage) {
					expectedExceptionKlass == actualException.class
				}
				actualException
			}
		end
		def assertInstanceOf(klass, object, message="")
			_wrap_assertion {
				assertEqual(Class, klass.class, "assertInstanceOf takes a Class as its first argument")
				fullMessage = buildMessage(message, object, klass) {
					| arg1, arg2 |
					"<#{arg1}> should have been an instance of <#{arg2}>"
				}
				assertEqual(klass, object.class, fullMessage)
			}
		end
		def assertNil(object, message="")
			assertEqual(nil, object, message)
		end
		def assertKindOf(klass, object, message="")
			_wrap_assertion {
				assertInstanceOf(Class, klass.class, "The first parameter to assertKindOf should be a Class.")
				fullMessage = buildMessage(message, object, klass) {
					| arg1, arg2 |
					"Expected <#{arg1}> to be kind_of?<#{arg2}>"
				}
				assertBlock(fullMessage) {
					object.kind_of?(klass)
				}
			}
		end
		def assertMatch(regexp, string, message="")
			_wrap_assertion {
				assertInstanceOf(Regexp, regexp, "The first argument to assertMatch should be a Regexp.")
				fullMessage = buildMessage(message, string, regexp.source) {
					| arg1, arg2 |
					"Expected <#{arg1}> to match regexp </#{arg2}/>"
				}
				assertBlock(fullMessage) {
					regexp =~ string
				}
			}
		end
		def assertSame(expected, actual, message="")
			_wrap_assertion {
				fullMessage = buildMessage(message, expected, expected.id, actual, actual.id) {
					| arg1, arg2, arg3, arg4 |
					"Expected <#{arg1}:#{arg2}> to be equal? to <#{arg3}:#{arg4}>"
				}
				assertBlock(fullMessage) {
					actual.equal?(expected)
				}
			}
		end
		def assertNothingRaised(message="")
			_wrap_assertion(message) {
				begin
					yield
				rescue Exception => thrownException
					fullMessage = buildMessage(message, thrownException.class.name, thrownException.message, thrownException.backtrace.join("\n\t")) {
						| arg1, arg2, arg3 |
						"Exception raised: <#{arg1}: #{arg2}\n\t#{arg3}>"
					}
					flunk(fullMessage)
				end
			}
		end
		def flunk(message="")
			assert(false, message)
		end
		def assertNotSame(expected, actual, message="")
			_wrap_assertion {
				fullMessage = buildMessage(message, expected, expected.id, actual, actual.id) {
					| arg1, arg2, arg3, arg4 |
					"Expected <#{arg1}:#{arg2}> to not be equal? to <#{arg3}:#{arg4}>"
				}
				assertBlock(fullMessage) {
					! actual.equal?(expected)
				}
			}
		end
		def assertNotEqual(expected, actual, message="")
			_wrap_assertion {
				fullMessage = buildMessage(message, expected, actual) {
					| arg1, arg2 |
					"Expected <#{arg1}> to be != to <#{arg2}>"
				}
				assertBlock(fullMessage) {
					expected != actual
				}
			}
		end
		def assertNotNil(object, message="")
			_wrap_assertion {
				fullMessage = buildMessage(message, object) {
					| arg |
					"Expected <#{arg}> to not be nil"
				}
				assertBlock(fullMessage) {
					! object.nil?
				}
			}
		end
		def assertDoesNotMatch(regexp, string, message="")
			_wrap_assertion(message) {
				assertInstanceOf(Regexp, regexp, "The first argument to assertDoesNotMatch should be a Regexp.")
				fullMessage = buildMessage(message, regexp.source, string) {
					| arg1, arg2 |
					"Expected </#{arg1}/> to not match <#{arg2}>"
				}
				assertBlock(fullMessage) {
					regexp !~ string
				}
			}
		end
		def assertThrows(expectedSymbol, message="", &proc)
			_wrap_assertion {
				assertInstanceOf(Symbol, expectedSymbol, "assertThrows expects the symbol that should be thrown for its first argument")
				assert(block_given?, "Should have passed a block to assertThrows")
				caught = true
				begin
					catch (expectedSymbol) {
						proc.call
						caught = false
					}
					fullMessage = buildMessage(message, expectedSymbol.to_s) {
						| arg |
						"<:#{arg}> should have been thrown"
					}
					assert(caught, fullMessage)
				rescue NameError => nameError
					if ( nameError.message !~ /^uncaught throw `(.+)'$/ )
						raise nameError
					end
					fullMessage = buildMessage(message, expectedSymbol.to_s, $1) {
						| arg1, arg2 |
						"Expected <:#{arg1}> to be thrown but <:#{arg2}> was thrown"
					}
					flunk(fullMessage)
				end				
			}
		end
		def assertNothingThrown(message="", &proc)
			_wrap_assertion {
				assert(block_given?, "Should have passed a block to assertNothingThrown")
				begin
					proc.call
				rescue NameError => nameError
					if (nameError.message !~ /^uncaught throw `(.+)'$/ )
						raise nameError
					end
					fullMessage = buildMessage(message, $1) {
						| arg |
						"Nothing should have been thrown but <:#{arg}> was thrown"
					}
					flunk(fullMessage)
				end
				fullMessage = buildMessage(message) { ||
					"Nothing should have been thrown"
				}
				assert(true, fullMessage)
			}
		end

		def method_missing(symbol, *arguments, &proc)
			if ( symbol.id2name =~ /assert_exception/ )
				assertRaises(*arguments, &proc)
			elsif ( symbol.id2name !~ /_[a-z]/ )
				super
			else
				send(symbol.id2name.gsub(/_([a-z])/) {$1.upcase}, *arguments, &proc)
			end
		end
		def buildMessage(message, *arguments, &proc)
			assertBlock("Should have passed a block to buildMessage") {
				block_given?
			}
			procArity = ((proc.arity == -1) ? 1 : proc.arity)
			assertBlock("Should have passed the same number of arguments as the message block expects. Expected <#{procArity}> but was <#{arguments.length}>") {
				procArity == arguments.length
			}
			messageParts = Array.new
			if (message != nil && message != "")
				if (message !~ /\.$/)
					message << "."
				end
				messageParts << message
			end
			arguments = arguments.collect {
				| argument |
				(argument == nil) ? "nil" : argument.to_s
			}
			messageParts << proc.call(*arguments)
			return messageParts.join(" ")
		end
		def _wrap_assertion(message = nil)
			@_assertionMessage = message if (message != nil)
			if (!@_assertionWrapped)
				@_assertionWrapped = true
				begin
					retValue = yield
					addSuccessfulAssertion(@_assertionMessage)
					return retValue
				ensure
					@_assertionWrapped = false
				end
			else
				return yield
			end
		end
	end
end

