=begin
=TC_TestCase.rb
Contains tests for the TestCase class
=end

#<standard_header>
#
# Copyright (C) 2000-2001 Nathaniel Talbott
#
#    This program is free software; you can redistribute it and/or modify
#    it under the terms of the GNU General Public License as published by
#    the Free Software Foundation; either version 2 of the License, or
#    (at your option) any later version.
#
#    This program is distributed in the hope that it will be useful,
#    but WITHOUT ANY WARRANTY; without even the implied warranty of
#    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#    GNU General Public License for more details.
#
#    You should have received a copy of the GNU General Public License
#    along with this program; if not, write to the Free Software
#    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
#
#</standard_header>

require 'Lapidary/TestResult'
require 'Lapidary/TF_Run'
require 'Lapidary/TestCase'
require 'Lapidary/TF_SetupTearDown'
require 'Lapidary/TestSuite'
require 'Lapidary/Assertions'
require 'Lapidary/TF_TestCaseAssertionHandling'
require 'Lapidary/Failure'
require 'Lapidary/Error'

module Lapidary
	class TC_TestCase
		include Assertions

		attr_reader(:result)

		def initialize(test)
			@test = test
		end
		def TC_TestCase.suite
			suite = TestSuite.new
			suite.add(TC_TestCase.new("testAssertions"))
			suite.add(TC_TestCase.new("testAddFailedAssertion"))
			suite.add(TC_TestCase.new("testAddError"))
			suite.add(TC_TestCase.new("testSuite"))
			suite.add(TC_TestCase.new("testSetupTearDown"))
			return suite
		end
		def size
			1
		end
		def run(result)
			@result = result
			send @test
		end
		def addSuccessfulAssertion(message)
			@myResult.addSuccessfulAssertion(message)
		end
		def addFailedAssertion(message)
			@myResult.addFailedAssertion(message)
		end
		def testAssertions
			@myResult = TestResult.new
			assertBlock {true}
			check("Should have no failures", @myResult.failures == 0)
			catch :assertionFailed do
				assertBlock {false}
				assertBlock {false}
			end
			check("Should have one failure", @myResult.failures == 1)
			catch :assertionFailed do
				assertBlock {true}
				assertBlock {false}
			end
			check("Should have two failures", @myResult.failures == 2)
			@result.addRun()
		end
		def testAddFailedAssertion
			testCase = TF_TestCaseAssertionHandling.new(:testFailedAssertion)
			result = TestResult.new
			called = false
			result.addFaultListener( proc {
				| fault |
				check("Should have a Failure", fault.instance_of?(Failure))
				check("The Failure should have the correct message", "failure" == fault.message)
				check("The Failure should have the correct location (was <#{fault.location})", fault.location =~ /testFailedAssertion\(Lapidary::TF_TestCaseAssertionHandling\) \[.*TF_TestCaseAssertionHandling\.rb:\d+\]/)
				called = true
			} )
			testCase.run(result)
			check("The failure should have triggered the listener", called)
		end
		def testAddError
			testCase = TF_TestCaseAssertionHandling.new(:testError)
			result = TestResult.new
			called = false
			result.addFaultListener( proc {
				| fault |
				check("Should have a TestError", fault.instance_of?(Error))
				check("The Error should have the correct message", "ZeroDivisionError: divided by 0" == fault.message)
				check("The Error should have the correct location", "testError(Lapidary::TF_TestCaseAssertionHandling)" == fault.location)
				check("The Error should have the correct exception", fault.exception.instance_of?(ZeroDivisionError))
				called = true
			} )
			testCase.run(result)
			check("The error should have triggered the listener", called)
		end
		def testSuite
			suite = TF_Run.suite()
			check("Should have a test suite", suite.instance_of?(TestSuite))
			check("Should have three tests", suite.size() == 3)

			result = TestResult.new()
			suite.run(result)
			check("Should have had three test runs", result.runs == 3)
			check("Should have had one test failure", result.failures == 1)
			check("Should have had one test error", result.errors == 1)
			@result.addRun()
		end
		def testSetupTearDown
			result = TestResult.new()

			test = TF_SetupTearDown.new(:testSucceed)
			test.run(result)
			check("Should have called setup the correct number of times", test.setupCalled)
			check("Should have called tearDown the correct number of times", test.tearDownCalled)

			test = TF_SetupTearDown.new(:testFail)
			test.run(result)
			check("Should have called setup the correct number of times", test.setupCalled)
			check("Should have called tearDown the correct number of times", test.tearDownCalled)

			test = TF_SetupTearDown.new(:testError)
			test.run(result)
			check("Should have called setup the correct number of times", test.setupCalled)
			check("Should have called tearDown the correct number of times", test.tearDownCalled)

			check("Should have had two test runs", result.runs == 3)
			check("Should have had a test failure", result.failures == 1)
			check("Should have had a test error", result.errors == 1)
			@result.addRun()
		end
		def check(message, passed)
			if ! passed
				@result.addFailedAssertion(Failure.new(name, message))
			end
		end
		def name
			"#{@test}(#{self.class.name})"
		end
	end
end
