= set.rb

# This library provides the Set class that deals with a collection of
# unordered values with no duplicates.  It is a hybrid of Array's
# intuitive inter-operation facilities and Hash's fast lookup.

集合を表す Set クラスを提供します。集合とは重複のないオブジェクトの
順序づけられていない集まりです。
Arrayの持つ演算機能とHashのような高速な検索を合わせ持ちます。

== Example

  require 'set'

  set1 = Set.new ["foo", "bar", "baz", "foo"]

  p set1			#=> #<Set: {"baz", "foo", "bar"}>

  p set1.include?("bar")	#=> true

  set1.add("heh")
  set1.delete("foo")

  p set1			#=> #<Set: {"heh", "baz", "bar"}>

== Set class

# Set implements a collection of unordered values with no duplicates.
# This is a hybrid of Array's intuitive inter-operation facilities and
# Hash's fast lookup.

Set は重複のないオブジェクトの順序づけられていない集まりの実装です。
Arrayの持つ演算機能とHashのような高速な検索を合わせ持ちます。

# The equality of each couple of elements is determined according to
# Object#eql? and Object#hash, since Set uses Hash as storage.

2つのオブジェクトの等価性は Object#eql? と Object#hash によります。
なぜなら、Set は Hash を記憶として使っているからです。

=== Included Modules
    Enumerable

=== Class Methods
--- Set::new(enum = nil)
#    Creates a new set containing the elements of the given enumerable
#    object.
    引数 enum で与えられた Enumerable の要素からなるあたらしい集合を作ります。

--- Set[*ary]
#    Creates a new set containing the given objects.
    与えられたオブジェクトからなる新しい集合を作ります。

=== Instance Methods
--- dup
#    Duplicates the set.
    集合を複製します。

--- size
--- length
#    Returns the number of elements.
    要素の個数を返します。

--- empty?
#    Returns true if the set contains no elements.
    要素を1つも持たないときに true を返します。

--- clear
#    Removes all elements and returns self.
    要素をすべて削除します。

--- replace(enum)
#    Replaces the contents of the set with the contents of the given
#    enumerable object and returns self.
    要素を enum のものと置き換えます。要素をすべて削除し、enum で与え
    られた Enumerable の要素を新しい要素とします。

--- flatten
#    Returns a new set that is a copy of the set, flattening each
#    containing set recursively.
    ネストした集合を平滑化して作られる新しい集合を返します。

    例:

      p s0 = Set[Set[1,2], 3]  #=> #<Set: {#<Set: {1, 2}>, 3}>
      p s0.flatten             #=> #<Set: {1, 2, 3}>
      p s1 = Set[Set[1,2], 1]  #=> #<Set: {#<Set: {1, 2}>, 1}>
      p s1.flatten             #=> #<Set: {1, 2}>

--- flatten!
#    Equivalent to Set#flatten, but replaces the receiver with the
#    result in place.  Returns nil if no modifications were made.
    flatten と同じですが、
    集合それ自身が平滑化して作られた新しい集合の要素で置き換えられます。
    ただし、要素の変更が起こらなかったときには nil を返します。

--- to_a
#    Converts the set to an array. (the order is uncertain)
    配列に変換します。順序は不定です。

--- include?(o)
--- member?(o)
#    Returns true if the set contains the given object.
    オブジェクト o がその集合に属す場合に true を返します。

--- superset?(set)
# Returns true if the set is a superset of the given set.
    集合が与えられた集合 set を含む場合に true を返します。

--- proper_superset?(set)
# Returns true if the set is a proper superset of the given set.
    集合が与えられた集合 set を真に含む場合に true を返します。つまり、
    ふたつの集合が等しい場合には false を返します。

--- subset?(set)
# Returns true if the set is a subset of the given set.
    集合が与えられた集合 set に含まれる場合に true を返します。

--- proper_subset?(set)
# Returns true if the set is a proper subset of the given set.
    集合が与えられた集合 set に真に含まれる場合に true を返します。つまり、
    ふたつの集合が等しい場合には false を返します。

--- each { |o| ... }
#    Calls the given block once for each element in the set, passing
#    the element as parameter.
    集合のすべての要素に対して1度ずつブロックを実行します。
    ブロック変数 o にはその要素が渡されます。

--- add(o)
--- << o
#    Adds the given object to the set and returns self.
    集合にオブジェクト o を加え、その集合自身を返します。

--- delete(o)
#    Deletes the given object from the set and returns the object.  If
#    the object is not found, returns nil.
    集合からオブジェクト o を削除し、その集合自身を返します。
    ただし、集合に o が属していない場合は nil を返します。

--- delete_if { |o| ... }
#    Deletes every element of the set for which block evaluates to
#    true, and returns self.
    ブロックで最後に評価された値が真であるようなすべての要素を削除し、
    その集合自身を返します。

--- reject! { |o| ... }
#    Equivalent to Set#delete_if, but returns nil if no changes were
#    made.
    delete_if と同じですが、要素が1つも削除されなかった場合は nil を返します。

--- merge(enum)
#    Merges the elements of the given enumerable object to the set and
#    returns self.
    enum で与えられた Enumerable の要素を追加し、その集合自身を返します。

--- subtract(enum)
#    Deletes every element that appears in the given enumerable object
#    and returns self.
    enum で与えられた Enumerable の要素を削除し、その集合自身を返します。

--- + enum
--- | enum
#    Returns a new set built by merging the set and the elements of the
#    given enumerable object.
    和集合、すなわち、2つの集合の少なくともどちらか一方に属するすべて
    の要素からなる新しい集合を作りそれを返します。

--- - enum
#    Returns a new set built by duplicating the set, removing every
#    element that appear in the given enumerable object.
    差集合、すなわち、前者に属し後者に属さないすべての要素からなる
    新しい集合を作りそれを返します。

--- & enum
#    Returns a new array containing elements common to the set and the
#    given enumerable object.
    共通部分、すなわち、2つの集合のいずれにも属するすべての要素からなる
    新しい集合を作りそれを返します。

--- ^ enum
#    Returns a new array containing elements exclusive between the set
#    and the given enumerable object.  (set ^ enum) is equivalent to
#    ((set | enum) - (set & enum)).
    対称差、すなわち、2つの集合のいずれか一方にだけ属するすべての要素からなる
    新しい集合を作りそれを返します。

--- == set
#    Returns true if two sets are equal.  The equality of each couple
#    of elements is defined according to Object#eql?.
    2つの集合が等しいときに true を返します。
    要素の等しさは Object#eql? で検査されます。

--- classify { |o| ... }
#    Classifies the set by the return value of the given block and
#    returns a hash of {value => set of elements} pairs.  The block is
#    called once for each element of the set, passing the element as
#    parameter.
    集合をブロックの値によって分類します。返される分類結果は、
    {値 => 要素の集合} という形をしたハッシュです。
    ブロックは集合のすべての要素に対して1度ずつ実行され、
    ブロック変数 o にはその要素が渡されます。

    例:

      require 'set'
      files = Set.new(Dir.glob("*.rb"))
      hash = files.classify { |f| File.mtime(f).year }
      p hash    #=> {2000=>#<Set: {"a.rb", "b.rb"}>,
                #    2001=>#<Set: {"c.rb", "d.rb", "e.rb"}>,
                #    2002=>#<Set: {"f.rb"}>}

--- divide { |o| ... }
--- divide { |o1, o2| ... }

#    Divides the set into a set of subsets according to the commonality
#    defined by the given block.

    商集合、すなわちブロックで定義される関係で分割した結果を集合として
    返します。

#    If the arity of the block is 2, elements o1 and o2 are in common
#    if block.call(o1, o2) is true.  Otherwise, elements o1 and o2 are
#    in common if block.call(o1) == block.call(o2).

    引数が2個のときは block.call(o1, o2) が真ならば o1 と o2 は同じ分
    割に属します。引数が1個のときは block.call(o1) == block.call(o2) 
    が真ならばo1 と o2 は同じ分割に属します。つまりブロックの引数が2個
    のときはブロックの値の真偽で決まる同値類に、引数が1個のときはブロッ
    クの値の == による等しさで定義される同値類に分割されます。

    引数が2個のときは、いわゆる同値関係のうち、
    対称律「(({block.call(o1, o2)})) ならば (({block.call(o2, o1)}))」、
    そして推移律「(({block.call(o1, o2)})) かつ 
    (({block.call(o2, o3)})) ならば
    (({block.call(o1, o3)}))」
    の2つを満たすことが仮定されています。

    例:

      require 'set'
      numbers = Set[1, 3, 4, 6, 9, 10, 11]
      set = numbers.divide { |i,j| (i - j).abs == 1 }
      p set     #=> #<Set: {#<Set: {1}>,
                #           #<Set: {11, 9, 10}>,
                #           #<Set: {3, 4}>,
                #           #<Set: {6}>}>

    応用例:

      # 8x2 のチェス盤上でナイトが到達できる位置に関する分類
      require "set"
      board = Set.new
      m, n = 8, 2
      for i in 1..m do for j in 1..n do board << [i,j] end end
      knight_move = Set[1,2]
      p board.divide { |i,j| 
        Set[(i[0]-j[0]).abs, (i[1]-j[1]).abs] == knight_move
      }  #=> #<Set: {#<Set: {[6, 2], [4, 1], [2, 2], [8, 1]}>, 
         #           #<Set: {[2, 1], [8, 2], [6, 1], [4, 2]}>, 
         #           #<Set: {[1, 1], [3, 2], [5, 1], [7, 2]}>, 
         #           #<Set: {[1, 2], [5, 2], [3, 1], [7, 1]}>}>

--- inspect
#    Returns a string containing a human-readable representation of the
#    set. ("#<Set: {element1, element2, ...}>")

    人間の読みやすい形に表現した文字列を返します。
