#!/usr/bin/env python
#
#  ninix-update.py - a command-line network update utility for ninix
#                    Tamito KAJIYAMA <14 October 2001>
#
#  $Id: ninix-update.py,v 1.8 2003/07/23 00:08:49 shy Exp $
#

import getopt
import os
import string
import sys
import tempfile
import StringIO

import codecs
import locale
locale.setlocale(locale.LC_ALL, '')

if os.environ.has_key("DISPLAY"):
    del os.environ["DISPLAY"]

import ninix.home
import ninix.update
import ninix.dll

PROGRAM_NAME = os.environ.get("NINIX_UPDATE", sys.argv[0])
locale_charset = locale.nl_langinfo(locale.CODESET)

USAGE = """\
Usage: %s [options] [URL] [directory]
Options:
  -l, --list          show a list of ghosts
  -s, --script        show a list of ghosts as a shell script
  -H, --homedir DIR   ninix home directory (default: ~/.ninix)
  -T, --tempdir DIR   temporary directory (default: /usr/tmp)
  -q, --quiet         suppress most messages
  -h, --help          show this message
"""

def usage():
    sys.stderr.write(USAGE % os.path.basename(PROGRAM_NAME))
    sys.exit(1)

def main():
    try:
        options, argv = getopt.getopt(sys.argv[1:], "lsH:T:qh",
            ["list", "script", "homedir=", "tempdir=", "quiet", "help"])
    except getopt.error, e:
        sys.stderr.write("Error: %s\n" % str(e))
        usage()
    list = script = 0
    homedir = os.environ.get("NINIX_HOME")
    tempdir = None
    quiet = 0
    for opt, val in options:
        if opt in ["-l", "--list"]:
            list = 1
        elif opt in ["-s", "--script"]:
            list = script = 1
        elif opt in ["-H", "--homedir"]:
            homedir = val
        elif opt in ["-T", "--tempdir"]:
            tempdir = val
        elif opt in ["-q", "--quiet"]:
            quiet = 1
        else:
            usage()
    if homedir:
        ninix.home.NINIX_HOME = homedir
    if tempdir:
        tempfile.tempdir = os.path.expanduser(tempdir)
    if list:
        list_ghosts(script)
    elif len(argv) == 2:
        update_ghost(argv[0], argv[1], quiet)
    else:
        usage()
        
def list_ghosts(script):
    sys.stderr.write("Searching ghosts...\n")
    config = ninix.home.load_config()
    if config is None:
        sys.stderr.write("Error: ninix home not found\n")
        sys.exit(1)
    ghosts, shells, balloons, plugins = config
    found = 0
    default_path = os.path.join(sys.path[0], 'ninix', 'dll')
    saori_lib = ninix.dll.Library('saori', default_path, sakura=None)
    dll = ninix.dll.Library('shiori', default_path, saori_lib=saori_lib)
    for desc, shiori_dir, use_makoto, surface_set, balloon, prefix, shiori_dll, shiori_name in ghosts:
        name = (shiori_dll, shiori_name)
        homeurl = ''
        shiori = dll.request(name)
        if shiori and shiori.load(shiori_dir):
            if getattr(shiori, 'is_oldtype', None):
                homeurl = shiori.getstring("homeurl")
                shiori.finalize()
            else:
                response = shiori.request('GET SHIORI/3.0\n' \
                                          'ID: homeurl\n' \
                                          'Sender: ninix-aya\n' \
                                          'SecurityLevel: local\n' \
                                          'Charset: ASCII\n\n')
                homeurl = get_value(response)
                shiori.unload()
        if not homeurl:
            continue
        if script:
            if not found:
                print "#!/bin/sh"
            print "echo " + "-" * 60
            print "echo 'Ghost: %s'" % desc.get("name", "").encode(locale_charset, 'ignore')
            print "echo 'Home URL: %s'" % homeurl
            print "echo 'Directory: %s'" % prefix
            print PROGRAM_NAME, homeurl, prefix
        else:
            print "-" * 60
            print "Ghost:", desc.get("name", "").encode(locale_charset, 'ignore')
            print "(%s,%s)" % (desc.get("sakura.name", "").encode(locale_charset, 'ignore'), desc.get("kero.name", "").encode(locale_charset, 'ignore'))
            print "Home URL:", homeurl
            print "Directory:", prefix
        found = 1
    if not found:
        sys.stderr.write("No ghost found\n")

def get_value(response):
    header = StringIO.StringIO(response)
    result = {}
    while 1:
        line = header.readline()
        if not line:
            break # EOF
        if line[-1] == '\n':
            line = line[:-1]
        line = string.strip(line)
        if not line:
            continue
        colon = string.find(line, ':')
        if colon >= 0:
            key = string.strip(line[:colon])
            result[key] = string.strip(line[colon+1:])
        else:
            continue
    if result.has_key('Charset'):
        charset = result['Charset']
        try:
            codecs.lookup(charset)
        except:
            sys.stderr.write("Unsupported charset %s" % repr(charset))
        else:
            for key in result.keys():
                result[key] = unicode(result[key], charset, 'ignore')
    if result.has_key('Value'):
        return result['Value']
    else:
        return ''

def update_ghost(homeurl, ghostdir, quiet):
    if not os.path.exists(os.path.expanduser(ghostdir)):
        sys.stderr.write("Error: %s not found\n" % ghostdir)
        sys.exit(1)
    updateman = ninix.update.NetworkUpdate(None)
    updateman.start(homeurl, ghostdir, timeout=60)
    while 1:
        code = updateman.run()
        while 1:
            event = updateman.get_event()
            if not event:
                break
            if quiet:
                continue
            elif event[0] in ["OnUpdateBegin",
                              "OnUpdate.OnMD5CompareComplete",
                              "OnUpdate.OnMD5CompareFailure"]:
                print event[0]
            elif event[0] in ["OnUpdateReady",
                              "OnUpdateComplete",
                              "OnUpdateFailure",
                              "OnUpdate.OnDownloadBegin"]:
                print "%s (%s)" % event[:2]
        if code == 0:
            break
        if updateman.state == 7 and updateman.schedule:
            print "File(s) to be updated:"
            for filename, checksum in updateman.schedule:
                print "   ", filename
    updateman.stop()

if __name__ == "__main__":
    main()
