# -*- coding: EUC-JP -*-
# Tamito KAJIYAMA <22 February 2002>
# $Id: misaka.py,v 1.10 2003/07/27 07:51:19 shy Exp $

# TODO:
# error recovery
# - ${foo}
# - $(foo) ?

import os
import re
import string
import StringIO
import sys
import time
import whrandom
from types import *

import kanjilib

builtin_open = open

class MisakaError(ValueError):
    pass

def lexical_error(path=None, position=None):
    if path is None:
        error = "lexical error"
    elif position is None:
        error = "lexical error in %s" % path
    else:
        at = "line %d, column %d" % (position)
        error = "lexical error at %s in %s" % (at, path)
    raise MisakaError, error

def syntax_error(message, path=None, position=None):
    if path is None:
        error = "syntax error (%s)" % message
    elif position is None:
        error = "syntax error in %s (%s)" % (path, message)
    else:
        at = "line %d, column %d" % position
        error = "syntax error at %s in %s (%s)" % (at, path, message)
    raise MisakaError, error

def list_dict(dir):
    buffer = []
    try:
        filelist, debug, error = read_misaka_ini(dir)
    except Exception, e:
        filelist = []
    for filename in filelist:
        buffer.append(os.path.join(dir, filename))
    return buffer

def read_misaka_ini(dir, _debug=0):
    path = os.path.join(dir, "misaka.ini")
    file = builtin_open(path)
    filelist = []
    debug = 0
    error = 0
    while 1:
        line = file.readline()
        if not line:
            break
        line = string.strip(line)
        if not line or line[:2] == "//":
            continue
        if line == "dictionaries":
            line = file.readline()
            if string.strip(line) != "{":
                syntax_error("expected an open brace", path)
            while 1:
                line = file.readline()
                if not line:
                    syntax_error("unexpected end of file", path)
                line = string.strip(line)
                if line == "}":
                    break
                if not line or line[:2] == "//":
                    continue
                filelist.append(string.lower(string.replace(line, "\\", "/")))
        elif line == "debug,0":
            debug = 0
        elif line == "debug,1":
            debug = 1
        elif line == "error,0":
            error = 0
        elif line == "error,1":
            error = 1
        elif line == "propertyhandler,0":
            pass
        elif line == "propertyhandler,1":
            pass
        elif _debug & 4:
            print "misaka.py: unknown directive '%s'" % line
    return filelist, debug, error

###   LEXER   ###

TOKEN_NEWLINE       = 1
TOKEN_WHITESPACE    = 2
TOKEN_OPEN_BRACE    = 3
TOKEN_CLOSE_BRACE   = 4
TOKEN_OPEN_PAREN    = 5
TOKEN_CLOSE_PAREN   = 6
TOKEN_OPEN_BRACKET  = 7
TOKEN_CLOSE_BRACKET = 8
TOKEN_DOLLAR        = 9
TOKEN_QUOTE         = 10
TOKEN_COMMA         = 11
TOKEN_SEMICOLON     = 12
TOKEN_OPERATOR      = 13
TOKEN_DIRECTIVE     = 14
TOKEN_TEXT          = 15

class Lexer:
    re_comment = re.compile(r"[ \t]*//[^\r\n]*")
    re_newline = re.compile(r"\r\n|\r|\n")
    patterns = [
        (TOKEN_NEWLINE,       re_newline),
        (TOKEN_WHITESPACE,    re.compile(r"[ \t]+")),
        (TOKEN_OPEN_BRACE,    re.compile(r"{")),
        (TOKEN_CLOSE_BRACE,   re.compile(r"}")),
        (TOKEN_OPEN_PAREN,    re.compile(r"\(")),
        (TOKEN_CLOSE_PAREN,   re.compile(r"\)")),
        (TOKEN_OPEN_BRACKET,  re.compile(r"\[")),
        (TOKEN_CLOSE_BRACKET, re.compile(r"\]")),
        (TOKEN_DOLLAR,        re.compile(r'\$')),
        #(TOKEN_QUOTE,         re.compile(r'"')),
        (TOKEN_COMMA,         re.compile(r",")),
        (TOKEN_SEMICOLON,     re.compile(r";")),
        (TOKEN_OPERATOR,      re.compile(r"[!=]=|[<>]=?|=[<>]|&&|\|\||\+\+|--|[+\-*/]?=|[+\-*/%\^]")),
        (TOKEN_DIRECTIVE,     re.compile(r"#[_A-Za-z][_A-Za-z0-9]*")),
        #(TOKEN_TEXT,          re.compile(r"[!&|]|(\\[,\"]|[\x01\x02#'.0-9:?@A-Z\\_`a-z~]|[\x80-\xff].)+")),
        (TOKEN_TEXT,          re.compile(r"(\"[^\"]*\")|[!&|]|(\\[,\"]|[\x01\x02#'.0-9:?@A-Z\\_`a-z~\"]|[\x80-\xff].)+")),
        ]
    token_names = {
        TOKEN_WHITESPACE:    "whitespace",
        TOKEN_NEWLINE:       "a newline",
        TOKEN_OPEN_BRACE:    "an open brace",
        TOKEN_CLOSE_BRACE:   "a close brace",
        TOKEN_OPEN_PAREN:    "an open paren",
        TOKEN_CLOSE_PAREN:   "a close paren",
        TOKEN_OPEN_BRACKET:  "an open bracket",
        TOKEN_CLOSE_BRACKET: "a close bracket",
        TOKEN_DOLLAR:        "a dollar",
        #TOKEN_QUOTE:         "a quote",
        TOKEN_COMMA:         "a comma",
        TOKEN_SEMICOLON:     "a semicolon",
        TOKEN_OPERATOR:      "an operator",
        TOKEN_DIRECTIVE:     "an directive",
        TOKEN_TEXT:          "text",
        }
    def __init__(self, debug=0):
        self.buffer = []
        self.debug = debug
    def read(self, file, path=None):
        line = 1
        column = 0
        data = kanjilib.sjis2euc(file.read(), "ignore")
        pos = 0
        end = len(data)
        while pos < end:
            if column == 0:
                match = self.re_comment.match(data, pos)
                if match:
                    column = column + len(match.group())
                    pos = match.end()
                    continue
            for token, pattern in self.patterns:
                match = pattern.match(data, pos)
                if match:
                    lexeme = match.group()
                    self.buffer.append((token, lexeme, (line, column)))
                    pos = match.end()
                    break
            else:
                ###print data[pos:pos+100]
                lexical_error(path, (line, column))
            if token == TOKEN_NEWLINE:
                line = line + 1
                column = 0
            else:
                column = column + len(lexeme)
        self.buffer.append((TOKEN_NEWLINE, "", (line, column)))
        self.path = path
    def get_position(self):
        return self.position
    def pop(self):
        try:
            token, lexeme, self.position = self.buffer.pop(0)
        except IndexError:
            syntax_error("unexpected end of file", self.path)
        ###print token, repr(lexeme)
        return token, lexeme
    def pop_check(self, expected):
        token, lexeme = self.pop()
        if token != expected:
            syntax_error("exptected " + self.token_names[expected],
                         self.path, self.get_position())
        return lexeme
    def look_ahead(self, index=0):
        try:
            token, lexeme, position = self.buffer[index]
        except IndexError:
            syntax_error("unexpected end of file", self.path)
        return token, lexeme
    def skip_space(self, accept_eof=0):
        while self.buffer:
            token, lexeme = self.look_ahead()
            if token not in [TOKEN_NEWLINE, TOKEN_WHITESPACE]:
                return 0
            self.pop()
        if not accept_eof:
            syntax_error("unexpected end of file", self.path)
        return 1
    def skip_line(self):
        while self.buffer:
            token, lexeme = self.pop()
            if token == TOKEN_NEWLINE:
                break

###   PARSER   ###

NODE_TEXT        = 1
NODE_STRING      = 2
NODE_VARIABLE    = 3
NODE_INDEXED     = 4
NODE_ASSIGNMENT  = 5
NODE_FUNCTION    = 6
NODE_IF          = 7
NODE_CALC        = 8
NODE_AND_EXPR    = 9
NODE_OR_EXPR     = 10
NODE_COMP_EXPR   = 11
NODE_ADD_EXPR    = 12
NODE_MUL_EXPR    = 13
NODE_POW_EXPR    = 14

class Parser:
    def __init__(self, debug=0):
        self.lexer = Lexer(debug)
        self.debug = debug
    def read(self, file, path=None):
        self.lexer.read(file, path)
        self.path = path
    def get_dict(self):
        common = None
        groups = []
        # skip junk tokens at the beginning of the file
        while 1:
            if self.lexer.skip_space(1):
                return common, groups
            token, lexeme = self.lexer.look_ahead()
            if token == TOKEN_DIRECTIVE and lexeme == "#_Common" or \
               token == TOKEN_DOLLAR:
                break
            self.lexer.skip_line()
        token, lexeme = self.lexer.look_ahead()
        if token == TOKEN_DIRECTIVE and lexeme == "#_Common":
            common = self.get_common()
            if self.lexer.skip_space(1):
                return common, groups
        while 1:
            groups.append(self.get_group())
            if self.lexer.skip_space(1):
                return common, groups
    def get_common(self):
        self.lexer.pop()
        token, lexeme = self.lexer.look_ahead()
        if token == TOKEN_WHITESPACE:
            self.lexer.pop()
        self.lexer.pop_check(TOKEN_NEWLINE)
        condition = self.get_brace_expr()
        self.lexer.pop_check(TOKEN_NEWLINE)
        return condition
    def get_group(self):
        # get group name
        buffer = []
        self.lexer.pop_check(TOKEN_DOLLAR)
        while 1:
            token, lexeme = self.lexer.look_ahead()
            if token == TOKEN_TEXT or \
               token == TOKEN_OPERATOR and lexeme in ["+", "-", "*", "/", "%"]:
                token, lexeme = self.lexer.pop()
                buffer.append(self.unescape(lexeme))
            else:
                break
        if len(buffer) == 0:
            syntax_error("null identifier",
                         self.path, self.lexer.get_position())
        name = "$" + string.join(buffer, "")
        # get group parameters
        parameters = []
        while 1:
            token, lexeme = self.lexer.look_ahead()
            if token == TOKEN_WHITESPACE:
                self.lexer.pop()
            token, lexeme = self.lexer.look_ahead()
            if token == TOKEN_NEWLINE:
                break
            elif token in [TOKEN_COMMA, TOKEN_SEMICOLON]:
                self.lexer.pop()
            else:
                syntax_error("expected a delimiter",
                             self.path, self.lexer.get_position())
            token, lexeme = self.lexer.look_ahead()
            if token == TOKEN_WHITESPACE:
                self.lexer.pop()
            token, lexeme = self.lexer.look_ahead()
            if token == TOKEN_NEWLINE:
                break
            token, lexeme = self.lexer.look_ahead()
            if token == TOKEN_OPEN_BRACE:
                parameters.append(self.get_brace_expr())
            elif token == TOKEN_TEXT:
                token, lexeme = self.lexer.pop()
                parameters.append([NODE_TEXT, self.unescape(lexeme)])
            else:
                syntax_error("expected a parameter or brace expression",
                             self.path, self.lexer.get_position())
        # get sentences
        self.lexer.pop_check(TOKEN_NEWLINE)
        sentences = []
        while 1:
            if self.lexer.skip_space(1):
                break
            token, lexeme = self.lexer.look_ahead()            
            if token == TOKEN_DOLLAR:
                break
            sentence = self.get_sentence()
            if not sentence:
                continue
            sentences.append(sentence)
        return (name, parameters, sentences)
    def get_sentence(self):
        token, lexeme = self.lexer.look_ahead()
        if token == TOKEN_OPEN_BRACE:
            token, lexeme = self.lexer.look_ahead(1)
            if token == TOKEN_NEWLINE:
                self.lexer.pop_check(TOKEN_OPEN_BRACE)
                token, lexeme = self.lexer.look_ahead()
                if token == TOKEN_WHITESPACE:
                    self.lexer.pop()
                self.lexer.pop_check(TOKEN_NEWLINE)
                line = []
                while 1:
                    self.lexer.skip_space()
                    token, lexeme = self.lexer.look_ahead()
                    if token == TOKEN_CLOSE_BRACE:
                        break
                    for node in self.get_line():
                        line.append(node)
                    self.lexer.pop_check(TOKEN_NEWLINE)
                self.lexer.pop_check(TOKEN_CLOSE_BRACE)
                token, lexeme = self.lexer.look_ahead()
                if token == TOKEN_WHITESPACE:
                    self.lexer.pop()
            else:
                try:
                    line = self.get_line() # beginning with brace expression
                except MisakaError, error:
                    if self.debug & 4:
                        print error
                    self.lexer.skip_line()
                    return None
        else:
            try:
                line = self.get_line()
            except MisakaError, error:
                if self.debug & 4:
                    print error
                self.lexer.skip_line()
                return None
        self.lexer.pop_check(TOKEN_NEWLINE)
        return line
    def is_whitespace(self, node):
        return node[0] == NODE_TEXT and not string.strip(node[1])
    def unescape(self, text):
        text = string.replace(text, r'\,', ',')
        text = string.replace(text, r'\"', '"')
        return text
    def get_word(self):
        buffer = []
        while 1:
            token, lexeme = self.lexer.look_ahead()
            if token == TOKEN_TEXT:
                token, lexeme = self.lexer.pop()
                if self.unescape(lexeme)[0] == '"' and \
                   self.unescape(lexeme)[-1] == '"':
                    buffer.append([NODE_STRING, self.unescape(lexeme)[1:-1]])
                else:
                    buffer.append([NODE_TEXT, self.unescape(lexeme)])
            elif token in [TOKEN_WHITESPACE, TOKEN_DOLLAR]:
                token, lexeme = self.lexer.pop()
                buffer.append([NODE_TEXT, lexeme])
            elif token == TOKEN_OPEN_BRACE:
                buffer.append(self.get_brace_expr())
            elif token == TOKEN_QUOTE:
                buffer.append(self.get_string())
            else:
                break
        # strip whitespace at the beginning and/or end of line
        if len(buffer) > 0 and self.is_whitespace(buffer[0]):
            del buffer[0]
        if len(buffer) > 0 and self.is_whitespace(buffer[-1]):
            del buffer[-1]
        return buffer
    def get_line(self):
        buffer = []
        while 1:
            token, lexeme = self.lexer.look_ahead()
            if token in [TOKEN_NEWLINE, TOKEN_CLOSE_BRACE]:
                break
            elif token == TOKEN_TEXT:
                token, lexeme = self.lexer.pop()
                buffer.append([NODE_TEXT, self.unescape(lexeme)])
            elif token in [TOKEN_WHITESPACE, TOKEN_DOLLAR, TOKEN_QUOTE,
                           TOKEN_OPEN_PAREN, TOKEN_CLOSE_PAREN,
                           TOKEN_OPEN_BRACKET, TOKEN_CLOSE_BRACKET,
                           TOKEN_OPERATOR, TOKEN_COMMA, TOKEN_SEMICOLON,
                           TOKEN_DIRECTIVE]:
                token, lexeme = self.lexer.pop()
                buffer.append([NODE_TEXT, lexeme])
            elif token == TOKEN_OPEN_BRACE:
                buffer.append(self.get_brace_expr())
            else:
                raise RuntimeError, "should not reach here"
        # strip whitespace at the beginning and/or end of line
        if len(buffer) > 0 and self.is_whitespace(buffer[0]):
            del buffer[0]
        if len(buffer) > 0 and self.is_whitespace(buffer[-1]):
            del buffer[-1]
        return buffer
    def get_string(self):
        buffer = []
        self.lexer.pop_check(TOKEN_QUOTE)
        while 1:
            token, lexeme = self.lexer.look_ahead()
            if token == TOKEN_QUOTE:
                break
            elif token == TOKEN_TEXT:
                token, lexeme = self.lexer.pop()
                buffer.append(self.unescape(lexeme))
            elif token in [TOKEN_WHITESPACE, TOKEN_DOLLAR,
                           TOKEN_OPEN_BRACE, TOKEN_CLOSE_BRACE,
                           TOKEN_OPEN_PAREN, TOKEN_CLOSE_PAREN,
                           TOKEN_OPEN_BRACKET, TOKEN_CLOSE_BRACKET,
                           TOKEN_OPERATOR, TOKEN_COMMA, TOKEN_SEMICOLON,
                           TOKEN_DIRECTIVE]:
                token, lexeme = self.lexer.pop()
                buffer.append(lexeme)
            elif token == TOKEN_NEWLINE:
                syntax_error("unexpected end of line", self.path)
            else:
                raise RuntimeError, "should not reach here"
        self.lexer.pop_check(TOKEN_QUOTE)
        return [NODE_STRING, string.join(buffer, "")]
    def get_brace_expr(self):
        self.lexer.pop_check(TOKEN_OPEN_BRACE)
        self.lexer.skip_space()
        self.lexer.pop_check(TOKEN_DOLLAR)
        self.lexer.skip_space()
        # get identifier (function or variable)
        nodelist = [[NODE_TEXT, "$"]]
        while 1:
            token, lexeme = self.lexer.look_ahead()
            if token == TOKEN_TEXT or \
               token == TOKEN_OPERATOR and lexeme in ["+", "-", "*", "/", "%"]:
                token, lexeme = self.lexer.pop()
                nodelist.append([NODE_TEXT, self.unescape(lexeme)])
            elif token == TOKEN_OPEN_BRACE:
                nodelist.append(self.get_brace_expr())
            else:
                break
        if len(nodelist) == 1:
            syntax_error("null identifier",
                         self.path, self.lexer.get_position())
        self.lexer.skip_space()
        token, lexeme = self.lexer.look_ahead()
        if token == TOKEN_OPEN_PAREN:
            # function
            name = string.join(map(lambda node: node[1], nodelist), "")
            if name == "$if":
                cond_expr = self.get_cond_expr()
                then_clause = [[NODE_TEXT, "true"]]
                else_clause = [[NODE_TEXT, "false"]]
                self.lexer.skip_space()
                token, lexeme = self.lexer.look_ahead()
                if token == TOKEN_OPEN_BRACE:
                    self.lexer.pop_check(TOKEN_OPEN_BRACE)
                    self.lexer.skip_space()
                    then_clause = []
                    while 1:
                        line = self.get_line()
                        for node in line:
                            then_clause.append(node)
                        token, lexem = self.lexer.pop()
                        if token == TOKEN_CLOSE_BRACE:
                            break
                        assert (token == TOKEN_NEWLINE)
                    self.lexer.skip_space()
                    token, lexeme = self.lexer.look_ahead()
                    if token == TOKEN_TEXT and lexeme == "else":
                        self.lexer.pop()
                        self.lexer.skip_space()
                        self.lexer.pop_check(TOKEN_OPEN_BRACE)
                        self.lexer.skip_space()
                        else_clause = []
                        while 1:
                            line = self.get_line()
                            for node in line:
                                else_clause.append(node)
                            token, lexem = self.lexer.pop()
                            if token == TOKEN_CLOSE_BRACE:
                                break
                            assert (token == TOKEN_NEWLINE)
                    else:
                        else_clause = [[NODE_TEXT, ""]]
                node = [NODE_IF, cond_expr, then_clause, else_clause]
            elif name == "$calc":
                node = [NODE_CALC, self.get_add_expr()]
            else:
                self.lexer.pop_check(TOKEN_OPEN_PAREN)
                self.lexer.skip_space()
                args = []
                token, lexeme = self.lexer.look_ahead()
                if token != TOKEN_CLOSE_PAREN:
                    while 1:
                        args.append(self.get_argument())
                        self.lexer.skip_space()
                        token, lexeme = self.lexer.look_ahead()
                        if token != TOKEN_COMMA:
                            break
                        self.lexer.pop()
                        self.lexer.skip_space()
                self.lexer.pop_check(TOKEN_CLOSE_PAREN)
                node = [NODE_FUNCTION, name, args]
        else:
            # variable
            token, lexeme = self.lexer.look_ahead()
            if token == TOKEN_OPERATOR:
                operator = self.lexer.pop_check(TOKEN_OPERATOR)
                if operator in ["=", "+=", "-=", "*=", "/="]:
                    self.lexer.skip_space()
                    value = self.get_add_expr()
                elif operator == "++":
                    operator = "+="
                    value = [[NODE_TEXT, "1"]]
                elif operator == "--":
                    operator = "-="
                    value = [[NODE_TEXT, "1"]]
                else:
                    syntax_error("bad operator " + operator,
                                 self.path, self.lexer.get_position())
                node = [NODE_ASSIGNMENT, nodelist, operator, value]
            elif token == TOKEN_OPEN_BRACKET:
                self.lexer.pop_check(TOKEN_OPEN_BRACKET)
                self.lexer.skip_space()
                index = self.get_word()
                self.lexer.skip_space()
                self.lexer.pop_check(TOKEN_CLOSE_BRACKET)
                node = [NODE_INDEXED, nodelist, index]
            else:
                node = [NODE_VARIABLE, nodelist]
        self.lexer.skip_space()
        self.lexer.pop_check(TOKEN_CLOSE_BRACE)
        return node
    def get_argument(self):
        buffer = []
        while 1:
            token, lexeme = self.lexer.look_ahead()
            if token == TOKEN_TEXT:
                token, lexeme = self.lexer.pop()
                if self.unescape(lexeme)[0] == '"' and \
                   self.unescape(lexeme)[-1] == '"':
                    buffer.append([NODE_STRING, self.unescape(lexeme)[1:-1]])
                else:
                    buffer.append([NODE_TEXT, self.unescape(lexeme)])
            elif token in [TOKEN_WHITESPACE, TOKEN_DOLLAR,
                           TOKEN_OPEN_BRACKET, TOKEN_CLOSE_BRACKET,
                           TOKEN_OPERATOR, TOKEN_SEMICOLON]:
                token, lexeme = self.lexer.pop()
                buffer.append([NODE_TEXT, lexeme])
            elif token == TOKEN_OPEN_BRACE:
                buffer.append(self.get_brace_expr())
            elif token == TOKEN_QUOTE:
                buffer.append(self.get_string())
            elif token == TOKEN_NEWLINE:
                self.lexer.skip_space()
            else:
                break
        # strip whitespace at the beginning and/or end of line
        if len(buffer) > 0 and self.is_whitespace(buffer[0]):
            del buffer[0]
        if len(buffer) > 0 and self.is_whitespace(buffer[-1]):
            del buffer[-1]
        return buffer
    def get_cond_expr(self):
        self.lexer.pop_check(TOKEN_OPEN_PAREN)
        self.lexer.skip_space()
        or_expr = self.get_or_expr()
        self.lexer.skip_space()
        self.lexer.pop_check(TOKEN_CLOSE_PAREN)
        return or_expr
    def get_or_expr(self):
        buffer = [NODE_OR_EXPR]
        buffer.append(self.get_and_expr())
        while 1:
            self.lexer.skip_space()
            token, lexeme = self.lexer.look_ahead()
            if not (token == TOKEN_OPERATOR and lexeme == "||"):
                break
            self.lexer.pop()
            self.lexer.skip_space()
            buffer.append(self.get_and_expr())
        if len(buffer) > 2:
            return buffer
        else:
            return buffer[1]
    def get_and_expr(self):
        buffer = [NODE_AND_EXPR]
        buffer.append(self.get_sub_expr())
        while 1:
            self.lexer.skip_space()
            token, lexeme = self.lexer.look_ahead()
            if not (token == TOKEN_OPERATOR and lexeme == "&&"):
                break
            self.lexer.pop()
            self.lexer.skip_space()
            buffer.append(self.get_sub_expr())
        if len(buffer) > 2:
            return buffer
        else:
            return buffer[1]
    def get_sub_expr(self):
        token, lexeme = self.lexer.look_ahead()
        if token == TOKEN_OPEN_PAREN:
            return self.get_cond_expr()
        else:
            return self.get_comp_expr()
    def get_comp_expr(self):
        buffer = [NODE_COMP_EXPR]
        buffer.append(self.get_add_expr())
        self.lexer.skip_space()
        token, lexeme = self.lexer.look_ahead()
        if token == TOKEN_OPERATOR and \
           lexeme in ["==", "!=", "<", "<=", "=<", ">", ">=", "=>"]:
            if lexeme == "=<":
                lexeme = "<="
            elif lexeme == "=>":
                lexeme = ">="
            buffer.append(lexeme)
            self.lexer.pop()
            self.lexer.skip_space()
            buffer.append(self.get_add_expr())
        if len(buffer) == 2:
            buffer.append("==")
            buffer.append([[NODE_TEXT, "true"]])
        return buffer
    def get_add_expr(self):
        buffer = [NODE_ADD_EXPR]
        buffer.append(self.get_mul_expr())
        while 1:
            self.lexer.skip_space()
            token, lexeme = self.lexer.look_ahead()
            if not (token == TOKEN_OPERATOR and lexeme in ["+", "-"]):
                break
            buffer.append(lexeme)
            self.lexer.pop()
            self.lexer.skip_space()
            buffer.append(self.get_mul_expr())
        if len(buffer) > 2:
            return [buffer]
        return buffer[1]
    def get_mul_expr(self):
        buffer = [NODE_MUL_EXPR]
        buffer.append(self.get_pow_expr())
        while 1:
            self.lexer.skip_space()
            token, lexeme = self.lexer.look_ahead()
            if not (token == TOKEN_OPERATOR and lexeme in ["*", "/", "%"]):
                break
            buffer.append(lexeme)
            self.lexer.pop()
            self.lexer.skip_space()
            buffer.append(self.get_pow_expr())
        if len(buffer) > 2:
            return [buffer]
        return buffer[1]
    def get_pow_expr(self):
        buffer = [NODE_POW_EXPR]
        buffer.append(self.get_unary_expr())
        while 1:
            self.lexer.skip_space()
            token, lexeme = self.lexer.look_ahead()
            if not (token == TOKEN_OPERATOR and lexeme == "^"):
                break
            self.lexer.pop()
            self.lexer.skip_space()
            buffer.append(self.get_unary_expr())
        if len(buffer) > 2:
            return [buffer]
        return buffer[1]
    def get_unary_expr(self):
        token, lexeme = self.lexer.look_ahead()
        if token == TOKEN_OPERATOR and lexeme in ["+", "-"]:
            buffer = [NODE_ADD_EXPR, [[NODE_TEXT, "0"]], lexeme]
            self.lexer.pop()
            self.lexer.skip_space()
            buffer.append(self.get_unary_expr())
            return [buffer]
        else:
            return self.get_factor()
    def get_factor(self):
        token, lexeme = self.lexer.look_ahead()
        if token == TOKEN_OPEN_PAREN:
            self.lexer.pop_check(TOKEN_OPEN_PAREN)
            self.lexer.skip_space()
            add_expr = self.get_add_expr()
            self.lexer.skip_space()
            self.lexer.pop_check(TOKEN_CLOSE_PAREN)
            return add_expr
        else:
            return self.get_word()
    # for debug
    def dump_list(self, nodelist, depth=0):
        for node in nodelist:
            self.dump_node(node, depth)
    def dump_node(self, node, depth):
        indent = "  " * depth
        if node[0] == NODE_TEXT:
            print indent + "TEXT", '"' + node[1] + '"'
        elif node[0] == NODE_STRING:
            print indent + "STRING", '"' + node[1] + '"'
        elif node[0] == NODE_VARIABLE:
            print indent + "VARIABLE"
            self.dump_list(node[1], depth+1)
        elif node[0] == NODE_INDEXED:
            print indent + "INDEXED"
            self.dump_list(node[1], depth+1)
            print indent + "index"
            self.dump_list(node[2], depth+1)
        elif node[0] == NODE_ASSIGNMENT:
            print indent + "ASSIGNMENT"
            self.dump_list(node[1], depth+1)
            print indent + "op", node[2]
            self.dump_list(node[3], depth+1)
        elif node[0] == NODE_FUNCTION:
            print indent + "FUNCTION", node[1]
            for i in range(len(node[2])):
                print indent + "args[%d]" % i
                self.dump_list(node[2][i], depth+1)
        elif node[0] == NODE_IF:
            print indent + "IF"
            self.dump_node(node[1], depth+1)
            print indent + "then"
            self.dump_list(node[2], depth+1)
            print indent + "else"
            self.dump_list(node[3], depth+1)
        elif node[0] == NODE_CALC:
            print indent + "CALC"
            self.dump_list(node[1], depth+1)
        elif node[0] == NODE_AND_EXPR:
            print indent + "AND_EXPR"
            for i in range(1, len(node)):
                self.dump_node(node[i], depth+1)
        elif node[0] == NODE_OR_EXPR:
            print indent + "OR_EXPR"
            for i in range(1, len(node)):
                self.dump_node(node[i], depth+1)
        elif node[0] == NODE_COMP_EXPR:
            print indent + "COMP_EXPR"
            self.dump_list(node[1], depth+1)
            print indent + "op", node[2]
            self.dump_list(node[3], depth+1)
        elif node[0] == NODE_ADD_EXPR:
            print indent + "ADD_EXPR"
            self.dump_list(node[1], depth+1)
            for i in range(2, len(node), 2):
                print indent + "op", node[i]
                self.dump_list(node[i+1], depth+1)
        elif node[0] == NODE_MUL_EXPR:
            print indent + "MUL_EXPR"
            self.dump_list(node[1], depth+1)
            for i in range(2, len(node), 2):
                print indent + "op", node[i]
                self.dump_list(node[i+1], depth+1)
        elif node[0] == NODE_POW_EXPR:
            print indent + "POW_EXPR"
            self.dump_list(node[1], depth+1)
            for i in range(2, len(node)):
                print indent + "op ^"
                self.dump_list(node[i], depth+1)
        else:
            print node
            raise RuntimeError, "should not reach here"

# <<< syntax >>>
# dict       := sp ( common sp )? ( group sp )*
# sp         := ( NEWLINE | WHITESPACE )*
# common     := "#_Common" NEWLINE brace_expr NEWLINE
# group      := group_name ( delimiter ( STRING | brace_expr ) )*
#               ( delimiter )? NEWLINE ( sentence )*
# group_name := DOLLAR ( TEXT )+
# delimiter  := ( WHITESPACE )? ( COMMA | SEMICOLON ) ( WHITESPACE )?
# sentence   := line NEWLINE |
#               OPEN_BRACE NEWLINE ( line NEWLINE )* CLOSE_BRACE NEWLINE
# word       := ( TEXT | WHITESPACE | DOLLAR | brace_expr | string )*
# line       := ( TEXT | WHITESPACE | DOLLAR | brace_expr | QUOTE |
#                 OPEN_PAREN | CLOSE_PAREN | OPEN_BRACKET | CLOSE_BRACKET |
#                 OPERATOR | COMMA | SEMICOLON | DIRECTIVE )*
# string     := QUOTE (
#                 TEXT | WHITESPACE | DOLLAR | OPEN_BRACE | CLOSE_BRACE |
#                 OPEN_PAREN | CLOSE_PAREN | OPEN_BRACKET | CLOSE_BRACKET |
#                 OPERATOR | COMMA | SEMICOLON | DIRECTIVE )*
#               QUOTE
# brace_expr := variable | indexed | assignment | increment |
#               function | if | calc
# variable   := OPEN_BRACE sp var_name sp CLOSE_BRACE
# indexed    := OPEN_BRACE sp var_name sp
#               OPEN_BRACKET sp word sp CLOSE_BRACKET sp CLOSE_BRACE
# var_name   := DOLLAR ( TEXT | brace_expr )+
# assignment := OPEN_BRACE sp var_name sp assign_ops sp add_expr sp CLOSE_BRACE
# assign_ops := "=" | "+=" | "-=" | "*=" | "/="
# increment  := OPEN_BRACE sp var_name sp inc_ops sp CLOSE_BRACE
# inc_ops    := "+" "+" | "-" "-"
# function   := OPEN_BRACE sp DOLLAR sp ( TEXT )+ sp OPEN_PAREN
#               ( sp argument ( sp COMMA sp argument )* sp )?
#               CLOSE_PAREN sp CLOSE_BRACE
# argument   := ( TEXT | WHITESPACE | DOLLAR | brace_expr | string |
#                 OPEN_BRACKET | CLOSE_BRACKET | OPERATOR | SEMICOLON )*
# if         := OPEN_BRACE sp DOLLAR sp "if" sp cond_expr
#               ( sp OPEN_BRACE ( sp line )* sp CLOSE_BRACE ( sp "else"
#                 sp OPEN_BRACE ( sp line )* sp CLOSE_BRACE )? sp )?
#               sp CLOSE_BRACE
# calc       := OPEN_BRACE sp DOLLAR sp "calc" sp add_expr sp CLOSE_BRACE
# cond_expr  := OPEN_PAREN sp or_expr sp CLOSE_PAREN
# or_expr    := and_expr ( sp "||" sp and_expr )*
# and_expr   := sub_expr ( sp "&&" sp sub_expr )*
# sub_expr   := comp_expr | cond_expr
# comp_expr  := add_expr ( sp comp_op sp add_expr )?
# comp_op    := "==" | "!=" | "<" | "<=" | ">" | ">="
# add_expr   := mul_expr (sp add_op sp mul_expr)*
# add_op     := "+" | "-"
# mul_expr   := pow_expr (sp mul_op sp pow_expr)*
# mul_op     := "*" | "/" | "%"
# pow_expr   := unary_expr (sp pow_op sp unary_expr)*
# pow_op     := "^"
# unary_expr := unary_op sp unary_expr | factor
# unary_op   := "+" | "-"
# factor     := OPEN_PAREN sp add_expr sp CLOSE_PAREN | word

###   INTERPRETER   ###

class Group:
    def __init__(self, misaka, name, list=None):
        self.misaka = misaka
        self.name = name
        self.list = list or []
    def __len__(self):
        return len(self.list)
    def __getitem__(self, index):
        return self.list[index]
    def get(self):
        if not self.list:
            return None
        return whrandom.choice(self.list)
    def copy(self, name):
        return Array(self.misaka, name, self.list[:])

class NonOverlapGroup(Group):
    def __init__(self, misaka, name, list=None):
        Group.__init__(self, misaka, name, list)
        self.indexes = []
    def get(self):
        if not self.list:
            return None
        if not self.indexes:
            self.indexes = range(len(self.list))
        index = self.indexes.pop(whrandom.choice(range(len(self.indexes))))
        return self.list[index]

class SequentialGroup(Group):
    def __init__(self, misaka, name, list=None):
        Group.__init__(self, misaka, name, list)
        self.indexes = []
    def get(self):
        if not self.list:
            return None
        if not self.indexes:
            self.indexes = range(len(self.list))
        index = self.indexes.pop(0)
        return self.list[index]

class Array(Group):
    def append(self, s):
        self.list.append([[NODE_TEXT, str(s)]])
    def index(self, s):
        for i in range(len(self.list)):
            if s == self.misaka.expand(self.list[i]):
                return i
        return -1
    def pop(self):
        if not self.list:
            return None
        return self.list.pop(whrandom.choice(range(len(self.list))))
    def popmatchl(self, s):
        buffer = []
        for i in range(len(self.list)):
            t = self.misaka.expand(self.list[i])
            if s == t[:len(s)]:
                buffer.append(i)
        if not buffer:
            return None
        return self.list.pop(whrandom.choice(buffer))

TYPE_SCHOLAR = 1
TYPE_ARRAY   = 2

class Shiori:
    DBNAME = "misaka.db"
    def __init__(self, dll_name, debug=0):
        self.dll_name = dll_name
        self.debug = debug
    def use_saori(self, saori):
        self.saori = saori
    def find(self, dir, dll_name):
        result = 0
        if list_dict(dir):
            result = 210
        elif os.path.isfile(os.path.join(dir, 'misaka.ini')):
            result = 200
        return result
    def show_description(self):
        sys.stdout.write('Shiori: MISAKA compatible module for ninix\n'
                         '        Copyright (C) 2002 by Tamito KAJIYAMA\n')
    def reset(self):
        self.dict = {}
        self.variable = {}
        self.constant = {}
        self.misaka_debug = 0
        self.misaka_error = 0
        self.reference = (None, None, None, None, None, None, None, None)
        self.mouse_move_count = {}
        self.random_talk = -1
        self.otherghost = []
        self.communicate = ["", ""]
        self.reset_request()
    def load(self, misaka_dir=os.curdir):
        self.misaka_dir = misaka_dir
        self.dbpath = os.path.join(self.misaka_dir, self.DBNAME)
        self.saori_library = SaoriLibrary(self.saori, self.misaka_dir)
        self.reset()
        try:
            filelist, debug, error = read_misaka_ini(self.misaka_dir, self.debug)
        except IOError:
            if self.debug & 4:
                print "cannot read misaka.ini"
            return 0
        except MisakaError, error:
            if self.debug & 4:
                print error
            return 0
        self.misaka_debug = debug
        self.misaka_error = error
        global_variables = []
        global_constants = []
        for filename in filelist:
            path = os.path.join(self.misaka_dir, filename)
            try:
                file = builtin_open(path)
            except IOError:
                if self.debug & 4:
                    print "cannot read", filename
                continue
            basename, suffix = os.path.splitext(filename)
            if suffix == ".__1":
                file = StringIO.StringIO(self.crypt(file.read()))
            try:
                variables, constants = self.read(file, path)
            except MisakaError, error:
                if self.debug & 4:
                    print error
                continue
            global_variables.extend(variables)
            global_constants.extend(constants)
        self.eval_globals(global_variables, 0)
        self.load_database()
        self.eval_globals(global_constants, 1)
        if self.dict.has_key("$_OnRandomTalk"):
            self.random_talk = 0
        return 1
    def crypt(self, data):
        return string.join(map(lambda c: chr(ord(c) ^ 0xff), data), "")
    def eval_globals(self, sentences, constant):
        for sentence in sentences:
            for node in sentence:
                if node[0] == NODE_ASSIGNMENT:
                    self.eval_assignment(node[1], node[2], node[3], constant)
                elif node[0] == NODE_FUNCTION:
                    self.eval_function(node[1], node[2])
    def read(self, file, path=None):
        parser = Parser(self.debug)
        parser.read(file, path)
        common, dict = parser.get_dict()
        variables = []
        constants = []
        for name, parameters, sentences in dict:
            if len(sentences) == 0:
                continue
            elif name == "$_Variable":
                variables.extend(sentences)
                continue
            elif name == "$_Constant":
                constants.extend(sentences)
                continue
            GroupClass = Group
            conditions = []
            if common is not None:
                conditions.append(common)
            for node in parameters:
                if node[0] == NODE_IF:
                    conditions.append(node)
                elif node[0] == NODE_TEXT and node[1] == "nonoverlap":
                    GroupClass = NonOverlapGroup
                elif node[0] == NODE_TEXT and node[1] == "sequential":
                    GroupClass = SequentialGroup
                else:
                    pass # ignore unknown parameters
            group = GroupClass(self, name, sentences)
            try:
                grouplist = self.dict[name]
            except KeyError:
                grouplist = self.dict[name] = []
            grouplist.append((group, conditions))
        return variables, constants
    def strip_newline(self, line):
        if line[-2:] == "\r\n":
            return line[:-2]
        elif line[-1] in "\r\n":
            return line[:-1]
        return line
    def load_database(self):
        try:
            file = builtin_open(self.dbpath)
        except IOError:
            return
        while 1:
            line = file.readline()
            if not line:
                break
            header = string.split(self.strip_newline(line))
            if header[0] == "SCHOLAR":
                value = self.strip_newline(file.readline())
                self.variable[header[1]] = (TYPE_SCHOLAR, value)
            elif header[0] == "ARRAY":
                try:
                    size = int(header[1])
                except ValueError:
                    if self.debug & 4:
                        print "misaka.py: malformed database (ignored)"
                    break
                array = Array(self, header[2])
                for i in range(size):
                    array.append(self.strip_newline(file.readline()))
                self.variable[header[2]] = (TYPE_ARRAY, array)
            else:
                raise RuntimeError, "should not reach here"
        file.close()
    def save_database(self):
        try:
            file = builtin_open(self.dbpath, "w")
        except IOError:
            if self.debug & 4:
                print "misaka.py: cannot write database (ignored)"
            return
        for name in self.variable.keys():
            type, value = self.variable[name]
            if type == TYPE_SCHOLAR:
                if value == "":
                    continue
                file.write("SCHOLAR %s\n%s\n" % (name, str(value)))
            elif type == TYPE_ARRAY:
                file.write("ARRAY %d %s\n" % (len(value), name))
                for item in value:
                    file.write("%s\n" % self.expand(item))
            else:
                raise RuntimeError, "should not reach here"
        file.close()
    def unload(self):
        self.save_database()
        self.saori_library.unload()
        return 1
    def reset_request(self):
        self.req_command = ''
        self.req_protocol = ''
        self.req_key = []
        self.req_header = {}
    def request(self, req_string):
        header = StringIO.StringIO(req_string)
        line = header.readline()
        if line:
            if line[-1] == '\n':
                line = line[:-1]
            line = string.strip(line)
            list = string.split(line)
            if len(list) >= 2:
                self.req_command = string.strip(list[0])
                self.req_protocol = string.strip(list[1])
            while 1:
                line = header.readline()
                if not line:
                    break # EOF
                if line[-1] == '\n':
                    line = line[:-1]
                line = string.strip(line)
                if not line:
                    continue
                colon = string.find(line, ':')
                if colon >= 0:
                    key = string.strip(line[:colon])
                    value = string.strip(line[colon+1:])
                    try:
                        value = int(value)
                    except:
                        value = str(value)
                    self.req_key.append(key)
                    self.req_header[key] = value
                else:
                    continue
            for key in self.req_header.keys():
                if type(self.req_header[key]) is StringType:
                    self.req_header[key] = kanjilib.sjis2euc(self.req_header[key], 'ignore')
        # FIXME
        ref0 = self.get_ref(0)
        ref1 = self.get_ref(1)
        ref2 = self.get_ref(2)
        ref3 = self.get_ref(3)
        ref4 = self.get_ref(4)
        ref5 = self.get_ref(5)
        ref6 = self.get_ref(6)
        ref7 = self.get_ref(7)
        event = self.req_header['ID']
        script = None
        if event == "otherghostname":
            n = 0
            refs = []
            while 1:
                ref = self.get_ref(n)
                if ref:
                    refs.append(ref)
                else:
                    break
                n = n + 1
            self.otherghost = []
            for ref in refs:
                name, s0, s1 = string.split(ref, chr(1))
                self.otherghost.append([name, s0, s1])
        if event == "OnMouseMove":
            key = (ref3, ref4) # side, part
            count = self.mouse_move_count.get(key, 0)
            self.mouse_move_count[key] = count + 5
        if event == "OnCommunicate":
            self.communicate = [ref0, ref1]
            script = self.get("$_OnGhostCommunicateReceive", default=None)
        else:
            self.reference = (ref0, ref1, ref2, ref3, ref4, ref5, ref6, ref7)
            script = self.get("$" + event, default=None)
        if event == "OnSecondChange":
            if self.random_talk == 0:
                self.reset_random_talk_interval()
            elif self.random_talk > 0:
                self.random_talk = self.random_talk - 1
                if self.random_talk == 0:
                    script = self.get("$_OnRandomTalk", default=None)
        if script is not None:
            script = string.strip(script)
        else:
            script = ''
        self.reset_request()
        response = 'SHIORI/3.0 200 OK\r\n' \
                   'Sender: Misaka\r\n' \
                   'Value: %s\r\n' % kanjilib.euc2sjis(script, 'ignore')
        to = self.eval_variable([[NODE_TEXT, "$to"]])
        if to:
            del self.variable["$to"]
            response = response + 'Reference0: %s\r\n' % kanjilib.euc2sjis(to, 'ignore')
        response = response + '\r\n'
        return response
    def get_ref(self, num):
        key = 'Reference'+str(num)
        if self.req_header.has_key(key):
            return self.req_header[key]
        else:
            return None
    # internal
    def reset_random_talk_interval(self):
        interval = 0
        type, value = self.variable.get("$_talkinterval", (TYPE_SCHOLAR, ""))
        if type == TYPE_SCHOLAR:
            try:
                interval = max(int(value), 0)
            except ValueError:
                pass
        self.random_talk = int(interval * whrandom.randint(5, 15) / 10.0)
    def get(self, name, default=""):
        grouplist = self.dict.get(name)
        if grouplist is None:
            return default
        for group, conditions in grouplist:
            if self.eval_and_expr(conditions) == "true":
                return self.expand(group.get())
        return default
    def expand(self, nodelist):
        if nodelist is None:
            return ""
        buffer = []
        for node in nodelist:
            buffer.append(self.eval(node))
        return string.join(buffer, "")
    def eval(self, node):
        if node[0] == NODE_TEXT:
            return node[1]
        elif node[0] == NODE_STRING:
            return node[1]
        elif node[0] == NODE_VARIABLE:
            return self.eval_variable(node[1])
        elif node[0] == NODE_INDEXED:
            return self.eval_indexed(node[1], node[2])
        elif node[0] == NODE_ASSIGNMENT:
            return self.eval_assignment(node[1], node[2], node[3])
        elif node[0] == NODE_FUNCTION:
            return self.eval_function(node[1], node[2])
        elif node[0] == NODE_IF:
            return self.eval_if(node[1], node[2], node[3])
        elif node[0] == NODE_CALC:
            return self.eval_calc(node[1])
        elif node[0] == NODE_COMP_EXPR:
            return self.eval_comp_expr(node[1], node[2], node[3])
        elif node[0] == NODE_AND_EXPR:
            return self.eval_and_expr(node[1:])
        elif node[0] == NODE_OR_EXPR:
            return self.eval_or_expr(node[1:])
        elif node[0] == NODE_ADD_EXPR:
            return self.eval_add_expr(node[1:])
        elif node[0] == NODE_MUL_EXPR:
            return self.eval_mul_expr(node[1:])
        elif node[0] == NODE_POW_EXPR:
            return self.eval_pow_expr(node[1:])
        ##print node
        raise RuntimeError, "should not reach here"
    SYSTEM_VARIABLES = [
        # current date and time
        "$year", "$month", "$day", "$hour", "$minute", "$second",
        # day of week (0 = Sunday)
        "$dayofweek",
        # ghost uptime
        "$elapsedhour", "$elapsedminute", "$elapsedsecond",
        # OS uptime
        "$elapsedhouros", "$elapsedminuteos", "$elapsedsecondos",
        # OS accumlative uptime
        "$elapsedhourtotal", "$elapsedminutetotal", "$elapsedsecondtotal",
        # system information
        "$os.version",
        "$os.name",
        "$os.phisicalmemorysize",
        "$os.freememorysize",
        "$os.totalmemorysize",
        "$cpu.vendorname",
        "$cpu.name",
        "$cpu.clockcycle",
        # number of days since the last network update
        "$daysfromlastupdate",
        # number of days since the first boot
        "$daysfromfirstboot",
        # name of the ghost with which it has communicated
        ##"$to",
        "$sender",
        "$lastsentence",
        "$otherghostlist",
        ]
    def eval_variable(self, name):
        now = time.localtime(time.time())
        name = self.expand(name)
        if name == "$year":
            return str(now[0])
        elif name == "$month":
            return str(now[1])
        elif name == "$day":
            return str(now[2])
        elif name == "$hour":
            return str(now[3])
        elif name == "$minute":
            return str(now[4])
        elif name == "$second":
            return str(now[5])
        elif name == "$dayofweek":
            return str([1, 2, 3, 4, 5, 6, 0][now[6]])
        elif name == "$lastsentence":
            return self.communicate[1]
        elif name == "$otherghostlist":
            if len(self.otherghost) > 0:
                ghost = whrandom.choice(self.otherghost)
                return ghost[0]
            else:
                return ""
        elif name == "$sender":
            return self.communicate[0]
        elif name in self.SYSTEM_VARIABLES:
            return ""
        elif self.dict.has_key(name):
            return self.get(name)
        elif self.constant.has_key(name):
            type, value = self.constant[name]
            if type == TYPE_ARRAY:
                return self.expand(value.get())
            return str(value)
        elif self.variable.has_key(name):
            type, value = self.variable[name]
            if type == TYPE_ARRAY:
                return self.expand(value.get())
            return str(value)
        return ""
    def eval_indexed(self, name, index):
        name = self.expand(name)
        index = self.expand(index)
        try:
            index = int(index)
        except ValueError:
            index = 0
        if name == "$otherghostlist":
            group = []
            for ghost in self.otherghost:
                group.append(ghost[0])
        elif name in self.SYSTEM_VARIABLES:
            return ""
        elif self.dict.has_key(name):
            grouplist = self.dict[name]
            group, constraints = grouplist[0]
        elif self.constant.has_key(name):
            return ""
        elif self.variable.has_key(name):
            type, group = self.variable[name]
            if type != TYPE_ARRAY:
                return ""
        else:
            return ""
        if index < 0 or index >= len(group):
            return ""
        return self.expand(group[index])
    def eval_assignment(self, name, operator, value, constant=0):
        name = self.expand(name)
        value = self.expand(value)
        if operator in ["+=", "-=", "*=", "/="]:
            try:
                operand = int(value)
            except ValueError:
                operand = 0
            type, value = self.variable.get(name, (TYPE_SCHOLAR, ""))
            if type == TYPE_ARRAY:
                return ""
            try:
                value = int(value)
            except ValueError:
                value = 0
            if operator == "+=":
                value = value + operand
            elif operator == "-=":
                value = value - operand
            elif operator == "*=":
                value = value * operand
            elif operator == "/=" and operand != 0:
                value = value / operand
        if constant:
            self.constant[name] = (TYPE_SCHOLAR, value)
        else:
            self.variable[name] = (TYPE_SCHOLAR, value)
        if name == "$_talkinterval":
            self.reset_random_talk_interval()
        return ""
    def eval_comp_expr(self, operand1, operator, operand2):
        value1 = self.expand(operand1)
        value2 = self.expand(operand2)
        try:
            operand1 = int(value1)
            operand2 = int(value2)
        except ValueError:
            operand1 = value1
            operand2 = value2
        if operator == "==" and operand1 == operand2 or \
           operator == "!=" and operand1 != operand2 or \
           operator == "<"  and operand1 <  operand2 or \
           operator == "<=" and operand1 <= operand2 or \
           operator == ">"  and operand1 >  operand2 or \
           operator == ">=" and operand1 >= operand2:
            return "true"
        return "false"
    def eval_and_expr(self, conditions):
        for condition in conditions:
            boolean = self.eval(condition)
            if string.strip(boolean) != "true":
                return "false"
        return "true"
    def eval_or_expr(self, conditions):
        for condition in conditions:
            boolean = self.eval(condition)
            if string.strip(boolean) == "true":
                return "true"
        return "false"
    def eval_add_expr(self, expression):
        value = self.expand(expression[0])
        try:
            value = int(value)
        except ValueError:
            value = 0
        for i in range(1, len(expression), 2):
            operand = self.expand(expression[i+1])
            try:
                operand = int(operand)
            except ValueError:
                operand = 0
            if expression[i] == "+":
                value = value + operand
            elif expression[i] == "-":
                value = value - operand
        return str(value)
    def eval_mul_expr(self, expression):
        value = self.expand(expression[0])
        try:
            value = int(value)
        except ValueError:
            value = 0
        for i in range(1, len(expression), 2):
            operand = self.expand(expression[i+1])
            try:
                operand = int(operand)
            except ValueError:
                operand = 0
            if expression[i] == "*":
                value = value * operand
            elif expression[i] == "/" and operand != 0:
                value = value / operand
            elif expression[i] == "%" and operand != 0:
                value = value % operand
        return str(value)
    def eval_pow_expr(self, expression):
        value = self.expand(expression[-1])
        try:
            value = int(value)
        except ValueError:
            value = 0
        for i in range(1, len(expression)):
            operand = self.expand(expression[-i-1])
            try:
                operand = int(operand)
            except ValueError:
                operand = 0
            value = operand ** value
        return str(value)
    def eval_if(self, condition, then_clause, else_clause):
        boolean = self.eval(condition)
        if string.strip(boolean) == "true":
            return self.expand(then_clause)
        else:
            return self.expand(else_clause)
    def eval_calc(self, expression):
        return self.expand(expression)
    def eval_function(self, name, args):
        function = self.SYSTEM_FUNCTIONS.get(name)
        if function is None:
            return ""
        return function(self, args)
    def is_number(self, s):
        return s and filter(lambda c: c in "0123456789", s) == s
    def split(self, s):
        buffer = []
        i, j = 0, len(s)
        while i < j:
            if ord(s[i]) < 0x80:
                buffer.append(s[i])
                i = i + 1
            else:
                buffer.append(s[i:i+2])
                i = i + 2
        return buffer
    def exec_reference(self, args):
        if len(args) != 1:
            return ""
        n = self.expand(args[0])
        if n in '01234567':
            value = self.reference[int(n)]
            if value is not None:
                return str(value)
        return ""
    def exec_random(self, args):
        if len(args) != 1:
            return ""
        n = self.expand(args[0])
        try:
            return str(whrandom.randrange(int(n)))
        except ValueError:
            return "" # n < 1 or not a number
    def exec_choice(self, args):
        if len(args) == 0:
            return ""
        return self.expand(whrandom.choice(args))
    def exec_getvalue(self, args):
        if len(args) != 2:
            return ""
        try:
            n = int(self.expand(args[1]))
        except ValueError:
            return ""
        if n < 0:
            return ""
        list = string.split(self.expand(args[0]), ",")
        try:
            return list[n]
        except IndexError:
            return ""
    def exec_search(self, args):
        namelist = []
        for arg in args:
            name = self.expand(arg)
            if string.strip(name):
                namelist.append(name)
        if len(namelist) == 0:
            return ""
        keylist = []
        for key in self.keys(): # dict, variable, constant
            for name in namelist:
                if string.find(key, name) < 0:
                    break
            else:
                keylist.append(key)
        if len(keylist) == 0:
            return ""
        return self.eval_variable([[NODE_TEXT, whrandom.choice(keylist)]])
    def keys(self):
        buffer = self.dict.keys()
        buffer.extend(self.variable.keys())
        buffer.extend(self.constant.keys())
        return buffer
    def exec_backup(self, args):
        if len(args) == 0:
            self.save_database()
        return ""
    def exec_getmousemovecount(self, args):
        if len(args) == 2:
            side = self.expand(args[0])
            part = self.expand(args[1])
            try:
                key = (int(side), part)
            except ValueError:
                pass
            else:
                return str(self.mouse_move_count.get(key, 0))
        return ""
    def exec_resetmousemovecount(self, args):
        if len(args) == 2:
            side = self.expand(args[0])
            part = self.expand(args[1])
            try:
                key = (int(side), part)
            except ValueError:
                pass
            else:
                self.mouse_move_count[key] = 0
        return ""
    def exec_substring(self, args):
        if len(args) != 3:
            return ""
        value = self.expand(args[2])
        try:
            count = int(value)
        except ValueError:
            return ""
        if count < 0:
            return ""
        value = self.expand(args[1])
        try:
            offset = int(value)
        except ValueError:
            return ""
        if offset < 0:
            return ""
        s = self.expand(args[0])
        return s[offset:offset+count]
    def exec_substringl(self, args):
        if len(args) != 2:
            return ""
        value = self.expand(args[1])
        try:
            count = int(value)
        except ValueError:
            return ""
        if count < 0:
            return ""
        s = self.expand(args[0])
        return s[:count]
    def exec_substringr(self, args):
        if len(args) != 2:
            return ""
        value = self.expand(args[1])
        try:
            count = int(value)
        except ValueError:
            return ""
        if count < 0:
            return ""
        s = self.expand(args[0])
        return s[len(s)-count:]
    def exec_substringfirst(self, args):
        if len(args) != 1:
            return ""
        buffer = self.split(self.expand(args[0]))
        if not buffer:
            return ""
        return buffer[0]
    def exec_substringlast(self, args):
        if len(args) != 1:
            return ""
        buffer = self.split(self.expand(args[0]))
        if not buffer:
            return ""
        return buffer[-1]
    def exec_length(self, args):
        if len(args) != 1:
            return ""
        return str(len(self.expand(args[0])))
    katakana2hiragana = {
        "": "", "": "", "": "", "": "", "": "",
        "": "", "": "", "": "", "": "", "": "",
        "": "", "": "", "": "", "": "", "": "",
        "": "", "": "", "": "", "": "", "": "",
        "": "", "": "", "": "", "": "", "": "",
        "": "", "": "", "": "", "": "", "": "",
        "": "", "": "", "": "", "": "", "": "",
        "": "", "": "", "": "", "": "", "": "",
        "": "", "": "", "": "", "": "", "": "",
        "": "", "": "", "": "", "": "", "": "",
        "": "", "": "", "": "", "": "", "": "",
        "": "", "": "", "": "", "": "", "": "",
        "": "", "": "", "": "", "": "", "": "",
        "": "", "": "", "": "", "": "", "": "",
        "": "", "": "", "": "", "": "", "": "",
        "": "", "": "", "": "", "": "", "": "",
        "": "", "": "", "": "", "": "",
        }
    def exec_hiraganacase(self, args):
        if len(args) != 1:
            return ""
        buffer = []
        for c in self.split(self.expand(args[0])):
            buffer.append(self.katakana2hiragana.get(c, c))
        return string.join(buffer, "")
    def exec_isequallastandfirst(self, args):
        if len(args) != 1:
            return ""
        buffer = self.split(self.expand(args[0]))
        if len(buffer) > 0 and buffer[0] == buffer[-1]:
            return "true"
        return "false"
    def exec_append(self, args):
        if len(args) == 2:
            name = self.expand(args[0])
            if name and name[0] == "$":
                type, value = self.variable.get(name, (TYPE_SCHOLAR, ""))
                if type == TYPE_SCHOLAR:
                    type = TYPE_ARRAY
                    array = Array(self, name)
                    if value != "":
                        array.append(value)
                    value = array
                list = string.split(self.expand(args[1]), chr(1))
                for member in list:
                    value.append(member)
                self.variable[name] = (type, value)
        return ""
    def exec_stringexists(self, args):
        if len(args) != 2:
            return ""
        name = self.expand(args[0])
        type, value = self.variable.get(name, (TYPE_SCHOLAR, ""))
        if type == TYPE_SCHOLAR:
            return ""
        if value.index(self.expand(args[1])) < 0:
            return "false"
        else:
            return "true"
    def exec_copy(self, args):
        if len(args) != 2:
            return ""
        src_name = self.expand(args[0])
        if not (src_name and src_name[0] == "$"):
            return ""
        new_name = self.expand(args[1])
        if not (new_name and new_name[0] == "$"):
            return ""
        source = None
        if self.dict.has_key(src_name):
            grouplist = self.dict[src_name]
            source, conditions = grouplist[0]
        elif self.variable.has_key(src_name):
            type, value = self.variable.get(src_name)
            if type == TYPE_ARRAY:
                source = value
        if source is None:
            value = Array(self, new_name)
        else:
            value = source.copy(new_name)
        self.variable[new_name] = (TYPE_ARRAY, value)
        return ""
    def exec_pop(self, args):
        if len(args) == 1:
            name = self.expand(args[0])
            type, value = self.variable.get(name, (TYPE_SCHOLAR, ""))
            if type == TYPE_ARRAY:
                return self.expand(value.pop())
        return ""
    def exec_popmatchl(self, args):
        if len(args) == 2:
            name = self.expand(args[0])
            type, value = self.variable.get(name, (TYPE_SCHOLAR, ""))
            if type == TYPE_ARRAY:
                return self.expand(value.popmatchl(self.expand(args[1])))
        return ""
    def exec_index(self, args):
        if len(args) == 2:
            pos = string.find(self.expand(args[1]), self.expand(args[0]))
            return str(pos)
        return ""
    def exec_insentence(self, args):
        if len(args) == 0:
            return ""
        s = self.expand(args[0])
        for i in range(1, len(args)):
            if string.find(s, self.expand(args[i])) < 0:
                return "false"
        return "true"
    def exec_substringw(self, args):
        if len(args) != 3:
            return ""
        value = self.expand(args[2])
        try:
            count = int(value)
        except ValueError:
            return ""
        if count < 0:
            return ""
        value = self.expand(args[1])
        try:
            offset = int(value)
        except ValueError:
            return ""
        if offset < 0:
            return ""
        buffer = self.split(self.expand(args[0]))
        return string.join(buffer[offset:offset+count], "")
    def exec_substringwl(self, args):
        if len(args) != 2:
            return ""
        value = self.expand(args[1])
        try:
            count = int(value)
        except ValueError:
            return ""
        if count < 0:
            return ""
        buffer = self.split(self.expand(args[0]))
        return string.join(buffer[:count], "")
    def exec_substringwr(self, args):
        if len(args) != 2:
            return ""
        value = self.expand(args[1])
        try:
            count = int(value)
        except ValueError:
            return ""
        if count < 0:
            return ""
        buffer = self.split(self.expand(args[0]))
        return string.join(buffer[len(buffer)-count:], "")
    prefixes = ["", "", "", "", "", "", "", ""]
    def exec_adjustprefix(self, args):
        if len(args) != 2:
            return ""
        s = self.expand(args[0])
        for prefix in self.prefixes:
            n = len(prefix)
            if s[-n:] == prefix:
                s = s[:-n] + self.expand(args[1])
                break
        return s
    def exec_count(self, args):
        if len(args) != 1:
            return ""
        name = self.expand(args[0])
        try:
            type, value = self.variable[name]
            if type == TYPE_SCHOLAR:
                return "1"
            return str(len(value))
        except KeyError:
            return "-1"
    def exec_inlastsentence(self, args):
        if len(args) == 0:
            return ""
        s = self.communicate[1]
        for i in range(1, len(args)):
            if string.find(s, self.expand(args[i])) < 0:
                return "false"
        return "true"
    def saori(self, args):
        saori_statuscode = ''
        saori_header = []
        saori_value = {}
        saori_protocol = ''
        req = 'EXECUTE SAORI/1.0\r\n' \
              'Sender: MISAKA\r\n' \
              'SecurityLevel: local\r\n' \
              'Charset: Shift_JIS\r\n'
        for i in range(1, len(args)):
              req = req + 'Argument%s: %s\r\n' % (i, self.expand(args[i]))
        req = req + '\r\n'
        response = self.saori_library.request(self.expand(args[0]), kanjilib.euc2sjis(req, 'ignore'))
        header = StringIO.StringIO(response)
        line = header.readline()
        if line:
            if line[-1] == '\n':
                line = line[:-1]
            line = string.strip(line)
            pos_space = string.find(line, ' ')
            if pos_space >= 0:
                saori_protocol = string.strip(line[:pos_space])
                saori_statuscode = string.strip(line[pos_space:])
            while 1:
                line = header.readline()
                if not line:
                    break # EOF
                if line[-1] == '\n':
                    line = line[:-1]
                line = string.strip(line)
                if not line:
                    continue
                colon = string.find(line, ':')
                if colon >= 0:
                    key = string.strip(line[:colon])
                    value = string.strip(line[colon+1:])
                    if key:
                        saori_header.append(key)
                        saori_value[key] = value
        if saori_value.has_key('Result'):
            return saori_value['Result']
        else:
            return ''
    def load_saori(self, args):
        if len(args) == 0:
            return ''
        else:
            result = self.saori_library.load(self.expand(args[0]), self.misaka_dir)
            if not result:
                result = ''
            return str(result)
    def unload_saori(self, args):
        if len(args) == 0:
            return ''
        else:
            result = self.saori_library.unload(self.expand(args[0]))
            if not result:
                result = ''
            return str(result)
    def exec_isghostexists(self, args):
        result = "false"
        if len(args) != 1:
            if len(self.otherghost):
                result = "true"
                self.variable["$to"] = (TYPE_SCHOLAR, whrandom.choice(self.otherghost)[0])
        else:
            for ghost in self.otherghost:
                if ghost[0] == self.expand(args[0]):
                    result = "true"
                    break
        return result
    SYSTEM_FUNCTIONS = {
        "$reference":           exec_reference,
        "$random":              exec_random,
        "$choice":              exec_choice,
        "$getvalue":            exec_getvalue,
        "$inlastsentence":      exec_inlastsentence,
        "$isghostexists":       exec_isghostexists,
        "$search":              exec_search,
        "$backup":              exec_backup,
        "$getmousemovecount":   exec_getmousemovecount,
        "$resetmousemovecount": exec_resetmousemovecount,
        "$substring":           exec_substring,
        "$substringl":          exec_substringl,
        "$substringr":          exec_substringr,
        "$substringfirst":      exec_substringfirst,
        "$substringlast":       exec_substringlast,
        "$length":              exec_length,
        "$hiraganacase":        exec_hiraganacase,
        "$isequallastandfirst": exec_isequallastandfirst,
        "$append":              exec_append,
        "$stringexists":        exec_stringexists,
        "$copy":                exec_copy,
        "$pop":                 exec_pop,
        "$popmatchl":           exec_popmatchl,
        "$index":               exec_index,
        "$insentece":           exec_insentence,
        "$substringw":          exec_substringw,
        "$substringwl":         exec_substringwl,
        "$substringwr":         exec_substringwr,
        "$adjustprefix":        exec_adjustprefix,
        "$count":               exec_count,
        "$saori":               saori,
        "$loadsaori":           load_saori,
        "$unloadsaori":         unload_saori,
        }

class SaoriLibrary:
    def __init__(self, saori, dir):
        self.saori_list = {}
        self.saori = saori
    def load(self, name, dir):
        result = 0
        if not self.saori_list.has_key(name):
            module = self.saori.request(name)
            if module:
                self.saori_list[name] = module
        if self.saori_list.has_key(name):
            result = self.saori_list[name].load(dir)
        return result
    def unload(self, name=None):
        if name:
            if self.saori_list.has_key(name):
                self.saori_list[name].unload()
                del self.saori_list[name]
        else:
            for key in self.saori_list.keys():
                self.saori_list[key].unload()
        return None
    def request(self, name, req):
        result = '' # FIXME
        if name and self.saori_list.has_key(name):
            result = self.saori_list[name].request(req)
        return result

def open(dir, debug=0):
    misaka = Shiori(dir, debug)
    misaka.load()
    return misaka

###   TEST   ###

def test_ini(dirlist):
    for dir in dirlist:
        print "Reading", os.path.join(dir, "misaka.ini"), "..."
        filelist, debug, error = read_misaka_ini(dir, 4)
        print "number of dictionaries =", len(filelist)
        for filename in filelist:
            print filename
        print "debug =", debug
        print "error =", error

def test_lexer(filelist):
    lexer = Lexer(debug=1)
    for filename in filelist:
        print "Reading", filename, "..."
        lexer.read(builtin_open(filename))
    token_names = {
        TOKEN_WHITESPACE:    "TOKEN_WHITESPACE",
        TOKEN_NEWLINE:       "TOKEN_NEWLINE",
        TOKEN_OPEN_BRACE:    "TOKEN_OPEN_BRACE",
        TOKEN_CLOSE_BRACE:   "TOKEN_CLOSE_BRACE",
        TOKEN_OPEN_PAREN:    "TOKEN_OPEN_PAREN",
        TOKEN_CLOSE_PAREN:   "TOKEN_CLOSE_PAREN",
        TOKEN_OPEN_BRACKET:  "TOKEN_OPEN_BRACKET",
        TOKEN_CLOSE_BRACKET: "TOKEN_CLOSE_BRACKET",
        TOKEN_DOLLAR:        "TOKEN_DOLLAR",
        TOKEN_QUOTE:         "TOKEN_QUOTE",
        TOKEN_COMMA:         "TOKEN_COMMA",
        TOKEN_SEMICOLON:     "TOKEN_SEMICOLON",
        TOKEN_OPERATOR:      "TOKEN_OPERATOR",
        TOKEN_DIRECTIVE:     "TOKEN_DIRECTIVE",
        TOKEN_TEXT:          "TOKEN_TEXT",
        }
    for token, lexeme, position in lexer.buffer:
        print "L%d : C%-3d :" % position, token_names[token], ":",
        if token in [TOKEN_WHITESPACE, TOKEN_NEWLINE,
                     TOKEN_OPEN_BRACE, TOKEN_CLOSE_BRACE,
                     TOKEN_OPEN_PAREN, TOKEN_CLOSE_PAREN,
                     TOKEN_OPEN_BRACKET, TOKEN_CLOSE_BRACKET,
                     TOKEN_DOLLAR, TOKEN_QUOTE,
                     TOKEN_COMMA, TOKEN_SEMICOLON]:
            print repr(lexeme)
        else:
            print lexeme

def test_parser(filelist):
    for filename in filelist:
        print "Reading", filename, "..."
        parser = Parser(debug=1)
        parser.read(builtin_open(filename))
        common, dict = parser.get_dict()
        if common is not None:
            print "Common"
            parser.dump_node(common, depth=2)
        for name, parameters, sentences in dict:
            print "Group", name
            if parameters:
                print "Parameter:"
                for parameter in parameters:
                    print ">>>", parameter
                    parser.dump_node(parameter, depth=2)
            if sentences:
                print "Sentence:"
                for sentence in sentences:
                    print ">>>", sentence
                    parser.dump_list(sentence, depth=2)

def test_interpreter(dir):
    import readline
    misaka = Shiori(dir, debug=1)
    misaka.load()
    while 1:
        try:
            line = raw_input(">>> ")
        except KeyboardInterrupt:
            print "Break"
            continue
        except EOFError:
            print
            break
        command = string.split(line)
        if len(command) == 0:
            continue
        elif len(command) == 1 and command[0][0] == "$":
            print misaka.eval_variable([[NODE_TEXT, command[0]]])
        elif len(command) == 1 and command[0] == "reload":
            misaka.load()
        else:
            print "list of commands:"
            print "  $id     get variable value"
            print "  reload  reload dictionaries"

if __name__ == "__main__":
    USAGE= "Usage: misaka.py [ini|lexer|parse|interp] ..."
    if len(sys.argv) == 1:
        print USAGE
    elif sys.argv[1] == "ini":
        test_ini(sys.argv[2:])
    elif sys.argv[1] == "lexer":
        test_lexer(sys.argv[2:])
    elif sys.argv[1] == "parser":
        test_parser(sys.argv[2:])
    elif sys.argv[1] == "interp":
        test_interpreter(sys.argv[2])
    else:
        print USAGE
