/*
 * kernel/power/compression.c
 *
 * Copyright (C) 2003-2006 Nigel Cunningham <nigel@suspend2.net>
 *
 * This file is released under the GPLv2.
 *
 * This file contains data compression routines for suspend,
 * using cryptoapi.
 *
 */

#include <linux/suspend.h>
#include <linux/module.h>
#include <linux/highmem.h>
#include <linux/vmalloc.h>
#include <linux/crypto.h>

#include "suspend.h"
#include "modules.h"
#include "sysfs.h"
#include "io.h"

#define S2C_WRITE 0
#define S2C_READ 1

static int suspend_expected_compression = 0;

static struct suspend_module_ops suspend_compression_ops;
static struct suspend_module_ops *next_driver;

static char suspend_compressor_name[32] = "lzf";
static struct crypto_tfm *suspend_compressor_transform;

static u8 *local_buffer = NULL;
static u8 *page_buffer = NULL;
static unsigned int bufofs;

static int position = 0;
       
/* ---- Local buffer management ---- */

/* 
 * suspend_compress_allocate_local_buffer
 *
 * Allocates a page of memory for buffering output.
 * Int: Zero if successful, -ENONEM otherwise.
 */
static int suspend_compress_allocate_local_buffer(void)
{
	if (!local_buffer) {
		local_buffer = (char *) get_zeroed_page(GFP_ATOMIC);
	
		if (!local_buffer) {
			printk(KERN_ERR
				"Failed to allocate the local buffer for "
				"suspend2 compression driver.\n");
			return -ENOMEM;
		}
	}

	if (!page_buffer) {
		page_buffer = (char *) get_zeroed_page(GFP_ATOMIC);
	
		if (!page_buffer) {
			printk(KERN_ERR
				"Failed to allocate the page buffer for "
				"suspend2 compression driver.\n");
			return -ENOMEM;
		}
	}

	return 0;
}

/* 
 * suspend_compress_free_local_buffer
 *
 * Frees memory allocated for buffering output.
 */
static inline void suspend_compress_free_local_buffer(void)
{
	if (local_buffer)
		free_page((unsigned long) local_buffer);

	local_buffer = NULL;

	if (page_buffer)
		free_page((unsigned long) page_buffer);

	page_buffer = NULL;
}

/* 
 * suspend_compress_cleanup
 *
 * Frees memory allocated for our labours.
 */
static void suspend_compress_cleanup(void)
{
	if (suspend_compressor_transform) {
		crypto_free_tfm(suspend_compressor_transform);
		suspend_compressor_transform = NULL;
	}
}

/* 
 * suspend_crypto_prepare
 *
 * Prepare to do some work by allocating buffers and transforms.
 * Returns: Int: Zero. Even if we can't set up compression, we still
 * seek to suspend.
 */
static int suspend_compress_crypto_prepare(void)
{
	if (!*suspend_compressor_name) {
		printk("Suspend2: Compression enabled but no compressor name set.\n");
		suspend_compression_ops.enabled = 0;
		return 0;
	}

	if (!(suspend_compressor_transform = crypto_alloc_tfm(suspend_compressor_name, 0))) {
		printk("Suspend2: Failed to initialise the %s compression transform.\n",
				suspend_compressor_name);
		suspend_compression_ops.enabled = 0;
		return 0;
	}

	return 0;
}

/* 
 * suspend_compress_write_cleanup(): Write unflushed data and free workspace.
 * 
 * Returns: Result of writing last page.
 */
static int suspend_compress_rw_cleanup(int rw)
{
	int ret = 0;
	
	if (rw == WRITE && suspend_compressor_transform)
		ret = next_driver->write_chunk(virt_to_page(local_buffer));

	suspend_compress_cleanup();
	suspend_compress_free_local_buffer();

	return ret;
}

/* 
 * suspend_compress_rw_init()
 * @stream_number:	Ignored.
 *
 * Allocate buffers and prepare to compress data.
 * Returns: Zero on success, -ENOMEM if unable to vmalloc.
 */
static int suspend_compress_rw_init(int rw, int stream_number)
{
	int result;
	
	next_driver = suspend_get_next_filter(&suspend_compression_ops);

	if (!next_driver) {
		printk("Compression Driver: Argh! Nothing follows me in"
				" the pipeline!");
		return -ECHILD;
	}

	if ((result = suspend_compress_crypto_prepare() ||
	     !suspend_compression_ops.enabled))
		return result;
	
	if ((result = suspend_compress_allocate_local_buffer()))
		return result;

	if (rw == READ)
		bufofs = PAGE_SIZE;
	else {
		/* Only reset the stats if starting to write an image */
		if (stream_number == 2)
			bytes_in = bytes_out = 0;
	
		bufofs = 0;
	}

	position = 0;

	return 0;
}

/* 
 * suspend_compress_write()
 * @u8*:		Output buffer to be written.
 * @unsigned int:	Length of buffer.
 *
 * Helper function for write_chunk. Write the compressed data.
 * Return: Int.	Result to be passed back to caller.
 */
static int suspend_compress_write (u8 *buffer, unsigned int len)
{
	int ret;

	bytes_out += len;

	while (len + bufofs > PAGE_SIZE) {
		unsigned int chunk = PAGE_SIZE - bufofs;
		memcpy (local_buffer + bufofs, buffer, chunk);
		buffer += chunk;
		len -= chunk;
		bufofs = 0;
		if ((ret = next_driver->write_chunk(virt_to_page(local_buffer))) < 0)
			return ret;
	}
	memcpy (local_buffer + bufofs, buffer, len);
	bufofs += len;
	return 0;
}

/* 
 * suspend_compress_write_chunk()
 *
 * Compress a page of data, buffering output and passing on filled
 * pages to the next module in the pipeline.
 * 
 * Buffer_page:	Pointer to a buffer of size PAGE_SIZE, containing
 * data to be compressed.
 *
 * Returns:	0 on success. Otherwise the error is that returned by later
 * 		modules, -ECHILD if we have a broken pipeline or -EIO if
 * 		zlib errs.
 */
static int suspend_compress_write_chunk(struct page *buffer_page)
{
	int ret; 
	unsigned int len;
	u16 len_written;
	char *buffer_start;
	
	if (!suspend_compressor_transform)
		return next_driver->write_chunk(buffer_page);

	buffer_start = kmap(buffer_page);

	bytes_in += PAGE_SIZE;

	len = PAGE_SIZE;

	ret = crypto_comp_compress(suspend_compressor_transform,
			buffer_start, PAGE_SIZE,
			page_buffer, &len);
	
	if (ret) {
		printk("Compression failed.\n");
		goto failure;
	}
	
	len_written = (u16) len;
		
	if ((ret = suspend_compress_write((u8 *)&len_written, 2)) >= 0) {
		if ((ret = suspend_compress_write((u8 *) &position, sizeof(position))))
			return -EIO;
		if (len < PAGE_SIZE) { /* some compression */
			position += len;
			ret = suspend_compress_write(page_buffer, len);
		} else {
			ret = suspend_compress_write(buffer_start, PAGE_SIZE);
			position += PAGE_SIZE;
		}
	}
	position += 2 + sizeof(int);


failure:
	kunmap(buffer_page);
	return ret;
}

/* 
 * suspend_compress_read()
 * @buffer: u8 *. Address of the buffer.
 * @len: unsigned int. Length.
 *
 * Description:	Read data into compression buffer.
 * Returns:	int:		Result of reading the image chunk.
 */
static int suspend_compress_read (u8 *buffer, unsigned int len)
{
	int ret;

	while (len + bufofs > PAGE_SIZE) {
		unsigned int chunk = PAGE_SIZE - bufofs;
		memcpy(buffer, local_buffer + bufofs, chunk);
		buffer += chunk;
		len -= chunk;
		bufofs = 0;
		if ((ret = next_driver->read_chunk(
				virt_to_page(local_buffer), SUSPEND_SYNC)) < 0) {
			return ret;
		}
	}
	memcpy (buffer, local_buffer + bufofs, len);
	bufofs += len;
	return 0;
}

/* 
 * suspend_compress_read_chunk()
 * @buffer_page: struct page *. Pointer to a buffer of size PAGE_SIZE.
 * @sync:	int. Whether the previous module (or core) wants its data synchronously.
 *
 * Retrieve data from later modules and decompress it until the input buffer
 * is filled.
 * Zero if successful. Error condition from me or from downstream on failure.
 */
static int suspend_compress_read_chunk(struct page *buffer_page, int sync)
{
	int ret, position_saved; 
	unsigned int len;
	u16 len_written;
	char *buffer_start;

	if (!suspend_compressor_transform)
		return next_driver->read_chunk(buffer_page, SUSPEND_ASYNC);

	/* 
	 * All our reads must be synchronous - we can't decompress
	 * data that hasn't been read yet.
	 */

	buffer_start = kmap(buffer_page);

	if ((ret = suspend_compress_read ((u8 *)&len_written, 2)) >= 0) {
		len = (unsigned int) len_written;
		ret = suspend_compress_read((u8 *) &position_saved, sizeof(position_saved));
		if (ret)
			return ret;

		if (position != position_saved) {
			printk("Position saved (%d) != position I'm at now (%d).\n",
					position_saved, position);
			BUG_ON(1);
		}
		if (len >= PAGE_SIZE) { /* uncompressed */
			ret = suspend_compress_read(buffer_start, PAGE_SIZE);
			if (ret)
				return ret;

			position += PAGE_SIZE;
		} else { /* compressed */
			if ((ret = suspend_compress_read(page_buffer, len)) >= 0) {
				int outlen = PAGE_SIZE;
				/* Important note.
				 *
				 * For Deflate, decompression return values may represent
				 * errors. Deflate complains when everything is alright, so
				 * we ignore the errors unless the number of output bytes is
				 * not PAGE_SIZE.
				 */
				crypto_comp_decompress(suspend_compressor_transform, 
						page_buffer, len,
						buffer_start, &outlen);
				if (outlen != PAGE_SIZE) {
					printk("Decompression yielded %d bytes instead of %ld.\n", outlen, PAGE_SIZE);
					ret = -EIO;
				} else
					ret = 0;
			}
			position += len;
		}
		position += 2 + sizeof(int);
	} else
		printk("Compress_read returned %d.", ret);
	kunmap(buffer_page);
	return ret;
}

/* 
 * suspend_compress_print_debug_stats
 * @buffer: Pointer to a buffer into which the debug info will be printed.
 * @size: Size of the buffer.
 *
 * Print information to be recorded for debugging purposes into a buffer.
 * Returns: Number of characters written to the buffer.
 */

static int suspend_compress_print_debug_stats(char *buffer, int size)
{
	int pages_in = bytes_in >> PAGE_SHIFT, 
		pages_out = bytes_out >> PAGE_SHIFT;
	int len;
	
	/* Output the compression ratio achieved. */
	if (*suspend_compressor_name)
		len = snprintf_used(buffer, size, "- Compressor is '%s'.\n",
				suspend_compressor_name);
	else
		len = snprintf_used(buffer, size, "- Compressor is not set.\n");

	if (pages_in)
		len+= snprintf_used(buffer+len, size - len,
		  "  Compressed %ld bytes into %ld (%d percent compression).\n",
		  bytes_in, bytes_out, (pages_in - pages_out) * 100 / pages_in);
	return len;
}

/* 
 * suspend_compress_compression_memory_needed
 *
 * Tell the caller how much memory we need to operate during suspend/resume.
 * Returns: Unsigned long. Maximum number of bytes of memory required for
 * operation.
 */
static unsigned long suspend_compress_memory_needed(void)
{
	return 2 * PAGE_SIZE;
}

static unsigned long suspend_compress_storage_needed(void)
{
	return 4 * sizeof(unsigned long) + strlen(suspend_compressor_name) + 1;
}

/* 
 * suspend_compress_save_config_info
 * @buffer: Pointer to a buffer of size PAGE_SIZE.
 *
 * Save informaton needed when reloading the image at resume time.
 * Returns: Number of bytes used for saving our data.
 */
static int suspend_compress_save_config_info(char *buffer)
{
	int namelen = strlen(suspend_compressor_name) + 1;
	int total_len;
	
	*((unsigned long *) buffer) = bytes_in;
	*((unsigned long *) (buffer + 1 * sizeof(unsigned long))) = bytes_out;
	*((unsigned long *) (buffer + 2 * sizeof(unsigned long))) =
		suspend_expected_compression;
	*((unsigned long *) (buffer + 3 * sizeof(unsigned long))) = namelen;
	strncpy(buffer + 4 * sizeof(unsigned long), suspend_compressor_name, 
								namelen);
	total_len = 4 * sizeof(unsigned long) + namelen;
	return total_len;
}

/* suspend_compress_load_config_info
 * @buffer: Pointer to the start of the data.
 * @size: Number of bytes that were saved.
 *
 * Description:	Reload information needed for decompressing the image at
 * resume time.
 */
static void suspend_compress_load_config_info(char *buffer, int size)
{
	int namelen;
	
	bytes_in = *((unsigned long *) buffer);
	bytes_out = *((unsigned long *) (buffer + 1 * sizeof(unsigned long)));
	suspend_expected_compression = *((unsigned long *) (buffer + 2 *
				sizeof(unsigned long)));
	namelen = *((unsigned long *) (buffer + 3 * sizeof(unsigned long)));
	strncpy(suspend_compressor_name, buffer + 4 * sizeof(unsigned long),
			namelen);
	return;
}

/* 
 * suspend_expected_compression_ratio
 * 
 * Description:	Returns the expected ratio between data passed into this module
 * 		and the amount of data output when writing.
 * Returns:	100 if the module is disabled. Otherwise the value set by the
 * 		user via our sysfs entry.
 */

int suspend_expected_compression_ratio(void)
{
	if (!suspend_compression_ops.enabled)
		return 100;
	else
		return 100 - suspend_expected_compression;
}

/*
 * data for our sysfs entries.
 */
static struct suspend_sysfs_data sysfs_params[] = {
	{
		SUSPEND2_ATTR("expected_compression", SYSFS_RW),
		SYSFS_INT(&suspend_expected_compression, 0, 99)
	},

	{
		SUSPEND2_ATTR("enabled", SYSFS_RW),
		SYSFS_INT(&suspend_compression_ops.enabled, 0, 1)
	},

	{
		SUSPEND2_ATTR("algorithm", SYSFS_RW),
		SYSFS_STRING(suspend_compressor_name, 31, SYSFS_SM_NOT_NEEDED)
	}
};

/*
 * Ops structure.
 */
static struct suspend_module_ops suspend_compression_ops = {
	.type			= FILTER_MODULE,
	.name			= "Suspend2 Compressor",
	.module			= THIS_MODULE,
	.memory_needed 		= suspend_compress_memory_needed,
	.print_debug_info	= suspend_compress_print_debug_stats,
	.save_config_info	= suspend_compress_save_config_info,
	.load_config_info	= suspend_compress_load_config_info,
	.storage_needed		= suspend_compress_storage_needed,
	
	.rw_init		= suspend_compress_rw_init,
	.rw_cleanup		= suspend_compress_rw_cleanup,

	.write_chunk		= suspend_compress_write_chunk,
	.read_chunk		= suspend_compress_read_chunk,
};

/* ---- Registration ---- */

static __init int suspend_compress_load(void)
{
	int result;
	int i, numfiles = sizeof(sysfs_params) / sizeof(struct suspend_sysfs_data);

	printk("Suspend2 Compression Driver loading.\n");
	if (!(result = suspend_register_module(&suspend_compression_ops))) {
		struct kobject *kobj = make_suspend2_sysdir("compression");
		for (i=0; i< numfiles; i++)
			suspend_register_sysfs_file(kobj, &sysfs_params[i]);
	} else
		printk("Suspend2 Compression Driver unable to register!\n");
	return result;
}

#ifdef MODULE
static __exit void suspend_compress_unload(void)
{
	printk("Suspend2 Compression Driver unloading.\n");
	for (i=0; i< numfiles; i++)
		suspend_unregister_sysfs_file(&sysfs_params[i]);
	suspend_unregister_module(&suspend_compression_ops);
}

module_init(suspend_compress_load);
module_exit(suspend_compress_unload);
MODULE_LICENSE("GPL");
MODULE_AUTHOR("Nigel Cunningham");
MODULE_DESCRIPTION("Compression Support for Suspend2");
#else
late_initcall(suspend_compress_load);
#endif
