/*
 * kernel/power/pagedir.c
 *
 * Copyright (C) 1998-2001 Gabor Kuti <seasons@fornax.hu>
 * Copyright (C) 1998,2001,2002 Pavel Machek <pavel@suse.cz>
 * Copyright (C) 2002-2003 Florent Chabaud <fchabaud@free.fr>
 * Copyright (C) 2002-2006 Nigel Cunningham <nigel@suspend2.net>
 *
 * This file is released under the GPLv2.
 *
 * Routines for handling pagesets.
 * Note that pbes aren't actually stored as such. They're stored as
 * bitmaps and extents.
 */

#include <linux/suspend.h>
#include <linux/highmem.h>
#include <linux/bootmem.h>
#include <linux/hardirq.h>
#include <linux/sched.h>
#include <asm/tlbflush.h>

#include "pageflags.h"
#include "ui.h"
#include "pagedir.h"

int extra_pagedir_pages_allocated;

struct extras {
	struct page *page;
	int order;
	struct extras *next;
} *extras_list;

/* suspend_free_extra_pagedir_memory
 *
 * Description:	Free previously allocated extra pagedir memory.
 */
void suspend_free_extra_pagedir_memory(void)
{
	/* Free allocated pages */
	while (extras_list) {
		struct extras *this = extras_list;
		int i;

		extras_list = this->next;

		for (i = 0; i < (1 << this->order); i++)
			ClearPageNosave(this->page + i);

		__free_pages(this->page, this->order);
		kfree(this);
	}

	extra_pagedir_pages_allocated = 0;
}

/* suspend_allocate_extra_pagedir_memory
 *
 * Description:	Allocate memory for making the atomic copy of pagedir1 in the
 * 		case where it is bigger than pagedir2.
 * Arguments:	struct pagedir *: 	The pagedir for which we should 
 * 					allocate memory.
 * 		int:			Size of pageset 1.
 * 		int:			Size of pageset 2.
 * Result:	int. Zero on success. One if unable to allocate enough memory.
 */
int suspend_allocate_extra_pagedir_memory(struct pagedir *p, int pageset_size,
		int alloc_from)
{
	int num_to_alloc = pageset_size - alloc_from - extra_pagedir_pages_allocated;
	int j, order, num_added = 0;

	if (num_to_alloc < 1)
		num_to_alloc = 0;

	if (num_to_alloc) {
		order = fls(num_to_alloc);
		if (order >= MAX_ORDER)
			order = MAX_ORDER - 1;

		while (num_added < num_to_alloc) {
			struct page *newpage;
			unsigned long virt;
			struct extras *extras_entry;
			
			while ((1 << order) > (num_to_alloc - num_added))
				order--;

			virt = __get_free_pages(GFP_ATOMIC | __GFP_NOWARN, order);
			while ((!virt) && (order > 0)) {
				order--;
				virt = __get_free_pages(GFP_ATOMIC | __GFP_NOWARN, order);
			}

			if (!virt) {
				p->pageset_size += num_added;
				extra_pagedir_pages_allocated += num_added;
				return 1;
			}

			newpage = virt_to_page(virt);

			extras_entry = (struct extras *) kmalloc(sizeof(struct extras), GFP_ATOMIC);

			if (!extras_entry) {
				__free_pages(newpage, order);
				extra_pagedir_pages_allocated += num_added;
				return 1;
			}

			extras_entry->page = newpage;
			extras_entry->order = order;
			extras_entry->next = NULL;

			if (extras_list)
				extras_entry->next = extras_list;

			extras_list = extras_entry;

			for (j = 0; j < (1 << order); j++) {
				SetPageNosave(newpage + j);
				SetPagePageset1Copy(newpage + j);
			}
			num_added+= (1 << order);
		}
	}

	extra_pagedir_pages_allocated += num_added;
	return 0;
}

int suspend_protect_pageset2(void)
{
#ifdef CONFIG_DEBUG_PAGEALLOC
	int pfn;

	BITMAP_FOR_EACH_SET(pageset2_map, pfn) {
		struct page *page = pfn_to_page(pfn);
		if (PageHighMem(page))
			continue;

		change_page_attr(page, 1, __pgprot(0));
		__flush_tlb_all();
	}
#endif
	
	return 0;
}

void suspend_restore_pageset2_permissions(void)
{
#ifdef CONFIG_DEBUG_PAGEALLOC
	int pfn;

	BITMAP_FOR_EACH_SET(pageset2_map, pfn) {
		struct page *page = pfn_to_page(pfn);
		if (PageHighMem(page))
			continue;

		change_page_attr(page, 1, PAGE_KERNEL);
		__flush_tlb_all();
	}
#endif
}

/*
 * suspend_mark_task_as_pageset1
 * Functionality   : Marks all the pages belonging to a given process as
 *                   pageset 1 pages.
 * Called From     : pagedir.c - mark_pages_for_pageset2
 *
 */
extern struct page *suspend2_follow_page(struct mm_struct *mm, unsigned long address);

static void suspend_mark_task_as_pageset1(struct task_struct *t)
{
	struct vm_area_struct *vma;
	struct mm_struct *mm;

	mm = t->active_mm;

	if (!mm || !mm->mmap) return;

	/* Don't try to take the sem when processes are frozen, 
	 * drivers are suspended and irqs are disabled. We're
	 * not racing with anything anyway.  */
	BUG_ON(in_atomic() && !irqs_disabled());

	if (!irqs_disabled())
		down_read(&mm->mmap_sem);
	
	for (vma = mm->mmap; vma; vma = vma->vm_next) {
		if (vma->vm_flags & VM_PFNMAP)
			continue;
		if (vma->vm_start) {
			unsigned long posn;
			for (posn = vma->vm_start; posn < vma->vm_end;
					posn += PAGE_SIZE) {
				struct page *page = 
					suspend2_follow_page(mm, posn);
				if (page) {
					ClearPagePageset2(page);
					SetPagePageset1(page);
				}
			}
		}
	}

	BUG_ON(in_atomic() && !irqs_disabled());

	if (!irqs_disabled())
		up_read(&mm->mmap_sem);
}

/* mark_pages_for_pageset2
 *
 * Description:	Mark unshared pages in processes not needed for suspend as
 * 		being able to be written out in a separate pagedir.
 * 		HighMem pages are simply marked as pageset2. They won't be
 * 		needed during suspend.
 */

struct attention_list {
	struct task_struct *task;
	struct attention_list *next;
};

void suspend_mark_pages_for_pageset2(void)
{
	struct zone *zone;
	struct task_struct *p;
	struct attention_list *attention_list = NULL, *last = NULL;
	unsigned long flags;

	BUG_ON(in_atomic() && !irqs_disabled());

	if (test_action_state(SUSPEND_NO_PAGESET2))
		return;

	clear_dyn_pageflags(pageset2_map);
	
	for_each_zone(zone) {
		spin_lock_irqsave(&zone->lru_lock, flags);
		if (zone->nr_inactive) {
			struct page *page;
			list_for_each_entry(page, &zone->inactive_list, lru)
				SetPagePageset2(page);
		}
		if (zone->nr_active) {
			struct page *page;
			list_for_each_entry(page, &zone->active_list, lru)
				SetPagePageset2(page);
		}
		spin_unlock_irqrestore(&zone->lru_lock, flags);
	}

	BUG_ON(in_atomic() && !irqs_disabled());

	/* Now we find all userspace process (with task->mm) marked PF_NOFREEZE
	 * and move them into pageset1.
	 */
	read_lock(&tasklist_lock);
	for_each_process(p)
		if ((p->flags & PF_NOFREEZE) || p == current) {
			struct attention_list *this = kmalloc(sizeof(struct attention_list), GFP_ATOMIC);
			BUG_ON(!this);
			this->task = p;
			this->next = NULL;
			if (attention_list) {
				last->next = this;
				last = this;
			} else
				attention_list = last = this;
		}
	read_unlock(&tasklist_lock);

	BUG_ON(in_atomic() && !irqs_disabled());

	/* Because the tasks in attention_list are ones related to suspending,
	 * we know that they won't go away under us.
	 */

	while (attention_list) {
		suspend_mark_task_as_pageset1(attention_list->task);
		last = attention_list;
		attention_list = attention_list->next;
		kfree(last);
	}

	BUG_ON(in_atomic() && !irqs_disabled());

}

/* suspend_get_nonconflicting_page
 *
 * Description: Gets order zero pages that won't be overwritten
 *		while copying the original pages.
 */

unsigned long suspend_get_nonconflicting_page(void)
{
	struct page *page;

	do {
		page = alloc_pages(GFP_ATOMIC | __GFP_NOWARN | __GFP_ZERO, 0);
		BUG_ON(!page);
	} while(PagePageset1(page));

	return (unsigned long) page_address(page);
}

/* relocate_page_if_required
 *
 * Description: Given the address of a pointer to a page, we check if the page
 * 		needs relocating and do so if needs be, adjusting the pointer
 * 		too.
 */

void suspend_relocate_if_required(unsigned long *current_value, unsigned int size)
{
	if (PagePageset1(virt_to_page(*current_value))) {
		unsigned long new_page = suspend_get_nonconflicting_page();
		memcpy((char *) new_page, (char *) *current_value, size);
		if (PageSlab(virt_to_page(*current_value)))
			kfree((void *) *current_value);
		else
			free_page((unsigned long) *current_value);
		*current_value = new_page;
	}
}

/* get_pageset1_load_addresses
 * 
 * Description: We check here that pagedir & pages it points to won't collide
 * 		with pages where we're going to restore from the loaded pages
 * 		later.
 * Returns:	Zero on success, one if couldn't find enough pages (shouldn't
 * 		happen).
 */

int suspend_get_pageset1_load_addresses(void)
{
	int i, result = 0;
	void *this;

	for(i=0; i < pagedir1.pageset_size; i++) {
		this = (void *) suspend_get_nonconflicting_page();
		if (!this) {
			abort_suspend("Error: Ran out of memory seeking locations for reloading data.");
			result = 1;
			break;
		}
		SetPagePageset1Copy(virt_to_page(this));
	}

	return result;
}
