/*
 * kernel/power/pageflags.h
 *
 * Copyright (C) 2004-2006 Nigel Cunningham <nigel@suspend2.net>
 *
 * This file is released under the GPLv2.
 *
 * Suspend2 needs a few pageflags while working that aren't otherwise
 * used. To save the struct page pageflags, we dynamically allocate
 * a bitmap and use that. These are the only non order-0 allocations
 * we do.
 *
 * NOTE!!!
 * We assume that PAGE_SIZE - sizeof(void *) is a multiple of
 * sizeof(unsigned long). Is this ever false?
 */

#include <linux/dyn_pageflags.h>
#include <linux/suspend.h>

extern dyn_pageflags_t in_use_map;
extern dyn_pageflags_t pageset2_map;
extern dyn_pageflags_t pageset2_rw_map;

/* 
 * inusemap is used in two ways: 
 * - During suspend, to tag pages which are not used (to speed up 
 *   count_data_pages);
 * - During resume, to tag pages which are in pagedir1. This does not tag 
 *   pagedir2 pages, so !== first use.
 */

#define PageInUse(page) (test_dynpageflag(&in_use_map, page))
#define SetPageInUse(page) (set_dynpageflag(&in_use_map, page))
#define ClearPageInUse(page) (clear_dynpageflag(&in_use_map, page))

#define PagePageset1(page) (test_dynpageflag(&pageset1_map, page))
#define SetPagePageset1(page) (set_dynpageflag(&pageset1_map, page))
#define ClearPagePageset1(page) (clear_dynpageflag(&pageset1_map, page))

#define PagePageset1Copy(page) (test_dynpageflag(&pageset1_copy_map, page))
#define SetPagePageset1Copy(page) (set_dynpageflag(&pageset1_copy_map, page))
#define ClearPagePageset1Copy(page) (clear_dynpageflag(&pageset1_copy_map, page))

#define PagePageset2(page) (test_dynpageflag(&pageset2_map, page))
#define SetPagePageset2(page) (set_dynpageflag(&pageset2_map, page))
#define ClearPagePageset2(page) (clear_dynpageflag(&pageset2_map, page))

#define PageWasRW(page) (test_dynpageflag(&pageset2_map, page))
#define SetPageWasRW(page) (set_dynpageflag(&pageset2_map, page))
#define ClearPageWasRW(page) (clear_dynpageflag(&pageset2_map, page))

extern void save_dyn_pageflags(dyn_pageflags_t pagemap);
extern void load_dyn_pageflags(dyn_pageflags_t pagemap);
extern void relocate_dyn_pageflags(dyn_pageflags_t *pagemap);
extern int suspend_pageflags_space_needed(void);
