/*
 * kernel/power/power_off.c
 *
 * Copyright (C) 2006 Nigel Cunningham <nigel@suspend2.net>
 *
 * This file is released under the GPLv2.
 *
 * Support for powering down.
 */

#include <linux/device.h>
#include <linux/suspend.h>
#include <linux/mm.h>
#include <linux/pm.h>
#include <linux/reboot.h>
#include "suspend.h"
#include "ui.h"

unsigned long suspend_powerdown_method = 0; /* 0 - Kernel power off */

extern struct pm_ops *pm_ops;

/* Use suspend_enter from main.c */
extern int suspend_enter(suspend_state_t state);

int try_pm_state_powerdown(void)
{
	if (pm_ops && pm_ops->prepare && suspend_powerdown_method &&
	    pm_ops->prepare(suspend_powerdown_method))
			return 0;

	if (suspend_powerdown_method > 3)
		kernel_shutdown_prepare(SYSTEM_SUSPEND_DISK);
	else {
		if (device_suspend(PMSG_SUSPEND)) {
			printk(KERN_ERR "Some devices failed to suspend\n");
			return 0;
		}
	}

	mdelay(1000); /* Give time for devices to power down */
	
	if (suspend_enter(suspend_powerdown_method))
		return 0;

	device_resume();

	if (pm_ops && pm_ops->finish && suspend_powerdown_method)
		pm_ops->finish(suspend_powerdown_method);

	return 1;
}

/*
 * suspend_power_down
 * Functionality   : Powers down or reboots the computer once the image
 *                   has been written to disk.
 * Key Assumptions : Able to reboot/power down via code called or that
 *                   the warning emitted if the calls fail will be visible
 *                   to the user (ie printk resumes devices).
 * Called From     : do_suspend2_suspend_2
 */

void suspend_power_down(void)
{
	if (test_action_state(SUSPEND_REBOOT)) {
		suspend_prepare_status(DONT_CLEAR_BAR, "Ready to reboot.");
		kernel_restart(NULL);
	}

	suspend_prepare_status(DONT_CLEAR_BAR, "Powering down.");

	if (pm_ops && pm_ops->enter && suspend_powerdown_method && try_pm_state_powerdown())
		return;

	kernel_shutdown_prepare(SYSTEM_POWER_OFF);

	mdelay(1000); /* Give time for devices to power down */

	machine_power_off();
	machine_halt();
	suspend_prepare_status(DONT_CLEAR_BAR, "Powerdown failed");
	while (1)
		cpu_relax();
}

