// Copyright (C) 1999-2004
// Smithsonian Astrophysical Observatory, Cambridge, MA, USA
// For conditions of distribution and use, see copyright notice in "copyright"

#include <math.h>
#include <iostream.h>

#include "contourscale.h"
#include "framebase.h"

ContourScale::ContourScale(int s) : size_(s)
{
  level_ = new float[size_];

  for (int i=0; i<size_; i++)
    level_[i] = 0;
}

ContourScale::ContourScale(int s, float* l) : size_(s)
{
  level_ = new float[size_];

  for (int i=0; i<size_; i++)
    level_[i] = l[i];
}

ContourScale::~ContourScale()
{
  if (level_)
    delete [] level_;
}

ostream& operator<<(ostream& s, const ContourScale& c)
{
  for (int i=0; i<c.size_; i++)
    s << c.level_[i] << endl;
  return s;
}

// Note: this an inverse algorithm. We have y, we need x

LinearContourScale::LinearContourScale(int s, float low, float high) 
  : ContourScale(s)
{
  if (size_==1) {
    level_[0] = high;
    return;
  }

  float diff = high-low;
  for (int i=0; i<size_; i++)
    level_[i] = (float)i/(size_-1)*diff + low;
}

LogContourScale::LogContourScale(int s, float low, float high) 
  : ContourScale(s)
{
  if (size_==1) {
    level_[0] = high;
    return;
  }

  double diff = log10(high)-log10(low);
  for (int i=0; i<size_; i++)
    level_[i] = pow(10,(float)i/(size_-1)*diff + log10(low));
}

SquaredContourScale::SquaredContourScale(int s, float low, float high)
  : ContourScale(s)
{
  if (size_==1) {
    level_[0] = high;
    return;
  }

  float diff = high-low;
  for (int i=0; i<size_; i++)
    level_[i] = sqrt((float)i/(size_-1))*diff+low;
}

SqrtContourScale::SqrtContourScale(int s, float low, float high) 
  : ContourScale(s)
{
  if (size_==1) {
    level_[0] = high;
    return;
  }

  float diff = high-low;
  for (int i=0; i<size_; i++) {
    float y = (float)i/(size_-1);
    level_[i] = (y*y)*diff+low;
  }
}

HistEquContourScale::HistEquContourScale(int s, float low, float high, 
					 double* hist, int histsize) 
  : ContourScale(s)
{
  if (size_==1) {
    level_[0] = high;
    return;
  }

  if (!hist) {
    float diff = high-low;
    for (int i=0; i<size_; i++)
      level_[i] = (float)i/(size_-1)*diff + low;
  }
  else {
    float diff = high-low;
    for (int i=0; i<size_; i++) {
      float v = float(i)/(size_-1);
      int j=0;
      while (j<histsize-1) {
	if (hist[j]>v)
	  break;
	j++;
      }
      level_[i] = float(j)/histsize*diff + low;
    }
  }
}

