// Copyright (C) 1999-2004
// Smithsonian Astrophysical Observatory, Cambridge, MA, USA
// For conditions of distribution and use, see copyright notice in "copyright"

#include "framergbtrue24.h"
#include "fitsimage.h"
#include "util.h"
#include "nan.h"

// Tk Canvas Widget Function Declarations

int FrameRGBTrueColor24CreateProc(Tcl_Interp*, Tk_Canvas, Tk_Item*, int, 
				  Tcl_Obj *const []);

// FrameRGBTrueColor24 Specs

static Tk_CustomOption tagsOption = {
  Tk_CanvasTagsParseProc, Tk_CanvasTagsPrintProc, NULL
};

static Tk_ConfigSpec frameRGBTrueColor24Specs[] = {

  {TK_CONFIG_STRING, "-command", NULL, NULL, "framergb",
   Tk_Offset(WidgetOptions, cmdName), TK_CONFIG_OPTION_SPECIFIED, NULL},
  {TK_CONFIG_INT, "-x", NULL, NULL, "1",
   Tk_Offset(WidgetOptions, x), TK_CONFIG_OPTION_SPECIFIED, NULL},
  {TK_CONFIG_INT, "-y", NULL, NULL, "1",
   Tk_Offset(WidgetOptions, y), TK_CONFIG_OPTION_SPECIFIED, NULL},
  {TK_CONFIG_INT, "-width", NULL, NULL, "512",
   Tk_Offset(WidgetOptions, width), TK_CONFIG_OPTION_SPECIFIED, NULL},
  {TK_CONFIG_INT, "-height", NULL, NULL, "512",
   Tk_Offset(WidgetOptions, height), TK_CONFIG_OPTION_SPECIFIED, NULL},
  {TK_CONFIG_ANCHOR, "-anchor", NULL, NULL, "nw",
   Tk_Offset(WidgetOptions, anchor), 0, NULL},
  {TK_CONFIG_CUSTOM, "-tags", NULL, NULL, NULL,
   0, TK_CONFIG_NULL_OK, &tagsOption},

  {TK_CONFIG_END, NULL, NULL, NULL, NULL, 0, 0, NULL},
};

// Tk Static Structure

static Tk_ItemType frameRGBTrueColor24Type = {
  "framergbtruecolor24",           // name
  sizeof(WidgetOptions),        // item size
  FrameRGBTrueColor24CreateProc,   // configProc
  frameRGBTrueColor24Specs,        // configSpecs
  WidgetConfigProc,             // configProc
  WidgetCoordProc,              // coordProc
  WidgetDeleteProc,             // deleteProc
  WidgetDisplayProc,            // displayProc
  0,                            // alwaysRedraw
  WidgetPointProc,              // pointProc
  WidgetAreaProc,               // areaProc
  WidgetPostscriptProc,         // postscriptProc
  WidgetScaleProc,              // scaleProc
  WidgetTranslateProc,          // translateProc
  (Tk_ItemIndexProc*)NULL,      // indexProc
  WidgetICursorProc,            // icursorProc
  (Tk_ItemSelectionProc*)NULL,  // selectionProc
  (Tk_ItemInsertProc*)NULL,     // insertProc
  (Tk_ItemDCharsProc*)NULL,     // dCharsProc
  (Tk_ItemType*)NULL            // nextPtr
};

// Non-Member Functions

int FrameRGBTrueColor24_Init(Tcl_Interp* interp)
{
  Tk_CreateItemType(&frameRGBTrueColor24Type);
  return TCL_OK;
}

int FrameRGBTrueColor24CreateProc(Tcl_Interp* interp, Tk_Canvas canvas, 
				  Tk_Item* item, int argc, 
				  Tcl_Obj *const argv[])
{
  FrameRGBTrueColor24* frame = new FrameRGBTrueColor24(interp, canvas, item);

  // and set default configuration

  if (frame->configure(argc, (const char**)argv, 0) != TCL_OK) {
    delete frame;
    Tcl_AppendResult(interp, " error occured while creating frame.", NULL);
    return TCL_ERROR;
  }

  return TCL_OK;
}

// FrameRGBTrueColor24 Member Functions

FrameRGBTrueColor24::FrameRGBTrueColor24(Tcl_Interp* i, Tk_Canvas c, 
					 Tk_Item* item)
  : FrameRGBTrueColor(i, c, item)
{
  configSpecs = frameRGBTrueColor24Specs;  // frame configure options
}

FrameRGBTrueColor24::~FrameRGBTrueColor24()
{
  // we must do this at this level, because updateColorScale is called
  unloadAllFits();
}

void FrameRGBTrueColor24::encodeTrueColor(XColor* src, char* dest)
{
  if (!baseXImage)
    return;

  switch (baseXImage->bits_per_pixel) {
  case 24:
    encodeTrueColor24(src,dest);
    break;
  case 32:
    encodeTrueColor32(src,dest);
    break;
  }
}

void FrameRGBTrueColor24::encodeTrueColor24(XColor* src, char* dest)
{
  int msb = baseXImage->byte_order;
#ifndef _WIN32
  int rs = decodeMask((unsigned long)visual->red_mask);
  int gs = decodeMask((unsigned long)visual->green_mask);
  int bs = decodeMask((unsigned long)visual->blue_mask);
#else
  int rs = decodeMask((unsigned long)0x00FF0000);
  int gs = decodeMask((unsigned long)0x0000FF00);
  int bs = decodeMask((unsigned long)0x000000FF);
#endif
      
  // we need to check to byteswap when we have cross platforms

  unsigned int r = (unsigned char)src->red;
  unsigned int g = (unsigned char)src->green;
  unsigned int b = (unsigned char)src->blue;
  unsigned int a = 0;
  a |= r << rs;
  a |= g << gs;
  a |= b << bs;

  if ((!msb && lsb()) || (msb && !lsb()))
    memcpy(dest, &a, 3);
  else {
    unsigned char* rr = (unsigned char*)(&a);
    *(dest) = *(rr+3);
    *(dest+1) = *(rr+2);
    *(dest+2) = *(rr+1);
  }
}

void FrameRGBTrueColor24::encodeTrueColor32(XColor* src, char* dest)
{
  int msb = baseXImage->byte_order;
#ifndef _WIN32
  int rs = decodeMask((unsigned long)visual->red_mask);
  int gs = decodeMask((unsigned long)visual->green_mask);
  int bs = decodeMask((unsigned long)visual->blue_mask);
#else
  int rs = decodeMask((unsigned long)0x00FF0000);
  int gs = decodeMask((unsigned long)0x0000FF00);
  int bs = decodeMask((unsigned long)0x000000FF);
#endif
      
  unsigned int r = (unsigned char)src->red;
  unsigned int g = (unsigned char)src->green;
  unsigned int b = (unsigned char)src->blue;
  unsigned int a = 0;
  a |= r << rs;
  a |= g << gs;
  a |= b << bs;

  if ((!msb && lsb()) || (msb && !lsb()))
    memcpy(dest, &a, 4);
  else {
    unsigned char* rr = (unsigned char*)(&a);
    *(dest) = *(rr+3);
    *(dest+1) = *(rr+2);
    *(dest+2) = *(rr+1);
    *(dest+3) = *(rr);
  }
}

void FrameRGBTrueColor24::buildXImage(XImage* xmap,
				      int x0, int y0, int x1, int y1,
				      double* (FitsImage::*getMatrix)())
{
  // we need a colorScale before we can render
  if (!validColorScale())
    return;

  // create img
  unsigned char* img = fillRGBImage(xmap->width, xmap->height, 
				    x0, y0, x1, y1, getMatrix);

  switch (xmap->bits_per_pixel) {
  case 24:
    fillXImage24(img, xmap);
    break;
  case 32:
    fillXImage32(img, xmap);
    break;
  }

  // clean up
  if (img)
    delete [] img;
}

void FrameRGBTrueColor24::buildColormapXM(const unsigned char* img)
{
  int& height = colormapXM->height;
  memset(colormapXM->data, 255, colormapXM->bytes_per_line * height);

  switch (colormapXM->bits_per_pixel) {
  case 24:
    fillXImage24(img, colormapXM);
    break;
  case 32:
    fillXImage32(img, colormapXM);
    break;
  }
}

void FrameRGBTrueColor24::fillXImage24(const unsigned char* img, XImage* xmap)
{
  int& width = xmap->width;
  int& height = xmap->height;
  int bytesPerPixel = xmap->bits_per_pixel/8;

#ifndef _WIN32
  int rs = decodeMask((unsigned long)visual->red_mask);
  int gs = decodeMask((unsigned long)visual->green_mask);
  int bs = decodeMask((unsigned long)visual->blue_mask);
#else
  int rs = decodeMask((unsigned long)0x00FF0000);
  int gs = decodeMask((unsigned long)0x0000FF00);
  int bs = decodeMask((unsigned long)0x000000FF);
#endif

  // we need to check to byteswap when we have cross platforms
  const unsigned char* ptr = img;
  int msb = xmap->byte_order;

  if ((!msb && lsb()) || (msb && !lsb())) {
    for (int j=0; j<height; j++) {
      // the line may be padded at the end
      char* dest = xmap->data + j*xmap->bytes_per_line;

      for (int i=0; i<width; i++, dest+=bytesPerPixel, ptr+=3) {
	unsigned int r = ptr[0];
	unsigned int g = ptr[1];
	unsigned int b = ptr[2];
	unsigned int a = 0;
	a |= r << rs;
	a |= g << gs;
	a |= b << bs;

	memcpy(dest, &a, 3);
      }
    }
  }
  else {
    for (int j=0; j<height; j++) {
      // the line may be padded at the end
      char* dest = xmap->data + j*xmap->bytes_per_line;

      for (int i=0; i<width; i++, dest+=bytesPerPixel, ptr+=3) {
	unsigned int r = ptr[0];
	unsigned int g = ptr[1];
	unsigned int b = ptr[2];
	unsigned int a = 0;
	a |= r << rs;
	a |= g << gs;
	a |= b << bs;

	unsigned char* rr = (unsigned char*)(&a);
	*(dest) = *(rr+3);
	*(dest+1) = *(rr+2);
	*(dest+2) = *(rr+1);
      }
    }
  }
}

void FrameRGBTrueColor24::fillXImage32(const unsigned char* img, XImage* xmap)
{
  int& width = xmap->width;
  int& height = xmap->height;
  int bytesPerPixel = xmap->bits_per_pixel/8;

#ifndef _WIN32
  int rs = decodeMask((unsigned long)visual->red_mask);
  int gs = decodeMask((unsigned long)visual->green_mask);
  int bs = decodeMask((unsigned long)visual->blue_mask);
#else
  int rs = decodeMask((unsigned long)0x00FF0000);
  int gs = decodeMask((unsigned long)0x0000FF00);
  int bs = decodeMask((unsigned long)0x000000FF);
#endif

  // we need to check to byteswap when we have cross platforms
  const unsigned char* ptr = img;
  int msb = xmap->byte_order;

  if ((!msb && lsb()) || (msb && !lsb())) {
    for (int j=0; j<height; j++) {
      // the line may be padded at the end
      char* dest = xmap->data + j*xmap->bytes_per_line;

      for (int i=0; i<width; i++, dest+=bytesPerPixel, ptr+=3) {
	unsigned int r = ptr[0];
	unsigned int g = ptr[1];
	unsigned int b = ptr[2];
	unsigned int a = 0;
	a |= r << rs;
	a |= g << gs;
	a |= b << bs;

	memcpy(dest, &a, 4);
      }
    }
  }
  else {
    for (int j=0; j<height; j++) {
      // the line may be padded at the end
      char* dest = xmap->data + j*xmap->bytes_per_line;

      for (int i=0; i<width; i++, dest+=bytesPerPixel, ptr+=3) {
	unsigned int r = ptr[0];
	unsigned int g = ptr[1];
	unsigned int b = ptr[2];
	unsigned int a = 0;
	a |= r << rs;
	a |= g << gs;
	a |= b << bs;

	unsigned char* rr = (unsigned char*)(&a);
	*(dest) = *(rr+3);
	*(dest+1) = *(rr+2);
	*(dest+2) = *(rr+1);
	*(dest+3) = *(rr);
      }
    }
  }
}



