/*
  ifctld: Manage network interface for netscript
  Author: Shun-ichi TAHARA <jado@flowernet.gr.jp>
  Time-stamp: <03/06/15 09:27:34 jado@sheira>

  Copyright (c) 2002 Shun-ichi TAHARA
*/

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <signal.h>
#include <sys/time.h>
#include <net/if.h>

#include "beep.h"
#include "cmdname.h"
#include "daemon.h"
#include "ioctl.h"
#include "lockfile.h"
#include "log.h"
#include "netif.h"

int trigger = 0;

void set_trigger(int sig)
{
    if (sig == SIGUSR1)
	trigger = 1;
    else if (sig == SIGUSR2)
	trigger = -1;
}

void usage(void)
{
    fprintf(stderr, "Usage: %s {-s|-d|-m} device\n", cmdname);
    exit(0);
}

int main(int argc, char **argv)
{
    int i;
    char *p;
    int dynamic = -1;
    int manual = -1;
    char *device = NULL;
    int sock;
    struct ifreq ifr;
    char *hwaddr;
    int phyid = 0;
    int link, link_last, status, status_last, triggered, count;
    struct timeval tm;

    /* SIGCHLD may be set to SIG_IGN */
    signal(SIGCHLD, SIG_DFL);
    /* Standard I/Os' may be closed */
    reopen_stdio();
    set_cmdname(*argv);

    for (i = 1; i < argc; i++) {
	p = argv[i];
	if (*p == '-') {
	    while (*(++p)) {
		switch (*p) {
		case 's':
		    dynamic = 0;
		    manual = 0;
		    break;
		case 'd':
		    dynamic = 1;
		    manual = 0;
		    break;
		case 'm':
		    dynamic = 0;
		    manual = 1;
		    break;
		default:
		    usage();
		}
	    }
	} else if (device == NULL) {
	    device = p;
	} else {
	    usage();
	}
    }
    if (dynamic < 0 || device == NULL)
	usage();

    init_log(0);
    set_lockfile(device);
    sock = init_ifreq(device, &ifr);

    if ((hwaddr = get_hwaddr(sock, &ifr)) == NULL) {
	print_log("%s: device not found.", device);
	close(sock);
	exit(1);
    }

    if (daemonize(device, hwaddr)) {
	print_log("%s: device already started.", device);
	close(sock);
	exit(1);
    }

    if (dynamic) {
	if ((phyid = mdio_init(sock, &ifr)) >= 0) {
	    status = 0;
	} else {
	    print_log("%s: cannot watch link status,"
		      " falling back to static mode.", device);
	    dynamic = 0;
	    status = 1;
	    close(sock);
	}
    } else {
	status = !manual;
	close(sock);
    }

    link_last = status_last = 0;
    triggered = 0;
    count = 0;

    signal(SIGUSR1, set_trigger);
    signal(SIGUSR2, set_trigger);

    while (1) {
	if (trigger > 0) {
	    status = 1;
	    triggered = 1;
	} else if (trigger < 0) {
	    status = 0;
	    triggered = 1;
	} else {
	    triggered = 0;
	}
	trigger = 0;

	if (dynamic) {
	    while ((link = mdio_read(sock, &ifr, phyid, 1)) < 0)
		sleep(3);
	    link = (link & 0x0004) >> 2;

	    if (link != link_last) {
		/* Wait 1500ms to avoid chattering */
		if (count++ >= 6) {
		    status = link;
		    link_last = link;
		}
	    } else if (count) {
		count = 0;
	    }
	}

	if (status != status_last) {
	    if (status) {
		if (dynamic || triggered) {
		    beep(150, 1920);
		    beep(150, 1440);
		    if (triggered)
			print_log("%s: enabled by hand.", device);
		    else
			print_log("%s: caught link, enabled.", device);
		}
		enable_if(device, 1, 1);
		lock_device();
	    } else {
		unlock_device();
		if (dynamic || triggered) {
		    beep(150, 1440);
		    beep(150, 1920);
		    if (triggered)
			print_log("%s: disabled by hand.", device);
		    else
			print_log("%s: lost link, disabled.", device);
		}
		disable_if(device, 1);
	    }
	    status_last = status;
	}

	if (dynamic) {
	    tm.tv_sec = 0;
	    tm.tv_usec = 250000;
	    select(0, 0, 0, 0, &tm);
	} else {
	    pause();
	}
    }
}
