/*
  lockfile.c: Handle lockfile
  Author: Shun-ichi TAHARA <jado@flowernet.gr.jp>
  Time-stamp: <03/06/16 13:39:24 jado@sheira>

  Copyright (c) 2003 Shun-ichi TAHARA
*/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <ctype.h>
#include <unistd.h>
#include <errno.h>
#include <fcntl.h>
#include <sys/file.h>

#include "defines.h"
#include "log.h"

static char *perm_lockfile = NULL, *actv_lockfile = NULL;
static int lockfd_daemon, lockfd_device, lockfd_global;

void set_lockfile(char *device)
{
    if (perm_lockfile != NULL)
	free(perm_lockfile);
    if ((perm_lockfile =
	 (char *)malloc(strlen(NETSTATE) + strlen(device) + 8)) == NULL) {
	error_log("set_lockfile: malloc");
	exit(1);
    }
    sprintf(perm_lockfile, NETSTATE "/.%s.lock", device);

    if (actv_lockfile != NULL)
	free(actv_lockfile);
    if ((actv_lockfile =
	 (char *)malloc(strlen(NETSTATE) + strlen(device) + 6)) == NULL) {
	error_log("set_lockfile: malloc");
	exit(1);
    }
    sprintf(actv_lockfile, NETSTATE "/%s.pid", device);
}

void clean_lockfile(void)
{
    unlink(perm_lockfile);
    unlink(actv_lockfile);
}

int lock_daemon(void)
{
    if ((lockfd_daemon =
	 open(perm_lockfile, O_WRONLY|O_CREAT|O_TRUNC, 0644)) < 0) {
	error_log(perm_lockfile);
	exit(1);
    }
    if (lockf(lockfd_daemon, F_TLOCK, 0)) {
	if (errno == EAGAIN) {
	    return -1;
	} else {
	    error_log("lock_daemon: lockf");
	    exit(1);
	}
    }

    return 0;
}

void unlock_daemon(void)
{
    lockf(lockfd_daemon, F_ULOCK, 0);
    close(lockfd_daemon);
}

void set_pid(pid_t pid, char *hwaddr)
{
    int fd;
    int len;
    char buf[256];

    if ((fd = open(actv_lockfile, O_WRONLY|O_CREAT|O_TRUNC, 0644)) < 0) {
	error_log(actv_lockfile);
	exit(1);
    }

    len = snprintf(buf, 256, "%d\n%s\n", pid, hwaddr);
    if (write(fd, buf, len) < 0) {
	error_log(actv_lockfile);
	close(fd);
	unlink(actv_lockfile);
	exit(1);
    }
    close(fd);
}

pid_t get_pid(void)
{
    int fd;
    pid_t pid;
    char c;

    if ((fd = open(actv_lockfile, O_RDONLY)) < 0) {
	if (errno == ENOENT) {
	    return -1;
	} else {
	    error_log(actv_lockfile);
	    exit(1);
	}
    }

    pid = 0;
    while (read(fd, &c, 1) > 0 && isdigit(c))
	pid = pid * 10 + c - '0';
    close(fd);

    return pid;
}

void lock_device(void)
{
    if ((lockfd_device = open(actv_lockfile, O_RDWR)) < 0) {
	error_log(actv_lockfile);
	exit(1);
    }
    if (lockf(lockfd_device, F_TLOCK, 0)) {
	error_log("lock_device: lockf");
	exit(1);
    }
}

int test_device(int noexit)
{
    int fd;

    if ((fd = open(actv_lockfile, O_RDWR)) < 0) {
	if (noexit) {
	    return -1;
	} else {
	    error_log(actv_lockfile);
	    exit(1);
	}
    }

    if (lockf(fd, F_TEST, 0)) {
	close(fd);
	return 1;
    }

    close(fd);
    return 0;
}

void unlock_device(void)
{
    lockf(lockfd_device, F_ULOCK, 0);
    close(lockfd_device);
}

int lock_global(int block)
{
    if ((lockfd_global = open(GLCKFILE, O_WRONLY|O_CREAT|O_TRUNC, 0644)) < 0) {
	error_log(GLCKFILE);
	exit(1);
    }
    if (lockf(lockfd_global, (block ? F_LOCK : F_TLOCK), 0)) {
	if (errno == EAGAIN) {
	    return -1;
	} else {
	    error_log("lock_global: lockf");
	    exit(1);
	}
    }

    return 0;
}

void unlock_global(void)
{
    lockf(lockfd_global, F_ULOCK, 0);
    close(lockfd_global);
}
