/*
  netif.c: Switch network interface
  Author: Shun-ichi TAHARA <jado@flowernet.gr.jp>
  Time-stamp: <03/06/16 13:42:24 jado@sheira>

  Copyright (c) 2002 Shun-ichi TAHARA
*/

#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <errno.h>
#include <signal.h>
#include <sys/wait.h>

#include "lockfile.h"
#include "log.h"

static pid_t monitor_pid;
static int monitor_clients = 0;
static int (*monitor_pipefd)[2];

static void wait_monitor(int signum)
{
    pid_t pid;

    while ((pid = waitpid(monitor_pid, NULL, WNOHANG)) < 0 && errno == EINTR)
	;

    if (pid == monitor_pid) {
	monitor_pid = 0;
	monitor_clients = 0;
	free(monitor_pipefd);
	signal(SIGCHLD, SIG_DFL);
    }
}

static int (*init_pipefd(int n))[2]
{
    int (*pipefd)[2];
    int i;

    if ((pipefd = (int (*)[2])malloc(sizeof(int[2]) * n)) == NULL) {
	error_log("init_pipefd: malloc");
	exit(1);
    }
    for (i = 0; i < n; i++) {
	if (pipe(pipefd[i])) {
	    error_log("init_pipefd: pipe");
	    exit(1);
	}
    }

    return pipefd;
}

static int init_fdset(fd_set *fds, int (*pipefd)[2], int n)
{
    int i;
    int maxfd = -1;

    FD_ZERO(fds);
    for (i = 0; i < n; i++) {
	if (pipefd[i][0] < 0)
	    continue;
	if (maxfd < pipefd[i][0])
	    maxfd = pipefd[i][0];
	FD_SET(pipefd[i][0], fds);
    }

    return maxfd;
}

static void monitor_logs(int (*pipefd)[2], int n)
{
    int i, len;
    fd_set fds;
    int maxfd;
    char *p, *q, *r;
    char buf[1024];

    while ((maxfd = init_fdset(&fds, pipefd, n)) >= 0) {
	if (select(maxfd + 1, &fds, NULL, NULL, NULL) < 0) {
	    if (errno != EINTR) {
		error_log("monitor_logs: select");
		exit(1);
	    }
	    continue;
	}

	for (i = 0; i < n; i++) {
	    if (pipefd[i][0] < 0)
		continue;
	    if (FD_ISSET(pipefd[i][0], &fds)) {
		if (!(len = read(pipefd[i][0], buf, 1023))) {
		    close(pipefd[i][0]);
		    pipefd[i][0] = -1;
		} else {
		    p = buf;
		    r = buf + len;
		    *r = '\0';
		    while (p < r) {
			if ((q = strchr(p, '\n')) == NULL) {
			    print_log("%s", p);
			    break;
			}
			*q = '\0';
			print_log("%s", p);
			p = q + 1;
		    }
		}
	    }
	}
    }
}

static void call_rc_ifctl(int fd, char *device, char *action)
{
    /* Connect pipe to stdout and stderr */
    dup2(fd, 1);
    dup2(fd, 2);
    close(fd);

    execl(RCDIR "/rc.ifctl", "rc.ifctl", device, action, NULL);

    error_log(RCDIR "/rc.ifctl");
    exit(1);
}

void enable_if(char *device, int lock, int n)
{
    static int max_clients;
    int i;
    pid_t pid;

    if (!monitor_clients)
	max_clients = n;

    if (monitor_clients >= max_clients) {
	print_log("Child processes limit (%d) exceeded.", max_clients);
	exit(1);
    }

    if (lock)
	lock_global(1);

    if (!monitor_clients) {
	monitor_pipefd = init_pipefd(n);

	/* The first child (monitor) is disposed by signal */
	signal(SIGCHLD, wait_monitor);

	if ((monitor_pid = fork()) < 0) {
	    error_log("enable_if: fork");
	    exit(1);
	}

	if (!monitor_pid) {
	    /* The first child monitors pipe with the second child */
	    for (i = 0; i < n; i++) {
		close(monitor_pipefd[i][1]);
	    }

	    monitor_logs(monitor_pipefd, n);

	    exit(0);
	} else {
	    /* Parent */
	    for (i = 0; i < n; i++) {
		close(monitor_pipefd[i][0]);
	    }
	}
    }

    if ((pid = fork()) < 0) {
	error_log("enable_if: fork");
	exit(1);
    }

    if (!pid) {
	/* The second child executes rc.ifctl */
	call_rc_ifctl(monitor_pipefd[monitor_clients][1], device, "up");
	/* NOTREACHED */
    }

    close(monitor_pipefd[monitor_clients++][1]);

    /* Parent waits the second child (rc.ifctld) to exit */
    while (waitpid(pid, NULL, 0) < 0 && errno == EINTR)
	;

    if (lock)
	unlock_global();
}

void disable_if(char *device, int lock)
{
    int (*pipefd)[2];
    pid_t pid;

    if (lock)
	lock_global(1);

    pipefd = init_pipefd(2);

    if ((pid = fork()) < 0) {
	error_log("disable_if: fork");
	exit(1);
    }

    if (!pid) {
	/* Child executes rc.ifctl */
	close(pipefd[0][0]);
	call_rc_ifctl(pipefd[0][1], device, "down");
	/* NOTREACHED */
    }

    /* Parent monitors pipe with child */
    close(pipefd[0][1]);
    monitor_logs(pipefd, 1);
    free(pipefd);

    /* Then waits child (rc.ifctld) to exit */
    while (waitpid(pid, NULL, 0) < 0 && errno == EINTR)
	;

    if (lock)
	unlock_global();
}
