/*
  netscheme: Change network configuration scheme for netscript
  Author: Shun-ichi TAHARA <jado@flowernet.gr.jp>
  Time-stamp: <03/06/16 14:21:44 jado@sheira>

  Copyright (c) 2002 Shun-ichi TAHARA
*/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <ctype.h>
#include <unistd.h>
#include <errno.h>
#include <signal.h>
#include <fcntl.h>
#include <dirent.h>
#include <sys/wait.h>

#include "defines.h"
#include "cmdname.h"
#include "daemon.h"
#include "lockfile.h"
#include "log.h"
#include "netif.h"
#include "strarray.h"

void usage(void)
{
    fprintf(stderr, "Usage: %s [-qv] [-l|new-scheme]\n", cmdname);
    exit(0);
}

void print_scm()
{
    char **schemes;
    int start, guess_case;
    FILE *fp;
    char *p, *q, **r;
    char buf[1024];

    if ((fp = fopen(CNFFILE, "r")) == NULL) {
	error_log(CNFFILE);
	exit(1);
    }

    schemes = strary_append(NULL, "default", 0);

    start = 0;
    guess_case = 0;

    while (fgets(p = buf, 1024, fp) != NULL) {
	while (isspace(*p))
	    p++;
	if (!*p || *p == '\n' || *p == '#')
	    continue;

	if (!start && !strncmp(p, "case \"$ADDRESS\" in", 18)) {
	    start = 1;
	    guess_case = 1;
	} else if (!guess_case && !strncmp(p, ";;", 2)) {
	    guess_case = 1;
	} else if ((q = strchr(p, ')')) != NULL) {
	    guess_case = 0;
	    *q = '\0';
	    if ((p = strchr(p, ',')) == NULL ||
		(q = strrchr(++p, ',')) == NULL)
		continue;
	    *q = '\0';
	    if (strchr(p, '*') != NULL || strchr(p, '?') != NULL ||
		strchr(p, '[') != NULL || strchr(p, ']') != NULL)
		continue;

	    schemes = strary_append(schemes, p, 0);
	}
    }

    printf("default");
    for (r = schemes + 1; *r != NULL; r++) {
	printf(" %s", *r);
    }

    strary_free(schemes);
}

char *get_scm()
{
    int fd;
    char *p;
    off_t len;

    if ((fd = open(SCMFILE, O_RDWR|O_CREAT, 0644)) < 0) {
	error_log(SCMFILE);
	exit(1);
    }

    if ((len = lseek(fd, 0, SEEK_END)) == 0) {
	if ((p = strdup("default")) == NULL) {
	    error_log("get_scm: strdup");
	    close(fd);
	    exit(1);
	}
    } else {
	if ((p = (char *)malloc(len + 1)) == NULL) {
	    error_log("get_scm: malloc");
	    close(fd);
	    exit(1);
	}
	lseek(fd, 0, SEEK_SET);
	if (read(fd, p, len) < 0) {
	    error_log("get_scm: read");
	    close(fd);
	    exit(1);
	}
	close(fd);
	if (*(p + len - 1) == '\n')
	    *(p + len - 1) = '\0';
	else
	    *(p + len) = '\0';
    }

    return p;
}

void update_scm(char *scheme)
{
    int fd;

    if ((fd = open(SCMFILE, O_WRONLY|O_CREAT|O_TRUNC, 0644)) < 0) {
	error_log(SCMFILE);
	exit(1);
    }
    if (write(fd, scheme, strlen(scheme)) < 0 ||
	write(fd, "\n", 1) < 0) {
	error_log("update_scm: write");
	close(fd);
	exit(1);
    }
    close(fd);
}

void rename_status(char *device, char *oldscm, char *newscm)
{
    char *old, *new;

    if ((old = malloc(strlen(NETSTATE) +
		      strlen(device) + strlen(oldscm) + 3)) == NULL ||
	(new = malloc(strlen(NETSTATE) +
		      strlen(device) + strlen(newscm) + 3)) == NULL) {
	error_log("rename_status: malloc");
	exit(1);
    }

    sprintf(old, NETSTATE "/%s-%s", device, oldscm);
    sprintf(new, NETSTATE "/%s-%s", device, newscm);

    if (rename(old, new) && errno != ENOENT) {
	error_log("rename_status: rename");
	exit(1);
    }

    free(old);
    free(new);
}

int main(int argc, char **argv)
{
    int i;
    char *p, **q;
    int verbose = 0;
    int list = 0;
    char *scheme = NULL;
    char *current;
    char **devices;
    int n;
    DIR *dir;
    struct dirent *entry;
    pid_t pid;
    int status;

    /* SIGCHLD may be set to SIG_IGN */
    signal(SIGCHLD, SIG_DFL);
    /* Standard I/Os' may be closed */
    reopen_stdio();
    set_cmdname(*argv);

    for (i = 1; i < argc; i++) {
	p = argv[i];
	if (*p == '-') {
	    while (*(++p)) {
		switch (*p) {
		case 'q':
		    verbose = -1;
		    break;
		case 'v':
		    verbose = 1;
		    break;
		case 'l':
		    list = 1;
		    break;
		default:
		    usage();
		}
	    }
	} else if (scheme == NULL) {
	    scheme = p;
	} else {
	    usage();
	}
    }

    if (list) {
	if (scheme != NULL)
	    usage();
	if (verbose >= 0)
	    printf("Available schemes: ");
	print_scm();
	if (verbose >= 0)
	    printf(".\n");
	else
	    putchar('\n');
	exit(0);
    }

    current = get_scm();

    if (scheme == NULL) {
	if (verbose >= 0)
	    printf("Current scheme: %s.\n", current);
	else
	    printf("%s\n", current);
	free(current);
	exit(0);
    }

    if (!strcmp(current, scheme)) {
	if (verbose >= 0)
	    printf("Scheme unchanged: %s.\n", current);
	free(current);
	exit(0);
    }

    if (geteuid() == 0 && getuid() != 0) {
	/* Set ruid to root so that sh script can export shell functions */
	setreuid(0, -1);
	setregid(0, 0);
    }

    if (lock_global(0)) {
	print_log("Cannot change the scheme now.");
	exit(1);
    }

    if ((dir = opendir(NETSTATE)) == NULL) {
	error_log(NETSTATE);
	exit(1);
    }

    devices = NULL;
    n = 0;
    while ((entry = readdir(dir)) != NULL) {
	if (entry->d_name[0] == '.')
	    continue;
	if ((p = strrchr(entry->d_name, '.')) == NULL || strcmp(p, ".pid"))
	    continue;
	*p = '\0';

	set_lockfile(entry->d_name);
	if (!test_device(0)) {
	    if (verbose > 0)
		printf("Checking %s: disabled - skip.\n", entry->d_name);
	    continue;
	}

	if ((pid = fork()) < 0) {
	    error_log("main: fork");
	    exit(1);
	}

	if (!pid) {
	    execl(RCDIR "/rc.netscheme",
		  "rc.netscheme", entry->d_name, current, scheme, NULL);

	    error_log(RCDIR "/rc.netscheme");
	    exit(1);
	}

	if (wait(&status) <= 0) {
	    error_log("main: wait");
	    exit(1);
	}
	if (WIFEXITED(status) && !WEXITSTATUS(status)) {
	    if (verbose > 0)
		printf("Checking %s: same config - skip.\n", entry->d_name);
	    rename_status(entry->d_name, current, scheme);
	    continue;
	}

	if (verbose > 0)
	    printf("Checking %s: will reset.\n", entry->d_name);
	devices = strary_append(devices, entry->d_name, 0);
    }
    closedir(dir);
    n = strary_len(devices);

    init_log(1);

    if (n) {
	if (verbose >= 0) {
	    printf("Stopping interfaces:");
	    fflush(stdout);
	}
	for (q = devices; *q != NULL; q++) {
	    if (verbose >= 0) {
		printf(" %s", *q);
		fflush(stdout);
	    }
	    disable_if(*q, 0);
	}
	if (verbose >= 0)
	    printf(".\n");
    }

    if (verbose >= 0)
	printf("Change scheme: %s --> %s.\n", current, scheme);
    print_log("Changing scheme: %s --> %s.", current, scheme);
    update_scm(scheme);

    if (n) {
	if (verbose >= 0) {
	    printf("Starting interfaces:");
	    fflush(stdout);
	}
	for (q = devices; *q != NULL; q++) {
	    if (verbose >= 0) {
		printf(" %s", *q);
		fflush(stdout);
	    }
	    print_log("%s: changed configuration, reset.", *q);
	    enable_if(*q, 0, n);
	}
	if (verbose >= 0)
	    printf(".\n");
    }

    strary_free(devices);

    unlock_global();

    exit(0);
}
