/**
* @license Apache-2.0
*
* Copyright (c) 2018 The Stdlib Authors.
*
* Licensed under the Apache License, Version 2.0 (the "License");
* you may not use this file except in compliance with the License.
* You may obtain a copy of the License at
*
*    http://www.apache.org/licenses/LICENSE-2.0
*
* Unless required by applicable law or agreed to in writing, software
* distributed under the License is distributed on an "AS IS" BASIS,
* WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
* See the License for the specific language governing permissions and
* limitations under the License.
*/

'use strict';

// MODULES //

var tape = require( 'tape' );
var validate = require( './../lib/validate.js' );


// FUNCTIONS //

/**
* Returns a set of default options.
*
* @private
* @returns {Object} options object
*/
function options() {
	return {
		'intercept': true,
		'lambda': 1.0e-4,
		'learningRate': [ 'basic' ],
		'loss': 'log'
	};
}


// TESTS //

tape( 'main export is a function', function test( t ) {
	t.ok( true, __filename );
	t.equal( typeof validate, 'function', 'main export is a function' );
	t.end();
});

tape( 'the function returns an error if provided an options argument which is not an object', function test( t ) {
	var values;
	var err;
	var i;

	values = [
		'5',
		5,
		true,
		void 0,
		null,
		NaN,
		[],
		function noop() {}
	];

	for ( i = 0; i < values.length; i++ ) {
		err = validate( options(), values[ i ] );
		t.equals( err instanceof TypeError, true, 'returns a type error when provided '+values[i] );
	}
	t.end();
});

tape( 'the function returns an error if provided an `intercept` option which is not a boolean', function test( t ) {
	var values;
	var err;
	var i;

	values = [
		5,
		'5',
		void 0,
		null,
		NaN,
		[],
		{},
		function noop() {}
	];

	for ( i = 0; i < values.length; i++ ) {
		err = validate( options(), {
			'intercept': values[ i ]
		});
		t.equal( err instanceof TypeError, true, 'returns a type error when provided '+values[i] );
	}
	t.end();
});

tape( 'the function returns an error if provided an `lambda` option which is not a nonnegative number', function test( t ) {
	var values;
	var err;
	var i;

	values = [
		-5,
		'5',
		true,
		false,
		void 0,
		null,
		NaN,
		[],
		{},
		function noop() {}
	];

	for ( i = 0; i < values.length; i++ ) {
		err = validate( options(), {
			'lambda': values[ i ]
		});
		t.equal( err instanceof TypeError, true, 'returns a type error when provided '+values[i] );
	}
	t.end();
});

tape( 'the function returns an error if provided a `learningRate` option which is not an array-like object', function test( t ) {
	var values;
	var err;
	var i;

	values = [
		5,
		'5',
		true,
		false,
		void 0,
		null,
		NaN,
		{},
		function noop() {}
	];

	for ( i = 0; i < values.length; i++ ) {
		err = validate( options(), {
			'learningRate': values[ i ]
		});
		t.equal( err instanceof TypeError, true, 'returns a type error when provided '+values[i] );
	}
	t.end();
});

tape( 'the function returns an error if provided a `learningRate` option whose first element is not a recognized string', function test( t ) {
	var values;
	var err;
	var i;

	values = [
		5,
		'5',
		true,
		false,
		void 0,
		null,
		NaN,
		{},
		[],
		function noop() {}
	];

	for ( i = 0; i < values.length; i++ ) {
		err = validate( options(), {
			'learningRate': [ values[ i ] ]
		});
		t.equal( err instanceof TypeError, true, 'returns a type error when provided '+values[i] );
	}
	t.end();
});

tape( 'the function returns an error if provided a `learningRate` option whose second element is not a positive number (constant, invscaling)', function test( t ) {
	var values;
	var err;
	var i;

	values = [
		-5,
		0,
		'5',
		void 0,
		null,
		NaN,
		[],
		{},
		function noop() {}
	];

	for ( i = 0; i < values.length; i++ ) {
		err = validate( options(), {
			'learningRate': [ 'constant', values[ i ] ]
		});
		t.equal( err instanceof TypeError, true, 'returns a type error when provided '+values[i] );

		err = validate( options(), {
			'learningRate': [ 'invscaling', values[ i ] ]
		});
		t.equal( err instanceof TypeError, true, 'returns a type error when provided '+values[i] );
	}
	t.end();
});

tape( 'the function returns an error if provided a `learningRate` option whose third element is not a number (invscaling)', function test( t ) {
	var values;
	var err;
	var i;

	values = [
		'5',
		true,
		false,
		void 0,
		null,
		[],
		{},
		function noop() {}
	];

	for ( i = 0; i < values.length; i++ ) {
		err = validate( options(), {
			'learningRate': [ 'invscaling', 4, values[ i ] ]
		});
		t.equal( err instanceof TypeError, true, 'returns a type error when provided '+values[i] );
	}
	t.end();
});

tape( 'the function returns an error if provided a `loss` option which is not a recognized string', function test( t ) {
	var values;
	var err;
	var i;

	values = [
		5,
		'5',
		true,
		false,
		void 0,
		null,
		NaN,
		[],
		{},
		function noop() {}
	];

	for ( i = 0; i < values.length; i++ ) {
		err = validate( options(), {
			'loss': values[ i ]
		});
		t.equal( err instanceof TypeError, true, 'returns a type error when provided '+values[i] );
	}
	t.end();
});

tape( 'the function returns `null` if all options are valid', function test( t ) {
	var opts1;
	var opts2;
	var err;

	opts1 = options();
	opts2 = {
		'loss': 'log',
		'lambda': 1e-6,
		'learningRate': [ 'pegasos' ],
		'intercept': true
	};
	err = validate( opts1, opts2 );

	t.equal( err, null, 'returns null' );
	t.deepEqual( opts1, opts2, 'sets option values' );

	t.end();
});

tape( 'the function ignores unrecognized/unsupported options', function test( t ) {
	var opts1;
	var opts2;
	var err;

	opts1 = options();
	opts2 = {
		'beep': true, // misc options
		'boop': 'bop'
	};
	err = validate( opts1, opts2 );

	t.equal( err, null, 'returns null' );
	t.deepEqual( opts1, options(), 'does not set any option values' );
	t.end();
});
