psANDpdf <- function(fname, EXPR,
                     horizontal=FALSE, width=7, height=7, ...)
{
  ## Purpose: postscript(..); {plot} ; pdf(..) ; {plot}
  ## ----------------------------------------------------------------------
  ## Author: Martin Maechler, Date:  3 Jun 2002, 21:32
    postscript(file = (f1 <- paste(fname, "eps", sep=".")),
               horizontal=horizontal, width=width, height=height, ...)
    ee <- eval.parent(EXPR)
    dev.off()

    pdf(file = (f2 <- paste(fname, "pdf", sep=".")),
        horizontal=horizontal, width=width, height=height, ...)
    ## Trick: EXPR is already evaluated, but we still have
    eval.parent(match.call()$EXPR)
    dev.off()

    cat("\n");system(paste("ls -lt", f1, f2))
    invisible(ee)
}

library(scatterplot3d)
if(.Platform $ OS.type == "unix") { ## Martin:
    setwd("/u/maechler/tex/o.peopl/scatterplot3d")
} else { ## Uwe:
    setwd("d:/uwe/tagungen/sfbs3d2002")
}

######################################
##### I  artificial data
### 1) HELIX
psANDpdf("helix",
     {
         z <- seq(-10, 10, 0.01)
         x <- cos(z)
         y <- sin(z)
         scatterplot3d(x, y, z, highlight.3d = TRUE, col.axis = "blue",
                       col.grid = "lightblue", main = "Helix", pch = 20)
     })


### 2) HEMISPHERE
psANDpdf("hemisphere",
     {
         temp <- seq(-pi, 0, length = 50)
         x <- c(rep(1, 50) %*% t(cos(temp)))
         y <- c(cos(temp) %*% t(sin(temp)))
         z <- c(sin(temp) %*% t(sin(temp)))
         scatterplot3d(x, y, z, highlight.3d=TRUE,  angle=120,
                       col.axis="blue", col.grid="lightblue",
                       cex.axis = 1.3, cex.lab = 1.1, main="Hemisphere", pch=20)
     })

### 3) 3D barplot
set.seed(321)# reproducible graphic for publication!
psANDpdf("barplot",
     {
         my.mat <- matrix(runif(25), nrow = 5)
         dimnames(my.mat) <- list(LETTERS[1:5], letters[11:15])
         s3d.dat <- data.frame(columns = c(col(my.mat)),
                               rows = c(row(my.mat)),
                               value = c(my.mat))
         scatterplot3d(s3d.dat, type = "h", lwd = 5, pch = " ",
                       x.ticklabs = colnames(my.mat),
                       y.ticklabs = rownames(my.mat),
                       color = grey(25:1 / 40), main = "3D barplot")
     })

### 4) Adding Elements
psANDpdf("elements",
     {
         data(trees)
         s3d <- scatterplot3d(trees, type = "h", color="blue",
                              angle = 55, scale.y = 0.7, pch = 16,
                              main = "Adding elements")
         my.lm <- lm(trees$Volume ~ trees$Girth + trees$Height)
         s3d$plane3d(my.lm)
         s3d$points3d(seq(10, 20, 2), seq(85, 60, -5), seq(60, 10, -10),
                      col = "red", type = "h", pch = 8)
     })

### 5) Binormal
library(mvtnorm)
x1 <- x2 <- seq(-10, 10, length = 51)
dens <- matrix(dmvnorm(expand.grid(x1, x2),
                       sigma = rbind(c(3, 2), c(2,3))),
               ncol = length(x1))

psANDpdf("binorm", {
    s3d <- scatterplot3d(x1, x2, seq(min(dens), max(dens), length = length(x1)),
                         type = "n", grid = FALSE, angle = 70,
                         zlab = expression(f(x[1],x[2])),
                         xlab = expression(x[1]), ylab = expression(x[2]),
                         main = "Bivariate normal distribution")
    text(s3d$xyz.convert(-1, 10, 0.07),
         labels = expression(f(x) == frac(1, sqrt((2 * pi)^n * phantom(".") *
             det(Sigma[X]))) * phantom(".") * exp *
             bgroup("(", - scriptstyle(frac(1, 2) * phantom(".")) * (x - mu)^T
                    * Sigma[X]^-1 * (x - mu), ")")))
    text(s3d$xyz.convert(1.5, 10, 0.05),
         labels = expression("with" * phantom("m") *
             mu == bgroup("(", atop(0, 0), ")") *phantom(".")* "," *phantom(0) *
             Sigma[X] ==  bgroup("(", atop(3 * phantom(0) * 2,
                                           2 * phantom(0) * 3), ")")))
    for(i in length(x1):1)
        s3d$points3d(rep(x1[i], length(x2)), x2, dens[i,], type = "l")
    for(i in length(x2):1)
        s3d$points3d(x1, rep(x2[i], length(x1)), dens[,i], type = "l")
})

######################################
##### II real world
### 1) Berkey (C3)
berkey <- read.table("berkey.txt", header = TRUE)
attach(berkey)
Risk <- Disease / Total
vac <- Vaccinated == 1
Prec <- log(Total / (Risk * (1 - Risk)), 10)
psANDpdf("meta", {
    layout(cbind(1:2, 1:2), heights = c(7, 1))
    prc <- hsv((prc <- 0.7 * Prec / diff(range(Prec))) - min(prc) + 0.3)
    s3d <- scatterplot3d(Year, Latitude, Risk, mar = c(5, 3, 4, 3),
                         type = "h", pch = " ", main = "Estimated TB risks")
    s3d$points(Year, Latitude, Risk, pch = ifelse(vac, 22, 21), bg = prc,
               cex = ifelse(vac, 2, 1.5))
    s3d.coords <- s3d$xyz.convert(Year, Latitude, Risk)
    al.char <- toupper(substr(as.character(Allocation), 1, 1))
    text(s3d.coords$x[!vac], s3d.coords$y[!vac], labels = al.char[!vac],
         pos = 2, offset = 0.5)
    legend(s3d$xyz.convert(80, 15, 0.21), pch = c("A", "R", "S"), yjust = 0,
           legend = c("alternate", "random", "systematic"), cex = 1.1)
    legend(s3d$xyz.convert(47, 60, 0.24), pch = 22:21, yjust = 0,
           legend = c("vaccinated", "not vaccinated"), cex = 1.1)
    par(mar=c(5, 3, 0, 3))
    plot(seq(min(Prec), max(Prec), length = 100), rep(0, 100), pch = 15,
         axes = FALSE, xlab = "color code of variable \"Precision\"",
         ylab = "", col = hsv(seq(0.3, 1, length = 100)))
    axis(1, at = 4:7, labels = expression(10^4, 10^5, 10^6, 10^7))
},  width=7, height=8)


## Von Guido Knapp, Zitat:
#Sind sechs Dimensionen mglich?
#1.Dim:  year
#2.Dim:  latitude
#3.Dim:  rel. Huf. disease / total
#4.Dim:  vaccinated 1=ja , 0=nein
#5.Dim:  allocation
#
#Gre bzw. Umfang der Knubbels
#6.Dim: Przision der Rel. Hufigkeiten:
#       total / ( (disease / total) * (1 - disease / total) )

### 2) i) Business cycles (B3)
B3 <- read.table("B3Daten.txt")
attach(B3)
psANDpdf("business", {
    layout(cbind(1:2, 1:2), heights = c(7, 1))
    temp <- hsv((temp <- 0.7 * IE / diff(range(IE))) - min(temp) + 0.3)
    s3d <- scatterplot3d(L, C, Y, pch = Phase, color = temp,
                         mar = c(5, 3, 4, 3), main = "Business cycle phases")
    legend(s3d$xyz.convert(-2, 0, 16), pch = 1:4, yjust = 0,
           legend = c("upswing", "upper turning points",
           "downswing", "lower turning points"))
    s3d$plane3d(my.lm <- lm(Y ~ L + C), lty="dotted")

    par(mar=c(5, 3, 0, 3))
    plot(seq(min(IE), max(IE), length = 100), rep(0, 100), pch = 15,
         axes = FALSE, xlab = "color code of variable \"IE\"", ylab = "",
         col = hsv(seq(0.3, 1, length = 100)))
    axis(1, at = seq(-20, 25, 5))
}, width=7, height=8)


### 2) ii) Residuals
psANDpdf("residuals",
     {
         s3d <- scatterplot3d(L, C, Y, pch = 20, mar = c(5, 3, 4, 3),
                              main = "Residuals")
         s3d$plane3d(my.lm, lty = "dotted")
         orig <- s3d$xyz.convert(L, C, Y)
         plane <- s3d$xyz.convert(L, C,  fitted(my.lm))
         i.negpos <- 1 + (resid(my.lm) > 0)
         segments(orig$x, orig$y, plane$x, plane$y,
                  col = c("blue","red")[i.negpos], lty = (2:1)[i.negpos])
     })

### 3) Drilling
drill1 <- scan("C5Daten_ratter.txt")
drill2 <- scan("C5Daten_still.txt")
ii <- 1:400
xl <- "drilling torque"
yl <- "drilling torque, lag 6"
zl <- "drilling torque, lag 31"

psANDpdf("drill1", {
    s3d <-
        scatterplot3d(drill1[ii], drill1[6 + ii], drill1[31 + ii],
                      color = "red", type = "l", angle = 120,
                      xlab = xl, ylab = yl, zlab = zl,
                      main = "Two deep hole drilling processes")
    s3d$points3d(drill2[ii], drill2[6 + ii], drill2[31 + ii],
                 col = "blue", type = "l")
    legend(s3d$xyz.convert(-400, 1000, 950), col= c("blue", "red"), lwd = 2,
           legend = c("regular process", "chattering process"), bg="white")
})

psANDpdf("drill2", {
    scatterplot3d(drill2[ii], drill2[6 + ii], drill2[31 + ii],
                  color = "blue", type = "l", angle = 120,
                  xlab = xl, ylab = yl, zlab = zl,
                  main = "Magnification of the regular process")
})
