/*
 * Copyright 2008 Sony Corporation
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 *   * Redistributions of source code must retain the above copyright notice,
 *     this list of conditions and the following disclaimer.
 *   * Redistributions in binary form must reproduce the above copyright
 *     notice, this list of conditions and the following disclaimer in the
 *     documentation and/or other materials provided with the distribution.
 *   * Neither the names of the copyright holders nor the names of their
 *     contributors may be used to endorse or promote products derived from this
 *     software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 */

#ifndef E_CELL_H
#define E_CELL_H

#include <stdio.h>
#include <stdint.h>
#include <assert.h>

/*** debugging ***/
#ifdef CELL_DEBUG
#  ifdef __GNUC__
#    define DPRINTF(fmt, ...)						\
	fprintf(stderr, "%s:%d:%s: " fmt ,				\
		__FILE__, __LINE__, __FUNCTION__, ##__VA_ARGS__)
#else /* !__GNUC__ */
#    define DPRINTF(...) fprintf(stderr, __VA_ARGS__)
#endif /* !__GNUC__ */
#else /* !CELL_DEBUG */
#  define DPRINTF(...) /* empty */
#endif /* !CELL_DEBUG */

#define ASSERT(e) assert(e)

/*** common part ***/
#define SHARED_DATA_ALIGN 16
#define SHARED_ATOMIC_DATA_ALIGN 128

#define ALIGNED(t, a) t __attribute__((aligned(a)))

#define ALIGN_CEIL(v, a)  (((v) + (a) - 1) & ~(uintptr_t)((a) - 1))
#define ALIGN_FLOOR(v, a) ((v) & ~(uintptr_t)((a) - 1))

#define IS_ALIGNED(v, a) (!((uintptr_t)(v) & ((a) - 1)))

#define CEIL(v, u) (((v) + (u) - 1) / (u) * (u))
#define FLOOR(v, u) ((v) / (u) * (u))

/*** parameters ***/
#define MIN_QUEUE_DEPTH        1
#define MAX_QUEUE_DEPTH     1024
#define DEFAULT_QUEUE_DEPTH   8

#define MIN_UNIT_SIZE     (1024 * 4) /* 4KiB */
#define MAX_UNIT_SIZE     (1024 * 1024 * 128) /* 128MiB */
#define DEFAULT_UNIT_SIZE (1024 * 8) /* 8KiB */

/* MD */
/* request (PPE -> SPE) */
#define SPE_DIGEST_COMMAND_INIT   1
#define SPE_DIGEST_COMMAND_UPDATE 2
#define SPE_DIGEST_COMMAND_FINAL  3
#define SPE_DIGEST_COMMAND_QUIT   4

typedef struct spe_digest_request_st
	{
	uint32_t request; /* command */
	uint32_t size; /* valid size of payload */
	uint64_t in_ea; /* EA of input data */
	uint32_t nid; /* algorithm */
	uint32_t pad[3];
	} spe_digest_request_t;

/* result (SPE -> PPE) */
#define SPE_DIGEST_SIZE_MAX 128
typedef struct spe_digest_result_st
	{
	uint8_t digest[SPE_DIGEST_SIZE_MAX];
	} spe_digest_result_t;

/* ciphers */
/* request (PPE -> SPE) */
#define SPE_CIPHER_COMMAND_INIT    1
#define SPE_CIPHER_COMMAND_DO      2 /* encrypt or decrypt */
#define SPE_CIPHER_COMMAND_QUIT    3

typedef struct spe_cipher_request_st
	{
	uint32_t request; /* command */
	uint32_t size; /* valid size of payload */
	uint32_t nid; /* algorithm */
	uint32_t encrypt; /* encryption or decryption */

	uint32_t flags; /* mode etc */
	uint32_t key_size;
	uint32_t iv_size;
	uint32_t block_size;

	uint64_t in_ea;  /* EA of payload */
	uint64_t out_ea; /* EA of output area */
	} spe_cipher_request_t;

/* result (SPE -> PPE) */
typedef struct spe_cipher_result_st
	{
	uint32_t size;
	uint32_t pad;
	uint64_t out_ea;
	} spe_cipher_result_t;


#ifdef __SPU__
/*** SPE specific part ***/

#include <spu_mfcio.h>

#define ALIGN_HINT(ptr, base, off) __align_hint(ptr, base, off)

/* main function */
int spe_main(uint64_t arg1, uint64_t arg2);

/* data queue (SPE side) */
int cell_queue_push(uint64_t q_ea, const void *buffer);
int cell_queue_pop(uint64_t q_ea, void *buffer);

#else /* !__SPU__ */
/*** PPE specific part ***/

#include <openssl/engine.h>
#include <libspe2.h>
#include <pthread.h>

/* global parameters */
extern int cell_queue_depth;
extern int cell_unit_size;
extern int cell_task_priority;

/* OpenSSL engine interface functions */
int cell_digests(ENGINE *e, const EVP_MD **digest, const int **nids, int nid);
int cell_ciphers(ENGINE *e, const EVP_CIPHER **cipher, const int **nids, int nid);

/* memory allocation */
#define aligned_malloc(a, s) aligned_malloc_internal(a, s, __FILE__, __LINE__)
void *aligned_malloc_internal(size_t align, size_t size,
	const char *file, int line);
void aligned_free(void *p);

/* SPE tasks */
typedef struct spe_task_st spe_task_t;

int spe_task_set_runtime_system(void *instance);

int spe_task_schedular_initialize(void);
int spe_task_schedular_finalize(void);

spe_task_t *spe_task_create(spe_program_handle_t *prog);
int spe_task_run(spe_task_t *spe, uint64_t in_q, uint64_t out_q);
int spe_task_wait(spe_task_t *spe);
int spe_task_destroy(spe_task_t *spe);

/* data queue (PPE side) */
uint64_t cell_queue_create(spe_task_t *task,
	int elm_size, int depth, int from_spe_to_ppe);
void cell_queue_destroy(uint64_t q);
int cell_queue_try_push(uint64_t q, const void *buffer);
int cell_queue_push(uint64_t q, const void *buffer);
int cell_queue_pop(uint64_t q, void *buffer);
int cell_queue_try_pop(uint64_t q, void *buffer);

/* internal status (PPE side) */
typedef struct cell_md_ctx_st
	{
	uint64_t q_in;
	uint64_t q_out;
	spe_task_t *spe;
	uint32_t queue_depth;
	uint32_t unit_size;
	void *in_buffer;
	unsigned int in_buffer_ptr;
	} cell_md_ctx_t;

typedef struct cell_cipher_ctx_st
	{
	uint64_t q_in;
	uint64_t q_out;
	spe_task_t *spe;
	uint32_t queue_depth;
	uint32_t unit_size;
	void *in_buffer;
	void *out_buffer;
	unsigned int in_buffer_ptr;
	unsigned int out_buffer_ptr;
	} cell_cipher_ctx_t;

#endif /* !__SPU__ */

#endif /* E_CELL_H */
