/*
 * Copyright 2008 Sony Corporation
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 *   * Redistributions of source code must retain the above copyright notice,
 *     this list of conditions and the following disclaimer.
 *   * Redistributions in binary form must reproduce the above copyright
 *     notice, this list of conditions and the following disclaimer in the
 *     documentation and/or other materials provided with the distribution.
 *   * Neither the names of the copyright holders nor the names of their
 *     contributors may be used to endorse or promote products derived from this
 *     software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 */

#include "e_cell.h"
#include "e_cell_err.h"

#include "libspe2_runtime.h"

#include <string.h>
#include <poll.h>
#include <unistd.h>

spe_task_t *spe_task_create(spe_program_handle_t *prog)
	{
	spe_task_t *task;
	unsigned int flags;

	task = OPENSSL_malloc(sizeof(*task));
	if (!task)
		{
		CELLerr(CELL_F_SPE_TASK_RUN,CELL_R_MEMORY_ALLOCATION_FAILED);
		return NULL;
		}

	memset(task, 0, sizeof(*task));

#if 1
	/* we don't use SPE_MAP_PS to avoid locking at mbox access
	   with SPU context switches */
	flags = 0;
#else
	flags = SPE_MAP_PS;
#endif
	task->spe = spe_context_create(flags, 0);
	if (!task->spe)
		{
		OPENSSL_free(task);
		CELLerr(CELL_F_SPE_TASK_RUN,CELL_R_SPE_CONTEXT_CREATION_FAILED);
		return NULL;
		}

	task->prog = prog;

	return task;
	}

int spe_task_destroy(spe_task_t *task)
	{
	if (task->spe)
		{
		spe_context_destroy(task->spe);
		}

	OPENSSL_free(task);

	return 1;
	}

static int spe_run(spe_context_ptr_t spe, spe_program_handle_t *prog,
		   uint64_t arg1, uint64_t arg2)
	{
	int ret;
	spe_stop_info_t si;
	unsigned int entry = SPE_DEFAULT_ENTRY;

	ret = spe_program_load(spe, prog);
	if (ret)
		{
		/* FIXME: appropriate error handling should be performed here. */
		CELLerr(CELL_F_SPE_RUN,CELL_R_SPE_PROGRAM_LOAD_FAILED);
		return 0;
		}

	for ( ; ; )
		{
		ret = spe_context_run(spe, &entry, 0,
			(void *)(uintptr_t)arg1, (void *)(uintptr_t)arg2, &si);
		if (ret == 0)
			{ /* normal exit */
			DPRINTF("SPE exit.\n");
			break;
			}
		else if (ret < 0)
			{
			CELLerr(CELL_F_SPE_RUN,CELL_R_SPE_CONTEXT_RUN_FAILED);
			/* FIXME: appropriate error handling should be performed here. */
			break;
			}
		else
			{
			/* user defined stop code */
			CELLerr(CELL_F_SPE_RUN,CELL_R_UNEXPECTED_STOP_AND_SIGNAL);
			}
		}

	return 1;
	}

static void *spe_thread_proc(void *arg)
	{
	spe_task_t *task = (spe_task_t *)arg;

	spe_run(task->spe, task->prog, task->arg1, task->arg2);

	return NULL;
	}

static int spe_task_handle_mbox(spe_task_t *task)
	{
	int ret;
	unsigned int ea_h, ea_l;
        void *ptr;

	ret = spe_out_intr_mbox_read(task->spe, &ea_l, 1,
		SPE_MBOX_ANY_BLOCKING);
	if (ret != 1)
		{
		CELLerr(CELL_F_SPE_TASK_HANDLE_MBOX, CELL_R_READ_FAILED);
		return 0;
		}

	ret = spe_out_mbox_read(task->spe, &ea_h, 1);
	if (ret != 1)
		{
		CELLerr(CELL_F_SPE_TASK_HANDLE_MBOX, CELL_R_READ_FAILED);
		return 0;
		}

        ptr = (void *)(uintptr_t)(((uint64_t)ea_h << 32) | ea_l);
        if (!ptr)
                {
                /* last notification */
                return -1;
                }

	sync_notify(ptr);

	return 1;
	}

#ifdef ENABLE_SYNC_FUTEX
static void *spe_handler_thread_proc(void *arg)
        {
        spe_task_t *task = (spe_task_t *)arg;

        while (spe_task_handle_mbox(task) > 0)
                {
                /* do nothing */
                }

        return NULL;
        }
#endif /* ENABLE_SYNC_FUTEX */

int spe_task_run(spe_task_t *task, uint64_t in_q, uint64_t out_q)
	{
	int ret;

	/* run */
	task->arg1 = in_q;
	task->arg2 = out_q;
	ret = pthread_create(&task->thread, NULL, spe_thread_proc, task);
	if (ret)
		{
		CELLerr(CELL_F_SPE_TASK_RUN,CELL_R_THREAD_CREATION_FAILED);
		return 0;
		}

#ifdef ENABLE_SYNC_FUTEX
        /* start notification handler thread */
	ret = pthread_create(&task->handler_thread, NULL,
                spe_handler_thread_proc, task);
	if (ret)
		{
                /* FIXME: appropriate error handling should be performed here. */
		CELLerr(CELL_F_SPE_TASK_RUN,CELL_R_THREAD_CREATION_FAILED);
		return 0;
		}
#endif /* ENABLE_SYNC_FUTEX */

	return 1;
	}

int spe_task_wait(spe_task_t *task)
	{
	/* wait for the threads */
#ifdef ENABLE_SYNC_FUTEX
	pthread_join(task->handler_thread, NULL);
#endif /* ENABLE_SYNC_FUTEX */
	pthread_join(task->thread, NULL);

	return 1;
	}
